/*
This file is part of SMAP.

SMAP is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

SMAP is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with SMAP.  If not, see <http://www.gnu.org/licenses/>.

*/

var gWait = 0;		// This javascript file only
var gCache = {};
var gCacheGroup = {};
var gCacheStatusQuestions = {};
var gCacheKeys = {};
var gEligibleUser;
var gSelectedOversightQuestion;
var gSelectedOversightSurvey;
var gConversationalSMS;	// Set true if a conversational SMS choice has been added to notification types

/*
 * Convert a choice list name into a valid jquery class name
 */
function jq(choiceList) {

	var c;

	c = choiceList.replace( /(:|\.|\[|\]|,)/g, "\\$1" );
	return c;
}

/* 
 * ==============================================================
 * Task Functions
 * ==============================================================
 */

function addPendingTask(taskId, assignmentId, status, source) {
	var i,
		duplicate = false,
		assignment;

	assignment = {
		assignment_id: assignmentId,
		assignment_status: status,
		task_id: taskId
	};
	globals.gPendingUpdates.push(assignment);

	if(source === "table") {
		updateMapTaskSelections(taskId, true);
	} else if(source === "map") {
		$('#tasks_table').find('[data-taskid=' + taskId + ']').prop("checked", true).closest('tr').addClass("info");
	}
}

function removePendingTask(taskId, source) {
	var i;
	for (i = 0; i < globals.gPendingUpdates.length; i++) {
		if(globals.gPendingUpdates[i].task_id === taskId) {
			globals.gPendingUpdates.splice(i,1);
			break;
		}
	}
	if(source === "table") {
		updateMapTaskSelections(taskId, false);
	} else if(source === "map") {
		$('#tasks_table').find('[data-taskid=' + taskId + ']').prop("checked", false).closest('tr').removeClass("info");
	}
}

/*
 * ===============================================================
 * Project Functions
 * ===============================================================
 */

/*
 * Update the list of available projects
 * Note when addAll is set to true the list is not used to change the default project
 *   In this case the value of the list should not be set to the default project
 */
function updateProjectList(addAll, projectId, callback, $projectSelect) {

	var i,
		h = [],
		idx = -1,
		updateCurrentProject;

	if(projectId > 0) {
		updateCurrentProject = true;		// Only save the current project if there it is set
	}

	if(addAll) {
		h[++idx] = '<option value="0">' + localise.set["c_all"] + '</option>';
		updateCurrentProject = false;
	}
	for(i = 0; i < globals.gProjectList.length; i++) {
		h[++idx] = '<option value="';
		h[++idx] = globals.gProjectList[i].id;
		h[++idx] = '">';
		h[++idx] = htmlEncode(globals.gProjectList[i].name);
		h[++idx] = '</option>';

		if(globals.gProjectList[i].id === projectId) {
			updateCurrentProject = false;   // Don't save the current project if it is already in the list
		}
	}
	$projectSelect.empty().append(h.join(''));

	// If for some reason the user's default project is no longer available then
	//  set the default project to the first project in the list
	//  if the list is empty then set the default project to undefined
	if(updateCurrentProject) {
		if (globals.gProjectList[0]) {
			globals.gCurrentProject = globals.gProjectList[0].id;		// Update the current project id
			globals.gCurrentSurvey = -1;
			globals.gCurrentTaskGroup = undefined;
		} else {
			globals.gCurrentProject = -1;		// Update the current project id
			globals.gCurrentSurvey = -1;
			globals.gCurrentTaskGroup = undefined;
		}

		saveCurrentProject(globals.gCurrentProject,
			globals.gCurrentSurvey,
			globals.gCurrentTaskGroup);
	}

	if(!addAll) {
		$projectSelect.val(globals.gCurrentProject);			// Set the initial project value
		$('#projectId').val(globals.gCurrentProject);			// Set the project value for the hidden field in template upload
	}

	if(typeof callback !== "undefined") {
		callback(globals.gCurrentProject);				// Call the callback with the correct current project
	}
}

/*
 * Get the list of available projects from the server
 */
function getMyProjects(projectId, callback, getAll) {
	addHourglass();
	$.ajax({
		url: "/surveyKPI/myProjectList",
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				globals.gProjectList = data;
				updateProjectList(getAll, projectId, callback, $('.project_list'));
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert("Error: Failed to get list of projects: " + err);
				}
			}
		}
	});
}

/*
 * Save the current project id in the user defaults
 */
function saveCurrentProject(projectId, surveyId, taskGroupId) {

	if(surveyId > 0 || projectId > 0 || taskGroupId > 0) {

		var user = {
			current_project_id: projectId,
			current_survey_id: surveyId,
			current_task_group_id: taskGroupId
		};

		var userString = JSON.stringify(user);

		addHourglass();
		$.ajax({
			type: "POST",
			contentType: "application/json",		// uses application/json
			url: "/surveyKPI/user/currentproject",
			cache: false,
			data: userString,
			success: function(data, status) {
				// Do not process a logout
				removeHourglass();
			}, error: function(data, status) {
				// Do not process a logout
				removeHourglass();
			}
		});
	}
}

/*
 * Save the current relationship between survey and surveyGroup
 */
function saveCurrentGroupSurvey(surveyId, gs, fName) {

	if (surveyId > 0) {

		var groupSurvey = {
			sId: surveyId,
			groupIdent: gs,
			fName: fName
		};

		addHourglass();
		$.ajax({
			type: "POST",
			contentType: "application/x-www-form-urlencoded",
			url: "/surveyKPI/user/groupsurvey",
			cache: false,
			data: JSON.stringify(groupSurvey),
			success: function (data, status) {
				removeHourglass();
				handleLogout(data);
			}, error: function (data, status) {
				removeHourglass();
			}
		});
	}
}

/*
 * ===============================================================
 * User Functions
 * ===============================================================
 */

/*
 * Add user details popup to the page
 * Legacy only used with non bootstrap pages - these should be replaced with bootstrap
 */
function addUserDetailsPopup() {
	var
		h =[],
		idx = -1;


	h[++idx] = '<div id="modify_me_popup" style="display:none;">';
	h[++idx] = '<div class="left_panel">';
	h[++idx] = '<form id="me_edit_form">';
	h[++idx] = '<label for="me_name">';
	h[++idx] = localise.set["c_name"];
	h[++idx] = '</label>';
	h[++idx] = '<input type="text" id="me_name" required><br/>';

	h[++idx] = '<label for="me_language">';
	h[++idx] = localise.set["c_lang"];
	h[++idx] = '</label>';
	h[++idx] = '<select class="language_select" id="me_language"></select><br/>';

	h[++idx] = '<label for="me_email">';
	h[++idx] = localise.set["c_email"];
	h[++idx] = '</label>';
	h[++idx] = '<input type="text" id="me_email" pattern="^[_A-Za-z0-9-\\+]+(\\.[_A-Za-z0-9-]+)*@[A-Za-z0-9-]+(\\.[A-Za-z0-9]+)*(\\.[A-Za-z]{2,})$"><br/>';

	h[++idx] = '<label for="me_organisation">';
	h[++idx] = localise.set["c_org"];
	h[++idx] = '</label>';
	h[++idx] = '<select class="organisation_select" id="me_organisation"></select><br/>';

	h[++idx] = '<label for="me_enterprise">';
	h[++idx] = localise.set["c_ent"];
	h[++idx] = '</label>';
	h[++idx] = '<div id="me_enterprise"></div><br/>';

	h[++idx] = '<label for="u_tz">';
	h[++idx] = localise.set["c_tz"];
	h[++idx] = '</label>';
	h[++idx] = '<select class="timezone_select" id="u_tz"></select>';

	h[++idx] = '</form>';
	h[++idx] = '</div>';
	h[++idx] = '</div>';

	$(document.body).append(h.join(''));

}

/*
 * Populate a language select widget
 */
function populateLanguageSelect(sId, $elem) {
	$.getJSON("/surveyKPI/languages/" + sId, function(data) {

		if(handleLogout(data)) {
			$elem.empty();
			$.each(data, function (j, item) {
				$elem.append('<option value="' + item + '">' + htmlEncode(item) + '</option>');
			});
		}
	});
}

/*
 * Populate a pdf select widget
 * Set the template set as the default to be selected
 * If there is no default template and there is a template specified in settings (legacy) then set that as the default
 */
function populatePdfSelect(sId, $elem) {
	var url = "/surveyKPI/surveys/templates/" + sId;
	url += addCacheBuster(url);

	$.getJSON(url, function(data) {

		if(handleLogout(data)) {
			var defaultTemplateId,
				fromSettingsTemplateId;

			$elem.empty();
			$elem.append('<option value="-2">' + localise.set["c_auto"] + '</option>');
			$elem.append('<option value="-1">' + localise.set["c_none"] + '</option>');
			$.each(data, function (j, item) {
				if (item.default_template) {
					defaultTemplateId = item.id;
				} else if (item.fromSettings) {
					fromSettingsTemplateId = item.id;
				}
				$elem.append('<option value="' + item.id + '">' + htmlEncode(item.name) + '</option>');
			});
			if (typeof defaultTemplateId !== "undefined") {
				$elem.val(defaultTemplateId);
			} else if (typeof fromSettingsTemplateId !== "undefined") {
				$elem.val(fromSettingsTemplateId)
			} else {
				$elem.val(-2);		// Set to auto
			}
		}

	});
}

/*
 * Add user details popup to the page
 */
function addUserDetailsPopupBootstrap4() {
	var	h =[],
		idx = -1;

	h[++idx] = '<div id="modify_me_popup" class="modal fade" tabindex="-1" role="dialog" aria-labelledby="modifyMeLabel" aria-hidden="true">';
	h[++idx] = '<div class="modal-dialog modal-lg">';
	h[++idx] = '<div class="modal-content">';
	h[++idx] = '<div class="modal-header">';
	h[++idx] = '<h4 class="modal-title" id="modifyMeLabel"></h4>';
	h[++idx] = '<button type="button" class="close" data-dismiss="modal" aria-label="Close"><span aria-hidden="true">&times;</span></button>';
	h[++idx] = '</div>';    // modal-headers

	h[++idx] = '<div class="modal-body">';
	h[++idx] = '<form role="form" id="me_edit_form">';
	h[++idx] = '<div class="form-group row">';
	h[++idx] = '<label for="me_name" class="col-sm-2 control-label">';
	h[++idx] = localise.set["c_name"];
	h[++idx] = '</label>';
	h[++idx] = '<div class="col-sm-10">';
	h[++idx] = '<input type="text" id="me_name" required class="form-control">';
	h[++idx] = '</div>';
	h[++idx] = '</div>';

	h[++idx] = '<div class="form-group row">';
	h[++idx] = '<label for="me_language" class="col-sm-2 control-label">';
	h[++idx] = localise.set["c_lang"];
	h[++idx] = '</label>';
	h[++idx] = '<div class="col-sm-10">';
	h[++idx] = '<select id="me_language" class="language_select form-control"></select>';
	h[++idx] = '</div>';
	h[++idx] = '</div>';

	h[++idx] = '<div class="form-group row">';
	h[++idx] = '<label for="me_email" class="col-sm-2 control-label">';
	h[++idx] = localise.set["c_email"];
	h[++idx] = '</label>';
	h[++idx] = '<div class="col-sm-10">';
	h[++idx] = '<input type="email" class="form-control" id="me_email"';
	h[++idx] = ' placeholder="Enter email"';
	h[++idx] = ' pattern="^[_A-Za-z0-9-\\+]+(\\.[_A-Za-z0-9-]+)*@[A-Za-z0-9-]+(\\.[A-Za-z0-9]+)*(\\.[A-Za-z]{2,})$">';
	h[++idx] = '</div>';
	h[++idx] = '</div>';

	h[++idx] = '<div class="form-group row">';
	h[++idx] = '<label for="me_organisation" class="col-sm-2 control-label">';
	h[++idx] = localise.set["c_org"];
	h[++idx] = '</label>';
	h[++idx] = '<div class="col-sm-10">';
	h[++idx] = '<select id="me_organisation" class="organisation_select form-control"></select>';
	h[++idx] = '</div>';
	h[++idx] = '</div>';

	h[++idx] = '<div class="form-group row">';
	h[++idx] = '<label for="me_enterprise" class="col-sm-2 control-label">';
	h[++idx] = localise.set["c_ent"];
	h[++idx] = '</label>';
	h[++idx] = '<div class="col-sm-10">';
	h[++idx] = '<div id="me_enterprise" class="form-control"></div>';
	h[++idx] = '</div>';
	h[++idx] = '</div>';

	h[++idx] = '<div class="form-group row">';
	h[++idx] = '<label for="u_tz" class="col-sm-2 control-label">';
	h[++idx] = localise.set["c_tz"];
	h[++idx] = '</label>';
	h[++idx] = '<div class="col-sm-10">';
	h[++idx] = '<select class="form-control timezone_select" id="u_tz"></select>';
	h[++idx] = '</div>';
	h[++idx] = '</div>';
	
	h[++idx] = '<div id="me_alert" class="alert d-none text-wrap text-break" role="alert"></div>';
	h[++idx] = '</form>';
	h[++idx] = '</div>';    // modal body

	h[++idx] = '<div class="modal-footer">';
	h[++idx] = '<button type="button" class="btn btn-default" data-dismiss="modal">';
	h[++idx] = localise.set["c_close"];
	h[++idx] = '</button>';

	h[++idx] = '<button id="userProfileSave" type="button" class="btn btn-primary">';
	h[++idx] = localise.set["c_save"];
	h[++idx] = '</button>';
	h[++idx] = '</div>';    // modal - footer
	h[++idx] = '</div>';        // modal - content
	h[++idx] = '</div>';            // modal - dialog
	h[++idx] = '</div>';                // popup

	$(document.body).append(h.join(''));

	enableUserProfileBS();
}

/*
 * Add user details popup to the page
 */
function addApiKeyPopup() {
	var	h =[],
		idx = -1;

	h[++idx] = '<div id="api_key_popup" class="modal fade" tabindex="-1" role="dialog" aria-labelledby="apiKeyLabel" aria-hidden="true">';
	h[++idx] = '<div class="modal-dialog modal-lg">';
	h[++idx] = '<div class="modal-content">';
	h[++idx] = '<div class="modal-header">';
	h[++idx] = '<h4 class="modal-title" id="apiKeyLabel"></h4>';
	h[++idx] = '<button type="button" class="close" data-dismiss="modal" aria-label="Close"><span aria-hidden="true">&times;</span></button>';
	h[++idx] = '</div>';    // modal-headers

	h[++idx] = '<div class="modal-body">';
	h[++idx] = '<form>';
	h[++idx] = '<div class="form-group">';
	h[++idx] = '<input type="text" id="apiKey" required class="form-control" readOnly>';
	h[++idx] = '</div>';
	h[++idx] = '</form>';
	h[++idx] = '<button id="getKey" type="button" class="btn btn-primary">';
	h[++idx] = localise.set["c_gak"];
	h[++idx] = '</button>';
	h[++idx] = '<button id="deleteKey" type="button" class="btn btn-danger ml-2">';
	h[++idx] = localise.set["c_del"];
	h[++idx] = '</button>';
	h[++idx] = '<button id="copyKey" type="button" class="btn btn-default has_tt ml-2" title="Copy Key">';
	h[++idx] = localise.set["c_ck"];
	h[++idx] = '</button>';
	h[++idx] = '</div>';

	h[++idx] = '<div class="modal-footer">';
	h[++idx] = '<button type="button" class="btn btn-default" data-dismiss="modal">';
	h[++idx] = localise.set["c_close"];
	h[++idx] = '</button>';

	h[++idx] = '</div>';    // modal - footer
	h[++idx] = '</div>';        // modal - content
	h[++idx] = '</div>';            // modal - dialog
	h[++idx] = '</div>';                // popup

	$(document.body).append(h.join(''));

	enableApiKeyPopup();
}


/*
 * Update the user details on the page
 */
function updateUserDetails(data, getOrganisationsFn, getEnterprisesFn, getServerDetailsFn) {

	var groups = data.groups,
		i,
		bootstrap_enabled = (typeof $().modal == 'function');

	if(data.language && data.language !== gUserLocale) {
		try {
			localStorage.setItem('user_locale', data.language);  // Write to storage may be disabled
			location.reload();
		} catch (e) {

		}

	} else if(data.o_id != globals.gOrgId) {
		location.reload();
	}

	globals.gLoggedInUser = data;
	globals.gOrgId = data.o_id;

	// Save the organisation name for the logon screen
	try {
		localStorage.setItem('org_name', data.organisation_name);
	} catch (e) {

	}

	if(bootstrap_enabled) {

		$('#modify_me_popup').on('show.bs.modal', function (event) {
			var $this = $(this)
			$this.find('.modal-title').text(data.ident + "@" + data.organisation_name)

			$("#me_alert").hide();

			$('#me_edit_form')[0].reset();
			$('#reset_me_password_fields').removeClass('d-none').show();
			$('#password_me_fields').hide();
			addLanguageOptions($('.language_select'), data.language);
			addOrganisationOptions($('.organisation_select'), data.o_id, data.orgs);
			$('#me_name').val(data.name);
			$('#me_email').val(data.email);
			$('#me_enterprise').text(globals.gEnterpriseName);
			$('#u_tz').val(globals.gTimezone);

			$(".navbar-collapse").removeClass("in").addClass("collapse");	// Remove drop down menu
		});


	} else {
		$('#username').text(data.name).button({ label: htmlEncode(data.name),
			icons: { primary: "ui-icon-person" }}).off().click(function(){
			$('#me_edit_form')[0].reset();

			$('#reset_me_password_fields').removeClass('d-none').show();
			$('#password_me_fields').hide();
			addLanguageOptions($('.language_select'), data.language);
			addOrganisationOptions($('.organisation_select'), data.o_id, data.orgs);
			$('#me_name').val(data.name);
			$('#me_email').val(data.email);
			$('#me_enterprise').text(globals.gEnterpriseName);
			$('#u_tz').val(globals.gTimezone);

			$('#modify_me_popup').dialog("option", "title", htmlEncode(data.name + "@" + data.organisation_name));
			$('#modify_me_popup').dialog("open");
		});
	}

	/*
	 * Show restricted functions
	 */
	if(groups) {
		for(i = 0; i < groups.length; i++) {
			if(groups[i].id === globals.GROUP_ADMIN) {
				globals.gIsAdministrator = true;

                if(data.billing_enabled) {
                    globals.gOrgBillingData = true;
                }

			} else if(groups[i].id === globals.GROUP_ORG_ADMIN) {
				globals.gIsOrgAdministrator = true;
				globals.gBillingData = true;

			} else if(groups[i].id === globals.GROUP_SECURITY) {
				globals.gIsSecurityAdministrator = true;

			} else if(groups[i].id === globals.GROUP_ENTERPRISE) {
                globals.gIsEnterpriseAdministrator = true;
				globals.gBillingData = true;

            } else if(groups[i].id === globals.GROUP_LINKAGES) {
				globals.gIsLinkFollower = true;

			} else if(groups[i].id === globals.GROUP_ANALYST) {
				globals.gIsAnalyst = true;

			} else if(groups[i].id === globals.GROUP_DASHBOARD) {
				globals.gIsDashboard = true;

			} else if(groups[i].id === globals.GROUP_MANAGE) {
				globals.gIsManage = true;

			} else if(groups[i].id === globals.GROUP_ENUM) {
				globals.gIsEnum = true;

			} else if(groups[i].id === globals.GROUP_VIEW_DATA) {
                globals.gViewData = true;

            } else if(groups[i].id === globals.GROUP_MANAGE_TASKS) {
				globals.gManageTasks = true;

			} else if(groups[i].id === globals.GROUP_OWNER) {
                globals.gIsServerOwner = true;

            } else if(groups[i].id === globals.GROUP_CONSOLE_ADMIN) {
				globals.gIsConsoleAdmin = true;
			}
		}
	}

	// Only show items relevant to a user
	$('.restrict_role').hide();
	if(globals.gIsEnum) {
		$('.enum_role').removeClass('d-none').show();
	}
	if(globals.gIsAnalyst) {
		$('.analyst_role').removeClass('d-none').show();
	}
	if(globals.gIsDashboard) {
		$('.dashboard_role').removeClass('d-none').show();
	}
	if(globals.gViewData) {
		$('.data_role').removeClass('d-none').show();
	}
	if(globals.gManageTasks) {
		$('.task_role').show();
	}
	if(globals.gIsAdministrator) {
		$('.admin_role').removeClass('d-none').show();
	}
	if(globals.gIsManage) {
		$('.manage_role').removeClass('d-none').show();
	}
	if(globals.gIsSecurityAdministrator) {
		$('.security_role').removeClass('d-none').show();
	}
	if(globals.gIsOrgAdministrator) {  // Admins can see their personal organisations
		$('.org_role').removeClass('d-none').show();
	}
	if(globals.gIsOrgAdministrator || globals.gIsAdministrator) {  // Admins can see their personal organisations
		if(typeof getOrganisationsFn === "function") {
			getOrganisationsFn();
		}
	}
	if(globals.gIsEnterpriseAdministrator) {
		$('.enterprise_role').removeClass('d-none').show();
		if(typeof getEnterprisesFn === "function") {
			getEnterprisesFn();
		}
	}
	if(globals.gIsServerOwner) {
		$('.owner_role').removeClass('d-none').show();
		if(typeof getServerDetailsFn === "function") {
			getServerDetailsFn();
		}
	}

	if(globals.gTraining) {
		$('#train_link').prop("href", globals.gTraining);
		$('#m_training').removeClass('d-none').show();
	}

	//TODO set logic for enabling disabling billing
	if(isBusinessServer() && (globals.gBillingData || globals.gOrgBillingData)) {
		$('.billing_role').removeClass('d-none').show();
	}

	// Other conditional elements
	if(globals.gSendTrail === 'off') {
		$('.user_trail').hide();
	}

	// 	Customer configurable details - the configurable part is TODO
	$('#my_name').val(data.name);			// Add the name to the configurable list

	if(data.settings) {
		var userDetails = JSON.parse(data.settings);
		$('#my_title').val(userDetails.title);
		$('#my_license').val(userDetails.license);
		$('#my_signature').attr("src", "/surveyKPI/file/" + data.signature + "/users?type=sig");
	}

	// Hide any menus that have been disabled by custom java scripts
	$('.perm_dis_menu').hide();
}

function addLanguageOptions($elem, current) {

	var h = [],
		idx = -1,
		i,
		languages = [
			{
				locale: "ar",
				name: "Arabic"
			},
			{
				locale: "en",
				name: "English"
			},
			{
				locale: "fr",
				name: "French"
			},
			{
				locale: "hi",
				name: "Hindi"
			},
			{
				locale: "pt",
				name: "Portugese"
			},
			{
				locale: "es",
				name: "Spanish"
			},
			{
				locale: "uk",
				name: "Ukrainian"
			}
		];

	for(i = 0; i < languages.length; i++) {
		h[++idx] = '<option value="';
		h[++idx] = languages[i].locale;
		h[++idx] = '">';
		h[++idx] = localise.set[languages[i].locale];
		h[++idx] = '</option>';
	}
	$elem.html(h.join(''));
	if(current) {
		$elem.val(current);
	} else {
		$elem.val("en");
	}
}

function addOrganisationOptions($elem, current, orgs) {

	var h = [],
		idx = -1,
		i;

	for(i = 0; i < orgs.length; i++) {
		h[++idx] = '<option value="';
		h[++idx] = orgs[i].id;
		h[++idx] = '">';
		h[++idx] = htmlEncode(orgs[i].name);
		h[++idx] = '</option>';
	}
	$elem.html(h.join(''));
	if(current) {
		$elem.val(current);
	}
}

/*
 * Enable the user profile button
 */
function enableUserProfile () {
	// Initialise the dialog for the user to edit their own account details
	$('#modify_me_popup').dialog(
		{
			autoOpen: false, closeOnEscape:true, draggable:true, modal:true,
			title:"User Profile",
			show:"drop",
			width:350,
			height:350,
			zIndex: 2000,
			buttons: [
				{
					text: "Cancel",
					click: function() {

						$(this).dialog("close");
					}
				}, {
					text: "Save",
					click: function() {

						var user = globals.gLoggedInUser;

						user.name = $('#me_name').val();
						user.language = $('#me_language').val();
						user.email = $('#me_email').val();
						if($('#me_password').is(':visible')) {
							user.password = $('#me_password').val();
							if($('#me_password_confirm').val() !== user.password) {
								user.password = undefined;
								alert("Passwords do not match");
								$('#me_password').focus();
								return false;
							}
						} else {
							user.password = undefined;
						}

						user.current_project_id = 0;	// Tell service to ignore project id and update other details
						user.current_survey_id = 0;
						user.current_task_group_id = 0;

						user.timezone = $('#u_tz').val();
						globals.gTimezone = user.timezone;

						user.o_id = $('#me_organisation').val();
						if(user.o_id == globals.gOrgId) {
							user.o_id = 0;	// No change
						}

						saveCurrentUser(user, undefined);			// Save the updated user details to disk
						$(this).dialog("close");
					},
				}
			]
		}
	);


	// Initialise the reset password checkbox
	$('#reset_me_password').click(function () {
		if($(this).is(':checked')) {
			$('#password_me_fields').removeClass('d-none').show();
		} else {
			$('#password_me_fields').hide();
		}
	});
}

/*
 * Enable the user profile button
 */
function enableUserProfileBS () {

	$("#modify_me_popup :input").keydown(function() {
		$("#me_alert").hide();
	});

	/*
	 * Save the user profile
	 */
	$('#userProfileSave').click(function() {
		var user = globals.gLoggedInUser;

		user.name = $('#me_name').val();
		user.language = $('#me_language').val();
		user.email = $('#me_email').val();
		if($('#me_password').is(':visible')) {
			user.password = $('#me_password').val();
			if($('#me_password_confirm').val() !== user.password) {
				user.password = undefined;
				$('#me_alert').removeClass('alert-success d-none').addClass('alert-danger').text(localise.set["msg_pwd_m"]).show();
				$('#me_password').focus();
				return false;
			}
		} else {
			user.password = undefined;
		}

		user.o_id = $('#me_organisation').val();
		if(user.o_id == globals.gOrgId) {
			user.o_id = 0;	// No change
		}

		globals.gTimezone = $('#u_tz').val();
		user.timezone = globals.gTimezone;

		user.current_project_id = 0;	// Tell service to ignore project id and update other details
		user.current_survey_id = 0;
		user.current_task_group_id = 0;

		saveCurrentUser(user, $('#modify_me_popup'));			// Save the updated user details to disk
	});


	// Initialise the reset password checkbox
	$('#reset_me_password').click(function () {
		if($(this).is(':checked')) {
			$('#password_me_fields').removeClass('d-none').show();
		} else {
			$('#password_me_fields').hide();
		}
	});
}

/*
 * Respond to events on the API key popup
 */
function enableApiKeyPopup() {

	$('#api_key_popup').on('show.bs.modal', function (event) {
		/*
		 * Get the current API key
		 */
		$('#getKey').prop('disabled', true);
		addHourglass();
		$.ajax({
			url: '/surveyKPI/user/api_key',
			cache: false,
			success: function (data) {
				removeHourglass();
				if (handleLogout(data)) {
					$('#apiKey').val(data.apiKey);
					$('#getKey').prop('disabled', false);
					if (data.apiKey) {
						$('#getKey').text(localise.set["c_rak"]);
						$('#deleteKey,#copyKey').prop('disabled', false);
					} else {
						$('#getKey').text(localise.set["c_gak"]);
						$('#deleteKey,#copyKey').prop('disabled', true);
					}
				}
			},
			error: function (xhr, textStatus, err) {
				removeHourglass();
				if (handleLogout(xhr.responseText)) {
					$('#getKey').prop('disabled', false);
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert(err);
						console.log("Error: Failed to get api key: " + err);
					}
				}
			}
		});
	});

	/*
	 * Delete a key
	 */
	$('#deleteKey').on("click",function () {
		addHourglass();
		$.ajax({
			type: "DELETE",
			url: '/surveyKPI/user/api_key',
			cache: false,
			success: function (data) {
				removeHourglass();
				if (handleLogout(data)) {
					$('#apiKey').val("");
					$('#getKey').prop('disabled', false);
					$('#getKey').text(localise.set["c_gak"]);
					$('#deleteKey,#copyKey').prop('disabled', true);
				}
			},
			error: function (xhr, textStatus, err) {
				removeHourglass();
				if (handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert(err);
						console.log("Error: Failed to delete api key: " + err);
					}
				}
			}
		});
	});

	/*
	 * Create a key
	 */
	$('#getKey').on("click", function () {
		addHourglass();
		$.ajax({
			type: "POST",
			cache: false,
			contentType: "application/x-www-form-urlencoded",
			dataType: 'json',
			url: "/surveyKPI/user/api_key/create",
			success: function (data) {
				removeHourglass();
				if (handleLogout(data)) {
					$('#apiKey').val(data.apiKey);
					$('#getKey').text(localise.set["c_rak"]);
					$('#deleteKey,#copyKey').prop('disabled', false);
				}
			},
			error: function (xhr, textStatus, err) {
				removeHourglass();
				if (handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert(err);
						console.log("Error: Failed to get api key: " + err);
					}
				}
			}
		});
	});

	// Respond to a user clicking copy api key
	$('.has_tt').tooltip();
	$('#copyKey').click(function () {
		var copyText = document.getElementById("apiKey");
		copyText.select();
		navigator.clipboard.writeText($('#apiKey').val());

		$('#copyKey').tooltip('dispose').tooltip({title: localise.set["c_c"] + ": " + copyText.value}).tooltip('show');

	});
	$('#copyKey').mouseout(function () {
		$('#copyKey').tooltip({title: localise.set["c_c"]});
	});
}

/*
 * Save the currently logged on user's details
 */
function saveCurrentUser(user, $dialog) {

	var fd = new FormData();
	fd.append("user", JSON.stringify(user));
	addHourglass();
	$.ajax({
		method: "POST",
		cache: false,
		contentType: false,
		processData: false,
		url: "/surveyKPI/user",
		data: fd,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				if (data.error) {
					if ($dialog) {
						$('#me_alert').removeClass('alert-success d-none').addClass('alert-danger').text(data.msg).show();
					} else {
						alert(localise.set["c_error"] + " : " + data.msg);  // legacy non bootstrap
					}
				} else if ($dialog) {
					$dialog.modal("hide");
				}
				updateUserDetails(data, undefined);
			}

		}, error: function(data, status) {
			removeHourglass();
			alert(localise.set["c_error"] + " : " + data.responseText);
		}
	});
}

function getAvailableTimeZones(callback) {
	addHourglass();
	$.ajax({
		url: "/surveyKPI/utility/timezones",
		cache: true,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				if (typeof callback == "function") {
					callback(data);
				}
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert(localise.set["c_error"] + ": " + err);
				}
			}
		}
	});
}

function showTimeZones(timeZones) {
	var h =[],
		idx = -1,
		i,
		tz;

	for (i = 0; i < timeZones.length; i++) {
		tz = timeZones[i];
		h[++idx] = '<option value="';
		h[++idx] = tz.id;
		h[++idx] = '">';
		h[++idx] = htmlEncode(tz.name);
		h[++idx] = '</option>';
	}
	$('.timezone_select').empty().html(h.join(''));
	if(!globals.gTimezone) {
		globals.gTimezone = Intl.DateTimeFormat().resolvedOptions().timeZone;      // Browser timezone
	}
	$('#u_tz').val(globals.gTimezone);   // Set time zone in user profile
	$('#timezone').text(localise.set["c_tz"] + ": " + globals.gTimezone);   // Show timezone where this is enabled
}

function addTimeZoneToUrl(url) {
	if(url) {
		if(url.indexOf("?") > 0) {
			url += "&";
		} else {
			url += "?";
		}
		url += "tz=";
		url += encodeURIComponent(globals.gTimezone);
	}
	return url;
}

/*
 * Create the user profile dialog and get any data it needs
 */
function setupUserProfile(bs4) {

	if(bs4) {
		addUserDetailsPopupBootstrap4();
		addApiKeyPopup();
	} else {
		addUserDetailsPopup();	// legacy
	}
	getAvailableTimeZones(showTimeZones);
}

function getLoggedInUser(callback, getAll, getProjects, getOrganisationsFn, hideUserDetails,
                         dontGetCurrentSurvey, getEnterprisesFn, getServerDetailsFn, getSMSNumbers) {

	globals.gIsAdministrator = false;

	addHourglass();
	$.ajax({
		url: "/surveyKPI/user",
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				var i;

				globals.gServerCanSendEmail = data.sendEmail;

				globals.gEmailEnabled = data.allow_email;
				globals.gFacebookEnabled = data.allow_facebook;
				globals.gTwitterEnabled = data.allow_twitter;
				globals.gCanEdit = data.can_edit;
				globals.gSendTrail = data.ft_send_location;
				globals.gAlertSeen = data.seen;		// Alerts have been acknowledged
				globals.gLastAlertTime = data.lastalert;
				globals.gOrgId = data.o_id;
				globals.gEntId = data.e_id;
				globals.gEnterpriseName = data.enterprise_name;
				globals.gSetAsTheme = data.set_as_theme;
				globals.gNavbarColor = data.navbar_color;
				globals.gNavbarTextColor = data.navbar_text_color;
				globals.gTraining = data.training;
				globals.gRefreshRate = data.refresh_rate;

				if (data.timezone) {
					globals.gTimezone = data.timezone;
				} else {
					globals.gTimezone = Intl.DateTimeFormat().resolvedOptions().timeZone;
				}
				$('#u_tz').val(globals.gTimezone);

				if (!hideUserDetails) {
					updateUserDetails(data, getOrganisationsFn, getEnterprisesFn, getServerDetailsFn);
				}

				if(getSMSNumbers) {
					getSMSNumbers();
				}

				if(data.totalTasks > 0) {
					$('.total_tasks').html('(' + htmlEncode(data.totalTasks) + ')').addClass('btn-danger');
				}

				if (!dontGetCurrentSurvey) {	// Hack, on edit screen current survey is set as parameter not from the user's defaults
					globals.gCurrentSurvey = data.current_survey_id;
					globals.gCurrentSurveyIdent = data.current_survey_ident;
				}
				globals.gCurrentProject = data.current_project_id;
				globals.gCurrentTaskGroup = data.current_task_group_id;
				$('#projectId').val(globals.gCurrentProject);		// Set the project value for the hidden field in template upload
				if (data.groupSurveys) {
					for (i = 0; i < data.groupSurveys.length; i++) {
						globals.gGroupSurveys[data.groupSurveys[i].sId] = data.groupSurveys[i].groupIdent;
						globals.gSubForms[data.groupSurveys[i].sId] = data.groupSurveys[i].fName;
					}
				}

				setOrganisationTheme();

				if (getProjects) {
					getMyProjects(globals.gCurrentProject, callback, getAll);	// Get projects
				} else {
					if (typeof callback !== "undefined") {
						callback(globals.gCurrentSurvey);				// Call the callback with the correct current survey
					}
				}
			}

		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0 || xhr.status == 401) {
					return;  // Not an error or an authorisation error which is handled by the service worker
				} else {
					console.log("Error: Failed to get user details: " + err);

					var msg = localise.set["c_error"] + ": ";
					if (err && err.message && err.message.indexOf('Unauthorized') >= 0) {
						msg += localise.set["c_auth"];
					} else {
						msg += err;
					}
					alert(msg);
				}
			}
		}
	});
}

/*
 * ===============================================================
 * Common functions for managing media (on both the edit page and shared resource page)
 * ===============================================================
 */

/*
 * Upload files to the server
 */
function uploadFiles(url, formName, callback1) {

	let f = document.forms.namedItem(formName),
		formData = new FormData(f);

	url = addUrlParam(url, "getlist=true");
	addHourglass();
	$('.submitFiles').addClass('disabled');
	$.ajax({
		url: url,
		type: 'POST',
		data: formData,
		cache: false,
		contentType: false,
		processData:false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				let cb1 = callback1;
				$('.upload_file_msg').show().removeClass('alert-danger').addClass('alert-success').html(localise.set["c_success"]);
				if (typeof cb1 === "function") {
					cb1(data);
				}
				document.forms.namedItem(formName).reset();
				$('#fileAddLocations').modal('hide');
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				document.forms.namedItem(formName).reset();
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					var msg = htmlEncode(xhr.responseText);
					if (msg && msg.indexOf("no tags") >= 0) {
						msg = localise.set["msg_u_nt"];
					} else {
						msg = localise.set["msg_u_f"] + " : " + msg;
					}
					$('.upload_file_msg').show().removeClass('alert-success').addClass('alert-danger').html(msg);
					$('#fileAddLocations').modal('hide');
				}
			}
		}
	});
}

/*
 * Add a parameter to a URL
 */
function addUrlParam(url, param) {
	if(url.indexOf("?") > 0) {
		url += "&" + param;
	} else {
		url += "?" + param;
	}
	return url;
}

/*
 * Refresh the media view and then set the mode to manage
 */
function refreshMediaViewManage(data, sId) {
	refreshMediaView(data, sId);
	$('.mediaManage').show();
	$('.mediaSelect').hide();
}
/*
 * Refresh the view of any attached media if the available media items has changed
 * sId is set if a resources for that survey are being viewed
 */
function refreshMediaView(data, sId) {

	var i,
		$elementMedia,
		$elementCsv,
		hCsv = [],
		idxCsv = -1,
		hMedia = [],
		idxMedia = -1;

	if(sId) {
		$('#survey_id').val(sId);
	}

	if(data) {
		window.gFiles = data.files;
		let files = data.files;

		$elementMedia = $('#filesOrg');
		$elementCsv = $('#csvOrg');

		if(files) {
			for (i = 0; i < files.length; i++) {
				if (files[i].type === 'csv') {
					hCsv[++idxCsv] = getMediaRecord(files[i], 'csv', i, sId > 0);
				} else {
					hMedia[++idxMedia] = getMediaRecord(files[i], 'media', i, sId > 0);
				}
			}
		}

		$elementMedia.html(hMedia.join(""));
		$elementCsv.html(hCsv.join(""));

		$('.media_delete').click(function () {
			let item = window.gFiles[$(this).val()];

			if(confirm(localise.set["msg_confirm_del"] + " " + htmlEncode(item.name))) {
				delete_media(item.name, sId);
			}
		});

		$('.media_history').click(function () {
			var item = window.gFiles[$(this).val()];
			var url = '/app/resource_history.html?resource=' + item.name;
			if(sId) {
				url += '&survey_id=' + sId;
			}
			window.location.href = url;
		});

		$('.csv_replace').click(function(e) {

			$('#fileCsv').show();
			$('#fileMedia').hide();

			replace(window.gFiles[$(this).val()]);
		});

		$('.media_replace').click(function(e) {

			$('#fileCsv').hide();
			$('#fileMedia').show();

			replace(window.gFiles[$(this).val()]);
		});

	}

	// If this is the organisational view we can refresh the list of choices for selecting vector maps
	if(!sId) {
		refreshVectorSelects(data);
	}
}

function replace(item) {

	$('#uploadAction').val('replace');
	$('#itemName').val(getBaseName(item.name));

	$('.upload_alert').hide();
	$('.notreplace').hide();
	$('#media_add_title').text(localise.set["tm_c_sr_rep"] + ": " + item.name);

	$('#fileAddPopup').modal('show');

}
function getBaseName(fileName) {
	let lastDot = fileName.lastIndexOf(".");
	let baseName = fileName;
	if (lastDot !== -1) {
		baseName = fileName.substr(0, lastDot);
	}
	return baseName;
}
function getMediaRecord(file, panel, record, surveyLevel) {
	var h = [],
		idx = -1;

	h[++idx] = '<tr class="';
	h[++idx] = htmlEncode(file.type);
	h[++idx] = '">';

	if(panel === 'media') {
		h[++idx] = '<td class="preview">';
		h[++idx] = '<a target="_blank" href="';
		h[++idx] = htmlEncode(file.url) + addCacheBuster(file.url);
		h[++idx] = '">';
		if (file.type == "audio") {
			h[++idx] = addAudioIcon();
		} else if (file.type == "geojson") {
			h[++idx] = addVectorMapIcon();
		} else {
			h[++idx] = '<img width="100" height="100" src="';
			h[++idx] = htmlEncode(file.thumbnailUrl) + addCacheBuster(file.thumbnailUrl);
			h[++idx] = '" alt="';
			h[++idx] = htmlEncode(file.name);
			h[++idx] = '">';
		}
		h[++idx] = '</a>';
		h[++idx] = '</td>';
	}

	h[++idx] = '<td class="filename">';
	h[++idx] = '<p>';
	h[++idx] = htmlEncode(file.name);
	h[++idx] = '</p>';
	h[++idx] = '</td>';

	h[++idx] = '<td class="mediaManage">';
	h[++idx] = localTime(file.modified);
	h[++idx] = '</td>';

	h[++idx] = '<td class="mediaManage">';
	h[++idx] = '<p>';
	h[++idx] = htmlEncode(file.size);
	h[++idx] = '</p>';
	h[++idx] = '</td>';

	h[++idx] = '<td class="mediaManage">';
	h[++idx] = '<button class="btn ';
	h[++idx] = (panel === 'csv') ? 'csv_replace' : 'media_replace';
	h[++idx] = '" value="';
	h[++idx] = record;
	h[++idx] = '">';
	h[++idx] = '<i class="fas fa-sync-alt"></i>';
	h[++idx] = '</button>';
	h[++idx] = '</td>';

	// Action Buttons
	let downloadUrl = '/surveyKPI/shared/latest/' + file.name
		+ (surveyLevel ? '?sIdent=' + globals.gCurrentSurveyIdent : '');
	h[++idx] = '<td class="mediaManage">';
	h[++idx] = '<a class="media_download btn btn-info" href="';					// Download
	h[++idx] = htmlEncode(downloadUrl + addCacheBuster(downloadUrl));
	h[++idx] = '">';
	h[++idx] = '<i class="fas fa-download"></i>'
	h[++idx] = '</a>';
	h[++idx] = '<button class="media_history btn btn-primary" value="';	// History
	h[++idx] = record;
	h[++idx] = '">';
	h[++idx] = '<i class="fas fa-landmark"></i>'
	h[++idx] = '</button>';
	h[++idx] = '<button class="media_delete btn btn-danger" value="';		// Delete
	h[++idx] = record;
	h[++idx] = '">';
	h[++idx] = '<i class="fas fa-trash-alt"></i>'
	h[++idx] = '</button>';

	h[++idx] = '</td>';

	h[++idx] = '<td class="mediaSelect">';
	h[++idx] = '<button class="mediaAdd btn btn-success">';
	h[++idx] = '<i class="fas fa-plus"></i> '
	h[++idx] = localise.set['c_add'];
	h[++idx] = '</button>';
	h[++idx] = '</td>';


	h[++idx] = '</tr>';

	return h.join('');
}
/*
 * Refresh the vector select lists
 */
function refreshVectorSelects(data) {

	var i,
		$vectorData = $('#vector_data'),
		$vectorStyle = $('#vector_style'),
		h_d = [],
		idx_d = -1,
		h_s = [],
		idx_s = -1,
		files;

	if(data) {
		files = data.files;

		for(i = 0; i < files.length; i++){
			if(files[i].type === "geojson") {
				h_d[++idx_d] = '<option value="';
				h_d[++idx_d] = files[i].name;
				h_d[++idx_d] = '">';
				h_d[++idx_d] = htmlEncode(files[i].name);
				h_d[++idx_d] = '</option>';
			}

			if(files[i].type === "TODO") {
				h_s[++idx_s] = '<option value="';
				h_s[++idx_s] = files[i].name;
				h_s[++idx_s] = '">';
				h_s[++idx_s] = htmlEncode(files[i].name);
				h_s[++idx_s] = '</option>';
			}

		}


		$vectorData.html(h_d.join(""));
		$vectorStyle.html(h_s.join(""));
	}
}

function addAudioIcon() {
	var h = [],
		idx = -1;

	h[++idx] = '<span class="has_tt" title="Audio">';
	h[++idx] = '<span class="glyphicon glyphicon-volume-up edit_type"></span>';
	h[++idx] = '</span>';

	return h.join('');
}

function addVectorMapIcon() {
	var h = [],
		idx = -1;

	h[++idx] = '<span class="has_tt" title="Audio">';
	h[++idx] = '<span class="glyphicon glyphicon glyphicon-map-marker edit_type"></span>';
	h[++idx] = '</span>';

	return h.join('');
}

function getFilesFromServer(sId, callback, getall) {

	let url = '/surveyKPI/upload/media';
	let hasParams = false;
	if(sId) {
		url += '?survey_id=' + sId;
		hasParams = true;
	}
	if(getall) {
		url += (hasParams ? '&' : '?') + 'getall=true';
	}

	url += addCacheBuster(url);

	addHourglass();
	$.ajax({
		url: url,
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				let surveyId = sId;
				callback(data, surveyId);
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					$('.upload_file_msg').removeClass('alert-success').addClass('alert-danger').html("Error: " + htmlEncode(err));
				}
			}
		}
	});
}

/*
 * Delete a media file
 */
function delete_media(filename, sId) {

	var url = "/surveyKPI/shared/file/" + encodeURIComponent(filename);

	if(sId > 0) {
		url += '?survey_id=' + sId;
	}

	addHourglass();
	$.ajax({
		url: url,
		type: 'DELETE',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				var surveyId = sId;
				getFilesFromServer(surveyId, refreshMediaViewManage, false);
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert("Error: " + err);
				}
			}
		}
	});

}
/*
 * ===============================================================
 * Hourglass Functions
 * ===============================================================
 */

function addHourglass() {

	if(gWait === 0) {

		$("#hour_glass,.hour_glass,.sk-spinner").show();
	}
	++gWait;
}

function removeHourglass() {

	--gWait;
	if(gWait === 0) {

		$("#hour_glass,.hour_glass,.sk-spinner").hide();
	}

}

/*
 * ===============================================================
 * Survey Functions
 * ===============================================================
 */

/*
 * Load the surveys from the server
 */
function loadSurveys(projectId, selector, getDeleted, addAll, callback, useIdx, sId, addNone, incReadOnly) {

	var url="/surveyKPI/surveys?projectId=" + projectId + "&blocked=true";

	if(selector === undefined) {
		selector = ".survey_select";	// Update the entire class of survey select controls
	}

	if(typeof projectId !== "undefined" && projectId > 0) {

		if(getDeleted) {
			url+="&deleted=true";
		}

		addHourglass();

		$.ajax({
			url: url,
			dataType: 'json',
			cache: false,
			success: function(data) {
				removeHourglass();
				if(handleLogout(data)) {
					var sel = selector;
					var all = addAll;

					showSurveyList(data, sel + ".data_survey", all, true, false, useIdx, sId, addNone, false, incReadOnly);
					showSurveyList(data, sel + ".oversight_survey", all, false, true, useIdx, sId, addNone, false, incReadOnly);
					showSurveyList(data, sel + ".data_oversight_survey", all, true, true, useIdx, sId, addNone, false, incReadOnly);
					showSurveyList(data, ".bundle_select", all, true, true, false, sId, addNone, true, incReadOnly);

					if (typeof callback == "function") {
						callback(data);
					}
				}
			},
			error: function(xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						console.log("Error: Failed to get list of surveys: " + err);
					}
				}
			}
		});
	} else {
		var $elem = $('.data_survey, .oversight_survey, .data_oversight_survey');
		$elem.empty();
		if(addAll) {
			$elem.append('<option value="_all">' + localise.set["c_all_s"] + '</option>');
		}

		if(callback) {
			callback();
		}

	}
}

/*
 * Load the surveys from the server
 */
function loadSurveyIdentList(projectId, sIdent, addAll, addNone) {

	var url="/surveyKPI/surveys/project/" + projectId;
	var selector = ".survey_select";

	if(typeof projectId !== "undefined" && projectId > 0) {
		addHourglass();
		$.ajax({
			url: url,
			dataType: 'json',
			cache: false,
			success: function(data) {
				removeHourglass();
				if(handleLogout(data)) {
					var sel = selector;

					showIdentSurveyList(data, sel, addAll, sIdent, addNone);

				}
			},
			error: function(xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						console.log("Error: Failed to get list of surveys: " + err);
					}
				}
			}
		});
	}
}

/*
 * Show the surveys in select controls
 */
function showSurveyList(data, selector, addAll, dataSurvey, oversightSurvey, useIdx, sId, addNone, bundle, incReadOnly) {

	var i,
		item,
		h = [],
		idx = -1,
		$elem,
		$elem_disable_blocked,
		selValue;

	$elem = $(selector);
	$elem_disable_blocked = $(selector + ".disable_blocked");

	$elem.empty();
	var valueSelected = false;
	if(addAll) {
		h[++idx] = '<option value="_all">';
		h[++idx] = localise.set["c_all_s"];		// All Surveys
		h[++idx] = '</option>';

		selValue = "_all";
		valueSelected = true;
	}
	if(addNone) {
		h[++idx] = '<option value="0">';
		h[++idx] = localise.set["c_none"];		// No survey
		h[++idx] = '</option>';
	}

	var bundleObj = {};
	for(i = 0; i < data.length; i++) {
		item = data[i];
		if(!bundle || !bundleObj[item.groupSurveyDetails]) {	// If this is for a bundle list remove duplicate entries
			if ((incReadOnly || !item.readOnlySurvey) && (item.dataSurvey && dataSurvey || item.oversightSurvey && oversightSurvey)) {
				h[++idx] = '<option';
				if (!valueSelected && !item.blocked) {
					valueSelected = true;
					selValue = useIdx ? i : item.id;
				}
				if (item.blocked && !bundle) {
					h[++idx] = ' class="blocked"';
				}
				h[++idx] = ' value="';
				if(bundle) {
					h[++idx] = useIdx ? i : item.groupSurveyIdent;
				} else {
					h[++idx] = useIdx ? i : item.id;
				}
				h[++idx] = '">';
				if(bundle){
					h[++idx] = htmlEncode(item.groupSurveyDetails);
					bundleObj[item.groupSurveyDetails] = '1';
				} else {
					h[++idx] = htmlEncode(item.displayName);

					if (item.blocked) {
						h[++idx] = ' (' + localise.set["c_blocked"] + ')';
					}
				}
				h[++idx] = '</option>';
			}
			if (typeof sid === 'undefined') {
				if (globals.gCurrentSurvey > 0 && globals.gCurrentSurvey === item.id) {
					selValue = useIdx ? i : item.id;
				}
			} else {
				if (sId > 0 && sId === item.id) {
					selValue = useIdx ? i : item.id;
				}
			}
		}
	}

	$elem.empty().append(h.join(''));
	$elem.val(selValue);
	$("option.blocked", $elem_disable_blocked).attr("disabled", "disabled");

}

/*
 * Show the surveys in select boxes
 */
function showIdentSurveyList(data, selector, addAll, sIdent, addNone) {

	var i,
		item,
		h = [],
		idx = -1,
		$elem;

	$elem = $(selector);

	$elem.empty();
	if(addAll) {
		h[++idx] = '<option value="_all">';
		h[++idx] = localise.set["c_all_s"];		// All Surveys
		h[++idx] = '</option>';
	}
	if(addNone) {
		h[++idx] = '<option value="_none">';
		h[++idx] = localise.set["c_none"];		// No Survey
		h[++idx] = '</option>';
	}

	for(i = 0; i < data.length; i++) {
		item = data[i];
		h[++idx] = '<option';
		h[++idx] = ' value="';
		h[++idx] = item.ident;
		h[++idx] = '">';
		h[++idx] = htmlEncode(item.name);
		h[++idx] = '</option>';
	}

	$elem.empty().append(h.join(''));
	if(sIdent) {
		$elem.val(sIdent);
	} else {
		$elem.val("_none");
	}

}


// Common Function to get the language and question list (for the default language)
function getLanguageList(sId, callback, addNone, selector, setGroupList, filterQuestion) {

	if(typeof sId === "undefined") {
		sId = globals.gCurrentSurvey;
	}

	if(typeof filterQuestion === "undefined") {
		filterQuestion = "-1";
	}

	function getAsyncLanguageList(sId, theCallback, selector, filterQuestion) {
		addHourglass();
		$.ajax({
			url: languageListUrl(sId),
			dataType: 'json',
			cache: false,
			success: function(data) {
				removeHourglass();
				if(handleLogout(data)) {
					globals.gSelector.setSurveyLanguages(sId, data);
					retrievedLanguages(sId, selector, data, theCallback, filterQuestion, setGroupList, addNone);
				}
			},
			error: function(xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert(localise.set["c_error"] + ": " + err);
					}
				}
			}
		});
	}

	var data = globals.gSelector.getSurveyLanguages(sId);
	if(data) {
		retrievedLanguages(sId, selector, data, callback, filterQuestion, setGroupList, addNone);
	} else {
		getAsyncLanguageList(sId, callback, selector, filterQuestion);
	}
}

/*
 * Called after languages have been retrieved
 */
function retrievedLanguages(sId, selector, data, theCallback, filterQuestion, setGroupList, addNone) {
	if(selector) {
		setSurveyViewLanguages(data, undefined, selector, addNone);
	} else {
		setSurveyViewLanguages(data, undefined, '#settings_language', false);
		setSurveyViewLanguages(data, undefined, '#export_language', true);
		setSurveyViewLanguages(data, undefined, '#language_name', false);
	}

	if(data[0]) {
		var dateQId = -1;
		if(typeof gTaskStart !== "undefined") {
			dateQId = gTaskStart;
		}
		getQuestionList(sId, data[0], filterQuestion, "-1", theCallback, setGroupList, undefined, dateQId, undefined, undefined, undefined);	// Default language to the first in the list
	} else {
		if(typeof theCallback === "function") {
			theCallback();
		}
	}
}

//Function to get the question list
function getQuestionList(sId, language, qId, groupId, callback, setGroupList, view, dateqId, qName, assignQuestion, scQuestion) {

	function getAsyncQuestionList(sId, language, theCallback, groupId, qId, view, dateqId, qName, assignQuestion, setGroupList, scQuestion) {

		var excludeReadOnly = true;
		if(setGroupList) {
			excludeReadOnly = false;		// Include read only questions in group list
		}
		addHourglass();
		$.ajax({
			url: questionListUrl(sId, language, excludeReadOnly),
			dataType: 'json',
			cache: false,
			success: function(data) {
				removeHourglass();
				if(handleLogout(data)) {
					globals.gSelector.setSurveyQuestions(sId, language, data);
					setSurveyViewQuestions(data, qId, view, dateqId, qName, assignQuestion, scQuestion);

					if (setGroupList && typeof setSurveyViewQuestionGroups === "function") {
						setSurveyViewQuestionGroups(data, groupId);
					}
					if (typeof theCallback === "function") {
						theCallback();
					}
				}
			},
			error: function(xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert("Error: Failed to get list of questions: " + err);
					}
				}
			}
		});
	}

	getAsyncQuestionList(sId, language, callback, groupId, qId, view, dateqId, qName, assignQuestion, setGroupList, scQuestion);
}

//Function to get the meta list
function getMetaList(sId, metaItem) {

	addHourglass();
	$.ajax({
		url: "/surveyKPI/metaList/" + sId,
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				globals.gSelector.setSurveyMeta(sId, data);
				setSurveyViewMeta(data, metaItem);
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert(localise.set["c_error"] + ": " + err);
				}
			}
		}
	});
}

/*
 * Function to get the list of notification alerts
 * These are extracted from the settings for the survey
 */
function getAlertList(sId, alertId) {

	addHourglass();
	$.ajax({
		url: "/surveyKPI/cases/settings/" + sId,
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				globals.gSelector.setSurveyAlerts(sId, data);
				setSurveyAlerts(data, alertId);
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert(localise.set["c_error"] + ": " + err);
				}
			}
		}
	});
}


//Set the language list in the survey view control
function setSurveyViewLanguages(list, language,elem, addNone) {

	var $languageSelect = $(elem),
		i;

	$languageSelect.empty();
	if(addNone) {
		$languageSelect.append('<option value="none">' + localise.set["c_none"] + '</option>');
	}

	for(i = 0; i < list.length; i++) {
		$languageSelect.append('<option value="' + list[i] + '">' + htmlEncode(list[i]) + '</option>');
	}

	if(language) {
		$languageSelect.val(language);
	}
}

// Set the question list in the survey view control
function setSurveyViewQuestions(list, qId, view, dateqId, qName, assignQuestion, scQuestion) {

	var $questionSelect = $('.selected_question'),
		$dateQuestions = $('.date_questions'),
		$scQuestions = $('#sc_question'),
		$questionNameSelect = $('.selected_name_question'),     // this should replace selected_question
		$assignQuestion = $('#assign_question'),
		label;

	$questionSelect.empty();
	$questionSelect.append('<option value="-1">' + localise.set["c_none"] + '</option>');

	$questionNameSelect.empty();
	$questionNameSelect.append('<option value="-1">' + localise.set["c_none"] + '</option>');

	$dateQuestions.empty();
	$dateQuestions.append('<option value="-1">' + localise.set["ed_i_c"] + '</option>');

	$scQuestions.empty();

	if(list) {
		$.each(list, function(j, item) {
			if(typeof item.q === "undefined") {
				label = "";
			} else {
				label = item.q;
			}

			$questionSelect.append('<option value="' + item.id + '">' + htmlEncode(item.name + " : " + label) + '</option>');
			$questionNameSelect.append('<option value="' + item.name + '">' + htmlEncode(item.name) + '</option>');
			if(item.type === 'timestamp' || item.type === 'dateTime' || item.type == 'date') {
				$dateQuestions.append('<option value="' + item.id + '">' + htmlEncode(item.name + " : " + label) + '</option>');
			}
			if(item.type === 'server_calculate') {
				let name = htmlEncode(item.name);
				$scQuestions.append(`<option value="${item.name}">${name}</option>`);
			}
		});
	}
	if(!qId) {
		qId = "-1";
	}
	$questionSelect.val(qId);

	if(!qName) {
		qName = "-1";
	}
	$questionNameSelect.val(qName);
	$assignQuestion.val(assignQuestion);

	if(!dateqId) {
		dateqId = "-1";
	}
	$dateQuestions.val(dateqId);

	// Server calculate question
	if(scQuestion) {
		$scQuestions.val(scQuestion);
	}

	if(view) {
		setFilterFromView(view);	// Set the filter dialog settings
	}

}

// Set the meta list in the survey view control
function setSurveyViewMeta(list, metaItem) {

	var $metaSelect = $('.selected_meta'),
		item,
		i;

	$metaSelect.empty();

	// Add none
	$metaSelect.append('<option value="-1">' + localise.set["c_none"] + '</option>');

	// Add the user who submitted the survey
	$metaSelect.append('<option value="_user">' + localise.set["c_submitter"] + '</option>');

	if(list) {
		for(i = 0; i < list.length; i++) {
			item = list[i];
			$metaSelect.append('<option value="' + item.name + '">' + htmlEncode(item.name) + '</option>');
		}
	}
	if(!metaItem) {
		metaItem = "-1";
	}
	$metaSelect.val(metaItem);

}

/*
 * Populate the alert list
 */
function setSurveyAlerts(settings, alertId) {

	var $elem = $('.alert_list'),
		item,
		i;

	$elem.empty();

	if(settings && settings.alerts) {
		for(i = 0; i < settings.alerts.length; i++) {
			item = settings.alerts[i];
			$elem.append('<option value="' + item.id + '">' + htmlEncode(item.name) + '</option>');
		}
	}
	if(alertId) {
		$elem.val(alertId);
	}


}

/*
 * ------------------------------------------------------------
 * Web service Functions
 */
function languageListUrl (sId) {

	var url = "/surveyKPI/languages/";
	url += sId;
	return url;
}

/*
 * Web service handler for retrieving available "count" questions for graph
 *  @param {string} survey
 */
function questionListUrl (sId, language, exc_read_only) {

	var url = "/surveyKPI/questionList/",
		ro_text;

	if(exc_read_only) {
		ro_text = "true";
	} else {
		ro_text = "false";
	}

	url += sId;
	url += "/" + encodeURIComponent(language);
	url += "?exc_read_only=" + ro_text;
	return url;
}

/**
 * Web service handler for question Meta Data
 * @param {string} survey id
 * @param {string} question id
 */
function questionMetaURL (sId, lang, qId) {

	var url = "/surveyKPI/question/";
	url += sId;
	url += "/" + lang;
	url += "/" + qId;
	url += "/getMeta";
	return url;
}

/*
 * Get a survey details - depends on globals being set
 */
function getSurveyDetails(callback, get_changes, hide_soft_deleted) {

	var tz = globals.gTimezone;
	var url="/surveyKPI/surveys/" + globals.gCurrentSurvey;
	if(get_changes) {
		url += "?get_changes=true";
		url += "&tz=" + encodeURIComponent(tz);
	} else {
		url += "?tz=" + encodeURIComponent(tz);
	}
	if(hide_soft_deleted) {
		url += "&get_soft_delete=false";
	}

	if(!globals.gCurrentSurvey) {
		alert("Error: Can't get survey details, Survey identifier not specified");
	} else {
		addHourglass();
		$.ajax({
			url: url,
			dataType: 'json',
			cache: false,
			success: function(data) {
				removeHourglass();
				if(handleLogout(data)) {
					globals.model.setSurveyData(data);
					globals.model.setSettings();
					setLanguages(data.languages, callback);

					if (typeof callback == "function") {
						callback();
					}
				}
			},
			error: function(xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						if (xhr.status == 404) {
							// The current survey has probably been deleted or the user no longer has access
							globals.gCurrentSurvey = undefined;
							return;
						}
						alert("Error: Failed to get survey: " + err);
					}
				}
			}
		});
	}
}

/*
 * Set the languages for the editor
 */
function setLanguages(languages, languageCallback) {

	var h = [],
		h2 = [],
		idx = -1,
		idx2 = -1,
		$lang_menu = $('.language_menu_list'),
		$lang = $('.language_list'),
		$lang1 = $('#language1'),
		$lang2 = $('#language2'),
		i;

	globals.gLanguage1 = 0;	// Language indexes used for translations
	globals.gLanguage2 = 0;
	if(languages.length > 1) {
		globals.gLanguage2 = 1;
	}

	for (i = 0; i < languages.length; i++) {
		h[++idx] = '<a data-lang="';
		h[++idx] = i;
		h[++idx] = '" class="dropdown-item" href="javascript:void(0)">';
		h[++idx] = htmlEncode(languages[i].name);
		h[++idx] = '</a>';

		h2[++idx2] = '<option value="';
		h2[++idx2] = i;
		h2[++idx2] = '">';
		h2[++idx2] = htmlEncode(languages[i].name);
		h2[++idx2] = '</option>';
	}

	$lang_menu.empty().append(h.join(""));
	$lang.empty().append(h2.join(""));

	$('#langSelected').text(languages[ globals.gLanguage].name);
	$('.language_menu_list a').click(function() {
		globals.gLanguage = $(this).data("lang");
		$('#langSelected').text(languages[ globals.gLanguage].name);
		languageCallback();
	});

	$lang1.val(globals.gLanguage1);
	$lang2.val(globals.gLanguage2)
}

/*
 * Get a survey details - depends on globals being set
 */
function createNewSurvey(name, existing, existing_survey, shared_results, callback) {

	console.log("create new: " + existing + " : " + existing_survey + " : " + shared_results);

	var url="/surveyKPI/surveys/new/" + globals.gCurrentProject + "/" + encodeURIComponent(name);
	if(!existing) {
		existing_survey = 0;
	}

	addHourglass();
	$.ajax({
		type: "POST",
		url: url,
		cache: false,
		dataType: 'json',
		data: {
			existing: existing,
			existing_survey: existing_survey,
			existing_form: 0,
			shared_results: shared_results
		},
		success: function(data) {
			removeHourglass();

			if(handleLogout(data)) {
				globals.model.setSurveyData(data);
				globals.model.setSettings();
				globals.gCurrentSurvey = data.id;

				saveCurrentProject(-1, globals.gCurrentSurvey, undefined);	// Save the current survey id

				setLanguages(data.languages, callback);

				if (typeof callback == "function") {
					callback();
				}
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					bootbox.alert(localise.set["c_error"] + " " + htmlEncode(xhr.responseText));
				}
			}
		}
	});
}

/*
 * Open a form for editing
 */
function openForm(type) {

	$('.reusing_form').hide();
	$('#base_on_existing').prop('checked', false);
	$('#shared_results').prop('checked', false);
	$('#new_form_name').val("");
	if(type === "new") {
		$('.existing_form').hide();
		$('.new_form').show();
		$('#openSurveyLabel').html(localise.set["tm_g_new"]);
		$('#get_form').html(localise.set["c_create"]);
		globals.gExistingSurvey = false;
	} else {
		$('.existing_form').show();
		$('.new_form').hide();
		$('#openSurveyLabel').html(localise.set["tm_g_open"]);
		$('#get_form').html(localise.set["m_open"]);
		globals.gExistingSurvey = true;
	}
	$('#openFormModal').modal('show');

}

/*
 * If this is a smap server return the subdomain
 */
function getServerSubDomainName() {

	var hostname = location.hostname;
	var sd = "";

	if(hostname.indexOf('.smap.com.au') > 0) {
		sd = hostname.substring(0, hostname.indexOf('.smap.com.au'));
	} else if(hostname === 'localhost') {
		sd = 'localhost';
	}

	return sd;
}

/*
 * Return true if this is a business server
 */
function isBusinessServer() {

	var hostname = location.hostname;
	var bs = true;

	if(hostname.indexOf('smap.com.au') > 0) {
		bs = false;
	}
	if(hostname.indexOf('sg.smap.com.au') >= 0 ||
		hostname.indexOf('ubuntu1804.smap.com.au') >= 0 ||
		hostname.indexOf('demo.smap.com.au') >= 0) {
		bs = true;
	}

	return bs;
}

/*
 * Returns the class of server that has custom menus
 */
function getCustomMenuClass() {

	var hostname = location.hostname;
	var classname = undefined;

	if(hostname.indexOf('cuso.smap.com.au') >= 0) {
		classname = '.xxxx1';
	} else if(hostname.indexOf('demo.smap.com.au') >= 0) {
		classname = '.xxxx1';
	} else {
		if(hostname === 'localhost') {
			classname = '.xxxx1';   // testing
		}
	}

	return classname;
}


/*
 * Return true if this is a self registration server
 */
function isSelfRegistrationServer() {
	var hostname = location.hostname;
	var sr = true;

	if(hostname !== 'localhost' &&
		hostname !== 'sg.smap.com.au' &&
		hostname.indexOf('reachnettechnologies.com') < 0 &&
		hostname.indexOf('.icanreach.com') < 0 &&
		hostname.indexOf('encontactone.com') < 0 &&
		hostname !== 'app.kontrolid.com' &&
		hostname !== 'kontrolid.smap.com.au') {
		sr = false;
	}
	return sr;
}

/*
 * Validate start and end dates
 */
function validDates() {
	var $d1 = $('#startDate'),
		$d2 = $('#endDate'),
		d1 = $d1.data("DateTimePicker").date(),
		d2 = $d2.data("DateTimePicker").date()

	if(!d1 || !d1.isValid()) {
		$('#ut_alert').show().text(localise.set["t_i_sd"]);
		setTimeout(function() {
			$('.form-control', '#startDate').focus();
		}, 0);
		return false;
	}

	if(!d2 || !d2.isValid()) {
		$('#ut_alert').show().text(localise.set["t_i_ed"]);
		setTimeout(function() {
			$('.form-control', '#endDate').focus();
		}, 0);
		return false;
	}

	if(d1 > d2) {
		$('#ut_alert').show().text("End date must be greater than or the same as the start date");
		setTimeout(function() {
			$('.form-control', '#startDate').focus();
		}, 0);
		return false;
	}

	$('#ut_alert').hide();
	return true;
}

/*
 * Convert a date into UTC
 */
function getUtcDate($element, start, end) {

	var theDate,
		utcDate;

	if(start) {
		theDate = $element.data("DateTimePicker").date().startOf('day');
	} else if (end) {
		theDate = $element.data("DateTimePicker").date().endOf('day');
	} else {
		theDate = $element.data("DateTimePicker").date();
	}

	utcDate = moment.utc(theDate);

	console.log("date:" + theDate.format("YYYY-MM-DD HH:mm:ss"));
	console.log("UTC:" + utcDate.format("YYYY-MM-DD HH:mm:ss"));

	return utcDate.valueOf();

}

/*
 * Get a description from a change made in the editor
 */
function getChangeDescription(change, version) {

	var h =[],
		idx = -1,
		oldVal,
		newVal,
		forms = globals.model.survey.forms,
		str;

	if(change.action === "external option") {
		/*
		 * Options added from a file
		 */
		h[++idx] = 'Choice <span style="color:blue;">';
		h[++idx] = htmlEncode(change.option.externalLabel);
		h[++idx] = '</span>';
		h[++idx] = ' from file: <span style="color:blue;">';
		h[++idx] = htmlEncode(change.fileName);
		h[++idx] = '</span>';

	} else if(change.action === "template_update") {
		h[++idx] = localise.set["ed_c_template"];
		h[++idx] = ' <span style="color:blue;">';
		h[++idx] = htmlEncode(change.msg);
		h[++idx] = '</span>';

	} else if(change.action === "template_add") {
		h[++idx] = localise.set["ed_a_template"];
		h[++idx] = ' <span style="color:blue;">';
		h[++idx] = htmlEncode(change.msg);
		h[++idx] = '</span>';

	} else if(change.action === "template_delete") {
		h[++idx] = ' <span style="color:red;">';
		h[++idx] = localise.set["ed_d_template"];
		h[++idx] = htmlEncode(change.msg);
		h[++idx] = '</span>';

	} else if(change.action === "settings_update") {
		h[++idx] = localise.set["ed_c_settings"];
		h[++idx] = ' <span style="color:blue;">';
		h[++idx] = htmlEncode(change.msg);
		h[++idx] = '</span>';

	} else if(change.action === "language_update") {
		h[++idx] = localise.set["ed_c_languages"];
		h[++idx] = ' <span style="color:blue;">';
		h[++idx] = htmlEncode(change.msg);
		h[++idx] = '</span>';

	} else if(change.action === "add_preload") {
		h[++idx] = ' <span style="color:blue;">';
		h[++idx] = htmlEncode(change.msg);
		h[++idx] = '</span>';

	} else if(change.action === "del_preload") {
		h[++idx] = ' <span style="color:red;">';
		h[++idx] = htmlEncode(change.msg);
		h[++idx] = '</span>';

	} else if(change.action === "update") {

		/*
		 * Updates to questions and options and list names
		 */
		if(change.property.prop === "type") {
			newVal = htmlEncode(translateType(change.property.newVal));
			oldVal = htmlEncode(translateType(change.property.oldVal));
		} else {
			newVal = htmlEncode(change.property.newVal);
			oldVal = htmlEncode(change.property.oldVal);
		}


		if(change.property.prop === "name") {

			// Deprecate the following when the structure of these log objects is made consistent
			if(typeof change.property.type === "optionList" || change.property.type === "unknown") {
				change.type = "choice list ";
			}

			h[++idx] = change.property.type;
			h[++idx] = ' ';
			h[++idx] = localise.set["msg_ren"],
				h[++idx] = ': <span style="color:blue;">';
			h[++idx] = newVal;		// Already encoded
			h[++idx] = '</span>';
			h[++idx] = ' from: <span style="color:red;">';
			h[++idx] = oldVal;	// Already encoded
			h[++idx] = '</span>';
		} else {
			str = localise.set["ed_c_chg_p"];
			if(change.property.propType === "constraint_msg" || change.property.propType === "required_msg" || change.property.propType === "guidance_hint") {
				str = str.replace("%s1", '"' + htmlEncode(change.property.propType) + '"');
			} else {
				str = str.replace("%s1", '"' + htmlEncode(change.property.prop) + '"');
			}
			str = str.replace("%s2", htmlEncode(change.property.name));
			str = str.replace("%s3", '<span style="color:blue;">' + newVal + '</span>');	// Already encoded
			str = str.replace("%s4", '<span style="color:red;">' + oldVal + '</span>');		// Already encoded
			h[++idx] = str;
		}

	} else if(change.action === "add")  {

		/*
		 * New questions or options
		 */
		if(change.type === "question" || change.changeType === "question"){  // deprecate checking of changeType

			str = localise.set["ed_c_add_q"];
			str = str.replace("%s1", '<span style="color:blue;">' + htmlEncode(change.question.name) + "</span>");
			var typeString;
			if(change.question.type === "string") {
				typeString = 'text';
			} else if(change.question.type === "select"){
				typeString = 'select_multiple';
			} else if(change.question.type === "select1"){
				typeString = 'select_one';
			} else {
				typeString = change.question.type;
			}
			str = str.replace("%s2", '<span style="color:red;">' + htmlEncode(typeString) + "</span>");
			h[++idx] = str;

		} else if(change.type === "option" || change.changeType === "option") {	// deprecate checking of changeType
			/*
			 * Options added or deleted from the editor
			 */
			str = localise.set["ed_c_add_o"];
			var valueStr = '<span style="color:blue;">' + change.option.value;
			if(change.option.labels && change.option.labels.length >= 1) {
				valueStr += ' (';
				valueStr += htmlEncode(change.option.labels[0].text);
				valueStr += ')';
			}
			valueStr += '</span>';
			str = str.replace("%s1", valueStr);
			str = str.replace("%s2", '<span style="color:blue;">' + htmlEncode(change.option.optionList) + '</span>');
			h[++idx] = str;
		}

	}  else if(change.action === "move")  {

		/*
		 * New questions or options
		 */
		h[++idx] = localise.set['c_moved'] + ' ';

		if(change.type === "question" || change.changeType === "question") {  // deprecate checking of changeType){

			h[++idx] = 'question <span style="color:blue;">';
			h[++idx] = htmlEncode(change.question.name);
			if(change.question.sourceSeq >= 0) {
				h[++idx] = '</span> from position <span style="color:red;">';
				h[++idx] = htmlEncode(change.question.sourceSeq);
				h[++idx] = '</span> in form ';
				h[++idx] = htmlEncode(forms[change.question.sourceFormIndex].name);
			} else {
				h[++idx] = '</span> from form ';
				h[++idx] = htmlEncode(forms[change.question.sourceFormIndex].name);
			}
			h[++idx] = '</span> to position <span style="color:red;">';
			h[++idx] = htmlEncode(change.question.seq);
			h[++idx] = '</span>';
			h[++idx] = ' in form ';
			if(change.question.formIndex < forms.length) {	// Allow for a form being deleted
				h[++idx] = htmlEncode(forms[change.question.formIndex].name);
			}


		} else if(change.type === "option") {

			h[++idx] = 'choice <span style="color:blue;">';
			h[++idx] = htmlEncode(change.option.value);
			if(change.option.labels && change.option.labels.length >= 1) {
				h[++idx] = ' (';
				h[++idx] = htmlEncode(change.option.labels[0].text);
				h[++idx] = ')';
			}
			h[++idx] = '</span>';
			h[++idx] = ' from choice list: <span style="color:blue;">';
			h[++idx] = htmlEncode(change.option.sourceOptionList);
			h[++idx] = '</span>';
			h[++idx] = ' to choice list: <span style="color:blue;">';
			h[++idx] = htmlEncode(change.option.optionList);
			h[++idx] = '</span>';
		}

	} else if(change.action === "delete")  {

		if(change.type === "question" || change.changeType === "question"){

			h[++idx] = localise.set["ed_c_del_q"];

			h[++idx] = ' <span style="color:blue;">';
			h[++idx] = htmlEncode(change.question.name);
			h[++idx] = '</span>';

		} else if(change.type === "option") {

			str = localise.set["ed_c_del_o"];
			var valueStr = '<span style="color:blue;">' + htmlEncode(change.option.value);
			if(change.option.labels && change.option.labels.length >= 1) {
				valueStr  += ' (';
				valueStr  += htmlEncode(change.option.labels[0].text);
				valueStr  += ')';
			}
			valueStr  += '</span>';
			str = str.replace("%s1", valueStr);
			str = str.replace("%s2", '<span style="color:blue;">' + htmlEncode(change.option.optionList) + '</span>');
			h[++idx] = str;
		}
	} else if(change.action === "set_required")  {
		if(change.msg.indexOf('not') < 0) {
			h[++idx] = localise.set["ed_c_sr"];
		} else {
			h[++idx] = localise.set["ed_c_snr"];
		}

	} else if(change.action === "upload_template")  {

		if(version > 1) {
			h[++idx] = localise.set["msg_survey_replaced"];
		} else {
			h[++idx] = localise.set["msg_survey_loaded"];
		}

	} else if(change.action === "role")  {

			h[++idx] = change.msg;

	} else {
		h[++idx] = htmlEncode(change.type);
		h[++idx] = ' ';
		h[++idx] = htmlEncode(change.name);
		h[++idx] = ' changed to: <span style="color:blue;">';
		h[++idx] = htmlEncode(change.newVal);
		h[++idx] = '</span>';
		h[++idx] = ' from: <span style="color:red;">';
		h[++idx] = htmlEncode(change.oldVal);
		h[++idx] = '</span>';
	}

	return h.join('');
}

// Translate types for use in change description
function translateType(input) {
	if(input === "string") {
		output = "text";
	} else {
		output = input;
	}
	return output;
}

/*
 * Get the shared locations from the server
 */
function getLocations(callback) {

	var url="/surveyKPI/tasks/locations";

	addHourglass();
	$.ajax({
		url: url,
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				if (typeof callback === "function") {
					callback(data);
				}
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					console.log("Error: Failed to get list of locations: " + err);
				}
			}
		}
	});

}

/*
 * update Location group list
 */
function refreshLocationGroups(tags, includeAll, currentGroup) {

	var g = undefined,
		h = [],
		idx = -1,
		i;

	var includeNfc = $('#includeNfc').prop('checked'),
		includeGeo = $('#includeGeo').prop('checked');

	if(tags) {
		for(i = 0; i < tags.length; i++) {
			if(includeAll || includeLocation(includeNfc, includeGeo, tags[i].uid, tags[i].lat, tags[i].lon)) {

				if (g != tags[i].group) {

					g = tags[i].group;
					if (typeof currentGroup === "undefined") {
						currentGroup = g;
					}

					if(includeAll) {
						if (currentGroup === g) {
							$('.location_group_list_sel').text(g);
						}
						h[++idx] = '<a class="dropdown-item" href="#">';
						h[++idx] = g;
						h[++idx] = '</a>';
					} else {
						h[++idx] = '<option';
						if (currentGroup === g) {
							h[++idx] = ' selected';
						}
						h[++idx] = ' value="';
						h[++idx] = g;
						h[++idx] = '">';
						h[++idx] = htmlEncode(g);
						h[++idx] = '</option>';
					}
				}
			}
		}
	}

	$('.location_group_list').empty().html(h.join(""));
	return currentGroup;
}

/*
 * Add the locations (NFC tags or geofence) to any drop down lists that use them
 */
function setLocationList(locns, current, currentGroup) {

	var h = [],
		idx = -1,
		i;

	if(locns && locns.length) {
		h[++idx] = '<option value="-1">';
		h[++idx] = localise.set["c_none"];
		h[++idx] = '</option>';
		for(i = 0; i < locns.length; i++) {
			if(locns[i].group === currentGroup) {
				h[++idx] = '<option value="';
				h[++idx] = i;
				h[++idx] = '">';
				h[++idx] = htmlEncode(locns[i].name);
				h[++idx] = '</option>';
			}
		}
	}

	$('.location_select').empty().append(h.join(""));
	$('.location_select').val(current);


}

/*
 * Test for whether or not a location should be shown in the resource page
 */
function includeLocation(includeNfc, includeGeo, uid, lat, lon) {
	var include = false;

	if(includeNfc && typeof uid !== 'undefined' && uid !== '') {
		include = true;
	}
	if(!include && includeGeo && lat != 0 && lon != 0) {
		include = true;
	}

	return include;
}

/*
 * Convert a timestamp in UTC to local time and return a date object
 */
function localTimeAsDate(utcTime) {
	var utcDate,
		localTime;

	if(utcTime) {
		if(utcTime.indexOf('+') > 0) {
			utcDate  = moment.utc(utcTime, 'YYYY-MM-DD HH:mm:ss Z').toDate();
		} else {
			utcDate  = moment.utc(utcTime, 'YYYY-MM-DD HH:mm:ss').toDate();
		}
		localTime = moment(utcDate);
	}
	return localTime;
}

/*
 * Convert a timestamp in UTC to local time
 */
function localTime(utcTime) {
	var utcDate,
		localTime;

	if(utcTime) {
		if(utcTime.indexOf('+') > 0) {
			utcDate  = moment.utc(utcTime, 'YYYY-MM-DD HH:mm:ss Z').toDate();
		} else {
			utcDate  = moment.utc(utcTime, 'YYYY-MM-DD HH:mm:ss').toDate();
		}
		localTime = moment(utcDate).format('YYYY-MM-DD HH:mm:ss');
	}
	return localTime;
}


function utcTime(localTime) {

	var utcTime,
		localDate;

	if(localTime) {
		localDate = moment(localTime).toDate();
		utcTime =  moment.utc(localDate).format('YYYY-MM-DD HH:mm:ss');
	}
	return utcTime;

}

function isLate(finish) {

	var late = false,
		current = new Date(),
		finishDate,
		localFinish;

	if(finish) {
		localFinish = localTime(finish);
		finishDate = new Date(localFinish);
		if(current > finishDate) {
			late = true;
		}
	}
	return late;

}

function downloadPdf(language, orientation, include_references, launched_only, sIdent, instanceId, pdfTemplateId) {

	var docURL = "/surveyKPI/pdf/" + sIdent
		+ "?language=" + language
		+ "&instance=" + instanceId
		+ "&pdftemplate=" + pdfTemplateId
		+ "&tz=" + globals.gTimezone;
	if(orientation === "landscape") {
		docURL += "&landscape=true";
	}
	if(include_references) {
		docURL += "&reference_surveys=true";
	}
	if(launched_only) {
		docURL += "&launched_only=true";
	}

	downloadFile(docURL);
}

function downloadFile(url) {

	url += addCacheBuster(url);
	$("body").append("<iframe src='" + url + "' style='display: none;' ></iframe>");
	// Check for errors allow 5 seconds for an error to be returned
	setTimeout(downloadFileErrorCheck, 5000);
}

// Show an error generated by file download
function downloadFileErrorCheck() {
	var msg = $("iframe").last().contents().find('body').html();
	if(handleLogout(msg)) {
		if (msg && msg.indexOf("Error:") === 0) {
			alert(msg.substring(7));	// Jump over "Error: "
		} else if (msg && msg.length > 0) {
			alert(msg);
		}
	}
}

/*
 * Post data to be converted into a file
 */
function generateFile(url, filename, format, mime, data, sId, groupSurvey, title, project, charts, chartData, settings, tz, form) {

	var fd = new FormData();
	fd.append("sId", sId);
	fd.append("format", format);
	if(groupSurvey) {
		fd.append("groupSurvey", groupSurvey)
	}
	if(form) {
		fd.append("form", form);
	}
	if(data) {
		var blob = new Blob([JSON.stringify(data)], { type: 'text/plain' });
		var file = new File([blob], "foo.txt", {type: "text/plain"});
		fd.append("data", file);
		//fd.append("data", JSON.stringify(data));
	}
	if(title) {
		fd.append("title", title);
	}
	if(project) {
		fd.append("project", project);
	}
	if(charts) {
		fd.append("charts", JSON.stringify(charts));
	}
	if(chartData) {
		fd.append("chartData", JSON.stringify(chartData));
	}
	if(settings) {
		fd.append("settings", JSON.stringify(settings));
	}
	if(tz) {
		fd.append("tz",JSON.stringify(tz));
	}

	var xhr = new XMLHttpRequest();
	url += addCacheBuster(url);
	xhr.open('POST', url, true);
	xhr.setRequestHeader("X-Requested-With", "XMLHttpRequest");
	xhr.responseType = 'blob';

	xhr.onload = function(e) {
		if(handleLogout(xhr.responseURL)) {
			if (this.status == 200) {
				// get binary data as a response
				var blob = new Blob([this.response], {type: mime});
				var downloadUrl = URL.createObjectURL(blob);
				var a = document.createElement("a");
				a.href = downloadUrl;
				a.download = filename;
				document.body.appendChild(a);
				a.click();
				setTimeout(function () {
					document.body.removeChild(a);
					window.URL.revokeObjectURL(url);
				}, 100);
			} else {
				alert(localise.set["c_error"] + ": " + this.statusText);
			}
		}
	};

	xhr.onerror = function(e) {
		if(handleLogout(this)) {
			alert("Error: Upload Failed");
		}
	}

	xhr.send(fd);

}

/*
 * Get the currently selected rows of datatable data as a json array
 * Also convert the JSON object into an array of Key values pairs. This allows easy converion
 * to a java object on the server
 */
function getTableData(table, columns, format) {

	var rows = table.rows({
		order:  'current',  // 'current', 'applied', 'index',  'original'
		page:   'all',      // 'all',     'current'
		search: 'applied',     // 'none',    'applied', 'removed'
	}).data();

	var data = [],
		cols = [],
		i, j;

	for(i = 0; i < rows.length; i++) {
		cols = [];
		for(j = 0; j < columns.length; j++) {
			if(format === "xlsx" || !columns[j].hide) {
				var k = columns[j].displayName;
				var v = rows[i][columns[j].column_name];

				if (typeof v !== "string") {
					v = JSON.stringify(v);
				}
				cols.push({
					k: k,
					v: v
				})
			}
		}
		data.push(cols);
	}

	return data;


}

/*
 * Get server settings
 */
function getMapboxDefault(callback, param) {

	if(!globals.gMapboxDefault) {
		addHourglass();
		$.ajax({
			url: '/surveyKPI/server/mapbox',
			cache: false,
			success: function(data) {
				removeHourglass();
				if(handleLogout(data)) {
					globals.gMapboxDefault = data;
					if (typeof callback === "function") {
						callback(param);
					}
				}
			},
			error: function(xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert(localise.set["error"] + ": " + err);
					}
				}
			}
		});
	} else {
		if(typeof callback === "function") {
			callback(param);
		}
	}
}


/*
 * Get google map api key
 */
function getGoogleMapApi(callback, map) {

	console.log("getGoogleMapApi");

	if(!window.smapLoadedGMaps && !window.smapGMapsLoading) {
		console.log("about to call server");

		window.smapGMapsLoading = true;

		window.smapGMapsToLoad = [];
		window.smapGMapsToLoad.push({
			fn: callback,
			locn: map
		});

		addHourglass();
		$.ajax({
			url: '/surveyKPI/server/googlemaps',
			cache: false,
			success: function(data) {
				removeHourglass();
				if(handleLogout(data)) {
					console.log("Retrieved map keys from server");

					var gElement = document.createElement('script');
					var key = "";
					if (data) {
						key = "?key=" + data;
					}
					//gElement.src = "//maps.google.com/maps/api/js?v=3.6&amp";
					gElement.src = "https://maps.googleapis.com/maps/api/js" + key;
					if (typeof callback === "function") {
						gElement.onload = onLoad;
					}
					document.getElementsByTagName('head')[0].appendChild(gElement);

					function onLoad() {

						var i;

						window.smapGMapsLoading = false;
						window.smapLoadedGMaps = true;

						console.log("Google map loaded");

						for (i = 0; i < window.smapGMapsToLoad.length; i++) {
							console.log("map callback");
							window.smapGMapsToLoad[i].fn(window.smapGMapsToLoad[i].locn);
						}
						delete window.smapGMapsToLoad;
					}
				}

			},
			error: function(xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert(localise.set["error"] + " " + err);
					}
				}
			}
		});

	} else if(window.smapLoadedGMaps) {
		console.log("Already loaded calling map callback");
		callback(map);
	} else {
		console.log("Adding callback to queue");
		window.smapGMapsToLoad.push({
			fn: callback,
			locn: map
		});
	}
}

/*
 * Add google layers to a map
 */
function addGoogleMapLayers(map) {
	try {
		map.addLayer(new OpenLayers.Layer.Google("Google Satellite",{type: google.maps.MapTypeId.SATELLITE, 'sphericalMercator': true, numZoomLevels: 22}));
		map.addLayer(new OpenLayers.Layer.Google("Google Maps",{type: google.maps.MapTypeId.ROADMAP, 'sphericalMercator': true, numZoomLevels: 22}));
		map.addLayer(new OpenLayers.Layer.Google("Google Hybrid",{type: google.maps.MapTypeId.HYBRID, 'sphericalMercator': true, numZoomLevels: 22}));
	} catch (err) {
		// Fail silently, the user may not want google maps - this is probably caused by a missing maps api key
	}
}

/*
 * Get a list of custom reports
 */
function getReports(callback1, callback2, type) {

	var url="/surveyKPI/custom_reports";

	if(type) {
		url += "?type=" + type;
	}

	addHourglass();
	$.ajax({
		url: url,
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				var cb1 = callback1,
					cb2 = callback2,
					t = type;
				globals.gReports = data;
				if (typeof cb1 === "function") {
					cb1(data, cb1, cb2, t);
				}
				if (typeof cb2 === "function") {
					cb2(data, cb1, cb2, t);
				}
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					console.log("Error: Failed to get list of reports: " + err);
				}
			}
		}
	});

}

/*
 * Allow the user to pick a report
 */
function showReportList(data) {
	var h = [],
		idx = -1,
		i;

	removeHourglass();

	if(data.length === 0) {

		// Enable / disable elements specifically for managed forms
		$('.selectmanaged').show();
		$('.no_oversight').show();
	} else {
		$('.no_oversight').hide();
		$('.selectmanaged').show();

		h[++idx] = '<option value="0">';
		h[++idx] = localise.set["c_none"];
		h[++idx] = '</option>';
		for(i = 0; i < data.length; i++) {
			h[++idx] = '<option value="';
			h[++idx] = data[i].id;
			h[++idx] = '">';
			h[++idx] = htmlEncode(data[i].name);
			h[++idx] = '</option>';
		}
		$('.customReportList').empty().html(h.join(''));
	}
}

/*
 * Show the Custom Reports in a table
 */
function refreshCustomReportView(data, callback1, callback2, type) {

	var $selector = $('#cr_list'),
		i,
		h = [],
		idx = -1;

	$('.panel_msg').show();
	$('#addReportPopup').modal("hide");

	data = data || [];
	globals.gReports = data;

	h[++idx] = '<table class="table">';
	h[++idx] = '<thead>';
	h[++idx] = '<tr>';
	h[++idx] = '<th>' + localise.set["c_name"], + '</th>';
	h[++idx] = '<th>' + localise.set["c_type"] + '</th>';
	h[++idx] = '</tr>';
	h[++idx] = '</thead>';
	h[++idx] = '<tbody class="table-striped">';

	for(i = 0; i < data.length; i++) {

		h[++idx] = '<tr>';

		// name
		h[++idx] = '<td>';
		h[++idx] = htmlEncode(data[i].name);
		h[++idx] = '</td>';

		// type
		h[++idx] = '<td>';
		h[++idx] = htmlEncode(data[i].type);
		h[++idx] = '</td>';

		// actions
		h[++idx] = '<td>';

		h[++idx] = '<button type="button" data-idx="';
		h[++idx] = i;
		h[++idx] = '" class="btn btn-default btn-sm rm_cr">';
		h[++idx] = '<i class="fa fa-trash-o"></i></button>';

		h[++idx] = '<button type="button" data-idx="';
		h[++idx] = i;
		h[++idx] = '" class="btn btn-default btn-sm download_cr">';
		h[++idx] = '<i class="fa fa-download"></i></button>';

		h[++idx] = '</td>';
		// end actions

		h[++idx] = '</tr>';
	}

	h[++idx] = '</tbody>';
	h[++idx] = '</table>';

	$selector.empty().append(h.join(''));

	$(".rm_cr", $selector).click(function(){
		var idx = $(this).data("idx");
		if(confirm(localise.set["msg_confirm_del"] + " " + htmlEncode(globals.gReports[idx].name))) {
			deleteCustomReport(globals.gReports[idx].id, type);
		}
	});

	$(".download_cr", $selector).click(function(){
		var idx = $(this).data("idx");
		downloadFile("/surveyKPI/custom_reports/xls/" + globals.gReports[idx].id +
			"?filetype=xls&filename=" + cleanFileName(globals.gReports[idx].name));
	});


}

function deleteCustomReport(id, type) {

	var url = "/surveyKPI/custom_reports/" + id;
	if(type) {
		url += "?type=" + type;
	}

	addHourglass();
	$.ajax({
		type: "DELETE",
		url: url,
		success: function(data, status) {
			removeHourglass();
			if(handleLogout(data)) {
				var t = type;
				console.log("delete: " + t + " : " + type);
				getReports(refreshCustomReportView, showReportList, t);
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert(localise.set["msg_err_del"] + " " + xhr.responseText);	// alerts htmlencode text
				}
			}
		}
	});
}

/*
 * Get the list of available roles from the server
 */
function getRoles(callback) {
	addHourglass();
	$.ajax({
		url: "/surveyKPI/role/roles",
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				globals.gRoleList = data;
				if (typeof callback === "function") {
					callback();
				}
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert(localise.set["msg_err_get_r"] + " " + err);
				}
			}
		}
	});
}

/*
 * Get the list of available case management settings from the server
 */
function getCms(callback) {
	addHourglass();
	$.ajax({
		url: "/surveyKPI/cases/settings/" + globals.gCurrentSurvey,
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				globals.gCmSettings = data;
				if (typeof callback === "function") {
					callback();
				}
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert(localise.set["msg_err_get_r"] + " " + err);
				}
			}
		}
	});
}

/*
 * Clean the filename so that it can be passed in a URL
 */
function cleanFileName(filename) {

	var n;

	n = filename.replace(/\//g, '_');	// remove slashes from the filename
	n = n.replace(/[#?&]/g, '_');		// Remove other characters that are not wanted
	n = n.replace("'", "", 'g');		// Remove apostrophes

	return n;
}

/*
 * Add a list of forms to pick from during export
 */
function addFormPickList(sMeta, checked_forms) {

	var h = [],
		idx = -1,
		i;

	// Start with the top level form
	for(i = 0; i < sMeta.forms.length; i++) {
		if(sMeta.forms[i].p_id == 0) {
			$(".osmforms").html(addFormToList(sMeta.forms[i], sMeta, 0, true, false, checked_forms, false));
			$(".selectforms").html(addFormToList(sMeta.forms[i], sMeta, 0, false, false, checked_forms, false));
			$(".shapeforms,.taforms").html(addFormToList(sMeta.forms[i], sMeta, 0, true, true, checked_forms, false));
			$(".shapeforms_bs4").html(addFormToList(sMeta.forms[i], sMeta, 0, true, true, checked_forms, true));
		}
	}

	$("button",".selectforms").click(function() {
		var $this = $(this),
			$check = $this.parent().find("input"),
			val,
			val_array = [];

		val = $check.val();
		val_array= val.split(":");
		if(val_array.length > 1) {
			if(val_array[1] === "true") {
				$check.val(val_array[0] + ":false");
				$this.text("Pivot");
			} else {
				$check.val(val_array[0] + ":true");
				$this.text("Flat");
			}
			$this.toggleClass('exportflat');
			$this.toggleClass('exportpivot');
		}

		return false;
	});
}

/*
 * Add a list of date questions to pick from
 */
function addDatePickList(sMeta, currentDate) {

	var h = [],
		idx = -1,
		i,
		value,
		key;

	if(sMeta && sMeta.dates) {
		for(i = 0; i < sMeta.dates.length; i++) {

			key = sMeta.dates[i].name;

			h[++idx] = '<option value="';
			h[++idx] = sMeta.dates[i].id;
			h[++idx] = '">';
			if(key === "Upload Time" || key === "_start" || key === "_end") {
				key = localise.set[key];
			} else if(key === "Scheduled Start") {
				key = localise.set["c_scheduled"]
			}
			h[++idx] = htmlEncode(key);
			h[++idx] = '</option>';

		}

		$(".date_question").empty().html((h.join('')));

		if(typeof currentDate !== "undefined" && currentDate != 0) {
			value = currentDate;
		} else {
			value = $("#settings_date_question").val();
		}
	}
}

/*
 * Add a list of geometry questions to pick from
 */
function addGeomPickList(sMeta) {

	var h = [],
		k = [],
		idx = -1,
		i,
		value,
		theForm;

	if(sMeta && sMeta.forms) {
		for(i = 0; i < sMeta.forms.length; i++) {

			theForm = sMeta.forms[i];

			k[++idx] = h[++idx] = '<div class="exportcontrol showshape showosm" style="display: block;">';
			k[++idx] = h[++idx] = '<label>' + htmlEncode(theForm.form) + '</label>';
			h[++idx] = '<select class="geomSelect" id="geomForm_' + theForm.f_id;            // export only
			k[++idx] = '<select class="geomSelect" id="geomSettingsForm_' + theForm.f_id;    // Settings only
			k[++idx] = h[++idx] = '" data-form="' + theForm.f_id + '">';
			if(theForm.geomQuestions) {
				for(j = 0; j < theForm.geomQuestions.length; j++) {
					k[++idx] = h[++idx] = '<option value="';
					k[++idx] = h[++idx] = theForm.geomQuestions[j];
					k[++idx] = h[++idx] = '">';
					k[++idx] = h[++idx] = htmlEncode(theForm.geomQuestions[j]);
					k[++idx] = h[++idx] = '</option>';
				}
			}
			k[++idx] = h[++idx] = '</select>';
			k[++idx] = h[++idx] = '</div>';

		}

		$(".geomselect_export").empty().html((h.join('')));
		$(".geomselect_settings").empty().html((k.join('')));

		shapeFormsChanged();

	}
}

function shapeFormsChanged() {
	var formId = getSelectedForm('.shapeforms', true);
	if(formId) {
		$('.geomSelect', '.geomselect_export').prop('disabled', true);
		$('#geomForm_' + formId, '.geomselect_export').prop('disabled', false);
	}
}

function getSelectedForm($forms, ignoreError) {
	var forms = $(':radio:checked', $forms).map(function() {
		return this.value;
	}).get();
	if(forms.length === 0) {
		if(!ignoreError) {
			alert(window.localise.set["msg_one_f2"]);
		}
		return 0;
	}
	return forms[0];
}

function addFormToList(form, sMeta, offset, osm, set_radio, checked_forms, bs4) {

	var h = [],
		idx = -1,
		i,
		type,
		checked;

	if (set_radio) {
		type = "radio";
	} else {
		type = "checkbox";
	}

	// Set checked value based on previous selections
	if(set_radio && offset == 0) {
		checked = 'checked="checked"';
	} else {
		if (offset == 0 && (!checked_forms || checked_forms.length == 0)) {
			checked = 'checked="checked"';
		} else {
			checked = '';
		}
	}
	if(checked_forms && checked_forms.length > 0) {
		for(i = 0; i < checked_forms.length; i++) {
			if(form.f_id == checked_forms[i]) {
				checked = 'checked="checked"';
				break;
			}
		}
	}

	h[++idx] = '<div class="' + type + '"';
	h[++idx] = '<span style="padding-left:';
	h[++idx]= offset + 'px;">';
	h[++idx] = '<label>';
	h[++idx] = '<input class="osmform" type="' + type + '" ' + checked + ' name="osmform" value="';
	h[++idx] = form.f_id;
	if(!osm) {
		h[++idx] = ':false"/>';
	} else {
		h[++idx] = '">';
	}
	if(bs4) {
		h[++idx] = '<span class="ml-2">';
	}
	h[++idx] = htmlEncode(form.form);
	if(bs4) {
		h[++idx] = '</span>';
	}
	h[++idx] = '</label>';
	if(form.p_id != 0 && !osm) {
		h[++idx] = ' <button class="exportpivot">' + localise.set["c_pivot"] + '</button>';
	}
	h[++idx]= '</div>';

	// Add the children (recursively)
	for(i = 0; i < sMeta.forms.length; i++) {
		if(sMeta.forms[i].p_id != 0  && sMeta.forms[i].p_id == form.f_id) {
			h[++idx] = addFormToList(sMeta.forms[i], sMeta, offset + 20, osm, set_radio, checked_forms, bs4);
		}
	}

	return h.join('');
}

function getViewLanguages(view) {

	if(view.sId != -1) {
		var url = languageListUrl(view.sId);
		$.getJSON(url, function(data) {
			globals.gSelector.setSurveyLanguages(view.sId, data);
			setSurveyViewLanguages(data, view.lang, '#settings_language', false);
			setSurveyViewLanguages(data, view.lang, '#export_language', true);
		});
	}

}

function validateEmails(emails) {
	var valid = true,
		i;
	if(emails && emails.trim().length > 0) {
		var emailArray = emails.split(",");
		for (i = 0; i < emailArray.length; i++) {
			var validEmail = /[A-Z0-9._%+-]+@[A-Z0-9.-]+.[A-Z]{2,4}/igm;
			if (!validEmail.test(emailArray[i])) {
				valid = false;
				break;
			}
		}
	}
	return valid;
}

/*
 * Get the roles for a survey
 */
function getSurveyRoles(sId, selectedRoles, setall, onlypriv) {

	if (!gTasks.cache.surveyRoles[sId]) {
		addHourglass();
		var url = "/surveyKPI/role/survey/" + sId + "?enabled=true";
		if(onlypriv) {
			url += "&onlypriv=true";
		}
		$.ajax({
			url: url,
			dataType: 'json',
			cache: false,
			success: function (data) {
				removeHourglass();
				if(handleLogout(data)) {
					var savedSelectedRoles = selectedRoles;
					gTasks.cache.surveyRoles[sId] = data;
					showRoles(gTasks.cache.surveyRoles[sId], savedSelectedRoles);
				}
			},
			error: function (xhr, textStatus, err) {

				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						console.log("Error: Failed to get roles for a survey: " + err);
					}
				}
			}
		});
	} else {
		showRoles(gTasks.cache.surveyRoles[sId], selectedRoles, setall);
	}
}

/*
 * Show the roles
 */
function showRoles(data, selectedRoles, setall) {

	var h = [],
		idx = -1,
		i,
		selId,
		selList = [];

	$('.role_select_roles').empty();
	if (data.length > 0) {
		for (i = 0; i < data.length; i++) {
			h[++idx] = '<div class="col-sm-10 custom-control custom-checkbox ml-2 mb-1">'
			h[++idx] = '<input type="checkbox"';
			selId = 'rolesel_' + i;
			h[++idx] = ' id="' + selId + '"';
			if(setall || roleSelected(data[i].id, selectedRoles)) {
				selList.push(selId);
			}
			h[++idx] = ' class="custom-control-input" value="';
			h[++idx] = data[i].id;
			h[++idx] = '">';

			h[++idx] = '<label class="custom-control-label"';
			h[++idx] = ' for="rolesel_' + i + '">';
			h[++idx] = 	htmlEncode(data[i].name);
			h[++idx] = '</label>';
			h[++idx] = '</div>';
		}
		$('.role_select').show();
		$('.role_select_roles').empty().append(h.join(''));
		for(i = 0; i < selList.length; i++) {
			selId = selList[i];
			$('#' + selId).prop('checked', true);
		}
	}
}

function roleSelected(roleId, selectedRoles) {
	var sel = false;
	if(selectedRoles) {
		for(var i = 0; i < selectedRoles.length; i++) {
			if(selectedRoles[i].id == roleId) {
				sel = true;
				break;
			}
		}
	}
	return sel;
}

 /*
  * Get all the surveys that a user can access
  */
function getAccessibleSurveys($elem, includeNone, includeBlocked, groupsOnly, includeSelf) {

	var url="/surveyKPI/surveys";
	var hasParam = false;
	if(includeBlocked) {
		url += hasParam ? '&' : '?';
		url += 'blocked=true';
		hasParam = true;
	}
	if(groupsOnly) {
		url += hasParam ? '&' : '?';
		url += 'groups=true';
		hasParam = true;
	}

	addHourglass();
	$.ajax({
		url: url,
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				var h = [],
					idx = -1,
					i;

				if (includeNone) {
					h[++idx] = '<option value="">';
					h[++idx] = localise.set["c_none"]
					h[++idx] = '</option>';
				}

				if (includeSelf) {
					h[++idx] = '<option value="self">';
					h[++idx] = localise.set["c_self"]
					h[++idx] = '</option>';
				}
				for (i = 0; i < data.length; i++) {
					h[++idx] = '<option value="';
					h[++idx] = htmlEncode(data[i].ident);
					h[++idx] = '">';
					h[++idx] = htmlEncode(data[i].projectName);
					h[++idx] = ' : ';
					h[++idx] = htmlEncode(data[i].displayName);
					h[++idx] = '</option>';
				}
				$elem.empty().append(h.join(''));
			}

		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					console.log("Error: Failed to get list of surveys: " + err);
				}
			}
		}
	});
}

/*
 * Get all the csv files that a user can access
 */
function getAccessibleCsvFiles($elem, includeNone) {

	var url="/surveyKPI/shared/csv/files";

	addHourglass();
	$.ajax({
		url: url,
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				globals.gCsvFiles = data;
				var h = [],
					idx = -1,
					i;

				if (includeNone) {
					h[++idx] = '<option value="">';
					h[++idx] = localise.set["c_none"]
					h[++idx] = '</option>';
				}
				for (i = 0; i < data.length; i++) {
					h[++idx] = '<option value="';
					h[++idx] = i;
					h[++idx] = '">';
					h[++idx] = htmlEncode(data[i].filename);
					h[++idx] = '</option>';
				}
				$elem.empty().append(h.join(''));
			}

		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					console.log("Error: Failed to get list of csv files: " + err);
				}
			}
		}
	});
}

 /*
  * Get the questions in a survey
  */
function getQuestionsInSurvey($elem, $elem_multiple, sIdent, includeNone, textOnly, callback, includeHrk) {

	function populateElement($elem, $elem_multiple, data) {
		var h = [],
			hm = [],
			idx = -1,
			idx_m = -1,
			i,
			setValueFn = callback;

		if (includeNone) {
			h[++idx] = '<option value="0">';
			h[++idx] = localise.set["c_none"];
			h[++idx] = '</option>';
		}
		if (includeHrk) {
			hm[++idx_m] = h[++idx] = '<option value="_hrk">';
			hm[++idx_m] = h[++idx] = localise.set["ed_hrk"];
			hm[++idx_m] = h[++idx] = '</option>';

			hm[++idx_m] = h[++idx] = '<option value="_assigned">';
			hm[++idx_m] = h[++idx] = localise.set["t_assigned"];
			hm[++idx_m] = h[++idx] = '</option>';
		}
		for (i = 0; i < data.length; i++) {
			if(!textOnly || isTextStorageType(data[i].type)) {
				hm[++idx_m] = h[++idx] = '<option value="';
				hm[++idx_m] = h[++idx] = data[i].name;
				hm[++idx_m] = h[++idx] = '">';
				hm[++idx_m] = h[++idx] = htmlEncode(data[i].name);
				hm[++idx_m] = h[++idx] = '</option>';
			}
		}
		if($elem) {
			$elem.empty().append(h.join(''));
		}
		if($elem_multiple) {
			$elem_multiple.empty().append(hm.join(''));
			$elem_multiple.multiselect('deselectAll', false);
			$elem_multiple.multiselect('rebuild');
		}

		if(typeof setValueFn === "function") {
			setValueFn();
		}
	}

	if(sIdent === 'self') {
		populateElement($elem, $elem_multiple, globals.model.survey.forms[globals.gFormIndex].questions);
	} else if(gCache[sIdent]) {
		populateElement($elem, $elem_multiple, gCache[sIdent]);
	} else {
		if (sIdent && sIdent !== "0" && sIdent !== '' && sIdent !== '_none') {
			addHourglass();
			$.ajax({
				url: "/surveyKPI/questionListIdent/" + sIdent + "/none?inc_meta=true",
				dataType: 'json',
				cache: false,
				success: function (data) {
					removeHourglass();
					if(handleLogout(data)) {
						var theIdent = sIdent;
						var $theElem = $elem;
						var $theElemMultiple = $elem_multiple;

						gCache[theIdent] = data;
						populateElement($theElem, $theElemMultiple, data);
					}
				},
				error: function (xhr, textStatus, err) {
					removeHourglass();
					if(handleLogout(xhr.responseText)) {
						if (xhr.readyState == 0 || xhr.status == 0) {
							return;  // Not an error
						} else {
							alert(localise.set["msg_err_get_q"] + ": " + err);
						}
					}
				}
			});
		} else {
			if (includeNone) {
				if($elem) {
					$elem.empty().append('option value="0">' + localise.set["c_none"] + '</option>');
				}
				if($elem_multiple) {
					$elem_multiple.empty().append('option value="0">' + localise.set["c_none"] + '</option>');
					$elem_multiple.multiselect('rebuild');
				}
			}
		}
	}

}

function getQuestionsInCsvFile($elem, $elem_multiple, index, includeNone) {
	var h = [],
		hm = [],
		idx = -1,
		idx_m = -1,
		i;

	if(globals.gCsvFiles[index]) {
		var data = globals.gCsvFiles[index].headers;

		if (includeNone) {		// Only include select none for single selects
			h[++idx] = '<option value="">';
			h[++idx] = localise.set["c_none"];
			h[++idx] = '</option>';
		}
		for (i = 0; i < data.length; i++) {
			hm[++idx_m] = h[++idx] = '<option value="';
			hm[++idx_m] = h[++idx] = data[i].fName;
			hm[++idx_m] = h[++idx] = '">';
			hm[++idx_m] = h[++idx] = htmlEncode(data[i].fName);
			hm[++idx_m] = h[++idx] = '</option>';
		}
		if ($elem) {
			$elem.empty().append(h.join(''));
		}
		if ($elem_multiple) {
			$elem_multiple.empty().append(hm.join(''));
			$elem_multiple.multiselect('deselectAll', false)
			$elem_multiple.multiselect('rebuild');
		}
	}
}

/*
 * Get the questions in a survey
 */
function getGroupQuestionsInSurvey($elem, sIdent) {

	function populateElement($elem, data) {
		var h = [],
			idx = -1,
			i;

		h[++idx] = '<option data-type="" value="">';
		h[++idx] = localise.set["c_none"];
		h[++idx] = '</option>';

		for (i = 0; i < data.length; i++) {
			h[++idx] = '<option data-type="';
			h[++idx] = data[i].type;
			h[++idx] = '" value="';
			h[++idx] = data[i].name;
			h[++idx] = '">';
			h[++idx] = htmlEncode(data[i].name);
			h[++idx] = '</option>';
		}
		$elem.empty().append(h.join(''));
	}

	if(gCacheGroup[sIdent]) {
		populateElement($elem, gCacheGroup[sIdent]);
	} else {
		if (sIdent !== "0") {
			addHourglass();
			$.ajax({
				url: "/surveyKPI/questionListIdent/" + sIdent + "/none/group",
				dataType: 'json',
				cache: false,
				success: function (data) {
					removeHourglass();
					if(handleLogout(data)) {
						var theIdent = sIdent;
						var $theElem = $elem;

						gCacheGroup[theIdent] = data;
						populateElement($theElem, data);
					}

				},
				error: function (xhr, textStatus, err) {
					removeHourglass();
					if(handleLogout(xhr.responseText)) {
						if (xhr.readyState == 0 || xhr.status == 0) {
							return;  // Not an error
						} else {
							alert(localise.set["msg_err_get_q"] + ": " + err);
						}
					}
				}
			});
		} else {
			if (includeNone) {
				$elem.empty().append('option value="0">' + localise.set["c_none"] + '</option>');
			}
		}
	}

}

/*
 * Get the questions suitable for use as a status in a survey group using the survey id as the key
 */
function getGroupStatusQuestions($elem, sId) {

	function populateElement($elem, data) {
		var h = [],
			idx = -1,
			i;

		for (i = 0; i < data.length; i++) {
			h[++idx] = '<option value="';
			h[++idx] = data[i].column_name;
			h[++idx] = '">';
			h[++idx] = htmlEncode(data[i].name);
			h[++idx] = '</option>';
		}
		$elem.empty().append(h.join(''));
	}

	if(gCacheStatusQuestions[sId]) {
		populateElement($elem, gCacheStatusQuestions[sId]);
	} else {
		addHourglass();
		$.ajax({
			url: "/surveyKPI/questionList/" + sId + "/none/group?status=true",
			dataType: 'json',
			cache: false,
			success: function (data) {
				removeHourglass();
				if(handleLogout(data)) {
					var theId = sId;
					var $theElem = $elem;

					gCacheStatusQuestions[theId] = data;
					populateElement($theElem, data);
				}

			},
			error: function (xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert(localise.set["msg_err_get_q"] + ": " + err);
					}
				}
			}
		});
	}
}

/*
 * Get the questions suitable for use as a status in a survey group using the survey id as the key
 */
function getGroupKeys($key, $key_policy, sId) {

	if(gCacheKeys[sId]) {
		$key.val(gCacheStatusQuestions[sId].key);
		$key_policy.val(gCacheStatusQuestions[sId].key_policy)
	} else {
		addHourglass();
		$.ajax({
			url: "/surveyKPI/cases/keys/" + sId,
			dataType: 'json',
			cache: false,
			success: function (data) {
				removeHourglass();
				if(handleLogout(data)) {
					var theId = sId;

					gCacheStatusQuestions[theId] = data;
					$key.val(gCacheStatusQuestions[sId].key);
					$key_policy.val(gCacheStatusQuestions[sId].key_policy);
				}

			},
			error: function (xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert(localise.set["c_error"] + ": " + err);
					}
				}
			}
		});
	}
}

function tokenizeAppearance(input) {
	var chunks = [];
	var tokens = [];
	var chunkTokens = [];
	var i;
	var j;
	var chunk;

	// only search/lookup_choices needs special treatment
	var idx1 = input.indexOf('search');
	if(idx1 < 0) {
		idx1 = input.indexOf('lookup_choices');
	}
	if(idx1 >= 0) {
		chunks.push({
			val:input.substring(0, idx1),
			type: "text"
		});
		if(idx1 < input.length) {
			var idx2 = input.lastIndexOf(')');
			if(idx2 >= 0) {
				chunks.push({
					val: input.substring(idx1, idx2 + 1),
					type: "fn"
				});
				if(idx2 < input.length) {
					chunks.push({
						val: input.substring(idx2 + 1),
						type: "text"
					});
				}
			}
		}
	} else {
		chunks.push({
			val: input,
			type: "text"
		});
	}
	for(i = 0; i < chunks.length; i++) {
		chunk = chunks[i].val.trim();
		if(chunk.length > 0) {
			if(chunks[i].type === "text") {
				chunkTokens = chunk.split(/(\s+)/);
			} else {
				chunkTokens = [];
				chunkTokens.push(chunk);
			}
			for(j = 0; j < chunkTokens.length; j++) {
				if(chunkTokens[j].trim().length > 0) {
					tokens.push(chunkTokens[j].trim());
				}
			}
		}
	}
	return tokens;
}

function setOrganisationTheme() {

	if(globals.gSetAsTheme && globals.gOrgId > 0) {

		var mainLogoSrc = getFromLocalStorage("main_logo");
		var logo = "/media/organisation/" + globals.gOrgId + '/settings/mainLogo';
		if(mainLogoSrc !== logo) {
			setInLocalStorage('main_logo', logo);
			$('.main_logo').attr("src", "/media/organisation/" + globals.gOrgId + '/settings/mainLogo');
		}

		// navbar color
		var navbarColor = getFromLocalStorage("navbar_color");
		if(navbarColor !== globals.gNavbarColor) {
			setInLocalStorage('navbar_color', globals.gNavbarColor);
		}
		// navbar color
		var navbarTextColor = getFromLocalStorage("navbar_text_color");
		if(navbarTextColor !== globals.gNavbarTextColor) {
			setInLocalStorage('navbar_text_color', globals.gNavbarTextColor);
		}
	} else {
		// remove styles
		var navbarColorElement = document. getElementById("navbar_color");
		if(navbarColorElement) {
			navbarColorElement.parentNode.removeChild(navbarColorElement);
		}
		setInLocalStorage('navbar_color', undefined);
		setInLocalStorage('navbar_text_color', undefined);
		setInLocalStorage('main_logo', undefined);

		// Set the default logo
		if(typeof setCustomMainLogo === "function") {
			setCustomMainLogo();
		}
	}
}

/*
 * Surround get / set from local storage in case user has disabled local sorage reading in browser settings
 */
function getFromLocalStorage(key) {
	var value;
	try {
		value = localStorage.getItem(key);
	} catch (e) {

	}
	return value;
}

function setInLocalStorage(key, value) {
	try {
		localStorage.setItem(key, value);
	} catch(e) {

	}
}

function populateTaskGroupList() {
	if (typeof globals.gCurrentProject !== "undefined" && globals.gCurrentProject != -1) {
		addHourglass();
		$.ajax({
			url: "/surveyKPI/tasks/taskgroups/" + globals.gCurrentProject,
			cache: false,
			dataType: 'json',
			success: function (taskgroups) {
				removeHourglass();
				if(handleLogout(taskgroups)) {
					var h = [],
						idx = -1,
						i,
						grp,
						firstTg,
						hasCurrentTg = false;

					window.gTaskGroups = taskgroups;   // Keep the task group list

					if (typeof taskgroups != "undefined" && taskgroups.length > 0) {

						for (i = 0; i < taskgroups.length; i++) {
							grp = taskgroups[i];
							h[++idx] = '<option value="';
							h[++idx] = i;
							h[++idx] = '">';
							h[++idx] = htmlEncode(grp.name);
							h[++idx] = '</option>';
						}
					}
					$('.task_group_select').html(h.join(''));
				}
			},
			error: function (xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert("Failed to get task group data");
					}
				}
			}
		});
	}
}

/*
 * Show a loaded file as an image
 * From https://codepen.io/adamrifai/pen/YXdEwz
 */
function displayAsImage(file, img) {

	var imgURL = URL.createObjectURL(file);
	img.onload = function() {
		URL.revokeObjectURL(imgURL);
	};

	img.src = imgURL;
}

/*
 * If debug=yes is passed as a parameter then enable debuging statement
 */
function enableDebugging() {

	if(location.search.indexOf("debug=yes") >= 0) {
		$(document).on('click', function(e) { console.log(e.target) });
	}

}

/*
 * ----------------------------------------------------
 * Common task functions shared between task managmeent page and console
 */
function setupAssignType(user_id, role_id, emails, email_question) {
	$('.assign_group').hide();
	$('.assign_type').removeClass('active');
	if(user_id != 0) {
		$('.user_type_checkbox').addClass('active');
		$('.assign_user').show();
	} else  if(role_id != 0) {
		$('.role_type_checkbox').addClass('active');
		$('.assign_role').show();
	} else if((typeof emails !== "undefined" && emails.trim().length > 0)
			|| (typeof email_question !== "undefined" && email_question.trim().length > 0)) {
		$('.email_type_checkbox').addClass('active');
		$('.assign_email').show();
	} else {        // Default to user
		$('.user_type_checkbox').addClass('active');
		$('.assign_user').show();
	}
}

// Convert a location name into a location index
function getLocationIndex(name, tags) {
	var idx = -1,
		i;

	if(tags) {
		for(i = 0; i < tags.length; i++) {
			if(tags[i].name == name) {
				idx = i;
				break;
			}

		}
	}
	return idx;

}

function saveTask(isConsole, currentTaskFeature, saveType, updateId, callback, tg_id) {
	var url = "/surveyKPI/api/tasks?preserveInitialData=true&tz=UTC",	// Assume we use UTC times in interface
		taskFeature = {
			properties: {}
		},
		fromDate,
		toDate,
		MIN_SHOW_RANGE = 10;

	taskFeature = $.extend(true, {}, currentTaskFeature);
	taskFeature.properties.assignee_ident = undefined;
	taskFeature.properties.assignee_name = undefined;

	/*
	 * Set the properties of the taskFeature from the dialog
	 */
	taskFeature.properties.p_id = globals.gCurrentProject;
	taskFeature.properties.tg_id = tg_id;

	if (!taskFeature.properties.id || taskFeature.properties.id == "") {
		taskFeature.properties["id"] = 0;
	}
	taskFeature.properties.name = $('#tp_name').val();		// task name
	var surveyIdentifier = $('#tp_form_name').val();
	if(!surveyIdentifier) {
		alert(localise.set["msg_pss"]);
		return false;
	}
	if(isConsole) {
		taskFeature.properties.survey_ident = surveyIdentifier;	// Survey Ident
		taskFeature.properties.form_id = undefined;
	} else {
		// old fashioned
		taskFeature.properties.form_id = surveyIdentifier;	// form id
		taskFeature.properties.survey_ident = undefined;
	}

	taskFeature.properties.assign_type = $("button.assign_type.active", "#task_properties").attr("id");
	if(taskFeature.properties.assign_type == 'tp_user_type') {
		taskFeature.properties.assignee = $('#tp_user').val();
		taskFeature.properties.emails = undefined;
	} else if(taskFeature.properties.assign_type == 'tp_email_type') {
		taskFeature.properties.assignee = 0;
		taskFeature.properties.emails = $('#tp_assign_emails').val();
		if(!validateEmails(taskFeature.properties.emails)) {
			alert(localise.set["msg_inv_email"]);
			return false;
		}
	}

	if(isConsole) {
		taskFeature.properties.update_id = updateId;
		taskFeature.properties.initial_data_source = 'survey';
	}

	taskFeature.properties.repeat = $('#tp_repeat').prop('checked');
	taskFeature.properties.complete_all = $('#tp_pol').prop('checked');
	taskFeature.properties.assign_auto = $('#tp_assign_auto').prop('checked');

	fromDate = $('#tp_from').data("DateTimePicker").date();
	toDate = $('#tp_to').data("DateTimePicker").date();

	// Validate dates
	if(toDate && !fromDate) {       // Can't have a to date without a from date
		alert(localise.set["msg_no_from"]);
		return false;
	}
	if(toDate && fromDate && fromDate > toDate) {       // To date must be after from date
		alert(localise.set["msg_sel_dates"]);
		return false;
	}

	if (fromDate) {
		taskFeature.properties.from = utcTime(fromDate.format("YYYY-MM-DD HH:mm:ss"));
	}
	if (toDate) {
		taskFeature.properties.to = utcTime(toDate.format("YYYY-MM-DD HH:mm:ss"));
	}

	taskFeature.properties.location_trigger = $('#nfc_uid').val();
	taskFeature.properties.guidance = $('#tp_guidance').val();
	taskFeature.properties.show_dist = $('#tp_show_dist').val();

	/*
	 * Save location group and location name
	 */
	var locationIdx = $('#location_select').val();
	if(saveType == "nl") {
		taskFeature.properties.location_group = $('#locationGroupSave').val();
		taskFeature.properties.location_name = $('#locationSave').val();
	} else if(saveType == "ul" && locationIdx != "-1") {
		taskFeature.properties.location_group = $('.location_group_list_sel').text();
		taskFeature.properties.location_name = window.gTags[locationIdx].name;
	} else {
		taskFeature.properties.location_group = undefined;
		taskFeature.properties.location_name = undefined;
	}
	taskFeature.properties.save_type = saveType;

	/*
	 * Convert the geoJson geometry into longitude and latitude for update
	 */
	if (currentTaskFeature.geometry) {
		if (currentTaskFeature.geometry.coordinates && currentTaskFeature.geometry.coordinates.length > 1) {
			//taskFeature.properties.location = "POINT(" + gCurrentTaskFeature.geometry.coordinates.join(" ") + ")";  // deprecate
			taskFeature.properties.lon = currentTaskFeature.geometry.coordinates[0];
			taskFeature.properties.lat = currentTaskFeature.geometry.coordinates[1];

		} else {
			//taskFeature.properties.location = "POINT(0 0)"; // deprecate
			taskFeature.properties.lon = 0;
			taskFeature.properties.lat = 0;
		}
	}

	// TODO task update details (updating existing record)

	// Validations
	if(typeof taskFeature.properties.show_dist === "undefined") {
		taskFeature.properties.show_dist = 0;
	} else {
		taskFeature.properties.show_dist = +taskFeature.properties.show_dist;
	}
	if (taskFeature.properties.show_dist && taskFeature.properties.show_dist < MIN_SHOW_RANGE) {
		alert(localise.set["msg_val_show_dist"]);
		$('#tp_show_dist').focus();
		return;
	}


	var tpString = JSON.stringify(taskFeature.properties);

	addHourglass();
	$.ajax({
		type: "POST",
		dataType: 'text',
		cache: false,
		contentType: "application/x-www-form-urlencoded",
		url: url,
		data: {task: tpString},
		success: function (data, status) {
			removeHourglass();
			if(handleLogout(data)) {
				$('#task_properties').modal("hide");
				callback();
			}
		},
		error: function (xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				alert(localise.set["msg_err_upd"] + " " + xhr.responseText);	// Alerts htmlencode text already
			}
		}
	});
}

/*
 * Get the list of users from the server so they can be assigned to tasks
 */
function getTaskUsers(projectId) {
	var $users = $('.users_select,#users_filter'),
		i, user,
		h = [],
		idx = -1;

	$users.empty();
	$('#users_filter').append('<option value="0">' + localise.set["t_au"] + '</options>');

	$('#users_select_new_task, #users_task_group, #users_select_user, #tp_user')
		.append('<option value="-1">' + localise.set["t_u"] + '</options>');

	$('#users_task_group').append('<option value="-2">' + localise.set["t_ad"] + '</options>');
	$.ajax({
		url: "/surveyKPI/userList",
		cache: false,
		success: function (data) {

			if(handleLogout(data)) {
				for (i = 0; i < data.length; i++) {
					user = data[i];
					// Check that this user has access to the project

					if (!projectId || userHasAccessToProject(user, projectId)) {
						h[++idx] = '<option value="';
						h[++idx] = user.id;
						h[++idx] = '">';
						h[++idx] = htmlEncode(user.name);
						h[++idx] = '</option>';
					}
				}
				$users.append(h.join(''));
			}
		},
		error: function (xhr, textStatus, err) {
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert(localise.set["c_error"] + err);
				}
			}
		}
	});
}

function userHasAccessToProject(user, projectId) {
	var i;
	if(user.projects) {
		for (i = 0; i < user.projects.length; i++) {
			if (user.projects[i].id == projectId) {
				return true;
			}
		}
	}
	return false;
}

function setupTaskDialog() {
	$('#tp_email_type, #assign_email_type').click(function() {
		$('.assign_type').removeClass('active');
		$(this).addClass('active');

		$('.assign_user, .assign_role,.assign_data').hide();
		$('.assign_email').show();
		$('#assign_data').prop('placeholder', localise.set['n_eqc']);
		$('.assign_data').show();
	});
	$('#tp_user_type, #assign_user_type').click(function() {
		$('.assign_type').removeClass('active');
		$(this).addClass('active');

		$('.assign_user').show();
		$('.assign_role,.assign_email').hide();
		if($('#users_task_group').val() == -2) {
			$('#assign_data').prop('placeholder', "");
			$('.assign_data').show();
		} else {
			$('.assign_data').hide();
		}
	});
	$('#tp_role_type, #assign_role_type').click(function() {
		$('.assign_type').removeClass('active');
		$(this).addClass('active');

		$('.assign_user, .assign_email').hide();
		$('.assign_role').show();
		if($('#roles_task_group').val() == -2) {
			$('#assign_data').prop('placeholder', "");
			$('.assign_data').show();
		} else {
			$('.assign_data').hide();
		}
	});

	$('#tp_from').datetimepicker({
		useCurrent: false,
		locale: gUserLocale || 'en'
	});

	$('#tp_to').datetimepicker({
		useCurrent: false,
		locale: gUserLocale || 'en'
	});

	$('#tp_from').on("dp.change", function () {

		var startDateLocal = $(this).data("DateTimePicker").date(),
			endDateLocal = $('#tp_to').data("DateTimePicker").date(),
			originalStart = gCurrentTaskFeature.properties.from,
			originalEnd = gCurrentTaskFeature.properties.to,
			newEndDate,
			duration;

		if (startDateLocal) {

			if (originalEnd && originalStart) {
				duration = moment(originalEnd, "YYYY-MM-DD HH:mm:ss").diff(moment(originalStart, "YYYY-MM-DD HH:mm:ss"), 'hours');
				newEndDate = startDateLocal.add(duration, 'hours');
				$('#tp_to').data("DateTimePicker").date(newEndDate);
			}
		}



	});

}

function getStatusClass(status, assign_auto) {

	var statusClass = "";

	if (status === "new") {
		if(assign_auto) {
			statusClass = "bg-orange";
		} else {
			statusClass = "bg-info";
		}
	} else if (status === "submitted" || status === "success") {
		statusClass = "bg-success";
	} else if (status === "late") {
		statusClass = "bg-danger";
	} else if (status === "accepted" || status === "pending") {
		statusClass = "bg-warning";
	} else 	if (status === "error" || status === "unsent" || status === "unsubscribed"
		|| status === "blocked" || status === "rejected" || status === "cancelled" || status === "deleted") {
		statusClass = "bg-rejected";
	} else {
		statusClass = "bg-success";
	}
	return statusClass;
}

/*
 *------------------------------------------------------------------
 * Common notification functions shared between console and notifications
 */
function edit_notification(edit, idx, inconsole) {

	var notification;
	var title;

	document.getElementById("notification_edit_form").reset();

	if(edit) {
		notification = window.gNotifications[idx];

		$('#bundle').prop('checked', notification.bundle);
		$('#addNotificationLabel').text(localise.set["msg_edit_notification"]);
		$('#trigger').val(notification.trigger);
		$('#target').val(notification.target);
		$('#name').val(notification.name);
		setTargetDependencies(notification.target);
		$('.assign_question').hide();
		if(notification.target === 'escalate' && notification.remote_user === '_data') {
			$('.assign_question').removeClass('d-none').show();
		}

		gSelectedOversightQuestion = notification.updateQuestion;
		gSelectedOversightSurvey = notification.updateSurvey;
		setTriggerDependencies(notification.trigger);
		setAttachDependencies(notification.notifyDetails.attach);

		if (notification.trigger !== "task_reminder") {
			if(notification.bundle) {
				$('#bundle_survey').val(notification.bundle_ident).change();
			} else {
				$('#survey').val(notification.s_id).change();
			}
		}
		$('#not_filter').val(notification.filter);
		$('#update_value').val(notification.updateValue);
		$('#alerts').val(notification.alert_id);
		$('#sc_question').val(notification.updateQuestion);
		$('#sc_value').val(notification.updateValue);

		// reminder settings
		if (!inconsole) {
			$('#task_group').val(getTaskGroupIndex(notification.tgId));
			if ((notification.period)) {
				var periodArray = notification.period.split(" ");
				if (periodArray.length > 1) {
					$('#r_period').val(periodArray[0]);
					$('#period_list_sel').val(periodArray[1]);
				}
			}
			if(notification.trigger === "task_reminder") {
				taskGroupChanged($('#task_group').val(), notification.notifyDetails.emailQuestionName, notification.notifyDetails.emailMeta);
			}
		}

		// Periodic settings
		$('#periodic_period').val(notification.periodic_period);
		$('#periodic_time').val(notification.periodic_time);
		$('#periodic_week_day').val(notification.periodic_week_day);
		$('#periodic_month_day').val(notification.periodic_month_day);
		$('#periodic_month, #periodic_month_quarter').val(notification.periodic_month);
		$('#report').val(notification.r_id);
		setPeriodDependencies(notification.periodic_period);

		if(notification.trigger !== "task_reminder" && (typeof notification.alert_id !== 'undefined'
			|| (notification.notify_details && (notification.notifyDetails.emailQuestionName || notification.notifyDetails.emailMeta)))) {

				surveyChangedNotification(notification.notifyDetails.emailQuestionName,
					notification.notifyDetails.assign_question,
					notification.notifyDetails.emailMeta,
					notification.alert_id,
					notification.updateQuestion,
					notification.notifyDetails.survey_case);
		}

		if (notification.notifyDetails) {

			if (notification.target == "email" || notification.target == "escalate") {
				if (notification.notifyDetails.emails) {
					$('#notify_emails').val(notification.notifyDetails.emails.join(","));
				}
				$('#assigned_user').prop('checked', notification.notifyDetails.emailAssigned);
				$('#email_subject').val(notification.notifyDetails.subject);
				$('#email_content').val(notification.notifyDetails.content);
				$('#email_attach').val(notification.notifyDetails.attach);
				$('#include_references').prop('checked', notification.notifyDetails.include_references);
				$('#launched_only').prop('checked', notification.notifyDetails.launched_only);
			} else if (notification.target == "sms") {
				if (notification.notifyDetails.emails) {
					$('#notify_sms').val(notification.notifyDetails.emails.join(","));
				}
				$('#sms_content').val(notification.notifyDetails.content);
				$('#sms_attach').val(notification.notifyDetails.attach);
				$('#sms_sender_id').val(notification.notifyDetails.subject);
			} else if (notification.target == "conversation") {
				if (notification.notifyDetails.emails) {
					$('#notify_sms').val(notification.notifyDetails.emails.join(","));
				}
				$('#conversation_text').val(notification.notifyDetails.content);
			} else if (notification.target == "webhook") {
				$('#callback_url').val(notification.notifyDetails.callback_url);
			}
		}
		if (!inconsole) {
			$('#fwd_user,#user_to_assign').val(notification.remote_user).change();
			$('#assign_question').val(notification.notifyDetails.assign_question);
			$('#survey_case').val(notification.notifyDetails.survey_case);
			gEligibleUser = notification.remote_user;
			// Password not returned from server - leave blank

			$('#fwd_host').val(notification.remote_host);

			// assign user from data
			if($('#user_to_assign').val() === '_data') {
				$('.assign_question').removeClass('d-none').show();
			}

			if (notification.enabled) {
				$('#nt_enabled').prop('checked', true);
			} else {
				$('#nt_enabled').prop('checked', false);
			}
		}

		window.gUpdateFwdPassword = false;
		window.gSelectedNotification = notification.id;
	} else {

		$('#fwd_host').val(window.gRemote_host);	// Set the values to the one's last used
		$('#fwd_user').val(window.gRemote_user);

		$('#survey').change();

		setTargetDependencies('email');
		setTriggerDependencies('submission');

		// Reminders
		$('#r_period').val(1);
		$('#period_list_sel').val('days');
		$('#nt_enabled').prop('checked',true);
		window.gUpdateFwdPassword = true;
		window.gSelectedNotification = -1;
	}
	bundleSelectChanged();

}

function bundleSelectChanged() {
	if($('#bundle').is(':checked')) {
		$('.bundle').show();
		$('.notbundle').hide();
	} else {
		$('.bundle').hide();
		$('.notbundle').show();
	}
}
function setTargetDependencies(target) {
	$('.sms_options, .webhook_options, .email_options, .escalate_options, .conv_options').hide();
	if(target === "email") {
		$('.email_options').show();
		initMsgNotPopup(target);
	} else if(target === "sms") {
		$('.sms_options').show();
	} else if(target  === "webhook") {
		$('.webhook_options').show();
	} else if(target  === "escalate") {
		$('.escalate_options,.email_options').show();
	} else if(target  === "conversation") {
		$('.conv_options').show();
		initMsgNotPopup(target);
	}
}

function setTriggerDependencies(trigger) {
	$('.task_reminder_options,.update_options, .submission_options, .cm_alert_options, .periodic_options, .sc_options').hide();
	if(trigger === "submission") {
		$('.submission_options').show();
	} else if(trigger === "task_reminder") {
		$('.task_reminder_options').show();
		$('#target').val('email');
		setTargetDependencies('email');
	} else if(trigger === "cm_alert") {
		$('.cm_alert_options').show();
	} else if(trigger === "periodic") {
		$('.periodic_options').show();
	} else if(trigger === "server_calc") {
		$('.sc_options').show();
	}
}

function setAttachDependencies(attach) {
	if(attach === "pdf" || attach === "pdf_landscape") {
		$('.pdf_options').show();
	} else  {
		$('.pdf_options').hide();
	}
}

function setPeriodDependencies(period) {
	$('.periodic_week_day, .periodic_month_day, .periodic_month, .periodic_month_quarter').hide();
	if(period === "weekly") {
		$('.periodic_week_day').show();
	} else if(period === "monthly") {
		$('.periodic_month_day').show();
	} else if(period === "yearly") {
		$('.periodic_month').show();
	} else if(period === "quarterly") {
		$('.periodic_month_quarter').show();
	}
}

/*
 * Initialise notification popup
 * Only required if the eDitRecord variable is set as used in immediate notifications
 */
function initMsgNotPopup(target) {
	if(window.gEditRecord) {
		var $msg = $('#msg_cur_nbr');
		var $email = $('#email_cur');
		var other = localise.set["c_other"];

		$('.other_msg').hide();
		$('.recvd_emails').hide();

		$msg.empty();
		var hasSelect = false;
		var hasEmailSelect = false;
		if (window.gEditRecord.contacts) {
			for (const [key, value] of Object.entries(window.gEditRecord.contacts)) {
				// Hack fix up channel for old entries, its either sms or email
				if (!value.channel) {
					value.channel = (key.indexOf("@") > 0) ? 'email' : 'sms';
				}

				if (!value.channel || value.channel === 'sms' || value.channel === 'whatsapp') {
					hasSelect = true;
					$msg.append(`<option data-channel="${value.channel}" value="${key}">${key} - ${value.channel} </option>`);
				} else {
					hasEmailSelect = true;
					$email.append(`<option value="${key}">${key}</option>`);
				}
				setOurNumbersList();
			}
		}
		$msg.append(`<option value="other">${other}</option>`);
		$email.append(`<option value="other">${other}</option>`);
		if(target === "conversation") {
			msgCurNbrChanged();
		}

		if(hasEmailSelect) {
			$('.recvd_emails').show();
		}

		$('#msg_cur_nbr').change(function () {
			msgCurNbrChanged();
		});

		$('#msg_channel').change(function () {
			setOurNumbersList();
		});

	}
}

/*
 * Change attribute visibility if the user select an existing number to message or selects other
 */
function msgCurNbrChanged($choice) {
	if ($('#msg_cur_nbr').val() === 'other') {
		$('.other_msg').show();
		$('#msg_channel').prop( "disabled", false);
	} else {
		$('.other_msg').hide();
		$('#msg_channel').val($('#msg_cur_nbr option:selected').attr('data-channel')).prop( "disabled", true).trigger("change");
	}
}

/*
 * Update the notification list
 */
function updateNotificationTypes(data) {

	var $selector=$('#target'),
		i,
		h = [],
		idx = -1;

	for(i = 0; i < data.length; i++) {

		h[++idx] = '<option value="';
		h[++idx] = data[i];
		h[++idx] = '">';
		h[++idx] = localise.set["c_" + data[i]];
		h[++idx] = '</option>';
	}

	$selector.empty().append(h.join(''));
	gConversationalSMS = false;

}

/*
 * Load the existing notifications from the server
 */
function getNotificationTypes(page) {

	addHourglass();
	$.ajax({
		url: '/surveyKPI/notifications/types?page=' + page,
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				window.gNotificationTypes = data;
				if (data) {
					updateNotificationTypes(data);
					if(gTasks && gTasks.cache && gTasks.cache.currentData) {
						updateConversationalSMS(gTasks.cache.currentData.sms);
					}
				}
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					console.log("Error: Failed to get list of notification types: " + err);
				}
			}
		}
	});
}

/*
 * Update anything related to using conversations and SMS
 */
function updateConversationalSMS(sms) {
	if(sms && !gConversationalSMS) {  // Add if there is SMS data associated with this survey and the type has not already been added
		var $selector=$('#target'),
			h = [],
			idx = -1;


		h[++idx] = '<option value="conversation">';
		h[++idx] = localise.set["c_conversation"];
		h[++idx] = '</option>';

		$selector.append(h.join(''));
		gConversationalSMS = true;
	}
}

function setupNotificationDialog() {

	// Set change function trigger
	$('#trigger').off().change(function() {
		var trigger = $(this).val();
		setTriggerDependencies(trigger);
		if(trigger === "task_reminder") {
			taskGroupChanged($('#task_group').val());
		}
		if(trigger === "console_update") {
			getGroupSurveys($('#survey').val(), showOversightSurveys);
		}
	});
	setTriggerDependencies("submission");

	// Set change function target
	$('#target').off().change(function() {
		setTargetDependencies($(this).val());
	});
	setTargetDependencies("email");

	// Set change function attach
	$('#email_attach').off().change(function() {
		setAttachDependencies($(this).val());
	});

	// Set dependencies on a periodic trigger period change
	setPeriodDependencies($('#period_period').val());
	$('#periodic_period').off().change(function() {
		setPeriodDependencies($(this).val());
	});


	// Set focus on notification name when edit notification is opened
	$('#addNotificationPopup').on('shown.bs.modal', function () {
		$('#name').focus();
	});

	/*
	 * Functions for forwarding
	 */
	$('#fwd_host').change(function(){
		var host = $(this).val();
		if(host.length === 0) {
			return false;
		} else if(host.substr(0, 4) !== "http") {
			alert(localise.set["msg_val_prot"]);
			return false;
		}
	});

	$('#fwd_password').change(function(){
		window.gUpdateFwdPassword = true;
	});

}

/*
 Get updated question names if the task group changes
 */
function taskGroupChanged(tgIndex, emailQuestionName, emailMetaName) {

	var tg = gTaskGroups[tgIndex];
	var language = "none";
	var qList;
	var metaList;

	if(tg && tg.source_s_id) {
		qList = globals.gSelector.getSurveyQuestions(tg.source_s_id, language);
		metaList = globals.gSelector.getSurveyMeta(tg.source_s_id);
	} else {
		qList = [];
		metaList = [];
	}

	if(!qList) {
		getQuestionList(tg.source_s_id, language, 0, "-1", undefined, false,
			undefined, undefined, emailQuestionName, undefined);
	} else {
		setSurveyViewQuestions(qList, undefined, undefined, undefined, emailQuestionName, undefined, undefined, undefined);
	}

	if(!metaList) {
		getMetaList(tg.source_s_id, undefined);
	} else {
		setSurveyViewMeta(metaList, undefined);
	}
}

/*
 * Process a save notification when the target is "email"
 */
function saveEmail() {

	var notification = {};
	var emails = $('#notify_emails').val();
	var emailQuestionName = $('#email_question').val();
	var emailMetaItem = $('#email_meta').val();
	var emailAssigned = $('#assigned_user').is(':checked');
	var emailArray;
	var i;

	// validate
	// Must specifify an email
	notification.error = false;
	if((!emails || emails.trim().length == 0) && (!emailQuestionName || emailQuestionName == "-1")
		&& (!emailMetaItem || emailMetaItem == "-1") && !emailAssigned) {
		notification.error = true;
		notification.errorMsg = localise.set["msg_inv_email"];
		notification.notifyDetails = {};
	}

	// Text email must be valid email addresses
	if(emails && emails.trim().length > 0) {
		emailArray = emails.split(",");
		for (i = 0; i < emailArray.length; i++) {
			if (!validateEmails(emailArray[i])) {
				notification.error = true;
				notification.errorMsg = localise.set["msg_inv_email"];
				notification.notifyDetails = {};
				break;
			}
		}
	}

	if(!notification.error) {
		notification.target = "email";
		notification.notifyDetails = {};
		notification.notifyDetails.emails = emailArray;
		notification.notifyDetails.emailQuestionName = emailQuestionName;
		notification.notifyDetails.emailAssigned = emailAssigned;
		notification.notifyDetails.emailMeta = emailMetaItem;
		notification.notifyDetails.subject = $('#email_subject').val();
		notification.notifyDetails.content = $('#email_content').val();
		notification.notifyDetails.attach = $('#email_attach').val();
		notification.notifyDetails.include_references = $('#include_references').prop('checked');
		notification.notifyDetails.launched_only = $('#launched_only').prop('checked');
	}

	return notification;
}

/*
 * Process a save notification when the target is "sms"
 */
function saveSMS() {

	var notification = {};

	notification.target = "sms";
	notification.notifyDetails = {};
	notification.notifyDetails.emails = $('#notify_sms').val().split(",");
	notification.notifyDetails.emailQuestionName = $('#sms_question').val();
	notification.notifyDetails.subject = $('#sms_sender_id').val();
	notification.notifyDetails.content = $('#sms_content').val();
	notification.notifyDetails.attach = $('#sms_attach').val();

	return notification;
}

/*
 * Process a save notification when the target is "document"
 */
function saveDocument() {

	var notification = {};

	notification.target = "document";
	notification.notifyDetails = {};

	return notification;
}

/*
 * Process a save notification when the target is "conversation"
 */
function saveConversation(columns, theirNumber, ourNumber, msgChannel, record) {

	var notification = {};

	notification.target = "conversation";
	notification.notifyDetails = {};
	notification.notifyDetails.content = $('#conversation_text').val();
	notification.notifyDetails.emails = [theirNumber];		// Must be sent as an array
	notification.notifyDetails.ourNumber = ourNumber;
	notification.notifyDetails.msgChannel = msgChannel;

	if(!theirNumber || theirNumber.length === 0) {
		notification.error = true;
		notification.errorMsg = localise.set["msg_no_nbr"];
	}
	return notification;
}

/*
 * Process a save notification when the target is "webhook"
 */
function saveWebhook() {

	var error = false,
		callback_url,
		notification = {};

	callback_url = $('#callback_url').val();

	if(!error) {

		notification.target = "webhook";
		notification.remote_user = $('#fwd_user').val();
		notification.remote_password = $('#fwd_password').val();
		notification.notifyDetails = {};
		notification.notifyDetails.callback_url = callback_url;
		notification.update_password = window.gUpdateFwdPassword;

	} else {
		notification.error = true;
	}

	return notification;
}

/*
 * Process a save notification when the target is "escalate"
 */
function saveEscalate() {

	var error = false,
		callback_url,
		notification = {};

	if(!error) {

		notification.target = "escalate";
		notification.remote_user = $('#user_to_assign').val();


		notification.notifyDetails = {};
		notification.notifyDetails.survey_case = $('#survey_case').val();
		notification.notifyDetails.assign_question = $('#assign_question').val();

	} else {
		notification.error = true;
	}

	return notification;
}

function getTaskGroupIndex(tgId) {
	var i;
	if(gTaskGroups && gTaskGroups.length > 0 && tgId) {
		for(i = 0; i < gTaskGroups.length; i++) {
			if(gTaskGroups[i].tg_id == tgId) {
				return i;
			}
		}
	}
	return 0;
}

function surveyChangedNotification(qName, assignQuestion, metaItem, alertId, updateQuestion, surveyVal) {

	var language = "none",
		bundle = $('#bundle').is(':checked'),
		sId = $('#survey').val() || 0,
		bundle_ident = $('#bundle_survey').val(),
		qList,
		metaList,
		alertList;

	if(bundle && bundle_ident) {
		getGroupSurveys(bundle_ident, setGroupSelector, surveyVal);		// Get the surveys in the group
	} else if(sId) {
		if(!qName) {
			qName = "-1";
		}

		getGroupSurveys(sId, setGroupSelector, surveyVal);		// Get the surveys in the group

		qList = globals.gSelector.getSurveyQuestions(sId, language);
		metaList = globals.gSelector.getSurveyMeta(sId);
		alertList = globals.gSelector.getSurveyAlerts(sId);

		if(!qList) {
			getQuestionList(sId, language, 0, "-1", undefined, false,
				undefined, undefined, qName, assignQuestion, updateQuestion);
		} else {
			setSurveyViewQuestions(qList, undefined, undefined, undefined, qName, assignQuestion, undefined, updateQuestion);
		}

		if(!metaList) {
			getMetaList(sId, metaItem);
		} else {
			setSurveyViewMeta(metaList, metaItem);
		}

		if(!alertList) {
			getAlertList(sId, alertId);
		} else {
			setSurveyAlerts(alertList, alertId);
		}

	}
}

function getInitialDataLink(task) {
	var tab = [];
	idx = -1;

	tab[++idx] = '<a href="';
	tab[++idx] = getWebFormUrl(task.properties.survey_ident,
		task.properties.update_id,
		task.properties.initial_data_source,
		task.properties.id,
		task.properties.a_id);
	tab[++idx] = '" target="_blank">'
	tab[++idx] = '<i class="fa fa-file-text"></i>';	// Edit existing data
	tab[++idx] = '</a>';

	return tab.join('');
}

function getWebFormUrl(form_ident, update_id, initial_data_source, taskId, assignmentId) {
	var url,
		hasParams = false;

	initial_data_souce = initial_data_source || 'none';

	url = "/webForm/" + form_ident;

	if (update_id && initial_data_source === 'survey') {
		url += "?datakey=instanceid&datakeyvalue=" + update_id;
		url += "&viewOnly=true"
		hasParams = true;
	} else {
		url += '?taskkey=';
		url += taskId;
		hasParams = true;
	}
	url += (hasParams ? '&' : '?');
	url += 'assignment_id=';
	url += assignmentId;

	return url;
}

function taskReport(taskGroup) {
	var tz = Intl.DateTimeFormat().resolvedOptions().timeZone,
		tzParam = "",
		url = '/surveyKPI/tasks/xls/' + taskGroup,
		hasParam = false,
		statusFilterArray = $('#status_filter').val(),
		period_filter = $('#period').val();

	// Add parameters
	if (tz) {
		url += (hasParam ? '&' : '?') + "tz=" + encodeURIComponent(tz);
		hasParam = true;
	}
	if(statusFilterArray) {
		url += (hasParam ? '&' : '?') + 'inc_status=' + statusFilterArray.join(',');
		hasParam = true;
	}
	if(period_filter) {
		url += (hasParam ? '&' : '?') + 'period=' + period_filter;
		hasParam = true;
	}

	downloadFile(url);
}

/*
 * Check to see if the status of the task means it should be included
 */
function includeByStatus(statusFilter, task, excludeZeroOrigin) {

	var include = statusFilter.indexOf(task.properties.status) >= 0;
	if(!include) {
		// check for late
		if(task.properties.status === 'accepted' && isLate(task.properties.to) && statusFilter.indexOf("late") >= 0) {
			include = true;
		}
	}
	if(include && excludeZeroOrigin) {
		// Remove points with 0,0 coordinates
		include = false;
		if(task.geometry) {
			include = true;
			if(task.geometry.type === "Point" && task.geometry.coordinates[0] == 0 && task.geometry.coordinates[1] == 0) {
				include = false;
			}
		}
	}

	return include;
}

/*
 * Return true if this question stores its data in a text type column
 */
function isTextStorageType(type) {
	return type === "string" || type === "select1" || type === "barcode" || type === "calculate"
		|| type === "conversation"
		|| type === "child_form" || type === "parent_form";
}

/*
 * Get surveys in the same bundle
 */
function getGroupSurveys(surveyId, callback, surveyVal) {

	var url = "/surveyKPI/surveyResults/" + surveyId + "/groups",
		survey = surveyId;

	if(surveyId) {

		if(gTasks.cache.groupSurveys[surveyId]) {
			if(typeof callback === 'function') {
				callback(gTasks.cache.groupSurveys[surveyId], surveyVal);
			}
		} else {
			addHourglass();
			$.ajax({
				url: url,
				dataType: 'json',
				cache: false,
				success: function (data) {
					removeHourglass();
					if(handleLogout(data)) {
						gTasks.cache.groupSurveys[surveyId] = data;
						if (typeof callback === 'function') {
							callback(data, surveyVal);
						}
					}
				},
				error: function (xhr, textStatus, err) {
					removeHourglass();
					if(handleLogout(xhr.responseText)) {
						if (xhr.readyState == 0 || xhr.status == 0) {
							return;  // Not an error
						} else {
							console.log(localise.set["c_error"] + ": " + err);
						}
					}
				}
			});
		}
	}
}

/*
 * Update a selector that is used for any data survey in a group that is not an oversight form
 */
function setGroupSelector(data, surveyVal) {
	var $elemGroups = $('#survey_case, #tp_form_name, #not_form_name');

	var i,
		item,
		h = [],
		idx = -1;

	for (i = 0; i < data.length; i++) {
		item = data[i];

		if (item.dataSurvey) {
			h[++idx] = '<option value="';
			h[++idx] = item.surveyIdent;
			h[++idx] = '">';
			h[++idx] = htmlEncode(item.surveyName);
			h[++idx] = '</option>';

			if(!surveyVal) {
				surveyVal = item.surveyIdent;
			}
		}
	}

	$elemGroups.empty().html(h.join(''));
	if(surveyVal) {
		$elemGroups.val(surveyVal).change();
	}

}

function showOversightSurveys(data) {
	var i,
		item,
		h = [],
		idx = -1,
		surveyId = $('#survey').val(),
		count = 0;

	$('#oversight_survey').empty();

	for (i = 0; i < data.length; i++) {
		item = data[i];

		if (item.oversightSurvey && item.sId != surveyId) {
			h[++idx] = '<option value="';
			h[++idx] = item.surveyIdent;
			h[++idx] = '">';
			h[++idx] = htmlEncode(item.surveyName);
			h[++idx] = '</option>';

			if(count == 0) {
				if(gSelectedOversightSurvey) {
					getOversightQuestionList(gSelectedOversightSurvey, showOversightQuestions);
				} else {
					getOversightQuestionList(item.surveyIdent, showOversightQuestions);
				}
			}
			count++;
		}
	}

	if(count == 0) {
		$('.update_options_msg').html(localise.set["n_no_oversight"]);
		$('.update_options_msg').show();
	} else {
		$('.update_options_msg').hide();
	}
	$('#oversight_survey').empty().html(h.join(''));
	if(gSelectedOversightSurvey) {
		$('#oversight_survey').val(gSelectedOversightSurvey);
	}
}

//Function to get the question list
function getOversightQuestionList(sIdent, callback) {

	var url = "/surveyKPI/questionListIdent/" + sIdent + "/none?exc_read_only=false&inc_meta=false";

	if(window.oversightQuestions[sIdent]) {
		callback(window.oversightQuestions[sIdent]);
	} else {
		addHourglass();
		$.ajax({
			url: url,
			dataType: 'json',
			cache: false,
			success: function(data) {
				removeHourglass();
				if(handleLogout(data)) {
					window.oversightQuestions[sIdent] = data;
					callback(data);
				}
			},
			error: function(xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert("Error: Failed to get list of questions: " + err);
					}
				}
			}
		});
	}

}

function showOversightQuestions(data) {
	var i,
		item,
		h = [],
		idx = -1;

	for (i = 0; i < data.length; i++) {
		item = data[i];

		h[++idx] = '<option value="';
		h[++idx] = item.name;
		h[++idx] = '">';
		h[++idx] = htmlEncode(item.name);
		h[++idx] = '</option>';

	}

	$('#update_question').empty().html(h.join(''));
	if(gSelectedOversightQuestion) {
		$('#update_question').val(gSelectedOversightQuestion);
	}
}

/*
 * Convert system names for meta data into human names
 */
function translateKey(key) {
	if(key === "_device") {
		key = localise.set["c_device"];  //"Device";
	} else if (key === "_user") {
		key = localise.set["c_user"];  // "Submitted By";
	} else if (key === "_start") {
		key = localise.set["_start"] + " (" + localise.set["c_lt"] +")"; // "Start Survey";
	} else if (key === "_end") {
		key = key = localise.set["_end"] + " (" + localise.set["c_lt"] +")";  // "End Survey";
	} else if (key === "Upload Time") {
		key = key = localise.set[key] + " (" + localise.set["c_lt"] +")";
	} else if (key === "_scheduled_start") {
		key = key = localise.set[key] + " (" + localise.set["c_lt"] +")";
	} else if (key === "_bad") {
		key = localise.set["a_mb"];         // "Marked Bad";
	} else if (key === "_bad_reason") {
		key = localise.set["c_reason"];     // "Reason";
	} else if (key === "_complete") {
		key = localise.set["c_complete"];	// "Complete";
	}

	return key;
}

/*
 * Convert system names for meta values into human values
 */
function translateKeyValue(key, value) {

	if (key === "_bad") {
		if(value === "t") {
			value = localise.set["c_yes"];   // "Yes";
		} else {
			value = localise.set["c_no"];   // "No";
		}
	} else if (key === "_complete") {
		value = (value === "t") ? localise.set["c_yes"] : localise.set["c_no"];
	}

	return value;

}

function addCacheBuster(url) {
	var cb;
	if(url.indexOf("?") >= 0) {
		cb = "&";
	} else {
		cb = "?";
	}
	return cb + "_v=" + new Date().getTime().toString();
}

function getAppearanceParams(appearance) {

	var response = {};

	var idx1 = appearance.indexOf('(');
	var idx2 = appearance.lastIndexOf(')');
	var params = appearance.substring(idx1 + 1, idx2);
	var paramsArray = [];
	if(params) {
		paramsArray = params.split(',');
	}

	response.length = paramsArray.length;
	if(paramsArray.length > 0) {

		// 1. First parameter is the filename
		var filename = paramsArray[0].trim();
		response.filename = filename.replace(/'/g, "");

		response.filter = '';    // default
		if(paramsArray.length > 1) {
			// Second parameter is the filter
			response.filter = paramsArray[1].trim();
			response.filter = response.filter.replace(/'/g, "");
		}

		if(response.filter === 'eval') {
			if (paramsArray.length > 2) {
				// Third parameter for an evaluation type function is the expression
				// For an expression type filter only remove the first and last single quote if they exist
				response.expression = paramsArray[2].trim();
				if(response.expression.charAt(0) == '\'') {
					response.expression = response.expression.substring(1);
				}
				if(response.expression.charAt(response.expression.length - 1) == '\'') {
					response.expression = response.expression.substring(0, response.expression.length - 1);
				}
			}
		} else {

			if (paramsArray.length > 2) {
				// Third parameter is the filter column
				response.filter_column = paramsArray[2].trim();
				response.filter_column = response.filter_column.replace(/'/g, "");
			}

			if (paramsArray.length > 3) {
				// Fourth parameter is the filter value
				response.filter_value = paramsArray[3].trim();
				response.filter_value = response.filter_value.replace(/'/g, "");
			}

			if (paramsArray.length > 4) {
				// Fifth parameter is the second filter column
				response.second_filter_column = paramsArray[4].trim();
				response.second_filter_column = response.second_filter_column.replace(/'/g, "");
			}


			if (paramsArray.length > 5) {
				// Sixth parameter is the filter value
				response.second_filter_value = paramsArray[5].trim();
				response.second_filter_value = response.second_filter_value.replace(/'/g, "");
			}
		}

	}
	return response;
}

function getQuestionType(schema, qname) {
	var i;
	for(i = 0; i < schema.columns.length; i++) {
		if(schema.columns[i].question_name == qname) {
			return schema.columns[i].type;
		}
	}
}

function getTrailData(projectId, userId, startDate, endDate, callback, tz, mps) {

	var url = '/surveyKPI/usertrail/trail' +
		'?userId=' + userId +
		'&startDate=' + startDate +
		'&endDate=' + endDate +
		'&mps=' + (mps || 0) +
		(tz ? "&tz=" + tz : "");

	addHourglass();
	$.ajax({
		url: url,
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				callback(data);
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert("Error: Failed to get user trail: " + err);
				}
			}
		}
	});
}

/*
 * Reports
 */
function executeUsageReport(oId) {

	var usageMsec = $('#usageDate').data("DateTimePicker").date(),
		d = new Date(usageMsec),
		month = d.getMonth() + 1,
		year = d.getFullYear(),
		incTemp = $('#usage_inc_temp').prop('checked'),
		incAllTime = $('#usage_inc_alltime').prop('checked'),
		byProject = $('#usage_by_project').prop('checked'),
		bySurvey = $('#usage_by_survey').prop('checked'),
		byDevice = $('#usage_by_device').prop('checked'),
		i;

	var reportName = localise.set["u_usage"] + "_";

	// Add the organisation name
	if(oId > 0 && globals.gLoggedInUser.orgs.length > 0) {
		for(i = 0; i < globals.gLoggedInUser.orgs.length; i++) {
			if(globals.gLoggedInUser.orgs[i].id == oId) {
				reportName += globals.gLoggedInUser.orgs[i].name + "_";
				break;
			}
		}
	}

	if(byProject) {
		reportName += localise.set["c_project"];
	} else if(bySurvey) {
		reportName += localise.set["c_survey"];
	} else if(byDevice) {
		reportName += localise.set["c_device"];
	} else {
		reportName += localise.set["c_user"];
	}
	reportName += "_" + year + "_" + month;
	reportName = reportName.replaceAll(' ', '_');

	var reportObj = {
		report_type: 'u_usage',
		report_name: reportName,
		pId: 0,
		params: {
			oId: oId,
			byProject: byProject,
			bySurvey: bySurvey,
			byDevice: byDevice,
			month: month,
			year: year,
			incTemp: incTemp,
			incAllTime: incAllTime
		}
	}

	var tzString = globals.gTimezone ? "?tz=" + encodeURIComponent(globals.gTimezone) : "";

	addHourglass();
	$.ajax({
		type: "POST",
		cache: false,
		dataType: 'text',
		contentType: "application/x-www-form-urlencoded",
		url: "/surveyKPI/background_report" + tzString,
		data: { report: JSON.stringify(reportObj) },
		success: function(data, status) {
			if(handleLogout(data)) {
				removeHourglass();
				alert(localise.set["msg_ds_s_r"]);
			}
		}, error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert(localise.set["msg_err_save"] + xhr.responseText);	// alerts htmlencode
				}
			}
		}
	});

}

function executeSurveyReport(oId) {

	var i;

	var reportName = localise.set["c_survey"];

	// Add the organisation name
	if(oId > 0 && globals.gLoggedInUser.orgs.length > 0) {
		for(i = 0; i < globals.gLoggedInUser.orgs.length; i++) {
			if(globals.gLoggedInUser.orgs[i].id == oId) {
				reportName += globals.gLoggedInUser.orgs[i].name + "_";
				break;
			}
		}
	}

	var reportObj = {
		report_type: 'survey',
		report_name: reportName,
		params: {
			oId: oId
		}
	}

	var tzString = globals.gTimezone ? "?tz=" + encodeURIComponent(globals.gTimezone) : "";

	addHourglass();
	$.ajax({
		type: "POST",
		cache: false,
		dataType: 'text',
		contentType: "application/x-www-form-urlencoded",
		url: "/surveyKPI/background_report" + tzString,
		data: { report: JSON.stringify(reportObj) },
		success: function(data, status) {
			if(handleLogout(data)) {
				removeHourglass();
				alert(localise.set["msg_ds_s_r"]);
			}
		}, error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert(localise.set["msg_err_save"] + xhr.responseText);	// alerts htmlencode
				}
			}
		}
	});

}
function executeAttendanceReport(oId) {

	var attendanceMsec = $('#attendanceDate').data("DateTimePicker").date(),
		d = new Date(attendanceMsec),
		day = d.getDate(),
		month = d.getMonth() + 1,
		year = d.getFullYear(),
		i;

	var reportName = localise.set["u_attendance"] + "_";

	// Add the organisation name
	if(oId > 0 && globals.gLoggedInUser.orgs.length > 0) {
		for(i = 0; i < globals.gLoggedInUser.orgs.length; i++) {
			if(globals.gLoggedInUser.orgs[i].id == oId) {
				reportName += globals.gLoggedInUser.orgs[i].name + "_";
				break;
			}
		}
	}

	reportName += "_" + year + "_" + month + "_" + day;
	reportName = reportName.replaceAll(' ', '_');

	var reportObj = {
		report_type: 'u_attendance',
		report_name: reportName,
		pId: 0,
		params: {
			oId: oId,
			month: month,
			year: year,
			day: day
		}
	}

	var tzString = globals.gTimezone ? "?tz=" + encodeURIComponent(globals.gTimezone) : "";

	addHourglass();
	$.ajax({
		type: "POST",
		cache: false,
		dataType: 'text',
		contentType: "application/x-www-form-urlencoded",
		url: "/surveyKPI/background_report" + tzString,
		data: { report: JSON.stringify(reportObj) },
		success: function(data, status) {
			removeHourglass();
			if(handleLogout(data)) {
				alert(localise.set["msg_ds_s_r"]);
			}
		}, error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert(localise.set["msg_err_save"] + " " + xhr.responseText);  // alerts htmlencode
				}
			}
		}
	});

}

/*
 * Decode escaped HTML
 * From https://stackoverflow.com/questions/1912501/unescape-html-entities-in-javascript
 */
function htmlDecode(input) {
	var doc = new DOMParser().parseFromString(input, "text/html");
	return doc.documentElement.textContent;
}

function htmlEncode(input) {
	if(input) {
		return $('<div>').text(input).html();
	}
}

/*
 * Get the list of users from the server
 */
function getEligibleUsers(sIdent, isNotification) {

	if(window.gTasks && window.gTasks.cache.eligibleUsers[sIdent]) {
		fillUsersList(isNotification, window.gTasks && window.gTasks.cache.eligibleUsers[sIdent]);
	} else if(sIdent) {
		addHourglass();
		$.ajax({
			url: "/surveyKPI/userList/survey/" + sIdent,
			dataType: 'json',
			cache: false,
			success: function (data) {
				removeHourglass();
				if(handleLogout(data)) {
					window.gTasks.cache.eligibleUsers[sIdent] = data;
					fillUsersList(isNotification, data);
				}
			},
			error: function (xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else if (err == 403) {
						return;  // Ignore errors where the survey cannot be found. The survey requested may be the global default current survey which may be out of date
					} else {
						alert(localise.set["error"] + ": " + err);
					}
				}
			}
		});
	}
}

/*
 * Fill a list with the users who can be selected
 */
function fillUsersList(isNotification, data) {
	var h = [],
		idx = -1,
		$elem = $('#user_to_assign');

	$elem.empty();

	h[++idx] = '<option value="_none">';
	h[++idx] = localise.set["c_none"];
	h[++idx] = '</option>';

	if (isNotification) {
		h[++idx] = '<option value="_submitter">';
		h[++idx] = localise.set["c_submitter"];
		h[++idx] = '</option>';

		h[++idx] = '<option value="_data">';
		h[++idx] = localise.set["t_ad"];
		h[++idx] = '</option>';
	}

	if (data && data.length > 0) {
		for (i = 0; i < data.length; i++) {
			h[++idx] = '<option value="';
			h[++idx] = data[i].ident;
			h[++idx] = '">';
			h[++idx] = htmlEncode(data[i].name);
			h[++idx] = '</option>';
		}
	}
	$elem.html(h.join(''));

	if (typeof gEligibleUser !== 'undefined') {
		$elem.val(gEligibleUser);
	}
}

/*
 * Return true if the passed in value is accepted by xlsFormConverter
 */
function isValidODKQuestionName(val) {

	var sqlCheck = /^[A-Za-z_][A-Za-z0-9_\-\.]*$/;
	return sqlCheck.test(val);
}

function isValidODKOptionName(val) {

	var sqlCheck = /^[A-Za-z0-9_@&\-\.\+\(\),%:\/ ]*$/;
	return sqlCheck.test(val);
}

/*
 * Check item names such as; a username or an organisation name for invalid characters
 */
function validGeneralName(val) {

	if(val.indexOf('<') >= 0 || val.indexOf('>') >= 0) {
		return false;
	}
	return true;
}

/*
 * Get the names of referenced questions in the passed in string
 */
function getReferenceNames(elem, refQuestions) {
	var names = [],
		reg = /\$\{[A-Za-z_][A-Za-z0-9_\-\.]*\}/g,
		i,
		name;

	if (elem) {
		names = elem.match(reg);
		if(names) {
			for(i = 0; i < names.length; i++) {
				if(names[i].length > 3) {
					name = names[i].substring(2, names[i].length - 1);		// Remove the curly brackets
					refQuestions[name] = {
						name: name,
						exists: false
					};
				}
			}
		}
	}
}

/*
 * Add an exists flag to each question in the references object
 */
function checkExistenceOfReferences(refQuestions, survey) {

	var refCount = 0,
		i = 0,
		j = 0,
		name,
		form;

	for (name in refQuestions) {
		if (refQuestions.hasOwnProperty(name)) {
			refCount++;
		}
	}

	if(refCount > 0) {

		for (i = 0; i < survey.forms.length; i++) {
			form = survey.forms[i];
			for (j = 0; j < form.questions.length; j++) {
				var otherItem = form.questions[j];
				var questionType = otherItem.type;
				if (!otherItem.deleted && !otherItem.soft_deleted && questionType !== "end group") {
					otherItem = form.questions[j];

					for (name in refQuestions) {
						if (refQuestions.hasOwnProperty(name)) {
							if (name === otherItem.name) {
								refQuestions[name].exists = true;
								break;
							}
						}
					}
				}
			}
		}

		// Check against preloads
		console.log("check against preloads");
		if (survey.meta) {
			for (i = 0; i < survey.meta.length; i++) {
				for (name in refQuestions) {
					if (name === survey.meta[i].name) {
						refQuestions[name].exists = true;
					}
				}
			}
		}
	}
	return refCount;
}

function checkLoggedIn(callback) {
	$.ajax({
		cache: false,
		url: "/authenticate/login.txt",
		success: function (data) {
			if(handleLogout(data)) {
				callback();
			}

		}, error: function (data, status) {
			if(handleLogout(data.responseText)) {
				alert(data);
			}
		}
	});
}

/*
 * Respond to a logged out redirect
 */
function handleLogout(data) {
	if(data) {
		if(    (data.code && data.code === 401)
			|| (data.status && data.status === 405)
			|| (data.status && data.status === 413)
			|| (typeof data === "string" && data.indexOf('"code": 401') >= 0)
			|| (typeof data === "string" && data.indexOf('Error: 401') >= 0)
			|| (typeof data === "string" && data.indexOf('Status 401 – Unauthorized') >= 0)
			|| (typeof data === "string" && data.indexOf('notloggedin.json') >= 0)
			|| (typeof data === "string" && data.toLowerCase().indexOf("method not allowed") >= 0)) {
				window.open("/login.html");
				return false;
		}

	} else if(data && (typeof data === "string" && data.indexOf('multilogon') >= 0)) {
		alert("Logon on another device detected - logging out");
		window.open("/dologout.html");
		return false;
	}
	return true;
}

/*
 * Load the sms numbers from the server
 */
function getOurNumbers() {

	var url="/surveyKPI/smsnumbers?org=true";
	addHourglass();
	$.ajax({
		url: url,
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				gNumbers = data;
				setOurNumbersList();
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					console.log("Error: Failed to get list of sms numbers: " + err);
				}
			}
		}
	});
}

function setOurNumbersList() {
	var i = 0;
	if(gNumbers && gNumbers.length > 0) {
		var $elem = $('#msg_our_nbr');
		var channel = $('#msg_channel').val();
		$elem.empty();
		for(i = 0; i < gNumbers.length; i++) {
			var n = gNumbers[i];
			if(n.channel === channel) {
				$elem.append(`<option value="${n.ourNumber}">${n.ourNumber} - ${n.channel} </option>`);
			}
		}
	}
}
;
define("app/common", function(){});

/*!
 * Modernizr v2.8.2
 * www.modernizr.com
 *
 * Copyright (c) Faruk Ates, Paul Irish, Alex Sexton
 * Available under the BSD and MIT licenses: www.modernizr.com/license/
 */
window.Modernizr=function(a,b,c){function d(a){t.cssText=a}function e(a,b){return d(x.join(a+";")+(b||""))}function f(a,b){return typeof a===b}function g(a,b){return!!~(""+a).indexOf(b)}function h(a,b){for(var d in a){var e=a[d];if(!g(e,"-")&&t[e]!==c)return"pfx"==b?e:!0}return!1}function i(a,b,d){for(var e in a){var g=b[a[e]];if(g!==c)return d===!1?a[e]:f(g,"function")?g.bind(d||b):g}return!1}function j(a,b,c){var d=a.charAt(0).toUpperCase()+a.slice(1),e=(a+" "+z.join(d+" ")+d).split(" ");return f(b,"string")||f(b,"undefined")?h(e,b):(e=(a+" "+A.join(d+" ")+d).split(" "),i(e,b,c))}function k(){o.input=function(c){for(var d=0,e=c.length;e>d;d++)E[c[d]]=!!(c[d]in u);return E.list&&(E.list=!(!b.createElement("datalist")||!a.HTMLDataListElement)),E}("autocomplete autofocus list placeholder max min multiple pattern required step".split(" ")),o.inputtypes=function(a){for(var d,e,f,g=0,h=a.length;h>g;g++)u.setAttribute("type",e=a[g]),d="text"!==u.type,d&&(u.value=v,u.style.cssText="position:absolute;visibility:hidden;",/^range$/.test(e)&&u.style.WebkitAppearance!==c?(q.appendChild(u),f=b.defaultView,d=f.getComputedStyle&&"textfield"!==f.getComputedStyle(u,null).WebkitAppearance&&0!==u.offsetHeight,q.removeChild(u)):/^(search|tel)$/.test(e)||(d=/^(url|email)$/.test(e)?u.checkValidity&&u.checkValidity()===!1:u.value!=v)),D[a[g]]=!!d;return D}("search tel url email datetime date month week time datetime-local number range color".split(" "))}var l,m,n="2.8.2",o={},p=!0,q=b.documentElement,r="modernizr",s=b.createElement(r),t=s.style,u=b.createElement("input"),v=":)",w={}.toString,x=" -webkit- -moz- -o- -ms- ".split(" "),y="Webkit Moz O ms",z=y.split(" "),A=y.toLowerCase().split(" "),B={svg:"http://www.w3.org/2000/svg"},C={},D={},E={},F=[],G=F.slice,H=function(a,c,d,e){var f,g,h,i,j=b.createElement("div"),k=b.body,l=k||b.createElement("body");if(parseInt(d,10))for(;d--;)h=b.createElement("div"),h.id=e?e[d]:r+(d+1),j.appendChild(h);return f=["&#173;",'<style id="s',r,'">',a,"</style>"].join(""),j.id=r,(k?j:l).innerHTML+=f,l.appendChild(j),k||(l.style.background="",l.style.overflow="hidden",i=q.style.overflow,q.style.overflow="hidden",q.appendChild(l)),g=c(j,a),k?j.parentNode.removeChild(j):(l.parentNode.removeChild(l),q.style.overflow=i),!!g},I=function(b){var c=a.matchMedia||a.msMatchMedia;if(c)return c(b)&&c(b).matches||!1;var d;return H("@media "+b+" { #"+r+" { position: absolute; } }",function(b){d="absolute"==(a.getComputedStyle?getComputedStyle(b,null):b.currentStyle).position}),d},J=function(){function a(a,e){e=e||b.createElement(d[a]||"div"),a="on"+a;var g=a in e;return g||(e.setAttribute||(e=b.createElement("div")),e.setAttribute&&e.removeAttribute&&(e.setAttribute(a,""),g=f(e[a],"function"),f(e[a],"undefined")||(e[a]=c),e.removeAttribute(a))),e=null,g}var d={select:"input",change:"input",submit:"form",reset:"form",error:"img",load:"img",abort:"img"};return a}(),K={}.hasOwnProperty;m=f(K,"undefined")||f(K.call,"undefined")?function(a,b){return b in a&&f(a.constructor.prototype[b],"undefined")}:function(a,b){return K.call(a,b)},Function.prototype.bind||(Function.prototype.bind=function(a){var b=this;if("function"!=typeof b)throw new TypeError;var c=G.call(arguments,1),d=function(){if(this instanceof d){var e=function(){};e.prototype=b.prototype;var f=new e,g=b.apply(f,c.concat(G.call(arguments)));return Object(g)===g?g:f}return b.apply(a,c.concat(G.call(arguments)))};return d}),C.flexbox=function(){return j("flexWrap")},C.flexboxlegacy=function(){return j("boxDirection")},C.canvas=function(){var a=b.createElement("canvas");return!(!a.getContext||!a.getContext("2d"))},C.canvastext=function(){return!(!o.canvas||!f(b.createElement("canvas").getContext("2d").fillText,"function"))},C.webgl=function(){return!!a.WebGLRenderingContext},C.touch=function(){var c;return"ontouchstart"in a||a.DocumentTouch&&b instanceof DocumentTouch?c=!0:H(["@media (",x.join("touch-enabled),("),r,")","{#modernizr{top:9px;position:absolute}}"].join(""),function(a){c=9===a.offsetTop}),c},C.geolocation=function(){return"geolocation"in navigator},C.postmessage=function(){return!!a.postMessage},C.websqldatabase=function(){return!!a.openDatabase},C.indexedDB=function(){return!!j("indexedDB",a)},C.hashchange=function(){return J("hashchange",a)&&(b.documentMode===c||b.documentMode>7)},C.history=function(){return!(!a.history||!history.pushState)},C.draganddrop=function(){var a=b.createElement("div");return"draggable"in a||"ondragstart"in a&&"ondrop"in a},C.websockets=function(){return"WebSocket"in a||"MozWebSocket"in a},C.rgba=function(){return d("background-color:rgba(150,255,150,.5)"),g(t.backgroundColor,"rgba")},C.hsla=function(){return d("background-color:hsla(120,40%,100%,.5)"),g(t.backgroundColor,"rgba")||g(t.backgroundColor,"hsla")},C.multiplebgs=function(){return d("background:url(https://),url(https://),red url(https://)"),/(url\s*\(.*?){3}/.test(t.background)},C.backgroundsize=function(){return j("backgroundSize")},C.borderimage=function(){return j("borderImage")},C.borderradius=function(){return j("borderRadius")},C.boxshadow=function(){return j("boxShadow")},C.textshadow=function(){return""===b.createElement("div").style.textShadow},C.opacity=function(){return e("opacity:.55"),/^0.55$/.test(t.opacity)},C.cssanimations=function(){return j("animationName")},C.csscolumns=function(){return j("columnCount")},C.cssgradients=function(){var a="background-image:",b="gradient(linear,left top,right bottom,from(#9f9),to(white));",c="linear-gradient(left top,#9f9, white);";return d((a+"-webkit- ".split(" ").join(b+a)+x.join(c+a)).slice(0,-a.length)),g(t.backgroundImage,"gradient")},C.cssreflections=function(){return j("boxReflect")},C.csstransforms=function(){return!!j("transform")},C.csstransforms3d=function(){var a=!!j("perspective");return a&&"webkitPerspective"in q.style&&H("@media (transform-3d),(-webkit-transform-3d){#modernizr{left:9px;position:absolute;height:3px;}}",function(b){a=9===b.offsetLeft&&3===b.offsetHeight}),a},C.csstransitions=function(){return j("transition")},C.fontface=function(){var a;return H('@font-face {font-family:"font";src:url("https://")}',function(c,d){var e=b.getElementById("smodernizr"),f=e.sheet||e.styleSheet,g=f?f.cssRules&&f.cssRules[0]?f.cssRules[0].cssText:f.cssText||"":"";a=/src/i.test(g)&&0===g.indexOf(d.split(" ")[0])}),a},C.generatedcontent=function(){var a;return H(["#",r,"{font:0/0 a}#",r,':after{content:"',v,'";visibility:hidden;font:3px/1 a}'].join(""),function(b){a=b.offsetHeight>=3}),a},C.video=function(){var a=b.createElement("video"),c=!1;try{(c=!!a.canPlayType)&&(c=new Boolean(c),c.ogg=a.canPlayType('video/ogg; codecs="theora"').replace(/^no$/,""),c.h264=a.canPlayType('video/mp4; codecs="avc1.42E01E"').replace(/^no$/,""),c.webm=a.canPlayType('video/webm; codecs="vp8, vorbis"').replace(/^no$/,""))}catch(d){}return c},C.audio=function(){var a=b.createElement("audio"),c=!1;try{(c=!!a.canPlayType)&&(c=new Boolean(c),c.ogg=a.canPlayType('audio/ogg; codecs="vorbis"').replace(/^no$/,""),c.mp3=a.canPlayType("audio/mpeg;").replace(/^no$/,""),c.wav=a.canPlayType('audio/wav; codecs="1"').replace(/^no$/,""),c.m4a=(a.canPlayType("audio/x-m4a;")||a.canPlayType("audio/aac;")).replace(/^no$/,""))}catch(d){}return c},C.localstorage=function(){try{return localStorage.setItem(r,r),localStorage.removeItem(r),!0}catch(a){return!1}},C.sessionstorage=function(){try{return sessionStorage.setItem(r,r),sessionStorage.removeItem(r),!0}catch(a){return!1}},C.webworkers=function(){return!!a.Worker},C.applicationcache=function(){return!!a.applicationCache},C.svg=function(){return!!b.createElementNS&&!!b.createElementNS(B.svg,"svg").createSVGRect},C.inlinesvg=function(){var a=b.createElement("div");return a.innerHTML="<svg/>",(a.firstChild&&a.firstChild.namespaceURI)==B.svg},C.smil=function(){return!!b.createElementNS&&/SVGAnimate/.test(w.call(b.createElementNS(B.svg,"animate")))},C.svgclippaths=function(){return!!b.createElementNS&&/SVGClipPath/.test(w.call(b.createElementNS(B.svg,"clipPath")))};for(var L in C)m(C,L)&&(l=L.toLowerCase(),o[l]=C[L](),F.push((o[l]?"":"no-")+l));return o.input||k(),o.addTest=function(a,b){if("object"==typeof a)for(var d in a)m(a,d)&&o.addTest(d,a[d]);else{if(a=a.toLowerCase(),o[a]!==c)return o;b="function"==typeof b?b():b,"undefined"!=typeof p&&p&&(q.className+=" "+(b?"":"no-")+a),o[a]=b}return o},d(""),s=u=null,function(a,b){function c(a,b){var c=a.createElement("p"),d=a.getElementsByTagName("head")[0]||a.documentElement;return c.innerHTML="x<style>"+b+"</style>",d.insertBefore(c.lastChild,d.firstChild)}function d(){var a=s.elements;return"string"==typeof a?a.split(" "):a}function e(a){var b=r[a[p]];return b||(b={},q++,a[p]=q,r[q]=b),b}function f(a,c,d){if(c||(c=b),k)return c.createElement(a);d||(d=e(c));var f;return f=d.cache[a]?d.cache[a].cloneNode():o.test(a)?(d.cache[a]=d.createElem(a)).cloneNode():d.createElem(a),!f.canHaveChildren||n.test(a)||f.tagUrn?f:d.frag.appendChild(f)}function g(a,c){if(a||(a=b),k)return a.createDocumentFragment();c=c||e(a);for(var f=c.frag.cloneNode(),g=0,h=d(),i=h.length;i>g;g++)f.createElement(h[g]);return f}function h(a,b){b.cache||(b.cache={},b.createElem=a.createElement,b.createFrag=a.createDocumentFragment,b.frag=b.createFrag()),a.createElement=function(c){return s.shivMethods?f(c,a,b):b.createElem(c)},a.createDocumentFragment=Function("h,f","return function(){var n=f.cloneNode(),c=n.createElement;h.shivMethods&&("+d().join().replace(/[\w\-]+/g,function(a){return b.createElem(a),b.frag.createElement(a),'c("'+a+'")'})+");return n}")(s,b.frag)}function i(a){a||(a=b);var d=e(a);return!s.shivCSS||j||d.hasCSS||(d.hasCSS=!!c(a,"article,aside,dialog,figcaption,figure,footer,header,hgroup,main,nav,section{display:block}mark{background:#FF0;color:#000}template{display:none}")),k||h(a,d),a}var j,k,l="3.7.0",m=a.html5||{},n=/^<|^(?:button|map|select|textarea|object|iframe|option|optgroup)$/i,o=/^(?:a|b|code|div|fieldset|h1|h2|h3|h4|h5|h6|i|label|li|ol|p|q|span|strong|style|table|tbody|td|th|tr|ul)$/i,p="_html5shiv",q=0,r={};!function(){try{var a=b.createElement("a");a.innerHTML="<xyz></xyz>",j="hidden"in a,k=1==a.childNodes.length||function(){b.createElement("a");var a=b.createDocumentFragment();return"undefined"==typeof a.cloneNode||"undefined"==typeof a.createDocumentFragment||"undefined"==typeof a.createElement}()}catch(c){j=!0,k=!0}}();var s={elements:m.elements||"abbr article aside audio bdi canvas data datalist details dialog figcaption figure footer header hgroup main mark meter nav output progress section summary template time video",version:l,shivCSS:m.shivCSS!==!1,supportsUnknownElements:k,shivMethods:m.shivMethods!==!1,type:"default",shivDocument:i,createElement:f,createDocumentFragment:g};a.html5=s,i(b)}(this,b),o._version=n,o._prefixes=x,o._domPrefixes=A,o._cssomPrefixes=z,o.mq=I,o.hasEvent=J,o.testProp=function(a){return h([a])},o.testAllProps=j,o.testStyles=H,o.prefixed=function(a,b,c){return b?j(a,b,c):j(a,"pfx")},q.className=q.className.replace(/(^|\s)no-js(\s|$)/,"$1$2")+(p?" js "+F.join(" "):""),o}(this,this.document);

define("modernizr", function(){});

/**
 * @license RequireJS i18n 2.0.4 Copyright (c) 2010-2012, The Dojo Foundation All Rights Reserved.
 * Available via the MIT or new BSD license.
 * see: http://github.com/requirejs/i18n for details
 */
/*jslint regexp: true */
/*global require: false, navigator: false, define: false */

/**
 * This plugin handles i18n! prefixed modules. It does the following:
 *
 * 1) A regular module can have a dependency on an i18n bundle, but the regular
 * module does not want to specify what locale to load. So it just specifies
 * the top-level bundle, like "i18n!nls/colors".
 *
 * This plugin will load the i18n bundle at nls/colors, see that it is a root/master
 * bundle since it does not have a locale in its name. It will then try to find
 * the best match locale available in that master bundle, then request all the
 * locale pieces for that best match locale. For instance, if the locale is "en-us",
 * then the plugin will ask for the "en-us", "en" and "root" bundles to be loaded
 * (but only if they are specified on the master bundle).
 *
 * Once all the bundles for the locale pieces load, then it mixes in all those
 * locale pieces into each other, then finally sets the context.defined value
 * for the nls/colors bundle to be that mixed in locale.
 *
 * 2) A regular module specifies a specific locale to load. For instance,
 * i18n!nls/fr-fr/colors. In this case, the plugin needs to load the master bundle
 * first, at nls/colors, then figure out what the best match locale is for fr-fr,
 * since maybe only fr or just root is defined for that locale. Once that best
 * fit is found, all of its locale pieces need to have their bundles loaded.
 *
 * Once all the bundles for the locale pieces load, then it mixes in all those
 * locale pieces into each other, then finally sets the context.defined value
 * for the nls/fr-fr/colors bundle to be that mixed in locale.
 */
(function () {
    'use strict';

    //regexp for reconstructing the master bundle name from parts of the regexp match
    //nlsRegExp.exec("foo/bar/baz/nls/en-ca/foo") gives:
    //["foo/bar/baz/nls/en-ca/foo", "foo/bar/baz/nls/", "/", "/", "en-ca", "foo"]
    //nlsRegExp.exec("foo/bar/baz/nls/foo") gives:
    //["foo/bar/baz/nls/foo", "foo/bar/baz/nls/", "/", "/", "foo", ""]
    //so, if match[5] is blank, it means this is the top bundle definition.
    var nlsRegExp = /(^.*(^|\/)nls(\/|$))([^\/]*)\/?([^\/]*)/;

    //Helper function to avoid repeating code. Lots of arguments in the
    //desire to stay functional and support RequireJS contexts without having
    //to know about the RequireJS contexts.
    function addPart(locale, master, needed, toLoad, prefix, suffix) {
        if (master[locale]) {
            needed.push(locale);
            if (master[locale] === true || master[locale] === 1) {
                toLoad.push(prefix + locale + '/' + suffix);
            }
        }
    }

    function addIfExists(req, locale, toLoad, prefix, suffix) {
        var fullName = prefix + locale + '/' + suffix;
        if (require._fileExists(req.toUrl(fullName + '.js'))) {
            toLoad.push(fullName);
        }
    }

    /**
     * Simple function to mix in properties from source into target,
     * but only if target does not already have a property of the same name.
     * This is not robust in IE for transferring methods that match
     * Object.prototype names, but the uses of mixin here seem unlikely to
     * trigger a problem related to that.
     */
    function mixin(target, source, force) {
        var prop;
        for (prop in source) {
            if (source.hasOwnProperty(prop) && (!target.hasOwnProperty(prop) || force)) {
                target[prop] = source[prop];
            } else if (typeof source[prop] === 'object') {
                if (!target[prop] && source[prop]) {
                    target[prop] = {};
                }
                mixin(target[prop], source[prop], force);
            }
        }
    }

    define('i18n',['module'], function (module) {
        var masterConfig = module.config ? module.config() : {};

        return {
            version: '2.0.4',
            /**
             * Called when a dependency needs to be loaded.
             */
            load: function (name, req, onLoad, config) {
                config = config || {};

                if (config.locale) {
                    masterConfig.locale = config.locale;
                }

                var masterName,
                    match = nlsRegExp.exec(name),
                    prefix = match[1],
                    locale = match[4],
                    suffix = match[5],
                    parts = locale.split("-"),
                    toLoad = [],
                    value = {},
                    i, part, current = "";

                //If match[5] is blank, it means this is the top bundle definition,
                //so it does not have to be handled. Locale-specific requests
                //will have a match[4] value but no match[5]
                if (match[5]) {
                    //locale-specific bundle
                    prefix = match[1];
                    masterName = prefix + suffix;
                } else {
                    //Top-level bundle.
                    masterName = name;
                    suffix = match[4];
                    locale = masterConfig.locale;
                    if (!locale) {
                        locale = masterConfig.locale =
                            typeof navigator === "undefined" ? "root" :
                            (navigator.language ||
                             navigator.userLanguage || "root").toLowerCase();
                    }
                    parts = locale.split("-");
                }

                if (config.isBuild) {
                    //Check for existence of all locale possible files and
                    //require them if exist.
                    toLoad.push(masterName);
                    addIfExists(req, "root", toLoad, prefix, suffix);
                    for (i = 0; i < parts.length; i++) {
                        part = parts[i];
                        current += (current ? "-" : "") + part;
                        addIfExists(req, current, toLoad, prefix, suffix);
                    }

                    req(toLoad, function () {
                        onLoad();
                    });
                } else {
                    //First, fetch the master bundle, it knows what locales are available.
                    req([masterName], function (master) {
                        //Figure out the best fit
                        var needed = [],
                            part;

                        //Always allow for root, then do the rest of the locale parts.
                        addPart("root", master, needed, toLoad, prefix, suffix);
                        for (i = 0; i < parts.length; i++) {
                            part = parts[i];
                            current += (current ? "-" : "") + part;
                            addPart(current, master, needed, toLoad, prefix, suffix);
                        }

                        //Load all the parts missing.
                        req(toLoad, function () {
                            var i, partBundle, part;
                            for (i = needed.length - 1; i > -1 && needed[i]; i--) {
                                part = needed[i];
                                partBundle = master[part];
                                if (partBundle === true || partBundle === 1) {
                                    partBundle = req(prefix + part + '/' + suffix);
                                }
                                mixin(value, partBundle);
                            }

                            //All done, notify the loader.
                            onLoad(value);
                        });
                    });
                }
            }
        };
    });
}());

define('lang_location/nls/lang',{
	"root": true,
	"ar": true,
	"fr": true,
	"pt": true,
	"es": true,
	"hi": true,
	"uk": true
});

define('lang_location/nls/root/lang',{
    "m_home": "Home",
    "m_admin": "Admin",
    "m_tm": "Forms",
    "m_analysis": "Analysis",
    "m_report": "Report",
    "m_reports": "Reports",
    "m_reports_def": "Default Reports",
    "m_reports_org": "Organisation Reports",
    "m_reports_leg": "Legacy Reports",
    "m_reports_new": "New Reports",
    "m_reports_public": "Public Reports",
    "m_refresh": "Refresh",
    "m_email_unsent": "Email Unsent",
	"m_email_gen": "Generate Links",
    "m_export": "Export",
    "m_export_media": "Export with Media",
    "m_backup": "Backup",
    "m_backup_media": "Backup with Media",
    "m_discuss": "Discuss",
    "m_modify": "Modify",
    "m_web_forms": "Web Forms",
    "m_docs": "Documentation",
    "m_monitor": "Monitoring",
    "m_user": "Users",
    "m_assign": "Tasks",
    "m_query": "Queries",
    "m_help": "Help",
    "m_train": "Training",
    "m_forgot_password": "Forgot Password",
    "m_register": "Register",
    "m_locations": "Locations",
    "m_section": "Section",
    "m_notify": "Notifications",
    "m_view": "View",
    "m_attach": "Load Attachments",
    "m_modules": "Modules",
    "m_lr": "Local Reports",
    "m_open": "Open",
    "m_new": "New",
    "m_import": "Import",
    "m_calculations": "Calculations",
    "m_resources": "Shared Resources",
    "m_review": "Review",
    "m_mf": "Managed Forms",
    "m_br": "Browse Results",
    "m_manage": "Manage",
    "m_dash": "Dashboard",
    "m_console": "Console",
    "m_s_m_f": "Setup Managed Forms",
    "m_task_m": "Task Management",
    "m_user_profile": "User Profile",
    "m_login": "Login",
    "m_logs": "Logs",
    "m_info": "Info",
    "m_os": "Oversight",
    "m_billing": "Billing",
    "m_rates": "Rates",
    "m_billing_server": "Server Billing",
    "m_billing_orgs": "Organisations Billing",
    "m_bill_level": "Bill Level",

    // Smap Server
    "ss_welcome": "Smap Server",
    "ss_welcome2": "Changes in this release",
    "ss_ft": "Get Field Task",
    "ss_su": "Get Smap Uploader",

    // Common
    "c_alert": "Alert",
    "c_alerts": "Alerts",
    "c_apply": "Apply",
    "c_api": "API",
    "c_api_builder": "API Builder",
    "c_audit": "Audit",
    "c_auto": "Auto",
    "c_before": "Before",
    "c_bundle": "Bundle",
    "c_cases": "Cases",
    "c_channel": "Channel",
    "c_closed": "Closed",
    "c_cm": "Case Management",
    "c_crit": "Criticality",
    "c_default": "Default",
    "c_desc": "Description",
    "c_fingerprint": "Fingerprint",
    "c_form": "Form",
    "c_forms": "Forms",
    "c_forward": "Forward",
    "c_iid": "Instance Id",
    "c_link": "Link",
    "c_links": "Links",
    "c_logon": "Server Logon",
    "c_moved" : "Moved",
    "c_opened": "Opened",
    "c_periodic": "Periodic",
    "c_project": "Project",
    "c_sub_form": "Sub Form",
    "c_threshold": "Threshold",
    "c_url": "URL",
    "c_webhook": "Web Hook",
    "c_forwarded": "Forwarded",
    "c_name": "Name",
    "c_company_name": "Company Name",
    "c_company_addr": "Company Address",
    "c_company_phone": "Company Phone",
    "c_company_email": "Company Email",
    "c_conversation": "Message Conversation",
    "c_type": "Type",
    "c_file": "File",
    "c_host": "Host",
    "c_rank": "Rank",
    "c_report": "Report",
    "c_user": "User",
    "c_self": "Self",
    "c_sunday": "Sunday",
    "c_monday": "Monday",
    "c_tuesday": "Tuesday",
    "c_wednesday": "Wednesday",
    "c_thursday": "Thursday",
    "c_friday": "Friday",
    "c_saturday": "Saturday",
    "c_vertical": "Vertical",
    "c_projects": "Projects",
    "c_email": "Email",
    "c_escalate": "Escalate",
    "c_emails": "Emails",
    "c_sms": "SMS",
    "c_document": "Document",
    "c_twitter": "Twitter",
    "c_lang": "Language",
    "c_langs": "Languages",
    "c_pulldata": "Pulldata",
    "c_pulldata_r": "Repeating Pulldata",
    "c_orientation": "Orientation",
    "c_password": "Password",
    "c_c_p": "Confirm Password",
    "c_n_p": "New Password",
    "c_c_d": "Confirm Delete",
    "c_r_p": "Reset Password",
    "c_change_p": "Change Password",
    "c_ak": "API key",
    "c_gak": "Create API key",
    "c_rak": "Replace API key",
    "c_cftk": "Create FieldTask Key",
    "c_rftk": "Replace FieldTask Key",
    "c_reset": "Reset",
    "c_en": "Enable",
    "c_dis": "Disable",
    "c_loc_f": "Local Form:",
    "c_rem_f": "Remote Form",
    "c_rem_u": "Remote User",
    "c_enabled": "Enabled",
    "c_edit": "Edit",
    "c_del": "Delete",
    "c_undel": "Undelete",
    "c_del_data": "Delete Data",
    "c_res_data": "Restore Data",
    "c_archive_data": "Archive Data",
    "c_download": "Download",
    "c_upload": "Upload",
    "c_done": "Done",
    "c_cancel": "Cancel",
    "c_save": "Save",
    "c_saved": "Saved",
    "c_submit": "Submit",
    "c_submitted": "Submitted",
    "c_test": "Test",
    "c_validate": "Validate",
    "c_close": "Close",
	"c_likert": "Likert",
    "c_logout": "Logout",
    "c_block": "Block",
    "c_blocked": "Blocked",
    "c_graph": "Graph",
    "c_map": "Map",
    "c_chart": "Chart",
    "c_charts": "Charts",
    "c_maps": "Maps",
    "c_map_layers": "Map Layers",
    "c_table": "Table",
    "c_tables": "Tables",
    "c_images": "Images",
    "c_title": "Title",
    "c_comment": "Comment",
    "c_country": "Country",
    "c_region": "Region",
    "c_district": "District",
    "c_community": "Community",
    "c_node": "Node",
    "c_relation": "Relation",
    "c_from": "From",
    "c_subject": "Subject",
    "c_to": "To",
    "c_properties": "Properties",
    "c_question": "Question",
    "c_questions": "Questions",
    "c_choices": "Choices",
    "c_include": "Include",
    "c_unique": "Unique",
    "c_survey": "Survey",
    "c_survey_tg": "Survey / Task Group / Report",
    "c_date": "Date",
    "c_dateq": "Date Question",
    "c_none": "None",
    "c_period": "Period",
    "c_hour": "Hour",
    "c_day": "Day",
    "c_week": "Week",
    "c_month": "Month",
    "c_year": "Year",
    "c_daily": "Daily",
    "c_weekly": "Weekly",
    "c_monthly": "Monthly",
    "c_quarterly": "Quarterly",
    "c_yearly": "Yearly",
    "c_matches": "matches",
    "c_filter": "Filter",
    "c_inclusive": "Inclusive",
    "c_value": "Value",
    "c_clear": "Clear",
    "c_options": "Options",
    "c_data": "Data",
    "c_layers": "Layers",
    "c_settings": "Settings",
    "c_start": "Start",
	"c_end": "End",
	"c_step": "Step",
    "c_back": "Back",
    "c_target": "Target",
    "c_trigger": "Trigger",
    "c_content": "Content",
    "c_today": "Today",
    "c_attach": "Attach",
    "c_details": "Details",
    "c_action": "Action",
    "c_actions": "Actions",
    "c_mapid": "Map Id",
    "c_zoom": "Zoom Levels",
    "c_zoom_data": "Zoom to Data",
    "c_vector_data": "Vector Data",
    "c_vector_style": "Vector Style",
    "c_style": "Style",
    "c_portrait": "Portrait",
    "c_landscape": "Landscape",
    "c_tools": "Tools",
    "c_required": "Make Required",
    "c_not_required": "Make Not Required",
    "c_del_sel": "Delete Selected",
    "c_assign_sel": "Set Selected as Assigned",
    "c_print": "Print",
    "c_repeat": "Repeat",
	"c_persist": "Persist",
    "c_repeats": "Repeats",
    "c_scheduled": "Scheduled",
    "c_t_c": "terms and conditions",
    "c_agree": "I agree to the",
    "c_sa": "Select All",
    "c_select": "Select",
    "c_status": "Status",
    "c_coords": "Coordinates",
    "c_task": "Task",
    "c_rec": "(Recommended)",
    "c_source": "Source",
    "c_show": "Show",
    "c_show_q": "Show Question",
    "c_hide": "Hide",
    "c_total": "Total",
    "c_totals": "Totals",
    "c_instances": "Instances",
    "c_device": "Device",
    "c_success": "Success",
    "c_errors": "Errors",
    "c_error": "Error",
    "c_warning": "Warning",
    "c_lt": "Local Time",
    "c_standard": "Standard",
    "c_simserial": "Sim Serial Number",
    "c_location": "Location",
    "c_ident": "Identifier",
    "c_version": "Version",
    "c_deleted": "Deleted",
    "c_yes": "Yes",
    "c_no": "No",
    "c_reason": "Reason",
    "c_complete": "Complete",
    "c_record": "Record",
    "c_records": "Records",
    "c_of": "of",
    "c_media": "Media",
    "c_resource": "Media / CSV",
    "c_change": "Change",
    "c_changes": "Changes",
    "c_templates": "PDF Templates",
    "c_template": "PDF Template",
    "c_meta": "Meta Items",
    "c_calc": "Calculations",
    "c_trans": "Translate",
    "c_org": "Organisation",
    "c_orgs": "Organisations",
    "c_ent": "Enterprise",
    "c_all_s": "All Surveys",
    "c_all": "All",
    "c_event": "Event",
    "c_note": "Note",
    "c_server": "Server",
    "c_cols": "Columns",
    "c_cols_packed": "Packed Columns",
    "c_add": "Add",
    "c_create": "Create",
    "c_generate": "Generate",
    "c_imp_xls": "Import Spreadsheet",
    "c_spread": "Spreadsheets",
    "c_format": "Format",
    "c_xlstype": "Excel Format",
    "c_roles": "Roles",
    "c_s_roles": "Survey Roles",
    "c_role": "Role",
    "c_id": "Id",
    "c_pivot": "Pivot",
    "c_insert": "Insert",
    "c_tz": "Time Zone",
    "c_undef": "Not Defined",
    "c_license": "License Number",
    "c_in": "in",
	"c_notin": "not in",
    "c_rev": "Reverse",
    "c_for": "For",
    "c_normal": "Normal",
    "c_heatmap": "Heatmap",
    "c_auth": "Authorized",
    "c_merge": "Merge",
    "c_append": "Append",
    "c_discard": "Discard",
    "c_keys": "Keys",
    "c_counts": "Counts",
    "c_unsav": "You have unsaved changes are you sure you want to leave?",
    "c_off": "Off",
    "c_on": "On",
    "c_rule": "Rule",
    "c_replace": "Replace",
    "c_retry": "Retry",
    "c_alink": "Anonymous Link",
    "c_excel": "Excel",
    "c_word": "Word",
    "c_csv": "CSV",
    "c_json": "JSON",
    "c_geojson": "GeoJSON",
    "c_osm": "Openstreetmap",
    "c_shape": "Shape",
    "c_kml": "KML / KMZ",
    "c_vrt": "VRT / CSV",
    "c_stata": "Stata / CSV",
    "c_spss": "SPSS / CSV",
    "c_pdf": "PDF",
    "c_phone": "Phone Number",
	"c_photo": "Photograph",
    "c_rep_type": "Sub Form",
    "c_cl": "Copy Link",
    "c_ck": "Copy Key",
    "c_codl": "Copy OData Link",
    "c_cb": "Copy to clipboard",
    "c_c": "Copied",
    "c_quantity": "Quantity",
    "c_after": "After",
    "c_units": "Units",
    "c_duration": "Duration",
    "c_msg": "Message",
    "c_completed": "Completed",
    "c_unsubscribed": "Unsubscribed",
    "c_unsubscribe": "Unsubscribe",
    "c_subscribe": "Subscribe",
    "c_expired": "Expired",
	"c_subscriberid": "Subscriber Id",
    "c_unsent": "Email not sent",
    "c_pending": "Pending",
    "c_ok": "OK",
    "c_move": "Move",
    "c_parent_form": "Parent Form",
    "c_child_form": "Child Form (Repeats)",
    "c_other": "Other",
    "c_random": "Randomize",
    "c_annotate": "Annotate",
    "c_draw": "Draw",
    "c_field_list": "Field List",
    "c_table_list": "Table List",
    "c_recover": "Recover Previous File",
    "c_lat": "Latitude",
    "c_lon": "Longitude",
    "c_browse": "Browse",
    "c_reject": "Reject",
    "c_history": "History",
    "c_events": "Events",
    "c_resend": "Resend",
    "c_late": "Late",
    "c_custom": "Custom",
    "c_access": "Access",
    "c_person": "Person",
    "c_people": "People",
    "c_contacts": "Contacts",
    "c_teams": "Teams",
    "c_mailouts": "Mail outs",
    "c_mailout": "Mail out",
    "c_campaigns": "Campaigns",
    "c_campaign": "Campaign",
    "c_manual": "Manual",
    "c_submitter": "Submitter",
    "c_code": "Code",
    "c_dirn": "Direction",
    "c_rtl": "RTL",
    "c_current": "Current",
    "c_quality": "Quality",
    "c_low": "Low",
    "c_external": "External",
    "c_preview": "Preview",
    "c_size": "Size",
    "c_meters": "Meters",
    "c_accuracy": "Accuracy",

    // File Types
    "ft": "File Type",
    "ft_xls": "XLS Form",
    "ft_xls_orig": "Original file",
    "ft_xml": "XML Form",
    "ft_codebook": "Codebook",
    "ft_pdf": "Default Report Template",
    "ft_spss": "SPS File (SPSS)",

    // Template Management
    "tm_s_del": "Show deleted forms",
    "tm_s_block": "Show blocked forms",
    "tm_d_forms": "Delete Forms",
    "tm_r_forms": "Restore Forms",
    "tm_e_forms": "Erase Forms",
    "tm_c_form": "Create a Form",
    "tm_c_form_rep": "Replace a Form",
    "tm_c_sr_rep": "Replace a shared resource file",
    "tm_c_xls": "Using the Spreadsheet editor",
    "tm_g_temp": "Download Template",
    "tm_g_new": "Create new Form",
    "tm_g_open": "Open a Form",
    "tm_g_temp_i1": "Download a template to create a new form on your personal computer. When you have finished, upload it to the server by clicking on the upload button below. You will first need to select the XLS file containing the form you have created and give it a name.",
    "tm_g_temp_i3": "Reference Guide",
    "tm_ul_f": "Upload the spreadsheet",
    "tm_ul_sr": "Upload the shared resource file",
    "tm_ul_sl": "Upload the shared locations file",
    "tm_si": "Specify survey identifier",
    "tm_up": "Upload Template",
    "tm_add": "Add Template",
    "tm_ups": "Upload Form",
    "tm_add_for": "Add Forward",
    "tm_ref_f": "Refresh Form List",
    "tm_o_u": "Original Upload Page",
    "tm_gw": "A bundle should only be specified for surveys that share tables, if in doubt select none",

    // Analysis
    "a_ref_rate": "Analysis auto refresh interval",
    "a_exp_title": "Export Results",
    "a_exp_leg1": "Select a survey",
    "a_locn": "Select a location question",
    "a_exp_leg2": "Output format:",
    "a_exp_notes": "Export notes:",
    "a_exp_meta": "Include Meta Data:",
    "a_exp_odata2": "Restrict Odata types to those supported by OData 2:",
    "a_exp_split": "Split lat / lon:",
    "a_exp_merge": "Compress select multiples:",
    "a_sel_ways": "Select ways (if any):",
    "a_sel_forms": "Select forms to include:",
    "a_sel_query": "There are no queries to export.  You will need to create one using the queries menu in the dashboard module",
    "a_sel_model": "Edit survey model",
    "a_type": "Chart Type",
    "a_showon": "Show on",
    "a_ss": "Select survey",
    "a_sq": "Select question",
    "a_gb": "Group by",
    "a_ts": "Time series",
    "a_sl": "Show Label",
    "a_g_gps": "GPS coordinates within region",
    "a_fbv": "Filter by value",
    "a_from_date": "From date",
    "a_to_date": "To date",
    "a_fq": "Filter question",
    "a_res": "Please provide a reason",
    "a_res_5": "Please provide a reason at least 5 characters long",
    "a_mb": "Marked Bad",
    "a_oo": "out of",
    "a_cm": "Cluster Members",
    "a_dv": "Show Details",
    "a_rd": "Report Definition",
    "a_rp": "Report Parameters",
    "a_lcr": "Launch Custom Report",
    "a_embed": "Embed Images in Excel File",
    "a_hxl": "HXL",
    "a_query": "Select a query",
    "a_fn": "Function",
    "a_ni": "You need to enable loading tasks for this form in the form settings in the editor page",
    "a_nx": "New XLSX",
    "a_lx": "Legacy XLSX",
	"a_lh": "Legacy XLS / HTML",
	"a_ua": "User Activity",
    "a_ul": "User Locations",
    "a_tc": "This Chart",
    "a_start_pk": "First Primary Key",
    "a_end_pk": "Last Primary Key",
    "a_inc_ro": "Include Read Only",

    "a_dd_sl": "Shows the results",
    "a_dd_ocha_map": "Shows the location of responses to %s1 in the question '%s2' in survey '%s3'.",
	"a_dd_percent_map": "Shows the percentage of responses that selected %s1 in the question '%s2' in survey '%s3'.",
	"a_dd_average_map": "Shows the average of responses to %s1 in the question '%s2' in survey '%s3'.",
	"a_dd_count_map": "Shows the count of responses to %s1 in the question '%s2' in survey '%s3'.",
	"a_dd_total_map": "Shows the total of responses to %s1 in the question '%s2' in survey '%s3'.",
	"a_dd_none_map": "Shows the responses to %s1 in the question '%s2' in survey '%s3'.",
	"a_dd_ocha_table": "Shows the location of responses to each choice in the question '%s2' in survey '%s3'.",
	"a_dd_percent_table": "Shows the percentage of responses that selected each choice in the question '%s2' in survey '%s3'.",
	"a_dd_average_table": "Shows the average of responses to each choice in the question '%s2' in survey '%s3'.",
	"a_dd_count_table": "Shows the count of responses to each choice in the question '%s2' in survey '%s3'.",
	"a_dd_total_table": "Shows the total of responses to each choice in the question '%s2' in survey '%s3'.",
	"a_dd_none_table": "Shows the responses to each choice in the question '%s2' in survey '%s3'.",
    "a_filter": "Showing %s1 records out of %s2",

    "a_dd_units": "(units are %s1)",

    "a_dd_group": "Grouped by the responses to the question '%s1'.",
	"a_dd_group_interval": "Grouped by the responses to the question '%s1' and by '%s2'.",
	"a_dd_interval": "Grouped by '%s1'",

    "a_dd_where": "<b>Where %s1 equals %s2</b>",
    "_scheduled_start": "Scheduled Start",
	"a_exp_media_msg": "Exports with media are limited to %s1 records. ",
	"a_exp_msg": "Exports without media are limited to %s1 records. ",
	"a_exp_msg2": "You are currently exporting %s2 records. Set the start and end records to conform to this limit",
	"a_exp_start": "The start record must be less than or equal to the end record",

    // Messages
    "msg_ipt": "Invalid panel title",
    "msg_pdft": "PDF Templates are now managed on the templates page which can be accessed from the tools menu",
    "msg_oet": "Only email can be specified as the target when periodic is selected as the trigger",
    "msg_dms": "The day of the month must be specified",
    "msg_pc": "The reminder interval must be greater than zero",
    "msg_pk_sel": "You can only select the record identifier if you have selected the \"Update a different question\" checkbox",
    "msg_wait": "Waiting for a response from the server. Please try again in a few seconds",
    "msg_reason": "Please provide a reason: ",
    "msg_rs": "Reminder notifications must be associated with a task group that has tasks generated from a survey.  You cannot send reminders for ad-hoc tasks",
    "msg_sel_survey": "You must select a survey",
    "msg_sel_dates": "From date is greater than To date",
	"msg_no_from": "If you specify the scheduled end date, then also specify the scheduled start date",
    "msg_sel_q": "You must select a question to show on a graph",
    "msg_so": "Number details can only be edited when you are in the organisation that the number is assigned to",
    "msg_val_prot": "Protocol (http:// or https://) must be specified with the hostname",
    "msg_val_p": "A project must be selected for this form",
    "msg_val_rf": "You must select a remote form",
    "msg_val_rh": "You must select a remote host",
    "msg_val_u_id": "You must specify a user id",
    "msg_val_pass": "You must specify a password",
    "msg_nv": "Please specify a new value",
    "msg_val_nm": "Name is required",
    "msg_val_inv_nm": "Name cannot include a / or a . character. Specify the name without an extension.",
    "msg_val_gen_nm": "The name cannot contain angle brackets <>",
    "msg_val_period": "An period greater than 0 must be specified",
    "msg_val_dl_dist": "If set then the download distance must be greater than 100 meters",
    "msg_val_show_dist": "If set then the show distance must be greater than 10 meters",
    "msg_val_ad": "If assigning a task using collected data then you need to specify the data",
    "msg_val_ad2": "The additional role should only be specified if the first role was specified",
    "msg_val_file": "A file must be selected",
    "msg_val_let": "Name must start with a letter or underscore",
    "msg_pwd_l": "Passwords, if specified, must be longer than 1 character",
    "msg_pwd_m": "Passwords do not match",
    "msg_add_notification": "Add Notification",
    "msg_edit_notification": "Edit Notification",
    "msg_send_notification": "Send Notification",
    "msg_nn": "No notifications",
    "msg_noi": "No opt in messages",
    "msg_add_map": "Add Map",
    "msg_edit_map": "Edit Map",
    "msg_sel_form": "Select Form:",
    "msg_sel_media": "Select Media Question:",
    "msg_mfn": "Create media file name with:",
    "msg_sel_media_f": "Select Media File",
    "msg_err_save": "Error: Save failed: ",
    "msg_err_del": "Error: Delete failed: ",
    "msg_err_cr": "Error: Create failed: ",
    "msg_err_upd": "Error: Update failed: ",
    "msg_err_res": "Error: Restore failed: ",
    "msg_err_block": "Error: Block failed: ",
    "msg_err_get_f": "Error: Get forms failed: ",
    "msg_err_get_r": "Error: Get roles failed",
    "msg_err_get_a": "Error: Get alerts failed",
    "msg_err_get_q": "Error: Get questions failed",
    "msg_err_get_s": "Error: Get surveys failed",
    "msg_err_get_c": "Error: Get case management settings failed",
    "msg_err_bll": "Blank Label for language",
    "msg_upd": "Update Succeeded",
    "msg_err_nc": "Name cannot include a comma",
    "msg_err_wait": "Either waiting for the server or there are no surveys in this project to assign to a user. If the project does have surveys then try again in a few seconds",
    "msg_err_cert": "Remote server does not have a signed certificate, try using http:// instead of https://",
    "msg_no_edit_rep": "You cannot edit a record that has been replaced by another record",
    "msg_confirm_del": "Are you sure you want to delete",
	"msg_confirm_del_one": "Delete %s1 only from %s2",
	"msg_confirm_del_all": "Delete %s1 from all organisations: %s2",
    "msg_confirm_tasks": "this group and all of its tasks?",
    "msg_del_r": "Are you sure you want to delete these regions?",
    "msg_erase": "Are you sure you want to permanently erase these surveys and all of their data?",
    "msg_restore": "Are you sure you want to restore these surveys?",
    "msg_del_recs": "There are %s1 data rows submitted for %s2. Are you sure you want to delete this data?",
    "msg_del_s": "Are you sure you want to delete these surveys?",
    "msg_del_data": "Are you sure you want to delete all the data in this survey?",
    "msg_del_groups": "This survey is part of a group and deleting its data will also delete the data for these other surveys",
    "msg_res_data": "Are you sure you want to restore all the data in this survey to the original submissions?",
    "msg_del_data2": "Are you really sure?",
    "msg_del_rep": "Are you sure you want to delete this report?",
    "msg_del_users": "Are you sure you want to delete these users?",
    "msg_del_projects": "Are you sure you want to delete this project?",
    "msg_del_roles": "Are you sure you want to delete these roles?",
    "msg_del_cms": "Are you sure you want to delete this setting?",
    "msg_del_orgs": "Are you sure you want to delete these organisations?",
	"msg_del_ents": "Are you sure you want to delete these enterprises?",
    "msg_del_q": "Are you sure you want to delete this question?",
    "msg_del_cl": "Are you sure you want to delete this choice list?",
    "msg_del_c": "Are you sure you want to delete this choice?",
    "msg_rep_f": "Are you sure you want to replace this choice filter",
    "msg_del_not": "Are you sure you want to delete notification",
    "msg_del_nbr": "Are you sure you want to delete the phone number",
    "msg_n_v": "Not valid",
    "msg_u_f": "Error upload failed",
    "msg_u_nt": "No nfc tags found",
    "msg_csv": "Only CSV files and Zip files containing CSV and media files are allowed",
    "msg_inv_email": "Not a valid email address",
    "msg_dup_email": "Duplicate email. Some other user has this email.",
    "msg_email_req": "If sending an email to the user then email address must be specified",
    "msg_email_dom": "Email user name should not include the email domain.  So for an email address of example@org.com the user name would be 'example'",
    "msg_dup_ident": "Duplicate user identification. Please change the user ident.",
    "msg_cs": "Click to filter or sort on this column",
    "msg_nrf": "No results forwarded to another server",
    "msg_saf": "Select a form to view the users that have downloaded that form",
    "msg_nf": "No forms",
    "msg_ns": "No Submissions",
    "msg_us": "Unknown Source",
    "msg_tg_ns": "Task group not selected",
    "msg_tg_rd": "The task group rule has been deleted",
    "msg_no_roles": "There are no enabled roles. Hence there are no restrictions on access",
    "msg_has_roles": "Only users who have one of the following enabled roles are allowed access",
    "msg_one_f": "At least one form must be selected",
    "msg_one_f2": "A form must be selected",
    "msg_embed": "The excel type must be xls or xlsx to embed images",
    "msg_pss": "Please select a survey",
    "msg_nm": "No images, video, audio found",
    "msg_pr": "Password has been reset",
    "msg_pex": "Your password has expired please set a new one",
    "msg_es": "If %s1 is associated with a user account then an email will be sent containing instructions to reset the password of that account.",
    "msg_refresh": "You have made some changes and not saved them, are you sure you want to refresh?",
    "msg_leave": "You have made some changes and not saved them, are you sure you want to leave?",
    "msg_test": "You have made some changes and will need to save them before testing the form.",
    "msg_not_f": "was not found",
    "msg_ren": "renamed to",
    "msg_nq": "You have no queries defined, use the 'add query' button to add one",
    "msg_dup_f": "This form is already in the query",
    "msg_ui": "Spaces are not allowed in the user ident",
    "msg_sp": "Select a project to see the status of forms",
    "msg_ab_ns": "Either adding tasks from already submitted data, or adding tasks from data yet to be submitted or both should be selected",
    "msg_n1": "emails in a question",
    "msg_n2": "emails in meta item",
    "msg_sms_n1": "Send SMS to ",
    "msg_sms_n2": "numbers entered in response to a question",
    "msg_ds_s": "Download started.  If there is a lot of data then this may take some time please be patient",
    "msg_ds_s_r": "Report generation started.  You will find the generated report in the reports module",
    "msg_survey_replaced": "Survey replaced",
    "msg_survey_loaded": "Survey loaded",
    "msg_reg": "Registration accepted.  An email has been sent to %s1 with a link that you can use to set your password.",
    "msg_uns": "Unsubscribe sucessful",
    "msg_s1": "An email has been sent to the address you provided.  Click on the link to confirm your subscription",
    "msg_s2": "Subscribed sucessfully",
    "msg_s3": "Enter your email address to re-subscribe to emails",
    "msg_pd_key": "Specify a data key ",
    "msg_one_role": "At least one role must be selected",
    "msg_dup_name": "Duplicate name",
    "msg_prev_select": "There are no previous select questions to get values from. You may want to set a custom filter",
    "msg_no_proj": "The user has no projects",
    "msg_pformat": "Parameters specified in the other field must be separated by semi colons and contain an equals sign. For example:  a=x;b=y",
    "msg_filter_col": "You have specified a filter Column of '%s1', hence you need to specify a filter to apply",
    "msg_filter_expr": "You must specify an expression for an 'eval' type of search",
    "msg_numb_ts": "Thousands separator will have no affect unless numbers is also selected",
	"msg_warn_likert_n": "Likert only works with the normal select type",
    "msg_choice_value": "The choice value must be specified",
	"msg_search_source": "You must select either survey or csv as the souce of the choice data",
    "msg_search_source2": "The source file or survey must be selected",
	"msg_pdfcols_count": "The total width of all the columns must add up to 10",
    "msg_pdfinv_zoom": "Zoom must be between 1 and 20",
    "msg_pdfinv_round": "Rounding of decimal places can be between 0 and 10",
	"msg_pdfcols_width": "A column width must be between 1 and 10 inclusive",
	"msg_pe": "The number of days before password expiry must be an integer greater than 0",
    "msg_restore_started": "Restoration of the survey data has started.  Press refresh to see progress.  This process may take minutes if the files to be retrieved are in off site storage",
	"msg_persist": "Persistent tasks stay on the device even after they are completed. The user should reject the task when they have finished with it",
    "msg_no_uns_orgs": "There are no organisations where that email address is unsubscribed",
    "msg_res_name": "%s1 is a reserved name",
    "msg_trans_to": "The translation request has timed out.  However it should still be working in the background and you can view the results by refreshing the page",
    "msg_subs_bg": "If you have un-subscribed from receiving email notifications, then using this page, you can re-subscribe to receive them again. The decision to send notifications to an email address is made by an administrator so although you can un-subscribe from these and then re-subscribe on this page you cannot request email notiifcations when you are not being sent any.",
    "msg_subs_u": "You can unsubscribe from receiving notifications from the organisation that sent you the email, by clicking the unsubscribe button below.  If you later want to receive emails from that organisation again, you can select the 'subscribe' menu on the home page.",
    "msg_subs_s": "Click on the button below to subscribe",
    "msg_fgt_p": "Enter the email address associated with your account. A link will be sent to that address allowing you to reset your password.",
    "msg_archive": "%s1 records on or before %s2 will be archived from the following survey(s); %s3. Do you wish to proceed?",
    "msg_archive_done": "Archiving completed",
    "msg_archive_none": "There is no data to archive",
    "msg_archive_data": "Are you sure you want to archive old submissions?",
    "msg_archive_before": "The date before which submissions will be archived must be specified",
    "msg_no_nbr": "You must specify their number",
    "msg_nc": "Do you want to move this message to a new case",

    // Editor
    "e_b_e": "Base on existing form",
    "e_s_r": "Store results in shared table",

    // Tasks
    "t_i_sd": "Invalid Start Date",
    "t_i_ed": "Invalid End Date",
    "t_b": "Trigger with any survey in the bundle",
    "t_complete": "Show Completed Tasks",
    "t_show": "Show tasks for:",
    "t_assign_type": "Assignment type",
    "t_assign": "Assign User",
    "t_assign_roles": "Assign to users with role",
    "t_fixed_role": "And this role",
    "t_add_group": "Add Task Group",
    "t_del_group": "Delete Task Group",
    "t_edit_group": "Edit Task Group",
    "t_lt_file": "Load Tasks From File",
    "t_choose_user": "Choose the user to assign to the selected tasks",
    "t_ce": "Clear Existing Tasks",
    "t_ce_warn": "This will clear all the existing data in the survey. Are you sure you want to do this?",
    "t_ceg": "Create Empty Task Group",
    "t_cfe": "Create from existing data",
    "t_ep": "Existing Project",
    "t_es": "Existing Survey",
    "t_ec": "Existing Choice",
    "t_et": "Existing Text",
    "t_nt": "New Text",
    "t_ns": "New Survey",
    "t_ft": "Filter Tasks",
    "t_aft": "Advanced Filter",
    "t_df": "Date Filter",
    "t_fq": "Filter by Question",
    "t_so": "Selected Option",
    "t_sn": "Set Numeric Value",
    "t_st": "Set Text Value",
    "t_ue": "Update Existing Results",
	"t_pp": "Pre-populate form with existing data",
    "t_nid": "No initial data",
    "t_ap": "Address Parameters",
    "t_nfc": "NFC / Location",
    "t_assigned": "Assigned",
    "t_eas": "Send Email to assigned user",
    "t_assignments": "Assignments",
    "t_iewe": "Emails can be sent to notify people about an escalation",
    "t_fl": "File Loaded",
    "t_tl": "Template Loaded",
    "t_efnl": "Error file not loaded",
    "t_ens": "Error not saved",
    "t_add_tasks": "Click on the map where you want to add new tasks. When finished click on 'Save Tasks' button",
    "t_add_task": "Add Task",
    "t_edit_task": "Edit Task",
    "t_new_case": "New Case",
    "t_add_tasks2": "Add Tasks",
    "t_save_tasks": "Save Tasks",
    "t_tasks_only": "Tasks Only",
    "t_tg": "Task Group",
    "t_tasks": "Tasks",
    "t_id": "Initial Data",
    "t_n_s": "Tasks without start time",
    "t_drag": "Drag a task and drop onto calendar",
    "t_g": "Guidance",
    "t_pol": "Complete All",
    "t_auto": "Allow users to assign themselves (if unassigned)",
    "t_auto2": "Self Assign",
    "t_ass_done": "Have you completed this assignment?",
    "t_defn": "Task Group Definition",
    "t_au": "All Users",
    "t_u": "Unassigned",
    "t_ad": "Assign from data values",
    "t_af": "Create tasks from results submitted after this task group is created",
    "t_ac": "Create tasks from results that have already been submitted",
    "t_start": "Set the start time and duration of the task relative to when it is created or a date in the source survey",
    "t_all": "All tasks",
    "t_dl_dist": "Download distance (meters)",
    "t_dl_dist_i": "The user will need to be within this distance from the geographic center of the tasks in this task group before they will be downloaded onto their device",
    "t_show_dist": "Show distance (meters)",
    "t_show_dist_i": "The user will need to be within this distance from the task before it is displayed on their device",
    "t_show_console": "Show in console",
    "new": "Unassigned",
    "accepted": "Assigned",
    "submitted": "Complete",
    "rejected": "Rejected",
    "cancelled": "Cancelled",
    "deleted": "Deleted",
    "restored": "Restored",
    "unsent": "Not Sent",
    "unsubscribed": "Unsubscribed",
    "restore": "Restore",
    "t_mt": "My Tasks",

    // template load errors
    "e_u_err": "Form Upload Error",
    "e_u_exists": "exists in project",
    "e_h_rename": "Change the name of the form or delete the old form",
    "e_u_sm_no_list": "select_multiple question without list name",
    "e_h_sm_no_list": "Check the survey sheet. Make sure you have specified a list name for all the select_multiple questions",
    "e_u_so_no_list": "select_one question without list name",
    "e_h_so_no_list": "Check the survey sheet. Make sure you have specified a list name for all the select_one questions",
    "e_unknown": "Unknown error",
    "e_get_help": "Contact the help desk",
    "e_calc": "Calculation error",
    "e_h_calc1": "Check the 'calculation' column for an invalid formula (A valid formula results in either true or false)",
    "e_h_calc2": "Otherwise check the 'relevant' column for a formula that does not result in a true or false result",
    "e_inv_f": "Invalid Function",
    "e_h_f1": "Check for capital letters, all letters should be lower case",
    "e_h_f2": "Check for spelling mistakes in the name of the function",
    "e_val": "Validation error",
    "e_circ": "Check for circular references",
    "e_in_q": "in question",
    "e_h_c1": "A 'relevant or calculation or choice filter' for question is referring to  itself instead of referring to another question",
    "e_h_c2": "Relevant statements are evaluated before a question is asked so they cannot refer to their own questions",
    "e_brackets": "Mismatched brackets in function",
    "e_text": "Error with the following text:",
    "c_mcm": "Multi case message",
    "c_mcmg": "Use place holder %s1 for the number of open cases and %s2 to hold the list of existing cases",

    // Register
    "r_o_n": "Organisation Name",
    "r_y_n": "Your Name",
    "r_y_e": "Your Email",
    "r_y_o": "Organisation Website",
    "r_f": "Registration Form",
    "r_s": "Email Subscriptions",

    // Monitor
    "mon_l200": "(last 200)",
    "mon_gtb": "Group Totals by",
    "mon_nl": "Not Loaded",
    "mon_ue": "Upload Errors",
    "mon_dup": "Duplicates",
    "mon_att": "Extra Attachments",
    "mon_sa": "Show As",
    "mon_zoom_grid": "Zoom to Grids",
    "mon_show_grid": "Show Grids",
    "mon_clear_grid": "Clear Grids",
    "mon_create_grid": "Create Grid",
    "mon_del_grid": "Delete Grids",
    "mon_sel_layer": "Select a layer to add to the map as an overlay",
    "mon_c_reg": "Create a new region",
    "mon_c_reg_i": "Enter a name for the region and the width of each cell in meters. Click on the map to set the centre of one of the cells. Press the \"shift\"button and drag the mouse to set the bounds.",
    "mon_name_cond": "(lowercase, no spaces or numbers)",
    "mon_cw": "Cell Width (in Meters. greater than 10)",
    "mon_centre": "Centre: (click on map)",
    "mon_bounds": "Boundary: (click and drag on map while pressing shift key)",
    "mon_sel_del": "Select the regions that you want to delete",
    "mon_ud": "Upload Date",
    "mon_pi": "Phone Identifier",
    "mon_fr": "Failure Reason",
    "mon_file": "File with raw results",
    "mon_optin_msg": "Opt in",
    "mon_send_count": "Send Count",
    "mon_pending_count": "Pending Messages",
    "mon_ignore_old": "Ignore issues older than 100 days",
    "error": "Error",
    "success": "Success",
    "mon_retry": "Re-apply failed uploads",
    "mon_page": "Page %s1 from %s2 to %s3",
    "mon_uid": "Upload Id",

    // Analysis, reports, modify
    "an_auth": "Author",
    "an_published": "Published",
    "an_publish": "Publish",
    "an_dr": "Date Range",
    "an_csel": "Clear Selection",
    "r_add": "Add Report",
    "r_ed": "Edit Report",
    "r_del": "Delete Reports",
    "an_nd": "No data available",
    "an_nmd": "No data matches the filter",
    "an_mod_hint": "The modify page allows you to make bulk changes to text.  You can select a specific text value and then change all occurrences of that to another value.  This is very useful for fixing spelling inconsistencies. You can also change text values in 'other' questions into one of the select choices. You can also update barcode and calculate questions. Other question types are not supported as, for example, it would make no sense to do a bulk change of an integer age from 25 to 26 wherever it occurred. You can review modifications by selecting the review page.<br/><br/> You can however update integer, decimal and select1 questions by selecting the checkbox 'Update a different question'. In this case the change will be applied to all questions that match the existing values of the two questions. In most cases you will not need this option.",
    "Upload Time": "Upload",
    "_start": "Start Time",
    "_end": "End Time",
    "percent": "Percentage",
    "count": "Count",
    "none": "None",
    "average": "Average",
    "max": "Max",
    "min": "Min",
    "sum": "Sum",
    "changes": "Update",
    "task": "Task",
    "notification": "Notification",
    "created": "Created",

    // Web forms
    "wf_sig": "Signature",

    // Editor
    "ed_gr": "Text rows to show",
    "ed_ba": "Background Audio",
    "ed_map_zoom": "Map Zoom",
    "ed_map_source": "Map Source",
    "ed_map_style": "Map Style",
    "ed_round": "Decimal places",
	"ed_hidden": "Hide in form",
    "ed_extendable": "Extendable",
	"ed_selfie": "Open camera in selfie mode",
    "ed_new": "Require a new picture be taken",
	"ed_read_nfc": "Read an NFC tag instead of a barcode",
    "ed_gt_0": "must be greater than 0, or not set",
	"ed_add_to": "Append to question",
    "ed_app": "Put in an appendix",
	"ed_barcode": "Show as a barcode",
    "ed_si": "Show choice images instead of label text",
    "ed_stretch": "Stretch images to take up all space allocated",
	"ed_hyperlink": "Show a hyperlink to the image",
	"ed_col_width": "width of column %s1",
	"ed_place_map": "Allow the user to select their location from a map",
	"ed_l_bold": "Make label bold",
	"ed_caps": "Make label all capitals",
	"ed_npb": "New Page Before",
	"ed_sa": "Spacing above",
	"ed_bgc": "Label Background Colour",
	"ed_vbgc": "Value Background Colour",
    "ed_mc": "Marker Colour",
	"ed_height": "Height allowed for answer",
    "ed_lw": "Width of label",
    "ed_pdfcols_help": "There can be a maximum of 10 columns. The widths of the columns must add up to 10.",
	"ed_pdf_layout": "PDF Layout",
	"ed_pdfno": "Hide question in pdf export",
	"ed_date_type": "Date Type",
	"ed_select_type": "Select Type",
    "ed_nc": "No Calendar",
	"ed_my": "Month and Year",
	"ed_bikram": "Bikram Sambat",
	"ed_coptic": "Coptic",
	"ed_ethiopian": "Ethiopian",
	"ed_islamic": "Islamic",
    "ed_myanmar": "Myanmar",
    "ed_persian": "Persian",
    "ed_r": "Rating",
	"ed_bear": "Bearing",
	"ed_sep": "Thousands Separator",
    "ed_ncol": "No Collapse Button (WebForms Only)",
    "ed_sn": "Show numbers keyboard",
    "ed_numb_cols": "Number of Columns",
	"ed_min": "Minimal",
	"ed_adv": "Auto Advance",
	"ed_ac": "Auto Complete",
    "ed_acm": "Minimal Auto Complete",
	"ed_compact": "Compact",
	"ed_compact_adv": "Compact with auto advance",
	"ed_image_map": "Image Map",
	"ed_pick": "Picker",
    "ed_as": "Add Search",
    "ed_rf": "Reference Form",
    "ed_in": "Instance Name",
	"ed_io": "Instance Order",
	"ed_ic": "Instance Count",
    "ed_ep": "Existing Project",
    "ed_es": "Existing Survey",
    "e_b_e": "Base on existing form",
    "e_s_r": "Store results in shared table",
    "ed_qt": "Question Text",
	"ed_dn": "Display Name",
    "ed_ct": "Choice Text",
    "ed_qh": "Question Hint",
    "ed_cl": "Choice List",
    "ed_appearance": "Appearance",
    "ed_parameters": "Parameters",
    "ed_con": "Constraint",
    "ed_con_msg": "Constraint Message",
    "ed_sct": "Save changes before running auto translate",
    "ed_rel": "Relevant",
    "ed_req": "Required",
    "ed_req_msg": "Required Message",
    "ed_nreq": "Not Required",
    "ed_ap": "Auto play",
    "ed_al": "Automatically Annotate",
	"ed_mat": "Medical Audio Type",
	"ed_mat_c": "Conversation",
	"ed_mat_d": "Dictation",
    "ed_al2": "Use in oversight forms to automatically annotate or translate text, images, audio and video",
	"ed_am": "Medical",
	"ed_am2": "Use when transcribing audio that contains medical terms",
    "ed_at": "GPS Accuracy Threshold",
    "ed_ls": "Linked Survey",
    "ed_ro": "Read Only",
    "ed_ls_add": "Add link",
    "ed_l": "Linked",
    "ed_nl": "Not Linked",
    "ed_r_msg": "Required Text Message",
    "ed_r_expression": "Optional expression to make this question required",
    "ed_r_readonly": "Optional expression to make this question read only",
	"ed_f_l": "The form to be launched must be specified as a parameter",
	"ed_emp_rep": "Sub forms and groups should have at least one question.",
    "ed_hrk": "Unique Key",
    "ed_dk": "Data Key",
    "ed_csv": "CSV File",
    "ed_csvs": "CSV Files",
    "ed_c_s": "Source of Choices",
    "ed_p_q": "Source question",
    "ed_cws": "Choices Worksheet",
    "ed_i_c": "Created",
    "ed_st": "Shared Table",
    "ed_ld": "Allow Loading data from file",
    "ed_pd": "Pull data from another survey",
    "ed_i_bo": "Based On",
    "ed_i_tn": "Table Names",
    "ed_tlf": "Top Level Form",
    "ed_namedia": "Media cannot be used with this question",
    "ed_ab": "Add Before",
    "ed_aa": "Add After",
    "ed_aq": "Add New Question",
    "ed_ancl": "Add New Choice List",
    "ed_anc": "Add New Choice",
    "ed_cs": "Cascading Select",
    "ed_cspq": "Previous Select Question",
    "ed_csp": "Previous Selection",
    "ed_cf": "Custom Filter",
    "ed_afc": "Add Filter Column",
	"ed_clabs": "Choice Labels",
    "ed_clab": "Choice Label",
    "ed_cval": "Choice Value",
    "ed_fv": "Filter Value",
	"ed_sfv": "Second Filter Value",
    "ed_choice_f": "Choice Filter",
	"ed_vc": "Choice names must only contain letters, numbers, spaces, underscores, dashes and periods",
	"ed_vq": "The question name must start with a letter, underscore or colon and only contain letters, numbers, underscores, dashes and periods",
    "ed_ns": "Add a new survey using the 'File' menu before adding questions",
    "ed_kp": "Key Policy",
    "ed_s1": "Select One",
    "ed_s": "Select Multiple",
    "ed_image": "Image",
    "ed_image_type": "Image Type",
    "ed_int": "Integer",
    "ed_gps": "GPS Point",
    "ed_calc": "Calculation",
    "ed_s_calc": "Server Calculation",
    "ed_ci": "Server Compound PDF Image",
    "ed_cm": "Compound Map",
    "ed_audio": "Audio",
    "ed_video": "Video",
    "ed_bc": "Barcode",
    "ed_dt": "Date and Time",
    "ed_t": "Time",
    "ed_dec": "Decimal",
    "ed_gps_line": "GPS Line",
    "ed_gps_area": "GPS Area",
    "ed_ack": "Acknowledge",
    "ed_range": "Range",
    "ed_graph": "Graph",
    "ed_qs": "Questions set ",
    "ed_ee": "Don't include empty answers in reports",
    "ed_cpdf": "Compress PDF (images will have lower resolution that should be adequate for printing)",
    "ed_er": "Cannot save until errors are fixed",
    "ed_blocked": "The survey has been blocked. Changes cannot be saved.  You can unblock the survey on the form management page.",
    "ed_cml": "Cannot modify languages while there are unsaved changes",
    "ed_cx": "Cannot export to an xlsForm while there are unsaved changes",
    "ed_csr": "Cannot set questions required, or not required, while there are unsaved changes",
    "ed_fi": "Form Identifier",
    "ed_dl": "Default Language",
    "ed_td": "Timing Data",
    "ed_rl": "Record location with questions",
    "ed_rc": "Audit changes to questions",
	"ed_hod": "Hide the form on devices",
    "ed_sld": "Lookup local, unsent, data on device",
    "ed_s_g": "Grid (Web Forms)",
    "ed_s_p": "Pages",
    "ed_c_sr": "All questions set to <span style=\"color:blue;\">required</span>",
    "ed_c_snr": "All questions set to <span style=\"color:red;\">not required</span>",
    "ed_c_del_q": "Deleted question",
    "ed_c_del_o": "Deleted choice %s1 from option list %s2",
    "ed_c_chg_p":  "%s1 property of question %s2 changed to: %s3 from: %s4",
    "ed_c_add_q": "Added question %s1 with type %s2",
    "ed_c_add_o": "Added choice %s1 to choice list: %s2",
    "ed_c_settings": "Settings changed",
    "ed_c_languages": "Languages changed",
    "ed_c_template": "PDF template changed",
    "ed_a_template": "PDF template added",
    "ed_d_template": "PDF template deleted",
    "ed_s_templates": "PDF templates added using the settings dialog cannot be set as 'not available' nor set as default",
    "ed_mmf": "Manage Media Files For this Form",
    "ed_sl": "Survey to launch",
    "ed_sc": "Survey to complete",
    "ed_slu": "Survey lookup",
    "ed_gsi": "Get Survey Ident",
    "ed_qk": "Question to store the returned key",
	"ed_qkc": "Question in the launched form to get this forms key",
    "ed_pl": "Page Layout",
	"ed_w": "Width in webform grid",
    "ed_m": "Filter",
    "ed_fc": "Filter Column",
	"ed_sfc": "Second Filter Column",
    "ed_addcalc": "Add calculation here",
    "contains": "contains",
    "startsWith": "startsWith",
    "endsWith": "endsWith",
    "ed_ds": "Data Survey",
    "ed_os": "Oversight Survey",
    "ed_mr": "Only allow an enumerator to access their own data using the search() or pulldata() functions. Data submitted by other users will not be downloaded as reference data.",
    "ed_nb": "No Buttons",
    "ed_offline": "Offline and Online",
    "ed_online": "Online Only",
    "ed_csa": "changes successfully applied",
    "ed_csf": "changes failed",
    "ed_transd": "Translations done from language %s1 to language %s2",
    "ed_ow": "Overwrite existing translations",
    "ed_fe": "Filter Expression",
    "ed_e": "Expression",
    "ed_drl": "Default report logo",
    "ed_voice": "Voice only",
    "ed_na": "Not available",

    // Managed Forms
    "mf_id": "Include Deleted",
    "mf_cc": "Include Completed Cases",
    "mf_fd": "Form Data",
    "mf_md": "Management Data",
    "mf_sc": "Visible Columns",
    "mf_sc_i": "Select the columns that you want to see in the table then press the 'Apply' button. You can then sort on these columns",
    "mf_qr": "QR Columns",
    "mf_bc_i": "Select the columns that you want to see as barcodes in the table and in reports",
    "mf_st": "Submission Tracking",
    "mf_gr": "Get related",
    "mf_of": "Oversight Survey",
    "mf_no": "There are no oversight forms. They can be added in the \"Available Oversight Forms\" tab",
    "mf_sel": "Select the form that you want to manage and the oversight form that you want to manage it with.",
    "mf_ao": "Available oversight forms",
    "mf_mf": "Managed forms",
    "mf_ld": "Layer details",
    "mf_rf": "Reset Filters",

    "mf_tr": "A title is required",
    "mf_sr": "Summary Report",
    "mf_lock": "Lock for editing",
    "mf_rel": "Release",
    "mf_mr": "My records",
    "mf_ur": "Unassigned records",
    "mf_or": "Other records",
    "mf_max": "Max records",
    "mf_clear": "Clear single record view",
    "mf_sc": "Show Controls",
    "mf_hc": "Hide Controls",
    "mf_ts": "Table Settings",

    // User trail
    "ut_mps": "Max point separation to include in a single line",
    "ut_mps_err": "Max point separation must be greater than 0",

    // Logout
    "lo_out": "You are logged out of ",
    "lo_lo": "Logged Out",
    "lo_loa": "You should now close your browser in order to ensure cached log in details are removed",

    // Users
    "u_sec_mgr_del": "Require security manager privilege to delete data",
    "u_other_msg": "These settings apply to the current organisation",
    "u_show_u": "Show users with security group",
    "u_show_p": "Show users in project",
    "u_show_r": "Show users with role",
	"u_show_o": "Show users with organisation",
    "u_add": "Add User",
    "u_del": "Delete User",
    "u_mv": "Move To Organisation",
    "u_cms": "Case management Settings",
	"u_mv_ent": "Move To Enterprise",
    "u_add_p": "Add Project",
    "u_del_p": "Delete Project",
    "u_add_o": "Add Organisation",
    "u_del_o": "Delete Organisation",
	"u_add_e": "Add Enterprise",
	"u_del_e": "Delete Enterprise",
    "u_add_r": "Add Role",
    "u_add_s": "Add Setting",
    "u_edit_r": "Edit Role",
    "u_del_r": "Delete Role",
    "u_det": "User Details",
    "u_ident": "Ident (lowercase, no spaces or an email address)",
    "u_email": "Send email to new user",
    "u_set_p": "Specify password",
    "u_sg": "Security Groups",
    "u_det_p": "Project Details",
    "u_det_ur": "User Role Details",
    "u_det_o": "Organisation Details",
    "u_det_e": "Enterprise Details",
    "u_det_o_rep": "Report Publishing Options",
    "u_det_o_ft": "Mobile App Options",
	"u_det_o_wf": "Webform Options",
    "u_det_pbc": "Page background color",
	"u_det_paperbc": "Paper background color",
    "u_det_buttonbc": "Button color",
    "u_det_buttontc": "Button text colors",
    "u_det_headertc": "Heading text colors",
	"u_det_fp": "Footer position (from right in pixels)",
    "u_wf_hd": "Hide 'save as draft' checkbox",
    "u_ft_del": "Delete submitted results from the phone",
    "u_ft_prev_track_dis": "Prevent the disabling of location tracking",
    "u_ft_img_size": "Maximum pixels of the long edge of an image",
    "u_ft_img_camera": "Original size from camera",
    "u_ft_im": "GeoShape and GeoTrace Input Method",
    "u_ft_im_pt": "Placement by Tapping",
    "u_ft_im_man": "Manual location recording",
    "u_ft_im_auto": "Automatic location recording",
    "u_ri": "Recording Interval",
    "u_ft_vs": "Very small (640px)",
    "u_ft_s": "Small (1024px)",
    "u_ft_m": "Medium (2048px)",
    "u_ft_l": "Large (3072px)",
    "u_ft_ld": "Send location data on path of user",
    "u_ft_os": "Enable ODK style menus to delete, submit, edit and get new forms",
    "u_ft_am": "Enable ODK Admin menu",
    "u_ft_ex": "Disable exit menu",
    "u_ft_bgs": "Prevent background audio from being disabled",
    "u_ft_in": "Allow user to set instance name",
    "u_ft_mf": "Allow user to mark  forms as not finalized",
    "u_ft_rv": "Allow finalised forms to be opened for review",
    "u_ft_ft": "Force the use of tokens for logon",
    "u_ft_as": "Automatically Synchronise",
    "u_ft_ms": "Manual Synchronisation",
    "u_ft_sw": "WIFI",
    "u_ft_hrv": "High Resolution Video",
    "u_ft_sw_c": "WIFI and cellular network",
    "u_ft_sop": "Set on phone",
    "u_ft_tasks": "Maximum number of tasks to download",
	"u_ft_back_nav": "Backward navigation",
	"u_ft_nav": "Screen Navigation",
	"u_ft_pw_timeout": "Number of days before password expiry",
    "u_ft_pw_policy": "Password policy",
	"u_ft_never": "Never require logon",
	"u_ft_always": "Always require logon on application start",
	"u_ft_periodically": "Require logon periodically",
    "u_ft_swipes": "Use horizontal swipes",
    "u_ft_buttons": "Use forward/backward buttons",
    "u_ft_swipes_buttons": "Use swipes and buttons",
    "u_ft_sa": "Yes - always shown",
    "u_ft_sc": "Yes - collapsed",
    "u_ft_ss": "Enable server settings menu",
    "u_ft_md": "Enable user and identity menu",
	"u_ft_eg": "Enable Geo-fence",
    "u_det_o_email": "Email Options",
    "u_det_o_other": "Other Options",
    "u_det_o_limits": "Monthly Usage Limits",
    "u_other_edit": "Allow editing of results",
    "u_email_tasks": "Allow sending of task emails",
	"u_use_api": "Allow API access",
	"u_submissions": "Allow submissions",
	"u_notifications": "Allow notifications",
    "u_sms": "Allow SMS",
    "u_ratelimit": "Max number of API requests per minute per organisation",
    "u_mb_k": "Mapbox Key",
    "u_mb_a": "Mapbox Account",
    "u_mb_s": "Mapbox Style Id",
    "u_g_k": "Google Maps Key",
    "u_mt_k": "MapTiler Maps Key",
    "u_v_k": "Vonage Application Id",
    "u_v_whs": "Vonage Webhook Secret",
    "u_ar": "AWS Region for Email Service",
    "u_smtp": "Smtp Host",
    "u_e_type": "Email Type",
    "u_e_dom": "Email Domain",
    "u_e_nm": "Email user name",
    "u_e_si": "Server Identifier",
    "u_e_p": "Email password",
    "u_e_port": "Email Server port",
    "u_h_e": "Email to get Help",
    "u_rap": "Restrict access",
    "u_rap_d": "Administrators will be able to assign the selected users only to this project",
    "u_chg": "Changed by",
	"u_usage": "Usage Report",
    "u_attendance": "Attendance Report",
    "u_notification": "Notification Report",
	"u_access": "Form Access Report",
    "u_b_access": "Bundle Access Report",
    "u_survey": "Surveys in this organisation",
    "u_r_u": "Resource Usage Report",
    "u_sms_url": "SMS Url",
    "u_sens_data": "Sensitive Data",
    "u_sens_ao": "Admin Only",
    "u_sens_sq": "Signature Questions",
    "u_sens_nr": "No restrictions",
    "u_org_admin": "Organisation Administrator",
    "u_ent_admin": "Enterprise Administator",
    "u_server_owner": "Server Owner",
    "u_check_mv_u": "Are you sure you want to move the following users (%s1) to %s2",
	"u_check_mv_p": "Are you sure you want to move the following projects (%s1) to %s2",
	"u_check_mv_o": "Are you sure you want to move organisation \"%s1\" to enterprise \"%s2\"",
    "u_only_one_ent": "Only one enterprise has been created.  You need to create another one to move an organisation there",
    "u_co": "Current Organisation",
    "u_gr": "Get report",
    "u_sat": "Set as Theme",
    "u_bl": "WebForm Banner Logo (Height 50px)",
    "u_ml": "Main Logo",
    "u_nbc": "Navigation bar color",
    "u_nbtc": "Navigation bar text color",
    "u_eoi": "Enable Opt In",
    "admin": "Administrator",
    "analyst": "Analyst",
    "enterprise admin": "Enterprise Admin",
    "enum": "Enumerator",
    "manage": "Manage Data",
    "console admin": "Manage Console",
    "org admin": "Organisational Admin",
    "security": "Security Manager",
    "view data": "View Data",
    "manage tasks": "Manage Tasks",
    "view own data": "View Own Data",
    "dashboard": "Dashboard",
    "links": "Follow Links",
    "u_css_f": "CSS file",
    "u_css": "Upload a CSS style sheet",
    "u_css_ss": "Upload a new CSS style sheet for the server",
    "u_css_ss_org": "Upload a new CSS style sheet for the current organisation",
    "u_css_del": "Delete CSS style sheet",
    "u_css_sel": "Select a style sheet",
    "u_css_info": "Create a custom style sheet and then upload to the server",
    "u_clear_p": "Clear existing empty imported projects",
    "u_clear_u": "Clear existing imported users",
    "u_clear_r": "Clear existing imported roles",
    "u_api_rl": "Rate limit for data API",
    "u_mps": "Minimum password strength",
    "u_oo": "The user is currently in a different organisation and only the list of organisations that they belong to can be changed.",
    "u_code": "App Code",

    // Browse Results
    "br_ci": "Case Insensitive",
    "br_exact": "Exact Match",
    "br_cc": "Select the columns that you want to test for similarities in the data. For each column that you want to test you can select a function to broaden the range of values that will match.",
    "br_s": "Search",
    "br_sdx": "Soundex",
    "br_cd": "Use the table of data for this survey view",
    "br_tf": "for text questions, including select and select_one",
    "br_nf": "for numeric questions, including integer and decimal",

    // Shared Resources
    "sr_res": "Images Video Audio Files",
    "sr_nfc": "NFC Tags",
    "sr_uid": "NFC UID",
    "sr_geo": "Geographic locations",
    "ar_af": "Add Files",
    "sr_g": "Group",
    "sr_fn": "File Name",
    "sr_m_ph": "Enter a name for your map",
    "sr_m_d": "Enter a description for this map (optional)",
    "sr_m_mb": "For Mapbox this identifies the map, for example: mapbox/light-v10",
    "sr_cr": "Custom Reports",
    "sr_al": "Use Anonymous Location in Task",
    "sr_ul": "Update Existing Location",
    "sr_nl": "Save as New Location",
    "sr_sl": "Save Location",
    "sr_sm": "Survey Media Files",
    "sr_si": "The NFC UID and GPS coordinates that you specify here can be saved as a named location or used in the task as an anonymous location",
    "sr_rh": "Resource History",

    // languages
    "ar": "Arabic",
    "en": "English",
    "fr": "French",
    "hi": "Hindi",
    "pt": "Portuguese",
    "uk": "Ukrainian",
    "es": "Spanish",

    // Notifications
    "n_wf": "Web Form",
    "n_pdfp": "PDF (Portrait)",
    "n_pdfl": "PDF (Landscape)",
    "n_eq": "Email Question",
    "n_ep": "Email Meta Item",
    "n_smsq": "SMS Question",
    "n_share": "Get link",
    "n_cs_e": "Comma separated list of email addresses",
    "n_ea": "Email addresses",
    "n_eq": "Email question",
    "n_eqc": "Question that will contain the email addresses to use",
    "n_uiqc": "Question that will contain the user ident to notify",
    "n_esc": "Subject for the email (optional)",
    "n_inc_ref": "Include referenced surveys",
	"n_lo": "Only reference data from forms that were launched from this form",
    "task_reminder": "Task Reminder",
    "submission": "Submission",
    "console_update": "Console Update",
    "n_p": "Send reminder after",
    "n_oa": "Email original assignee",
    "n_ap": "Add placeholder",
    "n_si": "SMS Sender Id",
    "n_sms_n_tt": "Comma separated list of phone numbers in format +ccnnnnnnnnnn",
    "n_sms_q_tt": "Question that will contain the SMS numbers to use",
    "n_e_p_tt": "Meta item or preload that will contain an email address to use",
    "n_uq": "Update Question",
    "n_uv": "Update Value",
    "n_no_oversight": "There are no oversight surveys that can be used to update the data in the console.  Select an oversight form or load a new one.",
    "m_no_q": "There are no questions in the oversight survey that can be edited in bulk.  These questions need to be of type text, or select one, or select multiple",
    "n_nv": "A value for the update must be specified",
    "n_nq": "An update question must be specified",
    "n_curl": "Callback URL",
    "n_aa": "After alert %s1.",
    "n_as": "Assign survey %s1 to %s2.",
    "n_dow": "Day of the week",
    "n_dom": "Day of the month",
    "n_sent": "Notification sent",
    "n_scc": "Server Calculation Question",
    "n_val": "A server calculation question and value that will cause the notification to be triggered must be specified",
    "n_their_nbr": "Select their Number",
    "n_our_nbr": "Our Number",
    "n_spec_nbr": "Enter their Number",
    "n_cur_emails": "Select Email",
    "n_spec_nbr_h": "Enter the number with no spaces or commas.  Include the country code.  Optionally include a +. For example: +442071838451",


    // Roles
    "ro_b": "Apply the role filters to all bundle surveys with the role",
    "ro_b_w": "Warning. Applying roles to the bundle will override the roles set in all other surveys in the bundle. Do you want to proceed?",
    "ro_b_d": "Identical roles have been applied to all surveys in the bundle",
    "ro_fr": "Filter rows",
    "ro_fc": "Filter columns",
    "ro_f_group": "Filter Group",
    "ro_fr_rule": "Rows to be shown",
    "ro_aq": "Available Questions",
    "ro_fr_i": "Enter a rule that determines which rows will be shown for this role. Enclose question names in ${...}. Enclose text in single quotes. Make sure there are spaces between the question names and operators such as =.<br/>Example 1: ${region} = 'northern'<br/>Example 2: ${age} > 16. If no rule is specified then all rows will be shown.",
    "ro_fc_i": "Select the columns to show with this filter. If no columns are selected then all columns will be shown.",
    "ro_transform": "Long to wide data transformation",

    // Intervals
    "i_m": "minutes",
    "i_h": "hours",
    "i_d": "days",
    "i_sec": "second ago",
    "i_secs": "seconds ago",
    "i_min": "minute ago",
    "i_mins": "minutes ago",
    "i_hour": "hour ago",
    "i_hours": "hours ago",
    "i_day": "day ago",
    "i_days": "days ago",

    // Dashboard
    "d_sound": "Play sound on new alert",
    "d_rep_def_freq": "Frequency Report",
    "bar_h": "Horizontal Bar Chart",
    "bar_v": "Vertical Bar Chart",
    "stacked": "Stacked",
    "normalized": "Normalized",
    "pie": "Pie Chart",
    "line": "Line Chart",
    "wordcloud": "Word Cloud",
    "length": "Count",
    "d_qtr": "25%",
    "d_half": "50%",
    "d_3qtr": "75%",
    "d_full": "100%",
    "d_sec": "Seconds",
    "d_min": "Minutes",
    "d_hour": "Hours",
    "d_day": "Days",
    "d_c_day": "Count by day",
    "d_c_month": "Coount by month",
    "d_c_year": "Count by year",
    "d_add": "Add",
    "d_add_c": "Add Chart",
    "d_add_q": "Add New Query",
    "d_add_f": "Add New Form",
    "d_from_f": "Link From Form",
    "d_from_q": "Link From Question",
    "d_edit_web": "Edit in Web Form",
    "d_d_pdf": "Download PDF",
    "d_layers": "Map Layers",

    // Review
    "rev_upd_diff": "Update a different question to the source question",
    "rev_sel_pk": "Select data to change using a primary key",
    "rev_q_upd": "Question to update",
    "rev_text": "Text",
    "rev_upd": "Update",
    "rev_upd_t": "Update target question",
    "rev_upd_text": "Update Text Questions",
    "rev_rt": "Replace text",
    "rev_rc": "records with choice",
    "rev_fq": "from question",
    "rev_rw": "records with",
    "rev_rdm": "Review Data Modifications",
    "rev_ci": "Change Id",
    "rev_rb": "Reversed By",
    "rev_cb": "Changed By",
    "rev_rcn": "Reverse Change Number",
    "rev_det": "Details for change number",
    "rev_cq": "Changed Question",
    "rev_sc": "Selcted Choice",
    "rev_usc": "Un-Selected Choice",

    // Chart Questions
    "ch_ds": "Chart Data",

    // Billing
    "quantity": "Quantity",
    "free": "Free",
    "unitCost": "Unit Cost",
    "amount": "Amount",
    "server": "Server",
    "submissions": "Submissions",
    "submissions_i": "A limit of zero is unlimited",
    "disk": "Disk",
    "rekognition": "AWS Rekognition",
    "rekognition_i": "images",
    "sentiment": "AWS Comprehend",
    "sentiment_i": "requests",
    "translate": "AWS Translate",
    "awsses": "AWS Simple Email Service",
    "translate_i": "letters",
	"transcribe_medical": "AWS Transcribe Medical",
	"transcribe_medical_i": "seconds of audio",
    "transcribe": "AWS Transcribe",
    "transcribe_i": "seconds of audio",
	"static_map": "Mapbox Static Map",
    "google_static_map": "Google Static Map",
    "maptiler_static_map": "MapTiler Static Map",
	"monthly": "Monthly Charge",
    "org_bill_rpt": "Organisation Usage Report",
    "bill_enable": "Enable Billing",
    "bill_from": "Applies From",
    "bill_chg_date": "Date Changed",
    "bill_norates": "No rates have been set for this %s1. If billing is enabled then the rates for the %s2 will be used",
	"billing_disabled_msg": "Billing has been disabled",

	// Reports
	"rep_usage_project": "Usage by project",
	"rep_usage_survey": "Usage by survey",
    "rep_usage_device": "Usage by device",
    "rep_values_question": "Values Question",
    "rep_values": "Values",
    "rep_wide_columns": "Wide Columns",
    "rep_msg_min_keys": "You must specify at least one key question if transform is enabled",
    "rep_msg_v_q": "You must specify a values question if transform is enabled",
	"rep_msg_min_values": "You must specify at least one value if transform is enabled",
	"rep_msg_min_wc": "You must specify at least one wide column if transform is enabled",
    "rep_ch": "Column Heading",
    "rep_inc_temp": "Include temporary users",
    "rep_inc_alltime": "Include data for 'all time' as well as selected month",
    "rep_gen": "Generated Reports",
    "rep_gen_ov": "Reports generated by a background process can be downloaded from here.  They will be kept for 2 weeks and then automatically deleted",
    "locations_distance": "Distance travelled",
    "locations_kml": "KML",
    "rep_au": "Include all users in project",
    "rep_d": "Generation time (seconds)",
    "rep_my": "Month and year of the report",

    // Console
    "co_dd": "Drill Down",
    "co_up": "Up",
    "co_tid": "Include text value in Download",
    "co_b": "Bulk Change",
    "co_drag": "Drag and drop questions to change the order in which they are shown",

    // Mailouts
    "mo_anon": "Anonymous",
    "mo_ce": "Clear Existing Unsent Emails",
    "mo_ns": "Mailout not selected",
    "mo_na": "Not Authorised",
    "mo_nf": "The survey form was not found",
    "mo_ss": "This survey form has been sucessfully submitted",
    "mo_exp": "This survey form has expired",
    "mo_del_done": "This survey form has been deleted",
    "mo_se": "System Error",
    "mo_del": "Are you sure that you want to delete mailout \"%s1\".  Recipients who have already been sent emails will no longer be able to respond. ",
    "mo_blocked": "This survey form has been blocked.  Try again later or contact the administrator to unblock",
    "mo_import": "Import Emails",
    "mo_export": "Export Emails",
    "mo_subject": "Email Subject",
    "mo_content": "Email Content",
    "mo_sent": "Sent",
    "mo_sd": "Status Details",
    "mo_ms": "Multiple Submissions",
    "mo_send": "Are you sure you are ready to send the emails?",
    "mo_gen": "Are you sure you are want to generate the links now?  They are created automatically when you use the email unsent button however you can generate them now if you want to send them via a different channel. For single submission links, once created they will expire in 30 days.",
    "mo_oi": "The following opt-in messages have been sent and are awaiting a response. ",

    // AWS Services
    "svc_at": "Auto Translate",
    "svc_from_l": "From Language",
    "svc_to_l": "To Language",

    // Logs
    "log_hr": "Hourly Summary",
    "log_sd": "Survey Details",

    // Login
    "li_success": "Login successful",
    "li_failure": "Login failure",
    "li_continue": "Return to applications",

    // Case Management
    "cm_fs": "Final status",
    "cm_sq": "Status Question",
    "cm_cq": "Criticality Question",
    "cm_aa": "Add an Alert",
    "cm_dcms": "There is already a case management event with that name for this survey group",
    "cm_alert_a": "Alert if not finalised within",
    "cm_a": "Case management alert",
    "cm_bar": "Bar",
    "cm_col": "Column",
    "cm_pie": "Pie",
    "cm_alert": "Case Management Alert",
    "cm_ns": "Select a survey to see case data",
    "cm_oc": "Count of cases opened and closed per day",
    "cm_p": "Show data for last",
    "cm_7days": "7 days",
    "cm_14days": "14 days",
    "cm_30days": "30 days",
    "cm_areq": "A status question and a final status value must be specified if you are using alerts",

    "fp_nm": "No matches found",

    "pw_gen": "Generate Password",
    "pw_mm": "The password and its confirm are not the same",

    // Server state
    "ss_title": "Server State",

    // sms
    "sms_their_q": "Question for calling number",
    "sms_conv_q": "Conversation Question"
});


/*
This file is part of SMAP.

SMAP is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

SMAP is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with SMAP.  If not, see <http://www.gnu.org/licenses/>.

*/

define('app/localise',['jquery', 'i18n!lang_location/nls/lang'], function($, lang) {
	
	var dtLangFiles = {
			en: "",
			es: "/js/libs/DataTables.i18n/es.json",
			ar: "/js/libs/DataTables.i18n/ar.json",
			fr: "/js/libs/DataTables.i18n/fr.json",
			pt: "/js/libs/DataTables.i18n/pt.json",
			hi: "/js/libs/DataTables.i18n/hi.json"
	}
	
	window.localise = {	// Set global while migrating app to AMD
		
		
		setlang: function () {

			// Content
			$(".lang").each(function(index) {
				var $this = $(this);
				var code = $this.data("lang");
				if(code) {		
					$this.html(lang[code]);
				}
			});

			// tooltips
			$(".lang_tt").each(function(index) {
				var $this = $(this);
				var code = $this.data("lang_tt");
				if(code) {
					$this.prop("title", lang[code]);
				}
			});

			// placeholders
			$(".lang_ph").each(function(index) {
				var $this = $(this);
				var code = $this.data("lang_ph");
				if(code) {
					$this.prop("placeholder", lang[code]);
				}
			});

			if(typeof responsiveMobileMenu === "function") {
				rmmResizeLabels();		// Resize menus
			}
		},
		set: lang,
		dt: function() {
			return dtLangFiles[gUserLocale];
		}
	}
	
	return localise;
});
/*
This file is part of SMAP.

SMAP is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

SMAP is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with SMAP.  If not, see <http://www.gnu.org/licenses/>.

*/
/*
 deprecated



define([
         'jquery',
         'modernizr',
         'app/localise',
         'app/globals'], 
		function($, modernizr, lang, globals) {

	return {
	
		init: init,
		setHtml: setHtml,
		populateFormList: populateFormList
	};
	



	function init() {
		
		var i;
		
		
		
		/*
		 * Add options for units
		 *
		$('#ssc_function').change(function () {
			var fn = $(this).val();
			$('#ssc_name').val(fn);		// Default name to name of function
			$("#ssc_form").empty();
			if(fn !== "") {
				populateFormList(fn);
				if(fn === "area") {
					$('#ssc_units').html(
							'<option value="hectares">Hectares</option>' +
							'<option value="square meters">Square Meters</option>'
							);
				} else if(fn === "length") {
					$('#ssc_units').html(
							'<option value="km">Kilometers</option>' +
							'<option value="meters">Meters</option>'
							);
				}
			} else {
				$('#ssc_units').html('');
			}
		});
			
		$('#add_ssc_modal').on('show.bs.modal', function (event) {
			  $(this).find("form")[0].reset();
			  
		});
	}


	function setHtml(selector, sscList) {
		var h =[],
			idx = -1,
			i;
		
		if(sscList) {
			for(i = 0; i < sscList.length; i++) {
				h[++idx] = '<tr>';
					h[++idx] = '<td>';
					h[++idx] = sscList[i].name;
					h[++idx] = '</td>';	
					h[++idx] = '<td>';
					h[++idx] = sscList[i].form;
					h[++idx] = '</td>';	
					h[++idx] = '<td>';
					h[++idx] = sscList[i].fn;
					h[++idx] = '</td>';	
					h[++idx] = '<td>';
					h[++idx] = sscList[i].units;
					h[++idx] = '</td>';	
					h[++idx] = '<td>';
					h[++idx] = sscList[i].parameters;
					h[++idx] = '</td>';	
					h[++idx] = '<td><button value="';
					h[++idx] = i;
					h[++idx] = '" class="btn btn-danger ssc_btn_rem"><span class="glyphicon glyphicon-trash edit_icon"></span></button></td>';
					
				h[++idx] = '</tr>';
			}
	
			$(selector).html(h.join(''));
		}
		
	}


	function populateFormList(fn) {
		var sId = $('#s_id').val(),
			i,
			h = [],
			idx = -1;
		
		addHourglass();
	 	$.ajax({
			url: "/surveyKPI/ssc/" + globals.gCurrentSurvey + "/" + fn + "/forms",
			dataType: 'json',
			cache: false,
			success: function(data) {
				removeHourglass();
				console.log("Forms");
				console.log(data);
				if(data.length === 0) {
					$('#ssc_alert').show().text("No forms found that can be used for function " + fn);
					setTimeout(function() {
						$('#ssc_function').focus();
						}, 0);		
				} else {
					
					for(i = 0; i < data.length; i++) {
						h[++idx] = '<option value="';
						h[++idx] = data[i].fId;
						h[++idx] = '">';
						h[++idx] = data[i].name;
						h[++idx] = '</option>';
					}
					$("#ssc_form").html(h.join(''));
				}
			},
			error: function(xhr, textStatus, err) {
				removeHourglass();
				if(xhr.readyState == 0 || xhr.status == 0) {
		              return;  // Not an error
				} else {
					$('#ssc_alert').show().text("Error: Failed to get list of forms: " + err);

				}
			}
		});
	}
	


});

 */;
define("app/ssc", function(){});

/*
 This file is part of SMAP.

 SMAP is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 SMAP is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with SMAP.  If not, see <http://www.gnu.org/licenses/>.

 */

/*
 * Quick solution to issue of legacy globals after migrating to AMD / require.js
 */
define('app/globals',[],function () {
    window.globals = {

        // Security groups
        GROUP_ADMIN: 1,
        GROUP_ANALYST: 2,
        GROUP_ENUM: 3,
        GROUP_ORG_ADMIN : 4,
        GROUP_MANAGE: 5,
        GROUP_SECURITY: 6,
        GROUP_VIEW_DATA: 7,
        GROUP_ENTERPRISE : 8,
        GROUP_OWNER : 9,
		GROUP_VIEW_OWN_DATA : 10,
	    GROUP_MANAGE_TASKS : 11,
	    GROUP_DASHBOARD : 12,
        GROUP_LINKAGES : 13,
        GROUP_CONSOLE_ADMIN : 14,

        REC_LIMIT: 200,     // Page size for table views in analysis
	    MAP_REC_LIMIT: 10000,    // Max size for map views in analysis

        gProjectList: undefined,
        gRoleList: undefined,
        gCmSettings: undefined,
        gCurrentProject: 0,
        gCurrentSurvey: 0,
        gCurrentSurveyIdent: undefined,
	    gGroupSurveys: {},
	    gSubForms: {},
        gCurrentForm: 0,
        gCurrentLayer: undefined,
        gLoggedInUser: undefined,
        gEditingReportProject: undefined,   		// Set if fieldAnalysis called to edit a report
        gIsAdministrator: false,
        gIsEnum: false,
        gIsAnalyst: false,
	    gIsDashboard: false,
        gIsManage: false,
        gIsOrgAdministrator: false,
        gIsSecurityAdministrator: false,
        gIsEnterpriseAdministrator: false,
        gIsLinkFollower: false,
        gIsServerOwner: false,
        gIsConsoleAdmin: false,
        gViewData: false,
	    gManageTasks: false,
        gBillingData: false,
        gOrgBillingData: false,
        gSendTrail: 'off',
        gViewIdx: 0,
        gSelector: new Selector(),
        gOrgId: 0,
        gTimezone: undefined,
	    gEnterpriseName: undefined,
	    gSetAsTheme: undefined,
	    gNavbarColor: undefined,

        gRegions: undefined,
        gRegion: {},

        gServerCanSendEmail: false,

        // Reports
        gEmailEnabled: false,
        gFacebookEnabled: false,
        gTwitterEnabled: false,

        // Tasks
        gCurrentUserId: undefined,
        gCurrentUserName: undefined,
        gAssignmentsLayer: undefined,
        gPendingUpdates: [],
        gCurrentTaskGroup: undefined,
	    gCurrentMailout: undefined,
        gTaskList: undefined,
        gCurrentSurveyIndex: 0,
	    gCurrentInstance: undefined,
        gAlertSeen: false,
        gLastAlertTime: undefined,

        // Editor
        gExistingSurvey: false,		// Set true if modifying an existing survey
        gElementIndex: 0,			// Manage creation of unique identifier for each element (question, option) in editor
        gHasItems: false,			// Set true if there are questions or choice lists in the survey
        gNewQuestionButtonIndex: 0,	// Manage creation of unique identifier for buttons that add new questions
        gNewOptionButtonIndex: 0,
        gSId: 0,
        gLanguage: 0,
        gLanguage1: 0,
        gLanguage2: 0,
        errors: [],
        changes: [],
        gErrorPosition: 0,
        gSelProperty: 'label',
        gSelLabel: 'Question Text',
        gSelQuestionProperty: 'label',
        gSelQuestionLabel: 'Question Text',
        gSelChoiceProperty: 'label',
        gSelChoiceLabel: 'Question Text',
        gIsQuestionView: true,
        gShowingChoices: false,
        gMaxOptionList: 0,
        gLatestOptionList: undefined,	// Hack to record the last option list name added
	    gCsvFiles: undefined,

        gListName: undefined,					// Choice Modal parameters, Set if started from choice list view
        gOptionList: undefined,					// The option list name applying to this set of choices
        gSelOptionId: undefined,				// Selected option index
        gFormIndex: undefined,					// Selected form index
        gItemIndex: undefined,					// Selected question index
        gSelectedFilters: undefined,
        gFilterArray: undefined,

        gSaveInProgress: false,

        // Dashboard
        gMainTable: undefined,			// Data tables
        gReports: undefined,			// reports
        gCharts: {},					// charts
	    gRecordMaps: [],                // Maps shown when editing a record
	    gRecordChangeMaps: [],          // Maps shown when viewing change history
        gMapLayersShown: false,
        gViewId: 0,						// Current survey view

	    gTraining: undefined,
	    gRefreshRate: 0,

        gMapboxDefault: undefined,		// Mapbox key
        
        model: new Model()

    }

    return window.globals;

    function Selector() {

        this.dataItems = new Object();
        this.surveys = new Object();
        this.surveysExtended = new Object();
        this.surveyLanguages = new Object();
        this.surveyQuestions = new Object();
        this.surveyMeta = new Object();
        this.surveyAlerts = new Object();
        this.questions = new Object();
        this.allSurveys;				// Simple list of surveys
        this.allRegions;
        this.sharedMaps;
        this.views = [];			// Simple list of views
        this.maps = {};				// map panels indexed by the panel id
        this.changed = false;
        this.SURVEY_KEY_PREFIX = "surveys";
        this.TASK_KEY = "tasks";
        this.TASK_COLOR = "#dd00aa";
        this.SURVEY_COLOR = "#00aa00";
        this.SELECTED_COLOR = "#0000aa";
        this.currentPanel = "map";

        /*
         * Get Functions
         */
        this.getAll = function () {
            return this.dataItems;
        };

        this.getItem = function (key) {
            return this.dataItems[key];
        };

        // Return all the table data available for a survey
        this.getFormItems = function (sId) {
            var tableItems = new Object();
            for (var key in this.dataItems) {
                var item = this.dataItems[key];
                if (item.table == true && item.sId == sId) {
                    tableItems[key] = item;
                }
            }
            return tableItems;
        };

        this.getSurvey = function (key) {
            return this.surveys[key];
        };

        this.getSurveyExtended = function (key) {
            return this.surveysExtended[key];
        };

        this.getSurveyQuestions = function (sId, language) {
            var langQ = this.surveyQuestions[sId];
            if (langQ) {
                return langQ[language];
            } else {
                return null;
            }
        };

        this.getSurveyMeta = function (key) {
            return this.surveyMeta[key];
        };

        this.getSurveyAlerts = function (key) {
            return this.surveyAlerts[key];
        };

        this.getSurveyLanguages = function (key) {
            return this.surveyLanguages[key];
        };

        // Returns the list of surveys on the home server
        this.getSurveyList = function () {
            return this.allSurveys;
        };

        this.getRegionList = function () {
            return this.allRegions;
        };

        this.getSharedMaps = function () {
            return this.sharedMaps;
        };

        // deprecate question meta should be replaced by all question details in the question list
        this.getQuestion = function (qId, language) {
            var langQ = this.questions[qId];
            if (langQ) {
                return langQ[language];
            } else {
                return null;
            }
        };

        /*
         * Get the question details that came with the question list
         * This approach should replace the concept of "question meta"
         */
        this.getQuestionDetails = function (sId, qId, language) {
            var qList = this.getSurveyQuestions(sId, language),
                i;

            if (qList) {
                for (i = 0; i < qList.length; i++) {
                    if (qList[i].id == qId) {
                        return qList[i];
                    }
                }
            }
            return null;
        };

        this.hasQuestion = function (key) {
            if (this.questions[key] != undefined) {
                return true;
            } else {
                return false;
            }
        };

        // Return the list of current views
        this.getViews = function () {
            return this.views;
        };

        // Return a map if it exists
        this.getMap = function (key) {
            return this.maps[key];
        };


        /*
         * Set Functions
         */
        this.addDataItem = function (key, value) {
            this.dataItems[key] = value;
            this.changed = true;
        };

        this.clearDataItems = function () {
            this.dataItems = new Object();
        };

        this.clearSurveys = function () {
            this.surveys = new Object();
            this.surveyLanguages = new Object();
            this.surveyQuestions = new Object();
            this.surveyMeta = new Object();
            this.surveyAlerts = new Object();
            this.questions = new Object();
            this.allSurveys = undefined;
            this.allRegions = undefined;
        };

        this.setSurveyList = function (list) {
            this.allSurveys = list;
            if (typeof list[0] !== "undefined") {
                this.selectedSurvey = list[0].sId;
            }
        };

        this.setSurveyLanguages = function (key, value) {
            this.surveyLanguages[key] = value;
        };

        this.setSurveyQuestions = function (sId, language, value) {
            var langQ = new Object();
            langQ[language] = value;
            this.surveyQuestions[sId] = langQ;
        };

        this.setSurveyMeta = function (key, value) {
            this.surveyMeta[key] = value;
        };

        this.setSurveyAlerts = function (key, value) {
            this.surveyAlerts[key] = value;
        };

        this.setRegionList = function (list) {
            this.allRegions = list;
        };

        this.setSharedMaps = function (list) {
            this.sharedMaps = list;
        };

        this.addSurvey = function (key, value) {
            this.surveys[key] = value;
        };

        this.addSurveyExtended = function (key, value) {
            this.surveysExtended[key] = value;
        };

        this.setSelectedSurvey = function (survey) {
            this.selectedSurvey = survey;
        };

        this.setSelectedQuestion = function (id) {
            this.selectedQuestion = id;
        };

        this.addQuestion = function (qId, language, value) {
            var langQ = this.questions[qId];
            if (!langQ) {
                this.questions[qId] = new Object();
                langQ = this.questions[qId];
            }
            langQ[language] = value;
        };

        // Set the list of views to the passed in array
        this.setViews = function (list) {
            this.views = list;
        };

        // Set the passed in map into the maps object indexed by key
        this.setMap = function (key, value) {
            this.maps[key] = value;
        };

    }

    /*
     * Model for Survey editing
     */
    function Model() {

        this.survey = undefined;
        this.translateChanges = [];
        this.currentTranslateChange = 0;
        this.savedSettings = undefined;
        this.forceSettingsChange = false;

	    // A list of valid appearances for each question type
	    this.qAppearances = {
		    'begin group': ['page', 'w', 'no-collapse'],
            'begin repeat': ['extendable', 'no-collapse'],
		    string: ['numbers', 'thousands-sep', 'w', 'url'],
		    note: ['w'],
            select1: ['select1_type', 'search', 'likert', 'no-buttons', 'w'],
            select: ['select_type', 'search', 'no-buttons', 'w'],
            image: ['image_type', 'selfie', 'new', 'w'],
            int:['thousands-sep', 'w'],
		    geopoint:['placement-map', 'w'],
		    audio:['w', 'new'],
		    video:['selfie', 'w', 'new'],
		    barcode:['read_nfc', 'w'],
		    date:['date_type', 'w'],
		    dateTime:['date_type', 'no-calendar', 'w'],
		    time:['w'],
            decimal:['thousands-sep', 'bearing', 'w'],
		    geotrace:['placement-map', 'w'],
		    geoshape:['placement-map', 'w'],
		    acknowledge:['w'],
		    range:['w', 'rating', 'vertical', 'picker'],
		    file:['w'],
		    rank:['w'],
            geocompound:['w', 'placement-map']
	    };

	    this.appearanceDetails = {
		    'page': {
			    field: 'a_page',
			    type: 'select',
                rex: 'field-list|table-list',
                valIsAppearance: true,
			    value_offset: 0,
                undef_value: ''
		    },
		    'image_type': {
			    field: 'a_image_type',
			    type: 'select',
			    rex: 'annotate|draw|signature',
			    valIsAppearance: true,
			    value_offset: 0,
			    undef_value: ''
		    },
		    'select1_type': {
			    field: 'a_select1_type',
			    type: 'form',
			    rex: 'minimal|quick$|autocomplete|columns|quickcompact|image-map|compact'
		    },
		    'select_type': {
			    field: 'a_select_type',
			    type: 'form',
			    rex: 'minimal|autocomplete|columns|image-map|compact|autocomplete-minimal'
		    },
		    'date_type': {
			    field: 'a_date_type',
			    type: 'select',
			    rex: 'no-calendar|month-year|year|coptic|ethiopian|islamic|myanmar|persian|bikram-sambat',
			    valIsAppearance: true,
			    value_offset: 0,
			    undef_value: ''
		    },
		    'no-calendar': {
			    field: 'a_no_calendar',
			    type: 'boolean',
			    rex: 'no-calendar'
		    },
		    'placement-map': {
			    field: 'a_placement-map',
			    type: 'boolean',
			    rex: 'placement-map'
		    },
		    'search': {
			    field: 'a_search',
			    type: 'form',
			    rex: 'search\\(|lookup_choices\\('
		    },
		    'rating': {
			    field: 'a_rating',
			    type: 'boolean',
			    rex: 'rating'
		    },
		    'likert': {
			    field: 'a_likert',
			    type: 'boolean',
			    rex: 'likert'
		    },
		    'no-buttons': {
			    field: 'a_no_buttons',
			    type: 'boolean',
			    rex: 'no-buttons'
		    },
		    'selfie': {
			    field: 'a_selfie',
			    type: 'boolean',
			    rex: 'selfie'
		    },
		    'new': {
			    field: 'a_new',
			    type: 'boolean',
			    rex: 'new'
		    },
		    'read_nfc': {
			    field: 'a_read_nfc',
			    type: 'boolean',
			    rex: 'read_nfc'
		    },
		    'vertical': {
			    field: 'a_vertical',
			    type: 'boolean',
			    rex: 'vertical'
		    },
		    'picker': {
			    field: 'a_picker',
			    type: 'boolean',
			    rex: 'picker'
		    },
		    'bearing': {
			    field: 'a_bearing',
			    type: 'boolean',
			    rex: 'bearing'
		    },
            'thousands-sep': {
                field: 'a_sep',
                type: 'boolean',
                rex: 'thousands-sep'
            },
		    'no-collapse': {
			    field: 'a_no_collapse',
			    type: 'boolean',
			    rex: 'no-collapse',
                value_offset: 0
		    },
            'extendable': {
                field: 'a_extendable',
                type: 'boolean',
                rex: 'extendable'
            },
		    'numbers': {
			    field: 'a_numbers',
			    type: 'boolean',
			    rex: 'numbers'
		    },
		    'url': {
			    field: 'a_url',
			    type: 'boolean',
			    rex: 'url'
		    },
		    'w': {
			    field: 'a_width',
			    type: 'select',
                rex: 'w10|w[1-9]',
                value_offset: 1,
                undef_value: ''
		    }
	    };

        // A list of valid parameters for each question type
        this.qParams = {
            string: ['rows', 'auto_annotate', 'source', 'from_lang', 'to_lang', 'medical', 'med_type'],
	        calculate: ['auto_annotate', 'source', 'from_lang', 'to_lang', 'medical', 'med_type'],
	        barcode: ['auto'],
            image: ['max-pixels', 'auto'],
	        video: ['auto'],
	        audio: ['auto', 'quality'],
            range: ['start', 'end', 'step'],
            select: ['randomize'],
            select1: ['randomize'],
            rank: ['randomize'],
            parent_form: ['form_identifier', 'key_question', 'auto'],
	        child_form: ['form_identifier', 'key_question', 'auto'],
	        geopoint: ['auto'],
            geotrace: ['geotextlength'],
            geoshape: ['geotextlength'],
            geocompound: ['geotextlength'],
            'begin repeat':['ref', 'instance_order', 'instance_count', 'key_policy'],
	        chart: ['chart_type', 'stacked', 'normalized']
        };

        this.paramDetails = {
	        rows: {
	            field: 'p_rows',
                type: 'integer'
            },
            geotextlength: {
                field: 'p_geotextlength',
                type: 'integer'
            },
            'max-pixels': {
	            field: 'p_max_pixels',
                type: 'integer'
            },
            start: {
	            field: 'p_start',
                type: 'number'
            },
	        end: {
		        field: 'p_end',
		        type: 'number'
	        },
	        step: {
		        field: 'p_step',
		        type: 'number'
	        },
	        randomize: {
		        field: 'p_randomize',
		        type: 'boolean'
	        },
	        auto: {
		        field: 'p_auto',
		        type: 'boolean'
	        },
	        quality: {
		        field: 'p_quality',
		        type: 'select'
	        },
	        auto_annotate: {
		        field: 'p_auto_annotate',
		        type: 'boolean'
	        },
	        medical: {
		        field: 'p_medical',
		        type: 'boolean'
	        },
	        med_type: {
		        field: 'p_med_type',
		        type: 'select'
	        },
	        source: {
		        field: 'p_source',
		        type: 'select'
	        },
	        from_lang: {
		        field: 'from_lang',
		        type: 'select'
	        },
	        to_lang: {
		        field: 'to_lang',
		        type: 'select'
	        },
	        form_identifier: {
		        field: 'p_form_identifier',
		        type: 'select'
	        },
	        key_question: {
		        field: 'p_key_question',
                type: 'select'
            },
	        ref: {
		        field: 'p_ref',
		        type: 'select'
	        },
	        instance_order: {
		        field: 'p_instance_order',
		        type: 'select'
	        },
	        instance_count: {
		        field: 'p_instance_count',
		        type: 'integer'
	        },
	        key_policy: {
		        field: 'p_key_policy',
		        type: 'select'
	        },
	        chart_type: {
		        field: 'p_chart_type',
		        type: 'select'
	        },
	        stacked: {
		        field: 'p_stacked',
		        type: 'boolean'
	        },
	        normalized: {
		        field: 'p_normalized',
		        type: 'boolean'
	        },
	        _other: {
		        field: 'p_other',
		        type: 'text'
	        }
        };

        this.qTypes = [{
	            name: "Text",
	            trans: "rev_text",
	            type: "string",
	            icon: "font",
	            canSelect: true,
	            visible: true,
		        source: "user",
		        compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field", "conversation", "phone"]
            },
            {
                name: "Note",
                type: "note",
                trans: "c_note",
                icon: "pencil-alt",
                canSelect: true,
                visible: true,
                source: "user",
	            compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field", "conversation", "phone"]
            },
            {
                name: "Select One",
                type: "select1",
                trans: "ed_s1",
                image: "/images/select1_64.png",
                canSelect: true,
                visible: true,
				source: "user",
	            compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field", "conversation", "phone"]
            },
            {
                name: "Select Multiple",
                type: "select",
                trans: "ed_s",
                image: "/images/select_64.png",
                canSelect: true,
                visible: true,
                source: "user",
	            compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field", "conversation", "phone"]
            },
            {
                name: "Form",
                type: "begin repeat",
                trans: "c_rep_type",
                icon: "redo-alt",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "Group",
                type: "begin group",
                trans: "sr_g",
                icon: "folder-open",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "Image",
                type: "image",
                trans: "ed_image",
                icon: "camera",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "Integer",
                type: "int",
                trans: "ed_int",
                text: "#",
                canSelect: true,
                visible: true,
                source: "user",
                compatTypes: ["pdf_field"]
            },
            {
                name: "Phone Number",
                type: "phone",
                trans: "c_phone",
                icon: "phone",
                canSelect: true,
                visible: true,
                source: "user",
                compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field", "conversation"]
            },
            {
                name: "Conversation",
                type: "conversation",
                trans: "ed_mat_c",
                icon: "sms",
                canSelect: true,
                visible: true,
                compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field", "phone"]
            },
            {
                name: "GPS Point",
                type: "geopoint",
                trans: "ed_gps",
                icon: "map-marker-alt",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "Calculation",
                type: "calculate",
                trans: "ed_calc",
                calculation: true,
                image: "/images/calc_64.png",
                canSelect: true,
                visible: true,
                source: "user",
	            compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field"]
            },
            {
                name: "Audio",
                type: "audio",
                trans: "ed_audio",
                icon: "volume-up",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "Video",
                type: "video",
                trans: "ed_video",
                icon: "video",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "Barcode",
                type: "barcode",
                trans: "ed_bc",
                icon: "barcode",
                canSelect: true,
                visible: true,
                source: "user",
                compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field"]
            },
            {
                name: "Date",
                type: "date",
                trans: "c_date",
                icon: "calendar-alt",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "Date and Time",
                type: "dateTime",
                trans: "ed_dt",
                icon: "calendar-alt, clock",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "Time",
                type: "time",
                trans: "ed_t",
                icon: "clock",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "Decimal",
                type: "decimal",
                trans: "ed_dec",
                text: "#.#",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "GPS Line",
                type: "geotrace",
                trans: "ed_gps_line",
                image: "/images/linestring_64.png",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "GPS Area",
                type: "geoshape",
                trans: "ed_gps_area",
                image: "/images/polygon_64.png",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "Acknowledge",
                type: "acknowledge",
                trans: "ed_ack",
                text: "OK",
                canSelect: true,
                visible: true,
                source: "user",
                compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field"]
            },
            {
                name: "Range",
                type: "range",
                trans: "ed_range",
                icon: "arrows-alt-h",
                text: "Range",
                canSelect: true,
                visible: true,
                source: "user",
	            compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field"]
            },
            {
                name: "Chart",
                type: "chart",
                trans: "c_chart",
                icon: "chart-bar",
                text: "Chart",
                canSelect: true,
                visible: true,
                source: "user",
                compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field"]
            },
	        {
		        name: "Parent Form",
		        type: "parent_form",
		        trans: "c_parent_form",
		        icon: "file-upload",
		        text: "Parent Form",
		        canSelect: true,
		        visible: true,
		        source: "user"
	        },
	        {
		        name: "Child Form",
		        type: "child_form",
		        trans: "c_child_form",
		        icon: "file-download",
		        text: "Child Form",
		        canSelect: true,
		        visible: true,
		        source: "user"
	        },
            {
                name: "File",
                type: "file",
                trans: "c_file",
                icon: "file",
                canSelect: true,
                visible: true,
                source: "user",
                compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field"]
            },
            {
                name: "Rank",
                type: "rank",
                trans: "c_rank",
                icon: "sort-amount-down-alt",
                canSelect: true,
                visible: true,
                source: "user",
                compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field"]
            },
	        {
		        name: "Server Calculation",
		        type: "server_calculate",
		        trans: "ed_s_calc",
		        calculation: true,
		        image: "/images/server_calc_64.png",
		        canSelect: true,
		        visible: true,
		        source: "user",
		        compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field"]
	        },
            {
                name: "Compound Pdf Image",
                type: "pdf_field",
                trans: "ed_ci",
                calculation: false,
                icon: "pallet",
                canSelect: true,
                visible: false,
            },
            {
                name: "Compound map",
                type: "geocompound",
                trans: "ed_cm",
                calculation: false,
                icon: "map-pin",
                canSelect: true,
                visible: false,
                compatTypes: ["geotrace"]
            },
            {
                name: "Unknown Type",
                icon: "ellipses-h",
                canSelect: false
            }
        ];

        // Set the survey model
        this.setSurveyData = function (data) {
            this.survey = data;
            this.survey.forms_orig = $.extend(true, {}, data.forms);
            this.survey.optionLists_orig = $.extend(true, {}, data.optionLists);
        }

        // Save the settings for the survey
        this.save_settings = function () {

            var settings = JSON.stringify(this.getSettings(true));

            addHourglass();
            $.ajax({
                type: "POST",
                contentType: "application/x-www-form-urlencoded",
                cache: false,
                data: { settings: settings },
                url: "/surveyKPI/surveys/save_settings/" + globals.gCurrentSurvey,
                success: function (data, status) {
                    removeHourglass();
                    if(handleLogout(data)) {
                        globals.model.savedSettings = settings;
                        globals.model.survey.pdfTemplateName = data;
                        globals.model.forceSettingsChange = false;
                        $('#save_settings').prop("disabled", true);

                        $('.formName').text(globals.model.survey.displayName);
                        $('#m_media').prop('href', '/app/resources.html?survey=true&survey_name=' + globals.model.survey.displayName);

                        $('#settingsModal').modal("hide");
                    }
                },
                error: function (xhr, textStatus, err) {
                    removeHourglass();
                    if(handleLogout(xhr.responseText)) {
                        if (xhr.readyState == 0 || xhr.status == 0) {
                            return;  // Not an error
                        } else {
                            bootbox.alert("Error saving settings. " + htmlEncode(xhr.responseText));
                        }
                    }
                }
            });

        };

        // Modify a label for a question or an option called from translate where multiple questions can be modified at once if the text is the same
        this.modLabel = function (language, changedQ, newVal, element, prop) {

            var labelMod = {
                changeType: prop,
                action: "update",
                items: []
            }

            var i,
                label = {},
                item,
                item_orig,
                qname,
                translation;


            for (i = 0; i < changedQ.length; i++) {
                translation = {
                    changeType: prop,
                    action: "update",
                    source: "editor"

                };

                // For questions
                if (typeof changedQ[i].form !== "undefined") {

                    label.formIndex = changedQ[i].form;
                    label.itemIndex = changedQ[i].question;
                    item = this.survey.forms[label.formIndex].questions[label.itemIndex];
                    item_orig = this.survey.forms_orig[label.formIndex].questions[label.itemIndex];

                    label.type = "question";
                    label.name = item.name;
                    label.prop = "label";
                    label.qId = item.id;

	                if(changedQ[i].constraint_msg) {
		                label.propType = "constraint_msg";
		                label.oldVal = item_orig.labels[language][label.propType];
	                } else if(changedQ[i].required_msg) {
		                label.propType = "required_msg";
		                label.oldVal = item_orig.labels[language][label.propType];
	                } else if(changedQ[i].guidance_hint) {
		                label.propType = "guidance_hint";
		                label.oldVal = item_orig.labels[language][label.propType];
	                } else if(changedQ[i].hint) {
		                label.propType = "hint";
		                label.oldVal = item_orig.labels[language][label.propType];
	                } else {
		                label.propType = "text";
		                label.oldVal = item_orig.labels[language][element];
	                }

                } else {
	                // For options
                    label.optionList = changedQ[i].optionList;
                    label.optionIdx = changedQ[i].option;

                    item = this.survey.optionLists[label.optionList].options[label.optionIdx];
                    item_orig = this.survey.optionLists_orig[label.optionList].options[label.optionIdx];

                    label.type = "option";
                    label.name = item.value;
	                label.propType = "text";
	                label.oldVal = item_orig.labels[language][element];
                }

                label.newVal = newVal;

                label.element = element;
                label.languageName = language;
                label.allLanguages = false;

                label.languageName = this.survey.languages[language].name;			// For logging the event
                var form = this.survey.forms[label.formIdx];

                if (item.text_id) {
                    label.key = item.text_id;
                } else {
                    // Create reference for this new Label
                    if (typeof changedQ[i].form !== "undefined") {
                        label.key = "/" + form.name + "/" + item.name + ":label";	// TODO hint
                    } else {
                        label.key = "/" + form.name + "/" + qname + "/" + item.name + ":label";
                    }
                }

                translation.property = label;

                labelMod.items.push(translation);
            }

            this.removeDuplicateTranslateChange(this.translateChanges, labelMod);
            if (labelMod.items[0].property.newVal !== labelMod.items[0].property.oldVal) {		// Add if the value has changed
                this.currentTranslateChange = this.translateChanges.push(labelMod) - 1;
                //this.doChange();				// Apply the current change
            }

            $('.m_save_survey').find('.badge').html(this.translateChanges.length);
            if (this.translateChanges.length > 0) {
                $('.m_save_survey').removeClass('disabled').prop('disabled', false);
	            $('#m_auto_translate').closest('li').addClass("disabled").prop("disabled", true);
            } else {
                $('.m_save_survey').addClass('disabled').prop('disabled', true);
	            $('#m_auto_translate').closest('li').removeClass("disabled").prop("disabled", false);
            }
        }

        // Clear the change list
        this.clearChanges = function () {
            this.translateChanges = [];
            $('.m_save_survey').find('.badge').html(this.translateChanges.length);
            if (this.translateChanges.length > 0) {
                $('.m_save_survey').removeClass('disabled').prop('disabled', false);
            } else {
                $('.m_save_survey').addClass('disabled').prop('disabled', true);
	            $('#m_auto_translate').closest('li').removeClass("disabled").prop("disabled", false);
            }
        }

        /*
         * If the label has been modified before then remove it from the change list
         */
        this.removeDuplicateTranslateChange = function () {
            // TODO
        }

        /*
         * Functions for managing settings
         */
        this.getSettings = function (save) {
            var current = this.createSettingsObject(
                $('#set_survey_name').val(),
                $('#set_instance_name').val(),
                $('#set_style').val(),
                $('#set_project_name option:selected').val(),
                $('#set_default_language option:selected').text(),
                $('#task_file').prop('checked'),
                $('#timing_data').prop('checked'),
	            $('#audit_location_data').prop('checked'),
	            $('#track_changes').prop('checked'),
	            $('#hide_on_device').prop('checked'),
	            $('#search_local_data').prop('checked'),
	            $('#data_survey').prop('checked'),
	            $('#oversight_survey').prop('checked'),
                $('#read_only_survey').prop('checked'),
                $('#my_reference_data').prop('checked'),
                $('#exclude_empty').prop('checked'),
                $('#compress_pdf').prop('checked'),
	            $('#default_logo').val()
            );

            // Update the model to reflect the current values
            if (save) {
                this.survey.displayName = current.displayName;
                this.survey.instanceNameDefn = current.instanceNameDefn;
                this.survey.surveyClass = current.surveyClass;
                this.survey.p_id = current.p_id;
                this.survey.def_lang = current.def_lang;
                this.survey.task_file = current.task_file;
                this.survey.timing_data = current.timing_data;
	            this.survey.audit_location_data = current.audit_location_data;
	            this.survey.track_changes = current.track_changes;
	            this.survey.hideOnDevice = current.hideOnDevice;
	            this.survey.searchLocalData = current.searchLocalData;
	            this.survey.dataSurvey = current.dataSurvey;
	            this.survey.oversightSurvey = current.oversightSurvey;
                this.survey.myReferenceData = current.myReferenceData;
                this.survey.readOnlySurvey = current.readOnlySurvey;
                this.survey.exclude_empty = current.exclude_empty;
                this.survey.compress_pdf = current.compress_pdf;
	            this.survey.default_logo = current.default_logo;
            }

            return current;
        }

        this.setSettings = function () {
            this.savedSettings = JSON.stringify(
                this.createSettingsObject(
                    this.survey.displayName,
                    this.survey.instanceNameDefn,
                    this.survey.surveyClass,
                    this.survey.p_id,
                    this.survey.def_lang,
                    this.survey.task_file,
                    this.survey.timing_data,
	                this.survey.audit_location_data,
	                this.survey.track_changes,
                    this.survey.hideOnDevice,
	                this.survey.searchLocalData,
	                this.survey.dataSurvey,
	                this.survey.oversightSurvey,
                    this.survey.myReferenceData,
                    this.survey.readOnlySurvey,
                    this.survey.exclude_empty,
                    this.survey.compress_pdf,
                    this.survey.hrk,
                    this.survey.key_policy,
	                this.survey.default_logo
                ));

            this.forceSettingsChange = false;
        }

        this.createSettingsObject = function (displayName, instanceNameDefn,
                                              surveyClass,
                                              p_id,
                                              def_lang,
                                              task_file,
                                              timing_data,
                                              audit_location_data,
                                              track_changes,
                                              hideOnDevice,
                                              searchLocalData,
                                              dataSurvey,
                                              oversightSurvey,
                                              readOnlySurvey,
                                              myReferenceData,
                                              exclude_empty,
                                              compress_pdf,
                                              default_logo) {

            var projId;
            if (typeof p_id === "string") {
                projId = parseInt(p_id);
            } else {
                projId = p_id;
            }
            return {
                displayName: displayName,
                instanceNameDefn: instanceNameDefn,
                surveyClass: surveyClass,
                p_id: projId,
                def_lang: def_lang,
                task_file: task_file,
                timing_data: timing_data,
	            audit_location_data: audit_location_data,
	            track_changes: track_changes,
                hideOnDevice: hideOnDevice,
                searchLocalData: searchLocalData,
	            dataSurvey: dataSurvey,
	            oversightSurvey: oversightSurvey,
                myReferenceData: myReferenceData,
                readOnlySurvey: readOnlySurvey,
                exclude_empty: exclude_empty,
                compress_pdf: compress_pdf,
	            default_logo: default_logo
            }
        }

        this.settingsChange = function () {
            var current = globals.model.getSettings(false);

            if (JSON.stringify(current) !== globals.model.savedSettings || globals.model.forceSettingsChange) {
                $('#save_settings').prop("disabled", false);
            } else {
                $('#save_settings').prop("disabled", true);
            }
        }

    }
});

/*
This file is part of SMAP.

SMAP is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.
uSMAP is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with SMAP.  If not, see <http://www.gnu.org/licenses/>.

*/

/*
 * Functions for manipulating a question in the editor
 */



define('app/editorMarkup',[
         'jquery',
         'modernizr',
         'app/localise',
         'app/globals'], 
		function($, modernizr, lang, globals) {

	var gGroupStacks = [];
	var linkedQuestions = {};
	
	return {	
		addOneQuestion: addOneQuestion,
		addPanelStyle: addPanelStyle,
		addQType: addQType,
		addQuestions: addQuestions,
		addMedia: addMedia,
		refresh: refresh,
		includeQuestion: includeQuestion,
		addQuestionSequence: addQuestionSequence,
		getLinkedQuestions: getLinkedQuestions,
		refreshQuestionFeaturedProperties: refreshQuestionFeaturedProperties
	};
	
	/*
	 * Add a single question
	 */
	function addOneQuestion(form, question, formIndex, qIndex, addNewButton, selProperty, update) {
		var h = [],
			idx = -1,
			questionId = "question" + formIndex + "_" + qIndex;
		
		// Support the legacy geopolygon and geolinestring types
		if(question.type === "geopolygon" || question.type === "geolinestring") {
			question.type = "begin repeat";
		}
		
		globals.gHasItems = true;
		if(!update) {					// Only increment maxQuestion if a new question is being added
			form.maxQuestion++;
		}
		
		selProperty = selProperty || globals.gSelProperty;
		
		if(addNewButton) {
			h[++idx] = addNewQuestionButton(false, false, formIndex, undefined, selProperty);
		}
		
		// Add the group to the group stack
		if(question.type === "begin group" && gGroupStacks[formIndex]) {
			gGroupStacks[formIndex].groupStack.push(question.name);
		}
		
		h[++idx] = addPanelStyle(question.type, formIndex, qIndex, question.error, questionId, undefined);
		h[++idx] = '<div class="card-body">';
			h[++idx] = addErrorMsg(question.errorMsg);
			h[++idx] = '<div class="row">';
				
			// Add question type
				h[++idx] = '<div class="col-sm-2 col-12 ">';
					h[++idx] = '<div class="question_type';
						if(question.published) {
							h[++idx] = ' readonly';
						}
						h[++idx] = '">';
						h[++idx] = addQType(question.type);
					h[++idx] = '</div>';
				h[++idx] = '</div>';	// End of question type cell

				// Add question name cell
				h[++idx] = '<div class="col-sm-3 col-12 "><input class="qname form-control has_tt" data-toggle="tooltip" title="';
				h[++idx] = question.name;
				h[++idx] = '" value="';
				h[++idx] = question.name;
				h[++idx] = '" ';
						
				if(question.published) {				// Mark disabled if the question has been published
					h[++idx] = 'readonly="true" ';
				}
				h[++idx] = 'type="text"></div>';		// End of name

				// Add feature property cell
				h[++idx] = addFeaturedProperty(question, formIndex, qIndex, undefined, undefined, questionId);
				
				// Add buttons
				h[++idx] = '<div class="col-sm-2 col-12 q_icons_col">';
					h[++idx] = '<div class="btn-group">';

						h[++idx] = '<button tabindex="-1" class="btn btn-light delete_question ml-1" data-id="';
						h[++idx] = questionId;
						h[++idx] = '">';
						h[++idx]='<i class="fas fa-trash-alt edit_icon"></i>';
						h[++idx]='</button>';

						if(question.type === "begin repeat" 
								|| question.type === "begin group") {
							
							h[++idx] = '<a button tabindex="-1" class="btn btn-light ml-1" data-toggle="collapse" href="#collapse';
							h[++idx] = globals.gElementIndex;
							h[++idx]='"><i class="fas fa-chevron-down edit_icon"></i></a>';
						} else if(question.type.indexOf("select") === 0 || question.type === "rank") {
						
							h[++idx] = '<a button tabindex="-1" class="btn btn-light edit_choice ml-1" ';
							h[++idx]='"><i class="fa fa-edit edit_icon"></i></a>';
							
						}
					 
					h[++idx] = '</div>';
				h[++idx] = '</div>';		// End of button cell
				h[++idx] = '</div>';		// End of row
		h[++idx] = '<div id="collapse';
		h[++idx] = globals.gElementIndex;
		h[++idx] = '" class="panel-body collapse';
		if(question.type.indexOf("select") === 0 || question.type === "rank") {
			h[++idx] = ' selectquestion';
		}
		h[++idx] = '">';
		if(question.type === "begin repeat") {
			
			h[++idx] = '<div class="question-controls">';
			h[++idx] = '<div class="row">';
				h[++idx] = '<div class="col-md-6"></div>';		
				// A control to set repeat count
					h[++idx] = '<div class="col-md-6">';
						h[++idx] = '<label>Repeat Count: </label>';
						h[++idx] = '<div class="input-group">';
							h[++idx] = '<input class="form-control repeat-counts" value="';
							h[++idx] = question.calculation;
							h[++idx] = '">';
						h[++idx] = '</div>';
					h[++idx] = '</div>';
				h[++idx] = '</div>';
			h[++idx] = '</div>';
		
			h[++idx] = addSubForm(formIndex, qIndex);
			
		} 
		
		if(question.type === "begin group") {	/* Add questions up to the end group to this panel */
			h[++idx] = '<ul class="list-unstyled">';
		} else { 
			h[++idx] = '</div>';
			h[++idx] = '</li>';
		}
		
		return h.join("");
	}
	

	/*
	 * Add a single option list element to the choices view
	 */
	function addOneOptionList(list_name, addNewButton, selProperty) {
		
		var h = [],
			idx = -1,
			itemId = "ol_" + list_name,
			optionList = globals.model.survey.optionLists[list_name];
		
		globals.gHasItems = true;
		
		selProperty = selProperty || globals.gSelProperty;
		
		if(addNewButton) {
			h[++idx] = addNewQuestionButton(false, false, undefined, undefined, selProperty);		
		}
		
		h[++idx] = addPanelStyle("choices", undefined, undefined, optionList.error, itemId, list_name);
		h[++idx] = '<div class="card-body">';
		h[++idx] = addErrorMsg(optionList.errorMsg);
		h[++idx] = '<div class="row">';

		// Add choice name cell
		h[++idx] = '<div class="col-10"><input class="olname form-control has_tt" data-toggle="tooltip" title="List Name" value="';
		h[++idx] = list_name;
		h[++idx] = '" type="text"></div>';

		// Add buttons
		h[++idx] = '<div class="col-2 q_icons_col">';
		h[++idx] = '<div class="btn-group">';

		h[++idx] = '<a button tabindex="-1" class="btn btn-light edit_choice" data-id="';
		h[++idx] = itemId;
		h[++idx] = '"><i class="fa fa-edit edit_icon"></i></a>';

		h[++idx] = '<a button tabindex="-1" class="btn btn-light delete_ol" data-id="';
		h[++idx] = itemId;
		h[++idx]= '"><i class="fa fa-times-circle edit_icon text-danger" aria-hidden="true"></i></a>';

		h[++idx] = '</div>';		// End of button group
		h[++idx] = '</div>';		// End of button group cell
		h[++idx] = '</div>';		// End of row
		h[++idx] = '</div>';		// End of card
		h[++idx] = '</li>';			// End of list item


		return h.join("");
	}
	
	function addErrorMsg(msg) {
		var h = [],
			idx = -1;
		
		h[++idx] = '<p class="error-msg">';
		h[++idx] = htmlEncode(msg);
		h[++idx] = '</p>';
		return h.join("");
	}
	
	function addNewQuestionButton(after, topLevelForm, formIndex, formName, selProperty) {
		var h = [],
			idx = -1,
			addButtonClass,
			locn;
		
		//console.log(" Group Stack: " + gGroupStacks[formIndex].groupStack.join(",") + " : Last group: " + gGroupStacks[formIndex].lastGroup);
		
		addButtonClass = after ? 'add_after_button add_button' : 'add_before_button add_button';
		locn = after ? 'after' : 'before';
		
		if(topLevelForm && locn === "after") {
			addButtonClass += ' add_final_button';
		}
		
		h[++idx] = '<li>';
		h[++idx] = '<span>';

		h[++idx] = '<button id="addnew_';
		h[++idx] = globals.gNewQuestionButtonIndex++;
		h[++idx] = '" type="button" class="add_question btn dropon ';
		h[++idx] = addButtonClass;
		if(globals.gNewQuestionButtonIndex == 1) { // First button in the form
			h[++idx] = ' first_element ';
		}
		if(!(topLevelForm && locn === "after")) {
			h[++idx] = '" tabindex="-1"';
		} else {
			h[++idx] = '"';
		}
		h[++idx] = ' data-locn="';
		h[++idx] = locn;
		h[++idx] = '" data-findex="';
		h[++idx] = formIndex;

		h[++idx] = '">';
		h[++idx] = localise.set['ed_aq'];
		if(formName) {
			h[++idx] = ' - ';
			h[++idx] = formName;
		}

		h[++idx] = '</button>';
		h[++idx] = '</span>';
		h[++idx] = '</li>';
		
		return h.join('');
	}
	
	function addNewOptionListButton() {
		var h = [],
			idx = -1,
			locn;
		
		h[++idx] = '<li>';
	
		h[++idx] = '<button tabindex="-1" id="addnew_optionlist" ';
		h[++idx] = 'type="button" class="add_option_list add_button add_after_button add_final_button btn">';
		h[++idx] = localise.set["ed_ancl"];
		h[++idx] = '</button>';
		h[++idx] = '</li>';
		
		return h.join('');
	}
	
	/*
	 * Add the container for an editable element including its styling
	 */
	function addPanelStyle(type, formIndex, qIndex, error, elementId, list_name) {
		
		var h = [],
			idx = -1;
		
		h[++idx] = '<li class="card editor_element question draggable';

		if(error) {
			h[++idx] = ' error ';
		}
		
		if(type === "begin repeat" || type === "begin group") {
			h[++idx] = ' bg-secondary" id="';
		} else {
			h[++idx] = ' bg-light" id="';
		}
		h[++idx] = elementId;
		++globals.gElementIndex;
		h[++idx] = '"';
		
		// Add the question index and form index
		if(typeof formIndex !== "undefined") {
			h[++idx] = ' data-fid="';
			h[++idx] = formIndex;
			h[++idx] = '"';
		}
		if(typeof qIndex !== "undefined") {
			h[++idx] = ' data-id="';
			h[++idx] = qIndex;
			h[++idx] = '"';
		}
		if(typeof list_name !== "undefined") {
			h[++idx] = ' data-list_name="';
			h[++idx] = list_name;
			h[++idx] = '"';
		}
		
		
		h[++idx] = '>';
		
		return h.join('');
	}
	
	function addQType(type) {
		
		var i,
			j,
			tArray,
			types = globals.model.qTypes,
			h = [],
			idx = -1,
			name;
	
		for(i = 0; i < types.length; i++) {
			if(types[i].type === type) {
				name = localise.set[types[i].trans];
				h[++idx] = '<span class="has_tt" data-toggle="tooltip" title="Question type: ';
				h[++idx] = name;
				h[++idx] = '">';
				if(types[i].icon) {
					tArray = types[i].icon.split(',');
					for(j = 0; j < tArray.length; j++) {
						h[++idx] = '<i class="fas fa-';
						h[++idx] = tArray[j].trim();
						h[++idx] = ' edit_type"></i>';
					}
				} else if(types[i].image) {
					h[++idx] = '<img class="edit_image" src="';
					h[++idx] = types[i].image; 
					h[++idx] = '">';
				} else if(types[i].text) {
					h[++idx] = '<span class="edit_type">';
					h[++idx] = types[i].text; 
					h[++idx] = '</span>';
				}
				h[++idx] = '</span>';
				break;
			}
		}	
		return h.join('');

	}
	
	/*
	 * One of the questions properties will be featured so that it can be edited in the header without expanding the question
	 */
	function addFeaturedProperty(question, fId, qIndex, list_name, qname, questionId) {
		
		var h = [],
			idx = -1,
			type = "question";
		
		if(list_name) {
			type = "option";
		}
		
		h[++idx] = '<div class="col-sm-5 col-12 ';
		h[++idx] = type;
	
		h[++idx] = '">';
		
		if(list_name) {
			type = "option";
		}
		h[++idx] = getFeaturedMarkup(question, type, questionId);
		
		h[++idx] = '</div>';
		return h.join("");
	}

	/*
	 * Get Featured Markup for the question
	 */
	function getFeaturedMarkup(question, type, questionId) {
		var h = [],
			idx = -1,
			i,
			//linkedSurveys = globals.model.survey.linkedSurveys,
			selProperty = globals.gSelProperty,
			selLabel = globals.gSelLabel,
			naMedia = '<div class="naMedia text-center">' + localise.set["ed_namedia"] + '</div>';
		
			if(selProperty === "required" && type === "question"
				&& question.type != "calculate") {		// Add a boolean type
				
				h[++idx] = '<div class="row">';
				
				h[++idx] = '<div class="col-sm-4">';
			    h[++idx] = '<button type="button" class="btn labelButton ';
			    if(question[selProperty]) {
			    	h[++idx] = 'prop_yes" ';
			    } else {
			    	h[++idx] = 'prop_no" ';
			    }
			    h[++idx] = ' data-prop="';
				h[++idx] = selProperty;
				h[++idx] = '">';
				if(question[selProperty]) {
					h[++idx] = '<i class="fa fa-check-circle" aria-hidden="true"></i> ';
					h[++idx] = localise.set["c_yes"];
				} else {
					h[++idx] = '<i class="fa fa-times-circle" aria-hidden="true"></i> ';
					h[++idx] = localise.set["c_no"];
				}
			    h[++idx] = '</button>';
			    h[++idx] = '</div>';

			    /*
			     * Add the text area for the required response text
			     */
			    h[++idx] = '<div class="col-sm-4';
				if(!question[selProperty]) {
					h[++idx] = ' d-none';
				}
				h[++idx] = '">';
			    h[++idx] = '<textarea class="labelProp has_tt" tabindex="0" title="';
			    h[++idx] = localise.set['ed_r_msg'];
				h[++idx] = '" data-prop="required_msg" placeholder="';
				h[++idx] = localise.set['ed_r_msg'];
				h[++idx] = '">';
				h[++idx] = question.labels[globals.gLanguage].required_msg;
				h[++idx] = '</textarea>';
			    h[++idx] = '</div>';

				/*
                * Add the text area for conditional required
                */
				h[++idx] = '<div class="col-sm-4';
				if(!question[selProperty]) {
					h[++idx] = ' d-none';
				}
				h[++idx] = '">';
				h[++idx] = '<textarea class="labelProp has_tt" tabindex="0" title="';
				h[++idx] = localise.set['ed_r_expression'];
				h[++idx] = '" data-prop="required_expression" placeholder="';
				h[++idx] = localise.set['ed_r_expression'];
				h[++idx] = '">';
				h[++idx] = question.required_expression;
				h[++idx] = '</textarea>';
				h[++idx] = '</div>';
			    
				h[++idx] ='</div>';		// End Row
				
			} else if(selProperty === "readonly" && type === "question"
				&& question.type != "calculate") {		// Add a boolean type
				
				h[++idx] = '<div class="row">';
				
				h[++idx] = '<div class="col-sm-6">';
			    h[++idx] = '<button type="button" class="btn labelButton ';
			    if(question[selProperty]) {
			    	h[++idx] = 'prop_yes" ';
			    } else {
			    	h[++idx] = 'prop_no" ';
			    }
			    h[++idx] = ' data-prop="';
				h[++idx] = selProperty;
				h[++idx] = '">';
				h[++idx] = '<span>';
				if(question[selProperty]) {
					h[++idx] = '<i class="fa fa-check-circle" aria-hidden="true"></i> ';
					h[++idx] = localise.set["c_yes"];
				} else {
					h[++idx] = '<i class="fa fa-times-circle" aria-hidden="true"></i> ';
					h[++idx] = localise.set["c_no"];
				}
			    h[++idx] = '</span></button>';
			    h[++idx] = '</div>';

				/*
				 * Add the text area for conditional required
				 */
				h[++idx] = '<div class="col-sm-6';
				if(!question[selProperty]) {
					h[++idx] = ' d-none';
				}
				h[++idx] = '">';
				h[++idx] = '<textarea class="labelProp has_tt" tabindex="0" title="';
				h[++idx] = localise.set['ed_r_readonly'];
				h[++idx] = '" data-prop="readonly_expression" placeholder="';
				h[++idx] = localise.set['ed_r_readonly'];
				h[++idx] = '">';
				h[++idx] = question.readonly_expression;
				h[++idx] = '</textarea>';
				h[++idx] = '</div>';
			    
				h[++idx] ='</div>';		// End Row
				
			} else if(selProperty === "autoplay" && type === "question"
				&& question.type != "calculate") {		// Add a radio buttons to select autoplay status
				
				h[++idx] = '<div class="btn-group" role="group" aria-label="Autoplay Selection" data-toggle="buttons-radio">';
					// Add "none" autoplay option
					h[++idx] = '<button data-prop="autoplay" type="button" class="btn btn-light labelButton ';
					if(question[selProperty] == "none") {
						h[++idx] = "active";
					}
					h[++idx] = '" value="none">None</button>';
					
					// Add video autoplay option
					h[++idx] = '<button data-prop="autoplay" type="button" class="btn btn-light labelButton ';
					if(question[selProperty] == "video") {
						h[++idx] = "active";
					}
					h[++idx] = '" value="video">Video</button>';
					
					// Add audio autoplay option
					h[++idx] = '<button data-prop="autoplay" type="button" class="btn btn-light labelButton ';
					if(question[selProperty] == "audio") {
						h[++idx] = "active";
					}
					h[++idx] = '" value="audio">';
					h[++idx] = localise.set["ed_audio"];
					h[++idx] = '</button>';
					
				h[++idx] = '</div>';


			} else if(selProperty === "parameters" && type === "question") {		// Add button to select the parameters dialog
				h[++idx] = '<div class="row">';

				h[++idx] = '<div class="col-sm-6">';	    // start col
				h[++idx] = '<button type="button" class="btn btn-primary parameterButton"';
				h[++idx] = ' data-prop="';
				h[++idx] = selProperty;
				h[++idx] = '">';
				h[++idx] = '<i class="fa fa-edit"></i> ';
				h[++idx] = localise.set["c_edit"];
				h[++idx] = '</button>';
				h[++idx] = '</div>';		// End Col

				/*
			     * Add the text area to display the parameters
			     */
				h[++idx] = '<div class="col-sm-6">';
				h[++idx] = '<textarea class="labelProp has_tt" data-toggle="tooltip" readonly title="';
				h[++idx] = selLabel;
				h[++idx] = '" data-prop="';
				h[++idx] = selProperty;
				h[++idx] = '">';
				h[++idx] = question["parameters"];
				h[++idx] = '</textarea>';
				h[++idx] = '</div>';    // End Col

				h[++idx] ='</div>';		// End Row



			} else if(selProperty === "appearance" && type === "question") {		// Add button to select the appearance dialog
				h[++idx] = '<div class="row">';

				h[++idx] = '<div class="col-sm-6">';	    // start col
				h[++idx] = '<button type="button" class="btn btn-info appearanceButton"';
				h[++idx] = ' data-prop="';
				h[++idx] = selProperty;
				h[++idx] = '">';
				h[++idx] = '<i class="fa fa-edit"></i> ';
				h[++idx] = localise.set["c_edit"];
				h[++idx] = '</button>';
				h[++idx] = '</div>';		// End Col

				/*
			     * Add the text area to display the appearances
			     */
				h[++idx] = '<div class="col-sm-6">';
				h[++idx] = '<textarea class="labelProp has_tt" data-toggle="tooltip" readonly title="';
				h[++idx] = selLabel;
				h[++idx] = '" data-prop="';
				h[++idx] = selProperty;
				h[++idx] = '">';
				h[++idx] = question["appearance"];
				h[++idx] = '</textarea>';
				h[++idx] = '</div>';    // End Col

				h[++idx] ='</div>';		// End Row


			} else if(selProperty === "media" && question.type != "calculate") {
				h[++idx] = '<div class="row">';
				if(type === "question" && (question.inMeta || question.source != "user" )) {
					h[++idx] = '<div class="col-sm-4 col-sm-offset-4">';
					h[++idx] = naMedia;
					h[++idx] = '</div>';
				} else {
					h[++idx] = addMedia("Image", 
							question.labels[globals.gLanguage].image, 
							question.labels[globals.gLanguage].imageUrl, 
							question.labels[globals.gLanguage].imageThumb);
			        
					h[++idx] = addMedia("Video", 
							question.labels[globals.gLanguage].video, 
							question.labels[globals.gLanguage].videoUrl, 
							question.labels[globals.gLanguage].videoThumb);
					
					h[++idx] = addMedia("Audio", 
							question.labels[globals.gLanguage].audio, 
							question.labels[globals.gLanguage].audioUrl, 
							question.labels[globals.gLanguage].audioThumb);		
				}
				
			h[++idx] = '</div>';		// End of row

		} else {
			h[++idx] = '<textarea tabindex="0" class="labelProp has_tt';
			if((question.type === 'calculate' || question.type === 'server_calculate') && selProperty !== "appearance" && selProperty !== "parameters" && selProperty !== "display_name") {
				h[++idx] = ' calculate';
			}
			h[++idx] = '" data-toggle="tooltip" title="';
			if((question.type === 'calculate' || question.type === 'server_calculate') && selProperty !== "appearance" && selProperty !== "parameters" && selProperty !== "display_name") {
				h[++idx] = localise.set["ed_addcalc"];
			} else {
				h[++idx] = type === "option" ? "Choice Label" : selLabel;
			}
			h[++idx] = '" data-prop="';
			h[++idx] = selProperty;
			h[++idx] = '"';
			if(type === "option" && selProperty !== "label") {
				h[++idx] = ' readonly tabindex="-1"';
			}
			if(type === "question" && selProperty !== "appearance" && selProperty !== "parameters" &&
					((question.source != "user" && 
						question.type != "begin group" &&
						question.type != "begin repeat" &&
						question.type != "server_calculate" &&
						question.type != "pdf_field"
						))) {
				h[++idx] = ' readonly tabindex="-1">';
				h[++idx] = selLabel;
				h[++idx] = localise.set["ed_nreq"];
			} else {
				h[++idx] = '>';
				if(question.type === 'calculate' && selProperty !== "appearance" && selProperty !== "parameters" && selProperty !== "display_name") {
					h[++idx] = question.calculation;
				} else if(question.type === 'server_calculate' && selProperty !== "appearance" && selProperty !== "parameters" && selProperty !== "display_name") {
					if(question.server_calculation && question.server_calculation.expression) {
						h[++idx] = question.server_calculation.expression;
					}
				} else {
					if(selProperty === "label") { 
						h[++idx] = question.labels[globals.gLanguage].text;
					} else if(selProperty === "hint") { 
						h[++idx] = question.labels[globals.gLanguage].hint;
					} else if(selProperty === "constraint_msg") {
						h[++idx] = question.labels[globals.gLanguage].constraint_msg;
					} else if(selProperty === "guidance_hint") {
						h[++idx] = question.labels[globals.gLanguage].guidance_hint;
					} else {
						h[++idx] = question[selProperty];
					}
				}
			}
			h[++idx] = '</textarea>';
		}
		
		return h.join("");
	}
	
	/*
	 * Add subform
	 */
	function addSubForm(parentFormIndex, parentQuestionIndex) {
		
		var h = [],
			idx = -1,
			//formName,
			survey = globals.model.survey,
			forms = survey.forms,
			i,
			form;
		
		h[++idx] = '<ol class="list-unstyled">';
		
		//formName = question.name;
		
		for(i = 0; i < forms.length; i++) {
			form = forms[i];
			if(forms[i].parentFormIndex === parentFormIndex && forms[i].parentQuestionIndex === parentQuestionIndex) {			
				h[++idx] = addQuestions(forms[i], i);
				break;
			}
		}
		
		h[++idx] = '</ol>';
		
		return h.join("");
	}

	/*
	 * Add the questions for a form
	 */
	function addQuestions(form, formIndex) {
		var i,
			question,
			h = [],
			idx = -1,
			topLevelForm = false,
			lastRealQuestionId = -1,
			finalButtonName,
			groupButtonName,
			selProperty = globals.gSelProperty;
		
		// Set the group counter for this form
		gGroupStacks[formIndex] = {
				groupStack: [],
				lastGroup: undefined
		}
		
		if(form) {
			addQuestionSequence(form);		// Add an array holding the question sequence if it does not already exist
			form.maxQuestion = 1;			// Add a number to use for the default next question in a form
			
			for(i = 0; i < form.qSeq.length; i++) {
				globals.gHasItems = true;
				question = form.questions[form.qSeq[i]];
				
				// Ignore property type questions, questions that have been deleted and meta questions like end repeat
				if(!includeQuestion(question)) {
					continue;
				}
				
				if(question.type === "end group") {
					groupButtonName = question.name.substring(0, question.name.indexOf('_groupEnd'));

					// Remove the group from the group stack and set the "last group" value
					gGroupStacks[formIndex].groupStack.pop();
					gGroupStacks[formIndex].lastGroup = groupButtonName;
					
					h[++idx] = addNewQuestionButton(true, false, formIndex, groupButtonName, selProperty);
					
					// End the group
					h[++idx] = '</ul>';
					h[++idx] = '</div>';
					h[++idx] = '</li>';
					
					// Add a dummy dom entry for this end group
					h[++idx] = '<li style="display:none;" id="';
					h[++idx] = "question" + formIndex + "_" + form.qSeq[i];
					h[++idx] = '"';
					h[++idx] = ' data-fid="';
					h[++idx] = formIndex;
					h[++idx] = '" data-id="';
					h[++idx] = form.qSeq[i];
					h[++idx] = '"></li>';
					
				
					continue;
				}

				h[++idx] = addOneQuestion(form, question, formIndex, form.qSeq[i], true, selProperty, false);
			}
			if(form.parentFormIndex == -1) {
				topLevelForm = true;
				finalButtonName = undefined;
			} else {
				finalButtonName = form.name;
			}
			h[++idx] = addNewQuestionButton(true, topLevelForm, formIndex, finalButtonName, selProperty); 	// Adds a question at the end of the form
		}
		return h.join("");
	}

	/*
	 * Get the display sequence of the question by name
	 */
	function getIndexQuestionByName(name, form) {
		var i;
		
		for(i = 0; i < form.qSeq.length; i++) {
			if(form.questions[form.qSeq[i]].name === name) {
				return form.qSeq[i];
			}
		}
		alert("Could not find question with name: " + name);
		return 0;
	}
	
	/*
	 * Add the array containing the question sequence
	 * This will initially be the same as the order of questions but as new questions are added in the editor
	 *  then these new questions will be at the end of the question array
	 */
	function addQuestionSequence(form) {
		
		var i;
		
		if(!form.qSeq) {
			form.qSeq = [];
			for(i = 0; i < form.questions.length; i++) {
				form.qSeq[i] = i;		// Assume initial sequence corresponds to order of questions
			}
		}
	}
	

	
	/*
	 * Add a media type
	 */
	function addMedia(label, mediaIdent, url, thumbUrl) {
		var h = [],
			idx = -1,
			emptyMedia = '<div class="emptyMedia text-center">Empty</div>',
			lcLabel = label.toLowerCase();
		
		h[++idx] = '<div class="col-sm-3 ';
		h[++idx] = lcLabel;
		h[++idx] = 'Element">';
		if(mediaIdent) {
			h[++idx] = '<a target="_blank" href="';
			h[++idx] = url
			h[++idx] = '"';
		} else {
			h[++idx] = "<div";
		}
		h[++idx] = ' class="thumbnail preview">';

		if(mediaIdent) {
			if(thumbUrl || (lcLabel === "image" && url)) {
				h[++idx] = '<img height="100" width="100" src="';
				if(thumbUrl) {
					h[++idx] = thumbUrl + addCacheBuster(thumbUrl);;
				} else {
					h[++idx] = url + addCacheBuster(url) ;
				}
				h[++idx] = '">';
			} else {
				h[++idx] = addQType(lcLabel)
			}
		} else {
			h[++idx] = emptyMedia;
		}

		if(mediaIdent) {
			h[++idx] = '</a>';
		} else {
			h[++idx] = '</div>';
		}
	    h[++idx] = '<a type="button" class="btn btn-primary mediaProp form-control" data-element="';
	    h[++idx] = label.toLowerCase();
	    h[++idx] = '">';
	    h[++idx] = localise.set["ed_" + lcLabel];
	    h[++idx] = '</a>';
	 
	    h[++idx] = '</div>';
	    
	    return h.join("");
	}
	
	/*
	 * Refresh the content
	 */
	function refresh() {
		
		var content,
			i,
        	collapsedPanels = [];
		
		globals.gElementIndex = 0;
		globals.gHasItems = false;
		globals.gNewQuestionButtonIndex = 0;
		globals.gNewOptionButtonIndex = 0;
		
		if(globals.gIsQuestionView) {
			content = refreshQuestions();
		} else {
			content = refreshChoiceListView();
		}
		
		// Get the current list of collapsed panels
		$('.collapse.show', '#formList').each(function(){
			collapsedPanels.push($(this).closest('li').attr("id"));
		});
		
		// Update the content view
		$('#formList').html(content);
		
		// Restore collapsed panels
		for(i = 0; i < collapsedPanels.length; i++) {
			var $collapsedPanel = $('#' + collapsedPanels[i]);
			$collapsedPanel.find('.collapse').first().addClass("show");
			$collapsedPanel.find('.edit_icon.fa-chevron-down').first().removeClass('fa-chevron-down').addClass('fa-chevron-up');
			
		}
		
		if(!globals.gHasItems) {
			// If there were no items then set focus to the add new item button
			$('.add_final_button').focus();
		} 
		
		return $('#formList');		// Return the context of the updated HTML so that events can be applied
	}

	/*
	 * Refresh the featured properties
	 */
	function refreshQuestionFeaturedProperties() {

        var survey = globals.model.survey;

		$('li.card.question').each(function() {
			var $this = $(this);
			var fId = $this.data("fid");
			var id = $this.data("id");
			var question = survey.forms[fId].questions[id];

			$this.find('.card-body .row .question').html(getFeaturedMarkup(question, "question"));

		});

        return $('#formList');
	}

	/*
	 * Show the choice List view
	 */
	function refreshChoiceListView() {
		var h = [],
			survey = globals.model.survey,
			optionLists = survey.optionLists,
			idx = -1,
			name,
			nameArray = [],
			i;
		
		/*
		 * Process the choice lists in sequential order
		 */
		if(survey) {
			if(optionLists) {
				
				for (name in optionLists) {
					if (optionLists.hasOwnProperty(name)) {
					    nameArray.push(name);
					}
				}
				// Sort array of list names
				nameArray.sort();
				for(i = 0; i < nameArray.length; i++) {
					h[++idx] = addOneOptionList(nameArray[i], false, undefined);
				}
				h[++idx] = addNewOptionListButton(); 
			}
		}
		
		return h.join("");
	}
	
	/*
	 * Show the form on the screen
	 */
	function refreshQuestions() {
		
		var i,
			survey = globals.model.survey,
			h = [],
			idx = -1;
		
		/*
		 * Process the questions in the top level form (parent is 0) 
		 *   Questions that are "begin repeat" type will link to sub level forms which are then processed in turn
		 * 
		 */
		if(survey) {
			if(survey.forms && survey.forms.length > 0) {
				for(i = 0; i < survey.forms.length; i++) {
					if(survey.forms[i].parentFormIndex == -1) {
						h[++idx] = addQuestions(survey.forms[i], i);
						break;
					}
				}
			}
		}


		gGroupStacks = [];		// save some memory
		
		return h.join("");


	}
	
	/*
	 * Return true for those questions that are of interest to the editor
	 */
	function includeQuestion(question) {
		if(question.propertyType || question.soft_deleted || question.type === "end repeat") {	
			return false;
		} else {
			return true;
		}
	}
	
	/*
	 * Get the questions for a linked survey
	 */
	function getLinkedQuestions(questionId, surveyId, qId) {
		if(linkedQuestions[surveyId]) {
			setTimeout(function() {
				showLinkedQuestions(questionId, linkedQuestions[surveyId], qId);
			}, 0);
		} else {
			addHourglass();
		 	$.ajax({
				url: "/surveyKPI/questionList/" + surveyId + "/none",	// Will use the default language
				cache: false,
				dataType: 'json',
				success: function(data) {
					linkedQuestions[surveyId] = data;
					showLinkedQuestions(questionId, data, qId);
					removeHourglass();
				},
				error: function(xhr, textStatus, err) {
					removeHourglass();
	  				if(xhr.readyState == 0 || xhr.status == 0) {
			              return;  // Not an error
					} else {
						bootbox.alert("Error failed to get questions for survey:" + surveyId);
						$("#" + questionId).find(".linkedQuestion").html("");
					}
				}
			});
		}
		
	}
	
	/*
	 * Show the linked questions in the drop down
	 */
	function showLinkedQuestions(questionId, data, qId) {
		
		var idx = -1,
			h = [],
			i;
		
		qId = +qId;
		h[++idx] = '<option value="0"';
		if(!qId) {
			h[++idx] = ' selected';
		}
		h[++idx] = '>Key</option>';
		if(data) {
			for(i = 0; i < data.length; i++) {
				if(data[i].q) {
					h[++idx] = '<option value="';
					h[++idx] = data[i].id;
					h[++idx] = '" ';
					if(data[i].id == qId) {
						h[++idx] = ' selected';
					} 
					h[++idx] = '>';
					h[++idx] = data[i].q;
					h[++idx] = '</option>';
				}
			}
		}
		
		$("#" + questionId).find(".linkedQuestion").html(h.join(''));
	}

});
/*
This file is part of SMAP.

SMAP is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

SMAP is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with SMAP.  If not, see <http://www.gnu.org/licenses/>.

*/

/*
 * Functions for manipulating a choice in the editor
 * 
 *  Filter Model
 *    globals.gFilterArray:				Filters in display order
 *    survey.filters	Filters with state of shown or not shown
 */



define('app/option',[
         'jquery',
         'modernizr',
         'app/localise',
         'app/globals',
         'app/editorMarkup',
         'app/changeset'], 
		function($, modernizr, lang, globals, markup, changeset) {

	return {	
		refreshOptionListControls: refreshOptionListControls,
		createChoiceView: createChoiceView,
		setupChoiceView: setupChoiceView,						// On create of option view
		addOneOption: addOneOption,
		resetFilterColumns: resetFilterColumns,
		setPreviousChoices: setPreviousChoices,
		addOptionTable: addOptionTable,
		addFilter: addFilter,
		addFilterSelectList: addFilterSelectList,
		addOptionSequence: addOptionSequence
	};
	
	globals.gFilterArray = [];			// Filters in repeatable order
	
	/*
	 * Refresh the select controls that show the available option lists
	 */
	function refreshOptionListControls() {
		var $selector = $(".option-lists");
		$selector.html(getOptionLists());
	}
	
	/*
	 * Create a choice view
	 */
	function createChoiceView() {
		
		var $cv = $('#choiceView').find('.choice-content'),
			survey = globals.model.survey,
			question,
			filter,
			filterType = "none",
			i;
		
		if(globals.gListName) {
			$cv.empty().append(addOptionContainer(undefined, undefined, undefined, globals.gListName));
		} else {
			// Choice list opened from a question
			question = survey.forms[globals.gFormIndex].questions[globals.gItemIndex];
			$cv.empty().append(addOptionContainer(question, globals.gFormIndex, globals.gItemIndex, undefined));
		}
		addFilterSelectList(survey.filters);
		
		/*
		 * Set the filter type to custom if there are filters with a name other than _smap_cascade
		 */
		if(question) {
			$('.filter_only').show();

			if(question.choice_filter && question.choice_filter.trim().length > 0) {
				if(globals.gFilterArray.length > 0) {
					filterType = "custom";
				}
				if(globals.gFilterArray.length === 1) {
					if (globals.gFilterArray[0] === "_smap_cascade") {
						filterType = "cascade";
					}
				}
			}
	
			// Set the custom filter view by default
			$('#filterType').val(filterType);
		}
		
		return $cv;
	}
	
	function setupChoiceView(filterType) {
		var survey = globals.model.survey,
			i;
		/*
		 * show custom filter columns that should be visible
		 */
		for (i = 0; i < globals.gFilterArray.length; i++) {
			if(survey.filters[globals.gFilterArray[i]] === true) {
				$('table', '#choiceView').removeClass("hide" + i);
			}
		}
		
		if(filterType === "custom") {
			$('.custom_filter_only').show();
			$('.cascade_filter_only').hide();
			$('#choiceView table').addClass("notcascade").removeClass("notcustom");	
		} else if(filterType === "cascade") {
			$('.custom_filter_only').hide();
			$('.cascade_filter_only').show();
			$('#choiceView table').removeClass("notcascade").addClass("notcustom");
		} else {
			$('.custom_filter_only').hide();
			$('.cascade_filter_only').hide();
			$('#choiceView table').addClass("notcascade").addClass("notcustom");	
		}

	}

	/*
	 * Add a single option
	 */
	function addOneOption(optionList, option, formIndex, index, list_name, qname) {
		
		var h = [],
			idx = -1,
			prevChoice = $("#previousSelectChoice").val(),
			filterType = $('#filterType').val();
		
		optionList.maxOption++;
		
		if(filterType !== "cascade" || showCascadeOption(option.cascade_filters, prevChoice,  option.value) ) {
			h[++idx] = '<tr class="editor_element option draggable dropon ';
			if(option.error) {
				h[++idx] = ' error';
			}
			
			// Add the option index 
			h[++idx] = '" data-id="';
			h[++idx] = index;
			h[++idx] = '" data-fid="';					
			h[++idx] = formIndex;
			h[++idx] = '" data-qname="';
			h[++idx] = qname;
			h[++idx] = '" data-list_name="';
			h[++idx] = list_name;
			h[++idx] = '" data-filters=';
			h[++idx] = JSON.stringify(option.cascade_filters);
			h[++idx] = '>';
			
				// Add select for cascade selects
				h[++idx] = '<td class="cascade">';
					h[++idx] = '<input type="checkbox" class="cascadeFilter" name="isSelected" value="';
					h[++idx] = "";
					h[++idx] = '" ';
					if(idx !== false) {
						h[++idx] = addCascadeChecked(option.cascade_filters, prevChoice);
					} 
					h[++idx] = '>';
				h[++idx] = '</td>';
				
				// Add option name and value cell
				h[++idx] = '<td>';
						h[++idx] = '<div class="dropdown">';
						h[++idx] = '<button type="button" class="btn btn-secondary btn-lg dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false"><i class="fa fa-bars" aria-hidden="true"></i> ';
						h[++idx] = '</button>';
							h[++idx] = '<div class="dropdown-menu">';
								h[++idx] = '<a href="javascript:void(0)" class="dropdown-item delete_option">';
								h[++idx] = localise.set["c_del"];
								h[++idx] = '</a>';
								h[++idx] = '<a href="javascript:void(0)" class="dropdown-item add_option_before">';
								h[++idx] = localise.set["ed_ab"];
								h[++idx] = '</a>';
								h[++idx] = '<a href="javascript:void(0)" class="dropdown-item add_option_after">';
								h[++idx] = localise.set["ed_aa"];
								h[++idx] = '</a>';
							h[++idx] = '</div>';
						h[++idx] = '</div>';
				h[++idx] = '</td>';	// End of menu button
				
				// Add option name cell
				h[++idx] = '<td>';
					
					h[++idx] = '<input class="oname form-control has_tt" value="';
					h[++idx] = option.value;
					h[++idx] = '" ';

					h[++idx] = ' type="text" title="';
					h[++idx] = localise.set["ed_cval"];
					h[++idx] = '">';
							
				h[++idx] = '</td>';	// End of option name and label cell

				h[++idx] = addFilterColumnBody(option.cascade_filters);
				h[++idx] = addOptionLabel(option);
				
			
			h[++idx] = '</tr>';
		}

	
		return h.join("");
	}
	
	function resetFilterColumns() {
		var i,
			survey = globals.model.survey;
		
		for (i = 0; i < globals.gFilterArray.length; i++) {
			if(survey.filters[globals.gFilterArray[i]] === true) {
				$('table', '#choiceView').removeClass("hide" + i);
			} else {
				$('table', '#choiceView').addClass("hide" + i);
			}
			
		}
	}
	
	/*
	 * Add an option container
	 */
	function addOptionContainer(question, formIndex, qIndex, listName) {
		var h = [],
			idx = -1,
			filter;
		
		h[++idx] = '<div class="question_head" data-fId="';
		h[++idx] = formIndex;
		h[++idx] = '" data-id="';
		h[++idx] = qIndex;
		h[++idx] = '">';
		h[++idx] = '<div class="question-controls">';
			h[++idx] = '<div class="row">';
				h[++idx] = '<div class="col-md-6">';
					h[++idx] = '<form role="form">';
						
						// A control to set option list name
						if(!listName) {
							h[++idx] = '<div class="form-group row">';
								h[++idx] = '<label class="col-sm-2">';
									h[++idx] = localise.set["ed_cl"];
								h[++idx] = '</label>';
								h[++idx] = '<div class="col-sm-10">';
									h[++idx] = '<select class="form-control option-lists">';
									h[++idx] = getOptionLists();
									h[++idx] = '</select>';
								h[++idx] = '</div>';
							h[++idx] = '</div>';
						}

						// A control to select the filter type
						h[++idx] = '<div class="form-group row filter_only" style="display:none;">';
							h[++idx] = '<label class="col-sm-2">';
								h[++idx] = localise.set["c_filter"];
							h[++idx] = '</label>';
							h[++idx] = '<div class="col-sm-10">';
								h[++idx] = '<select class="form-control" id="filterType">';
								
								h[++idx] = '<option value="none">';
								h[++idx] = localise.set["c_none"];
								h[++idx] = '</option>';
								
								h[++idx] = '<option value="cascade">';
								h[++idx] = localise.set["ed_cs"];
								h[++idx] = '</option>';
								
								h[++idx] = '<option value="custom">';
								h[++idx] = localise.set["ed_cf"];
								h[++idx] = '</option>';
								
								h[++idx] = '</select>';
							h[++idx] = '</div>';
						h[++idx] = '</div>';
						
					h[++idx] = '</form>';
				h[++idx] = '</div>';
				h[++idx] = '<div class="col-md-6">';
					h[++idx] = '<div class="text-right">';
				    h[++idx] = '<button class="btn btn-primary exitOptions mb-2">';
						h[++idx] = localise.set["c_back"];
					h[++idx] = '</button>';
					h[++idx] = '</div>';
					h[++idx] = '<div class="custom_filter_only" style="display:none;">';
						h[++idx] = '<form role="form">';
						
							if(!listName) {
								h[++idx] = '<div class="form-group row">';
									h[++idx] = '<label class="col-sm-2">';
									h[++idx] = localise.set["ed_choice_f"];
									h[++idx] = '</label>';
									h[++idx] = '<div class="col-sm-8">';
										h[++idx] = '<textarea id="choiceFilter" class="input-block" form-control has_tt" type="text">';
										h[++idx] = question.choice_filter;
										h[++idx] = '</textarea>';
										h[++idx] = addChoiceFilterError(question.errorMsg);
									h[++idx] = '</div>';
								h[++idx] = '</div>';
							}
							
							h[++idx] = '<div id="custom_filters">';
							h[++idx] = '</div>';
						h[++idx] = '</form>';
						h[++idx] = '<button id="addFilter" class="btn btn-light mt-2">';
						h[++idx] = localise.set["ed_afc"];
						h[++idx] = '</button>';
					h[++idx] = '</div>';  // Custom filter only
					
					h[++idx] = '<div class="cascade_filter_only" style="display:none;">';
						h[++idx] = '<form role="form">';

							h[++idx] = '<div class="form-group row">';
								h[++idx] = '<label class="col-sm-2">';
									h[++idx] = localise.set["ed_cspq"];
								h[++idx] = '</label>';
								h[++idx] = '<div class="col-sm-10">';
									h[++idx] = '<select class="form-control" id="previousSelect">';
										h[++idx] = addSelectQuestions(question);
									h[++idx] = '</select>';
								h[++idx] = '</div>';
							h[++idx] = '</div>';
			
							h[++idx] = '<div class="form-group row">';
								h[++idx] = '<label class="col-sm-2">';
									h[++idx] = localise.set["ed_csp"];
								h[++idx] = '</label>';
								h[++idx] = '<div class="col-sm-10">';
									h[++idx] = '<select class="form-control" id="previousSelectChoice">';	
									h[++idx] = '</select>';
								h[++idx] = '</div>';
							h[++idx] = '</div>';
						h[++idx] = '</form>';
					h[++idx] = '</div>';	// Cascade filter only
				h[++idx] = '</div>';
			h[++idx] = '</div>';
		h[++idx] = '</div>';
	
		h[++idx] = '<div id="optionTable">';	
		h[++idx] = '</div>';	// Option Table
		h[++idx] = '</div>';		// Question Head
		
		return h.join("");
	}

	/*
	 * Refresh the table of options
	 */
	function addOptionTable(question, formIndex, listName) {
		var h = [],
			idx = -1,
			$element = $('#optionTable');
		
		if(listName) {
			h[++idx] = addOptions(undefined, undefined, listName);
			$('#choiceViewQuestion').html(localise.set["ed_cl"] + ": " + listName);
		} else {
			// Opened from a specific question
			h[++idx] = addOptions(question, formIndex, undefined);
			$('#choiceViewQuestion').html(localise.set["c_question"] + ": " + question.name);
		}
		
		$element.html(h.join());
		
		// Add styling to cascading select checkboxes
		$('[type="checkbox"]', '#optionTable').iCheck({
		    checkboxClass: 'icheckbox_square-green',
		    radioClass: 'iradio_square-green'
		});
	}
	
	/*
	 * Add filter to filters if not already there
	 */
	function addFilter(filter) {
		var i,
			hasFilter = false,
			survey = globals.model.survey;
		
		for(i = 0; i < globals.gFilterArray.length; i++) {
			if(globals.gFilterArray[i] === filter) {
				hasFilter = true;
			}
		}
		if(!hasFilter) {
			globals.gFilterArray.push(filter);
		}
		
		if(!survey.filters) {
			survey.filters = {};
		}
		survey.filters[filter] = true;
	}
	
	/*
	 * Add the filters as  a select list
	 */
	function addFilterSelectList(filters) {
		var h = [],
			idx = -1,
			survey = globals.model.survey,
			filter;
		
		globals.gFilterArray = [];
		for(filter in filters) {
			if (filters.hasOwnProperty(filter)) {
				globals.gFilterArray.push(filter);	// Save filters as ordered array
				h[++idx] = '<div class="checkbox">';
			      h[++idx] = '<input type="checkbox" ';
			      if(survey.filters[filter]) {
			    	  h[++idx] = 'checked=true ';
			      }
			      h[++idx] = 'value="';
			      h[++idx] = filter;
			      h[++idx] = '"> ';
				  h[++idx] = '<label>';
			      h[++idx] = filter
			    h[++idx] = '</label>';
			    h[++idx] = '</div>';
			}
		}
		
		$('#custom_filters').html(h.join(""));
		
		// Style checkboxes
		$('[type="checkbox"]', '#custom_filters').iCheck({
		    checkboxClass: 'icheckbox_square-green',
		    radioClass: 'iradio_square-green'
		});
	}
	
	/*
	 * Show the options
	 */
	function addOptions(question, formIndex, list_name) {
		var survey = globals.model.survey,
			optionList,
			questionName,
			oSeq,
			maxIndex,
			h = [],
			idx = -1,
			i,
			initialiseFilters = false;
		
		if(!list_name) {		// Options attached to a question
			list_name = question.list_name;
			questionName = question.name;
		} 
		optionList = survey.optionLists[list_name];
		
		if(typeof optionList !== "undefined") {
			optionList.maxOption = 0;
			
			addOptionSequence(optionList);		// Add an array holding the option sequence if it does not already exist
			oSeq = optionList.oSeq;
			
			h[++idx] = '<div class="table-responsive">';
			h[++idx] = '<table class="table notcustom hide0 hide1 hide2 hide3 hide4">';
			
			h[++idx] = '<thead class="thead-default"><tr>';
			
				h[++idx] = '<th class="cascade">';
				h[++idx] = '</th>';
			
				h[++idx] = '<th>';
				h[++idx] = '</th>';
				
				h[++idx] = '<th>';
				h[++idx] = localise.set["ed_cval"];
				h[++idx] = '</th>';

				//h[++idx] = '<th>';
				//h[++idx] = localise.set["ed_dn"];
				//h[++idx] = '</th>';

				h[++idx] = addFilterColumnHeadings();
				
				h[++idx] = '<th>';
				if(globals.gSelProperty === 'media') {
					h[++idx] = localise.set["c_media"];
				} else if(globals.gSelProperty === 'display_name') {
					h[++idx] = localise.set["ed_dn"];
				} else {
					h[++idx] = localise.set["ed_clab"];
				}
				h[++idx] = '</th>';
				
			h[++idx] = '</tr></thead>';
			h[++idx] = '<tbody>';
			if(oSeq) {
				maxIndex = 0;
				for(i = 0; i < oSeq.length; i++) {
					h[++idx] = addOneOption(optionList,
							optionList.options[oSeq[i]], 
							formIndex, 
							oSeq[i], 
							list_name, 
							questionName);
					if(oSeq[i] >= maxIndex) {
						maxIndex = oSeq[i] + 1;
					}
				}
			}
			h[++idx] = '</tbody>';
			h[++idx] = '</table>';
			h[++idx]= '</div>';
			h[++idx] = addNewOptionButton(true, list_name, formIndex, questionName, -1); 
		}
		return h.join("");
	}
	
	/*
	 * Add the array containing the option sequence
	 */
	function addOptionSequence(optionList) {
		var i;
		
		if(!optionList.oSeq) {
			optionList.oSeq = [];
			for(i = 0; i < optionList.options.length; i++) {
				optionList.oSeq[i] = i;
			}
		}
	}
	
	
	/*
	 * One of the questions properties will be featured so that it can be edited in the header without expanding the question
	 */
	function addOptionLabel(option) {
		
		var h = [],
			idx = -1,
			selProperty = globals.gSelProperty;
		
		h[++idx] = '<td class="option">';
			if(selProperty === "media") {
				h[++idx] = '<div class="row">';
				h[++idx] = markup.addMedia("Image", 
						option.labels[globals.gLanguage].image, 
						option.labels[globals.gLanguage].imageUrl, 
						option.labels[globals.gLanguage].imageThumb);
		        
				h[++idx] = markup.addMedia("Video", 
						option.labels[globals.gLanguage].video, 
						option.labels[globals.gLanguage].videoUrl, 
						option.labels[globals.gLanguage].videoThumb);
				
				h[++idx] = markup.addMedia("Audio", 
						option.labels[globals.gLanguage].audio, 
						option.labels[globals.gLanguage].audioUrl, 
						option.labels[globals.gLanguage].audioThumb);
				h[++idx] = '</div>';

			} else if(selProperty === "display_name") {
				h[++idx] = '<textarea class="odisplayname has_tt" title="';
				h[++idx] = localise.set["ed_dn"];
				h[++idx] = '">';
				h[++idx] = option.display_name;
				h[++idx] = '</textarea>';
				h[++idx] = addOptionErrorMsg(option.errorMsg);

			} else {
				h[++idx] = '<textarea data-toggle="tooltip" class="labelProp has_tt" title="';
				h[++idx] = localise.set["ed_clab"];
				h[++idx] = '">';
				h[++idx] = option.labels[globals.gLanguage].text;
				h[++idx] = '</textarea>';
				h[++idx] = addOptionErrorMsg(option.errorMsg);
			}
		
		h[++idx] = '</td>';
		return h.join("");
	}
	
	function addNewOptionButton(after, list_name, formIndex, qname, index) {
		var h = [],
			idx = -1,
			addButtonClass,
			locn;
		
		addButtonClass = after ? 'add_after_button add_button' : 'add_before_button add_button';
		locn = after ? 'after' : 'before';
		
		h[++idx] = '<button type="button" class="add_option btn btn-primary dropon option editor_element add_after_button"';
		if(typeof index !== "undefined") {
			h[++idx] = ' data-id="';
			h[++idx] = index;
			h[++idx] = '"';
		}
		if(typeof list_name !== "undefined") {
			h[++idx] = ' data-list_name="';
			h[++idx] = list_name;
			h[++idx] = '"';
		}
		if(typeof qname !== "undefined") {
			h[++idx] = ' data-qname="';
			h[++idx] = qname;
			h[++idx] = '"';
		}
		if(typeof formIndex !== "undefined") {
			h[++idx] = ' data-fid="';
			h[++idx] = formIndex;
			h[++idx] = '"';
		}
		h[++idx] = '>';
		h[++idx] = localise.set["ed_anc"];
		h[++idx] = ' </button>';

		
		return h.join('');
	}
	
	/*
	 * Get an array of option list names sorted alphabetically
	 */
	function getOptionLists() {
		
		var lists = globals.model.survey.optionLists,
			name,
			nameArray = [],
			h = [],
			idx = -1,
			i;

		// get the names into an array so they can be sorted
		for (name in lists) {
			if (lists.hasOwnProperty(name)) {
			    nameArray.push(name);
			}
		}
		// Sort array of list names
		nameArray.sort();
		
		// Create html
		for(i = 0; i < nameArray.length; i++) {
			h[++idx] = '<option value ="';
			h[++idx] = nameArray[i];
			h[++idx] = '">';
			h[++idx] = nameArray[i];
			h[++idx] = '</option>';
		}
		return h.join("");
	}
	
	/*
	 * Add the table headings for filter columns
	 */
	function addFilterColumnHeadings() {
		
		var i,
			h = [],
			idx = -1;
		
		for(i = 0; i < globals.gFilterArray.length; i++) {
			h[++idx] = '<th class="f';		// Class to hide / show filter
			h[++idx] = i;
			h[++idx] = '">';
			h[++idx] = globals.gFilterArray[i];
			h[++idx] = '</th>';
		}
		
		return h.join("");
	}
	
	/*
	 * Add the table body for filter columns
	 */
	function addFilterColumnBody(filters) {
		
		var i,
			h = [],
			idx = -1;
		
		for(i = 0; i < globals.gFilterArray.length; i++) {
			h[++idx] = '<td class="f';		// Class to hide / show filter
			h[++idx] = i;
			h[++idx] = '" data-f_name="';
			h[++idx] = globals.gFilterArray[i];
			h[++idx] = '">';
			h[++idx] = '<input class="filter form-control data-toggle="tooltip" has_tt" value="';
			h[++idx] = filters ? filters[globals.gFilterArray[i]] : "";
			h[++idx] = '" ';	
			h[++idx] = ' type="text" title="';
			h[++idx] = localise.set["ed_fv"];
			h[++idx] = '">';
			
			h[++idx] = '</td>';
			
		}
		
		return h.join("");
	}
	
	/*
	 * Add a media type
	 */
	function addOptionMedia(label, mediaIdent, url, thumbUrl) {
		var h = [],
			idx = -1,
			emptyMedia = '<div class="emptyMedia text-center">Empty</div>',
			lcLabel = label.toLowerCase();
		
		h[++idx] = '<div class="col-sm-3 ';
		h[++idx] = lcLabel;
		h[++idx] = 'Element">';
		if(mediaIdent) {
			h[++idx] = '<a target="_blank" href="';
			h[++idx] = url
			h[++idx] = '"';
		} else {
			h[++idx] = "<div";
		}
		h[++idx] = ' class="thumbnail preview">';

		if(mediaIdent) {
			if(thumbUrl || (lcLabel === "image" && url)) {
				h[++idx] = '<img height="100" width="100" src="';
				if(thumbUrl) {
					h[++idx] = thumbUrl;
				} else {
					h[++idx] = url;
				}
				h[++idx] = '">';
			} else {
				h[++idx] = addQType(lcLabel)
			}
		} else {
			h[++idx] = emptyMedia;
		}

		if(mediaIdent) {
			h[++idx] = '</a>';
		} else {
			h[++idx] = '</div>';
		}
	    h[++idx] = '<a type="button" class="btn btn-default mediaProp form-control" data-element="';
	    h[++idx] = label.toLowerCase();
	    h[++idx] = '">';
	    h[++idx] = lcLabel;
	    h[++idx] = '</a>';
	 
	    h[++idx] = '</div>';
	    
	    return h.join("");
	}
	
	function addOptionErrorMsg(msg) {
		var h = [],
			idx = -1;
		
		h[++idx] = '<div class="error-msg pull-right">';
		h[++idx] = msg;
		h[++idx] = '</div>';
		return h.join("");
	}
	
	function addChoiceFilterError(msg) {
		var h = [],
			idx = -1;
		
		h[++idx] = '<div class="error-msg choice_filter_error pull-right">';
		h[++idx] = msg;
		h[++idx] = '</div>';
		return h.join("");
	}
	
	/*
	 * Add a list of select questions that can be selected as the preceeding question for a cascade select
	 */
	function addSelectQuestions(choiceQuestion) {
		var survey = globals.model.survey,
			h = [],
			idx = -1,
			i, j,
			form,
			question,
			first = true;
			
		for(i = 0; i < survey.forms.length; i++) {
			form = survey.forms[i];
			markup.addQuestionSequence(form);		// Add the sequence for the questions if it has not already been added
			for(j = 0; j < form.qSeq.length; j++) {
				question = form.questions[form.qSeq[j]];
				if(!question.soft_deleted && question.type === "select1") {
					if(choiceQuestion && choiceQuestion.name === question.name) {
						continue;		// Skip question being edited
					}
					h[++idx] = '<option ';
					if(first) {
						h[++idx] = 'selected ';
					}
					h[++idx] = 'value="';
					h[++idx] = question.list_name;
					h[++idx] = '">';
					h[++idx] = question.name;
					h[++idx] = '</option>';
				}
			}
		}
	
		return h.join("");
	}
	
	/*
	 * Set the choices that can be selected as the preceeding choice for a cascade select
	 */
	function setPreviousChoices(listname) {
		
		var survey = globals.model.survey,
			h = [],
			idx = -1,
			i,
			option,
			optionList = survey.optionLists[listname],
			oSeq;
			
		addOptionSequence(optionList);		// Add an array holding the option sequence if it does not already exist
		oSeq = optionList.oSeq;
		
		for(i = 0; i < oSeq.length; i++) {

			option = optionList.options[oSeq[i]];
				
			h[++idx] = '<option ';
				if(i == 0) {
					h[++idx] = 'selected ';
				}
				h[++idx] = 'value="';
				h[++idx] = option.value;
				h[++idx] = '">';
				h[++idx] = option.value;
			h[++idx] = '</option>';
		}
		$("#previousSelectChoice").html(h.join(""));
	}
	
	/*
	 * Set the checkbox checked if the cascade filter matches the previously selected choice
	 */
	function addCascadeChecked (filters, prevChoice) {
		
		var match = false;
		
		if(filters) {
			if(filters["_smap_cascade"] && filters["_smap_cascade"] == prevChoice) {
				match = true;
			}
		}
		
		if(match) {
			return 'checked="checked"';
		} else {
			return '';
		}
	}
	
	/*
	 * As this is the cascade view options should only be shown if their filter value is empty or 
	 *  matches the prevChoice
	 */
	function showCascadeOption (filters, prevChoice, optionValue) {
		
		var show = false,
			i,
			inOptionList = false,
			listname = $('#previousSelect').val(),
			survey = globals.model.survey,
			optionList = survey.optionLists[listname],
			oSeq,
			option;
		
		if(listname) {
			addOptionSequence(optionList);		// Add an array holding the option sequence if it does not already exist
			oSeq = optionList.oSeq
			for(i = 0; i < oSeq.length; i++) {
				option = optionList.options[oSeq[i]];
				if(filters["_smap_cascade"] == option.value) {
					inOptionList = true;
					break;
				}
			}
						
			if(!filters["_smap_cascade"] || filters["_smap_cascade"] == prevChoice || !inOptionList) {
				show = true;
			}
		} else {
			show = true;		// No previous list set
		}
		
		return show;
	}
	
});

/*
 This file is part of SMAP.

 SMAP is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 SMAP is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with SMAP.  If not, see <http://www.gnu.org/licenses/>.

 */

/*
 * Module to manage the changeset of changes for editor
 */



define('app/changeset',[
        'jquery',
        'modernizr',
        'app/localise',
        'app/globals',
        'app/editorMarkup',
        'app/option'],
    function($, modernizr, lang, globals, markup, option) {

        var modelGeneratedChanges = [];

        return {
            add: add,
            undo: undo,
            save: save,
            setHasChanges: setHasChanges,
            addValidationError: addValidationError,
            removeValidationError: removeValidationError,
            validateItem: validateItem,
            validateName: validateName,
            updateModelWithErrorStatus: updateModelWithErrorStatus,
            validateAll: validateAll,
            numberIssues: numberIssues,
            addUpdateMessage: addUpdateMessage,
            updateViewControls: updateViewControls
        };

		/*
		 * Add a new change item
		 */
        function add(change) {
            var refresh,
                $context,
                container,
                itemIndex,
                itemType;

            // Apply to model
            modelGeneratedChanges = [];
            refresh = updateModel(change);

			/*
			 * Validate updated model
			 */
            if(change.changeType === "property" || change.changeType === "label") {

                if (change.property.type === "option") {
                    container = change.property.optionList;
                    itemType = "option";
                } else {
                    container = change.property.formIndex;
                    itemType = "question";
                }
                itemIndex = change.property.itemIndex;

            } else if(change.changeType === "option") {

                container = change.option.optionList;
                itemIndex = change.option.itemIndex;
                itemType = "option";

            } else if(change.changeType === "question") {

                container = change.question.formIndex;
                itemIndex = change.question.itemIndex;
                itemType = "question";

            }
            if(typeof itemIndex !== "undefined") {		// Note some changes do not have an itemIndex and don't need to be validated

                validateItem(container, itemIndex, itemType, true);
            }

			/*
			 * Apply any HTML changes either directly to the changed element, or by refreshing the whole form using
			 * the updated model
			 */
            if(change.action === "delete" && typeof change.question !== "undefined") {
                $context = undefined;
                change.question.$deletedElement.prev().remove();		// remove the add new button
                change.question.$deletedElement.remove();
            } else {
                if(refresh) {
                    if(change.question || change.changeType === "optionlist" || (change.property && change.property.type === "question")) {
                        $context = markup.refresh();
	                    validateAll();
                    } else {
                        $context = option.createChoiceView();
                        var survey = globals.model.survey;
                        var question = survey.forms[globals.gFormIndex].questions[globals.gItemIndex];
                        option.addOptionTable(question, globals.gFormIndex, globals.gListName);
                        option.setupChoiceView($('#filterType').val());
                    }
                } else {
                    $context = updateHtmlElement(change);
                }
            }

            // Add to changeset array ready for writing to the database
            addToChangesetArray(change);

            // Add any additional changes generated in the model
            for(var i = 0; i < modelGeneratedChanges.length; i++) {
                addToChangesetArray(modelGeneratedChanges[i]);
            }

            return $context;

        }

        function undo() {
        }

        // Save the survey
        function save(callback) {

            var url="/surveyKPI/surveys/save/" + globals.gCurrentSurvey,
                changesString,
                changesInSeq = resequenceChangeSet(globals.changes),
                i, j,
                item;

            // newVal and oldVal cannot be objects so if this is a property update
            // for a server calculation then they need to be stringified
            if(changesInSeq) {
                for(i = 0; i < changesInSeq.length; i++) {
                    if(changesInSeq[i].changeType === "property") {
                        for(j = 0; j < changesInSeq[i].items.length; j++) {
                            item = changesInSeq[i].items[j];
                            if(item.property && item.property.prop == 'server_calculation') {
                                if(item.property.oldVal && item.property.oldVal.expression) {
                                    item.property.oldVal = JSON.stringify(item.property.oldVal);
                                }
                                if(item.property.newVal  && item.property.newVal.expression) {
                                    item.property.newVal = JSON.stringify(item.property.newVal);
                                }
                            }
                        }
                    }
                }
            }
            changesString = JSON.stringify(changesInSeq);

            addHourglass();
            $.ajax({
                url: url,
                type: 'PUT',
                dataType: 'json',
                cache: false,
                data: { changes: changesString },
                success: function(data) {
                    var responseFn = callback,
                        h = [],
                        idx = -1,
                        i;

                    removeHourglass();
                    globals.gSaveInProgress = false;
                    if(handleLogout(data)) {

                        setHasChanges(0);

                        if (typeof responseFn === "function") {
                            responseFn();
                        }

                        // Report success and failure
                        globals.model.lastChanges = data.changeSet;
                        $('#successLabel .counter').html(data.success);
                        $('#failedLabel .counter').html(data.failed);

                        if (data.success > 0) {
                            h[++idx] = '<div class="alert alert-success" role="alert">';
                            h[++idx] = '<p>';
                            h[++idx] = data.success;
                            h[++idx] = localise.set["msg_upd"];
                            h[++idx] = '</p>'
                            h[++idx] = '<ol>';
                            for (i = 0; i < data.changeSet.length; i++) {
                                h[++idx] = addUpdateMessage(data.changeSet[i], false);
                            }
                            h[++idx] = '</ol>';
                            h[++idx] = '</div>';
                        }
                        if (data.failed > 0) {
                            h[++idx] = '<div class="alert alert-danger" role="alert">';
                            h[++idx] = data.failed;
                            h[++idx] = " changes failed";
                            h[++idx] = '<ol>';
                            for (i = 0; i < data.changeSet.length; i++) {
                                h[++idx] = addUpdateMessage(data.changeSet[i], true);
                            }
                            h[++idx] = '</ol>';
                            h[++idx] = '</div>';
                        }

                        bootbox.alert(h.join(""));
                    }

                },
                error: function(xhr, textStatus, err) {
                    removeHourglass();
                    if(handleLogout(xhr.responseText)) {
                        globals.gSaveInProgress = false;

                        if (typeof responseFn === "function") {
                            responseFn();
                        }

                        if (xhr.readyState == 0 || xhr.status == 0) {
                            // Not an error
                        } else {
                            bootbox.alert(localise.set["msg_err_save"] + ' ' + err);
                        }
                    }
                }
            });

        }

        /*
         * Resequence a change array to allign with the sequences in the model
         */
        function resequenceChangeSet(changes) {

            var fo = [],
                i,
                newSeq = [];

            for(i = 0; i < globals.model.survey.forms.length; i++) {
                fo.push({});
            }

            // Set the sequence nuber as per the model
            for(i = 0; i < changes.length; i++) {
                if(globals.changes[i].changeType === "question") {
                    var question = globals.changes[i].items[0].question;
                    if(question && globals.changes[i].items[0].action === "add") {
                        var form = globals.model.survey.forms[question.formIndex];
                        for (var j = 0; j < form.qSeq.length; j++) {
                            if(form.qSeq[j] == question.itemIndex) {
                                question.seq = j;
                                fo[question.formIndex]["q"+j] = globals.changes[i];
                                break;
                            }
                        }
                    }
                }
            }

            // Put the question changes in the order of their sequence numbers
            for(i = 0; i < globals.model.survey.forms.length; i++) {
                var form = globals.model.survey.forms[i];
                for (var j = 0; j < form.qSeq.length; j++) {
                    if(fo[i]["q"+j]) {
                        newSeq.push(fo[i]["q"+j]);
                    }
                }
            }

            // Add any other changes
            for(i = 0; i < changes.length; i++) {
                if (globals.changes[i].changeType !== "question"
                    || typeof globals.changes[i].items[0].question === "undefined"
                    || globals.changes[i].items[0].action !== "add"
                ) {
                    newSeq.push(changes[i]);
                }
            }

            return newSeq;

        }

		/*
		 * Add a single change item to the array of changes (changeset)
		 */
        function addToChangesetArray(change) {

            var type = change.property ? change.property.type : undefined;

            var ci = {
                    changeType: change.changeType,
                    type: type,
                    action: change.action,
                    items: []
                },
                survey = globals.model.survey,
                item,
                item_orig,
                forms_orig,
                optionListOrig,
                form,
                applyChange = true,
                changes = globals.changes;

            // Delete any items containing jquery elements
            if(change.question) {
                delete change.question.$relatedElement;
                delete change.question.$deletedElement;
            }

			/*
			 * Add additional parameters to change object
			 */
            if(change.changeType === "label" || change.changeType === "property") {
                if(change.property.type === "question") {
                    item = survey.forms[change.property.formIndex].questions[change.property.itemIndex];
                    if(survey.forms_orig) {
                        forms_orig = survey.forms_orig[change.property.formIndex];
                        if (forms_orig) {
                            item_orig = forms_orig.questions[change.property.itemIndex];
                        }
                    }
                    change.property.name = item.name;
                    change.property.qId = item.id;
                    change.property.fId = item.fId;
                    change.property.childFormIndex = survey.forms.length - 1;
                    if(change.changeType === "property") {
                        setTypeSpecificChanges(change.property.prop, change, survey);
                    }
                } else if(change.property.type === "option") {
                    item = survey.optionLists[change.property.optionList].options[change.property.itemIndex];
                    optionListOrig = survey.optionLists_orig[change.property.optionList];
                    if(optionListOrig) {
                        item_orig = optionListOrig.options[change.property.itemIndex];
                    }
                    change.property.name = item.value;
                    change.property.o_id = item.id;
                    //if(change.changeType === "property") {
                    //	setOptionTypeSpecificChanges(change.property.prop, change, survey);
                    //}
                }

                if(change.changeType === "label") {
                    if(item_orig) {
                        change.property.oldVal = item_orig.labels[change.property.language][change.property.propType];
                    }

                    // Add a reference for the label
                    form = survey.forms[change.property.formIndex];
                    if(change.property.prop === "label" || change.property.prop === "media" && item.text_id) {
                        change.property.key = item.text_id;
                    } else if(change.property.prop === "hint" && item.hint_id) {
                        change.property.key = item.hint_id;
                    } else {
                        // Create reference for this new Label
                        if(change.property.type === "question") {
                            change.property.key = getFormPath(form) + "/" + item.name +
                                (change.property.prop === "hint" ? ":hint" : ":label");
                        } else if(change.property.type === "option") {
                            change.property.key = getFormPath(form) + "/" + change.property.qname + "/" + item.value + ":label";
                            change.property.o_id = item.id;
                        }
                    }
                } else {
                    if(item_orig) {
                        if(change.property.prop === "app_choices") {
                            change.property.oldVal = window.gAppChoiceArray.join(' ');
                        } else {
                            change.property.oldVal = item_orig[change.property.prop];
                        }
                    }
                }

                change.property.languageName = survey.languages[change.property.language].name;			// For logging the event

            } else if(change.changeType === "question") {
                form = survey.forms[change.question.formIndex];
                change.question.fId = form.id;
                if(change.action === "delete") {
                    item = survey.forms[change.question.formIndex].questions[change.question.itemIndex];
                    change.question.id = item.id;
                    change.question.name = item.name;
                    change.question.path = item.path;
                }
                //else {
                //	change.question.path = getFormPath(form) + "/" + change.question.name;
                //}
            } else if(change.changeType === "option") {
                if(change.action === "delete") {
                    item = survey.optionLists[change.option.optionList].options[change.option.itemIndex];
                    change.option.value = item.value;
                }
            }


			/*
			 * Add the change item to the array
			 */
            ci.items.push(change);
            if(change.action !== "add") {
                applyChange = removeDuplicateChange(changes, ci);
            }

			/*
			 * If this is a property update and the property is cascade_filters then convert to a string
			 * Filter values should be saved as a json string
			 */
            if(change.property) {
                if(change.property.type === "option" && change.property.prop === "cascade_filters") {
                    change.property.newVal = JSON.stringify(change.property.newVal);
                    change.property.oldVal = JSON.stringify(change.property.oldVal);
                }
            }

            if(applyChange) {
                if(change.property && (change.property.newVal !== change.property.oldVal)) {		// Add if the value has changed
                    changes.push(ci);
                } else if(change.action === "add" || change.action === "delete" || change.action === "move") {
                    changes.push(ci);
                }
            }
            setHasChanges(changes.length);

        }

		/*
		 * Annotate a change item with changes that are dependent on the type of the property
		 */
        function setTypeSpecificChanges(type, change, survey) {
            var i;
            if(type === "type") {
                var typeList = globals.model.qTypes;

                for(i = 0; i < typeList.length; i++) {
                    if(change.property.newVal === typeList[i].type) {
                        change.property.setVisible = true;
                        change.property.visibleValue = typeList[i].visible;
                        change.property.sourceValue = typeList[i].source;
                        break;
                    }
                }
            }
        }

		/*
		 * Remove duplicate updates.  This simplifies the analysis of changes to a survey
		 * For example if the user sets the label to "x", then changes it to "y" the change recorded should only be "y"
		 */
        function removeDuplicateChange(changes, change) {

            var j,
                item,
                element,
                newItem,
                newElement,
                newElementType,
                elementType;

            newItem = change.items[0];
            if(newItem.question) {
                newElement = newItem.question;
                newElementType = "question";
            } else if(newItem.option) {
                newElement = newItem.option;
                newElementType = "option";
            } else if(newItem.property) {
                newElement = newItem.property;
                newElementType = newElement.type;
            } else if(newItem.changeType === "optionlist") {
                newElement = {
                    itemIndex: newItem.name
                };
                newElementType = newItem.changeType;
            }

            for(j = 0; j < changes.length; j++) {

                item = changes[j].items[0];
                if(item.question) {
                    element = item.question;
                    elementType = "question";
                } else if(item.option) {
                    element = item.option;
                    elementType = "option";
                } else if(item.property) {
                    element = item.property;
                    elementType = element.type;
                } else if(item.changeType === "optionlist") {
                    element = {
                        itemIndex: item.name
                    };
                    elementType = newItem.changeType;
                }

				/*
				 * Only remove duplicates if
				 * 		- The new and existing element types are the same, ie both question or both option
				 * 		- They refer to the same item
				 */
                if(elementType === newElementType && newElement.itemIndex === element.itemIndex) {
                    if((newElementType === "question" && newElement.formIndex === element.formIndex) ||
                        (newElementType === "option" && newElement.optionList === element.optionList) ||
                        (newElementType === "optionlist")) {

                        console.log("Update to an item that is already in the update queue");


						/*
						 * 1. If this is a property update and there is already another update to
						 * 		- the same property
						 *    then remove the first update
						 */
                        if(newItem.action === "update" 	&& newItem.changeType === "property"
                            && item.changeType === "property") {

                            if(newElement.prop === element.prop) {
                                changes.splice(j,1);	// Remove the old item and apply the new one
                                return true;
                            }

							/*
							 * 2. If this is a label update and there is already another update to
							 * 		- the same language
							 * 		- the same label type (in the propType attribute)
							 *    then remove the first update
							 */
                        } else if(newItem.action === "update" 	&& newItem.changeType === "label"
                            && item.changeType === "label") {

                            if(newElement.languageName === element.languageName &&
                                newElement.propType === element.propType) {
                                changes.splice(j,1);	// Remove the old item and apply the new one
                                return true;
                            }

							/*
							 * 3. If this is an update to a property or label and
							 *      - the existing item is newly added then
							 * 		- merge the update into the added item
							 */
                        } else if(newItem.action === "update"
                            && ((elementType === "question" && typeof element.qId === "undefined") ||
                            (elementType === "option" && item.action === "add"))
                            && (newItem.changeType === "label" || newItem.changeType === "property")) {

                            if(newItem.changeType === "label") {

                                if(!element.labels) {
                                    element.labels = [];
                                }
                                if(!element.labels[newElement.language]) {
                                    element.labels[newElement.language] = {};
                                }
                                element.labels[newElement.language][newElement.propType] = newElement.newVal;

                            } else {

                                element[newElement.prop] = newElement.newVal;

                            }

                            if(elementType === "question") {
                                if(newElement.prop === "name") {
                                    element["path"] = newElement.path;
                                }
                            } else {
                                if(newElement.prop === "value") {
                                    element["path"] = newElement.path;
                                }
                            }

                            return false;		// Don't apply the change it has been merged

							/*
							 * 4. If this is a move of an item
							 */
                        } else if(newItem.action === "move") {
							/*
							 * Remove any modifications to this deleted element
							 */
                            element.seq = newElement.seq;
                            return false;


							/*
							 * 5. If this is a delete of an item
							 */
                        } else if(newItem.action === "delete") {
							/*
							 * Remove any modifications to this deleted element
							 * If the item is a group question then also remove its group end
							 */
                            changes.splice(j,1);	// Remove this item
                            return true;

                        }
                    }
                }
            }

            return true;
        }

		/*
		 * Update settings when the number of changes to apply changes
		 */
        function setHasChanges(numberChanges) {

            if(numberChanges === 0) {
                globals.changes = [];
                $('.m_save_survey').addClass("disabled").prop("disabled", true).find('.badge').html(numberChanges);
                $('.m_languages,#m_required,#m_not_required').removeClass("disabled").prop("disabled", false);
            } else {
                $('.m_save_survey').find('.badge').html(numberChanges);
                $('.m_languages,#m_required,#m_not_required').addClass("disabled").prop("disabled", true);
                $('.m_validate').removeClass("disabled").prop("disabled", false);
                $('.m_save_survey').removeClass("disabled").prop("disabled", false);
                //if(numberIssues("error") === 0) {
                //	$('.m_save_survey').removeClass("disabled").prop("disabled", false);
                //}
            }

        }


		/*
		 * Update the in memory model
		 */
        function updateModel(change) {
            var refresh = false,		// Set to true if the page needs to be refreshed with this change
                survey = globals.model.survey,
                question,
                theOption,
                property,
                length,
                i, j;

            if(change.property) {
				/*
				 * Update the property values
				 */
                property = change.property;

                if(property.type === "question") {			// Change to a question
                    question = survey.forms[property.formIndex].questions[property.itemIndex];
                    if(property.propType === "text" || property.propType === "hint") {		// Not a media change

                        if(property.prop === "label" || property.prop === "hint") {	   // Changing the label or hint

                            question.labels[property.language][property.propType] = property.newVal;

                        } else {						// Other properties, such as constraints, relevance question name

                            var oldVal;
                            if(property.prop === "app_choices") {
                                oldVal = window.gAppChoiceArray.join[' '];
                            } else {
                                oldVal = question[property.prop];
                            }
                            question[property.prop] = property.newVal;


                            if(property.setVisible) {
                                question["visible"] = property.visibleValue;
                                question["source"] = property.sourceValue;
                            }

							/*
							 * Set type dependent properties
							 */
                            if(property.prop === "type" || property.prop === "name") {

                                if(property.newVal.indexOf("select") == 0 || question.type.indexOf("select") == 0 ||
                                    property.newVal === "rank" || question.type === "rank") {	// Select question

                                    // Ensure there is a list name for this question
                                    if(!question.list_name && question.name) {
                                        question.list_name = question.name;
                                    }

                                    // Ensure there is a list of choices
                                    if(question.list_name) {
                                        var optionList = survey.optionLists[question.list_name];
                                        if(!optionList) {
                                            survey.optionLists[question.list_name] = {
                                                oSeq: [],
                                                options: []
                                            };
                                            option.refreshOptionListControls();
                                        }
                                    }

                                }
                            }


                            if(property.prop === "type") {

                                if(property.newVal == "begin repeat") {
                                    // New sub form
                                    survey.forms.push({
                                        id: undefined,
                                        name: survey.forms[property.formIndex].questions[property.itemIndex].name,  // name of question
                                        parentQuestionIndex: property.itemIndex,
                                        parentFormIndex: property.formIndex,
                                        questions: [],
                                        qSeq: []
                                    });
                                    question.childFormIndex = survey.forms.length - 1;

                                }

                                // Fix ups depending on oldVal
                                var newFormIdx;
                                var oldFormIdx;
                                var endQuestionIdx;
                                var groupMembers = [];
                                var mvStartIdx;
                                var itemIndex;
                                if(oldVal === "note") {
                                    // Remove the readonly status
                                    modelGeneratedChanges.push({
                                        changeType: "property",
                                        action: "update",
                                        source: "editor",
                                        property: {
                                            type: "question",
                                            prop: "readonly",
                                            newVal: false,
                                            language: property.language,
                                            formIndex: property.formIndex,
                                            itemIndex: property.itemIndex
                                        }
                                    });
                                    question.readonly = false;
                                } else if(oldVal === "begin group") {
                                    console.log("xxxxxxx fix up begin group");
                                    if(property.newVal === "begin repeat") {
                                        console.log("xxxxxxx move questions to repeat");
                                        // Move the questions in the group to the new form
                                        newFormIdx = question.childFormIndex;
                                        oldFormIdx = property.formIndex;
                                        oldLocation = property.itemIndex;
                                        newLocation = 0;
                                        endQuestionIdx = 0;

                                        name = question.name;
                                        endName = name + "_groupEnd";

										/*
										 * Get the questions to move
										 * Start from one past the begin group
										 */
                                        oldLocation++;
                                        for(i = oldLocation; i < survey.forms[oldFormIdx].qSeq.length; i++) {

                                            // Don't include the group end or any questions after it
                                            if(survey.forms[oldFormIdx].questions[survey.forms[oldFormIdx].qSeq[i]].name.toLowerCase() === endName.toLowerCase()) {
                                                endQuestionIdx = i;
                                                break;
                                            }
                                            groupMembers.push(survey.forms[oldFormIdx].qSeq[i]);
                                        }

										/*
										 * Move the group members
										 */
                                        mvStartIdx = oldLocation;
                                        for(i = 0; i < groupMembers.length; i++) {

                                            itemIndex = moveQuestion(survey, survey.forms[oldFormIdx].questions[groupMembers[i]],
                                                newFormIdx,
                                                newLocation++,
                                                oldFormIdx,
                                                mvStartIdx++);

                                            // Add the move to the changeset
                                            modelGeneratedChanges.push({
                                                changeType: "question",
                                                action: "move",
                                                source: "editor",
                                                question: {

                                                    type: survey.forms[oldFormIdx].questions[groupMembers[i]].type,
                                                    formIndex: newFormIdx,
                                                    itemIndex: itemIndex,
                                                    sourceFormIndex: oldFormIdx,
                                                    sourceItemIndex: survey.forms[oldFormIdx].questions[groupMembers[i]].itemIndex,
                                                    sourceFormId: survey.forms[oldFormIdx].id,
                                                    name: survey.forms[oldFormIdx].questions[groupMembers[i]].name
                                                }
                                            });

                                            console.log("xxxxx moved");
                                        }

                                    }
                                    console.log("xxxxxxx remove end group");
                                    applyToEndGroup(survey.forms[oldFormIdx], question.name, oldLocation, "delete", undefined, undefined);
                                    refresh = true;
                                } else if(oldVal === "begin repeat") {
                                    console.log("xxxxxxx fix up begin repeat");
                                    if(property.newVal === "begin group") {
                                        console.log("xxxxxxx move questions to group");

                                        // Move the questions in the repeat to the group
                                        newFormIdx = property.formIndex;
                                        newLocation = property.itemIndex + 1; // After the begin group
                                        oldLocation = 0;
                                        endQuestionIdx = 0;
                                        oldFormIdx = getSubFormIndex(newFormIdx, property.itemIndex);

                                        name = question.name;
                                        endName = name + "_groupEnd";

										/*
										 * Get the questions to move
										 */
                                        groupMembers = [];
                                        for(i = oldLocation; i < survey.forms[oldFormIdx].qSeq.length; i++) {
                                            groupMembers.push(survey.forms[oldFormIdx].qSeq[i]);
                                        }

										/*
										 * Move the group members
										 */
                                        mvStartIdx = oldLocation;
                                        for(i = 0; i < groupMembers.length; i++) {

                                            itemIndex = moveQuestion(survey, survey.forms[oldFormIdx].questions[groupMembers[i]],
                                                newFormIdx,
                                                newLocation++,
                                                oldFormIdx,
                                                mvStartIdx++);

                                            // Record the group that this question has been added to
                                            survey.forms[newFormIdx].questions[survey.forms[newFormIdx].qSeq[newLocation - 1]].memberGroup = name;

                                            // Add the move to the changeset
                                            modelGeneratedChanges.push({
                                                changeType: "question",
                                                action: "move",
                                                source: "editor",
                                                question: {

                                                    type: survey.forms[oldFormIdx].questions[groupMembers[i]].type,
                                                    formIndex: newFormIdx,
                                                    itemIndex: itemIndex,
                                                    sourceFormIndex: oldFormIdx,
                                                    sourceItemIndex: survey.forms[oldFormIdx].questions[groupMembers[i]].itemIndex,
                                                    sourceFormId: survey.forms[oldFormIdx].id,
                                                    name: survey.forms[oldFormIdx].questions[groupMembers[i]].name
                                                }
                                            });

                                            console.log("yyyy moved");
                                        }

                                    }
                                    refresh = true;
                                }


                            } else if(property.prop === "name") {
                                // update the end group name
                                if(survey.forms[property.formIndex].questions[property.itemIndex].type === "begin group") {
                                    applyToEndGroup(survey.forms[property.formIndex],
                                        oldVal, 0, "rename", property.newVal, undefined);

                                }
                            }

							/*
							 * If this is a change to a begin group then refresh the entire form
							 */
                            if(survey.forms[property.formIndex].questions[property.itemIndex].type === "begin group") {
                                refresh = true;
                            }


                        }

                    } else {
                        // For non text changes update all languages
                        for(i = 0; i < survey.forms[property.formIndex].questions[property.itemIndex].labels.length; i++) {
                            survey.forms[property.formIndex].questions[property.itemIndex].labels[i][property.propType] = property.newVal;
                            survey.forms[property.formIndex].questions[property.itemIndex].labels[i][property.propType + "Url"] =
                                _getUrl(property.newVal, false, property.propType, property.isSurveyLevel, survey.id);
                        }
                    }
                } else if(property.type === "option") {	// Change to an option

                    theOption = survey.optionLists[property.optionList].options[property.itemIndex];
                    theOption[property.prop] = property.newVal;

                    if(property.propType === "text") {
                        if(property.prop === "label") {
                            survey.optionLists[property.optionList].options[property.itemIndex].labels[property.language][property.propType] = property.newVal;
                        }
                    } else {
                        // For non text changes update all languages
                        for(i = 0; i < survey.optionLists[property.optionList].options[property.itemIndex].labels.length; i++) {
                            survey.optionLists[property.optionList].options[property.itemIndex].labels[i][property.propType] = property.newVal;
                            survey.optionLists[property.optionList].options[property.itemIndex].labels[i][property.propType + "Url"] =
                                _getUrl(property.newVal, false, property.propType, property.isSurveyLevel, survey.id);
                        }
                    }
                } else if(property.type === "optionlist") {	// Change to an optionlist
                    // 1. Rename the option list
                    survey.optionLists[property.newVal] = survey.optionLists[property.oldVal];
                    delete survey.optionLists[property.oldVal];

                    // 2. Update all questions that refer to this option list so they reference the new one
                    for(i = 0; i < survey.forms.length; i++) {
                        for(j = 0; j < survey.forms[i].questions.length; j++) {
                            if(survey.forms[i].questions[j].list_name === property.oldVal) {
                                survey.forms[i].questions[j].list_name = property.newVal
                            }
                        }
                    }

                    //3.  Update the optionList property for all options in the changed optionList
                    for(i = 0; i < survey.optionLists[property.newVal].options.length; i++) {
                        survey.optionLists[property.newVal].options[i].optionList = property.newVal;
                    }
                } else {
                    console.log("Error: unknown property type: " + property.type);
                }

            } else if(change.changeType === "question") {		// Not a change to a property
                if(change.action === "move") {

                    var sourceForm = change.question.sourceFormIndex;
                    var sourceItem = change.question.sourceItemIndex;
                    var targetForm = change.question.formIndex;
                    var oldLocation = change.question.sourceSeq;
                    var newLocation = change.question.seq;
                    var question = survey.forms[sourceForm].questions[sourceItem];
                    var name = question.name;
                    var endName;
                    var form;

                    if(question.type === "begin group") {	// Move all the group members
                        name = question.name;
                        endName = name + "_groupEnd";
                        form = survey.forms[sourceForm];

						/*
						 * Get the questions to move
						 */
                        var groupMembers = [];
                        for(i = oldLocation; i < form.qSeq.length; i++) {

                            groupMembers.push(form.qSeq[i]);
                            if(form.questions[form.qSeq[i]].name.toLowerCase() === endName.toLowerCase()) {
                                break;
                            }
                        }

						/*
						 * Move the group members
						 */
                        for(i = 0; i < groupMembers.length; i++) {

                            moveQuestion(survey, form.questions[groupMembers[i]],
                                targetForm,
                                newLocation++,
                                sourceForm,
                                oldLocation++);

                            console.log("moved");
                        }

                    } else {

                        // Move the single question
                        change.question.itemIndex = moveQuestion(survey, question,
                            targetForm,
                            newLocation,
                            sourceForm,
                            oldLocation);
                        
                        // Fix up the form being moved to point to its new parent
                        var movedForm;
                        if(question.type === "begin repeat") {
                            // Get the form being moved
                            for(i = 0; i < survey.forms.length; i++) {
                                movedForm = survey.forms[i];
                                if(movedForm.parentFormIndex === sourceForm && movedForm.parentQuestionIndex === sourceItem) {
                                    movedForm.parentFormIndex = targetForm;
                                    movedForm.parentform = survey.forms[targetForm].id;
                                    movedForm.parentQuestionIndex = change.question.itemIndex;
                                    break;
                                }
                            }
                        }

                    }
                    refresh = true;		// Do a complete refresh after moving questions

                } else if(change.action === "add") {
                    length = survey.forms[change.question.formIndex].questions.push(change.question);			// Add the new question to the end of the array of questions
                    change.question.itemIndex = length -1;
                    survey.forms[change.question.formIndex].qSeq.splice(change.question.seq, 0, length - 1);	// Update the question sequence array

                    if(change.question.firstQuestion) {
                        refresh = true;		// Refresh all the questions when adding the first question to a form
                    } else if(change.question.type === "end group") {
                        refresh = false;
                    }

                    // Add a subform if required
                    if(change.question.type === "begin repeat") {
                        // New sub form
                        survey.forms.push({
                            id: undefined,
                            name: change.question.name,
                            parentQuestionIndex: change.question.itemIndex,
                            parentFormIndex: change.question.formIndex,
                            questions: [],
                            qSeq: []
                        });
                        change.question.childFormIndex = survey.forms.length - 1;
                    }

                    refresh = true;			// DEBUG - remove after fixing update of html

                } else if(change.action === "delete") {
                    var form = survey.forms[change.question.formIndex];
                    var question = form.questions[change.question.itemIndex];
                    form.qSeq.splice(change.question.seq, 1);	// Remove item from the sequence array
                    question.deleted = true;	// Mark deleted
                    if(question.type === "begin group") {
                        applyToEndGroup(form, question.name, change.question.seq, "delete", undefined, change.question.formIndex);
                        refresh = true;
                    }
                } else {
                    console.log("Unknown action: " + change.action);
                }


            } else if(change.changeType === "option") {				// Change to an option
                if(change.action === "move") {

                    var sourceOptionList = survey.optionLists[change.option.sourceOptionList];
                    var targetOptionList = survey.optionLists[change.option.optionList];

                    var theOption = sourceOptionList.options[change.option.sourceItemIndex];
                    var newOption = jQuery.extend(true, {}, theOption);
                    var oldLocation = change.option.sourceSeq;
                    var newLocation = change.option.seq;

                    // 1. Add the option in the new location
                    length = targetOptionList.options.push(newOption);
                    change.option.itemIndex = length -1;
                    change.option.value = theOption.value;
                    targetOptionList.oSeq.splice(change.option.seq, 0, length - 1);

                    // 2. Remove the option from the old location
                    // The old location may have changed if the new location was inserted before it
                    if(newLocation < oldLocation && change.option.sourceOptionList == change.option.optionList) {
                        oldLocation++;
                    }
                    sourceOptionList.oSeq.splice(oldLocation, 1);
                    theOption.deleted = true;

                    // 3. Update any items in the change list to the new location
                    if(globals.changes) {
                        for(i = 0; i < globals.changes.length; i++) {
                            var existingChange = globals.changes[i];
                            if(existingChange.changeType === "option") {
                                for(j = 0; j < existingChange.items.length; j++) {
                                    var existingItem = existingChange.items[j];
                                    if(existingItem.option.optionList === theOption.optionList &&
                                        existingItem.option.itemIndex === theOption.itemIndex) {
                                        // We moved an option thats in the change queue
                                        existingItem.option.optionList = change.option.optionList;
                                        existingItem.option.itemIndex = change.option.itemIndex;
                                    }
                                }
                            }
                        }
                    }
                    refresh = true;

                } else if(change.action === "add") {
                    length = survey.optionLists[change.option.optionList].options.push(change.option);			// Add the new option to the end of the array of options
                    change.option.itemIndex = length -1;
                    survey.optionLists[change.option.optionList].oSeq.splice(change.option.seq, 0, length - 1);	// Update the option sequence array

                } else if(change.action === "delete") {
                    survey.optionLists[change.option.optionList].oSeq.splice(change.option.seq, 1);	// Remove item from the sequence array
                    survey.optionLists[change.option.optionList].options[change.option.itemIndex].deleted = true;
                } else {
                    console.log("Unknown action: " + change.action);
                }

            } else if(change.changeType === "optionlist") {				// Change to an option list
                if(change.action === "add") {
                    survey.optionLists[change.name] = {
                        oSeq: [],
                        options: []
                    };
                    refresh = true;
                } else if(change.action === "delete") {
                    delete survey.optionLists[change.name];
                    refresh = true;
                } else {
                    console.log("Unknown action: " + change.action);
                }
            }
            return refresh;
        }

		/*
		 * Move a question
		 */
        function moveQuestion(survey, question, targetForm, newLocation, sourceForm, oldLocation) {

            var newQuestion = jQuery.extend(true, {}, question),
                itemIndex,
                i,j,
                change,
                item,
                length;

            // 1. Add the question in the new location
            length = survey.forms[targetForm].questions.push(newQuestion);			// Add the new question to the end of the array of questions
            itemIndex = length - 1;
            survey.forms[targetForm].qSeq.splice(newLocation, 0, length - 1);	// Update the question sequence array

            // 1.5 Update the sequences stored in the question objects
            //for(i = 0; i < survey.forms[targetForm].qSeq.length; i++) {
            //	var idx = survey.forms[targetForm].qSeq[i];
            //    survey.forms[targetForm].questions[idx].seq = i;
            //}

            // 2. Remove the question from the old location
            // The old location may have changed if the new location was inserted before it
            if(newLocation < oldLocation && sourceForm === targetForm) {
                oldLocation++;
            }
            survey.forms[sourceForm].qSeq.splice(oldLocation, 1);
            question.deleted = true;

            // 3. Update any items in the change list to put to the new location
            if(globals.changes) {
                for(i = 0; i < globals.changes.length; i++) {
                    change = globals.changes[i];
                    if(change.changeType === "question") {
                        for(j = 0; j < change.items.length; j++) {
                            item = change.items[j];
                            if(item.question.formIndex === question.formIndex &&
                                item.question.itemIndex === question.itemIndex) {
                                // We moved a question thats in the change queue
                                item.question.formIndex = targetForm;
                                item.question.itemIndex = itemIndex;
                                item.question.fId = survey.forms[targetForm].id;
                            }
                        }
                    }
                }
            }

            // Update the model
            question.itemIndex = itemIndex;
            question.formIndex = targetForm;

            return itemIndex;
        }
		/*
		 * Apply a change to the "end group" of a group
		 */
        function applyToEndGroup(form, name, start_seq, action, new_name, form_index) {
            var i,
                end_name = name + "_groupEnd";

            // Remove the group from the model
            for(i = start_seq; i < form.qSeq.length; i++) {
                if(form.questions[form.qSeq[i]].name === end_name) {
                    if(action === "delete") {
                        form.questions[form.qSeq[i]].deleted = true;
                        form.qSeq.splice(i, 1);
                    } else if(action === "rename") {
                        form.questions[form.qSeq[i]].name = new_name + "_groupEnd";
                    }
                    break;
                } else {

                    // Delete the member
                    if(action === "delete"  && typeof form_index !== "undefined") {
                        modelGeneratedChanges.push({
                            changeType: "question",
                            action: "delete",
                            source: "editor",
                            question: {
                                seq: i,
                                formIndex: form_index,
                                itemIndex: i
                            }
                        });
                    }
                }
            }

            // Remove group end from the pending changes list
            for(i = 0; i < globals.changes.length; i++) {
                if(globals.changes[i].items[0].question) {
                    if(globals.changes[i].items[0].question.name === end_name) {
                        if(action === "delete") {
                            globals.changes.splice(i, 1);
                        } else if(action === "rename") {
                            globals.changes[i].items[0].question.name = new_name + "_groupEnd";
                        }
                    }
                }
            }

        }

		/*
		 * Update the modified HTML element
		 */
        function updateHtmlElement(change) {
            var newMarkup,
                survey = globals.model.survey,
                $changedRow,
                i,
                collapsedPanels = [];

            if(change.changeType === "label") {
                if(change.property.propType === "image") {

                    newMarkup = markup.addMedia("Image",
                        change.property.newVal,
                        _getUrl(change.property.newVal, false, 'image', change.property.isSurveyLevel, survey.id),
                        _getUrl(change.property.newVal, true, 'image', change.property.isSurveyLevel, survey.id)
                    );

                } else if(change.property.propType === "video") {

                    newMarkup = markup.addMedia("Video",
                        change.property.newVal,
                        _getUrl(change.property.newVal, false, 'video', change.property.isSurveyLevel, survey.id),
                        _getUrl(change.property.newVal, true, 'video', change.property.isSurveyLevel, survey.id)
                    );

                } else if(change.property.propType === "audio") {

                    newMarkup = markup.addMedia("Audio",
                        change.property.newVal,
                        _getUrl(change.property.newVal, false, 'audio', change.property.isSurveyLevel, survey.id),
                        undefined
                    );

                }

                if(newMarkup) {
                    if(change.property.type === "question") {
                        $changedRow = $('#question' + change.property.formIndex + '_' + change.property.itemIndex);
						/*
						 $changedRow = $('#formList').find('li.question').filter(function(index){
						 var $this = $(this);
						 return $this.data("fid") == change.property.formIndex && $this.data("id") == change.property.itemIndex;
						 });
						 */
                    } else {

                        $changedRow = $('#choiceView').find('tr').filter(function(index){
                            var $this = $(this);
                            return $this.data("id") == change.property.itemIndex;
                        });
                    }

                    if($changedRow) {
                        $changedRow.find('.' + change.property.propType + 'Element').replaceWith(newMarkup);
                    }
                }
            } else if(change.changeType === "question") {

                var form = survey.forms[change.question.formIndex];

                if(change.action === "add") {
                    var preceedingQuestion = form.questions[form.qSeq[change.question.seq-1]];
                    if(change.question.locn === "after") {
                        change.question.$relatedElement.after(markup.addOneQuestion(form, change.question, change.question.formIndex, change.question.itemIndex, true, undefined, false));
                    } else {
                        change.question.$relatedElement.prev().before(markup.addOneQuestion(form, change.question, change.question.formIndex, change.question.itemIndex, true, undefined, false));
                    }
                    $changedRow = $("#question" + change.question.formIndex + "_" + change.question.itemIndex);

                } else if(change.action === "delete") {
                    change.question.$deletedElement.prev().remove();	// Remove the add before button
                    change.question.$deletedElement.remove();
                }
            } else if(change.changeType === "option") {
                if(change.action === "add") {
                    var optionList,
                        $ref;


                    optionList = survey.optionLists[change.option.optionList];

                    // get the new markup
                    newMarkup = option.addOneOption(optionList,
                        change.option,
                        change.option.formIndex,
                        change.option.itemIndex,
                        change.option.optionList,
                        change.option.qName);


                    $ref = $('#choiceView').find('.editor_element').
                    filter(function(index) {
                        var $this = $(this);
                        return $this.data("id") == change.option.optionIndex;
                    });

                    if(change.option.locn == "after") {
                        $ref.after(newMarkup);
                        $changedRow = $ref.next();
                    } else if(change.option.locn == "before") {
                        $ref.before(newMarkup);
                        $changedRow = $ref.prev();
                    } else {
                        // put at end of table
                        $('#choiceView tbody').append(newMarkup);
                        $changedRow = $('#choiceView tbody tr:last');
                    }

                } else if(change.action === "delete") {
                    change.option.$deletedElement.remove();
                }
            } else if(change.changeType === "property") {
                // Apply any markup changes that result from a property change


                // 1. Update the question / option
                if(change.property.type === "option") {
					/*
					 * No option property changes result in markup changes
					 */
                } else if(change.property.type === "question") {

                    // Get the changed question row
                    $changedRow = $('#formList').find('li.question').filter(function(index){
                        var $this = $(this);
                        return $this.data("fid") == change.property.formIndex && $this.data("id") == change.property.itemIndex;
                    });

                    newMarkup = markup.addOneQuestion(
                        survey.forms[change.property.formIndex],
                        survey.forms[change.property.formIndex].questions[change.property.itemIndex],
                        change.property.formIndex,
                        change.property.itemIndex,
                        false, undefined, true);

                }

				/*
				 * Apply the update
				 */
                if($changedRow) {

                    // Get the current list of collapsed panels
                    $('.collapse.show', $changedRow).each(function(){
                        collapsedPanels.push($(this).closest('li').attr("id"));
                    });

                    // Update the content view
                    $changedRow.replaceWith(newMarkup);

                    // Restore collapsed panels
                    for(i = 0; i < collapsedPanels.length; i++) {
                        $('#' + collapsedPanels[i]).find('.collapse').addClass("show");
                    }

                    // Since we replaced the row we had better get the replaced row so that actions can be reapplied
                    if(change.property.type === "option") {
                        $changedRow = $('#formList').find('li.option').filter(function(index){
                            var $this = $(this);
                            return $this.data("fid") == change.property.formIndex && $this.data("id") == change.property.itemIndex;
                        });
                    } else {
                        $changedRow = $('#formList').find('li.question').filter(function(index){
                            var $this = $(this);
                            return $this.data("fid") == change.property.formIndex && $this.data("id") == change.property.itemIndex;
                        });
                    }
                }

            }


            return $changedRow;
        }

        function getQuestionsUsingOptionList(list) {
            var i, j,
                survey = globals.model.survey,
                forms = survey.forms,
                question,
                nameArray = [];

            for(i = 0; i < forms.length; i++) {
                for(j = 0; j < forms[i].questions.length; j++) {
                    question = forms[i].questions[j];
                    if(!question.deleted &&  !question.soft_deleted &&
                        (question.type.indexOf("select") === 0 || question.type === "rank") &&
                        question.list_name === list) {

                        nameArray.push(question.name);
                    }
                }
            }
            return nameArray;
        }

		/*
		 * Media functions
		 */
        function _getUrl(newVal, thumbs, type, isSurveyLevel, sId) {
            var url = "/surveyKPI/file/",
                filebase,
                ext,
                index;

            if(newVal) {

                // Videos and other derived thumbnails will have type jpg
                if(thumbs && type !== "image") {
                    index = newVal.lastIndexOf('.');
                    filebase = url.substr(0, index);
                    newVal = filebase + ".jpg";
                }

                url += newVal;

                if(isSurveyLevel) {
                    url += "/survey/"
                    url += sId;
                } else {
                    url += "/organisation";
                }

                if(thumbs) {
                    url += "?thumbs=true";
                }


            } else {
                url = undefined;
            }

            return url;
        }

		/*
		 * Create a message from response to a save request
		 */
        function addUpdateMessage(data, forError) {
            var h = [],
                idx = -1,
                j,
                action,
                result,
                name;

            if(data.updateFailed && forError) {
                h[++idx] = '<li>';
                h[++idx] = data.errorMsg;
                h[++idx] = '</li>';
            } else if(!data.updateFailed && !forError) {
                for(j = 0; j < data.items.length; j++) {
                    h[++idx] = '<li>';
                    h[++idx] = getChangeDescription(data.items[j]);
                    h[++idx] = '</li>';
                }
            }

            return h.join("");
        }

		/*
		 * ***************************************************************
		 * Validate each change to a question or option
		 */
        function validateItem(container, itemIndex, itemType, removeExisting) {

            var i, j,
                form,
                survey = globals.model.survey,
                item,
                name,
                isDeleted,
                isValid = true,
                hasDuplicate = false,
                changes = globals.changes,
                numberErrors,
                numberWarnings;

            if(itemType === "question") {
                item = survey.forms[container].questions[itemIndex];
                name = item.name;
            } else if(itemType === "option") {
                item = survey.optionLists[container].options[itemIndex];
                name = item.value;
            } else if(itemType === "optionlist") {
                item = survey.optionLists[container];
                name = container;
            }

			/*
			 * Remove the existing error
			 * If this validation has been called by validateAll() then this removal would already have been done
			 */
            if(removeExisting) {
                removeValidationError(container, itemIndex,	"item", itemType);
            }

            if(!item.deleted &&
                ((itemType === "question" && markup.includeQuestion(item)) ||
                (itemType === "optionlist") ||
                (itemType === "option"))) {

                // Validate the name
                isValid = validateName(container, itemIndex, name, itemType, removeExisting);

				/*
				 * Question specific validations
				 */
                if(itemType === "question") {

                    // Check references to other questions
                    isValid = checkReferences(container, itemIndex, itemType, item);
                    isValid = checkSelfReferences(container, itemIndex, itemType, item);

                    if(isValid) {	// Check parenthesis on relevant
                        isValid = checkParentheisis(container, itemIndex, itemType, item.relevant);
                    }

                    if(isValid) {	// Check parenthesis on constraint
                        isValid = checkParentheisis(container, itemIndex, itemType, item.constraint);
                    }

                    if(isValid) {	// Check parenthesis on calculation
                        isValid = checkParentheisis(container, itemIndex, itemType, item.calculation);
                    }

                    if(isValid) {	// Check parenthesis on choiceFilter
                        isValid = checkParentheisis(container, itemIndex, itemType, item.choice_filter);
                    }

	                if(isValid && (item.type === "child_form" || item.type === "parent_form")) {	// Check that form launch types have a reference form

	                    if(item.parameters && item.parameters.indexOf("form_identifier") >= 0) {
	                        isValid = true;
	                    } else {
	                        isValid = false;
		                    addValidationError(
			                    container,
			                    itemIndex,
			                    "item",
			                    localise.set["ed_f_l"],	// Only one geometry question can be added to a form
			                    itemType,
			                    "error");
	                    }

	                }

	                if(isValid && item.type === "begin repeat") {	// Check that a repeat group has children

	                    var itemx = item;
		                var childFormIndex = getSubFormIndex(item.formIndex, itemIndex);
		                var questions = survey.forms[childFormIndex].questions;
                        var isEmpty = true;

                        if(questions.length > 0) {
                            isEmpty = false;
                        }

                        if(isEmpty) {
	                        isValid = false;
	                        addValidationError(
		                        container,
		                        itemIndex,
		                        "item",
		                        localise.set["ed_emp_rep"],
		                        itemType,
		                        "warning");
                        }
	                }

	                if(isValid && item.type === "begin group") {	// Check that a group has children

		                var endName = item.name + "_groupEnd";
		                var questions = survey.forms[item.formIndex].questions;
		                var qSeq = survey.forms[item.formIndex].qSeq;
		                var isEmpty = true;
		                var inGroup = false;

                        if(!qSeq) {
                            qSeq = [];
                        }
		                /*
						 * Get the questions in the group
						 */
		                for(j = 0; j < qSeq.length; j++) {

		                    if(inGroup === true) {
			                    if (questions[qSeq[j]].name.toLowerCase() === endName.toLowerCase()) {
				                    break;
			                    } else {
				                    isEmpty = false;
				                    break;
			                    }
		                    }

			                if(qSeq[j] === itemIndex) {
				                inGroup = true;
			                }

		                }

		                if(isEmpty && inGroup) {
			                isValid = false;
			                addValidationError(
				                container,
				                itemIndex,
				                "item",
				                localise.set["ed_emp_rep"],
				                itemType,
				                "warning");
		                }
	                }


                } else if(itemType === "option") {
                    // Check references to other questions
                    isValid = checkReferences(container, itemIndex, itemType, item);
                }

                if(!isValid) {
                    updateModelWithErrorStatus(container, itemIndex, itemType);	// Update model and DOM
                }

				/*
				 * If there were no errors check for warnings
				 * Only do this on validateAll as otherwise it is just annoying to get the warnings
				 */
                if(!removeExisting) {
                    if(isValid) {
                        if(item.visible || itemType === "option") {
                            isValid = checkBlankLabels(container, itemIndex, itemType, item, "warning");
                        }
                    }

                    if(isValid) {
                        if(item.visible && itemType === "question") {
                            isValid = checkMissingChoices(container, itemIndex, itemType, item, "warning");
                        }
                    }
                }
            }

            // Set the control buttons
            numberErrors = numberIssues("error");
            numberWarnings = numberIssues("warning");

            $('.error-count').html(numberErrors);
            $('.warning-count').html(numberWarnings);

            //if(numberErrors > 0) {
            //	$('.m_save_survey').addClass("disabled").prop("disabled", true);
            //} else
            if(changes.length > 0) {
                $('.m_save_survey').removeClass("disabled").prop("disabled", false);
            }

            if(numberErrors > 0 || numberWarnings > 0) {

                $('#error-nav-btns').show();
                if(numberErrors > 0) {
                    $('#next-error').removeClass("disabled");
                } else {
                    $('#next-error').addClass("disabled");
                }
                if(numberWarnings > 0) {
                    $('#next-warning').removeClass("disabled");
                } else {
                    $('#next-warning').addClass("disabled");
                }
            }  else {
                $('#error-nav-btns').hide();
            }

        }

		/*
		 * Check for blank labels
		 */
        function checkBlankLabels(container, itemIndex, itemType, item) {
            var i,
                valid = true;

            for(i = 0; i < item.labels.length; i++) {
                if(typeof item.labels[i].text === "undefined" ||
                    (typeof item.labels[i].text === "undefined" && item.labels[i].text.trim().length === 0)) {
                    if(itemType === "question" && (item.type === "begin repeat" || item.type === "begin group")) {
                        continue;		// Don't report warnings on blank labels for these question types
                    }
                    addValidationError(
                        container,
                        itemIndex,
                        "item",
                        localise.set["msg_err_bll"] + " " + htmlEncode(globals.model.survey.languages[i].name),
                        itemType,
                        "warning");
                    valid = false;
                    break;
                }
            }

            return valid;
        }

		/*
		 * Check for Missing choices in a select questions
		 */
        function checkMissingChoices(container, itemIndex, itemType, item) {
            var i,
                valid = true,
                survey;

            if(item.type.indexOf('select') === 0 || item.type === "rank") {
                if(typeof globals.model.survey.optionLists[item.list_name] === "undefined") {
                    valid = false;
                }
                if(valid) {
                    var list = globals.model.survey.optionLists[item.list_name].options;
                    valid = false;
                    for(i = 0; i < list.length; i++) {
                        if(!list[i].deleted) {
                            valid = true;
                            break;
                        }
                    }
                }

                if(!valid) {
                    addValidationError(
                        container,
                        itemIndex,
                        "item",
                        "No choices in the choice list ",
                        itemType,
                        "warning");
                    valid = false;
                }

            }


            return valid;
        }

		/*
		 * Check for mismatched parenthesis
		 */
        function checkParentheisis(container, itemIndex, itemType, elem) {
            var i,
                c,
                depth = 0,
                lastOpen,
                errorText,
                isValid = true,
                locn;

            if(elem) {
                for(i = 0; i < elem.length; i++) {
                    c = elem.charAt(i);
                    if( c === '(') {
                        depth++;
                        locn = i;
                    } else if( c === ')') {
                        depth--;
                        locn = i;
                    }
                    if(depth < 0) {
                        break;
                    }
                }

                if(depth != 0) {
                    errorText = "Mis-matched parenthesis: " + elem.slice(0, locn) + '<b><span style="color:red;">' +
                        elem.slice(locn, locn + 1) +
                        '</span></b>' +
                        elem.slice(locn + 1);

                    addValidationError(
                        container,
                        itemIndex,
                        "item",
                        "Mis-matched parenthesis: " + errorText,
                        itemType,
                        "error");
                    isValid = false;
                }
            }

            return isValid;
        }

		/*
		 * Return false if the calculation is not valid
		 */
        function checkReferences(container, itemIndex, itemType, item) {

            var refQuestions = {},
                survey = globals.model.survey,
                form,
                i, j,
                name;

            // Get a list of references to other questions
            if(itemType  === "question") {
                getReferenceNames(item.relevant, refQuestions);
                getReferenceNames(item.constraint, refQuestions);
                getReferenceNames(item.calculation, refQuestions);
                getReferenceNames(item.choice_filter, refQuestions);
            }
            for(i = 0; i < item.labels.length; i++) {
                var text = item.labels[i].text;
                if(typeof text === "string") {
                    getReferenceNames(item.labels[i].text, refQuestions);
                }
            }

            if(checkExistenceOfReferences(refQuestions, survey)) {

                for (name in refQuestions) {
                    if (refQuestions.hasOwnProperty(name)) {
                        if (!refQuestions[name].exists) {
                            addValidationError(
                                container,
                                itemIndex,
                                "item",
                                localise.set["c_question"] + " ${" + name + "} " + localise.set["msg_not_f"],
                                itemType,
                                "error");
                            return false;
                        }
                    }
                }
            }

            return true;

        }

        /*
         * Return false if the calculation is not valid
         */
        function checkSelfReferences(container, itemIndex, itemType, item) {

            var refQuestions = {},
                survey = globals.model.survey,
                form,
                i, j,
                name;

            // Get a list of references to other where self cannot be references
            if(itemType  === "question") {
                getReferenceNames(item.relevant, refQuestions);
                getReferenceNames(item.calculation, refQuestions);
                getReferenceNames(item.choice_filter, refQuestions);
            }
            for(i = 0; i < item.labels.length; i++) {
                var text = item.labels[i].text;
                if(typeof text === "string") {
                    getReferenceNames(item.labels[i].text, refQuestions);
                }
            }

            var refCount = 0;
            for (name in refQuestions) {
                if (refQuestions.hasOwnProperty(name)) {
                    refCount++;
                }
            }

            if(refCount > 0) {

                form = survey.forms[container];
                var otherItem = form.questions[itemIndex];
                var questionType = otherItem.type;
                if (!otherItem.deleted && !otherItem.soft_deleted && questionType !== "end group") {

                    for (name in refQuestions) {
                        if (refQuestions.hasOwnProperty(name)) {
                            if (name === otherItem.name) {
                                addValidationError(
                                    container,
                                    itemIndex,
                                    "item",
                                    localise.set["c_question"] + " ${" + name + "} " + localise.set["e_h_c1"],
                                    itemType,
                                    "error");
                                return false;
                            }
                        }
                    }

                }

            }

            return true;

        }

		/*
		 * Error types:  name || dupname || noname || dupgeom
		 */
        function addValidationError(container, itemIndex, errorType, msg, itemType, severity) {

            // Push error into validation array
            globals.errors.push({
                container: container,
                itemIndex: itemIndex,
                errorType: errorType,
                itemType: itemType,
                msg: msg,
                severity: severity
            });

            updateModelWithErrorStatus(container, itemIndex, itemType);

        }

		/*
		 * Update the model and the DOM to report any erors on the question
		 */
        function updateModelWithErrorStatus(container, itemIndex, itemType) {

            var $changedRow,
                survey = globals.model.survey,
                hasError = false,
                hasWarning = false,
                msg = "",
                i,
                errors = globals.errors,
                item;

            if(itemType === "question") {
                item = survey.forms[container].questions[itemIndex];
                $changedRow = $('#question' + container + '_' + itemIndex);
            } else if(itemType === "option" || itemType === "optionlist") {
                item = survey.optionLists[container];
                $changedRow = $('#ol_' + container);
            }

            for(i = errors.length - 1; i >= 0; i--) {
                if(errors[i].itemType === itemType && errors[i].container === container &&
                    errors[i].itemIndex === itemIndex) {

                    if(errors[i].severity === "error") {
                        hasError = true;
                        msg = errors[i].msg;
                    } else {
                        hasWarning = true;
                        if(!hasError) {
                            msg = errors[i].msg;
                        }
                    }

                    if(errors[i].errorType === "name") {	// Break on name errors as this is the highest level error to show
                        break;
                    }

                }
            }

            // Update Model
            if(typeof item !== "undefined") {
                item.error = hasError;
                item.warning = hasWarning;
                item.errorMsg = msg;
            }

            // Update DOM
            $changedRow.find('.error-msg').html(msg);	// Add message
            if(globals.gShowingChoices) {
                $(".choice_filter_error").html(msg);
            }

            $changedRow.removeClass("error warning");
            if(hasError) {
                $changedRow.addClass("error");
                if(itemType === "question") {
                    $changedRow.find('.question_type').addClass("disabled");
                }
            } else {

                if(itemType === "question" && !item.published) {
                    $changedRow.find('.question_type').removeClass("disabled");
                }

                if(hasWarning) {
                    $changedRow.addClass("warning");
                }

            }
        }

		/*
		 * Remove the specified error type (see addValidationError for error types)
		 */
        function removeValidationError(container, itemIndex, errorType, itemType) {

            var i,
                $changedRow,
                survey = globals.model.survey;

            // Remove error
            for(i = globals.errors.length - 1; i >= 0; i--) {
                if(globals.errors[i].itemType === itemType &&
                    globals.errors[i].container === container &&
                    globals.errors[i].itemIndex === itemIndex &&
                    globals.errors[i].errorType === errorType) {
                    globals.errors.splice(i, 1);
                }
            }

            updateModelWithErrorStatus(container, itemIndex, itemType);  // Update model

        }

		/*
		 * Validate a question or option name
		 */
        function validateName(container, itemIndex, val, itemType, removeExisting) {

            var i, j,
                form,
                optionList,
                otherItem,
                survey = globals.model.survey,
                isValid = true,
                hasDuplicate = false,
                hasDuplicateOption = false,
                itemDesc,
                questionType,
                question,
                valLower;

            if(itemType === "question") {
                question = survey.forms[container].questions[itemIndex];
            }

            // Clear the existing name validation errors
            if(removeExisting) {
                removeValidationError(
                    container,
                    itemIndex,
                    "name",
                    itemType);
            }

            if(itemType === "question" ) {
                itemDesc = "question";
            } else if(itemType === "option" ) {
                itemDesc = "choice";
            } else if(itemType === "optionlist" ) {
                itemDesc = "choice list";
            } else {
                itemDesc = "unknown";
            }

            // Check for empty name
            if(typeof val === "undefined" || val === "") {
                addValidationError(
                    container,
                    itemIndex,
                    "name",
                    "This " + itemDesc + " does not have a name.  Specify a unique name.",
                    itemType,
                    "error");
                isValid = false;
            }

            // Check for invalid characters
            if(isValid) {
                if(itemType === "question") {
                    isValid = isValidODKQuestionName(val);

                    if(!isValid) {
                        addValidationError(
                            container,
                            itemIndex,
                            "name",
                            localise.set["ed_vq"],
                            itemType,
                            "error");

                    }

                } else {
                    isValid = isValidODKOptionName(val);

                    if(!isValid) {
                        addValidationError(
                            container,
                            itemIndex,
                            "name",
                            localise.set["ed_vc"] + " : "  + htmlEncode(val),
                            itemType,
                            "error");

                    }
                }
            }

			/*
			 * Question name change require the questions in all the forms to be validated for duplicates
			 */
            if(isValid) {

                valLower = val.toLowerCase();

                if(itemType === "question") {
                    for(i = 0; i < survey.forms.length; i++) {
                        form = survey.forms[i];
                        for(j = 0; j < form.questions.length; j++) {
                            questionType = form.questions[j].type;
                            if(questionType !== "end group") {
                                if(!(i === container && j === itemIndex)) {	// Don't test the question against itself!
                                    otherItem = form.questions[j];
                                    if(otherItem.name.toLowerCase() === valLower) {
                                        if((!otherItem.deleted && !otherItem.soft_deleted)
                                            || (otherItem.soft_deleted && otherItem.type !== questionType)) {
                                            hasDuplicate = true;
                                            break;
                                        }
                                    }
                                }
                            }
                        }
                    }

                } else if(itemType === "option") {
                    optionList = survey.optionLists[container];
                    for(j = 0; j < optionList.options.length; j++) {
                        if(j !== itemIndex) {
                            otherItem = optionList.options[j];
                            if(!otherItem.deleted && otherItem.value.toLowerCase() === valLower) {
                                hasDuplicateOption = true;
                                break;
                            }
                        }
                    }
                } else if(itemType === "optionlist") {
                    if(container === val) {
                        // no change its valid
                    } else {
                        optionList = survey.optionLists[val];
                        if(typeof optionList !== "undefined") {
                            hasDuplicate = true;
                        }
                    }
                }
                if(hasDuplicate || hasDuplicateOption) {
                    addValidationError(
                        container,
                        itemIndex,
                        "name",
                        "The " + itemDesc + " name is the same as the name of another " + itemDesc +".  Specify a unique name.",
                        itemType,
                        hasDuplicate ? "error" : "warning");
                    isValid = false;
                }

            }

            return isValid;

        }

		/*
		 * Get the path for the form from the top level form down
		 */
        function getFormPath(form) {
            var path = form.name,
                forms = globals.model.survey.forms,
                i;
            while(form.parentFormIndex !== -1) {
                for(i = 0; i < forms.length; i++) {
                    if(i === form.parentFormIndex) {
                        form = forms[i];
                        path = form.name + "/" + path;
                        break;
                    }
                }
            }
            path = "/" + path;
            return path;
        }

		/*
		 * Validate the entire survey
		 */
        function validateAll() {
            var i,
                j,
                forms = globals.model.survey.forms,
                optionLists = globals.model.survey.optionLists,
                list,
                numberErrors = 0,
                numberWarnings = 0;

            globals.errors = [];		// Clear the existing errors
            $('tr.editor_element.error', '#choiceView').removeClass("error");
            $('tr.editor_element.warning', '#choiceView').removeClass("warning");
            $('li.card.error', '#formList').removeClass("error");
            $('li.card.warning', '#formList').removeClass("warning");

            for(i = 0; i < forms.length; i++) {
                var deleted = false;
                var parentForm = forms[i].parentFormIndex;
                var parentQuestion = forms[i].parentQuestionIndex;
                if(parentForm > -1 && parentQuestion > -1) {
                    deleted = forms[parentForm].questions[parentQuestion].deleted;
                }
                if(!deleted) {
                    var validate = true;
                    var groupName;
                    for (j = 0; j < forms[i].questions.length; j++) {
                        // Skip deleted groups
                        if(validate) {
                            if (!forms[i].questions[j].deleted && !forms[i].questions[j].soft_deleted) {
                                validateItem(i, j, "question", false);		// Validate the question
                            } else {
                                if(forms[i].questions[j].type === "begin group") {
                                    groupName = forms[i].questions[j].name;
                                    validate = false;
                                }
                            }
                        } else {
                            if(forms[i].questions[j].type === "end group") {
                                if(forms[i].questions[j].name.indexOf(groupName) > 0) {
                                    validate = true;
                                }
                            }
                        }
                    }
                }
            }
            for(list in optionLists) {
                if(optionLists.hasOwnProperty(list)) {
                    for(j = 0; j < optionLists[list].options.length; j++) {
                        validateItem(list, j, "option", false);		// Validate the option
                    }
                }

                // Validate the option list itself
                validateItem(list, "ol_" + list, "optionlist", false);
            }

        }

        function numberIssues(severity) {
            var i,
                count = 0;

            for(i = 0; i < globals.errors.length; i++) {
                if(globals.errors[i].severity === severity) {
                    count++;
                }
            }
            return count;
        }

        function getSubFormIndex(parentFormIndex, parentQuestionIndex) {

            var survey = globals.model.survey,
                forms = survey.forms,
                i,
                form;

            for(i = 0; i < forms.length; i++) {
                form = forms[i];
                if(forms[i].parentFormIndex === parentFormIndex && forms[i].parentQuestionIndex === parentQuestionIndex) {
                    return i;
                }
            }

            return 0;
        }

        /*
         * Modify controls that are dependent on the view being either for questions or choices
         */
        function updateViewControls() {

            //if(globals.gSelProperty !== "media") {		// media is the only common attribute between question and option view
            //	globals.gSelProperty = "label";
            //}
            if(globals.gIsQuestionView && !globals.gShowingChoices) {
                $('.q_only').show();
                $('.o_only').hide();
                globals.gSelLabel = $('#selProperty > a.q_only.default').text();
            } else {
                $('.q_only').hide();
                $('.o_only').show();
                globals.gSelLabel = $('#selProperty > a.o_only.default').text();
            }
            globals.gSelProperty = "label";
            $('#propSelected').text(globals.gSelLabel);
        }

    });
/**
 * bootbox.js v4.3.0
 *
 * http://bootboxjs.com/license.txt
 */
!function(a,b){"use strict";"function"==typeof define&&define.amd?define('bootbox',["jquery"],b):"object"==typeof exports?module.exports=b(require("jquery")):a.bootbox=b(a.jQuery)}(this,function a(b,c){"use strict";function d(a){var b=q[o.locale];return b?b[a]:q.en[a]}function e(a,c,d){a.stopPropagation(),a.preventDefault();var e=b.isFunction(d)&&d(a)===!1;e||c.modal("hide")}function f(a){var b,c=0;for(b in a)c++;return c}function g(a,c){var d=0;b.each(a,function(a,b){c(a,b,d++)})}function h(a){var c,d;if("object"!=typeof a)throw new Error("Please supply an object of options");if(!a.message)throw new Error("Please specify a message");return a=b.extend({},o,a),a.buttons||(a.buttons={}),a.backdrop=a.backdrop?"static":!1,c=a.buttons,d=f(c),g(c,function(a,e,f){if(b.isFunction(e)&&(e=c[a]={callback:e}),"object"!==b.type(e))throw new Error("button with key "+a+" must be an object");e.label||(e.label=a),e.className||(e.className=2>=d&&f===d-1?"btn-primary":"btn-default")}),a}function i(a,b){var c=a.length,d={};if(1>c||c>2)throw new Error("Invalid argument length");return 2===c||"string"==typeof a[0]?(d[b[0]]=a[0],d[b[1]]=a[1]):d=a[0],d}function j(a,c,d){return b.extend(!0,{},a,i(c,d))}function k(a,b,c,d){var e={className:"bootbox-"+a,buttons:l.apply(null,b)};return m(j(e,d,c),b)}function l(){for(var a={},b=0,c=arguments.length;c>b;b++){var e=arguments[b],f=e.toLowerCase(),g=e.toUpperCase();a[f]={label:d(g)}}return a}function m(a,b){var d={};return g(b,function(a,b){d[b]=!0}),g(a.buttons,function(a){if(d[a]===c)throw new Error("button key "+a+" is not allowed (options are "+b.join("\n")+")")}),a}var n={dialog:"<div class='bootbox modal' tabindex='-1' role='dialog'><div class='modal-dialog'><div class='modal-content'><div class='modal-body'><div class='bootbox-body'></div></div></div></div></div>",header:"<div class='modal-header'><h4 class='modal-title'></h4></div>",footer:"<div class='modal-footer'></div>",closeButton:"<button type='button' class='bootbox-close-button close' data-dismiss='modal' aria-hidden='true'>&times;</button>",form:"<form class='bootbox-form'></form>",inputs:{text:"<input class='bootbox-input bootbox-input-text form-control' autocomplete=off type=text />",textarea:"<textarea class='bootbox-input bootbox-input-textarea form-control'></textarea>",email:"<input class='bootbox-input bootbox-input-email form-control' autocomplete='off' type='email' />",select:"<select class='bootbox-input bootbox-input-select form-control'></select>",checkbox:"<div class='checkbox'><label><input class='bootbox-input bootbox-input-checkbox' type='checkbox' /></label></div>",date:"<input class='bootbox-input bootbox-input-date form-control' autocomplete=off type='date' />",time:"<input class='bootbox-input bootbox-input-time form-control' autocomplete=off type='time' />",number:"<input class='bootbox-input bootbox-input-number form-control' autocomplete=off type='number' />",password:"<input class='bootbox-input bootbox-input-password form-control' autocomplete='off' type='password' />"}},o={locale:"en",backdrop:!0,animate:!0,className:null,closeButton:!0,show:!0,container:"body"},p={};p.alert=function(){var a;if(a=k("alert",["ok"],["message","callback"],arguments),a.callback&&!b.isFunction(a.callback))throw new Error("alert requires callback property to be a function when provided");return a.buttons.ok.callback=a.onEscape=function(){return b.isFunction(a.callback)?a.callback():!0},p.dialog(a)},p.confirm=function(){var a;if(a=k("confirm",["cancel","confirm"],["message","callback"],arguments),a.buttons.cancel.callback=a.onEscape=function(){return a.callback(!1)},a.buttons.confirm.callback=function(){return a.callback(!0)},!b.isFunction(a.callback))throw new Error("confirm requires a callback");return p.dialog(a)},p.prompt=function(){var a,d,e,f,h,i,k;if(f=b(n.form),d={className:"bootbox-prompt",buttons:l("cancel","confirm"),value:"",inputType:"text"},a=m(j(d,arguments,["title","callback"]),["cancel","confirm"]),i=a.show===c?!0:a.show,a.message=f,a.buttons.cancel.callback=a.onEscape=function(){return a.callback(null)},a.buttons.confirm.callback=function(){var c;switch(a.inputType){case"text":case"textarea":case"email":case"select":case"date":case"time":case"number":case"password":c=h.val();break;case"checkbox":var d=h.find("input:checked");c=[],g(d,function(a,d){c.push(b(d).val())})}return a.callback(c)},a.show=!1,!a.title)throw new Error("prompt requires a title");if(!b.isFunction(a.callback))throw new Error("prompt requires a callback");if(!n.inputs[a.inputType])throw new Error("invalid prompt type");switch(h=b(n.inputs[a.inputType]),a.inputType){case"text":case"textarea":case"email":case"date":case"time":case"number":case"password":h.val(a.value);break;case"select":var o={};if(k=a.inputOptions||[],!k.length)throw new Error("prompt with select requires options");g(k,function(a,d){var e=h;if(d.value===c||d.text===c)throw new Error("given options in wrong format");d.group&&(o[d.group]||(o[d.group]=b("<optgroup/>").attr("label",d.group)),e=o[d.group]),e.append("<option value='"+d.value+"'>"+d.text+"</option>")}),g(o,function(a,b){h.append(b)}),h.val(a.value);break;case"checkbox":var q=b.isArray(a.value)?a.value:[a.value];if(k=a.inputOptions||[],!k.length)throw new Error("prompt with checkbox requires options");if(!k[0].value||!k[0].text)throw new Error("given options in wrong format");h=b("<div/>"),g(k,function(c,d){var e=b(n.inputs[a.inputType]);e.find("input").attr("value",d.value),e.find("label").append(d.text),g(q,function(a,b){b===d.value&&e.find("input").prop("checked",!0)}),h.append(e)})}return a.placeholder&&h.attr("placeholder",a.placeholder),a.pattern&&h.attr("pattern",a.pattern),f.append(h),f.on("submit",function(a){a.preventDefault(),a.stopPropagation(),e.find(".btn-primary").click()}),e=p.dialog(a),e.off("shown.bs.modal"),e.on("shown.bs.modal",function(){h.focus()}),i===!0&&e.modal("show"),e},p.dialog=function(a){a=h(a);var c=b(n.dialog),d=c.find(".modal-dialog"),f=c.find(".modal-body"),i=a.buttons,j="",k={onEscape:a.onEscape};if(g(i,function(a,b){j+="<button data-bb-handler='"+a+"' type='button' class='btn "+b.className+"'>"+b.label+"</button>",k[a]=b.callback}),f.find(".bootbox-body").html(a.message),a.animate===!0&&c.addClass("fade"),a.className&&c.addClass(a.className),"large"===a.size&&d.addClass("modal-lg"),"small"===a.size&&d.addClass("modal-sm"),a.title&&f.before(n.header),a.closeButton){var l=b(n.closeButton);a.title?c.find(".modal-header").prepend(l):l.css("margin-top","-10px").prependTo(f)}return a.title&&c.find(".modal-title").html(a.title),j.length&&(f.after(n.footer),c.find(".modal-footer").html(j)),c.on("hidden.bs.modal",function(a){a.target===this&&c.remove()}),c.on("shown.bs.modal",function(){c.find(".btn-primary:first").focus()}),c.on("escape.close.bb",function(a){k.onEscape&&e(a,c,k.onEscape)}),c.on("click",".modal-footer button",function(a){var d=b(this).data("bb-handler");e(a,c,k[d])}),c.on("click",".bootbox-close-button",function(a){e(a,c,k.onEscape)}),c.on("keyup",function(a){27===a.which&&c.trigger("escape.close.bb")}),b(a.container).append(c),c.modal({backdrop:a.backdrop,keyboard:!1,show:!1}),a.show&&c.modal("show"),c},p.setDefaults=function(){var a={};2===arguments.length?a[arguments[0]]=arguments[1]:a=arguments[0],b.extend(o,a)},p.hideAll=function(){return b(".bootbox").modal("hide"),p};var q={br:{OK:"OK",CANCEL:"Cancelar",CONFIRM:"Sim"},cs:{OK:"OK",CANCEL:"Zrušit",CONFIRM:"Potvrdit"},da:{OK:"OK",CANCEL:"Annuller",CONFIRM:"Accepter"},de:{OK:"OK",CANCEL:"Abbrechen",CONFIRM:"Akzeptieren"},el:{OK:"Εντάξει",CANCEL:"Ακύρωση",CONFIRM:"Επιβεβαίωση"},en:{OK:"OK",CANCEL:"Cancel",CONFIRM:"OK"},es:{OK:"OK",CANCEL:"Cancelar",CONFIRM:"Aceptar"},et:{OK:"OK",CANCEL:"Katkesta",CONFIRM:"OK"},fi:{OK:"OK",CANCEL:"Peruuta",CONFIRM:"OK"},fr:{OK:"OK",CANCEL:"Annuler",CONFIRM:"D'accord"},he:{OK:"אישור",CANCEL:"ביטול",CONFIRM:"אישור"},id:{OK:"OK",CANCEL:"Batal",CONFIRM:"OK"},it:{OK:"OK",CANCEL:"Annulla",CONFIRM:"Conferma"},ja:{OK:"OK",CANCEL:"キャンセル",CONFIRM:"確認"},lt:{OK:"Gerai",CANCEL:"Atšaukti",CONFIRM:"Patvirtinti"},lv:{OK:"Labi",CANCEL:"Atcelt",CONFIRM:"Apstiprināt"},nl:{OK:"OK",CANCEL:"Annuleren",CONFIRM:"Accepteren"},no:{OK:"OK",CANCEL:"Avbryt",CONFIRM:"OK"},pl:{OK:"OK",CANCEL:"Anuluj",CONFIRM:"Potwierdź"},pt:{OK:"OK",CANCEL:"Cancelar",CONFIRM:"Confirmar"},ru:{OK:"OK",CANCEL:"Отмена",CONFIRM:"Применить"},sv:{OK:"OK",CANCEL:"Avbryt",CONFIRM:"OK"},tr:{OK:"Tamam",CANCEL:"İptal",CONFIRM:"Onayla"},zh_CN:{OK:"OK",CANCEL:"取消",CONFIRM:"确认"},zh_TW:{OK:"OK",CANCEL:"取消",CONFIRM:"確認"}};return p.init=function(c){return a(c||b)},p});
/*
This file is part of SMAP.

SMAP is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

SMAP is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with SMAP.  If not, see <http://www.gnu.org/licenses/>.

*/

/*
 * Functions for using aws services
 *
 */



var gLanguages;

define('app/aws',[
		'jquery',
		'modernizr',
		'app/localise',
		'app/globals'],
	function($, modernizr, lang, globals) {

		return {
			setLanguageSelect: setLanguageSelect
		};

		function setLanguageSelect ($elem, type, callback) {

			if(gLanguages) {
					updateSelection($elem, type, callback);
			} else {
					getLanguages($elem, type, callback);
			}
		}

		function updateSelection($elem, type, callback) {
			var h = [],
				idx = -1,
				i;

			for(i = 0; i < gLanguages.length; i++) {
				if(gLanguages[i][type]) {
					h[++idx] = '<option value="';
					h[++idx] = gLanguages[i].code;
					h[++idx] = '">';
					h[++idx] = htmlEncode(gLanguages[i].name);
					h[++idx] = '</option>';
				}
			}
			$elem.empty().html(h.join(''));
			if(typeof callback === "function") {
				callback();
			}
		}

		function getLanguages($elem, type, callback) {
			addHourglass();
			$.ajax({
				url: "/surveyKPI/language_codes",
				dataType: 'json',
				cache: false,
				success: function(data) {
					removeHourglass();
					var $e = $elem;
					var t = type;
					var cb = callback;
					gLanguages = data;
					updateSelection($e, t, cb);
				},
				error: function(xhr, textStatus, err) {
					removeHourglass();
					if(xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert(localise.set["c_error"] + ": " + err);
					}
				}
			});
		}
	});
/*!
	Autosize v1.18.9 - 2014-05-27
	Automatically adjust textarea height based on user input.
	(c) 2014 Jack Moore - http://www.jacklmoore.com/autosize
	license: http://www.opensource.org/licenses/mit-license.php
*/
(function(e){var t,o={className:"autosizejs",id:"autosizejs",append:"\n",callback:!1,resizeDelay:10,placeholder:!0},i='<textarea tabindex="-1" style="position:absolute; top:-999px; left:0; right:auto; bottom:auto; border:0; padding: 0; -moz-box-sizing:content-box; -webkit-box-sizing:content-box; box-sizing:content-box; word-wrap:break-word; height:0 !important; min-height:0 !important; overflow:hidden; transition:none; -webkit-transition:none; -moz-transition:none;"/>',n=["fontFamily","fontSize","fontWeight","fontStyle","letterSpacing","textTransform","wordSpacing","textIndent"],s=e(i).data("autosize",!0)[0];s.style.lineHeight="99px","99px"===e(s).css("lineHeight")&&n.push("lineHeight"),s.style.lineHeight="",e.fn.autosize=function(i){return this.length?(i=e.extend({},o,i||{}),s.parentNode!==document.body&&e(document.body).append(s),this.each(function(){function o(){var t,o=window.getComputedStyle?window.getComputedStyle(u,null):!1;o?(t=u.getBoundingClientRect().width,(0===t||"number"!=typeof t)&&(t=parseInt(o.width,10)),e.each(["paddingLeft","paddingRight","borderLeftWidth","borderRightWidth"],function(e,i){t-=parseInt(o[i],10)})):t=p.width(),s.style.width=Math.max(t,0)+"px"}function a(){var a={};if(t=u,s.className=i.className,s.id=i.id,d=parseInt(p.css("maxHeight"),10),e.each(n,function(e,t){a[t]=p.css(t)}),e(s).css(a).attr("wrap",p.attr("wrap")),o(),window.chrome){var r=u.style.width;u.style.width="0px",u.offsetWidth,u.style.width=r}}function r(){var e,n;t!==u?a():o(),s.value=!u.value&&i.placeholder?(p.attr("placeholder")||"")+i.append:u.value+i.append,s.style.overflowY=u.style.overflowY,n=parseInt(u.style.height,10),s.scrollTop=0,s.scrollTop=9e4,e=s.scrollTop,d&&e>d?(u.style.overflowY="scroll",e=d):(u.style.overflowY="hidden",c>e&&(e=c)),e+=w,n!==e&&(u.style.height=e+"px",f&&i.callback.call(u,u))}function l(){clearTimeout(h),h=setTimeout(function(){var e=p.width();e!==g&&(g=e,r())},parseInt(i.resizeDelay,10))}var d,c,h,u=this,p=e(u),w=0,f=e.isFunction(i.callback),z={height:u.style.height,overflow:u.style.overflow,overflowY:u.style.overflowY,wordWrap:u.style.wordWrap,resize:u.style.resize},g=p.width(),y=p.css("resize");p.data("autosize")||(p.data("autosize",!0),("border-box"===p.css("box-sizing")||"border-box"===p.css("-moz-box-sizing")||"border-box"===p.css("-webkit-box-sizing"))&&(w=p.outerHeight()-p.height()),c=Math.max(parseInt(p.css("minHeight"),10)-w||0,p.height()),p.css({overflow:"hidden",overflowY:"hidden",wordWrap:"break-word"}),"vertical"===y?p.css("resize","none"):"both"===y&&p.css("resize","horizontal"),"onpropertychange"in u?"oninput"in u?p.on("input.autosize keyup.autosize",r):p.on("propertychange.autosize",function(){"value"===event.propertyName&&r()}):p.on("input.autosize",r),i.resizeDelay!==!1&&e(window).on("resize.autosize",l),p.on("autosize.resize",r),p.on("autosize.resizeIncludeStyle",function(){t=null,r()}),p.on("autosize.destroy",function(){t=null,clearTimeout(h),e(window).off("resize",l),p.off("autosize").off(".autosize").css(z).removeData("autosize")}),r())})):this}})(window.jQuery||window.$);
define("jquery.autosize.min", function(){});

/*
This file is part of SMAP.

SMAP is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

SMAP is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with SMAP.  If not, see <http://www.gnu.org/licenses/>.

*/

var gUserLocale = navigator.language;
if (Modernizr.localstorage) {
	gUserLocale = localStorage.getItem('user_locale') || navigator.language;
} 


require.config({
    baseUrl: 'js/libs',
    waitSeconds: 0,
    locale: gUserLocale,
    paths: {
    	app: '../app',
    	bootbox: 'bootbox.min',
    	lang_location: '..'
    },
    shim: {
    	'app/common': ['jquery'],
        'jquery.autosize.min': ['jquery']
    }
});

require([
         'jquery',
         'app/common',
         'modernizr',
         'app/localise',
         'app/ssc',
         'app/globals',
         'app/changeset',
         'bootbox',
         'app/aws',
         'jquery.autosize.min'], 
		function($, common, modernizr, lang, ssc, globals, changeset, bootbox, aws) {


var	gMode = "survey",
	gTempLanguageItems = [];

$(document).ready(function() {

	setCustomEdit();
	setTheme();
	setupUserProfile(true);
	localise.setlang();		// Localise HTML
	
	// Get the user details
	getLoggedInUser(surveyListDone, false, true, undefined, false, false);
	
	// Add menu functions
	$('#m_open').off().click(function() {	// Open an existing form
		openForm("existing");
	});
	
	$('.save_form').off().click(function() {	// Save a survey to Smap
		saveTranslations(getSurveyDetails(undefined, false, true));
	});
	
	$('#m_settings').off().click(function() {	// Get a survey from Smap
		gMode = "settings";
		refreshView(gMode);
	});
	$('#m_changes').off().click(function() {	// View the changes to this survey
		gMode = "changes";
		refreshView(gMode);
	});
	$('#m_undo').off().click(function() {	// Undo last change
		globals.model.undo();
		refreshView(gMode);
	});
	$('#m_redo').off().click(function() {	// Redo last change
		globals.model.redo();
		refreshView(gMode);
	});

	$('#m_auto_translate').click(function(e) {
		e.preventDefault();
		if(!$(this).parent().hasClass("disabled")) {
			var survey = globals.model.survey;
			if (survey.languages.length > 1) {
				aws.setLanguageSelect($('.translate_select'), 'translate', setTranslateValues);
				$('#overwrite').prop("checked", false);
				$('#autoTranslateModal').modal("show");
			}
		} else {
			alert(localise.set["ed_sct"]);
		}
	});
	
	$('.language_list').off().change(function() {
		globals.gLanguage1 = $('#language1').val();
		globals.gLanguage2 = $('#language2').val();
		refreshView(gMode);
 	 });
	

	// Check for selection of the label indicating successful updates and the one indicating failed
	$('#successLabel').off().click(function() {
		alert("success");
	});
	// Check for selection of the label indicating successful updates and the one indicating failed
	$('#failedLabel').off().click(function() {
		alert("failed");
	});
	
    /*
     * Open a new form
     */
	$('#get_form').off().click(function() {
		globals.gCurrentSurvey = $('#survey_name option:selected').val();
		saveCurrentProject(globals.gCurrentProject, globals.gCurrentSurvey);	// Save the current survey id
		getSurveyDetails(refreshView, false, true);
 	 });

	$('#translateGo').off().click(function() {
		if(!$(this).hasClass('disabled')) {
			autoTranslate();
		}
	});

	/*
 	 * Add check prior to the user leaving the screen
 	 */
	window.onbeforeunload = function() {
		if(globals.changes.length > 0) {
			return localise.set["msg_leave"];
		} else {
			return;
		}
	};

});

function surveyListDone() {
	getSurveyDetails(refreshView, false, true);
}


function setTranslateValues() {
	var survey = globals.model.survey;
	if (survey.languages[globals.gLanguage1].code) {
		$('#from_lang').val(survey.languages[globals.gLanguage1].code);
	} else {
		$('#from_lang').val("en");
	}
	if (survey.languages[globals.gLanguage2].code) {
		$('#to_lang').val(survey.languages[globals.gLanguage2].code);
	}  else {
		$('#to_lang').val("en");
	}
}

// Save the survey
function saveTranslations(callback) {
	
	var url="/surveyKPI/surveys/save/" + globals.gCurrentSurvey,
		changes = globals.model.translateChanges,
		changesString = JSON.stringify(changes);		
	
	addHourglass();
	$.ajax({
		url: url,
		type: 'PUT',
		dataType: 'json',
		cache: false,
		data: { changes: changesString },
		success: function(data) {
			var responseFn = callback,
				h = [],
				idx = -1,
				i;
			
			removeHourglass();			
			
			globals.model.clearChanges();
			
			if(typeof responseFn === "function") { 
				responseFn();
			}
			
			// Report success and failure
			globals.model.lastChanges = data.changeSet;
			//$('#successLabel .counter').html(data.success);
			//$('#failedLabel .counter').html(data.failed);	
			
			if(data.success > 0) {
				h[++idx] = '<div class="alert alert-success" role="alert">';
				h[++idx] = '<p>';
				h[++idx] = data.success;
				h[++idx] = " " + localise.set["ed_csa"];
				h[++idx] = '</p>';
				h[++idx] = '<ol>';
				for(i = 0; i < data.changeSet.length; i++) {
					h[++idx] = htmlEncode(changeset.addUpdateMessage(data.changeSet[i], false));
				}
				h[++idx] = '</ol>';
				h[++idx] = '</div>';
			}
			if(data.failed > 0) {
				h[++idx] = '<div class="alert alert-danger" role="alert">';
				h[++idx] = data.failed;
				h[++idx] = " " + localise.set["ed_csf"];
				h[++idx] = '<ol>';
				for(i = 0; i < data.changeSet.length; i++) {
					h[++idx] = htmlEncode(changeset.addUpdateMessage(data.changeSet[i], true));
				}
				h[++idx] = '</ol>';
				h[++idx] = '</div>';
			}

			bootbox.alert(h.join(""));

		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			
			if(xhr.readyState === 0 || xhr.status === 0) {
	              return;  // Not an error
			} else {
				alert(localise.set["msg_err_save"] + ' ' + err);
			}
					
			if(typeof responseFn === "function") { 
				responseFn();
			}
		}
	});	
	
};

function refreshView() {
	
	var i,
		j,
		itemList = [],
		index = -1,
		survey = globals.model.survey,
		numberLanguages,
		key,
		options = [];

	gTempLanguageItems = [];
	
	if(survey) {
		numberLanguages = survey.languages.length;
	}
	
	if(globals.gLanguage1 >= numberLanguages) {
		globals.gLanguage1 = 0;
	}
	if(globals.gLanguage2 >= numberLanguages) {
		globals.gLanguage2 = 0;
	}
	
	// Set the display name
	$('.formName').text(survey.displayName);
			
	// Add all unique questions from all forms
	for(i = 0; i < survey.forms.length; i++) {
		console.log("Form name: " + survey.forms[i].name);
		var formQuestions = survey.forms[i].questions; 
		for(j = 0; j < formQuestions.length; j++) {

			// Question Labels
			if(formQuestions[j].labels[globals.gLanguage1].text) {
				if((index = $.inArray(formQuestions[j].labels[globals.gLanguage1].text, itemList)) > -1) {
					gTempLanguageItems[index].indexes.push({
						form: i,
						question: j
					});
				} else {
					itemList.push(formQuestions[j].labels[globals.gLanguage1].text);
					gTempLanguageItems.push({
						label_a: formQuestions[j].labels[globals.gLanguage1].text,
						label_b: formQuestions[j].labels[globals.gLanguage2].text,
						indexes: [{
							form: i,
							question: j
						}]
					});
				}
			}

			// Hints
			if(formQuestions[j].labels[globals.gLanguage1].hint) {
				if((index = $.inArray(formQuestions[j].labels[globals.gLanguage1].hint, itemList)) > -1) {
					gTempLanguageItems[index].indexes.push({
						form: i,
						question: j,
						hint: true
					});
				} else {
					itemList.push(formQuestions[j].labels[globals.gLanguage1].hint);
					gTempLanguageItems.push({
						label_a: formQuestions[j].labels[globals.gLanguage1].hint,
						label_b: formQuestions[j].labels[globals.gLanguage2].hint,
						indexes: [{
							form: i,
							question: j,
							hint: true
						}]
					});
				}
			}

			// Constraint Messages
			if(formQuestions[j].labels[globals.gLanguage1].constraint_msg) {
				if((index = $.inArray(formQuestions[j].labels[globals.gLanguage1].constraint_msg, itemList)) > -1) {
					gTempLanguageItems[index].indexes.push({
						form: i,
						question: j,
						constraint_msg: true
					});
				} else {
					itemList.push(formQuestions[j].labels[globals.gLanguage1].constraint_msg);
					gTempLanguageItems.push({
						label_a: formQuestions[j].labels[globals.gLanguage1].constraint_msg,
						label_b: formQuestions[j].labels[globals.gLanguage2].constraint_msg,
						indexes: [{
							form: i,
							question: j,
							constraint_msg: true
						}]
					});
				}
			}

			// Required Messages
			if(formQuestions[j].labels[globals.gLanguage1].required_msg) {
				if((index = $.inArray(formQuestions[j].labels[globals.gLanguage1].required_msg, itemList)) > -1) {
					gTempLanguageItems[index].indexes.push({
						form: i,
						question: j,
						required_msg: true
					});
				} else {
					itemList.push(formQuestions[j].labels[globals.gLanguage1].required_msg);
					gTempLanguageItems.push({
						label_a: formQuestions[j].labels[globals.gLanguage1].required_msg,
						label_b: formQuestions[j].labels[globals.gLanguage2].required_msg,
						indexes: [{
							form: i,
							question: j,
							required_msg: true
						}]
					});
				}
			}

			// Guidance Hint
			if(formQuestions[j].labels[globals.gLanguage1].guidance_hint) {
				if((index = $.inArray(formQuestions[j].labels[globals.gLanguage1].guidance_hint, itemList)) > -1) {
					gTempLanguageItems[index].indexes.push({
						form: i,
						question: j,
						guidance_hint: true
					});
				} else {
					itemList.push(formQuestions[j].labels[globals.gLanguage1].guidance_hint);
					gTempLanguageItems.push({
						label_a: formQuestions[j].labels[globals.gLanguage1].guidance_hint,
						label_b: formQuestions[j].labels[globals.gLanguage2].guidance_hint,
						indexes: [{
							form: i,
							question: j,
							guidance_hint: true
						}]
					});
				}
			}
		}
	}
	// Add all unique options from all option lists
	for(key in survey.optionLists) {

		options = survey.optionLists[key].options;
		for(j = 0; j < options.length; j++) {

			if(options[j].labels[globals.gLanguage1].text) {
				if((index = $.inArray(options[j].labels[globals.gLanguage1].text, itemList)) > -1) {
					console.log(options[j].labels[globals.gLanguage1].text);
					gTempLanguageItems[index].indexes.push({
						optionList: key,
						option: j
					});
					
				} else {
					itemList.push(options[j].labels[globals.gLanguage1].text);
					gTempLanguageItems.push({
						label_a: options[j].labels[globals.gLanguage1].text,
						label_b: options[j].labels[globals.gLanguage2].text,
						indexes: [{
							optionList: key,
							option: j
						}]
					});
				}
			}
		}
	}

	// Add the HTML
	setTranslateHtml($('.questions'), gTempLanguageItems, survey);

	// Respond to text changes
	$(".lang_b").first().focus();
	$(".lang_b").change(function(e){
		e.preventDefault();
		var $this = $(this);
		var index = $this.data("index");
		var newVal = $this.val();
		console.log(gTempLanguageItems[index]);
		console.log("New val:" + newVal);
		globals.model.modLabel(globals.gLanguage2, gTempLanguageItems[index].indexes, newVal, "text", "label");
	});
}

/*
 * Convert JSON to html
 */
function setTranslateHtml($element, language_items, survey) {
	var i;

	$element.empty();
	for(i = 0; i < language_items.length; i++) {
		var label_a = language_items[i].label_a;
		var label_b = language_items[i].label_b || "";
		var tabidx = i + 1;
		var content = `
			<div class="col-6">
				<textarea class="lang_a" tabindex="-1" readonly>${label_a}</textarea>
			</div>
			<div class="col-6">
				<textarea class="lang_b" tabindex="${tabidx}" data-index="${i}">${label_b}</textarea>
			<div>`;
		$element.append(content);
	}

}

/*
 * Call AWS services to translate automatically
 */
function autoTranslate() {

	var url="/surveyKPI/surveys/translate/" + globals.gCurrentSurvey
		+ "/" + globals.gLanguage1
		+ "/" + globals.gLanguage2
		+ "/" + $("#from_lang").val()
		+ "/" + $('#to_lang').val();
	var overwrite = $('#overwrite').prop("checked");

	if(overwrite) {
		url += "?overwrite=true";
	}

	$('#translateGo').addClass("disabled");

	addHourglass();
	$.ajax({
		url: url,
		type: 'PUT',
		cache: false,
		timeout: 0,
		success: function() {
			removeHourglass();
			$('#autoTranslateModal').modal("hide");
			$('#translateGo').removeClass("disabled");
			getSurveyDetails(refreshView, false, true);

			var h = [],
				idx = -1,
				msg = localise.set["ed_transd"];

			msg = msg.replace("%s1", globals.model.survey.languages[globals.gLanguage1].name);
			msg = msg.replace("%s2", globals.model.survey.languages[globals.gLanguage2].name);

			h[++idx] = '<div class="alert alert-success" role="alert">';
			h[++idx] = '<p>';
			h[++idx] = msg;
			h[++idx] = '</p>';

			h[++idx] = '</div>';
			bootbox.alert(h.join(""));

		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			$('#translateGo').removeClass("disabled");

			if(xhr.readyState === 0 || xhr.status === 0) {
				return;  // Not an error
			} else {
				if(xhr.responseText.indexOf("<html>") > 0) {
					alert(localise.set["msg_trans_to"]);
				} else {
					alert(xhr.responseText);
				}

			}
		}
	});
}

});
define("../translate", function(){});

