/*
This file is part of SMAP.

SMAP is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

SMAP is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with SMAP.  If not, see <http://www.gnu.org/licenses/>.

*/

var gWait = 0;		// This javascript file only
var gCache = {};
var gCacheGroup = {};
var gCacheStatusQuestions = {};
var gCacheKeys = {};
var gEligibleUser;
var gSelectedOversightQuestion;
var gSelectedOversightSurvey;
var gConversationalSMS;	// Set true if a conversational SMS choice has been added to notification types

/*
 * Convert a choice list name into a valid jquery class name
 */
function jq(choiceList) {

	var c;

	c = choiceList.replace( /(:|\.|\[|\]|,)/g, "\\$1" );
	return c;
}

/* 
 * ==============================================================
 * Task Functions
 * ==============================================================
 */

function addPendingTask(taskId, assignmentId, status, source) {
	var i,
		duplicate = false,
		assignment;

	assignment = {
		assignment_id: assignmentId,
		assignment_status: status,
		task_id: taskId
	};
	globals.gPendingUpdates.push(assignment);

	if(source === "table") {
		updateMapTaskSelections(taskId, true);
	} else if(source === "map") {
		$('#tasks_table').find('[data-taskid=' + taskId + ']').prop("checked", true).closest('tr').addClass("info");
	}
}

function removePendingTask(taskId, source) {
	var i;
	for (i = 0; i < globals.gPendingUpdates.length; i++) {
		if(globals.gPendingUpdates[i].task_id === taskId) {
			globals.gPendingUpdates.splice(i,1);
			break;
		}
	}
	if(source === "table") {
		updateMapTaskSelections(taskId, false);
	} else if(source === "map") {
		$('#tasks_table').find('[data-taskid=' + taskId + ']').prop("checked", false).closest('tr').removeClass("info");
	}
}

/*
 * ===============================================================
 * Project Functions
 * ===============================================================
 */

/*
 * Update the list of available projects
 * Note when addAll is set to true the list is not used to change the default project
 *   In this case the value of the list should not be set to the default project
 */
function updateProjectList(addAll, projectId, callback, $projectSelect) {

	var i,
		h = [],
		idx = -1,
		updateCurrentProject;

	if(projectId > 0) {
		updateCurrentProject = true;		// Only save the current project if there it is set
	}

	if(addAll) {
		h[++idx] = '<option value="0">' + localise.set["c_all"] + '</option>';
		updateCurrentProject = false;
	}
	for(i = 0; i < globals.gProjectList.length; i++) {
		h[++idx] = '<option value="';
		h[++idx] = globals.gProjectList[i].id;
		h[++idx] = '">';
		h[++idx] = htmlEncode(globals.gProjectList[i].name);
		h[++idx] = '</option>';

		if(globals.gProjectList[i].id === projectId) {
			updateCurrentProject = false;   // Don't save the current project if it is already in the list
		}
	}
	$projectSelect.empty().append(h.join(''));

	// If for some reason the user's default project is no longer available then
	//  set the default project to the first project in the list
	//  if the list is empty then set the default project to undefined
	if(updateCurrentProject) {
		if (globals.gProjectList[0]) {
			globals.gCurrentProject = globals.gProjectList[0].id;		// Update the current project id
			globals.gCurrentSurvey = -1;
			globals.gCurrentTaskGroup = undefined;
		} else {
			globals.gCurrentProject = -1;		// Update the current project id
			globals.gCurrentSurvey = -1;
			globals.gCurrentTaskGroup = undefined;
		}

		saveCurrentProject(globals.gCurrentProject,
			globals.gCurrentSurvey,
			globals.gCurrentTaskGroup);
	}

	if(!addAll) {
		$projectSelect.val(globals.gCurrentProject);			// Set the initial project value
		$('#projectId').val(globals.gCurrentProject);			// Set the project value for the hidden field in template upload
	}

	if(typeof callback !== "undefined") {
		callback(globals.gCurrentProject);				// Call the callback with the correct current project
	}
}

/*
 * Get the list of available projects from the server
 */
function getMyProjects(projectId, callback, getAll) {
	addHourglass();
	$.ajax({
		url: "/surveyKPI/myProjectList",
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				globals.gProjectList = data;
				updateProjectList(getAll, projectId, callback, $('.project_list'));
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert("Error: Failed to get list of projects: " + err);
				}
			}
		}
	});
}

/*
 * Save the current project id in the user defaults
 */
function saveCurrentProject(projectId, surveyId, taskGroupId) {

	if(surveyId > 0 || projectId > 0 || taskGroupId > 0) {

		var user = {
			current_project_id: projectId,
			current_survey_id: surveyId,
			current_task_group_id: taskGroupId
		};

		var userString = JSON.stringify(user);

		addHourglass();
		$.ajax({
			type: "POST",
			contentType: "application/json",		// uses application/json
			url: "/surveyKPI/user/currentproject",
			cache: false,
			data: userString,
			success: function(data, status) {
				// Do not process a logout
				removeHourglass();
			}, error: function(data, status) {
				// Do not process a logout
				removeHourglass();
			}
		});
	}
}

/*
 * Save the current relationship between survey and surveyGroup
 */
function saveCurrentGroupSurvey(surveyId, gs, fName) {

	if (surveyId > 0) {

		var groupSurvey = {
			sId: surveyId,
			groupIdent: gs,
			fName: fName
		};

		addHourglass();
		$.ajax({
			type: "POST",
			contentType: "application/x-www-form-urlencoded",
			url: "/surveyKPI/user/groupsurvey",
			cache: false,
			data: JSON.stringify(groupSurvey),
			success: function (data, status) {
				removeHourglass();
				handleLogout(data);
			}, error: function (data, status) {
				removeHourglass();
			}
		});
	}
}

/*
 * ===============================================================
 * User Functions
 * ===============================================================
 */

/*
 * Add user details popup to the page
 * Legacy only used with non bootstrap pages - these should be replaced with bootstrap
 */
function addUserDetailsPopup() {
	var
		h =[],
		idx = -1;


	h[++idx] = '<div id="modify_me_popup" style="display:none;">';
	h[++idx] = '<div class="left_panel">';
	h[++idx] = '<form id="me_edit_form">';
	h[++idx] = '<label for="me_name">';
	h[++idx] = localise.set["c_name"];
	h[++idx] = '</label>';
	h[++idx] = '<input type="text" id="me_name" required><br/>';

	h[++idx] = '<label for="me_language">';
	h[++idx] = localise.set["c_lang"];
	h[++idx] = '</label>';
	h[++idx] = '<select class="language_select" id="me_language"></select><br/>';

	h[++idx] = '<label for="me_email">';
	h[++idx] = localise.set["c_email"];
	h[++idx] = '</label>';
	h[++idx] = '<input type="text" id="me_email" pattern="^[_A-Za-z0-9-\\+]+(\\.[_A-Za-z0-9-]+)*@[A-Za-z0-9-]+(\\.[A-Za-z0-9]+)*(\\.[A-Za-z]{2,})$"><br/>';

	h[++idx] = '<label for="me_organisation">';
	h[++idx] = localise.set["c_org"];
	h[++idx] = '</label>';
	h[++idx] = '<select class="organisation_select" id="me_organisation"></select><br/>';

	h[++idx] = '<label for="me_enterprise">';
	h[++idx] = localise.set["c_ent"];
	h[++idx] = '</label>';
	h[++idx] = '<div id="me_enterprise"></div><br/>';

	h[++idx] = '<label for="u_tz">';
	h[++idx] = localise.set["c_tz"];
	h[++idx] = '</label>';
	h[++idx] = '<select class="timezone_select" id="u_tz"></select>';

	h[++idx] = '</form>';
	h[++idx] = '</div>';
	h[++idx] = '</div>';

	$(document.body).append(h.join(''));

}

/*
 * Populate a language select widget
 */
function populateLanguageSelect(sId, $elem) {
	$.getJSON("/surveyKPI/languages/" + sId, function(data) {

		if(handleLogout(data)) {
			$elem.empty();
			$.each(data, function (j, item) {
				$elem.append('<option value="' + item + '">' + htmlEncode(item) + '</option>');
			});
		}
	});
}

/*
 * Populate a pdf select widget
 * Set the template set as the default to be selected
 * If there is no default template and there is a template specified in settings (legacy) then set that as the default
 */
function populatePdfSelect(sId, $elem) {
	var url = "/surveyKPI/surveys/templates/" + sId;
	url += addCacheBuster(url);

	$.getJSON(url, function(data) {

		if(handleLogout(data)) {
			var defaultTemplateId,
				fromSettingsTemplateId;

			$elem.empty();
			$elem.append('<option value="-2">' + localise.set["c_auto"] + '</option>');
			$elem.append('<option value="-1">' + localise.set["c_none"] + '</option>');
			$.each(data, function (j, item) {
				if (item.default_template) {
					defaultTemplateId = item.id;
				} else if (item.fromSettings) {
					fromSettingsTemplateId = item.id;
				}
				$elem.append('<option value="' + item.id + '">' + htmlEncode(item.name) + '</option>');
			});
			if (typeof defaultTemplateId !== "undefined") {
				$elem.val(defaultTemplateId);
			} else if (typeof fromSettingsTemplateId !== "undefined") {
				$elem.val(fromSettingsTemplateId)
			} else {
				$elem.val(-2);		// Set to auto
			}
		}

	});
}

/*
 * Add user details popup to the page
 */
function addUserDetailsPopupBootstrap4() {
	var	h =[],
		idx = -1;

	h[++idx] = '<div id="modify_me_popup" class="modal fade" tabindex="-1" role="dialog" aria-labelledby="modifyMeLabel" aria-hidden="true">';
	h[++idx] = '<div class="modal-dialog modal-lg">';
	h[++idx] = '<div class="modal-content">';
	h[++idx] = '<div class="modal-header">';
	h[++idx] = '<h4 class="modal-title" id="modifyMeLabel"></h4>';
	h[++idx] = '<button type="button" class="close" data-dismiss="modal" aria-label="Close"><span aria-hidden="true">&times;</span></button>';
	h[++idx] = '</div>';    // modal-headers

	h[++idx] = '<div class="modal-body">';
	h[++idx] = '<form role="form" id="me_edit_form">';
	h[++idx] = '<div class="form-group row">';
	h[++idx] = '<label for="me_name" class="col-sm-2 control-label">';
	h[++idx] = localise.set["c_name"];
	h[++idx] = '</label>';
	h[++idx] = '<div class="col-sm-10">';
	h[++idx] = '<input type="text" id="me_name" required class="form-control">';
	h[++idx] = '</div>';
	h[++idx] = '</div>';

	h[++idx] = '<div class="form-group row">';
	h[++idx] = '<label for="me_language" class="col-sm-2 control-label">';
	h[++idx] = localise.set["c_lang"];
	h[++idx] = '</label>';
	h[++idx] = '<div class="col-sm-10">';
	h[++idx] = '<select id="me_language" class="language_select form-control"></select>';
	h[++idx] = '</div>';
	h[++idx] = '</div>';

	h[++idx] = '<div class="form-group row">';
	h[++idx] = '<label for="me_email" class="col-sm-2 control-label">';
	h[++idx] = localise.set["c_email"];
	h[++idx] = '</label>';
	h[++idx] = '<div class="col-sm-10">';
	h[++idx] = '<input type="email" class="form-control" id="me_email"';
	h[++idx] = ' placeholder="Enter email"';
	h[++idx] = ' pattern="^[_A-Za-z0-9-\\+]+(\\.[_A-Za-z0-9-]+)*@[A-Za-z0-9-]+(\\.[A-Za-z0-9]+)*(\\.[A-Za-z]{2,})$">';
	h[++idx] = '</div>';
	h[++idx] = '</div>';

	h[++idx] = '<div class="form-group row">';
	h[++idx] = '<label for="me_organisation" class="col-sm-2 control-label">';
	h[++idx] = localise.set["c_org"];
	h[++idx] = '</label>';
	h[++idx] = '<div class="col-sm-10">';
	h[++idx] = '<select id="me_organisation" class="organisation_select form-control"></select>';
	h[++idx] = '</div>';
	h[++idx] = '</div>';

	h[++idx] = '<div class="form-group row">';
	h[++idx] = '<label for="me_enterprise" class="col-sm-2 control-label">';
	h[++idx] = localise.set["c_ent"];
	h[++idx] = '</label>';
	h[++idx] = '<div class="col-sm-10">';
	h[++idx] = '<div id="me_enterprise" class="form-control"></div>';
	h[++idx] = '</div>';
	h[++idx] = '</div>';

	h[++idx] = '<div class="form-group row">';
	h[++idx] = '<label for="u_tz" class="col-sm-2 control-label">';
	h[++idx] = localise.set["c_tz"];
	h[++idx] = '</label>';
	h[++idx] = '<div class="col-sm-10">';
	h[++idx] = '<select class="form-control timezone_select" id="u_tz"></select>';
	h[++idx] = '</div>';
	h[++idx] = '</div>';
	
	h[++idx] = '<div id="me_alert" class="alert d-none text-wrap text-break" role="alert"></div>';
	h[++idx] = '</form>';
	h[++idx] = '</div>';    // modal body

	h[++idx] = '<div class="modal-footer">';
	h[++idx] = '<button type="button" class="btn btn-default" data-dismiss="modal">';
	h[++idx] = localise.set["c_close"];
	h[++idx] = '</button>';

	h[++idx] = '<button id="userProfileSave" type="button" class="btn btn-primary">';
	h[++idx] = localise.set["c_save"];
	h[++idx] = '</button>';
	h[++idx] = '</div>';    // modal - footer
	h[++idx] = '</div>';        // modal - content
	h[++idx] = '</div>';            // modal - dialog
	h[++idx] = '</div>';                // popup

	$(document.body).append(h.join(''));

	enableUserProfileBS();
}

/*
 * Add user details popup to the page
 */
function addApiKeyPopup() {
	var	h =[],
		idx = -1;

	h[++idx] = '<div id="api_key_popup" class="modal fade" tabindex="-1" role="dialog" aria-labelledby="apiKeyLabel" aria-hidden="true">';
	h[++idx] = '<div class="modal-dialog modal-lg">';
	h[++idx] = '<div class="modal-content">';
	h[++idx] = '<div class="modal-header">';
	h[++idx] = '<h4 class="modal-title" id="apiKeyLabel"></h4>';
	h[++idx] = '<button type="button" class="close" data-dismiss="modal" aria-label="Close"><span aria-hidden="true">&times;</span></button>';
	h[++idx] = '</div>';    // modal-headers

	h[++idx] = '<div class="modal-body">';
	h[++idx] = '<form>';
	h[++idx] = '<div class="form-group">';
	h[++idx] = '<input type="text" id="apiKey" required class="form-control" readOnly>';
	h[++idx] = '</div>';
	h[++idx] = '</form>';
	h[++idx] = '<button id="getKey" type="button" class="btn btn-primary">';
	h[++idx] = localise.set["c_gak"];
	h[++idx] = '</button>';
	h[++idx] = '<button id="deleteKey" type="button" class="btn btn-danger ml-2">';
	h[++idx] = localise.set["c_del"];
	h[++idx] = '</button>';
	h[++idx] = '<button id="copyKey" type="button" class="btn btn-default has_tt ml-2" title="Copy Key">';
	h[++idx] = localise.set["c_ck"];
	h[++idx] = '</button>';
	h[++idx] = '</div>';

	h[++idx] = '<div class="modal-footer">';
	h[++idx] = '<button type="button" class="btn btn-default" data-dismiss="modal">';
	h[++idx] = localise.set["c_close"];
	h[++idx] = '</button>';

	h[++idx] = '</div>';    // modal - footer
	h[++idx] = '</div>';        // modal - content
	h[++idx] = '</div>';            // modal - dialog
	h[++idx] = '</div>';                // popup

	$(document.body).append(h.join(''));

	enableApiKeyPopup();
}


/*
 * Update the user details on the page
 */
function updateUserDetails(data, getOrganisationsFn, getEnterprisesFn, getServerDetailsFn) {

	var groups = data.groups,
		i,
		bootstrap_enabled = (typeof $().modal == 'function');

	if(data.language && data.language !== gUserLocale) {
		try {
			localStorage.setItem('user_locale', data.language);  // Write to storage may be disabled
			location.reload();
		} catch (e) {

		}

	} else if(data.o_id != globals.gOrgId) {
		location.reload();
	}

	globals.gLoggedInUser = data;
	globals.gOrgId = data.o_id;

	// Save the organisation name for the logon screen
	try {
		localStorage.setItem('org_name', data.organisation_name);
	} catch (e) {

	}

	if(bootstrap_enabled) {

		$('#modify_me_popup').on('show.bs.modal', function (event) {
			var $this = $(this)
			$this.find('.modal-title').text(data.ident + "@" + data.organisation_name)

			$("#me_alert").hide();

			$('#me_edit_form')[0].reset();
			$('#reset_me_password_fields').removeClass('d-none').show();
			$('#password_me_fields').hide();
			addLanguageOptions($('.language_select'), data.language);
			addOrganisationOptions($('.organisation_select'), data.o_id, data.orgs);
			$('#me_name').val(data.name);
			$('#me_email').val(data.email);
			$('#me_enterprise').text(globals.gEnterpriseName);
			$('#u_tz').val(globals.gTimezone);

			$(".navbar-collapse").removeClass("in").addClass("collapse");	// Remove drop down menu
		});


	} else {
		$('#username').text(data.name).button({ label: htmlEncode(data.name),
			icons: { primary: "ui-icon-person" }}).off().click(function(){
			$('#me_edit_form')[0].reset();

			$('#reset_me_password_fields').removeClass('d-none').show();
			$('#password_me_fields').hide();
			addLanguageOptions($('.language_select'), data.language);
			addOrganisationOptions($('.organisation_select'), data.o_id, data.orgs);
			$('#me_name').val(data.name);
			$('#me_email').val(data.email);
			$('#me_enterprise').text(globals.gEnterpriseName);
			$('#u_tz').val(globals.gTimezone);

			$('#modify_me_popup').dialog("option", "title", htmlEncode(data.name + "@" + data.organisation_name));
			$('#modify_me_popup').dialog("open");
		});
	}

	/*
	 * Show restricted functions
	 */
	if(groups) {
		for(i = 0; i < groups.length; i++) {
			if(groups[i].id === globals.GROUP_ADMIN) {
				globals.gIsAdministrator = true;

                if(data.billing_enabled) {
                    globals.gOrgBillingData = true;
                }

			} else if(groups[i].id === globals.GROUP_ORG_ADMIN) {
				globals.gIsOrgAdministrator = true;
				globals.gBillingData = true;

			} else if(groups[i].id === globals.GROUP_SECURITY) {
				globals.gIsSecurityAdministrator = true;

			} else if(groups[i].id === globals.GROUP_ENTERPRISE) {
                globals.gIsEnterpriseAdministrator = true;
				globals.gBillingData = true;

            } else if(groups[i].id === globals.GROUP_LINKAGES) {
				globals.gIsLinkFollower = true;

			} else if(groups[i].id === globals.GROUP_ANALYST) {
				globals.gIsAnalyst = true;

			} else if(groups[i].id === globals.GROUP_DASHBOARD) {
				globals.gIsDashboard = true;

			} else if(groups[i].id === globals.GROUP_MANAGE) {
				globals.gIsManage = true;

			} else if(groups[i].id === globals.GROUP_ENUM) {
				globals.gIsEnum = true;

			} else if(groups[i].id === globals.GROUP_VIEW_DATA) {
                globals.gViewData = true;

            } else if(groups[i].id === globals.GROUP_MANAGE_TASKS) {
				globals.gManageTasks = true;

			} else if(groups[i].id === globals.GROUP_OWNER) {
                globals.gIsServerOwner = true;

            } else if(groups[i].id === globals.GROUP_CONSOLE_ADMIN) {
				globals.gIsConsoleAdmin = true;
			}
		}
	}

	// Only show items relevant to a user
	$('.restrict_role').hide();
	if(globals.gIsEnum) {
		$('.enum_role').removeClass('d-none').show();
	}
	if(globals.gIsAnalyst) {
		$('.analyst_role').removeClass('d-none').show();
	}
	if(globals.gIsDashboard) {
		$('.dashboard_role').removeClass('d-none').show();
	}
	if(globals.gViewData) {
		$('.data_role').removeClass('d-none').show();
	}
	if(globals.gManageTasks) {
		$('.task_role').show();
	}
	if(globals.gIsAdministrator) {
		$('.admin_role').removeClass('d-none').show();
	}
	if(globals.gIsManage) {
		$('.manage_role').removeClass('d-none').show();
	}
	if(globals.gIsSecurityAdministrator) {
		$('.security_role').removeClass('d-none').show();
	}
	if(globals.gIsOrgAdministrator) {  // Admins can see their personal organisations
		$('.org_role').removeClass('d-none').show();
	}
	if(globals.gIsOrgAdministrator || globals.gIsAdministrator) {  // Admins can see their personal organisations
		if(typeof getOrganisationsFn === "function") {
			getOrganisationsFn();
		}
	}
	if(globals.gIsEnterpriseAdministrator) {
		$('.enterprise_role').removeClass('d-none').show();
		if(typeof getEnterprisesFn === "function") {
			getEnterprisesFn();
		}
	}
	if(globals.gIsServerOwner) {
		$('.owner_role').removeClass('d-none').show();
		if(typeof getServerDetailsFn === "function") {
			getServerDetailsFn();
		}
	}

	if(globals.gTraining) {
		$('#train_link').prop("href", globals.gTraining);
		$('#m_training').removeClass('d-none').show();
	}

	//TODO set logic for enabling disabling billing
	if(isBusinessServer() && (globals.gBillingData || globals.gOrgBillingData)) {
		$('.billing_role').removeClass('d-none').show();
	}

	// Other conditional elements
	if(globals.gSendTrail === 'off') {
		$('.user_trail').hide();
	}

	// 	Customer configurable details - the configurable part is TODO
	$('#my_name').val(data.name);			// Add the name to the configurable list

	if(data.settings) {
		var userDetails = JSON.parse(data.settings);
		$('#my_title').val(userDetails.title);
		$('#my_license').val(userDetails.license);
		$('#my_signature').attr("src", "/surveyKPI/file/" + data.signature + "/users?type=sig");
	}

	// Hide any menus that have been disabled by custom java scripts
	$('.perm_dis_menu').hide();
}

function addLanguageOptions($elem, current) {

	var h = [],
		idx = -1,
		i,
		languages = [
			{
				locale: "ar",
				name: "Arabic"
			},
			{
				locale: "en",
				name: "English"
			},
			{
				locale: "fr",
				name: "French"
			},
			{
				locale: "hi",
				name: "Hindi"
			},
			{
				locale: "pt",
				name: "Portugese"
			},
			{
				locale: "es",
				name: "Spanish"
			},
			{
				locale: "uk",
				name: "Ukrainian"
			}
		];

	for(i = 0; i < languages.length; i++) {
		h[++idx] = '<option value="';
		h[++idx] = languages[i].locale;
		h[++idx] = '">';
		h[++idx] = localise.set[languages[i].locale];
		h[++idx] = '</option>';
	}
	$elem.html(h.join(''));
	if(current) {
		$elem.val(current);
	} else {
		$elem.val("en");
	}
}

function addOrganisationOptions($elem, current, orgs) {

	var h = [],
		idx = -1,
		i;

	for(i = 0; i < orgs.length; i++) {
		h[++idx] = '<option value="';
		h[++idx] = orgs[i].id;
		h[++idx] = '">';
		h[++idx] = htmlEncode(orgs[i].name);
		h[++idx] = '</option>';
	}
	$elem.html(h.join(''));
	if(current) {
		$elem.val(current);
	}
}

/*
 * Enable the user profile button
 */
function enableUserProfile () {
	// Initialise the dialog for the user to edit their own account details
	$('#modify_me_popup').dialog(
		{
			autoOpen: false, closeOnEscape:true, draggable:true, modal:true,
			title:"User Profile",
			show:"drop",
			width:350,
			height:350,
			zIndex: 2000,
			buttons: [
				{
					text: "Cancel",
					click: function() {

						$(this).dialog("close");
					}
				}, {
					text: "Save",
					click: function() {

						var user = globals.gLoggedInUser;

						user.name = $('#me_name').val();
						user.language = $('#me_language').val();
						user.email = $('#me_email').val();
						if($('#me_password').is(':visible')) {
							user.password = $('#me_password').val();
							if($('#me_password_confirm').val() !== user.password) {
								user.password = undefined;
								alert("Passwords do not match");
								$('#me_password').focus();
								return false;
							}
						} else {
							user.password = undefined;
						}

						user.current_project_id = 0;	// Tell service to ignore project id and update other details
						user.current_survey_id = 0;
						user.current_task_group_id = 0;

						user.timezone = $('#u_tz').val();
						globals.gTimezone = user.timezone;

						user.o_id = $('#me_organisation').val();
						if(user.o_id == globals.gOrgId) {
							user.o_id = 0;	// No change
						}

						saveCurrentUser(user, undefined);			// Save the updated user details to disk
						$(this).dialog("close");
					},
				}
			]
		}
	);


	// Initialise the reset password checkbox
	$('#reset_me_password').click(function () {
		if($(this).is(':checked')) {
			$('#password_me_fields').removeClass('d-none').show();
		} else {
			$('#password_me_fields').hide();
		}
	});
}

/*
 * Enable the user profile button
 */
function enableUserProfileBS () {

	$("#modify_me_popup :input").keydown(function() {
		$("#me_alert").hide();
	});

	/*
	 * Save the user profile
	 */
	$('#userProfileSave').click(function() {
		var user = globals.gLoggedInUser;

		user.name = $('#me_name').val();
		user.language = $('#me_language').val();
		user.email = $('#me_email').val();
		if($('#me_password').is(':visible')) {
			user.password = $('#me_password').val();
			if($('#me_password_confirm').val() !== user.password) {
				user.password = undefined;
				$('#me_alert').removeClass('alert-success d-none').addClass('alert-danger').text(localise.set["msg_pwd_m"]).show();
				$('#me_password').focus();
				return false;
			}
		} else {
			user.password = undefined;
		}

		user.o_id = $('#me_organisation').val();
		if(user.o_id == globals.gOrgId) {
			user.o_id = 0;	// No change
		}

		globals.gTimezone = $('#u_tz').val();
		user.timezone = globals.gTimezone;

		user.current_project_id = 0;	// Tell service to ignore project id and update other details
		user.current_survey_id = 0;
		user.current_task_group_id = 0;

		saveCurrentUser(user, $('#modify_me_popup'));			// Save the updated user details to disk
	});


	// Initialise the reset password checkbox
	$('#reset_me_password').click(function () {
		if($(this).is(':checked')) {
			$('#password_me_fields').removeClass('d-none').show();
		} else {
			$('#password_me_fields').hide();
		}
	});
}

/*
 * Respond to events on the API key popup
 */
function enableApiKeyPopup() {

	$('#api_key_popup').on('show.bs.modal', function (event) {
		/*
		 * Get the current API key
		 */
		$('#getKey').prop('disabled', true);
		addHourglass();
		$.ajax({
			url: '/surveyKPI/user/api_key',
			cache: false,
			success: function (data) {
				removeHourglass();
				if (handleLogout(data)) {
					$('#apiKey').val(data.apiKey);
					$('#getKey').prop('disabled', false);
					if (data.apiKey) {
						$('#getKey').text(localise.set["c_rak"]);
						$('#deleteKey,#copyKey').prop('disabled', false);
					} else {
						$('#getKey').text(localise.set["c_gak"]);
						$('#deleteKey,#copyKey').prop('disabled', true);
					}
				}
			},
			error: function (xhr, textStatus, err) {
				removeHourglass();
				if (handleLogout(xhr.responseText)) {
					$('#getKey').prop('disabled', false);
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert(err);
						console.log("Error: Failed to get api key: " + err);
					}
				}
			}
		});
	});

	/*
	 * Delete a key
	 */
	$('#deleteKey').on("click",function () {
		addHourglass();
		$.ajax({
			type: "DELETE",
			url: '/surveyKPI/user/api_key',
			cache: false,
			success: function (data) {
				removeHourglass();
				if (handleLogout(data)) {
					$('#apiKey').val("");
					$('#getKey').prop('disabled', false);
					$('#getKey').text(localise.set["c_gak"]);
					$('#deleteKey,#copyKey').prop('disabled', true);
				}
			},
			error: function (xhr, textStatus, err) {
				removeHourglass();
				if (handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert(err);
						console.log("Error: Failed to delete api key: " + err);
					}
				}
			}
		});
	});

	/*
	 * Create a key
	 */
	$('#getKey').on("click", function () {
		addHourglass();
		$.ajax({
			type: "POST",
			cache: false,
			contentType: "application/x-www-form-urlencoded",
			dataType: 'json',
			url: "/surveyKPI/user/api_key/create",
			success: function (data) {
				removeHourglass();
				if (handleLogout(data)) {
					$('#apiKey').val(data.apiKey);
					$('#getKey').text(localise.set["c_rak"]);
					$('#deleteKey,#copyKey').prop('disabled', false);
				}
			},
			error: function (xhr, textStatus, err) {
				removeHourglass();
				if (handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert(err);
						console.log("Error: Failed to get api key: " + err);
					}
				}
			}
		});
	});

	// Respond to a user clicking copy api key
	$('.has_tt').tooltip();
	$('#copyKey').click(function () {
		var copyText = document.getElementById("apiKey");
		copyText.select();
		navigator.clipboard.writeText($('#apiKey').val());

		$('#copyKey').tooltip('dispose').tooltip({title: localise.set["c_c"] + ": " + copyText.value}).tooltip('show');

	});
	$('#copyKey').mouseout(function () {
		$('#copyKey').tooltip({title: localise.set["c_c"]});
	});
}

/*
 * Save the currently logged on user's details
 */
function saveCurrentUser(user, $dialog) {

	var fd = new FormData();
	fd.append("user", JSON.stringify(user));
	addHourglass();
	$.ajax({
		method: "POST",
		cache: false,
		contentType: false,
		processData: false,
		url: "/surveyKPI/user",
		data: fd,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				if (data.error) {
					if ($dialog) {
						$('#me_alert').removeClass('alert-success d-none').addClass('alert-danger').text(data.msg).show();
					} else {
						alert(localise.set["c_error"] + " : " + data.msg);  // legacy non bootstrap
					}
				} else if ($dialog) {
					$dialog.modal("hide");
				}
				updateUserDetails(data, undefined);
			}

		}, error: function(data, status) {
			removeHourglass();
			alert(localise.set["c_error"] + " : " + data.responseText);
		}
	});
}

function getAvailableTimeZones(callback) {
	addHourglass();
	$.ajax({
		url: "/surveyKPI/utility/timezones",
		cache: true,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				if (typeof callback == "function") {
					callback(data);
				}
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert(localise.set["c_error"] + ": " + err);
				}
			}
		}
	});
}

function showTimeZones(timeZones) {
	var h =[],
		idx = -1,
		i,
		tz;

	for (i = 0; i < timeZones.length; i++) {
		tz = timeZones[i];
		h[++idx] = '<option value="';
		h[++idx] = tz.id;
		h[++idx] = '">';
		h[++idx] = htmlEncode(tz.name);
		h[++idx] = '</option>';
	}
	$('.timezone_select').empty().html(h.join(''));
	if(!globals.gTimezone) {
		globals.gTimezone = Intl.DateTimeFormat().resolvedOptions().timeZone;      // Browser timezone
	}
	$('#u_tz').val(globals.gTimezone);   // Set time zone in user profile
	$('#timezone').text(localise.set["c_tz"] + ": " + globals.gTimezone);   // Show timezone where this is enabled
}

function addTimeZoneToUrl(url) {
	if(url) {
		if(url.indexOf("?") > 0) {
			url += "&";
		} else {
			url += "?";
		}
		url += "tz=";
		url += encodeURIComponent(globals.gTimezone);
	}
	return url;
}

/*
 * Create the user profile dialog and get any data it needs
 */
function setupUserProfile(bs4) {

	if(bs4) {
		addUserDetailsPopupBootstrap4();
		addApiKeyPopup();
	} else {
		addUserDetailsPopup();	// legacy
	}
	getAvailableTimeZones(showTimeZones);
}

function getLoggedInUser(callback, getAll, getProjects, getOrganisationsFn, hideUserDetails,
                         dontGetCurrentSurvey, getEnterprisesFn, getServerDetailsFn, getSMSNumbers) {

	globals.gIsAdministrator = false;

	addHourglass();
	$.ajax({
		url: "/surveyKPI/user",
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				var i;

				globals.gServerCanSendEmail = data.sendEmail;

				globals.gEmailEnabled = data.allow_email;
				globals.gFacebookEnabled = data.allow_facebook;
				globals.gTwitterEnabled = data.allow_twitter;
				globals.gCanEdit = data.can_edit;
				globals.gSendTrail = data.ft_send_location;
				globals.gAlertSeen = data.seen;		// Alerts have been acknowledged
				globals.gLastAlertTime = data.lastalert;
				globals.gOrgId = data.o_id;
				globals.gEntId = data.e_id;
				globals.gEnterpriseName = data.enterprise_name;
				globals.gSetAsTheme = data.set_as_theme;
				globals.gNavbarColor = data.navbar_color;
				globals.gNavbarTextColor = data.navbar_text_color;
				globals.gTraining = data.training;
				globals.gRefreshRate = data.refresh_rate;

				if (data.timezone) {
					globals.gTimezone = data.timezone;
				} else {
					globals.gTimezone = Intl.DateTimeFormat().resolvedOptions().timeZone;
				}
				$('#u_tz').val(globals.gTimezone);

				if (!hideUserDetails) {
					updateUserDetails(data, getOrganisationsFn, getEnterprisesFn, getServerDetailsFn);
				}

				if(getSMSNumbers) {
					getSMSNumbers();
				}

				if(data.totalTasks > 0) {
					$('.total_tasks').html('(' + htmlEncode(data.totalTasks) + ')').addClass('btn-danger');
				}

				if (!dontGetCurrentSurvey) {	// Hack, on edit screen current survey is set as parameter not from the user's defaults
					globals.gCurrentSurvey = data.current_survey_id;
					globals.gCurrentSurveyIdent = data.current_survey_ident;
				}
				globals.gCurrentProject = data.current_project_id;
				globals.gCurrentTaskGroup = data.current_task_group_id;
				$('#projectId').val(globals.gCurrentProject);		// Set the project value for the hidden field in template upload
				if (data.groupSurveys) {
					for (i = 0; i < data.groupSurveys.length; i++) {
						globals.gGroupSurveys[data.groupSurveys[i].sId] = data.groupSurveys[i].groupIdent;
						globals.gSubForms[data.groupSurveys[i].sId] = data.groupSurveys[i].fName;
					}
				}

				setOrganisationTheme();

				if (getProjects) {
					getMyProjects(globals.gCurrentProject, callback, getAll);	// Get projects
				} else {
					if (typeof callback !== "undefined") {
						callback(globals.gCurrentSurvey);				// Call the callback with the correct current survey
					}
				}
			}

		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0 || xhr.status == 401) {
					return;  // Not an error or an authorisation error which is handled by the service worker
				} else {
					console.log("Error: Failed to get user details: " + err);

					var msg = localise.set["c_error"] + ": ";
					if (err && err.message && err.message.indexOf('Unauthorized') >= 0) {
						msg += localise.set["c_auth"];
					} else {
						msg += err;
					}
					alert(msg);
				}
			}
		}
	});
}

/*
 * ===============================================================
 * Common functions for managing media (on both the edit page and shared resource page)
 * ===============================================================
 */

/*
 * Upload files to the server
 */
function uploadFiles(url, formName, callback1) {

	let f = document.forms.namedItem(formName),
		formData = new FormData(f);

	url = addUrlParam(url, "getlist=true");
	addHourglass();
	$('.submitFiles').addClass('disabled');
	$.ajax({
		url: url,
		type: 'POST',
		data: formData,
		cache: false,
		contentType: false,
		processData:false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				let cb1 = callback1;
				$('.upload_file_msg').show().removeClass('alert-danger').addClass('alert-success').html(localise.set["c_success"]);
				if (typeof cb1 === "function") {
					cb1(data);
				}
				document.forms.namedItem(formName).reset();
				$('#fileAddLocations').modal('hide');
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				document.forms.namedItem(formName).reset();
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					var msg = htmlEncode(xhr.responseText);
					if (msg && msg.indexOf("no tags") >= 0) {
						msg = localise.set["msg_u_nt"];
					} else {
						msg = localise.set["msg_u_f"] + " : " + msg;
					}
					$('.upload_file_msg').show().removeClass('alert-success').addClass('alert-danger').html(msg);
					$('#fileAddLocations').modal('hide');
				}
			}
		}
	});
}

/*
 * Add a parameter to a URL
 */
function addUrlParam(url, param) {
	if(url.indexOf("?") > 0) {
		url += "&" + param;
	} else {
		url += "?" + param;
	}
	return url;
}

/*
 * Refresh the media view and then set the mode to manage
 */
function refreshMediaViewManage(data, sId) {
	refreshMediaView(data, sId);
	$('.mediaManage').show();
	$('.mediaSelect').hide();
}
/*
 * Refresh the view of any attached media if the available media items has changed
 * sId is set if a resources for that survey are being viewed
 */
function refreshMediaView(data, sId) {

	var i,
		$elementMedia,
		$elementCsv,
		hCsv = [],
		idxCsv = -1,
		hMedia = [],
		idxMedia = -1;

	if(sId) {
		$('#survey_id').val(sId);
	}

	if(data) {
		window.gFiles = data.files;
		let files = data.files;

		$elementMedia = $('#filesOrg');
		$elementCsv = $('#csvOrg');

		if(files) {
			for (i = 0; i < files.length; i++) {
				if (files[i].type === 'csv') {
					hCsv[++idxCsv] = getMediaRecord(files[i], 'csv', i, sId > 0);
				} else {
					hMedia[++idxMedia] = getMediaRecord(files[i], 'media', i, sId > 0);
				}
			}
		}

		$elementMedia.html(hMedia.join(""));
		$elementCsv.html(hCsv.join(""));

		$('.media_delete').click(function () {
			let item = window.gFiles[$(this).val()];

			if(confirm(localise.set["msg_confirm_del"] + " " + htmlEncode(item.name))) {
				delete_media(item.name, sId);
			}
		});

		$('.media_history').click(function () {
			var item = window.gFiles[$(this).val()];
			var url = '/app/resource_history.html?resource=' + item.name;
			if(sId) {
				url += '&survey_id=' + sId;
			}
			window.location.href = url;
		});

		$('.csv_replace').click(function(e) {

			$('#fileCsv').show();
			$('#fileMedia').hide();

			replace(window.gFiles[$(this).val()]);
		});

		$('.media_replace').click(function(e) {

			$('#fileCsv').hide();
			$('#fileMedia').show();

			replace(window.gFiles[$(this).val()]);
		});

	}

	// If this is the organisational view we can refresh the list of choices for selecting vector maps
	if(!sId) {
		refreshVectorSelects(data);
	}
}

function replace(item) {

	$('#uploadAction').val('replace');
	$('#itemName').val(getBaseName(item.name));

	$('.upload_alert').hide();
	$('.notreplace').hide();
	$('#media_add_title').text(localise.set["tm_c_sr_rep"] + ": " + item.name);

	$('#fileAddPopup').modal('show');

}
function getBaseName(fileName) {
	let lastDot = fileName.lastIndexOf(".");
	let baseName = fileName;
	if (lastDot !== -1) {
		baseName = fileName.substr(0, lastDot);
	}
	return baseName;
}
function getMediaRecord(file, panel, record, surveyLevel) {
	var h = [],
		idx = -1;

	h[++idx] = '<tr class="';
	h[++idx] = htmlEncode(file.type);
	h[++idx] = '">';

	if(panel === 'media') {
		h[++idx] = '<td class="preview">';
		h[++idx] = '<a target="_blank" href="';
		h[++idx] = htmlEncode(file.url) + addCacheBuster(file.url);
		h[++idx] = '">';
		if (file.type == "audio") {
			h[++idx] = addAudioIcon();
		} else if (file.type == "geojson") {
			h[++idx] = addVectorMapIcon();
		} else {
			h[++idx] = '<img width="100" height="100" src="';
			h[++idx] = htmlEncode(file.thumbnailUrl) + addCacheBuster(file.thumbnailUrl);
			h[++idx] = '" alt="';
			h[++idx] = htmlEncode(file.name);
			h[++idx] = '">';
		}
		h[++idx] = '</a>';
		h[++idx] = '</td>';
	}

	h[++idx] = '<td class="filename">';
	h[++idx] = '<p>';
	h[++idx] = htmlEncode(file.name);
	h[++idx] = '</p>';
	h[++idx] = '</td>';

	h[++idx] = '<td class="mediaManage">';
	h[++idx] = localTime(file.modified);
	h[++idx] = '</td>';

	h[++idx] = '<td class="mediaManage">';
	h[++idx] = '<p>';
	h[++idx] = htmlEncode(file.size);
	h[++idx] = '</p>';
	h[++idx] = '</td>';

	h[++idx] = '<td class="mediaManage">';
	h[++idx] = '<button class="btn ';
	h[++idx] = (panel === 'csv') ? 'csv_replace' : 'media_replace';
	h[++idx] = '" value="';
	h[++idx] = record;
	h[++idx] = '">';
	h[++idx] = '<i class="fas fa-sync-alt"></i>';
	h[++idx] = '</button>';
	h[++idx] = '</td>';

	// Action Buttons
	let downloadUrl = '/surveyKPI/shared/latest/' + file.name
		+ (surveyLevel ? '?sIdent=' + globals.gCurrentSurveyIdent : '');
	h[++idx] = '<td class="mediaManage">';
	h[++idx] = '<a class="media_download btn btn-info" href="';					// Download
	h[++idx] = htmlEncode(downloadUrl + addCacheBuster(downloadUrl));
	h[++idx] = '">';
	h[++idx] = '<i class="fas fa-download"></i>'
	h[++idx] = '</a>';
	h[++idx] = '<button class="media_history btn btn-primary" value="';	// History
	h[++idx] = record;
	h[++idx] = '">';
	h[++idx] = '<i class="fas fa-landmark"></i>'
	h[++idx] = '</button>';
	h[++idx] = '<button class="media_delete btn btn-danger" value="';		// Delete
	h[++idx] = record;
	h[++idx] = '">';
	h[++idx] = '<i class="fas fa-trash-alt"></i>'
	h[++idx] = '</button>';

	h[++idx] = '</td>';

	h[++idx] = '<td class="mediaSelect">';
	h[++idx] = '<button class="mediaAdd btn btn-success">';
	h[++idx] = '<i class="fas fa-plus"></i> '
	h[++idx] = localise.set['c_add'];
	h[++idx] = '</button>';
	h[++idx] = '</td>';


	h[++idx] = '</tr>';

	return h.join('');
}
/*
 * Refresh the vector select lists
 */
function refreshVectorSelects(data) {

	var i,
		$vectorData = $('#vector_data'),
		$vectorStyle = $('#vector_style'),
		h_d = [],
		idx_d = -1,
		h_s = [],
		idx_s = -1,
		files;

	if(data) {
		files = data.files;

		for(i = 0; i < files.length; i++){
			if(files[i].type === "geojson") {
				h_d[++idx_d] = '<option value="';
				h_d[++idx_d] = files[i].name;
				h_d[++idx_d] = '">';
				h_d[++idx_d] = htmlEncode(files[i].name);
				h_d[++idx_d] = '</option>';
			}

			if(files[i].type === "TODO") {
				h_s[++idx_s] = '<option value="';
				h_s[++idx_s] = files[i].name;
				h_s[++idx_s] = '">';
				h_s[++idx_s] = htmlEncode(files[i].name);
				h_s[++idx_s] = '</option>';
			}

		}


		$vectorData.html(h_d.join(""));
		$vectorStyle.html(h_s.join(""));
	}
}

function addAudioIcon() {
	var h = [],
		idx = -1;

	h[++idx] = '<span class="has_tt" title="Audio">';
	h[++idx] = '<span class="glyphicon glyphicon-volume-up edit_type"></span>';
	h[++idx] = '</span>';

	return h.join('');
}

function addVectorMapIcon() {
	var h = [],
		idx = -1;

	h[++idx] = '<span class="has_tt" title="Audio">';
	h[++idx] = '<span class="glyphicon glyphicon glyphicon-map-marker edit_type"></span>';
	h[++idx] = '</span>';

	return h.join('');
}

function getFilesFromServer(sId, callback, getall) {

	let url = '/surveyKPI/upload/media';
	let hasParams = false;
	if(sId) {
		url += '?survey_id=' + sId;
		hasParams = true;
	}
	if(getall) {
		url += (hasParams ? '&' : '?') + 'getall=true';
	}

	url += addCacheBuster(url);

	addHourglass();
	$.ajax({
		url: url,
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				let surveyId = sId;
				callback(data, surveyId);
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					$('.upload_file_msg').removeClass('alert-success').addClass('alert-danger').html("Error: " + htmlEncode(err));
				}
			}
		}
	});
}

/*
 * Delete a media file
 */
function delete_media(filename, sId) {

	var url = "/surveyKPI/shared/file/" + encodeURIComponent(filename);

	if(sId > 0) {
		url += '?survey_id=' + sId;
	}

	addHourglass();
	$.ajax({
		url: url,
		type: 'DELETE',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				var surveyId = sId;
				getFilesFromServer(surveyId, refreshMediaViewManage, false);
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert("Error: " + err);
				}
			}
		}
	});

}
/*
 * ===============================================================
 * Hourglass Functions
 * ===============================================================
 */

function addHourglass() {

	if(gWait === 0) {

		$("#hour_glass,.hour_glass,.sk-spinner").show();
	}
	++gWait;
}

function removeHourglass() {

	--gWait;
	if(gWait === 0) {

		$("#hour_glass,.hour_glass,.sk-spinner").hide();
	}

}

/*
 * ===============================================================
 * Survey Functions
 * ===============================================================
 */

/*
 * Load the surveys from the server
 */
function loadSurveys(projectId, selector, getDeleted, addAll, callback, useIdx, sId, addNone, incReadOnly) {

	var url="/surveyKPI/surveys?projectId=" + projectId + "&blocked=true";

	if(selector === undefined) {
		selector = ".survey_select";	// Update the entire class of survey select controls
	}

	if(typeof projectId !== "undefined" && projectId > 0) {

		if(getDeleted) {
			url+="&deleted=true";
		}

		addHourglass();

		$.ajax({
			url: url,
			dataType: 'json',
			cache: false,
			success: function(data) {
				removeHourglass();
				if(handleLogout(data)) {
					var sel = selector;
					var all = addAll;

					showSurveyList(data, sel + ".data_survey", all, true, false, useIdx, sId, addNone, false, incReadOnly);
					showSurveyList(data, sel + ".oversight_survey", all, false, true, useIdx, sId, addNone, false, incReadOnly);
					showSurveyList(data, sel + ".data_oversight_survey", all, true, true, useIdx, sId, addNone, false, incReadOnly);
					showSurveyList(data, ".bundle_select", all, true, true, false, sId, addNone, true, incReadOnly);

					if (typeof callback == "function") {
						callback(data);
					}
				}
			},
			error: function(xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						console.log("Error: Failed to get list of surveys: " + err);
					}
				}
			}
		});
	} else {
		var $elem = $('.data_survey, .oversight_survey, .data_oversight_survey');
		$elem.empty();
		if(addAll) {
			$elem.append('<option value="_all">' + localise.set["c_all_s"] + '</option>');
		}

		if(callback) {
			callback();
		}

	}
}

/*
 * Load the surveys from the server
 */
function loadSurveyIdentList(projectId, sIdent, addAll, addNone) {

	var url="/surveyKPI/surveys/project/" + projectId;
	var selector = ".survey_select";

	if(typeof projectId !== "undefined" && projectId > 0) {
		addHourglass();
		$.ajax({
			url: url,
			dataType: 'json',
			cache: false,
			success: function(data) {
				removeHourglass();
				if(handleLogout(data)) {
					var sel = selector;

					showIdentSurveyList(data, sel, addAll, sIdent, addNone);

				}
			},
			error: function(xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						console.log("Error: Failed to get list of surveys: " + err);
					}
				}
			}
		});
	}
}

/*
 * Show the surveys in select controls
 */
function showSurveyList(data, selector, addAll, dataSurvey, oversightSurvey, useIdx, sId, addNone, bundle, incReadOnly) {

	var i,
		item,
		h = [],
		idx = -1,
		$elem,
		$elem_disable_blocked,
		selValue;

	$elem = $(selector);
	$elem_disable_blocked = $(selector + ".disable_blocked");

	$elem.empty();
	var valueSelected = false;
	if(addAll) {
		h[++idx] = '<option value="_all">';
		h[++idx] = localise.set["c_all_s"];		// All Surveys
		h[++idx] = '</option>';

		selValue = "_all";
		valueSelected = true;
	}
	if(addNone) {
		h[++idx] = '<option value="0">';
		h[++idx] = localise.set["c_none"];		// No survey
		h[++idx] = '</option>';
	}

	var bundleObj = {};
	for(i = 0; i < data.length; i++) {
		item = data[i];
		if(!bundle || !bundleObj[item.groupSurveyDetails]) {	// If this is for a bundle list remove duplicate entries
			if ((incReadOnly || !item.readOnlySurvey) && (item.dataSurvey && dataSurvey || item.oversightSurvey && oversightSurvey)) {
				h[++idx] = '<option';
				if (!valueSelected && !item.blocked) {
					valueSelected = true;
					selValue = useIdx ? i : item.id;
				}
				if (item.blocked && !bundle) {
					h[++idx] = ' class="blocked"';
				}
				h[++idx] = ' value="';
				if(bundle) {
					h[++idx] = useIdx ? i : item.groupSurveyIdent;
				} else {
					h[++idx] = useIdx ? i : item.id;
				}
				h[++idx] = '">';
				if(bundle){
					h[++idx] = htmlEncode(item.groupSurveyDetails);
					bundleObj[item.groupSurveyDetails] = '1';
				} else {
					h[++idx] = htmlEncode(item.displayName);

					if (item.blocked) {
						h[++idx] = ' (' + localise.set["c_blocked"] + ')';
					}
				}
				h[++idx] = '</option>';
			}
			if (typeof sid === 'undefined') {
				if (globals.gCurrentSurvey > 0 && globals.gCurrentSurvey === item.id) {
					selValue = useIdx ? i : item.id;
				}
			} else {
				if (sId > 0 && sId === item.id) {
					selValue = useIdx ? i : item.id;
				}
			}
		}
	}

	$elem.empty().append(h.join(''));
	$elem.val(selValue);
	$("option.blocked", $elem_disable_blocked).attr("disabled", "disabled");

}

/*
 * Show the surveys in select boxes
 */
function showIdentSurveyList(data, selector, addAll, sIdent, addNone) {

	var i,
		item,
		h = [],
		idx = -1,
		$elem;

	$elem = $(selector);

	$elem.empty();
	if(addAll) {
		h[++idx] = '<option value="_all">';
		h[++idx] = localise.set["c_all_s"];		// All Surveys
		h[++idx] = '</option>';
	}
	if(addNone) {
		h[++idx] = '<option value="_none">';
		h[++idx] = localise.set["c_none"];		// No Survey
		h[++idx] = '</option>';
	}

	for(i = 0; i < data.length; i++) {
		item = data[i];
		h[++idx] = '<option';
		h[++idx] = ' value="';
		h[++idx] = item.ident;
		h[++idx] = '">';
		h[++idx] = htmlEncode(item.name);
		h[++idx] = '</option>';
	}

	$elem.empty().append(h.join(''));
	if(sIdent) {
		$elem.val(sIdent);
	} else {
		$elem.val("_none");
	}

}


// Common Function to get the language and question list (for the default language)
function getLanguageList(sId, callback, addNone, selector, setGroupList, filterQuestion) {

	if(typeof sId === "undefined") {
		sId = globals.gCurrentSurvey;
	}

	if(typeof filterQuestion === "undefined") {
		filterQuestion = "-1";
	}

	function getAsyncLanguageList(sId, theCallback, selector, filterQuestion) {
		addHourglass();
		$.ajax({
			url: languageListUrl(sId),
			dataType: 'json',
			cache: false,
			success: function(data) {
				removeHourglass();
				if(handleLogout(data)) {
					globals.gSelector.setSurveyLanguages(sId, data);
					retrievedLanguages(sId, selector, data, theCallback, filterQuestion, setGroupList, addNone);
				}
			},
			error: function(xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert(localise.set["c_error"] + ": " + err);
					}
				}
			}
		});
	}

	var data = globals.gSelector.getSurveyLanguages(sId);
	if(data) {
		retrievedLanguages(sId, selector, data, callback, filterQuestion, setGroupList, addNone);
	} else {
		getAsyncLanguageList(sId, callback, selector, filterQuestion);
	}
}

/*
 * Called after languages have been retrieved
 */
function retrievedLanguages(sId, selector, data, theCallback, filterQuestion, setGroupList, addNone) {
	if(selector) {
		setSurveyViewLanguages(data, undefined, selector, addNone);
	} else {
		setSurveyViewLanguages(data, undefined, '#settings_language', false);
		setSurveyViewLanguages(data, undefined, '#export_language', true);
		setSurveyViewLanguages(data, undefined, '#language_name', false);
	}

	if(data[0]) {
		var dateQId = -1;
		if(typeof gTaskStart !== "undefined") {
			dateQId = gTaskStart;
		}
		getQuestionList(sId, data[0], filterQuestion, "-1", theCallback, setGroupList, undefined, dateQId, undefined, undefined, undefined);	// Default language to the first in the list
	} else {
		if(typeof theCallback === "function") {
			theCallback();
		}
	}
}

//Function to get the question list
function getQuestionList(sId, language, qId, groupId, callback, setGroupList, view, dateqId, qName, assignQuestion, scQuestion) {

	function getAsyncQuestionList(sId, language, theCallback, groupId, qId, view, dateqId, qName, assignQuestion, setGroupList, scQuestion) {

		var excludeReadOnly = true;
		if(setGroupList) {
			excludeReadOnly = false;		// Include read only questions in group list
		}
		addHourglass();
		$.ajax({
			url: questionListUrl(sId, language, excludeReadOnly),
			dataType: 'json',
			cache: false,
			success: function(data) {
				removeHourglass();
				if(handleLogout(data)) {
					globals.gSelector.setSurveyQuestions(sId, language, data);
					setSurveyViewQuestions(data, qId, view, dateqId, qName, assignQuestion, scQuestion);

					if (setGroupList && typeof setSurveyViewQuestionGroups === "function") {
						setSurveyViewQuestionGroups(data, groupId);
					}
					if (typeof theCallback === "function") {
						theCallback();
					}
				}
			},
			error: function(xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert("Error: Failed to get list of questions: " + err);
					}
				}
			}
		});
	}

	getAsyncQuestionList(sId, language, callback, groupId, qId, view, dateqId, qName, assignQuestion, setGroupList, scQuestion);
}

//Function to get the meta list
function getMetaList(sId, metaItem) {

	addHourglass();
	$.ajax({
		url: "/surveyKPI/metaList/" + sId,
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				globals.gSelector.setSurveyMeta(sId, data);
				setSurveyViewMeta(data, metaItem);
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert(localise.set["c_error"] + ": " + err);
				}
			}
		}
	});
}

/*
 * Function to get the list of notification alerts
 * These are extracted from the settings for the survey
 */
function getAlertList(sId, alertId) {

	addHourglass();
	$.ajax({
		url: "/surveyKPI/cases/settings/" + sId,
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				globals.gSelector.setSurveyAlerts(sId, data);
				setSurveyAlerts(data, alertId);
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert(localise.set["c_error"] + ": " + err);
				}
			}
		}
	});
}


//Set the language list in the survey view control
function setSurveyViewLanguages(list, language,elem, addNone) {

	var $languageSelect = $(elem),
		i;

	$languageSelect.empty();
	if(addNone) {
		$languageSelect.append('<option value="none">' + localise.set["c_none"] + '</option>');
	}

	for(i = 0; i < list.length; i++) {
		$languageSelect.append('<option value="' + list[i] + '">' + htmlEncode(list[i]) + '</option>');
	}

	if(language) {
		$languageSelect.val(language);
	}
}

// Set the question list in the survey view control
function setSurveyViewQuestions(list, qId, view, dateqId, qName, assignQuestion, scQuestion) {

	var $questionSelect = $('.selected_question'),
		$dateQuestions = $('.date_questions'),
		$scQuestions = $('#sc_question'),
		$questionNameSelect = $('.selected_name_question'),     // this should replace selected_question
		$assignQuestion = $('#assign_question'),
		label;

	$questionSelect.empty();
	$questionSelect.append('<option value="-1">' + localise.set["c_none"] + '</option>');

	$questionNameSelect.empty();
	$questionNameSelect.append('<option value="-1">' + localise.set["c_none"] + '</option>');

	$dateQuestions.empty();
	$dateQuestions.append('<option value="-1">' + localise.set["ed_i_c"] + '</option>');

	$scQuestions.empty();

	if(list) {
		$.each(list, function(j, item) {
			if(typeof item.q === "undefined") {
				label = "";
			} else {
				label = item.q;
			}

			$questionSelect.append('<option value="' + item.id + '">' + htmlEncode(item.name + " : " + label) + '</option>');
			$questionNameSelect.append('<option value="' + item.name + '">' + htmlEncode(item.name) + '</option>');
			if(item.type === 'timestamp' || item.type === 'dateTime' || item.type == 'date') {
				$dateQuestions.append('<option value="' + item.id + '">' + htmlEncode(item.name + " : " + label) + '</option>');
			}
			if(item.type === 'server_calculate') {
				let name = htmlEncode(item.name);
				$scQuestions.append(`<option value="${item.name}">${name}</option>`);
			}
		});
	}
	if(!qId) {
		qId = "-1";
	}
	$questionSelect.val(qId);

	if(!qName) {
		qName = "-1";
	}
	$questionNameSelect.val(qName);
	$assignQuestion.val(assignQuestion);

	if(!dateqId) {
		dateqId = "-1";
	}
	$dateQuestions.val(dateqId);

	// Server calculate question
	if(scQuestion) {
		$scQuestions.val(scQuestion);
	}

	if(view) {
		setFilterFromView(view);	// Set the filter dialog settings
	}

}

// Set the meta list in the survey view control
function setSurveyViewMeta(list, metaItem) {

	var $metaSelect = $('.selected_meta'),
		item,
		i;

	$metaSelect.empty();

	// Add none
	$metaSelect.append('<option value="-1">' + localise.set["c_none"] + '</option>');

	// Add the user who submitted the survey
	$metaSelect.append('<option value="_user">' + localise.set["c_submitter"] + '</option>');

	if(list) {
		for(i = 0; i < list.length; i++) {
			item = list[i];
			$metaSelect.append('<option value="' + item.name + '">' + htmlEncode(item.name) + '</option>');
		}
	}
	if(!metaItem) {
		metaItem = "-1";
	}
	$metaSelect.val(metaItem);

}

/*
 * Populate the alert list
 */
function setSurveyAlerts(settings, alertId) {

	var $elem = $('.alert_list'),
		item,
		i;

	$elem.empty();

	if(settings && settings.alerts) {
		for(i = 0; i < settings.alerts.length; i++) {
			item = settings.alerts[i];
			$elem.append('<option value="' + item.id + '">' + htmlEncode(item.name) + '</option>');
		}
	}
	if(alertId) {
		$elem.val(alertId);
	}


}

/*
 * ------------------------------------------------------------
 * Web service Functions
 */
function languageListUrl (sId) {

	var url = "/surveyKPI/languages/";
	url += sId;
	return url;
}

/*
 * Web service handler for retrieving available "count" questions for graph
 *  @param {string} survey
 */
function questionListUrl (sId, language, exc_read_only) {

	var url = "/surveyKPI/questionList/",
		ro_text;

	if(exc_read_only) {
		ro_text = "true";
	} else {
		ro_text = "false";
	}

	url += sId;
	url += "/" + encodeURIComponent(language);
	url += "?exc_read_only=" + ro_text;
	return url;
}

/**
 * Web service handler for question Meta Data
 * @param {string} survey id
 * @param {string} question id
 */
function questionMetaURL (sId, lang, qId) {

	var url = "/surveyKPI/question/";
	url += sId;
	url += "/" + lang;
	url += "/" + qId;
	url += "/getMeta";
	return url;
}

/*
 * Get a survey details - depends on globals being set
 */
function getSurveyDetails(callback, get_changes, hide_soft_deleted) {

	var tz = globals.gTimezone;
	var url="/surveyKPI/surveys/" + globals.gCurrentSurvey;
	if(get_changes) {
		url += "?get_changes=true";
		url += "&tz=" + encodeURIComponent(tz);
	} else {
		url += "?tz=" + encodeURIComponent(tz);
	}
	if(hide_soft_deleted) {
		url += "&get_soft_delete=false";
	}

	if(!globals.gCurrentSurvey) {
		alert("Error: Can't get survey details, Survey identifier not specified");
	} else {
		addHourglass();
		$.ajax({
			url: url,
			dataType: 'json',
			cache: false,
			success: function(data) {
				removeHourglass();
				if(handleLogout(data)) {
					globals.model.setSurveyData(data);
					globals.model.setSettings();
					setLanguages(data.languages, callback);

					if (typeof callback == "function") {
						callback();
					}
				}
			},
			error: function(xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						if (xhr.status == 404) {
							// The current survey has probably been deleted or the user no longer has access
							globals.gCurrentSurvey = undefined;
							return;
						}
						alert("Error: Failed to get survey: " + err);
					}
				}
			}
		});
	}
}

/*
 * Set the languages for the editor
 */
function setLanguages(languages, languageCallback) {

	var h = [],
		h2 = [],
		idx = -1,
		idx2 = -1,
		$lang_menu = $('.language_menu_list'),
		$lang = $('.language_list'),
		$lang1 = $('#language1'),
		$lang2 = $('#language2'),
		i;

	globals.gLanguage1 = 0;	// Language indexes used for translations
	globals.gLanguage2 = 0;
	if(languages.length > 1) {
		globals.gLanguage2 = 1;
	}

	for (i = 0; i < languages.length; i++) {
		h[++idx] = '<a data-lang="';
		h[++idx] = i;
		h[++idx] = '" class="dropdown-item" href="javascript:void(0)">';
		h[++idx] = htmlEncode(languages[i].name);
		h[++idx] = '</a>';

		h2[++idx2] = '<option value="';
		h2[++idx2] = i;
		h2[++idx2] = '">';
		h2[++idx2] = htmlEncode(languages[i].name);
		h2[++idx2] = '</option>';
	}

	$lang_menu.empty().append(h.join(""));
	$lang.empty().append(h2.join(""));

	$('#langSelected').text(languages[ globals.gLanguage].name);
	$('.language_menu_list a').click(function() {
		globals.gLanguage = $(this).data("lang");
		$('#langSelected').text(languages[ globals.gLanguage].name);
		languageCallback();
	});

	$lang1.val(globals.gLanguage1);
	$lang2.val(globals.gLanguage2)
}

/*
 * Get a survey details - depends on globals being set
 */
function createNewSurvey(name, existing, existing_survey, shared_results, callback) {

	console.log("create new: " + existing + " : " + existing_survey + " : " + shared_results);

	var url="/surveyKPI/surveys/new/" + globals.gCurrentProject + "/" + encodeURIComponent(name);
	if(!existing) {
		existing_survey = 0;
	}

	addHourglass();
	$.ajax({
		type: "POST",
		url: url,
		cache: false,
		dataType: 'json',
		data: {
			existing: existing,
			existing_survey: existing_survey,
			existing_form: 0,
			shared_results: shared_results
		},
		success: function(data) {
			removeHourglass();

			if(handleLogout(data)) {
				globals.model.setSurveyData(data);
				globals.model.setSettings();
				globals.gCurrentSurvey = data.id;

				saveCurrentProject(-1, globals.gCurrentSurvey, undefined);	// Save the current survey id

				setLanguages(data.languages, callback);

				if (typeof callback == "function") {
					callback();
				}
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					bootbox.alert(localise.set["c_error"] + " " + htmlEncode(xhr.responseText));
				}
			}
		}
	});
}

/*
 * Open a form for editing
 */
function openForm(type) {

	$('.reusing_form').hide();
	$('#base_on_existing').prop('checked', false);
	$('#shared_results').prop('checked', false);
	$('#new_form_name').val("");
	if(type === "new") {
		$('.existing_form').hide();
		$('.new_form').show();
		$('#openSurveyLabel').html(localise.set["tm_g_new"]);
		$('#get_form').html(localise.set["c_create"]);
		globals.gExistingSurvey = false;
	} else {
		$('.existing_form').show();
		$('.new_form').hide();
		$('#openSurveyLabel').html(localise.set["tm_g_open"]);
		$('#get_form').html(localise.set["m_open"]);
		globals.gExistingSurvey = true;
	}
	$('#openFormModal').modal('show');

}

/*
 * If this is a smap server return the subdomain
 */
function getServerSubDomainName() {

	var hostname = location.hostname;
	var sd = "";

	if(hostname.indexOf('.smap.com.au') > 0) {
		sd = hostname.substring(0, hostname.indexOf('.smap.com.au'));
	} else if(hostname === 'localhost') {
		sd = 'localhost';
	}

	return sd;
}

/*
 * Return true if this is a business server
 */
function isBusinessServer() {

	var hostname = location.hostname;
	var bs = true;

	if(hostname.indexOf('smap.com.au') > 0) {
		bs = false;
	}
	if(hostname.indexOf('sg.smap.com.au') >= 0 ||
		hostname.indexOf('ubuntu1804.smap.com.au') >= 0 ||
		hostname.indexOf('demo.smap.com.au') >= 0) {
		bs = true;
	}

	return bs;
}

/*
 * Returns the class of server that has custom menus
 */
function getCustomMenuClass() {

	var hostname = location.hostname;
	var classname = undefined;

	if(hostname.indexOf('cuso.smap.com.au') >= 0) {
		classname = '.xxxx1';
	} else if(hostname.indexOf('demo.smap.com.au') >= 0) {
		classname = '.xxxx1';
	} else {
		if(hostname === 'localhost') {
			classname = '.xxxx1';   // testing
		}
	}

	return classname;
}


/*
 * Return true if this is a self registration server
 */
function isSelfRegistrationServer() {
	var hostname = location.hostname;
	var sr = true;

	if(hostname !== 'localhost' &&
		hostname !== 'sg.smap.com.au' &&
		hostname.indexOf('reachnettechnologies.com') < 0 &&
		hostname.indexOf('.icanreach.com') < 0 &&
		hostname.indexOf('encontactone.com') < 0 &&
		hostname !== 'app.kontrolid.com' &&
		hostname !== 'kontrolid.smap.com.au') {
		sr = false;
	}
	return sr;
}

/*
 * Validate start and end dates
 */
function validDates() {
	var $d1 = $('#startDate'),
		$d2 = $('#endDate'),
		d1 = $d1.data("DateTimePicker").date(),
		d2 = $d2.data("DateTimePicker").date()

	if(!d1 || !d1.isValid()) {
		$('#ut_alert').show().text(localise.set["t_i_sd"]);
		setTimeout(function() {
			$('.form-control', '#startDate').focus();
		}, 0);
		return false;
	}

	if(!d2 || !d2.isValid()) {
		$('#ut_alert').show().text(localise.set["t_i_ed"]);
		setTimeout(function() {
			$('.form-control', '#endDate').focus();
		}, 0);
		return false;
	}

	if(d1 > d2) {
		$('#ut_alert').show().text("End date must be greater than or the same as the start date");
		setTimeout(function() {
			$('.form-control', '#startDate').focus();
		}, 0);
		return false;
	}

	$('#ut_alert').hide();
	return true;
}

/*
 * Convert a date into UTC
 */
function getUtcDate($element, start, end) {

	var theDate,
		utcDate;

	if(start) {
		theDate = $element.data("DateTimePicker").date().startOf('day');
	} else if (end) {
		theDate = $element.data("DateTimePicker").date().endOf('day');
	} else {
		theDate = $element.data("DateTimePicker").date();
	}

	utcDate = moment.utc(theDate);

	console.log("date:" + theDate.format("YYYY-MM-DD HH:mm:ss"));
	console.log("UTC:" + utcDate.format("YYYY-MM-DD HH:mm:ss"));

	return utcDate.valueOf();

}

/*
 * Get a description from a change made in the editor
 */
function getChangeDescription(change, version) {

	var h =[],
		idx = -1,
		oldVal,
		newVal,
		forms = globals.model.survey.forms,
		str;

	if(change.action === "external option") {
		/*
		 * Options added from a file
		 */
		h[++idx] = 'Choice <span style="color:blue;">';
		h[++idx] = htmlEncode(change.option.externalLabel);
		h[++idx] = '</span>';
		h[++idx] = ' from file: <span style="color:blue;">';
		h[++idx] = htmlEncode(change.fileName);
		h[++idx] = '</span>';

	} else if(change.action === "template_update") {
		h[++idx] = localise.set["ed_c_template"];
		h[++idx] = ' <span style="color:blue;">';
		h[++idx] = htmlEncode(change.msg);
		h[++idx] = '</span>';

	} else if(change.action === "template_add") {
		h[++idx] = localise.set["ed_a_template"];
		h[++idx] = ' <span style="color:blue;">';
		h[++idx] = htmlEncode(change.msg);
		h[++idx] = '</span>';

	} else if(change.action === "template_delete") {
		h[++idx] = ' <span style="color:red;">';
		h[++idx] = localise.set["ed_d_template"];
		h[++idx] = htmlEncode(change.msg);
		h[++idx] = '</span>';

	} else if(change.action === "settings_update") {
		h[++idx] = localise.set["ed_c_settings"];
		h[++idx] = ' <span style="color:blue;">';
		h[++idx] = htmlEncode(change.msg);
		h[++idx] = '</span>';

	} else if(change.action === "language_update") {
		h[++idx] = localise.set["ed_c_languages"];
		h[++idx] = ' <span style="color:blue;">';
		h[++idx] = htmlEncode(change.msg);
		h[++idx] = '</span>';

	} else if(change.action === "add_preload") {
		h[++idx] = ' <span style="color:blue;">';
		h[++idx] = htmlEncode(change.msg);
		h[++idx] = '</span>';

	} else if(change.action === "del_preload") {
		h[++idx] = ' <span style="color:red;">';
		h[++idx] = htmlEncode(change.msg);
		h[++idx] = '</span>';

	} else if(change.action === "update") {

		/*
		 * Updates to questions and options and list names
		 */
		if(change.property.prop === "type") {
			newVal = htmlEncode(translateType(change.property.newVal));
			oldVal = htmlEncode(translateType(change.property.oldVal));
		} else {
			newVal = htmlEncode(change.property.newVal);
			oldVal = htmlEncode(change.property.oldVal);
		}


		if(change.property.prop === "name") {

			// Deprecate the following when the structure of these log objects is made consistent
			if(typeof change.property.type === "optionList" || change.property.type === "unknown") {
				change.type = "choice list ";
			}

			h[++idx] = change.property.type;
			h[++idx] = ' ';
			h[++idx] = localise.set["msg_ren"],
				h[++idx] = ': <span style="color:blue;">';
			h[++idx] = newVal;		// Already encoded
			h[++idx] = '</span>';
			h[++idx] = ' from: <span style="color:red;">';
			h[++idx] = oldVal;	// Already encoded
			h[++idx] = '</span>';
		} else {
			str = localise.set["ed_c_chg_p"];
			if(change.property.propType === "constraint_msg" || change.property.propType === "required_msg" || change.property.propType === "guidance_hint") {
				str = str.replace("%s1", '"' + htmlEncode(change.property.propType) + '"');
			} else {
				str = str.replace("%s1", '"' + htmlEncode(change.property.prop) + '"');
			}
			str = str.replace("%s2", htmlEncode(change.property.name));
			str = str.replace("%s3", '<span style="color:blue;">' + newVal + '</span>');	// Already encoded
			str = str.replace("%s4", '<span style="color:red;">' + oldVal + '</span>');		// Already encoded
			h[++idx] = str;
		}

	} else if(change.action === "add")  {

		/*
		 * New questions or options
		 */
		if(change.type === "question" || change.changeType === "question"){  // deprecate checking of changeType

			str = localise.set["ed_c_add_q"];
			str = str.replace("%s1", '<span style="color:blue;">' + htmlEncode(change.question.name) + "</span>");
			var typeString;
			if(change.question.type === "string") {
				typeString = 'text';
			} else if(change.question.type === "select"){
				typeString = 'select_multiple';
			} else if(change.question.type === "select1"){
				typeString = 'select_one';
			} else {
				typeString = change.question.type;
			}
			str = str.replace("%s2", '<span style="color:red;">' + htmlEncode(typeString) + "</span>");
			h[++idx] = str;

		} else if(change.type === "option" || change.changeType === "option") {	// deprecate checking of changeType
			/*
			 * Options added or deleted from the editor
			 */
			str = localise.set["ed_c_add_o"];
			var valueStr = '<span style="color:blue;">' + change.option.value;
			if(change.option.labels && change.option.labels.length >= 1) {
				valueStr += ' (';
				valueStr += htmlEncode(change.option.labels[0].text);
				valueStr += ')';
			}
			valueStr += '</span>';
			str = str.replace("%s1", valueStr);
			str = str.replace("%s2", '<span style="color:blue;">' + htmlEncode(change.option.optionList) + '</span>');
			h[++idx] = str;
		}

	}  else if(change.action === "move")  {

		/*
		 * New questions or options
		 */
		h[++idx] = localise.set['c_moved'] + ' ';

		if(change.type === "question" || change.changeType === "question") {  // deprecate checking of changeType){

			h[++idx] = 'question <span style="color:blue;">';
			h[++idx] = htmlEncode(change.question.name);
			if(change.question.sourceSeq >= 0) {
				h[++idx] = '</span> from position <span style="color:red;">';
				h[++idx] = htmlEncode(change.question.sourceSeq);
				h[++idx] = '</span> in form ';
				h[++idx] = htmlEncode(forms[change.question.sourceFormIndex].name);
			} else {
				h[++idx] = '</span> from form ';
				h[++idx] = htmlEncode(forms[change.question.sourceFormIndex].name);
			}
			h[++idx] = '</span> to position <span style="color:red;">';
			h[++idx] = htmlEncode(change.question.seq);
			h[++idx] = '</span>';
			h[++idx] = ' in form ';
			if(change.question.formIndex < forms.length) {	// Allow for a form being deleted
				h[++idx] = htmlEncode(forms[change.question.formIndex].name);
			}


		} else if(change.type === "option") {

			h[++idx] = 'choice <span style="color:blue;">';
			h[++idx] = htmlEncode(change.option.value);
			if(change.option.labels && change.option.labels.length >= 1) {
				h[++idx] = ' (';
				h[++idx] = htmlEncode(change.option.labels[0].text);
				h[++idx] = ')';
			}
			h[++idx] = '</span>';
			h[++idx] = ' from choice list: <span style="color:blue;">';
			h[++idx] = htmlEncode(change.option.sourceOptionList);
			h[++idx] = '</span>';
			h[++idx] = ' to choice list: <span style="color:blue;">';
			h[++idx] = htmlEncode(change.option.optionList);
			h[++idx] = '</span>';
		}

	} else if(change.action === "delete")  {

		if(change.type === "question" || change.changeType === "question"){

			h[++idx] = localise.set["ed_c_del_q"];

			h[++idx] = ' <span style="color:blue;">';
			h[++idx] = htmlEncode(change.question.name);
			h[++idx] = '</span>';

		} else if(change.type === "option") {

			str = localise.set["ed_c_del_o"];
			var valueStr = '<span style="color:blue;">' + htmlEncode(change.option.value);
			if(change.option.labels && change.option.labels.length >= 1) {
				valueStr  += ' (';
				valueStr  += htmlEncode(change.option.labels[0].text);
				valueStr  += ')';
			}
			valueStr  += '</span>';
			str = str.replace("%s1", valueStr);
			str = str.replace("%s2", '<span style="color:blue;">' + htmlEncode(change.option.optionList) + '</span>');
			h[++idx] = str;
		}
	} else if(change.action === "set_required")  {
		if(change.msg.indexOf('not') < 0) {
			h[++idx] = localise.set["ed_c_sr"];
		} else {
			h[++idx] = localise.set["ed_c_snr"];
		}

	} else if(change.action === "upload_template")  {

		if(version > 1) {
			h[++idx] = localise.set["msg_survey_replaced"];
		} else {
			h[++idx] = localise.set["msg_survey_loaded"];
		}

	} else if(change.action === "role")  {

			h[++idx] = change.msg;

	} else {
		h[++idx] = htmlEncode(change.type);
		h[++idx] = ' ';
		h[++idx] = htmlEncode(change.name);
		h[++idx] = ' changed to: <span style="color:blue;">';
		h[++idx] = htmlEncode(change.newVal);
		h[++idx] = '</span>';
		h[++idx] = ' from: <span style="color:red;">';
		h[++idx] = htmlEncode(change.oldVal);
		h[++idx] = '</span>';
	}

	return h.join('');
}

// Translate types for use in change description
function translateType(input) {
	if(input === "string") {
		output = "text";
	} else {
		output = input;
	}
	return output;
}

/*
 * Get the shared locations from the server
 */
function getLocations(callback) {

	var url="/surveyKPI/tasks/locations";

	addHourglass();
	$.ajax({
		url: url,
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				if (typeof callback === "function") {
					callback(data);
				}
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					console.log("Error: Failed to get list of locations: " + err);
				}
			}
		}
	});

}

/*
 * update Location group list
 */
function refreshLocationGroups(tags, includeAll, currentGroup) {

	var g = undefined,
		h = [],
		idx = -1,
		i;

	var includeNfc = $('#includeNfc').prop('checked'),
		includeGeo = $('#includeGeo').prop('checked');

	if(tags) {
		for(i = 0; i < tags.length; i++) {
			if(includeAll || includeLocation(includeNfc, includeGeo, tags[i].uid, tags[i].lat, tags[i].lon)) {

				if (g != tags[i].group) {

					g = tags[i].group;
					if (typeof currentGroup === "undefined") {
						currentGroup = g;
					}

					if(includeAll) {
						if (currentGroup === g) {
							$('.location_group_list_sel').text(g);
						}
						h[++idx] = '<a class="dropdown-item" href="#">';
						h[++idx] = g;
						h[++idx] = '</a>';
					} else {
						h[++idx] = '<option';
						if (currentGroup === g) {
							h[++idx] = ' selected';
						}
						h[++idx] = ' value="';
						h[++idx] = g;
						h[++idx] = '">';
						h[++idx] = htmlEncode(g);
						h[++idx] = '</option>';
					}
				}
			}
		}
	}

	$('.location_group_list').empty().html(h.join(""));
	return currentGroup;
}

/*
 * Add the locations (NFC tags or geofence) to any drop down lists that use them
 */
function setLocationList(locns, current, currentGroup) {

	var h = [],
		idx = -1,
		i;

	if(locns && locns.length) {
		h[++idx] = '<option value="-1">';
		h[++idx] = localise.set["c_none"];
		h[++idx] = '</option>';
		for(i = 0; i < locns.length; i++) {
			if(locns[i].group === currentGroup) {
				h[++idx] = '<option value="';
				h[++idx] = i;
				h[++idx] = '">';
				h[++idx] = htmlEncode(locns[i].name);
				h[++idx] = '</option>';
			}
		}
	}

	$('.location_select').empty().append(h.join(""));
	$('.location_select').val(current);


}

/*
 * Test for whether or not a location should be shown in the resource page
 */
function includeLocation(includeNfc, includeGeo, uid, lat, lon) {
	var include = false;

	if(includeNfc && typeof uid !== 'undefined' && uid !== '') {
		include = true;
	}
	if(!include && includeGeo && lat != 0 && lon != 0) {
		include = true;
	}

	return include;
}

/*
 * Convert a timestamp in UTC to local time and return a date object
 */
function localTimeAsDate(utcTime) {
	var utcDate,
		localTime;

	if(utcTime) {
		if(utcTime.indexOf('+') > 0) {
			utcDate  = moment.utc(utcTime, 'YYYY-MM-DD HH:mm:ss Z').toDate();
		} else {
			utcDate  = moment.utc(utcTime, 'YYYY-MM-DD HH:mm:ss').toDate();
		}
		localTime = moment(utcDate);
	}
	return localTime;
}

/*
 * Convert a timestamp in UTC to local time
 */
function localTime(utcTime) {
	var utcDate,
		localTime;

	if(utcTime) {
		if(utcTime.indexOf('+') > 0) {
			utcDate  = moment.utc(utcTime, 'YYYY-MM-DD HH:mm:ss Z').toDate();
		} else {
			utcDate  = moment.utc(utcTime, 'YYYY-MM-DD HH:mm:ss').toDate();
		}
		localTime = moment(utcDate).format('YYYY-MM-DD HH:mm:ss');
	}
	return localTime;
}


function utcTime(localTime) {

	var utcTime,
		localDate;

	if(localTime) {
		localDate = moment(localTime).toDate();
		utcTime =  moment.utc(localDate).format('YYYY-MM-DD HH:mm:ss');
	}
	return utcTime;

}

function isLate(finish) {

	var late = false,
		current = new Date(),
		finishDate,
		localFinish;

	if(finish) {
		localFinish = localTime(finish);
		finishDate = new Date(localFinish);
		if(current > finishDate) {
			late = true;
		}
	}
	return late;

}

function downloadPdf(language, orientation, include_references, launched_only, sIdent, instanceId, pdfTemplateId) {

	var docURL = "/surveyKPI/pdf/" + sIdent
		+ "?language=" + language
		+ "&instance=" + instanceId
		+ "&pdftemplate=" + pdfTemplateId
		+ "&tz=" + globals.gTimezone;
	if(orientation === "landscape") {
		docURL += "&landscape=true";
	}
	if(include_references) {
		docURL += "&reference_surveys=true";
	}
	if(launched_only) {
		docURL += "&launched_only=true";
	}

	downloadFile(docURL);
}

function downloadFile(url) {

	url += addCacheBuster(url);
	$("body").append("<iframe src='" + url + "' style='display: none;' ></iframe>");
	// Check for errors allow 5 seconds for an error to be returned
	setTimeout(downloadFileErrorCheck, 5000);
}

// Show an error generated by file download
function downloadFileErrorCheck() {
	var msg = $("iframe").last().contents().find('body').html();
	if(handleLogout(msg)) {
		if (msg && msg.indexOf("Error:") === 0) {
			alert(msg.substring(7));	// Jump over "Error: "
		} else if (msg && msg.length > 0) {
			alert(msg);
		}
	}
}

/*
 * Post data to be converted into a file
 */
function generateFile(url, filename, format, mime, data, sId, groupSurvey, title, project, charts, chartData, settings, tz, form) {

	var fd = new FormData();
	fd.append("sId", sId);
	fd.append("format", format);
	if(groupSurvey) {
		fd.append("groupSurvey", groupSurvey)
	}
	if(form) {
		fd.append("form", form);
	}
	if(data) {
		var blob = new Blob([JSON.stringify(data)], { type: 'text/plain' });
		var file = new File([blob], "foo.txt", {type: "text/plain"});
		fd.append("data", file);
		//fd.append("data", JSON.stringify(data));
	}
	if(title) {
		fd.append("title", title);
	}
	if(project) {
		fd.append("project", project);
	}
	if(charts) {
		fd.append("charts", JSON.stringify(charts));
	}
	if(chartData) {
		fd.append("chartData", JSON.stringify(chartData));
	}
	if(settings) {
		fd.append("settings", JSON.stringify(settings));
	}
	if(tz) {
		fd.append("tz",JSON.stringify(tz));
	}

	var xhr = new XMLHttpRequest();
	url += addCacheBuster(url);
	xhr.open('POST', url, true);
	xhr.setRequestHeader("X-Requested-With", "XMLHttpRequest");
	xhr.responseType = 'blob';

	xhr.onload = function(e) {
		if(handleLogout(xhr.responseURL)) {
			if (this.status == 200) {
				// get binary data as a response
				var blob = new Blob([this.response], {type: mime});
				var downloadUrl = URL.createObjectURL(blob);
				var a = document.createElement("a");
				a.href = downloadUrl;
				a.download = filename;
				document.body.appendChild(a);
				a.click();
				setTimeout(function () {
					document.body.removeChild(a);
					window.URL.revokeObjectURL(url);
				}, 100);
			} else {
				alert(localise.set["c_error"] + ": " + this.statusText);
			}
		}
	};

	xhr.onerror = function(e) {
		if(handleLogout(this)) {
			alert("Error: Upload Failed");
		}
	}

	xhr.send(fd);

}

/*
 * Get the currently selected rows of datatable data as a json array
 * Also convert the JSON object into an array of Key values pairs. This allows easy converion
 * to a java object on the server
 */
function getTableData(table, columns, format) {

	var rows = table.rows({
		order:  'current',  // 'current', 'applied', 'index',  'original'
		page:   'all',      // 'all',     'current'
		search: 'applied',     // 'none',    'applied', 'removed'
	}).data();

	var data = [],
		cols = [],
		i, j;

	for(i = 0; i < rows.length; i++) {
		cols = [];
		for(j = 0; j < columns.length; j++) {
			if(format === "xlsx" || !columns[j].hide) {
				var k = columns[j].displayName;
				var v = rows[i][columns[j].column_name];

				if (typeof v !== "string") {
					v = JSON.stringify(v);
				}
				cols.push({
					k: k,
					v: v
				})
			}
		}
		data.push(cols);
	}

	return data;


}

/*
 * Get server settings
 */
function getMapboxDefault(callback, param) {

	if(!globals.gMapboxDefault) {
		addHourglass();
		$.ajax({
			url: '/surveyKPI/server/mapbox',
			cache: false,
			success: function(data) {
				removeHourglass();
				if(handleLogout(data)) {
					globals.gMapboxDefault = data;
					if (typeof callback === "function") {
						callback(param);
					}
				}
			},
			error: function(xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert(localise.set["error"] + ": " + err);
					}
				}
			}
		});
	} else {
		if(typeof callback === "function") {
			callback(param);
		}
	}
}


/*
 * Get google map api key
 */
function getGoogleMapApi(callback, map) {

	console.log("getGoogleMapApi");

	if(!window.smapLoadedGMaps && !window.smapGMapsLoading) {
		console.log("about to call server");

		window.smapGMapsLoading = true;

		window.smapGMapsToLoad = [];
		window.smapGMapsToLoad.push({
			fn: callback,
			locn: map
		});

		addHourglass();
		$.ajax({
			url: '/surveyKPI/server/googlemaps',
			cache: false,
			success: function(data) {
				removeHourglass();
				if(handleLogout(data)) {
					console.log("Retrieved map keys from server");

					var gElement = document.createElement('script');
					var key = "";
					if (data) {
						key = "?key=" + data;
					}
					//gElement.src = "//maps.google.com/maps/api/js?v=3.6&amp";
					gElement.src = "https://maps.googleapis.com/maps/api/js" + key;
					if (typeof callback === "function") {
						gElement.onload = onLoad;
					}
					document.getElementsByTagName('head')[0].appendChild(gElement);

					function onLoad() {

						var i;

						window.smapGMapsLoading = false;
						window.smapLoadedGMaps = true;

						console.log("Google map loaded");

						for (i = 0; i < window.smapGMapsToLoad.length; i++) {
							console.log("map callback");
							window.smapGMapsToLoad[i].fn(window.smapGMapsToLoad[i].locn);
						}
						delete window.smapGMapsToLoad;
					}
				}

			},
			error: function(xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert(localise.set["error"] + " " + err);
					}
				}
			}
		});

	} else if(window.smapLoadedGMaps) {
		console.log("Already loaded calling map callback");
		callback(map);
	} else {
		console.log("Adding callback to queue");
		window.smapGMapsToLoad.push({
			fn: callback,
			locn: map
		});
	}
}

/*
 * Add google layers to a map
 */
function addGoogleMapLayers(map) {
	try {
		map.addLayer(new OpenLayers.Layer.Google("Google Satellite",{type: google.maps.MapTypeId.SATELLITE, 'sphericalMercator': true, numZoomLevels: 22}));
		map.addLayer(new OpenLayers.Layer.Google("Google Maps",{type: google.maps.MapTypeId.ROADMAP, 'sphericalMercator': true, numZoomLevels: 22}));
		map.addLayer(new OpenLayers.Layer.Google("Google Hybrid",{type: google.maps.MapTypeId.HYBRID, 'sphericalMercator': true, numZoomLevels: 22}));
	} catch (err) {
		// Fail silently, the user may not want google maps - this is probably caused by a missing maps api key
	}
}

/*
 * Get a list of custom reports
 */
function getReports(callback1, callback2, type) {

	var url="/surveyKPI/custom_reports";

	if(type) {
		url += "?type=" + type;
	}

	addHourglass();
	$.ajax({
		url: url,
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				var cb1 = callback1,
					cb2 = callback2,
					t = type;
				globals.gReports = data;
				if (typeof cb1 === "function") {
					cb1(data, cb1, cb2, t);
				}
				if (typeof cb2 === "function") {
					cb2(data, cb1, cb2, t);
				}
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					console.log("Error: Failed to get list of reports: " + err);
				}
			}
		}
	});

}

/*
 * Allow the user to pick a report
 */
function showReportList(data) {
	var h = [],
		idx = -1,
		i;

	removeHourglass();

	if(data.length === 0) {

		// Enable / disable elements specifically for managed forms
		$('.selectmanaged').show();
		$('.no_oversight').show();
	} else {
		$('.no_oversight').hide();
		$('.selectmanaged').show();

		h[++idx] = '<option value="0">';
		h[++idx] = localise.set["c_none"];
		h[++idx] = '</option>';
		for(i = 0; i < data.length; i++) {
			h[++idx] = '<option value="';
			h[++idx] = data[i].id;
			h[++idx] = '">';
			h[++idx] = htmlEncode(data[i].name);
			h[++idx] = '</option>';
		}
		$('.customReportList').empty().html(h.join(''));
	}
}

/*
 * Show the Custom Reports in a table
 */
function refreshCustomReportView(data, callback1, callback2, type) {

	var $selector = $('#cr_list'),
		i,
		h = [],
		idx = -1;

	$('.panel_msg').show();
	$('#addReportPopup').modal("hide");

	data = data || [];
	globals.gReports = data;

	h[++idx] = '<table class="table">';
	h[++idx] = '<thead>';
	h[++idx] = '<tr>';
	h[++idx] = '<th>' + localise.set["c_name"], + '</th>';
	h[++idx] = '<th>' + localise.set["c_type"] + '</th>';
	h[++idx] = '</tr>';
	h[++idx] = '</thead>';
	h[++idx] = '<tbody class="table-striped">';

	for(i = 0; i < data.length; i++) {

		h[++idx] = '<tr>';

		// name
		h[++idx] = '<td>';
		h[++idx] = htmlEncode(data[i].name);
		h[++idx] = '</td>';

		// type
		h[++idx] = '<td>';
		h[++idx] = htmlEncode(data[i].type);
		h[++idx] = '</td>';

		// actions
		h[++idx] = '<td>';

		h[++idx] = '<button type="button" data-idx="';
		h[++idx] = i;
		h[++idx] = '" class="btn btn-default btn-sm rm_cr">';
		h[++idx] = '<i class="fa fa-trash-o"></i></button>';

		h[++idx] = '<button type="button" data-idx="';
		h[++idx] = i;
		h[++idx] = '" class="btn btn-default btn-sm download_cr">';
		h[++idx] = '<i class="fa fa-download"></i></button>';

		h[++idx] = '</td>';
		// end actions

		h[++idx] = '</tr>';
	}

	h[++idx] = '</tbody>';
	h[++idx] = '</table>';

	$selector.empty().append(h.join(''));

	$(".rm_cr", $selector).click(function(){
		var idx = $(this).data("idx");
		if(confirm(localise.set["msg_confirm_del"] + " " + htmlEncode(globals.gReports[idx].name))) {
			deleteCustomReport(globals.gReports[idx].id, type);
		}
	});

	$(".download_cr", $selector).click(function(){
		var idx = $(this).data("idx");
		downloadFile("/surveyKPI/custom_reports/xls/" + globals.gReports[idx].id +
			"?filetype=xls&filename=" + cleanFileName(globals.gReports[idx].name));
	});


}

function deleteCustomReport(id, type) {

	var url = "/surveyKPI/custom_reports/" + id;
	if(type) {
		url += "?type=" + type;
	}

	addHourglass();
	$.ajax({
		type: "DELETE",
		url: url,
		success: function(data, status) {
			removeHourglass();
			if(handleLogout(data)) {
				var t = type;
				console.log("delete: " + t + " : " + type);
				getReports(refreshCustomReportView, showReportList, t);
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert(localise.set["msg_err_del"] + " " + xhr.responseText);	// alerts htmlencode text
				}
			}
		}
	});
}

/*
 * Get the list of available roles from the server
 */
function getRoles(callback) {
	addHourglass();
	$.ajax({
		url: "/surveyKPI/role/roles",
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				globals.gRoleList = data;
				if (typeof callback === "function") {
					callback();
				}
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert(localise.set["msg_err_get_r"] + " " + err);
				}
			}
		}
	});
}

/*
 * Get the list of available case management settings from the server
 */
function getCms(callback) {
	addHourglass();
	$.ajax({
		url: "/surveyKPI/cases/settings/" + globals.gCurrentSurvey,
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				globals.gCmSettings = data;
				if (typeof callback === "function") {
					callback();
				}
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert(localise.set["msg_err_get_r"] + " " + err);
				}
			}
		}
	});
}

/*
 * Clean the filename so that it can be passed in a URL
 */
function cleanFileName(filename) {

	var n;

	n = filename.replace(/\//g, '_');	// remove slashes from the filename
	n = n.replace(/[#?&]/g, '_');		// Remove other characters that are not wanted
	n = n.replace("'", "", 'g');		// Remove apostrophes

	return n;
}

/*
 * Add a list of forms to pick from during export
 */
function addFormPickList(sMeta, checked_forms) {

	var h = [],
		idx = -1,
		i;

	// Start with the top level form
	for(i = 0; i < sMeta.forms.length; i++) {
		if(sMeta.forms[i].p_id == 0) {
			$(".osmforms").html(addFormToList(sMeta.forms[i], sMeta, 0, true, false, checked_forms, false));
			$(".selectforms").html(addFormToList(sMeta.forms[i], sMeta, 0, false, false, checked_forms, false));
			$(".shapeforms,.taforms").html(addFormToList(sMeta.forms[i], sMeta, 0, true, true, checked_forms, false));
			$(".shapeforms_bs4").html(addFormToList(sMeta.forms[i], sMeta, 0, true, true, checked_forms, true));
		}
	}

	$("button",".selectforms").click(function() {
		var $this = $(this),
			$check = $this.parent().find("input"),
			val,
			val_array = [];

		val = $check.val();
		val_array= val.split(":");
		if(val_array.length > 1) {
			if(val_array[1] === "true") {
				$check.val(val_array[0] + ":false");
				$this.text("Pivot");
			} else {
				$check.val(val_array[0] + ":true");
				$this.text("Flat");
			}
			$this.toggleClass('exportflat');
			$this.toggleClass('exportpivot');
		}

		return false;
	});
}

/*
 * Add a list of date questions to pick from
 */
function addDatePickList(sMeta, currentDate) {

	var h = [],
		idx = -1,
		i,
		value,
		key;

	if(sMeta && sMeta.dates) {
		for(i = 0; i < sMeta.dates.length; i++) {

			key = sMeta.dates[i].name;

			h[++idx] = '<option value="';
			h[++idx] = sMeta.dates[i].id;
			h[++idx] = '">';
			if(key === "Upload Time" || key === "_start" || key === "_end") {
				key = localise.set[key];
			} else if(key === "Scheduled Start") {
				key = localise.set["c_scheduled"]
			}
			h[++idx] = htmlEncode(key);
			h[++idx] = '</option>';

		}

		$(".date_question").empty().html((h.join('')));

		if(typeof currentDate !== "undefined" && currentDate != 0) {
			value = currentDate;
		} else {
			value = $("#settings_date_question").val();
		}
	}
}

/*
 * Add a list of geometry questions to pick from
 */
function addGeomPickList(sMeta) {

	var h = [],
		k = [],
		idx = -1,
		i,
		value,
		theForm;

	if(sMeta && sMeta.forms) {
		for(i = 0; i < sMeta.forms.length; i++) {

			theForm = sMeta.forms[i];

			k[++idx] = h[++idx] = '<div class="exportcontrol showshape showosm" style="display: block;">';
			k[++idx] = h[++idx] = '<label>' + htmlEncode(theForm.form) + '</label>';
			h[++idx] = '<select class="geomSelect" id="geomForm_' + theForm.f_id;            // export only
			k[++idx] = '<select class="geomSelect" id="geomSettingsForm_' + theForm.f_id;    // Settings only
			k[++idx] = h[++idx] = '" data-form="' + theForm.f_id + '">';
			if(theForm.geomQuestions) {
				for(j = 0; j < theForm.geomQuestions.length; j++) {
					k[++idx] = h[++idx] = '<option value="';
					k[++idx] = h[++idx] = theForm.geomQuestions[j];
					k[++idx] = h[++idx] = '">';
					k[++idx] = h[++idx] = htmlEncode(theForm.geomQuestions[j]);
					k[++idx] = h[++idx] = '</option>';
				}
			}
			k[++idx] = h[++idx] = '</select>';
			k[++idx] = h[++idx] = '</div>';

		}

		$(".geomselect_export").empty().html((h.join('')));
		$(".geomselect_settings").empty().html((k.join('')));

		shapeFormsChanged();

	}
}

function shapeFormsChanged() {
	var formId = getSelectedForm('.shapeforms', true);
	if(formId) {
		$('.geomSelect', '.geomselect_export').prop('disabled', true);
		$('#geomForm_' + formId, '.geomselect_export').prop('disabled', false);
	}
}

function getSelectedForm($forms, ignoreError) {
	var forms = $(':radio:checked', $forms).map(function() {
		return this.value;
	}).get();
	if(forms.length === 0) {
		if(!ignoreError) {
			alert(window.localise.set["msg_one_f2"]);
		}
		return 0;
	}
	return forms[0];
}

function addFormToList(form, sMeta, offset, osm, set_radio, checked_forms, bs4) {

	var h = [],
		idx = -1,
		i,
		type,
		checked;

	if (set_radio) {
		type = "radio";
	} else {
		type = "checkbox";
	}

	// Set checked value based on previous selections
	if(set_radio && offset == 0) {
		checked = 'checked="checked"';
	} else {
		if (offset == 0 && (!checked_forms || checked_forms.length == 0)) {
			checked = 'checked="checked"';
		} else {
			checked = '';
		}
	}
	if(checked_forms && checked_forms.length > 0) {
		for(i = 0; i < checked_forms.length; i++) {
			if(form.f_id == checked_forms[i]) {
				checked = 'checked="checked"';
				break;
			}
		}
	}

	h[++idx] = '<div class="' + type + '"';
	h[++idx] = '<span style="padding-left:';
	h[++idx]= offset + 'px;">';
	h[++idx] = '<label>';
	h[++idx] = '<input class="osmform" type="' + type + '" ' + checked + ' name="osmform" value="';
	h[++idx] = form.f_id;
	if(!osm) {
		h[++idx] = ':false"/>';
	} else {
		h[++idx] = '">';
	}
	if(bs4) {
		h[++idx] = '<span class="ml-2">';
	}
	h[++idx] = htmlEncode(form.form);
	if(bs4) {
		h[++idx] = '</span>';
	}
	h[++idx] = '</label>';
	if(form.p_id != 0 && !osm) {
		h[++idx] = ' <button class="exportpivot">' + localise.set["c_pivot"] + '</button>';
	}
	h[++idx]= '</div>';

	// Add the children (recursively)
	for(i = 0; i < sMeta.forms.length; i++) {
		if(sMeta.forms[i].p_id != 0  && sMeta.forms[i].p_id == form.f_id) {
			h[++idx] = addFormToList(sMeta.forms[i], sMeta, offset + 20, osm, set_radio, checked_forms, bs4);
		}
	}

	return h.join('');
}

function getViewLanguages(view) {

	if(view.sId != -1) {
		var url = languageListUrl(view.sId);
		$.getJSON(url, function(data) {
			globals.gSelector.setSurveyLanguages(view.sId, data);
			setSurveyViewLanguages(data, view.lang, '#settings_language', false);
			setSurveyViewLanguages(data, view.lang, '#export_language', true);
		});
	}

}

function validateEmails(emails) {
	var valid = true,
		i;
	if(emails && emails.trim().length > 0) {
		var emailArray = emails.split(",");
		for (i = 0; i < emailArray.length; i++) {
			var validEmail = /[A-Z0-9._%+-]+@[A-Z0-9.-]+.[A-Z]{2,4}/igm;
			if (!validEmail.test(emailArray[i])) {
				valid = false;
				break;
			}
		}
	}
	return valid;
}

/*
 * Get the roles for a survey
 */
function getSurveyRoles(sId, selectedRoles, setall, onlypriv) {

	if (!gTasks.cache.surveyRoles[sId]) {
		addHourglass();
		var url = "/surveyKPI/role/survey/" + sId + "?enabled=true";
		if(onlypriv) {
			url += "&onlypriv=true";
		}
		$.ajax({
			url: url,
			dataType: 'json',
			cache: false,
			success: function (data) {
				removeHourglass();
				if(handleLogout(data)) {
					var savedSelectedRoles = selectedRoles;
					gTasks.cache.surveyRoles[sId] = data;
					showRoles(gTasks.cache.surveyRoles[sId], savedSelectedRoles);
				}
			},
			error: function (xhr, textStatus, err) {

				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						console.log("Error: Failed to get roles for a survey: " + err);
					}
				}
			}
		});
	} else {
		showRoles(gTasks.cache.surveyRoles[sId], selectedRoles, setall);
	}
}

/*
 * Show the roles
 */
function showRoles(data, selectedRoles, setall) {

	var h = [],
		idx = -1,
		i,
		selId,
		selList = [];

	$('.role_select_roles').empty();
	if (data.length > 0) {
		for (i = 0; i < data.length; i++) {
			h[++idx] = '<div class="col-sm-10 custom-control custom-checkbox ml-2 mb-1">'
			h[++idx] = '<input type="checkbox"';
			selId = 'rolesel_' + i;
			h[++idx] = ' id="' + selId + '"';
			if(setall || roleSelected(data[i].id, selectedRoles)) {
				selList.push(selId);
			}
			h[++idx] = ' class="custom-control-input" value="';
			h[++idx] = data[i].id;
			h[++idx] = '">';

			h[++idx] = '<label class="custom-control-label"';
			h[++idx] = ' for="rolesel_' + i + '">';
			h[++idx] = 	htmlEncode(data[i].name);
			h[++idx] = '</label>';
			h[++idx] = '</div>';
		}
		$('.role_select').show();
		$('.role_select_roles').empty().append(h.join(''));
		for(i = 0; i < selList.length; i++) {
			selId = selList[i];
			$('#' + selId).prop('checked', true);
		}
	}
}

function roleSelected(roleId, selectedRoles) {
	var sel = false;
	if(selectedRoles) {
		for(var i = 0; i < selectedRoles.length; i++) {
			if(selectedRoles[i].id == roleId) {
				sel = true;
				break;
			}
		}
	}
	return sel;
}

 /*
  * Get all the surveys that a user can access
  */
function getAccessibleSurveys($elem, includeNone, includeBlocked, groupsOnly, includeSelf) {

	var url="/surveyKPI/surveys";
	var hasParam = false;
	if(includeBlocked) {
		url += hasParam ? '&' : '?';
		url += 'blocked=true';
		hasParam = true;
	}
	if(groupsOnly) {
		url += hasParam ? '&' : '?';
		url += 'groups=true';
		hasParam = true;
	}

	addHourglass();
	$.ajax({
		url: url,
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				var h = [],
					idx = -1,
					i;

				if (includeNone) {
					h[++idx] = '<option value="">';
					h[++idx] = localise.set["c_none"]
					h[++idx] = '</option>';
				}

				if (includeSelf) {
					h[++idx] = '<option value="self">';
					h[++idx] = localise.set["c_self"]
					h[++idx] = '</option>';
				}
				for (i = 0; i < data.length; i++) {
					h[++idx] = '<option value="';
					h[++idx] = htmlEncode(data[i].ident);
					h[++idx] = '">';
					h[++idx] = htmlEncode(data[i].projectName);
					h[++idx] = ' : ';
					h[++idx] = htmlEncode(data[i].displayName);
					h[++idx] = '</option>';
				}
				$elem.empty().append(h.join(''));
			}

		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					console.log("Error: Failed to get list of surveys: " + err);
				}
			}
		}
	});
}

/*
 * Get all the csv files that a user can access
 */
function getAccessibleCsvFiles($elem, includeNone) {

	var url="/surveyKPI/shared/csv/files";

	addHourglass();
	$.ajax({
		url: url,
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				globals.gCsvFiles = data;
				var h = [],
					idx = -1,
					i;

				if (includeNone) {
					h[++idx] = '<option value="">';
					h[++idx] = localise.set["c_none"]
					h[++idx] = '</option>';
				}
				for (i = 0; i < data.length; i++) {
					h[++idx] = '<option value="';
					h[++idx] = i;
					h[++idx] = '">';
					h[++idx] = htmlEncode(data[i].filename);
					h[++idx] = '</option>';
				}
				$elem.empty().append(h.join(''));
			}

		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					console.log("Error: Failed to get list of csv files: " + err);
				}
			}
		}
	});
}

 /*
  * Get the questions in a survey
  */
function getQuestionsInSurvey($elem, $elem_multiple, sIdent, includeNone, textOnly, callback, includeHrk) {

	function populateElement($elem, $elem_multiple, data) {
		var h = [],
			hm = [],
			idx = -1,
			idx_m = -1,
			i,
			setValueFn = callback;

		if (includeNone) {
			h[++idx] = '<option value="0">';
			h[++idx] = localise.set["c_none"];
			h[++idx] = '</option>';
		}
		if (includeHrk) {
			hm[++idx_m] = h[++idx] = '<option value="_hrk">';
			hm[++idx_m] = h[++idx] = localise.set["ed_hrk"];
			hm[++idx_m] = h[++idx] = '</option>';

			hm[++idx_m] = h[++idx] = '<option value="_assigned">';
			hm[++idx_m] = h[++idx] = localise.set["t_assigned"];
			hm[++idx_m] = h[++idx] = '</option>';
		}
		for (i = 0; i < data.length; i++) {
			if(!textOnly || isTextStorageType(data[i].type)) {
				hm[++idx_m] = h[++idx] = '<option value="';
				hm[++idx_m] = h[++idx] = data[i].name;
				hm[++idx_m] = h[++idx] = '">';
				hm[++idx_m] = h[++idx] = htmlEncode(data[i].name);
				hm[++idx_m] = h[++idx] = '</option>';
			}
		}
		if($elem) {
			$elem.empty().append(h.join(''));
		}
		if($elem_multiple) {
			$elem_multiple.empty().append(hm.join(''));
			$elem_multiple.multiselect('deselectAll', false);
			$elem_multiple.multiselect('rebuild');
		}

		if(typeof setValueFn === "function") {
			setValueFn();
		}
	}

	if(sIdent === 'self') {
		populateElement($elem, $elem_multiple, globals.model.survey.forms[globals.gFormIndex].questions);
	} else if(gCache[sIdent]) {
		populateElement($elem, $elem_multiple, gCache[sIdent]);
	} else {
		if (sIdent && sIdent !== "0" && sIdent !== '' && sIdent !== '_none') {
			addHourglass();
			$.ajax({
				url: "/surveyKPI/questionListIdent/" + sIdent + "/none?inc_meta=true",
				dataType: 'json',
				cache: false,
				success: function (data) {
					removeHourglass();
					if(handleLogout(data)) {
						var theIdent = sIdent;
						var $theElem = $elem;
						var $theElemMultiple = $elem_multiple;

						gCache[theIdent] = data;
						populateElement($theElem, $theElemMultiple, data);
					}
				},
				error: function (xhr, textStatus, err) {
					removeHourglass();
					if(handleLogout(xhr.responseText)) {
						if (xhr.readyState == 0 || xhr.status == 0) {
							return;  // Not an error
						} else {
							alert(localise.set["msg_err_get_q"] + ": " + err);
						}
					}
				}
			});
		} else {
			if (includeNone) {
				if($elem) {
					$elem.empty().append('option value="0">' + localise.set["c_none"] + '</option>');
				}
				if($elem_multiple) {
					$elem_multiple.empty().append('option value="0">' + localise.set["c_none"] + '</option>');
					$elem_multiple.multiselect('rebuild');
				}
			}
		}
	}

}

function getQuestionsInCsvFile($elem, $elem_multiple, index, includeNone) {
	var h = [],
		hm = [],
		idx = -1,
		idx_m = -1,
		i;

	if(globals.gCsvFiles[index]) {
		var data = globals.gCsvFiles[index].headers;

		if (includeNone) {		// Only include select none for single selects
			h[++idx] = '<option value="">';
			h[++idx] = localise.set["c_none"];
			h[++idx] = '</option>';
		}
		for (i = 0; i < data.length; i++) {
			hm[++idx_m] = h[++idx] = '<option value="';
			hm[++idx_m] = h[++idx] = data[i].fName;
			hm[++idx_m] = h[++idx] = '">';
			hm[++idx_m] = h[++idx] = htmlEncode(data[i].fName);
			hm[++idx_m] = h[++idx] = '</option>';
		}
		if ($elem) {
			$elem.empty().append(h.join(''));
		}
		if ($elem_multiple) {
			$elem_multiple.empty().append(hm.join(''));
			$elem_multiple.multiselect('deselectAll', false)
			$elem_multiple.multiselect('rebuild');
		}
	}
}

/*
 * Get the questions in a survey
 */
function getGroupQuestionsInSurvey($elem, sIdent) {

	function populateElement($elem, data) {
		var h = [],
			idx = -1,
			i;

		h[++idx] = '<option data-type="" value="">';
		h[++idx] = localise.set["c_none"];
		h[++idx] = '</option>';

		for (i = 0; i < data.length; i++) {
			h[++idx] = '<option data-type="';
			h[++idx] = data[i].type;
			h[++idx] = '" value="';
			h[++idx] = data[i].name;
			h[++idx] = '">';
			h[++idx] = htmlEncode(data[i].name);
			h[++idx] = '</option>';
		}
		$elem.empty().append(h.join(''));
	}

	if(gCacheGroup[sIdent]) {
		populateElement($elem, gCacheGroup[sIdent]);
	} else {
		if (sIdent !== "0") {
			addHourglass();
			$.ajax({
				url: "/surveyKPI/questionListIdent/" + sIdent + "/none/group",
				dataType: 'json',
				cache: false,
				success: function (data) {
					removeHourglass();
					if(handleLogout(data)) {
						var theIdent = sIdent;
						var $theElem = $elem;

						gCacheGroup[theIdent] = data;
						populateElement($theElem, data);
					}

				},
				error: function (xhr, textStatus, err) {
					removeHourglass();
					if(handleLogout(xhr.responseText)) {
						if (xhr.readyState == 0 || xhr.status == 0) {
							return;  // Not an error
						} else {
							alert(localise.set["msg_err_get_q"] + ": " + err);
						}
					}
				}
			});
		} else {
			if (includeNone) {
				$elem.empty().append('option value="0">' + localise.set["c_none"] + '</option>');
			}
		}
	}

}

/*
 * Get the questions suitable for use as a status in a survey group using the survey id as the key
 */
function getGroupStatusQuestions($elem, sId) {

	function populateElement($elem, data) {
		var h = [],
			idx = -1,
			i;

		for (i = 0; i < data.length; i++) {
			h[++idx] = '<option value="';
			h[++idx] = data[i].column_name;
			h[++idx] = '">';
			h[++idx] = htmlEncode(data[i].name);
			h[++idx] = '</option>';
		}
		$elem.empty().append(h.join(''));
	}

	if(gCacheStatusQuestions[sId]) {
		populateElement($elem, gCacheStatusQuestions[sId]);
	} else {
		addHourglass();
		$.ajax({
			url: "/surveyKPI/questionList/" + sId + "/none/group?status=true",
			dataType: 'json',
			cache: false,
			success: function (data) {
				removeHourglass();
				if(handleLogout(data)) {
					var theId = sId;
					var $theElem = $elem;

					gCacheStatusQuestions[theId] = data;
					populateElement($theElem, data);
				}

			},
			error: function (xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert(localise.set["msg_err_get_q"] + ": " + err);
					}
				}
			}
		});
	}
}

/*
 * Get the questions suitable for use as a status in a survey group using the survey id as the key
 */
function getGroupKeys($key, $key_policy, sId) {

	if(gCacheKeys[sId]) {
		$key.val(gCacheStatusQuestions[sId].key);
		$key_policy.val(gCacheStatusQuestions[sId].key_policy)
	} else {
		addHourglass();
		$.ajax({
			url: "/surveyKPI/cases/keys/" + sId,
			dataType: 'json',
			cache: false,
			success: function (data) {
				removeHourglass();
				if(handleLogout(data)) {
					var theId = sId;

					gCacheStatusQuestions[theId] = data;
					$key.val(gCacheStatusQuestions[sId].key);
					$key_policy.val(gCacheStatusQuestions[sId].key_policy);
				}

			},
			error: function (xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert(localise.set["c_error"] + ": " + err);
					}
				}
			}
		});
	}
}

function tokenizeAppearance(input) {
	var chunks = [];
	var tokens = [];
	var chunkTokens = [];
	var i;
	var j;
	var chunk;

	// only search/lookup_choices needs special treatment
	var idx1 = input.indexOf('search');
	if(idx1 < 0) {
		idx1 = input.indexOf('lookup_choices');
	}
	if(idx1 >= 0) {
		chunks.push({
			val:input.substring(0, idx1),
			type: "text"
		});
		if(idx1 < input.length) {
			var idx2 = input.lastIndexOf(')');
			if(idx2 >= 0) {
				chunks.push({
					val: input.substring(idx1, idx2 + 1),
					type: "fn"
				});
				if(idx2 < input.length) {
					chunks.push({
						val: input.substring(idx2 + 1),
						type: "text"
					});
				}
			}
		}
	} else {
		chunks.push({
			val: input,
			type: "text"
		});
	}
	for(i = 0; i < chunks.length; i++) {
		chunk = chunks[i].val.trim();
		if(chunk.length > 0) {
			if(chunks[i].type === "text") {
				chunkTokens = chunk.split(/(\s+)/);
			} else {
				chunkTokens = [];
				chunkTokens.push(chunk);
			}
			for(j = 0; j < chunkTokens.length; j++) {
				if(chunkTokens[j].trim().length > 0) {
					tokens.push(chunkTokens[j].trim());
				}
			}
		}
	}
	return tokens;
}

function setOrganisationTheme() {

	if(globals.gSetAsTheme && globals.gOrgId > 0) {

		var mainLogoSrc = getFromLocalStorage("main_logo");
		var logo = "/media/organisation/" + globals.gOrgId + '/settings/mainLogo';
		if(mainLogoSrc !== logo) {
			setInLocalStorage('main_logo', logo);
			$('.main_logo').attr("src", "/media/organisation/" + globals.gOrgId + '/settings/mainLogo');
		}

		// navbar color
		var navbarColor = getFromLocalStorage("navbar_color");
		if(navbarColor !== globals.gNavbarColor) {
			setInLocalStorage('navbar_color', globals.gNavbarColor);
		}
		// navbar color
		var navbarTextColor = getFromLocalStorage("navbar_text_color");
		if(navbarTextColor !== globals.gNavbarTextColor) {
			setInLocalStorage('navbar_text_color', globals.gNavbarTextColor);
		}
	} else {
		// remove styles
		var navbarColorElement = document. getElementById("navbar_color");
		if(navbarColorElement) {
			navbarColorElement.parentNode.removeChild(navbarColorElement);
		}
		setInLocalStorage('navbar_color', undefined);
		setInLocalStorage('navbar_text_color', undefined);
		setInLocalStorage('main_logo', undefined);

		// Set the default logo
		if(typeof setCustomMainLogo === "function") {
			setCustomMainLogo();
		}
	}
}

/*
 * Surround get / set from local storage in case user has disabled local sorage reading in browser settings
 */
function getFromLocalStorage(key) {
	var value;
	try {
		value = localStorage.getItem(key);
	} catch (e) {

	}
	return value;
}

function setInLocalStorage(key, value) {
	try {
		localStorage.setItem(key, value);
	} catch(e) {

	}
}

function populateTaskGroupList() {
	if (typeof globals.gCurrentProject !== "undefined" && globals.gCurrentProject != -1) {
		addHourglass();
		$.ajax({
			url: "/surveyKPI/tasks/taskgroups/" + globals.gCurrentProject,
			cache: false,
			dataType: 'json',
			success: function (taskgroups) {
				removeHourglass();
				if(handleLogout(taskgroups)) {
					var h = [],
						idx = -1,
						i,
						grp,
						firstTg,
						hasCurrentTg = false;

					window.gTaskGroups = taskgroups;   // Keep the task group list

					if (typeof taskgroups != "undefined" && taskgroups.length > 0) {

						for (i = 0; i < taskgroups.length; i++) {
							grp = taskgroups[i];
							h[++idx] = '<option value="';
							h[++idx] = i;
							h[++idx] = '">';
							h[++idx] = htmlEncode(grp.name);
							h[++idx] = '</option>';
						}
					}
					$('.task_group_select').html(h.join(''));
				}
			},
			error: function (xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert("Failed to get task group data");
					}
				}
			}
		});
	}
}

/*
 * Show a loaded file as an image
 * From https://codepen.io/adamrifai/pen/YXdEwz
 */
function displayAsImage(file, img) {

	var imgURL = URL.createObjectURL(file);
	img.onload = function() {
		URL.revokeObjectURL(imgURL);
	};

	img.src = imgURL;
}

/*
 * If debug=yes is passed as a parameter then enable debuging statement
 */
function enableDebugging() {

	if(location.search.indexOf("debug=yes") >= 0) {
		$(document).on('click', function(e) { console.log(e.target) });
	}

}

/*
 * ----------------------------------------------------
 * Common task functions shared between task managmeent page and console
 */
function setupAssignType(user_id, role_id, emails, email_question) {
	$('.assign_group').hide();
	$('.assign_type').removeClass('active');
	if(user_id != 0) {
		$('.user_type_checkbox').addClass('active');
		$('.assign_user').show();
	} else  if(role_id != 0) {
		$('.role_type_checkbox').addClass('active');
		$('.assign_role').show();
	} else if((typeof emails !== "undefined" && emails.trim().length > 0)
			|| (typeof email_question !== "undefined" && email_question.trim().length > 0)) {
		$('.email_type_checkbox').addClass('active');
		$('.assign_email').show();
	} else {        // Default to user
		$('.user_type_checkbox').addClass('active');
		$('.assign_user').show();
	}
}

// Convert a location name into a location index
function getLocationIndex(name, tags) {
	var idx = -1,
		i;

	if(tags) {
		for(i = 0; i < tags.length; i++) {
			if(tags[i].name == name) {
				idx = i;
				break;
			}

		}
	}
	return idx;

}

function saveTask(isConsole, currentTaskFeature, saveType, updateId, callback, tg_id) {
	var url = "/surveyKPI/api/tasks?preserveInitialData=true&tz=UTC",	// Assume we use UTC times in interface
		taskFeature = {
			properties: {}
		},
		fromDate,
		toDate,
		MIN_SHOW_RANGE = 10;

	taskFeature = $.extend(true, {}, currentTaskFeature);
	taskFeature.properties.assignee_ident = undefined;
	taskFeature.properties.assignee_name = undefined;

	/*
	 * Set the properties of the taskFeature from the dialog
	 */
	taskFeature.properties.p_id = globals.gCurrentProject;
	taskFeature.properties.tg_id = tg_id;

	if (!taskFeature.properties.id || taskFeature.properties.id == "") {
		taskFeature.properties["id"] = 0;
	}
	taskFeature.properties.name = $('#tp_name').val();		// task name
	var surveyIdentifier = $('#tp_form_name').val();
	if(!surveyIdentifier) {
		alert(localise.set["msg_pss"]);
		return false;
	}
	if(isConsole) {
		taskFeature.properties.survey_ident = surveyIdentifier;	// Survey Ident
		taskFeature.properties.form_id = undefined;
	} else {
		// old fashioned
		taskFeature.properties.form_id = surveyIdentifier;	// form id
		taskFeature.properties.survey_ident = undefined;
	}

	taskFeature.properties.assign_type = $("button.assign_type.active", "#task_properties").attr("id");
	if(taskFeature.properties.assign_type == 'tp_user_type') {
		taskFeature.properties.assignee = $('#tp_user').val();
		taskFeature.properties.emails = undefined;
	} else if(taskFeature.properties.assign_type == 'tp_email_type') {
		taskFeature.properties.assignee = 0;
		taskFeature.properties.emails = $('#tp_assign_emails').val();
		if(!validateEmails(taskFeature.properties.emails)) {
			alert(localise.set["msg_inv_email"]);
			return false;
		}
	}

	if(isConsole) {
		taskFeature.properties.update_id = updateId;
		taskFeature.properties.initial_data_source = 'survey';
	}

	taskFeature.properties.repeat = $('#tp_repeat').prop('checked');
	taskFeature.properties.complete_all = $('#tp_pol').prop('checked');
	taskFeature.properties.assign_auto = $('#tp_assign_auto').prop('checked');

	fromDate = $('#tp_from').data("DateTimePicker").date();
	toDate = $('#tp_to').data("DateTimePicker").date();

	// Validate dates
	if(toDate && !fromDate) {       // Can't have a to date without a from date
		alert(localise.set["msg_no_from"]);
		return false;
	}
	if(toDate && fromDate && fromDate > toDate) {       // To date must be after from date
		alert(localise.set["msg_sel_dates"]);
		return false;
	}

	if (fromDate) {
		taskFeature.properties.from = utcTime(fromDate.format("YYYY-MM-DD HH:mm:ss"));
	}
	if (toDate) {
		taskFeature.properties.to = utcTime(toDate.format("YYYY-MM-DD HH:mm:ss"));
	}

	taskFeature.properties.location_trigger = $('#nfc_uid').val();
	taskFeature.properties.guidance = $('#tp_guidance').val();
	taskFeature.properties.show_dist = $('#tp_show_dist').val();

	/*
	 * Save location group and location name
	 */
	var locationIdx = $('#location_select').val();
	if(saveType == "nl") {
		taskFeature.properties.location_group = $('#locationGroupSave').val();
		taskFeature.properties.location_name = $('#locationSave').val();
	} else if(saveType == "ul" && locationIdx != "-1") {
		taskFeature.properties.location_group = $('.location_group_list_sel').text();
		taskFeature.properties.location_name = window.gTags[locationIdx].name;
	} else {
		taskFeature.properties.location_group = undefined;
		taskFeature.properties.location_name = undefined;
	}
	taskFeature.properties.save_type = saveType;

	/*
	 * Convert the geoJson geometry into longitude and latitude for update
	 */
	if (currentTaskFeature.geometry) {
		if (currentTaskFeature.geometry.coordinates && currentTaskFeature.geometry.coordinates.length > 1) {
			//taskFeature.properties.location = "POINT(" + gCurrentTaskFeature.geometry.coordinates.join(" ") + ")";  // deprecate
			taskFeature.properties.lon = currentTaskFeature.geometry.coordinates[0];
			taskFeature.properties.lat = currentTaskFeature.geometry.coordinates[1];

		} else {
			//taskFeature.properties.location = "POINT(0 0)"; // deprecate
			taskFeature.properties.lon = 0;
			taskFeature.properties.lat = 0;
		}
	}

	// TODO task update details (updating existing record)

	// Validations
	if(typeof taskFeature.properties.show_dist === "undefined") {
		taskFeature.properties.show_dist = 0;
	} else {
		taskFeature.properties.show_dist = +taskFeature.properties.show_dist;
	}
	if (taskFeature.properties.show_dist && taskFeature.properties.show_dist < MIN_SHOW_RANGE) {
		alert(localise.set["msg_val_show_dist"]);
		$('#tp_show_dist').focus();
		return;
	}


	var tpString = JSON.stringify(taskFeature.properties);

	addHourglass();
	$.ajax({
		type: "POST",
		dataType: 'text',
		cache: false,
		contentType: "application/x-www-form-urlencoded",
		url: url,
		data: {task: tpString},
		success: function (data, status) {
			removeHourglass();
			if(handleLogout(data)) {
				$('#task_properties').modal("hide");
				callback();
			}
		},
		error: function (xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				alert(localise.set["msg_err_upd"] + " " + xhr.responseText);	// Alerts htmlencode text already
			}
		}
	});
}

/*
 * Get the list of users from the server so they can be assigned to tasks
 */
function getTaskUsers(projectId) {
	var $users = $('.users_select,#users_filter'),
		i, user,
		h = [],
		idx = -1;

	$users.empty();
	$('#users_filter').append('<option value="0">' + localise.set["t_au"] + '</options>');

	$('#users_select_new_task, #users_task_group, #users_select_user, #tp_user')
		.append('<option value="-1">' + localise.set["t_u"] + '</options>');

	$('#users_task_group').append('<option value="-2">' + localise.set["t_ad"] + '</options>');
	$.ajax({
		url: "/surveyKPI/userList",
		cache: false,
		success: function (data) {

			if(handleLogout(data)) {
				for (i = 0; i < data.length; i++) {
					user = data[i];
					// Check that this user has access to the project

					if (!projectId || userHasAccessToProject(user, projectId)) {
						h[++idx] = '<option value="';
						h[++idx] = user.id;
						h[++idx] = '">';
						h[++idx] = htmlEncode(user.name);
						h[++idx] = '</option>';
					}
				}
				$users.append(h.join(''));
			}
		},
		error: function (xhr, textStatus, err) {
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert(localise.set["c_error"] + err);
				}
			}
		}
	});
}

function userHasAccessToProject(user, projectId) {
	var i;
	if(user.projects) {
		for (i = 0; i < user.projects.length; i++) {
			if (user.projects[i].id == projectId) {
				return true;
			}
		}
	}
	return false;
}

function setupTaskDialog() {
	$('#tp_email_type, #assign_email_type').click(function() {
		$('.assign_type').removeClass('active');
		$(this).addClass('active');

		$('.assign_user, .assign_role,.assign_data').hide();
		$('.assign_email').show();
		$('#assign_data').prop('placeholder', localise.set['n_eqc']);
		$('.assign_data').show();
	});
	$('#tp_user_type, #assign_user_type').click(function() {
		$('.assign_type').removeClass('active');
		$(this).addClass('active');

		$('.assign_user').show();
		$('.assign_role,.assign_email').hide();
		if($('#users_task_group').val() == -2) {
			$('#assign_data').prop('placeholder', "");
			$('.assign_data').show();
		} else {
			$('.assign_data').hide();
		}
	});
	$('#tp_role_type, #assign_role_type').click(function() {
		$('.assign_type').removeClass('active');
		$(this).addClass('active');

		$('.assign_user, .assign_email').hide();
		$('.assign_role').show();
		if($('#roles_task_group').val() == -2) {
			$('#assign_data').prop('placeholder', "");
			$('.assign_data').show();
		} else {
			$('.assign_data').hide();
		}
	});

	$('#tp_from').datetimepicker({
		useCurrent: false,
		locale: gUserLocale || 'en'
	});

	$('#tp_to').datetimepicker({
		useCurrent: false,
		locale: gUserLocale || 'en'
	});

	$('#tp_from').on("dp.change", function () {

		var startDateLocal = $(this).data("DateTimePicker").date(),
			endDateLocal = $('#tp_to').data("DateTimePicker").date(),
			originalStart = gCurrentTaskFeature.properties.from,
			originalEnd = gCurrentTaskFeature.properties.to,
			newEndDate,
			duration;

		if (startDateLocal) {

			if (originalEnd && originalStart) {
				duration = moment(originalEnd, "YYYY-MM-DD HH:mm:ss").diff(moment(originalStart, "YYYY-MM-DD HH:mm:ss"), 'hours');
				newEndDate = startDateLocal.add(duration, 'hours');
				$('#tp_to').data("DateTimePicker").date(newEndDate);
			}
		}



	});

}

function getStatusClass(status, assign_auto) {

	var statusClass = "";

	if (status === "new") {
		if(assign_auto) {
			statusClass = "bg-orange";
		} else {
			statusClass = "bg-info";
		}
	} else if (status === "submitted" || status === "success") {
		statusClass = "bg-success";
	} else if (status === "late") {
		statusClass = "bg-danger";
	} else if (status === "accepted" || status === "pending") {
		statusClass = "bg-warning";
	} else 	if (status === "error" || status === "unsent" || status === "unsubscribed"
		|| status === "blocked" || status === "rejected" || status === "cancelled" || status === "deleted") {
		statusClass = "bg-rejected";
	} else {
		statusClass = "bg-success";
	}
	return statusClass;
}

/*
 *------------------------------------------------------------------
 * Common notification functions shared between console and notifications
 */
function edit_notification(edit, idx, inconsole) {

	var notification;
	var title;

	document.getElementById("notification_edit_form").reset();

	if(edit) {
		notification = window.gNotifications[idx];

		$('#bundle').prop('checked', notification.bundle);
		$('#addNotificationLabel').text(localise.set["msg_edit_notification"]);
		$('#trigger').val(notification.trigger);
		$('#target').val(notification.target);
		$('#name').val(notification.name);
		setTargetDependencies(notification.target);
		$('.assign_question').hide();
		if(notification.target === 'escalate' && notification.remote_user === '_data') {
			$('.assign_question').removeClass('d-none').show();
		}

		gSelectedOversightQuestion = notification.updateQuestion;
		gSelectedOversightSurvey = notification.updateSurvey;
		setTriggerDependencies(notification.trigger);
		setAttachDependencies(notification.notifyDetails.attach);

		if (notification.trigger !== "task_reminder") {
			if(notification.bundle) {
				$('#bundle_survey').val(notification.bundle_ident).change();
			} else {
				$('#survey').val(notification.s_id).change();
			}
		}
		$('#not_filter').val(notification.filter);
		$('#update_value').val(notification.updateValue);
		$('#alerts').val(notification.alert_id);
		$('#sc_question').val(notification.updateQuestion);
		$('#sc_value').val(notification.updateValue);

		// reminder settings
		if (!inconsole) {
			$('#task_group').val(getTaskGroupIndex(notification.tgId));
			if ((notification.period)) {
				var periodArray = notification.period.split(" ");
				if (periodArray.length > 1) {
					$('#r_period').val(periodArray[0]);
					$('#period_list_sel').val(periodArray[1]);
				}
			}
			if(notification.trigger === "task_reminder") {
				taskGroupChanged($('#task_group').val(), notification.notifyDetails.emailQuestionName, notification.notifyDetails.emailMeta);
			}
		}

		// Periodic settings
		$('#periodic_period').val(notification.periodic_period);
		$('#periodic_time').val(notification.periodic_time);
		$('#periodic_week_day').val(notification.periodic_week_day);
		$('#periodic_month_day').val(notification.periodic_month_day);
		$('#periodic_month, #periodic_month_quarter').val(notification.periodic_month);
		$('#report').val(notification.r_id);
		setPeriodDependencies(notification.periodic_period);

		if(notification.trigger !== "task_reminder" && (typeof notification.alert_id !== 'undefined'
			|| (notification.notify_details && (notification.notifyDetails.emailQuestionName || notification.notifyDetails.emailMeta)))) {

				surveyChangedNotification(notification.notifyDetails.emailQuestionName,
					notification.notifyDetails.assign_question,
					notification.notifyDetails.emailMeta,
					notification.alert_id,
					notification.updateQuestion,
					notification.notifyDetails.survey_case);
		}

		if (notification.notifyDetails) {

			if (notification.target == "email" || notification.target == "escalate") {
				if (notification.notifyDetails.emails) {
					$('#notify_emails').val(notification.notifyDetails.emails.join(","));
				}
				$('#assigned_user').prop('checked', notification.notifyDetails.emailAssigned);
				$('#email_subject').val(notification.notifyDetails.subject);
				$('#email_content').val(notification.notifyDetails.content);
				$('#email_attach').val(notification.notifyDetails.attach);
				$('#include_references').prop('checked', notification.notifyDetails.include_references);
				$('#launched_only').prop('checked', notification.notifyDetails.launched_only);
			} else if (notification.target == "sms") {
				if (notification.notifyDetails.emails) {
					$('#notify_sms').val(notification.notifyDetails.emails.join(","));
				}
				$('#sms_content').val(notification.notifyDetails.content);
				$('#sms_attach').val(notification.notifyDetails.attach);
				$('#sms_sender_id').val(notification.notifyDetails.subject);
			} else if (notification.target == "conversation") {
				if (notification.notifyDetails.emails) {
					$('#notify_sms').val(notification.notifyDetails.emails.join(","));
				}
				$('#conversation_text').val(notification.notifyDetails.content);
			} else if (notification.target == "webhook") {
				$('#callback_url').val(notification.notifyDetails.callback_url);
			}
		}
		if (!inconsole) {
			$('#fwd_user,#user_to_assign').val(notification.remote_user).change();
			$('#assign_question').val(notification.notifyDetails.assign_question);
			$('#survey_case').val(notification.notifyDetails.survey_case);
			gEligibleUser = notification.remote_user;
			// Password not returned from server - leave blank

			$('#fwd_host').val(notification.remote_host);

			// assign user from data
			if($('#user_to_assign').val() === '_data') {
				$('.assign_question').removeClass('d-none').show();
			}

			if (notification.enabled) {
				$('#nt_enabled').prop('checked', true);
			} else {
				$('#nt_enabled').prop('checked', false);
			}
		}

		window.gUpdateFwdPassword = false;
		window.gSelectedNotification = notification.id;
	} else {

		$('#fwd_host').val(window.gRemote_host);	// Set the values to the one's last used
		$('#fwd_user').val(window.gRemote_user);

		$('#survey').change();

		setTargetDependencies('email');
		setTriggerDependencies('submission');

		// Reminders
		$('#r_period').val(1);
		$('#period_list_sel').val('days');
		$('#nt_enabled').prop('checked',true);
		window.gUpdateFwdPassword = true;
		window.gSelectedNotification = -1;
	}
	bundleSelectChanged();

}

function bundleSelectChanged() {
	if($('#bundle').is(':checked')) {
		$('.bundle').show();
		$('.notbundle').hide();
	} else {
		$('.bundle').hide();
		$('.notbundle').show();
	}
}
function setTargetDependencies(target) {
	$('.sms_options, .webhook_options, .email_options, .escalate_options, .conv_options').hide();
	if(target === "email") {
		$('.email_options').show();
		initMsgNotPopup(target);
	} else if(target === "sms") {
		$('.sms_options').show();
	} else if(target  === "webhook") {
		$('.webhook_options').show();
	} else if(target  === "escalate") {
		$('.escalate_options,.email_options').show();
	} else if(target  === "conversation") {
		$('.conv_options').show();
		initMsgNotPopup(target);
	}
}

function setTriggerDependencies(trigger) {
	$('.task_reminder_options,.update_options, .submission_options, .cm_alert_options, .periodic_options, .sc_options').hide();
	if(trigger === "submission") {
		$('.submission_options').show();
	} else if(trigger === "task_reminder") {
		$('.task_reminder_options').show();
		$('#target').val('email');
		setTargetDependencies('email');
	} else if(trigger === "cm_alert") {
		$('.cm_alert_options').show();
	} else if(trigger === "periodic") {
		$('.periodic_options').show();
	} else if(trigger === "server_calc") {
		$('.sc_options').show();
	}
}

function setAttachDependencies(attach) {
	if(attach === "pdf" || attach === "pdf_landscape") {
		$('.pdf_options').show();
	} else  {
		$('.pdf_options').hide();
	}
}

function setPeriodDependencies(period) {
	$('.periodic_week_day, .periodic_month_day, .periodic_month, .periodic_month_quarter').hide();
	if(period === "weekly") {
		$('.periodic_week_day').show();
	} else if(period === "monthly") {
		$('.periodic_month_day').show();
	} else if(period === "yearly") {
		$('.periodic_month').show();
	} else if(period === "quarterly") {
		$('.periodic_month_quarter').show();
	}
}

/*
 * Initialise notification popup
 * Only required if the eDitRecord variable is set as used in immediate notifications
 */
function initMsgNotPopup(target) {
	if(window.gEditRecord) {
		var $msg = $('#msg_cur_nbr');
		var $email = $('#email_cur');
		var other = localise.set["c_other"];

		$('.other_msg').hide();
		$('.recvd_emails').hide();

		$msg.empty();
		var hasSelect = false;
		var hasEmailSelect = false;
		if (window.gEditRecord.contacts) {
			for (const [key, value] of Object.entries(window.gEditRecord.contacts)) {
				// Hack fix up channel for old entries, its either sms or email
				if (!value.channel) {
					value.channel = (key.indexOf("@") > 0) ? 'email' : 'sms';
				}

				if (!value.channel || value.channel === 'sms' || value.channel === 'whatsapp') {
					hasSelect = true;
					$msg.append(`<option data-channel="${value.channel}" value="${key}">${key} - ${value.channel} </option>`);
				} else {
					hasEmailSelect = true;
					$email.append(`<option value="${key}">${key}</option>`);
				}
				setOurNumbersList();
			}
		}
		$msg.append(`<option value="other">${other}</option>`);
		$email.append(`<option value="other">${other}</option>`);
		if(target === "conversation") {
			msgCurNbrChanged();
		}

		if(hasEmailSelect) {
			$('.recvd_emails').show();
		}

		$('#msg_cur_nbr').change(function () {
			msgCurNbrChanged();
		});

		$('#msg_channel').change(function () {
			setOurNumbersList();
		});

	}
}

/*
 * Change attribute visibility if the user select an existing number to message or selects other
 */
function msgCurNbrChanged($choice) {
	if ($('#msg_cur_nbr').val() === 'other') {
		$('.other_msg').show();
		$('#msg_channel').prop( "disabled", false);
	} else {
		$('.other_msg').hide();
		$('#msg_channel').val($('#msg_cur_nbr option:selected').attr('data-channel')).prop( "disabled", true).trigger("change");
	}
}

/*
 * Update the notification list
 */
function updateNotificationTypes(data) {

	var $selector=$('#target'),
		i,
		h = [],
		idx = -1;

	for(i = 0; i < data.length; i++) {

		h[++idx] = '<option value="';
		h[++idx] = data[i];
		h[++idx] = '">';
		h[++idx] = localise.set["c_" + data[i]];
		h[++idx] = '</option>';
	}

	$selector.empty().append(h.join(''));
	gConversationalSMS = false;

}

/*
 * Load the existing notifications from the server
 */
function getNotificationTypes(page) {

	addHourglass();
	$.ajax({
		url: '/surveyKPI/notifications/types?page=' + page,
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				window.gNotificationTypes = data;
				if (data) {
					updateNotificationTypes(data);
					if(gTasks && gTasks.cache && gTasks.cache.currentData) {
						updateConversationalSMS(gTasks.cache.currentData.sms);
					}
				}
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					console.log("Error: Failed to get list of notification types: " + err);
				}
			}
		}
	});
}

/*
 * Update anything related to using conversations and SMS
 */
function updateConversationalSMS(sms) {
	if(sms && !gConversationalSMS) {  // Add if there is SMS data associated with this survey and the type has not already been added
		var $selector=$('#target'),
			h = [],
			idx = -1;


		h[++idx] = '<option value="conversation">';
		h[++idx] = localise.set["c_conversation"];
		h[++idx] = '</option>';

		$selector.append(h.join(''));
		gConversationalSMS = true;
	}
}

function setupNotificationDialog() {

	// Set change function trigger
	$('#trigger').off().change(function() {
		var trigger = $(this).val();
		setTriggerDependencies(trigger);
		if(trigger === "task_reminder") {
			taskGroupChanged($('#task_group').val());
		}
		if(trigger === "console_update") {
			getGroupSurveys($('#survey').val(), showOversightSurveys);
		}
	});
	setTriggerDependencies("submission");

	// Set change function target
	$('#target').off().change(function() {
		setTargetDependencies($(this).val());
	});
	setTargetDependencies("email");

	// Set change function attach
	$('#email_attach').off().change(function() {
		setAttachDependencies($(this).val());
	});

	// Set dependencies on a periodic trigger period change
	setPeriodDependencies($('#period_period').val());
	$('#periodic_period').off().change(function() {
		setPeriodDependencies($(this).val());
	});


	// Set focus on notification name when edit notification is opened
	$('#addNotificationPopup').on('shown.bs.modal', function () {
		$('#name').focus();
	});

	/*
	 * Functions for forwarding
	 */
	$('#fwd_host').change(function(){
		var host = $(this).val();
		if(host.length === 0) {
			return false;
		} else if(host.substr(0, 4) !== "http") {
			alert(localise.set["msg_val_prot"]);
			return false;
		}
	});

	$('#fwd_password').change(function(){
		window.gUpdateFwdPassword = true;
	});

}

/*
 Get updated question names if the task group changes
 */
function taskGroupChanged(tgIndex, emailQuestionName, emailMetaName) {

	var tg = gTaskGroups[tgIndex];
	var language = "none";
	var qList;
	var metaList;

	if(tg && tg.source_s_id) {
		qList = globals.gSelector.getSurveyQuestions(tg.source_s_id, language);
		metaList = globals.gSelector.getSurveyMeta(tg.source_s_id);
	} else {
		qList = [];
		metaList = [];
	}

	if(!qList) {
		getQuestionList(tg.source_s_id, language, 0, "-1", undefined, false,
			undefined, undefined, emailQuestionName, undefined);
	} else {
		setSurveyViewQuestions(qList, undefined, undefined, undefined, emailQuestionName, undefined, undefined, undefined);
	}

	if(!metaList) {
		getMetaList(tg.source_s_id, undefined);
	} else {
		setSurveyViewMeta(metaList, undefined);
	}
}

/*
 * Process a save notification when the target is "email"
 */
function saveEmail() {

	var notification = {};
	var emails = $('#notify_emails').val();
	var emailQuestionName = $('#email_question').val();
	var emailMetaItem = $('#email_meta').val();
	var emailAssigned = $('#assigned_user').is(':checked');
	var emailArray;
	var i;

	// validate
	// Must specifify an email
	notification.error = false;
	if((!emails || emails.trim().length == 0) && (!emailQuestionName || emailQuestionName == "-1")
		&& (!emailMetaItem || emailMetaItem == "-1") && !emailAssigned) {
		notification.error = true;
		notification.errorMsg = localise.set["msg_inv_email"];
		notification.notifyDetails = {};
	}

	// Text email must be valid email addresses
	if(emails && emails.trim().length > 0) {
		emailArray = emails.split(",");
		for (i = 0; i < emailArray.length; i++) {
			if (!validateEmails(emailArray[i])) {
				notification.error = true;
				notification.errorMsg = localise.set["msg_inv_email"];
				notification.notifyDetails = {};
				break;
			}
		}
	}

	if(!notification.error) {
		notification.target = "email";
		notification.notifyDetails = {};
		notification.notifyDetails.emails = emailArray;
		notification.notifyDetails.emailQuestionName = emailQuestionName;
		notification.notifyDetails.emailAssigned = emailAssigned;
		notification.notifyDetails.emailMeta = emailMetaItem;
		notification.notifyDetails.subject = $('#email_subject').val();
		notification.notifyDetails.content = $('#email_content').val();
		notification.notifyDetails.attach = $('#email_attach').val();
		notification.notifyDetails.include_references = $('#include_references').prop('checked');
		notification.notifyDetails.launched_only = $('#launched_only').prop('checked');
	}

	return notification;
}

/*
 * Process a save notification when the target is "sms"
 */
function saveSMS() {

	var notification = {};

	notification.target = "sms";
	notification.notifyDetails = {};
	notification.notifyDetails.emails = $('#notify_sms').val().split(",");
	notification.notifyDetails.emailQuestionName = $('#sms_question').val();
	notification.notifyDetails.subject = $('#sms_sender_id').val();
	notification.notifyDetails.content = $('#sms_content').val();
	notification.notifyDetails.attach = $('#sms_attach').val();

	return notification;
}

/*
 * Process a save notification when the target is "document"
 */
function saveDocument() {

	var notification = {};

	notification.target = "document";
	notification.notifyDetails = {};

	return notification;
}

/*
 * Process a save notification when the target is "conversation"
 */
function saveConversation(columns, theirNumber, ourNumber, msgChannel, record) {

	var notification = {};

	notification.target = "conversation";
	notification.notifyDetails = {};
	notification.notifyDetails.content = $('#conversation_text').val();
	notification.notifyDetails.emails = [theirNumber];		// Must be sent as an array
	notification.notifyDetails.ourNumber = ourNumber;
	notification.notifyDetails.msgChannel = msgChannel;

	if(!theirNumber || theirNumber.length === 0) {
		notification.error = true;
		notification.errorMsg = localise.set["msg_no_nbr"];
	}
	return notification;
}

/*
 * Process a save notification when the target is "webhook"
 */
function saveWebhook() {

	var error = false,
		callback_url,
		notification = {};

	callback_url = $('#callback_url').val();

	if(!error) {

		notification.target = "webhook";
		notification.remote_user = $('#fwd_user').val();
		notification.remote_password = $('#fwd_password').val();
		notification.notifyDetails = {};
		notification.notifyDetails.callback_url = callback_url;
		notification.update_password = window.gUpdateFwdPassword;

	} else {
		notification.error = true;
	}

	return notification;
}

/*
 * Process a save notification when the target is "escalate"
 */
function saveEscalate() {

	var error = false,
		callback_url,
		notification = {};

	if(!error) {

		notification.target = "escalate";
		notification.remote_user = $('#user_to_assign').val();


		notification.notifyDetails = {};
		notification.notifyDetails.survey_case = $('#survey_case').val();
		notification.notifyDetails.assign_question = $('#assign_question').val();

	} else {
		notification.error = true;
	}

	return notification;
}

function getTaskGroupIndex(tgId) {
	var i;
	if(gTaskGroups && gTaskGroups.length > 0 && tgId) {
		for(i = 0; i < gTaskGroups.length; i++) {
			if(gTaskGroups[i].tg_id == tgId) {
				return i;
			}
		}
	}
	return 0;
}

function surveyChangedNotification(qName, assignQuestion, metaItem, alertId, updateQuestion, surveyVal) {

	var language = "none",
		bundle = $('#bundle').is(':checked'),
		sId = $('#survey').val() || 0,
		bundle_ident = $('#bundle_survey').val(),
		qList,
		metaList,
		alertList;

	if(bundle && bundle_ident) {
		getGroupSurveys(bundle_ident, setGroupSelector, surveyVal);		// Get the surveys in the group
	} else if(sId) {
		if(!qName) {
			qName = "-1";
		}

		getGroupSurveys(sId, setGroupSelector, surveyVal);		// Get the surveys in the group

		qList = globals.gSelector.getSurveyQuestions(sId, language);
		metaList = globals.gSelector.getSurveyMeta(sId);
		alertList = globals.gSelector.getSurveyAlerts(sId);

		if(!qList) {
			getQuestionList(sId, language, 0, "-1", undefined, false,
				undefined, undefined, qName, assignQuestion, updateQuestion);
		} else {
			setSurveyViewQuestions(qList, undefined, undefined, undefined, qName, assignQuestion, undefined, updateQuestion);
		}

		if(!metaList) {
			getMetaList(sId, metaItem);
		} else {
			setSurveyViewMeta(metaList, metaItem);
		}

		if(!alertList) {
			getAlertList(sId, alertId);
		} else {
			setSurveyAlerts(alertList, alertId);
		}

	}
}

function getInitialDataLink(task) {
	var tab = [];
	idx = -1;

	tab[++idx] = '<a href="';
	tab[++idx] = getWebFormUrl(task.properties.survey_ident,
		task.properties.update_id,
		task.properties.initial_data_source,
		task.properties.id,
		task.properties.a_id);
	tab[++idx] = '" target="_blank">'
	tab[++idx] = '<i class="fa fa-file-text"></i>';	// Edit existing data
	tab[++idx] = '</a>';

	return tab.join('');
}

function getWebFormUrl(form_ident, update_id, initial_data_source, taskId, assignmentId) {
	var url,
		hasParams = false;

	initial_data_souce = initial_data_source || 'none';

	url = "/webForm/" + form_ident;

	if (update_id && initial_data_source === 'survey') {
		url += "?datakey=instanceid&datakeyvalue=" + update_id;
		url += "&viewOnly=true"
		hasParams = true;
	} else {
		url += '?taskkey=';
		url += taskId;
		hasParams = true;
	}
	url += (hasParams ? '&' : '?');
	url += 'assignment_id=';
	url += assignmentId;

	return url;
}

function taskReport(taskGroup) {
	var tz = Intl.DateTimeFormat().resolvedOptions().timeZone,
		tzParam = "",
		url = '/surveyKPI/tasks/xls/' + taskGroup,
		hasParam = false,
		statusFilterArray = $('#status_filter').val(),
		period_filter = $('#period').val();

	// Add parameters
	if (tz) {
		url += (hasParam ? '&' : '?') + "tz=" + encodeURIComponent(tz);
		hasParam = true;
	}
	if(statusFilterArray) {
		url += (hasParam ? '&' : '?') + 'inc_status=' + statusFilterArray.join(',');
		hasParam = true;
	}
	if(period_filter) {
		url += (hasParam ? '&' : '?') + 'period=' + period_filter;
		hasParam = true;
	}

	downloadFile(url);
}

/*
 * Check to see if the status of the task means it should be included
 */
function includeByStatus(statusFilter, task, excludeZeroOrigin) {

	var include = statusFilter.indexOf(task.properties.status) >= 0;
	if(!include) {
		// check for late
		if(task.properties.status === 'accepted' && isLate(task.properties.to) && statusFilter.indexOf("late") >= 0) {
			include = true;
		}
	}
	if(include && excludeZeroOrigin) {
		// Remove points with 0,0 coordinates
		include = false;
		if(task.geometry) {
			include = true;
			if(task.geometry.type === "Point" && task.geometry.coordinates[0] == 0 && task.geometry.coordinates[1] == 0) {
				include = false;
			}
		}
	}

	return include;
}

/*
 * Return true if this question stores its data in a text type column
 */
function isTextStorageType(type) {
	return type === "string" || type === "select1" || type === "barcode" || type === "calculate"
		|| type === "conversation"
		|| type === "child_form" || type === "parent_form";
}

/*
 * Get surveys in the same bundle
 */
function getGroupSurveys(surveyId, callback, surveyVal) {

	var url = "/surveyKPI/surveyResults/" + surveyId + "/groups",
		survey = surveyId;

	if(surveyId) {

		if(gTasks.cache.groupSurveys[surveyId]) {
			if(typeof callback === 'function') {
				callback(gTasks.cache.groupSurveys[surveyId], surveyVal);
			}
		} else {
			addHourglass();
			$.ajax({
				url: url,
				dataType: 'json',
				cache: false,
				success: function (data) {
					removeHourglass();
					if(handleLogout(data)) {
						gTasks.cache.groupSurveys[surveyId] = data;
						if (typeof callback === 'function') {
							callback(data, surveyVal);
						}
					}
				},
				error: function (xhr, textStatus, err) {
					removeHourglass();
					if(handleLogout(xhr.responseText)) {
						if (xhr.readyState == 0 || xhr.status == 0) {
							return;  // Not an error
						} else {
							console.log(localise.set["c_error"] + ": " + err);
						}
					}
				}
			});
		}
	}
}

/*
 * Update a selector that is used for any data survey in a group that is not an oversight form
 */
function setGroupSelector(data, surveyVal) {
	var $elemGroups = $('#survey_case, #tp_form_name, #not_form_name');

	var i,
		item,
		h = [],
		idx = -1;

	for (i = 0; i < data.length; i++) {
		item = data[i];

		if (item.dataSurvey) {
			h[++idx] = '<option value="';
			h[++idx] = item.surveyIdent;
			h[++idx] = '">';
			h[++idx] = htmlEncode(item.surveyName);
			h[++idx] = '</option>';

			if(!surveyVal) {
				surveyVal = item.surveyIdent;
			}
		}
	}

	$elemGroups.empty().html(h.join(''));
	if(surveyVal) {
		$elemGroups.val(surveyVal).change();
	}

}

function showOversightSurveys(data) {
	var i,
		item,
		h = [],
		idx = -1,
		surveyId = $('#survey').val(),
		count = 0;

	$('#oversight_survey').empty();

	for (i = 0; i < data.length; i++) {
		item = data[i];

		if (item.oversightSurvey && item.sId != surveyId) {
			h[++idx] = '<option value="';
			h[++idx] = item.surveyIdent;
			h[++idx] = '">';
			h[++idx] = htmlEncode(item.surveyName);
			h[++idx] = '</option>';

			if(count == 0) {
				if(gSelectedOversightSurvey) {
					getOversightQuestionList(gSelectedOversightSurvey, showOversightQuestions);
				} else {
					getOversightQuestionList(item.surveyIdent, showOversightQuestions);
				}
			}
			count++;
		}
	}

	if(count == 0) {
		$('.update_options_msg').html(localise.set["n_no_oversight"]);
		$('.update_options_msg').show();
	} else {
		$('.update_options_msg').hide();
	}
	$('#oversight_survey').empty().html(h.join(''));
	if(gSelectedOversightSurvey) {
		$('#oversight_survey').val(gSelectedOversightSurvey);
	}
}

//Function to get the question list
function getOversightQuestionList(sIdent, callback) {

	var url = "/surveyKPI/questionListIdent/" + sIdent + "/none?exc_read_only=false&inc_meta=false";

	if(window.oversightQuestions[sIdent]) {
		callback(window.oversightQuestions[sIdent]);
	} else {
		addHourglass();
		$.ajax({
			url: url,
			dataType: 'json',
			cache: false,
			success: function(data) {
				removeHourglass();
				if(handleLogout(data)) {
					window.oversightQuestions[sIdent] = data;
					callback(data);
				}
			},
			error: function(xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert("Error: Failed to get list of questions: " + err);
					}
				}
			}
		});
	}

}

function showOversightQuestions(data) {
	var i,
		item,
		h = [],
		idx = -1;

	for (i = 0; i < data.length; i++) {
		item = data[i];

		h[++idx] = '<option value="';
		h[++idx] = item.name;
		h[++idx] = '">';
		h[++idx] = htmlEncode(item.name);
		h[++idx] = '</option>';

	}

	$('#update_question').empty().html(h.join(''));
	if(gSelectedOversightQuestion) {
		$('#update_question').val(gSelectedOversightQuestion);
	}
}

/*
 * Convert system names for meta data into human names
 */
function translateKey(key) {
	if(key === "_device") {
		key = localise.set["c_device"];  //"Device";
	} else if (key === "_user") {
		key = localise.set["c_user"];  // "Submitted By";
	} else if (key === "_start") {
		key = localise.set["_start"] + " (" + localise.set["c_lt"] +")"; // "Start Survey";
	} else if (key === "_end") {
		key = key = localise.set["_end"] + " (" + localise.set["c_lt"] +")";  // "End Survey";
	} else if (key === "Upload Time") {
		key = key = localise.set[key] + " (" + localise.set["c_lt"] +")";
	} else if (key === "_scheduled_start") {
		key = key = localise.set[key] + " (" + localise.set["c_lt"] +")";
	} else if (key === "_bad") {
		key = localise.set["a_mb"];         // "Marked Bad";
	} else if (key === "_bad_reason") {
		key = localise.set["c_reason"];     // "Reason";
	} else if (key === "_complete") {
		key = localise.set["c_complete"];	// "Complete";
	}

	return key;
}

/*
 * Convert system names for meta values into human values
 */
function translateKeyValue(key, value) {

	if (key === "_bad") {
		if(value === "t") {
			value = localise.set["c_yes"];   // "Yes";
		} else {
			value = localise.set["c_no"];   // "No";
		}
	} else if (key === "_complete") {
		value = (value === "t") ? localise.set["c_yes"] : localise.set["c_no"];
	}

	return value;

}

function addCacheBuster(url) {
	var cb;
	if(url.indexOf("?") >= 0) {
		cb = "&";
	} else {
		cb = "?";
	}
	return cb + "_v=" + new Date().getTime().toString();
}

function getAppearanceParams(appearance) {

	var response = {};

	var idx1 = appearance.indexOf('(');
	var idx2 = appearance.lastIndexOf(')');
	var params = appearance.substring(idx1 + 1, idx2);
	var paramsArray = [];
	if(params) {
		paramsArray = params.split(',');
	}

	response.length = paramsArray.length;
	if(paramsArray.length > 0) {

		// 1. First parameter is the filename
		var filename = paramsArray[0].trim();
		response.filename = filename.replace(/'/g, "");

		response.filter = '';    // default
		if(paramsArray.length > 1) {
			// Second parameter is the filter
			response.filter = paramsArray[1].trim();
			response.filter = response.filter.replace(/'/g, "");
		}

		if(response.filter === 'eval') {
			if (paramsArray.length > 2) {
				// Third parameter for an evaluation type function is the expression
				// For an expression type filter only remove the first and last single quote if they exist
				response.expression = paramsArray[2].trim();
				if(response.expression.charAt(0) == '\'') {
					response.expression = response.expression.substring(1);
				}
				if(response.expression.charAt(response.expression.length - 1) == '\'') {
					response.expression = response.expression.substring(0, response.expression.length - 1);
				}
			}
		} else {

			if (paramsArray.length > 2) {
				// Third parameter is the filter column
				response.filter_column = paramsArray[2].trim();
				response.filter_column = response.filter_column.replace(/'/g, "");
			}

			if (paramsArray.length > 3) {
				// Fourth parameter is the filter value
				response.filter_value = paramsArray[3].trim();
				response.filter_value = response.filter_value.replace(/'/g, "");
			}

			if (paramsArray.length > 4) {
				// Fifth parameter is the second filter column
				response.second_filter_column = paramsArray[4].trim();
				response.second_filter_column = response.second_filter_column.replace(/'/g, "");
			}


			if (paramsArray.length > 5) {
				// Sixth parameter is the filter value
				response.second_filter_value = paramsArray[5].trim();
				response.second_filter_value = response.second_filter_value.replace(/'/g, "");
			}
		}

	}
	return response;
}

function getQuestionType(schema, qname) {
	var i;
	for(i = 0; i < schema.columns.length; i++) {
		if(schema.columns[i].question_name == qname) {
			return schema.columns[i].type;
		}
	}
}

function getTrailData(projectId, userId, startDate, endDate, callback, tz, mps) {

	var url = '/surveyKPI/usertrail/trail' +
		'?userId=' + userId +
		'&startDate=' + startDate +
		'&endDate=' + endDate +
		'&mps=' + (mps || 0) +
		(tz ? "&tz=" + tz : "");

	addHourglass();
	$.ajax({
		url: url,
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				callback(data);
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert("Error: Failed to get user trail: " + err);
				}
			}
		}
	});
}

/*
 * Reports
 */
function executeUsageReport(oId) {

	var usageMsec = $('#usageDate').data("DateTimePicker").date(),
		d = new Date(usageMsec),
		month = d.getMonth() + 1,
		year = d.getFullYear(),
		incTemp = $('#usage_inc_temp').prop('checked'),
		incAllTime = $('#usage_inc_alltime').prop('checked'),
		byProject = $('#usage_by_project').prop('checked'),
		bySurvey = $('#usage_by_survey').prop('checked'),
		byDevice = $('#usage_by_device').prop('checked'),
		i;

	var reportName = localise.set["u_usage"] + "_";

	// Add the organisation name
	if(oId > 0 && globals.gLoggedInUser.orgs.length > 0) {
		for(i = 0; i < globals.gLoggedInUser.orgs.length; i++) {
			if(globals.gLoggedInUser.orgs[i].id == oId) {
				reportName += globals.gLoggedInUser.orgs[i].name + "_";
				break;
			}
		}
	}

	if(byProject) {
		reportName += localise.set["c_project"];
	} else if(bySurvey) {
		reportName += localise.set["c_survey"];
	} else if(byDevice) {
		reportName += localise.set["c_device"];
	} else {
		reportName += localise.set["c_user"];
	}
	reportName += "_" + year + "_" + month;
	reportName = reportName.replaceAll(' ', '_');

	var reportObj = {
		report_type: 'u_usage',
		report_name: reportName,
		pId: 0,
		params: {
			oId: oId,
			byProject: byProject,
			bySurvey: bySurvey,
			byDevice: byDevice,
			month: month,
			year: year,
			incTemp: incTemp,
			incAllTime: incAllTime
		}
	}

	var tzString = globals.gTimezone ? "?tz=" + encodeURIComponent(globals.gTimezone) : "";

	addHourglass();
	$.ajax({
		type: "POST",
		cache: false,
		dataType: 'text',
		contentType: "application/x-www-form-urlencoded",
		url: "/surveyKPI/background_report" + tzString,
		data: { report: JSON.stringify(reportObj) },
		success: function(data, status) {
			if(handleLogout(data)) {
				removeHourglass();
				alert(localise.set["msg_ds_s_r"]);
			}
		}, error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert(localise.set["msg_err_save"] + xhr.responseText);	// alerts htmlencode
				}
			}
		}
	});

}

function executeSurveyReport(oId) {

	var i;

	var reportName = localise.set["c_survey"];

	// Add the organisation name
	if(oId > 0 && globals.gLoggedInUser.orgs.length > 0) {
		for(i = 0; i < globals.gLoggedInUser.orgs.length; i++) {
			if(globals.gLoggedInUser.orgs[i].id == oId) {
				reportName += globals.gLoggedInUser.orgs[i].name + "_";
				break;
			}
		}
	}

	var reportObj = {
		report_type: 'survey',
		report_name: reportName,
		params: {
			oId: oId
		}
	}

	var tzString = globals.gTimezone ? "?tz=" + encodeURIComponent(globals.gTimezone) : "";

	addHourglass();
	$.ajax({
		type: "POST",
		cache: false,
		dataType: 'text',
		contentType: "application/x-www-form-urlencoded",
		url: "/surveyKPI/background_report" + tzString,
		data: { report: JSON.stringify(reportObj) },
		success: function(data, status) {
			if(handleLogout(data)) {
				removeHourglass();
				alert(localise.set["msg_ds_s_r"]);
			}
		}, error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert(localise.set["msg_err_save"] + xhr.responseText);	// alerts htmlencode
				}
			}
		}
	});

}
function executeAttendanceReport(oId) {

	var attendanceMsec = $('#attendanceDate').data("DateTimePicker").date(),
		d = new Date(attendanceMsec),
		day = d.getDate(),
		month = d.getMonth() + 1,
		year = d.getFullYear(),
		i;

	var reportName = localise.set["u_attendance"] + "_";

	// Add the organisation name
	if(oId > 0 && globals.gLoggedInUser.orgs.length > 0) {
		for(i = 0; i < globals.gLoggedInUser.orgs.length; i++) {
			if(globals.gLoggedInUser.orgs[i].id == oId) {
				reportName += globals.gLoggedInUser.orgs[i].name + "_";
				break;
			}
		}
	}

	reportName += "_" + year + "_" + month + "_" + day;
	reportName = reportName.replaceAll(' ', '_');

	var reportObj = {
		report_type: 'u_attendance',
		report_name: reportName,
		pId: 0,
		params: {
			oId: oId,
			month: month,
			year: year,
			day: day
		}
	}

	var tzString = globals.gTimezone ? "?tz=" + encodeURIComponent(globals.gTimezone) : "";

	addHourglass();
	$.ajax({
		type: "POST",
		cache: false,
		dataType: 'text',
		contentType: "application/x-www-form-urlencoded",
		url: "/surveyKPI/background_report" + tzString,
		data: { report: JSON.stringify(reportObj) },
		success: function(data, status) {
			removeHourglass();
			if(handleLogout(data)) {
				alert(localise.set["msg_ds_s_r"]);
			}
		}, error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert(localise.set["msg_err_save"] + " " + xhr.responseText);  // alerts htmlencode
				}
			}
		}
	});

}

/*
 * Decode escaped HTML
 * From https://stackoverflow.com/questions/1912501/unescape-html-entities-in-javascript
 */
function htmlDecode(input) {
	var doc = new DOMParser().parseFromString(input, "text/html");
	return doc.documentElement.textContent;
}

function htmlEncode(input) {
	if(input) {
		return $('<div>').text(input).html();
	}
}

/*
 * Get the list of users from the server
 */
function getEligibleUsers(sIdent, isNotification) {

	if(window.gTasks && window.gTasks.cache.eligibleUsers[sIdent]) {
		fillUsersList(isNotification, window.gTasks && window.gTasks.cache.eligibleUsers[sIdent]);
	} else if(sIdent) {
		addHourglass();
		$.ajax({
			url: "/surveyKPI/userList/survey/" + sIdent,
			dataType: 'json',
			cache: false,
			success: function (data) {
				removeHourglass();
				if(handleLogout(data)) {
					window.gTasks.cache.eligibleUsers[sIdent] = data;
					fillUsersList(isNotification, data);
				}
			},
			error: function (xhr, textStatus, err) {
				removeHourglass();
				if(handleLogout(xhr.responseText)) {
					if (xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else if (err == 403) {
						return;  // Ignore errors where the survey cannot be found. The survey requested may be the global default current survey which may be out of date
					} else {
						alert(localise.set["error"] + ": " + err);
					}
				}
			}
		});
	}
}

/*
 * Fill a list with the users who can be selected
 */
function fillUsersList(isNotification, data) {
	var h = [],
		idx = -1,
		$elem = $('#user_to_assign');

	$elem.empty();

	h[++idx] = '<option value="_none">';
	h[++idx] = localise.set["c_none"];
	h[++idx] = '</option>';

	if (isNotification) {
		h[++idx] = '<option value="_submitter">';
		h[++idx] = localise.set["c_submitter"];
		h[++idx] = '</option>';

		h[++idx] = '<option value="_data">';
		h[++idx] = localise.set["t_ad"];
		h[++idx] = '</option>';
	}

	if (data && data.length > 0) {
		for (i = 0; i < data.length; i++) {
			h[++idx] = '<option value="';
			h[++idx] = data[i].ident;
			h[++idx] = '">';
			h[++idx] = htmlEncode(data[i].name);
			h[++idx] = '</option>';
		}
	}
	$elem.html(h.join(''));

	if (typeof gEligibleUser !== 'undefined') {
		$elem.val(gEligibleUser);
	}
}

/*
 * Return true if the passed in value is accepted by xlsFormConverter
 */
function isValidODKQuestionName(val) {

	var sqlCheck = /^[A-Za-z_][A-Za-z0-9_\-\.]*$/;
	return sqlCheck.test(val);
}

function isValidODKOptionName(val) {

	var sqlCheck = /^[A-Za-z0-9_@&\-\.\+\(\),%:\/ ]*$/;
	return sqlCheck.test(val);
}

/*
 * Check item names such as; a username or an organisation name for invalid characters
 */
function validGeneralName(val) {

	if(val.indexOf('<') >= 0 || val.indexOf('>') >= 0) {
		return false;
	}
	return true;
}

/*
 * Get the names of referenced questions in the passed in string
 */
function getReferenceNames(elem, refQuestions) {
	var names = [],
		reg = /\$\{[A-Za-z_][A-Za-z0-9_\-\.]*\}/g,
		i,
		name;

	if (elem) {
		names = elem.match(reg);
		if(names) {
			for(i = 0; i < names.length; i++) {
				if(names[i].length > 3) {
					name = names[i].substring(2, names[i].length - 1);		// Remove the curly brackets
					refQuestions[name] = {
						name: name,
						exists: false
					};
				}
			}
		}
	}
}

/*
 * Add an exists flag to each question in the references object
 */
function checkExistenceOfReferences(refQuestions, survey) {

	var refCount = 0,
		i = 0,
		j = 0,
		name,
		form;

	for (name in refQuestions) {
		if (refQuestions.hasOwnProperty(name)) {
			refCount++;
		}
	}

	if(refCount > 0) {

		for (i = 0; i < survey.forms.length; i++) {
			form = survey.forms[i];
			for (j = 0; j < form.questions.length; j++) {
				var otherItem = form.questions[j];
				var questionType = otherItem.type;
				if (!otherItem.deleted && !otherItem.soft_deleted && questionType !== "end group") {
					otherItem = form.questions[j];

					for (name in refQuestions) {
						if (refQuestions.hasOwnProperty(name)) {
							if (name === otherItem.name) {
								refQuestions[name].exists = true;
								break;
							}
						}
					}
				}
			}
		}

		// Check against preloads
		console.log("check against preloads");
		if (survey.meta) {
			for (i = 0; i < survey.meta.length; i++) {
				for (name in refQuestions) {
					if (name === survey.meta[i].name) {
						refQuestions[name].exists = true;
					}
				}
			}
		}
	}
	return refCount;
}

function checkLoggedIn(callback) {
	$.ajax({
		cache: false,
		url: "/authenticate/login.txt",
		success: function (data) {
			if(handleLogout(data)) {
				callback();
			}

		}, error: function (data, status) {
			if(handleLogout(data.responseText)) {
				alert(data);
			}
		}
	});
}

/*
 * Respond to a logged out redirect
 */
function handleLogout(data) {
	if(data) {
		if(    (data.code && data.code === 401)
			|| (data.status && data.status === 405)
			|| (data.status && data.status === 413)
			|| (typeof data === "string" && data.indexOf('"code": 401') >= 0)
			|| (typeof data === "string" && data.indexOf('Error: 401') >= 0)
			|| (typeof data === "string" && data.indexOf('Status 401 – Unauthorized') >= 0)
			|| (typeof data === "string" && data.indexOf('notloggedin.json') >= 0)
			|| (typeof data === "string" && data.toLowerCase().indexOf("method not allowed") >= 0)) {
				window.open("/login.html");
				return false;
		}

	} else if(data && (typeof data === "string" && data.indexOf('multilogon') >= 0)) {
		alert("Logon on another device detected - logging out");
		window.open("/dologout.html");
		return false;
	}
	return true;
}

/*
 * Load the sms numbers from the server
 */
function getOurNumbers() {

	var url="/surveyKPI/smsnumbers?org=true";
	addHourglass();
	$.ajax({
		url: url,
		dataType: 'json',
		cache: false,
		success: function(data) {
			removeHourglass();
			if(handleLogout(data)) {
				gNumbers = data;
				setOurNumbersList();
			}
		},
		error: function(xhr, textStatus, err) {
			removeHourglass();
			if(handleLogout(xhr.responseText)) {
				if (xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					console.log("Error: Failed to get list of sms numbers: " + err);
				}
			}
		}
	});
}

function setOurNumbersList() {
	var i = 0;
	if(gNumbers && gNumbers.length > 0) {
		var $elem = $('#msg_our_nbr');
		var channel = $('#msg_channel').val();
		$elem.empty();
		for(i = 0; i < gNumbers.length; i++) {
			var n = gNumbers[i];
			if(n.channel === channel) {
				$elem.append(`<option value="${n.ourNumber}">${n.ourNumber} - ${n.channel} </option>`);
			}
		}
	}
}
;
define("common", function(){});

/**
 * @license RequireJS i18n 2.0.4 Copyright (c) 2010-2012, The Dojo Foundation All Rights Reserved.
 * Available via the MIT or new BSD license.
 * see: http://github.com/requirejs/i18n for details
 */
/*jslint regexp: true */
/*global require: false, navigator: false, define: false */

/**
 * This plugin handles i18n! prefixed modules. It does the following:
 *
 * 1) A regular module can have a dependency on an i18n bundle, but the regular
 * module does not want to specify what locale to load. So it just specifies
 * the top-level bundle, like "i18n!nls/colors".
 *
 * This plugin will load the i18n bundle at nls/colors, see that it is a root/master
 * bundle since it does not have a locale in its name. It will then try to find
 * the best match locale available in that master bundle, then request all the
 * locale pieces for that best match locale. For instance, if the locale is "en-us",
 * then the plugin will ask for the "en-us", "en" and "root" bundles to be loaded
 * (but only if they are specified on the master bundle).
 *
 * Once all the bundles for the locale pieces load, then it mixes in all those
 * locale pieces into each other, then finally sets the context.defined value
 * for the nls/colors bundle to be that mixed in locale.
 *
 * 2) A regular module specifies a specific locale to load. For instance,
 * i18n!nls/fr-fr/colors. In this case, the plugin needs to load the master bundle
 * first, at nls/colors, then figure out what the best match locale is for fr-fr,
 * since maybe only fr or just root is defined for that locale. Once that best
 * fit is found, all of its locale pieces need to have their bundles loaded.
 *
 * Once all the bundles for the locale pieces load, then it mixes in all those
 * locale pieces into each other, then finally sets the context.defined value
 * for the nls/fr-fr/colors bundle to be that mixed in locale.
 */
(function () {
    'use strict';

    //regexp for reconstructing the master bundle name from parts of the regexp match
    //nlsRegExp.exec("foo/bar/baz/nls/en-ca/foo") gives:
    //["foo/bar/baz/nls/en-ca/foo", "foo/bar/baz/nls/", "/", "/", "en-ca", "foo"]
    //nlsRegExp.exec("foo/bar/baz/nls/foo") gives:
    //["foo/bar/baz/nls/foo", "foo/bar/baz/nls/", "/", "/", "foo", ""]
    //so, if match[5] is blank, it means this is the top bundle definition.
    var nlsRegExp = /(^.*(^|\/)nls(\/|$))([^\/]*)\/?([^\/]*)/;

    //Helper function to avoid repeating code. Lots of arguments in the
    //desire to stay functional and support RequireJS contexts without having
    //to know about the RequireJS contexts.
    function addPart(locale, master, needed, toLoad, prefix, suffix) {
        if (master[locale]) {
            needed.push(locale);
            if (master[locale] === true || master[locale] === 1) {
                toLoad.push(prefix + locale + '/' + suffix);
            }
        }
    }

    function addIfExists(req, locale, toLoad, prefix, suffix) {
        var fullName = prefix + locale + '/' + suffix;
        if (require._fileExists(req.toUrl(fullName + '.js'))) {
            toLoad.push(fullName);
        }
    }

    /**
     * Simple function to mix in properties from source into target,
     * but only if target does not already have a property of the same name.
     * This is not robust in IE for transferring methods that match
     * Object.prototype names, but the uses of mixin here seem unlikely to
     * trigger a problem related to that.
     */
    function mixin(target, source, force) {
        var prop;
        for (prop in source) {
            if (source.hasOwnProperty(prop) && (!target.hasOwnProperty(prop) || force)) {
                target[prop] = source[prop];
            } else if (typeof source[prop] === 'object') {
                if (!target[prop] && source[prop]) {
                    target[prop] = {};
                }
                mixin(target[prop], source[prop], force);
            }
        }
    }

    define('i18n',['module'], function (module) {
        var masterConfig = module.config ? module.config() : {};

        return {
            version: '2.0.4',
            /**
             * Called when a dependency needs to be loaded.
             */
            load: function (name, req, onLoad, config) {
                config = config || {};

                if (config.locale) {
                    masterConfig.locale = config.locale;
                }

                var masterName,
                    match = nlsRegExp.exec(name),
                    prefix = match[1],
                    locale = match[4],
                    suffix = match[5],
                    parts = locale.split("-"),
                    toLoad = [],
                    value = {},
                    i, part, current = "";

                //If match[5] is blank, it means this is the top bundle definition,
                //so it does not have to be handled. Locale-specific requests
                //will have a match[4] value but no match[5]
                if (match[5]) {
                    //locale-specific bundle
                    prefix = match[1];
                    masterName = prefix + suffix;
                } else {
                    //Top-level bundle.
                    masterName = name;
                    suffix = match[4];
                    locale = masterConfig.locale;
                    if (!locale) {
                        locale = masterConfig.locale =
                            typeof navigator === "undefined" ? "root" :
                            (navigator.language ||
                             navigator.userLanguage || "root").toLowerCase();
                    }
                    parts = locale.split("-");
                }

                if (config.isBuild) {
                    //Check for existence of all locale possible files and
                    //require them if exist.
                    toLoad.push(masterName);
                    addIfExists(req, "root", toLoad, prefix, suffix);
                    for (i = 0; i < parts.length; i++) {
                        part = parts[i];
                        current += (current ? "-" : "") + part;
                        addIfExists(req, current, toLoad, prefix, suffix);
                    }

                    req(toLoad, function () {
                        onLoad();
                    });
                } else {
                    //First, fetch the master bundle, it knows what locales are available.
                    req([masterName], function (master) {
                        //Figure out the best fit
                        var needed = [],
                            part;

                        //Always allow for root, then do the rest of the locale parts.
                        addPart("root", master, needed, toLoad, prefix, suffix);
                        for (i = 0; i < parts.length; i++) {
                            part = parts[i];
                            current += (current ? "-" : "") + part;
                            addPart(current, master, needed, toLoad, prefix, suffix);
                        }

                        //Load all the parts missing.
                        req(toLoad, function () {
                            var i, partBundle, part;
                            for (i = needed.length - 1; i > -1 && needed[i]; i--) {
                                part = needed[i];
                                partBundle = master[part];
                                if (partBundle === true || partBundle === 1) {
                                    partBundle = req(prefix + part + '/' + suffix);
                                }
                                mixin(value, partBundle);
                            }

                            //All done, notify the loader.
                            onLoad(value);
                        });
                    });
                }
            }
        };
    });
}());

define('lang_location/nls/lang',{
	"root": true,
	"ar": true,
	"fr": true,
	"pt": true,
	"es": true,
	"hi": true,
	"uk": true
});

define('lang_location/nls/root/lang',{
    "m_home": "Home",
    "m_admin": "Admin",
    "m_tm": "Forms",
    "m_analysis": "Analysis",
    "m_report": "Report",
    "m_reports": "Reports",
    "m_reports_def": "Default Reports",
    "m_reports_org": "Organisation Reports",
    "m_reports_leg": "Legacy Reports",
    "m_reports_new": "New Reports",
    "m_reports_public": "Public Reports",
    "m_refresh": "Refresh",
    "m_email_unsent": "Email Unsent",
	"m_email_gen": "Generate Links",
    "m_export": "Export",
    "m_export_media": "Export with Media",
    "m_backup": "Backup",
    "m_backup_media": "Backup with Media",
    "m_discuss": "Discuss",
    "m_modify": "Modify",
    "m_web_forms": "Web Forms",
    "m_docs": "Documentation",
    "m_monitor": "Monitoring",
    "m_user": "Users",
    "m_assign": "Tasks",
    "m_query": "Queries",
    "m_help": "Help",
    "m_train": "Training",
    "m_forgot_password": "Forgot Password",
    "m_register": "Register",
    "m_locations": "Locations",
    "m_section": "Section",
    "m_notify": "Notifications",
    "m_view": "View",
    "m_attach": "Load Attachments",
    "m_modules": "Modules",
    "m_lr": "Local Reports",
    "m_open": "Open",
    "m_new": "New",
    "m_import": "Import",
    "m_calculations": "Calculations",
    "m_resources": "Shared Resources",
    "m_review": "Review",
    "m_mf": "Managed Forms",
    "m_br": "Browse Results",
    "m_manage": "Manage",
    "m_dash": "Dashboard",
    "m_console": "Console",
    "m_s_m_f": "Setup Managed Forms",
    "m_task_m": "Task Management",
    "m_user_profile": "User Profile",
    "m_login": "Login",
    "m_logs": "Logs",
    "m_info": "Info",
    "m_os": "Oversight",
    "m_billing": "Billing",
    "m_rates": "Rates",
    "m_billing_server": "Server Billing",
    "m_billing_orgs": "Organisations Billing",
    "m_bill_level": "Bill Level",

    // Smap Server
    "ss_welcome": "Smap Server",
    "ss_welcome2": "Changes in this release",
    "ss_ft": "Get Field Task",
    "ss_su": "Get Smap Uploader",

    // Common
    "c_alert": "Alert",
    "c_alerts": "Alerts",
    "c_apply": "Apply",
    "c_api": "API",
    "c_api_builder": "API Builder",
    "c_audit": "Audit",
    "c_auto": "Auto",
    "c_before": "Before",
    "c_bundle": "Bundle",
    "c_cases": "Cases",
    "c_channel": "Channel",
    "c_closed": "Closed",
    "c_cm": "Bundle and Case Management",
    "c_crit": "Criticality",
    "c_default": "Default",
    "c_desc": "Description",
    "c_fingerprint": "Fingerprint",
    "c_form": "Form",
    "c_forms": "Forms",
    "c_forward": "Forward",
    "c_iid": "Instance Id",
    "c_link": "Link",
    "c_links": "Links",
    "c_logon": "Server Logon",
    "c_moved" : "Moved",
    "c_opened": "Opened",
    "c_periodic": "Periodic",
    "c_project": "Project",
    "c_sub_form": "Sub Form",
    "c_threshold": "Threshold",
    "c_url": "URL",
    "c_webhook": "Web Hook",
    "c_forwarded": "Forwarded",
    "c_name": "Name",
    "c_company_name": "Company Name",
    "c_company_addr": "Company Address",
    "c_company_phone": "Company Phone",
    "c_company_email": "Company Email",
    "c_conversation": "Message Conversation",
    "c_type": "Type",
    "c_file": "File",
    "c_host": "Host",
    "c_rank": "Rank",
    "c_report": "Report",
    "c_user": "User",
    "c_self": "Self",
    "c_sunday": "Sunday",
    "c_monday": "Monday",
    "c_tuesday": "Tuesday",
    "c_wednesday": "Wednesday",
    "c_thursday": "Thursday",
    "c_friday": "Friday",
    "c_saturday": "Saturday",
    "c_vertical": "Vertical",
    "c_projects": "Projects",
    "c_email": "Email",
    "c_escalate": "Escalate",
    "c_emails": "Emails",
    "c_sms": "SMS",
    "c_document": "Document",
    "c_twitter": "Twitter",
    "c_lang": "Language",
    "c_langs": "Languages",
    "c_pulldata": "Pulldata",
    "c_pulldata_r": "Repeating Pulldata",
    "c_orientation": "Orientation",
    "c_password": "Password",
    "c_c_p": "Confirm Password",
    "c_n_p": "New Password",
    "c_c_d": "Confirm Delete",
    "c_r_p": "Reset Password",
    "c_change_p": "Change Password",
    "c_ak": "API key",
    "c_gak": "Create API key",
    "c_rak": "Replace API key",
    "c_cftk": "Create FieldTask Key",
    "c_rftk": "Replace FieldTask Key",
    "c_reset": "Reset",
    "c_en": "Enable",
    "c_dis": "Disable",
    "c_loc_f": "Local Form:",
    "c_rem_f": "Remote Form",
    "c_rem_u": "Remote User",
    "c_enabled": "Enabled",
    "c_edit": "Edit",
    "c_del": "Delete",
    "c_undel": "Undelete",
    "c_del_data": "Delete Data",
    "c_res_data": "Restore Data",
    "c_archive_data": "Archive Data",
    "c_download": "Download",
    "c_upload": "Upload",
    "c_done": "Done",
    "c_cancel": "Cancel",
    "c_save": "Save",
    "c_saved": "Saved",
    "c_submit": "Submit",
    "c_submitted": "Submitted",
    "c_test": "Test",
    "c_validate": "Validate",
    "c_close": "Close",
	"c_likert": "Likert",
    "c_logout": "Logout",
    "c_block": "Block",
    "c_blocked": "Blocked",
    "c_graph": "Graph",
    "c_map": "Map",
    "c_chart": "Chart",
    "c_charts": "Charts",
    "c_maps": "Maps",
    "c_map_layers": "Map Layers",
    "c_table": "Table",
    "c_tables": "Tables",
    "c_images": "Images",
    "c_title": "Title",
    "c_comment": "Comment",
    "c_country": "Country",
    "c_region": "Region",
    "c_district": "District",
    "c_community": "Community",
    "c_node": "Node",
    "c_relation": "Relation",
    "c_from": "From",
    "c_subject": "Subject",
    "c_to": "To",
    "c_properties": "Properties",
    "c_question": "Question",
    "c_questions": "Questions",
    "c_choices": "Choices",
    "c_include": "Include",
    "c_unique": "Unique",
    "c_survey": "Survey",
    "c_survey_tg": "Survey / Task Group / Report",
    "c_date": "Date",
    "c_dateq": "Date Question",
    "c_none": "None",
    "c_period": "Period",
    "c_hour": "Hour",
    "c_day": "Day",
    "c_week": "Week",
    "c_month": "Month",
    "c_year": "Year",
    "c_daily": "Daily",
    "c_weekly": "Weekly",
    "c_monthly": "Monthly",
    "c_quarterly": "Quarterly",
    "c_yearly": "Yearly",
    "c_matches": "matches",
    "c_filter": "Filter",
    "c_inclusive": "Inclusive",
    "c_value": "Value",
    "c_clear": "Clear",
    "c_options": "Options",
    "c_data": "Data",
    "c_layers": "Layers",
    "c_settings": "Settings",
    "c_start": "Start",
	"c_end": "End",
	"c_step": "Step",
    "c_back": "Back",
    "c_target": "Target",
    "c_trigger": "Trigger",
    "c_content": "Content",
    "c_today": "Today",
    "c_attach": "Attach",
    "c_details": "Details",
    "c_action": "Action",
    "c_actions": "Actions",
    "c_mapid": "Map Id",
    "c_zoom": "Zoom Levels",
    "c_zoom_data": "Zoom to Data",
    "c_vector_data": "Vector Data",
    "c_vector_style": "Vector Style",
    "c_style": "Style",
    "c_portrait": "Portrait",
    "c_landscape": "Landscape",
    "c_tools": "Tools",
    "c_required": "Make Required",
    "c_not_required": "Make Not Required",
    "c_del_sel": "Delete Selected",
    "c_assign_sel": "Set Selected as Assigned",
    "c_print": "Print",
    "c_repeat": "Repeat",
	"c_persist": "Persist",
    "c_repeats": "Repeats",
    "c_scheduled": "Scheduled",
    "c_t_c": "terms and conditions",
    "c_agree": "I agree to the",
    "c_sa": "Select All",
    "c_select": "Select",
    "c_status": "Status",
    "c_coords": "Coordinates",
    "c_task": "Task",
    "c_rec": "(Recommended)",
    "c_source": "Source",
    "c_show": "Show",
    "c_show_q": "Show Question",
    "c_hide": "Hide",
    "c_total": "Total",
    "c_totals": "Totals",
    "c_instances": "Instances",
    "c_device": "Device",
    "c_success": "Success",
    "c_errors": "Errors",
    "c_error": "Error",
    "c_warning": "Warning",
    "c_lt": "Local Time",
    "c_standard": "Standard",
    "c_simserial": "Sim Serial Number",
    "c_location": "Location",
    "c_ident": "Identifier",
    "c_version": "Version",
    "c_deleted": "Deleted",
    "c_yes": "Yes",
    "c_no": "No",
    "c_reason": "Reason",
    "c_complete": "Complete",
    "c_record": "Record",
    "c_records": "Records",
    "c_of": "of",
    "c_media": "Media",
    "c_resource": "Media / CSV",
    "c_change": "Change",
    "c_changes": "Changes",
    "c_templates": "PDF Templates",
    "c_template": "PDF Template",
    "c_meta": "Meta Items",
    "c_calc": "Calculations",
    "c_trans": "Translate",
    "c_org": "Organisation",
    "c_orgs": "Organisations",
    "c_ent": "Enterprise",
    "c_all_s": "All Surveys",
    "c_all": "All",
    "c_event": "Event",
    "c_note": "Note",
    "c_server": "Server",
    "c_cols": "Columns",
    "c_cols_packed": "Packed Columns",
    "c_add": "Add",
    "c_create": "Create",
    "c_generate": "Generate",
    "c_imp_xls": "Import Spreadsheet",
    "c_spread": "Spreadsheets",
    "c_format": "Format",
    "c_xlstype": "Excel Format",
    "c_roles": "Roles",
    "c_s_roles": "Survey Roles",
    "c_role": "Role",
    "c_id": "Id",
    "c_pivot": "Pivot",
    "c_insert": "Insert",
    "c_tz": "Time Zone",
    "c_undef": "Not Defined",
    "c_license": "License Number",
    "c_in": "in",
	"c_notin": "not in",
    "c_rev": "Reverse",
    "c_for": "For",
    "c_normal": "Normal",
    "c_heatmap": "Heatmap",
    "c_auth": "Authorized",
    "c_merge": "Merge",
    "c_append": "Append",
    "c_discard": "Discard",
    "c_keys": "Keys",
    "c_counts": "Counts",
    "c_unsav": "You have unsaved changes are you sure you want to leave?",
    "c_off": "Off",
    "c_on": "On",
    "c_rule": "Rule",
    "c_replace": "Replace",
    "c_retry": "Retry",
    "c_alink": "Anonymous Link",
    "c_excel": "Excel",
    "c_word": "Word",
    "c_csv": "CSV",
    "c_json": "JSON",
    "c_geojson": "GeoJSON",
    "c_osm": "Openstreetmap",
    "c_shape": "Shape",
    "c_kml": "KML / KMZ",
    "c_vrt": "VRT / CSV",
    "c_stata": "Stata / CSV",
    "c_spss": "SPSS / CSV",
    "c_pdf": "PDF",
    "c_phone": "Phone Number",
	"c_photo": "Photograph",
    "c_rep_type": "Sub Form",
    "c_cl": "Copy Link",
    "c_ck": "Copy Key",
    "c_codl": "Copy OData Link",
    "c_cb": "Copy to clipboard",
    "c_c": "Copied",
    "c_quantity": "Quantity",
    "c_after": "After",
    "c_units": "Units",
    "c_duration": "Duration",
    "c_msg": "Message",
    "c_completed": "Completed",
    "c_unsubscribed": "Unsubscribed",
    "c_unsubscribe": "Unsubscribe",
    "c_subscribe": "Subscribe",
    "c_expired": "Expired",
	"c_subscriberid": "Subscriber Id",
    "c_unsent": "Email not sent",
    "c_pending": "Pending",
    "c_ok": "OK",
    "c_move": "Move",
    "c_parent_form": "Parent Form",
    "c_child_form": "Child Form (Repeats)",
    "c_other": "Other",
    "c_random": "Randomize",
    "c_annotate": "Annotate",
    "c_draw": "Draw",
    "c_field_list": "Field List",
    "c_table_list": "Table List",
    "c_recover": "Recover Previous File",
    "c_lat": "Latitude",
    "c_lon": "Longitude",
    "c_browse": "Browse",
    "c_reject": "Reject",
    "c_history": "History",
    "c_events": "Events",
    "c_resend": "Resend",
    "c_late": "Late",
    "c_custom": "Custom",
    "c_access": "Access",
    "c_person": "Person",
    "c_people": "People",
    "c_contacts": "Contacts",
    "c_teams": "Teams",
    "c_mailouts": "Mail outs",
    "c_mailout": "Mail out",
    "c_campaigns": "Campaigns",
    "c_campaign": "Campaign",
    "c_manual": "Manual",
    "c_submitter": "Submitter",
    "c_code": "Code",
    "c_dirn": "Direction",
    "c_rtl": "RTL",
    "c_current": "Current",
    "c_quality": "Quality",
    "c_low": "Low",
    "c_external": "External",
    "c_preview": "Preview",
    "c_size": "Size",
    "c_meters": "Meters",
    "c_accuracy": "Accuracy",

    // File Types
    "ft": "File Type",
    "ft_xls": "XLS Form",
    "ft_xls_orig": "Original file",
    "ft_xml": "XML Form",
    "ft_codebook": "Codebook",
    "ft_pdf": "Default Report Template",
    "ft_spss": "SPS File (SPSS)",

    // Template Management
    "tm_s_del": "Show deleted forms",
    "tm_s_block": "Show blocked forms",
    "tm_d_forms": "Delete Forms",
    "tm_r_forms": "Restore Forms",
    "tm_e_forms": "Erase Forms",
    "tm_c_form": "Create a Form",
    "tm_c_form_rep": "Replace a Form",
    "tm_c_sr_rep": "Replace a shared resource file",
    "tm_c_xls": "Using the Spreadsheet editor",
    "tm_g_temp": "Download Template",
    "tm_g_new": "Create new Form",
    "tm_g_open": "Open a Form",
    "tm_g_temp_i1": "Download a template to create a new form on your personal computer. When you have finished, upload it to the server by clicking on the upload button below. You will first need to select the XLS file containing the form you have created and give it a name.",
    "tm_g_temp_i3": "Reference Guide",
    "tm_ul_f": "Upload the spreadsheet",
    "tm_ul_sr": "Upload the shared resource file",
    "tm_ul_sl": "Upload the shared locations file",
    "tm_si": "Specify survey identifier",
    "tm_up": "Upload Template",
    "tm_add": "Add Template",
    "tm_ups": "Upload Form",
    "tm_add_for": "Add Forward",
    "tm_ref_f": "Refresh Form List",
    "tm_o_u": "Original Upload Page",
    "tm_gw": "A bundle should only be specified for surveys that share tables, if in doubt select none",

    // Analysis
    "a_ref_rate": "Analysis auto refresh interval",
    "a_exp_title": "Export Results",
    "a_exp_leg1": "Select a survey",
    "a_locn": "Select a location question",
    "a_exp_leg2": "Output format:",
    "a_exp_notes": "Export notes:",
    "a_exp_meta": "Include Meta Data:",
    "a_exp_odata2": "Restrict Odata types to those supported by OData 2:",
    "a_exp_split": "Split lat / lon:",
    "a_exp_merge": "Compress select multiples:",
    "a_sel_ways": "Select ways (if any):",
    "a_sel_forms": "Select forms to include:",
    "a_sel_query": "There are no queries to export.  You will need to create one using the queries menu in the dashboard module",
    "a_sel_model": "Edit survey model",
    "a_type": "Chart Type",
    "a_showon": "Show on",
    "a_ss": "Select survey",
    "a_sq": "Select question",
    "a_gb": "Group by",
    "a_ts": "Time series",
    "a_sl": "Show Label",
    "a_g_gps": "GPS coordinates within region",
    "a_fbv": "Filter by value",
    "a_from_date": "From date",
    "a_to_date": "To date",
    "a_fq": "Filter question",
    "a_res": "Please provide a reason",
    "a_res_5": "Please provide a reason at least 5 characters long",
    "a_mb": "Marked Bad",
    "a_oo": "out of",
    "a_cm": "Cluster Members",
    "a_dv": "Show Details",
    "a_rd": "Report Definition",
    "a_rp": "Report Parameters",
    "a_lcr": "Launch Custom Report",
    "a_embed": "Embed Images in Excel File",
    "a_hxl": "HXL",
    "a_query": "Select a query",
    "a_fn": "Function",
    "a_ni": "You need to enable loading tasks for this form in the form settings in the editor page",
    "a_nx": "New XLSX",
    "a_lx": "Legacy XLSX",
	"a_lh": "Legacy XLS / HTML",
	"a_ua": "User Activity",
    "a_ul": "User Locations",
    "a_tc": "This Chart",
    "a_start_pk": "First Primary Key",
    "a_end_pk": "Last Primary Key",
    "a_inc_ro": "Include Read Only",

    "a_dd_sl": "Shows the results",
    "a_dd_ocha_map": "Shows the location of responses to %s1 in the question '%s2' in survey '%s3'.",
	"a_dd_percent_map": "Shows the percentage of responses that selected %s1 in the question '%s2' in survey '%s3'.",
	"a_dd_average_map": "Shows the average of responses to %s1 in the question '%s2' in survey '%s3'.",
	"a_dd_count_map": "Shows the count of responses to %s1 in the question '%s2' in survey '%s3'.",
	"a_dd_total_map": "Shows the total of responses to %s1 in the question '%s2' in survey '%s3'.",
	"a_dd_none_map": "Shows the responses to %s1 in the question '%s2' in survey '%s3'.",
	"a_dd_ocha_table": "Shows the location of responses to each choice in the question '%s2' in survey '%s3'.",
	"a_dd_percent_table": "Shows the percentage of responses that selected each choice in the question '%s2' in survey '%s3'.",
	"a_dd_average_table": "Shows the average of responses to each choice in the question '%s2' in survey '%s3'.",
	"a_dd_count_table": "Shows the count of responses to each choice in the question '%s2' in survey '%s3'.",
	"a_dd_total_table": "Shows the total of responses to each choice in the question '%s2' in survey '%s3'.",
	"a_dd_none_table": "Shows the responses to each choice in the question '%s2' in survey '%s3'.",
    "a_filter": "Showing %s1 records out of %s2",

    "a_dd_units": "(units are %s1)",

    "a_dd_group": "Grouped by the responses to the question '%s1'.",
	"a_dd_group_interval": "Grouped by the responses to the question '%s1' and by '%s2'.",
	"a_dd_interval": "Grouped by '%s1'",

    "a_dd_where": "<b>Where %s1 equals %s2</b>",
    "_scheduled_start": "Scheduled Start",
	"a_exp_media_msg": "Exports with media are limited to %s1 records. ",
	"a_exp_msg": "Exports without media are limited to %s1 records. ",
	"a_exp_msg2": "You are currently exporting %s2 records. Set the start and end records to conform to this limit",
	"a_exp_start": "The start record must be less than or equal to the end record",

    // Messages
    "msg_ipt": "Invalid panel title",
    "msg_pdft": "PDF Templates are now managed on the templates page which can be accessed from the tools menu",
    "msg_oet": "Only email can be specified as the target when periodic is selected as the trigger",
    "msg_dms": "The day of the month must be specified",
    "msg_pc": "The reminder interval must be greater than zero",
    "msg_pk_sel": "You can only select the record identifier if you have selected the \"Update a different question\" checkbox",
    "msg_wait": "Waiting for a response from the server. Please try again in a few seconds",
    "msg_reason": "Please provide a reason: ",
    "msg_rs": "Reminder notifications must be associated with a task group that has tasks generated from a survey.  You cannot send reminders for ad-hoc tasks",
    "msg_sel_survey": "You must select a survey",
    "msg_sel_dates": "From date is greater than To date",
	"msg_no_from": "If you specify the scheduled end date, then also specify the scheduled start date",
    "msg_sel_q": "You must select a question to show on a graph",
    "msg_so": "Number details can only be edited when you are in the organisation that the number is assigned to",
    "msg_val_prot": "Protocol (http:// or https://) must be specified with the hostname",
    "msg_val_p": "A project must be selected for this form",
    "msg_val_rf": "You must select a remote form",
    "msg_val_rh": "You must select a remote host",
    "msg_val_u_id": "You must specify a user id",
    "msg_val_pass": "You must specify a password",
    "msg_nv": "Please specify a new value",
    "msg_val_nm": "Name is required",
    "msg_val_inv_nm": "Name cannot include a / or a . character. Specify the name without an extension.",
    "msg_val_gen_nm": "The name cannot contain angle brackets <>",
    "msg_val_period": "An period greater than 0 must be specified",
    "msg_val_dl_dist": "If set then the download distance must be greater than 100 meters",
    "msg_val_show_dist": "If set then the show distance must be greater than 10 meters",
    "msg_val_ad": "If assigning a task using collected data then you need to specify the data",
    "msg_val_ad2": "The additional role should only be specified if the first role was specified",
    "msg_val_file": "A file must be selected",
    "msg_val_let": "Name must start with a letter or underscore",
    "msg_pwd_l": "Passwords, if specified, must be longer than 1 character",
    "msg_pwd_m": "Passwords do not match",
    "msg_add_notification": "Add Notification",
    "msg_edit_notification": "Edit Notification",
    "msg_send_notification": "Send Notification",
    "msg_nn": "No notifications",
    "msg_noi": "No opt in messages",
    "msg_add_map": "Add Map",
    "msg_edit_map": "Edit Map",
    "msg_sel_form": "Select Form:",
    "msg_sel_media": "Select Media Question:",
    "msg_mfn": "Create media file name with:",
    "msg_sel_media_f": "Select Media File",
    "msg_err_save": "Error: Save failed: ",
    "msg_err_del": "Error: Delete failed: ",
    "msg_err_cr": "Error: Create failed: ",
    "msg_err_upd": "Error: Update failed: ",
    "msg_err_res": "Error: Restore failed: ",
    "msg_err_block": "Error: Block failed: ",
    "msg_err_get_f": "Error: Get forms failed: ",
    "msg_err_get_r": "Error: Get roles failed",
    "msg_err_get_a": "Error: Get alerts failed",
    "msg_err_get_q": "Error: Get questions failed",
    "msg_err_get_s": "Error: Get surveys failed",
    "msg_err_get_c": "Error: Get case management settings failed",
    "msg_err_bll": "Blank Label for language",
    "msg_upd": "Update Succeeded",
    "msg_err_nc": "Name cannot include a comma",
    "msg_err_wait": "Either waiting for the server or there are no surveys in this project to assign to a user. If the project does have surveys then try again in a few seconds",
    "msg_err_cert": "Remote server does not have a signed certificate, try using http:// instead of https://",
    "msg_no_edit_rep": "You cannot edit a record that has been replaced by another record",
    "msg_confirm_del": "Are you sure you want to delete",
	"msg_confirm_del_one": "Delete %s1 only from %s2",
	"msg_confirm_del_all": "Delete %s1 from all organisations: %s2",
    "msg_confirm_tasks": "this group and all of its tasks?",
    "msg_del_r": "Are you sure you want to delete these regions?",
    "msg_erase": "Are you sure you want to permanently erase these surveys and all of their data?",
    "msg_restore": "Are you sure you want to restore these surveys?",
    "msg_del_recs": "There are %s1 data rows submitted for %s2. Are you sure you want to delete this data?",
    "msg_del_s": "Are you sure you want to delete these surveys?",
    "msg_del_data": "Are you sure you want to delete all the data in this survey?",
    "msg_del_groups": "This survey is part of a group and deleting its data will also delete the data for these other surveys",
    "msg_res_data": "Are you sure you want to restore all the data in this survey to the original submissions?",
    "msg_del_data2": "Are you really sure?",
    "msg_del_rep": "Are you sure you want to delete this report?",
    "msg_del_users": "Are you sure you want to delete these users?",
    "msg_del_projects": "Are you sure you want to delete this project?",
    "msg_del_roles": "Are you sure you want to delete these roles?",
    "msg_del_cms": "Are you sure you want to delete this setting?",
    "msg_del_orgs": "Are you sure you want to delete these organisations?",
	"msg_del_ents": "Are you sure you want to delete these enterprises?",
    "msg_del_q": "Are you sure you want to delete this question?",
    "msg_del_cl": "Are you sure you want to delete this choice list?",
    "msg_del_c": "Are you sure you want to delete this choice?",
    "msg_rep_f": "Are you sure you want to replace this choice filter",
    "msg_del_not": "Are you sure you want to delete notification",
    "msg_del_nbr": "Are you sure you want to delete the phone number",
    "msg_n_v": "Not valid",
    "msg_u_f": "Error upload failed",
    "msg_u_nt": "No nfc tags found",
    "msg_csv": "Only CSV files and Zip files containing CSV and media files are allowed",
    "msg_inv_email": "Not a valid email address",
    "msg_dup_email": "Duplicate email. Some other user has this email.",
    "msg_email_req": "If sending an email to the user then email address must be specified",
    "msg_email_dom": "Email user name should not include the email domain.  So for an email address of example@org.com the user name would be 'example'",
    "msg_dup_ident": "Duplicate user identification. Please change the user ident.",
    "msg_cs": "Click to filter or sort on this column",
    "msg_nrf": "No results forwarded to another server",
    "msg_saf": "Select a form to view the users that have downloaded that form",
    "msg_nf": "No forms",
    "msg_ns": "No Submissions",
    "msg_us": "Unknown Source",
    "msg_tg_ns": "Task group not selected",
    "msg_tg_rd": "The task group rule has been deleted",
    "msg_no_roles": "There are no enabled roles. Hence there are no restrictions on access",
    "msg_has_roles": "Only users who have one of the following enabled roles are allowed access",
    "msg_one_f": "At least one form must be selected",
    "msg_one_f2": "A form must be selected",
    "msg_embed": "The excel type must be xls or xlsx to embed images",
    "msg_pss": "Please select a survey",
    "msg_nm": "No images, video, audio found",
    "msg_pr": "Password has been reset",
    "msg_pex": "Your password has expired please set a new one",
    "msg_es": "If %s1 is associated with a user account then an email will be sent containing instructions to reset the password of that account.",
    "msg_refresh": "You have made some changes and not saved them, are you sure you want to refresh?",
    "msg_leave": "You have made some changes and not saved them, are you sure you want to leave?",
    "msg_test": "You have made some changes and will need to save them before testing the form.",
    "msg_not_f": "was not found",
    "msg_ren": "renamed to",
    "msg_nq": "You have no queries defined, use the 'add query' button to add one",
    "msg_dup_f": "This form is already in the query",
    "msg_ui": "Spaces are not allowed in the user ident",
    "msg_sp": "Select a project to see the status of forms",
    "msg_ab_ns": "Either adding tasks from already submitted data, or adding tasks from data yet to be submitted or both should be selected",
    "msg_n1": "emails in a question",
    "msg_n2": "emails in meta item",
    "msg_sms_n1": "Send SMS to ",
    "msg_sms_n2": "numbers entered in response to a question",
    "msg_ds_s": "Download started.  If there is a lot of data then this may take some time please be patient",
    "msg_ds_s_r": "Report generation started.  You will find the generated report in the reports module",
    "msg_survey_replaced": "Survey replaced",
    "msg_survey_loaded": "Survey loaded",
    "msg_reg": "Registration accepted.  An email has been sent to %s1 with a link that you can use to set your password.",
    "msg_uns": "Unsubscribe sucessful",
    "msg_s1": "An email has been sent to the address you provided.  Click on the link to confirm your subscription",
    "msg_s2": "Subscribed sucessfully",
    "msg_s3": "Enter your email address to re-subscribe to emails",
    "msg_pd_key": "Specify a data key ",
    "msg_one_role": "At least one role must be selected",
    "msg_dup_name": "Duplicate name",
    "msg_prev_select": "There are no previous select questions to get values from. You may want to set a custom filter",
    "msg_no_proj": "The user has no projects",
    "msg_pformat": "Parameters specified in the other field must be separated by semi colons and contain an equals sign. For example:  a=x;b=y",
    "msg_filter_col": "You have specified a filter Column of '%s1', hence you need to specify a filter to apply",
    "msg_filter_expr": "You must specify an expression for an 'eval' type of search",
    "msg_numb_ts": "Thousands separator will have no affect unless numbers is also selected",
	"msg_warn_likert_n": "Likert only works with the normal select type",
    "msg_choice_value": "The choice value must be specified",
	"msg_search_source": "You must select either survey or csv as the souce of the choice data",
    "msg_search_source2": "The source file or survey must be selected",
	"msg_pdfcols_count": "The total width of all the columns must add up to 10",
    "msg_pdfinv_zoom": "Zoom must be between 1 and 20",
    "msg_pdfinv_round": "Rounding of decimal places can be between 0 and 10",
	"msg_pdfcols_width": "A column width must be between 1 and 10 inclusive",
	"msg_pe": "The number of days before password expiry must be an integer greater than 0",
    "msg_restore_started": "Restoration of the survey data has started.  Press refresh to see progress.  This process may take minutes if the files to be retrieved are in off site storage",
	"msg_persist": "Persistent tasks stay on the device even after they are completed. The user should reject the task when they have finished with it",
    "msg_no_uns_orgs": "There are no organisations where that email address is unsubscribed",
    "msg_res_name": "%s1 is a reserved name",
    "msg_trans_to": "The translation request has timed out.  However it should still be working in the background and you can view the results by refreshing the page",
    "msg_subs_bg": "If you have un-subscribed from receiving email notifications, then using this page, you can re-subscribe to receive them again. The decision to send notifications to an email address is made by an administrator so although you can un-subscribe from these and then re-subscribe on this page you cannot request email notiifcations when you are not being sent any.",
    "msg_subs_u": "You can unsubscribe from receiving notifications from the organisation that sent you the email, by clicking the unsubscribe button below.  If you later want to receive emails from that organisation again, you can select the 'subscribe' menu on the home page.",
    "msg_subs_s": "Click on the button below to subscribe",
    "msg_fgt_p": "Enter the email address associated with your account. A link will be sent to that address allowing you to reset your password.",
    "msg_archive": "%s1 records on or before %s2 will be archived from the following survey(s); %s3. Do you wish to proceed?",
    "msg_archive_done": "Archiving completed",
    "msg_archive_none": "There is no data to archive",
    "msg_archive_data": "Are you sure you want to archive old submissions?",
    "msg_archive_before": "The date before which submissions will be archived must be specified",
    "msg_no_nbr": "You must specify their number",
    "msg_nc": "Do you want to move this message to a new case",

    // Editor
    "e_b_e": "Base on existing form",
    "e_s_r": "Store results in shared table",

    // Tasks
    "t_i_sd": "Invalid Start Date",
    "t_i_ed": "Invalid End Date",
    "t_b": "Trigger with any survey in the bundle",
    "t_complete": "Show Completed Tasks",
    "t_show": "Show tasks for:",
    "t_assign_type": "Assignment type",
    "t_assign": "Assign User",
    "t_assign_roles": "Assign to users with role",
    "t_fixed_role": "And this role",
    "t_add_group": "Add Task Group",
    "t_del_group": "Delete Task Group",
    "t_edit_group": "Edit Task Group",
    "t_lt_file": "Load Tasks From File",
    "t_choose_user": "Choose the user to assign to the selected tasks",
    "t_ce": "Clear Existing Tasks",
    "t_ce_warn": "This will clear all the existing data in the survey. Are you sure you want to do this?",
    "t_ceg": "Create Empty Task Group",
    "t_cfe": "Create from existing data",
    "t_ep": "Existing Project",
    "t_es": "Existing Survey",
    "t_ec": "Existing Choice",
    "t_et": "Existing Text",
    "t_nt": "New Text",
    "t_ns": "New Survey",
    "t_ft": "Filter Tasks",
    "t_aft": "Advanced Filter",
    "t_df": "Date Filter",
    "t_fq": "Filter by Question",
    "t_so": "Selected Option",
    "t_sn": "Set Numeric Value",
    "t_st": "Set Text Value",
    "t_ue": "Update Existing Results",
	"t_pp": "Pre-populate form with existing data",
    "t_nid": "No initial data",
    "t_ap": "Address Parameters",
    "t_nfc": "NFC / Location",
    "t_assigned": "Assigned",
    "t_eas": "Send Email to assigned user",
    "t_assignments": "Assignments",
    "t_iewe": "Emails can be sent to notify people about an escalation",
    "t_fl": "File Loaded",
    "t_tl": "Template Loaded",
    "t_efnl": "Error file not loaded",
    "t_ens": "Error not saved",
    "t_add_tasks": "Click on the map where you want to add new tasks. When finished click on 'Save Tasks' button",
    "t_add_task": "Add Task",
    "t_edit_task": "Edit Task",
    "t_new_case": "New Case",
    "t_add_tasks2": "Add Tasks",
    "t_save_tasks": "Save Tasks",
    "t_tasks_only": "Tasks Only",
    "t_tg": "Task Group",
    "t_tasks": "Tasks",
    "t_id": "Initial Data",
    "t_n_s": "Tasks without start time",
    "t_drag": "Drag a task and drop onto calendar",
    "t_g": "Guidance",
    "t_pol": "Complete All",
    "t_auto": "Allow users to assign themselves (if unassigned)",
    "t_auto2": "Self Assign",
    "t_ass_done": "Have you completed this assignment?",
    "t_defn": "Task Group Definition",
    "t_au": "All Users",
    "t_u": "Unassigned",
    "t_ad": "Assign from data values",
    "t_af": "Create tasks from results submitted after this task group is created",
    "t_ac": "Create tasks from results that have already been submitted",
    "t_start": "Set the start time and duration of the task relative to when it is created or a date in the source survey",
    "t_all": "All tasks",
    "t_dl_dist": "Download distance (meters)",
    "t_dl_dist_i": "The user will need to be within this distance from the geographic center of the tasks in this task group before they will be downloaded onto their device",
    "t_show_dist": "Show distance (meters)",
    "t_show_dist_i": "The user will need to be within this distance from the task before it is displayed on their device",
    "t_show_console": "Show in console",
    "new": "Unassigned",
    "accepted": "Assigned",
    "submitted": "Complete",
    "rejected": "Rejected",
    "cancelled": "Cancelled",
    "deleted": "Deleted",
    "restored": "Restored",
    "unsent": "Not Sent",
    "unsubscribed": "Unsubscribed",
    "restore": "Restore",
    "t_mt": "My Tasks",

    // template load errors
    "e_u_err": "Form Upload Error",
    "e_u_exists": "exists in project",
    "e_h_rename": "Change the name of the form or delete the old form",
    "e_u_sm_no_list": "select_multiple question without list name",
    "e_h_sm_no_list": "Check the survey sheet. Make sure you have specified a list name for all the select_multiple questions",
    "e_u_so_no_list": "select_one question without list name",
    "e_h_so_no_list": "Check the survey sheet. Make sure you have specified a list name for all the select_one questions",
    "e_unknown": "Unknown error",
    "e_get_help": "Contact the help desk",
    "e_calc": "Calculation error",
    "e_h_calc1": "Check the 'calculation' column for an invalid formula (A valid formula results in either true or false)",
    "e_h_calc2": "Otherwise check the 'relevant' column for a formula that does not result in a true or false result",
    "e_inv_f": "Invalid Function",
    "e_h_f1": "Check for capital letters, all letters should be lower case",
    "e_h_f2": "Check for spelling mistakes in the name of the function",
    "e_val": "Validation error",
    "e_circ": "Check for circular references",
    "e_in_q": "in question",
    "e_h_c1": "A 'relevant or calculation or choice filter' for question is referring to  itself instead of referring to another question",
    "e_h_c2": "Relevant statements are evaluated before a question is asked so they cannot refer to their own questions",
    "e_brackets": "Mismatched brackets in function",
    "e_text": "Error with the following text:",
    "c_mcm": "Multi case message",
    "c_mcmg": "Use place holder %s1 for the number of open cases and %s2 to hold the list of existing cases",

    // Register
    "r_o_n": "Organisation Name",
    "r_y_n": "Your Name",
    "r_y_e": "Your Email",
    "r_y_o": "Organisation Website",
    "r_f": "Registration Form",
    "r_s": "Email Subscriptions",

    // Monitor
    "mon_l200": "(last 200)",
    "mon_gtb": "Group Totals by",
    "mon_nl": "Not Loaded",
    "mon_ue": "Upload Errors",
    "mon_dup": "Duplicates",
    "mon_att": "Extra Attachments",
    "mon_sa": "Show As",
    "mon_zoom_grid": "Zoom to Grids",
    "mon_show_grid": "Show Grids",
    "mon_clear_grid": "Clear Grids",
    "mon_create_grid": "Create Grid",
    "mon_del_grid": "Delete Grids",
    "mon_sel_layer": "Select a layer to add to the map as an overlay",
    "mon_c_reg": "Create a new region",
    "mon_c_reg_i": "Enter a name for the region and the width of each cell in meters. Click on the map to set the centre of one of the cells. Press the \"shift\"button and drag the mouse to set the bounds.",
    "mon_name_cond": "(lowercase, no spaces or numbers)",
    "mon_cw": "Cell Width (in Meters. greater than 10)",
    "mon_centre": "Centre: (click on map)",
    "mon_bounds": "Boundary: (click and drag on map while pressing shift key)",
    "mon_sel_del": "Select the regions that you want to delete",
    "mon_ud": "Upload Date",
    "mon_pi": "Phone Identifier",
    "mon_fr": "Failure Reason",
    "mon_file": "File with raw results",
    "mon_optin_msg": "Opt in",
    "mon_send_count": "Send Count",
    "mon_pending_count": "Pending Messages",
    "mon_ignore_old": "Ignore issues older than 100 days",
    "error": "Error",
    "success": "Success",
    "mon_retry": "Re-apply failed uploads",
    "mon_page": "Page %s1 from %s2 to %s3",
    "mon_uid": "Upload Id",

    // Analysis, reports, modify
    "an_auth": "Author",
    "an_published": "Published",
    "an_publish": "Publish",
    "an_dr": "Date Range",
    "an_csel": "Clear Selection",
    "r_add": "Add Report",
    "r_ed": "Edit Report",
    "r_del": "Delete Reports",
    "an_nd": "No data available",
    "an_nmd": "No data matches the filter",
    "an_mod_hint": "The modify page allows you to make bulk changes to text.  You can select a specific text value and then change all occurrences of that to another value.  This is very useful for fixing spelling inconsistencies. You can also change text values in 'other' questions into one of the select choices. You can also update barcode and calculate questions. Other question types are not supported as, for example, it would make no sense to do a bulk change of an integer age from 25 to 26 wherever it occurred. You can review modifications by selecting the review page.<br/><br/> You can however update integer, decimal and select1 questions by selecting the checkbox 'Update a different question'. In this case the change will be applied to all questions that match the existing values of the two questions. In most cases you will not need this option.",
    "Upload Time": "Upload",
    "_start": "Start Time",
    "_end": "End Time",
    "percent": "Percentage",
    "count": "Count",
    "none": "None",
    "average": "Average",
    "max": "Max",
    "min": "Min",
    "sum": "Sum",
    "changes": "Update",
    "task": "Task",
    "notification": "Notification",
    "created": "Created",

    // Web forms
    "wf_sig": "Signature",

    // Editor
    "ed_gr": "Text rows to show",
    "ed_ba": "Background Audio",
    "ed_map_zoom": "Map Zoom",
    "ed_map_source": "Map Source",
    "ed_map_style": "Map Style",
    "ed_round": "Decimal places",
	"ed_hidden": "Hide in form",
    "ed_extendable": "Extendable",
	"ed_selfie": "Open camera in selfie mode",
    "ed_new": "Require a new picture be taken",
	"ed_read_nfc": "Read an NFC tag instead of a barcode",
    "ed_gt_0": "must be greater than 0, or not set",
	"ed_add_to": "Append to question",
    "ed_app": "Put in an appendix",
	"ed_barcode": "Show as a barcode",
    "ed_si": "Show choice images instead of label text",
    "ed_stretch": "Stretch images to take up all space allocated",
	"ed_hyperlink": "Show a hyperlink to the image",
	"ed_col_width": "width of column %s1",
	"ed_place_map": "Allow the user to select their location from a map",
	"ed_l_bold": "Make label bold",
	"ed_caps": "Make label all capitals",
	"ed_npb": "New Page Before",
	"ed_sa": "Spacing above",
	"ed_bgc": "Label Background Colour",
	"ed_vbgc": "Value Background Colour",
    "ed_mc": "Marker Colour",
	"ed_height": "Height allowed for answer",
    "ed_lw": "Width of label",
    "ed_pdfcols_help": "There can be a maximum of 10 columns. The widths of the columns must add up to 10.",
	"ed_pdf_layout": "PDF Layout",
	"ed_pdfno": "Hide question in pdf export",
	"ed_date_type": "Date Type",
	"ed_select_type": "Select Type",
    "ed_nc": "No Calendar",
	"ed_my": "Month and Year",
	"ed_bikram": "Bikram Sambat",
	"ed_coptic": "Coptic",
	"ed_ethiopian": "Ethiopian",
	"ed_islamic": "Islamic",
    "ed_myanmar": "Myanmar",
    "ed_persian": "Persian",
    "ed_r": "Rating",
	"ed_bear": "Bearing",
	"ed_sep": "Thousands Separator",
    "ed_ncol": "No Collapse Button (WebForms Only)",
    "ed_sn": "Show numbers keyboard",
    "ed_numb_cols": "Number of Columns",
	"ed_min": "Minimal",
	"ed_adv": "Auto Advance",
	"ed_ac": "Auto Complete",
    "ed_acm": "Minimal Auto Complete",
	"ed_compact": "Compact",
	"ed_compact_adv": "Compact with auto advance",
	"ed_image_map": "Image Map",
	"ed_pick": "Picker",
    "ed_as": "Add Search",
    "ed_rf": "Reference Form",
    "ed_in": "Instance Name",
	"ed_io": "Instance Order",
	"ed_ic": "Instance Count",
    "ed_ep": "Existing Project",
    "ed_es": "Existing Survey",
    "e_b_e": "Base on existing form",
    "e_s_r": "Store results in shared table",
    "ed_qt": "Question Text",
	"ed_dn": "Display Name",
    "ed_ct": "Choice Text",
    "ed_qh": "Question Hint",
    "ed_cl": "Choice List",
    "ed_appearance": "Appearance",
    "ed_parameters": "Parameters",
    "ed_con": "Constraint",
    "ed_con_msg": "Constraint Message",
    "ed_sct": "Save changes before running auto translate",
    "ed_rel": "Relevant",
    "ed_req": "Required",
    "ed_req_msg": "Required Message",
    "ed_nreq": "Not Required",
    "ed_ap": "Auto play",
    "ed_al": "Automatically Annotate",
	"ed_mat": "Medical Audio Type",
	"ed_mat_c": "Conversation",
	"ed_mat_d": "Dictation",
    "ed_al2": "Use in oversight forms to automatically annotate or translate text, images, audio and video",
	"ed_am": "Medical",
	"ed_am2": "Use when transcribing audio that contains medical terms",
    "ed_at": "GPS Accuracy Threshold",
    "ed_ls": "Linked Survey",
    "ed_ro": "Read Only",
    "ed_ls_add": "Add link",
    "ed_l": "Linked",
    "ed_nl": "Not Linked",
    "ed_r_msg": "Required Text Message",
    "ed_r_expression": "Optional expression to make this question required",
    "ed_r_readonly": "Optional expression to make this question read only",
	"ed_f_l": "The form to be launched must be specified as a parameter",
	"ed_emp_rep": "Sub forms and groups should have at least one question.",
    "ed_hrk": "Unique Key",
    "ed_dk": "Data Key",
    "ed_csv": "CSV File",
    "ed_csvs": "CSV Files",
    "ed_c_s": "Source of Choices",
    "ed_p_q": "Source question",
    "ed_cws": "Choices Worksheet",
    "ed_i_c": "Created",
    "ed_st": "Shared Table",
    "ed_ld": "Allow Loading data from file",
    "ed_pd": "Pull data from another survey",
    "ed_i_bo": "Based On",
    "ed_i_tn": "Table Names",
    "ed_tlf": "Top Level Form",
    "ed_namedia": "Media cannot be used with this question",
    "ed_ab": "Add Before",
    "ed_aa": "Add After",
    "ed_aq": "Add New Question",
    "ed_ancl": "Add New Choice List",
    "ed_anc": "Add New Choice",
    "ed_cs": "Cascading Select",
    "ed_cspq": "Previous Select Question",
    "ed_csp": "Previous Selection",
    "ed_cf": "Custom Filter",
    "ed_afc": "Add Filter Column",
	"ed_clabs": "Choice Labels",
    "ed_clab": "Choice Label",
    "ed_cval": "Choice Value",
    "ed_fv": "Filter Value",
	"ed_sfv": "Second Filter Value",
    "ed_choice_f": "Choice Filter",
	"ed_vc": "Choice names must only contain letters, numbers, spaces, underscores, dashes and periods",
	"ed_vq": "The question name must start with a letter, underscore or colon and only contain letters, numbers, underscores, dashes and periods",
    "ed_ns": "Add a new survey using the 'File' menu before adding questions",
    "ed_kp": "Key Policy",
    "ed_s1": "Select One",
    "ed_s": "Select Multiple",
    "ed_image": "Image",
    "ed_image_type": "Image Type",
    "ed_int": "Integer",
    "ed_gps": "GPS Point",
    "ed_calc": "Calculation",
    "ed_s_calc": "Server Calculation",
    "ed_ci": "Server Compound PDF Image",
    "ed_cm": "Compound Map",
    "ed_audio": "Audio",
    "ed_video": "Video",
    "ed_bc": "Barcode",
    "ed_dt": "Date and Time",
    "ed_t": "Time",
    "ed_dec": "Decimal",
    "ed_gps_line": "GPS Line",
    "ed_gps_area": "GPS Area",
    "ed_ack": "Acknowledge",
    "ed_range": "Range",
    "ed_graph": "Graph",
    "ed_qs": "Questions set ",
    "ed_ee": "Don't include empty answers in reports",
    "ed_cpdf": "Compress PDF (images will have lower resolution that should be adequate for printing)",
    "ed_er": "Cannot save until errors are fixed",
    "ed_blocked": "The survey has been blocked. Changes cannot be saved.  You can unblock the survey on the form management page.",
    "ed_cml": "Cannot modify languages while there are unsaved changes",
    "ed_cx": "Cannot export to an xlsForm while there are unsaved changes",
    "ed_csr": "Cannot set questions required, or not required, while there are unsaved changes",
    "ed_fi": "Form Identifier",
    "ed_dl": "Default Language",
    "ed_td": "Timing Data",
    "ed_rl": "Record location with questions",
    "ed_rc": "Audit changes to questions",
	"ed_hod": "Hide the form on devices",
    "ed_sld": "Lookup local, unsent, data on device",
    "ed_s_g": "Grid (Web Forms)",
    "ed_s_p": "Pages",
    "ed_c_sr": "All questions set to <span style=\"color:blue;\">required</span>",
    "ed_c_snr": "All questions set to <span style=\"color:red;\">not required</span>",
    "ed_c_del_q": "Deleted question",
    "ed_c_del_o": "Deleted choice %s1 from option list %s2",
    "ed_c_chg_p":  "%s1 property of question %s2 changed to: %s3 from: %s4",
    "ed_c_add_q": "Added question %s1 with type %s2",
    "ed_c_add_o": "Added choice %s1 to choice list: %s2",
    "ed_c_settings": "Settings changed",
    "ed_c_languages": "Languages changed",
    "ed_c_template": "PDF template changed",
    "ed_a_template": "PDF template added",
    "ed_d_template": "PDF template deleted",
    "ed_s_templates": "PDF templates added using the settings dialog cannot be set as 'not available' nor set as default",
    "ed_mmf": "Manage Media Files For this Form",
    "ed_sl": "Survey to launch",
    "ed_sc": "Survey to complete",
    "ed_slu": "Survey lookup",
    "ed_gsi": "Get Survey Ident",
    "ed_qk": "Question to store the returned key",
	"ed_qkc": "Question in the launched form to get this forms key",
    "ed_pl": "Page Layout",
	"ed_w": "Width in webform grid",
    "ed_m": "Filter",
    "ed_fc": "Filter Column",
	"ed_sfc": "Second Filter Column",
    "ed_addcalc": "Add calculation here",
    "contains": "contains",
    "startsWith": "startsWith",
    "endsWith": "endsWith",
    "ed_ds": "Data Survey",
    "ed_os": "Oversight Survey",
    "ed_mr": "Only allow an enumerator to access their own data using the search() or pulldata() functions. Data submitted by other users will not be downloaded as reference data.",
    "ed_nb": "No Buttons",
    "ed_offline": "Offline and Online",
    "ed_online": "Online Only",
    "ed_csa": "changes successfully applied",
    "ed_csf": "changes failed",
    "ed_transd": "Translations done from language %s1 to language %s2",
    "ed_ow": "Overwrite existing translations",
    "ed_fe": "Filter Expression",
    "ed_e": "Expression",
    "ed_drl": "Default report logo",
    "ed_voice": "Voice only",
    "ed_na": "Not available",

    // Managed Forms
    "mf_id": "Include Deleted",
    "mf_cc": "Include Completed Cases",
    "mf_fd": "Form Data",
    "mf_md": "Management Data",
    "mf_sc": "Visible Columns",
    "mf_sc_i": "Select the columns that you want to see in the table then press the 'Apply' button. You can then sort on these columns",
    "mf_qr": "QR Columns",
    "mf_bc_i": "Select the columns that you want to see as barcodes in the table and in reports",
    "mf_st": "Submission Tracking",
    "mf_gr": "Get related",
    "mf_of": "Oversight Survey",
    "mf_no": "There are no oversight forms. They can be added in the \"Available Oversight Forms\" tab",
    "mf_sel": "Select the form that you want to manage and the oversight form that you want to manage it with.",
    "mf_ao": "Available oversight forms",
    "mf_mf": "Managed forms",
    "mf_ld": "Layer details",
    "mf_rf": "Reset Filters",

    "mf_tr": "A title is required",
    "mf_sr": "Summary Report",
    "mf_lock": "Lock for editing",
    "mf_rel": "Release",
    "mf_mr": "My records",
    "mf_ur": "Unassigned records",
    "mf_or": "Other records",
    "mf_max": "Max records",
    "mf_clear": "Clear single record view",
    "mf_sc": "Show Controls",
    "mf_hc": "Hide Controls",
    "mf_ts": "Table Settings",

    // User trail
    "ut_mps": "Max point separation to include in a single line",
    "ut_mps_err": "Max point separation must be greater than 0",

    // Logout
    "lo_out": "You are logged out of ",
    "lo_lo": "Logged Out",
    "lo_loa": "You should now close your browser in order to ensure cached log in details are removed",

    // Users
    "u_sec_mgr_del": "Require security manager privilege to delete data",
    "u_other_msg": "These settings apply to the current organisation",
    "u_show_u": "Show users with security group",
    "u_show_p": "Show users in project",
    "u_show_r": "Show users with role",
	"u_show_o": "Show users with organisation",
    "u_add": "Add User",
    "u_del": "Delete User",
    "u_mv": "Move To Organisation",
    "u_cms": "Case management Settings",
	"u_mv_ent": "Move To Enterprise",
    "u_add_p": "Add Project",
    "u_del_p": "Delete Project",
    "u_add_o": "Add Organisation",
    "u_del_o": "Delete Organisation",
	"u_add_e": "Add Enterprise",
	"u_del_e": "Delete Enterprise",
    "u_add_r": "Add Role",
    "u_add_s": "Add Setting",
    "u_edit_r": "Edit Role",
    "u_del_r": "Delete Role",
    "u_det": "User Details",
    "u_ident": "Ident (lowercase, no spaces or an email address)",
    "u_email": "Send email to new user",
    "u_set_p": "Specify password",
    "u_sg": "Security Groups",
    "u_det_p": "Project Details",
    "u_det_ur": "User Role Details",
    "u_det_o": "Organisation Details",
    "u_det_e": "Enterprise Details",
    "u_det_o_rep": "Report Publishing Options",
    "u_det_o_ft": "Mobile App Options",
	"u_det_o_wf": "Webform Options",
    "u_det_pbc": "Page background color",
	"u_det_paperbc": "Paper background color",
    "u_det_buttonbc": "Button color",
    "u_det_buttontc": "Button text colors",
    "u_det_headertc": "Heading text colors",
	"u_det_fp": "Footer position (from right in pixels)",
    "u_wf_hd": "Hide 'save as draft' checkbox",
    "u_ft_del": "Delete submitted results from the phone",
    "u_ft_prev_track_dis": "Prevent the disabling of location tracking",
    "u_ft_img_size": "Maximum pixels of the long edge of an image",
    "u_ft_img_camera": "Original size from camera",
    "u_ft_im": "GeoShape and GeoTrace Input Method",
    "u_ft_im_pt": "Placement by Tapping",
    "u_ft_im_man": "Manual location recording",
    "u_ft_im_auto": "Automatic location recording",
    "u_ri": "Recording Interval",
    "u_ft_vs": "Very small (640px)",
    "u_ft_s": "Small (1024px)",
    "u_ft_m": "Medium (2048px)",
    "u_ft_l": "Large (3072px)",
    "u_ft_ld": "Send location data on path of user",
    "u_ft_os": "Enable ODK style menus to delete, submit, edit and get new forms",
    "u_ft_am": "Enable ODK Admin menu",
    "u_ft_ex": "Disable exit menu",
    "u_ft_bgs": "Prevent background audio from being disabled",
    "u_ft_in": "Allow user to set instance name",
    "u_ft_mf": "Allow user to mark  forms as not finalized",
    "u_ft_rv": "Allow finalised forms to be opened for review",
    "u_ft_ft": "Force the use of tokens for logon",
    "u_ft_as": "Automatically Synchronise",
    "u_ft_ms": "Manual Synchronisation",
    "u_ft_sw": "WIFI",
    "u_ft_hrv": "High Resolution Video",
    "u_ft_sw_c": "WIFI and cellular network",
    "u_ft_sop": "Set on phone",
    "u_ft_tasks": "Maximum number of tasks to download",
	"u_ft_back_nav": "Backward navigation",
    "u_ft_nav": "Screen Navigation",
    "u_ft_pw_timeout": "Number of days before password expiry",
    "u_mobile_security": "Security",
    "u_mobile_security_h": "Access protection and authentication policies.",
    "u_mobile_menus": "Menus",
    "u_mobile_menus_h": "Expose or hide in-app menus. You must restart the mobile device to see changes in these options.",
    "u_mobile_completion": "Form Completion",
    "u_mobile_completion_h": "How forms are reviewed, finalized, and cleaned up.",
    "u_mobile_sync": "Sync & Connectivity",
    "u_mobile_sync_h": "Network and sync behavior for submissions.",
    "u_mobile_media": "Media",
    "u_mobile_media_h": "Capture quality and file sizing for media.",
    "u_mobile_tracking": "Tracking & Geolocation",
    "u_mobile_tracking_h": "Location capture, geofencing, and tracking safeguards.",

    "u_ft_pw_policy": "Password policy",
	"u_ft_never": "Never require logon",
	"u_ft_always": "Always require logon on application start",
	"u_ft_periodically": "Require logon periodically",
    "u_ft_swipes": "Use horizontal swipes",
    "u_ft_buttons": "Use forward/backward buttons",
    "u_ft_swipes_buttons": "Use swipes and buttons",
    "u_ft_sa": "Yes - always shown",
    "u_ft_sc": "Yes - collapsed",
    "u_ft_ss": "Enable server settings menu",
    "u_ft_md": "Enable user and identity menu",
	"u_ft_eg": "Enable Geo-fence",
    "u_det_o_email": "Email Options",
    "u_det_o_other": "Other Options",
    "u_det_o_limits": "Monthly Usage Limits",
    "u_other_edit": "Allow editing of results",
    "u_email_tasks": "Allow sending of task emails",
	"u_use_api": "Allow API access",
	"u_submissions": "Allow submissions",
	"u_notifications": "Allow notifications",
    "u_sms": "Allow SMS",
    "u_ratelimit": "Max number of API requests per minute per organisation",
    "u_load_mgmt": "Load Management",
    "u_api_max_records": "Max records per API request",
    "u_server_map_services": "Map Services",
    "u_server_messaging": "Messaging",
    "u_server_email": "Email Server",
    "u_mb_k": "Mapbox Key",
    "u_mb_a": "Mapbox Account",
    "u_mb_s": "Mapbox Style Id",
    "u_g_k": "Google Maps Key",
    "u_mt_k": "MapTiler Maps Key",
    "u_v_k": "Vonage Application Id",
    "u_v_whs": "Vonage Webhook Secret",
    "u_ar": "AWS Region for Email Service",
    "u_smtp": "Smtp Host",
    "u_e_type": "Email Type",
    "u_e_dom": "Email Domain",
    "u_e_nm": "Email user name",
    "u_e_si": "Server Identifier",
    "u_e_p": "Email password",
    "u_e_port": "Email Server port",
    "u_h_e": "Email to get Help",
    "u_rap": "Restrict access",
    "u_rap_d": "Administrators will be able to assign the selected users only to this project",
    "u_chg": "Changed by",
	"u_usage": "Usage Report",
    "u_attendance": "Attendance Report",
    "u_notification": "Notification Report",
	"u_access": "Form Access Report",
    "u_b_access": "Bundle Access Report",
    "u_survey": "Surveys in this organisation",
    "u_os": "Organisational Structure",
    "u_eou": "Enterprise and Organisation Administrators",
    "u_r_u": "Resource Usage Report",
    "u_sms_url": "SMS Url",
    "u_sens_data": "Sensitive Data",
    "u_sens_ao": "Admin Only",
    "u_sens_sq": "Signature Questions",
    "u_sens_nr": "No restrictions",
    "u_org_admin": "Organisation Administrator",
    "u_ent_admin": "Enterprise Administator",
    "u_server_owner": "Server Owner",
    "u_check_mv_u": "Are you sure you want to move the following users (%s1) to %s2",
	"u_check_mv_p": "Are you sure you want to move the following projects (%s1) to %s2",
	"u_check_mv_o": "Are you sure you want to move organisation \"%s1\" to enterprise \"%s2\"",
    "u_only_one_ent": "Only one enterprise has been created.  You need to create another one to move an organisation there",
    "u_co": "Current Organisation",
    "u_gr": "Get report",
    "u_sat": "Set as Theme",
    "u_bl": "WebForm Banner Logo (Height 50px)",
    "u_ml": "Main Logo",
    "u_nbc": "Navigation bar color",
    "u_nbtc": "Navigation bar text color",
    "u_eoi": "Enable Opt In",
    "admin": "Administrator",
    "analyst": "Analyst",
    "enterprise admin": "Enterprise Admin",
    "enum": "Enumerator",
    "manage": "Manage Data",
    "console admin": "Manage Console",
    "org admin": "Organisational Admin",
    "security": "Security Manager",
    "view data": "View Data",
    "manage tasks": "Manage Tasks",
    "view own data": "View Own Data",
    "dashboard": "Dashboard",
    "links": "Follow Links",
    "mcp access": "AI Access",
    "u_css_f": "CSS file",
    "u_css": "Upload a CSS style sheet",
    "u_css_ss": "Upload a new CSS style sheet for the server",
    "u_css_ss_org": "Upload a new CSS style sheet for the current organisation",
    "u_css_del": "Delete CSS style sheet",
    "u_css_sel": "Select a style sheet",
    "u_css_info": "Create a custom style sheet and then upload to the server",
    "u_clear_p": "Clear existing empty imported projects",
    "u_clear_u": "Clear existing imported users",
    "u_clear_r": "Clear existing imported roles",
    "u_api_rl": "Rate limit for data API",
    "u_mps": "Minimum password strength",
    "u_oo": "The user is currently in a different organisation and only the list of organisations that they belong to can be changed.",
    "u_code": "App Code",

    // Browse Results
    "br_ci": "Case Insensitive",
    "br_exact": "Exact Match",
    "br_cc": "Select the columns that you want to test for similarities in the data. For each column that you want to test you can select a function to broaden the range of values that will match.",
    "br_s": "Search",
    "br_sdx": "Soundex",
    "br_cd": "Use the table of data for this survey view",
    "br_tf": "for text questions, including select and select_one",
    "br_nf": "for numeric questions, including integer and decimal",

    // Shared Resources
    "sr_res": "Images Video Audio Files",
    "sr_nfc": "NFC Tags",
    "sr_uid": "NFC UID",
    "sr_geo": "Geographic locations",
    "ar_af": "Add Files",
    "sr_g": "Group",
    "sr_fn": "File Name",
    "sr_m_ph": "Enter a name for your map",
    "sr_m_d": "Enter a description for this map (optional)",
    "sr_m_mb": "For Mapbox this identifies the map, for example: mapbox/light-v10",
    "sr_cr": "Custom Reports",
    "sr_al": "Use Anonymous Location in Task",
    "sr_ul": "Update Existing Location",
    "sr_nl": "Save as New Location",
    "sr_sl": "Save Location",
    "sr_sm": "Survey Media Files",
    "sr_si": "The NFC UID and GPS coordinates that you specify here can be saved as a named location or used in the task as an anonymous location",
    "sr_rh": "Resource History",

    // languages
    "ar": "Arabic",
    "en": "English",
    "fr": "French",
    "hi": "Hindi",
    "pt": "Portuguese",
    "uk": "Ukrainian",
    "es": "Spanish",

    // Notifications
    "n_wf": "Web Form",
    "n_pdfp": "PDF (Portrait)",
    "n_pdfl": "PDF (Landscape)",
    "n_eq": "Email Question",
    "n_ep": "Email Meta Item",
    "n_smsq": "SMS Question",
    "n_share": "Get link",
    "n_cs_e": "Comma separated list of email addresses",
    "n_ea": "Email addresses",
    "n_eq": "Email question",
    "n_eqc": "Question that will contain the email addresses to use",
    "n_uiqc": "Question that will contain the user ident to notify",
    "n_esc": "Subject for the email (optional)",
    "n_inc_ref": "Include referenced surveys",
	"n_lo": "Only reference data from forms that were launched from this form",
    "task_reminder": "Task Reminder",
    "submission": "Submission",
    "console_update": "Console Update",
    "n_p": "Send reminder after",
    "n_oa": "Email original assignee",
    "n_ap": "Add placeholder",
    "n_si": "SMS Sender Id",
    "n_sms_n_tt": "Comma separated list of phone numbers in format +ccnnnnnnnnnn",
    "n_sms_q_tt": "Question that will contain the SMS numbers to use",
    "n_e_p_tt": "Meta item or preload that will contain an email address to use",
    "n_uq": "Update Question",
    "n_uv": "Update Value",
    "n_no_oversight": "There are no oversight surveys that can be used to update the data in the console.  Select an oversight form or load a new one.",
    "m_no_q": "There are no questions in the oversight survey that can be edited in bulk.  These questions need to be of type text, or select one, or select multiple",
    "n_nv": "A value for the update must be specified",
    "n_nq": "An update question must be specified",
    "n_curl": "Callback URL",
    "n_aa": "After alert %s1.",
    "n_as": "Assign survey %s1 to %s2.",
    "n_dow": "Day of the week",
    "n_dom": "Day of the month",
    "n_sent": "Notification sent",
    "n_scc": "Server Calculation Question",
    "n_val": "A server calculation question and value that will cause the notification to be triggered must be specified",
    "n_their_nbr": "Select their Number",
    "n_our_nbr": "Our Number",
    "n_spec_nbr": "Enter their Number",
    "n_cur_emails": "Select Email",
    "n_spec_nbr_h": "Enter the number with no spaces or commas.  Include the country code.  Optionally include a +. For example: +442071838451",


    // Roles
    "ro_b": "Apply the role filters to all bundle surveys with the role",
    "ro_b_w": "Warning. Applying roles to the bundle will override the roles set in all other surveys in the bundle. Do you want to proceed?",
    "ro_b_d": "Identical roles have been applied to all surveys in the bundle",
    "ro_fr": "Filter rows",
    "ro_fc": "Filter columns",
    "ro_f_group": "Filter Group",
    "ro_fr_rule": "Rows to be shown",
    "ro_aq": "Available Questions",
    "ro_fr_i": "Enter a rule that determines which rows will be shown for this role. Enclose question names in ${...}. Enclose text in single quotes. Make sure there are spaces between the question names and operators such as =.<br/>Example 1: ${region} = 'northern'<br/>Example 2: ${age} > 16. If no rule is specified then all rows will be shown.",
    "ro_fc_i": "Select the columns to show with this filter. If no columns are selected then all columns will be shown.",
    "ro_transform": "Long to wide data transformation",

    // Intervals
    "i_m": "minutes",
    "i_h": "hours",
    "i_d": "days",
    "i_sec": "second ago",
    "i_secs": "seconds ago",
    "i_min": "minute ago",
    "i_mins": "minutes ago",
    "i_hour": "hour ago",
    "i_hours": "hours ago",
    "i_day": "day ago",
    "i_days": "days ago",

    // Dashboard
    "d_sound": "Play sound on new alert",
    "d_rep_def_freq": "Frequency Report",
    "bar_h": "Horizontal Bar Chart",
    "bar_v": "Vertical Bar Chart",
    "stacked": "Stacked",
    "normalized": "Normalized",
    "pie": "Pie Chart",
    "line": "Line Chart",
    "wordcloud": "Word Cloud",
    "length": "Count",
    "d_qtr": "25%",
    "d_half": "50%",
    "d_3qtr": "75%",
    "d_full": "100%",
    "d_sec": "Seconds",
    "d_min": "Minutes",
    "d_hour": "Hours",
    "d_day": "Days",
    "d_c_day": "Count by day",
    "d_c_month": "Coount by month",
    "d_c_year": "Count by year",
    "d_add": "Add",
    "d_add_c": "Add Chart",
    "d_add_q": "Add New Query",
    "d_add_f": "Add New Form",
    "d_from_f": "Link From Form",
    "d_from_q": "Link From Question",
    "d_edit_web": "Edit in Web Form",
    "d_d_pdf": "Download PDF",
    "d_layers": "Map Layers",

    // Review
    "rev_upd_diff": "Update a different question to the source question",
    "rev_sel_pk": "Select data to change using a primary key",
    "rev_q_upd": "Question to update",
    "rev_text": "Text",
    "rev_upd": "Update",
    "rev_upd_t": "Update target question",
    "rev_upd_text": "Update Text Questions",
    "rev_rt": "Replace text",
    "rev_rc": "records with choice",
    "rev_fq": "from question",
    "rev_rw": "records with",
    "rev_rdm": "Review Data Modifications",
    "rev_ci": "Change Id",
    "rev_rb": "Reversed By",
    "rev_cb": "Changed By",
    "rev_rcn": "Reverse Change Number",
    "rev_det": "Details for change number",
    "rev_cq": "Changed Question",
    "rev_sc": "Selcted Choice",
    "rev_usc": "Un-Selected Choice",

    // Chart Questions
    "ch_ds": "Chart Data",

    // Billing
    "quantity": "Quantity",
    "free": "Free",
    "unitCost": "Unit Cost",
    "amount": "Amount",
    "server": "Server",
    "submissions": "Submissions",
    "submissions_i": "A limit of zero is unlimited",
    "disk": "Disk",
    "rekognition": "AWS Rekognition",
    "rekognition_i": "images",
    "sentiment": "AWS Comprehend",
    "sentiment_i": "requests",
    "translate": "AWS Translate",
    "awsses": "AWS Simple Email Service",
    "translate_i": "letters",
	"transcribe_medical": "AWS Transcribe Medical",
	"transcribe_medical_i": "seconds of audio",
    "transcribe": "AWS Transcribe",
    "transcribe_i": "seconds of audio",
	"static_map": "Mapbox Static Map",
    "google_static_map": "Google Static Map",
    "maptiler_static_map": "MapTiler Static Map",
	"monthly": "Monthly Charge",
    "org_bill_rpt": "Organisation Usage Report",
    "bill_enable": "Enable Billing",
    "bill_from": "Applies From",
    "bill_chg_date": "Date Changed",
    "bill_norates": "No rates have been set for this %s1. If billing is enabled then the rates for the %s2 will be used",
	"billing_disabled_msg": "Billing has been disabled",

	// Reports
	"rep_usage_project": "Usage by project",
	"rep_usage_survey": "Usage by survey",
    "rep_usage_device": "Usage by device",
    "rep_values_question": "Values Question",
    "rep_values": "Values",
    "rep_wide_columns": "Wide Columns",
    "rep_msg_min_keys": "You must specify at least one key question if transform is enabled",
    "rep_msg_v_q": "You must specify a values question if transform is enabled",
	"rep_msg_min_values": "You must specify at least one value if transform is enabled",
	"rep_msg_min_wc": "You must specify at least one wide column if transform is enabled",
    "rep_ch": "Column Heading",
    "rep_inc_temp": "Include temporary users",
    "rep_inc_alltime": "Include data for 'all time' as well as selected month",
    "rep_gen": "Generated Reports",
    "rep_gen_ov": "Reports generated by a background process can be downloaded from here.  They will be kept for 2 weeks and then automatically deleted",
    "locations_distance": "Distance travelled",
    "locations_kml": "KML",
    "rep_au": "Include all users in project",
    "rep_d": "Generation time (seconds)",
    "rep_my": "Month and year of the report",

    // Console
    "co_dd": "Drill Down",
    "co_up": "Up",
    "co_tid": "Include text value in Download",
    "co_b": "Bulk Change",
    "co_drag": "Drag and drop questions to change the order in which they are shown",

    // Mailouts
    "mo_anon": "Anonymous",
    "mo_ce": "Clear Existing Unsent Emails",
    "mo_ns": "Mailout not selected",
    "mo_na": "Not Authorised",
    "mo_nf": "The survey form was not found",
    "mo_ss": "This survey form has been sucessfully submitted",
    "mo_exp": "This survey form has expired",
    "mo_del_done": "This survey form has been deleted",
    "mo_se": "System Error",
    "mo_del": "Are you sure that you want to delete mailout \"%s1\".  Recipients who have already been sent emails will no longer be able to respond. ",
    "mo_blocked": "This survey form has been blocked.  Try again later or contact the administrator to unblock",
    "mo_import": "Import Emails",
    "mo_export": "Export Emails",
    "mo_subject": "Email Subject",
    "mo_content": "Email Content",
    "mo_sent": "Sent",
    "mo_sd": "Status Details",
    "mo_ms": "Multiple Submissions",
    "mo_send": "Are you sure you are ready to send the emails?",
    "mo_gen": "Are you sure you are want to generate the links now?  They are created automatically when you use the email unsent button however you can generate them now if you want to send them via a different channel. For single submission links, once created they will expire in 30 days.",
    "mo_oi": "The following opt-in messages have been sent and are awaiting a response. ",

    // AWS Services
    "svc_at": "Auto Translate",
    "svc_from_l": "From Language",
    "svc_to_l": "To Language",

    // Logs
    "log_hr": "Hourly Summary",
    "log_sd": "Survey Details",

    // Login
    "li_success": "Login successful",
    "li_failure": "Login failure",
    "li_continue": "Return to applications",

    // Case Management
    "cm_fs": "Final status",
    "cm_sq": "Status Question",
    "cm_cq": "Criticality Question",
    "cm_aa": "Add an Alert",
    "cm_dcms": "There is already a case management event with that name for this survey group",
    "cm_alert_a": "Alert if not finalised within",
    "cm_a": "Case management alert",
    "cm_bar": "Bar",
    "cm_col": "Column",
    "cm_pie": "Pie",
    "cm_alert": "Case Management Alert",
    "cm_ns": "Select a survey to see case data",
    "cm_oc": "Count of cases opened and closed per day",
    "cm_p": "Show data for last",
    "cm_7days": "7 days",
    "cm_14days": "14 days",
    "cm_30days": "30 days",
    "cm_areq": "A status question and a final status value must be specified if you are using alerts",
    "cm_bn": "Bundle Name (Topic)",
    "cm_bd": "Bundle Description",

    "fp_nm": "No matches found",

    "pw_gen": "Generate Password",
    "pw_mm": "The password and its confirm are not the same",

    // Server state
    "ss_title": "Server State",

    // sms
    "sms_their_q": "Question for calling number",
    "sms_conv_q": "Conversation Question"
});


/*
This file is part of SMAP.

SMAP is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

SMAP is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with SMAP.  If not, see <http://www.gnu.org/licenses/>.

*/

define('localise',['jquery', 'i18n!lang_location/nls/lang'], function($, lang) {
	
	var dtLangFiles = {
			en: "",
			es: "/js/libs/DataTables.i18n/es.json",
			ar: "/js/libs/DataTables.i18n/ar.json",
			fr: "/js/libs/DataTables.i18n/fr.json",
			pt: "/js/libs/DataTables.i18n/pt.json",
			hi: "/js/libs/DataTables.i18n/hi.json"
	}
	
	window.localise = {	// Set global while migrating app to AMD
		
		
		setlang: function () {

			// Content
			$(".lang").each(function(index) {
				var $this = $(this);
				var code = $this.data("lang");
				if(code) {		
					$this.html(lang[code]);
				}
			});

			// tooltips
			$(".lang_tt").each(function(index) {
				var $this = $(this);
				var code = $this.data("lang_tt");
				if(code) {
					$this.prop("title", lang[code]);
				}
			});

			// placeholders
			$(".lang_ph").each(function(index) {
				var $this = $(this);
				var code = $this.data("lang_ph");
				if(code) {
					$this.prop("placeholder", lang[code]);
				}
			});

			if(typeof responsiveMobileMenu === "function") {
				rmmResizeLabels();		// Resize menus
			}
		},
		set: lang,
		dt: function() {
			return dtLangFiles[gUserLocale];
		}
	}
	
	return localise;
});
/*
 This file is part of SMAP.

 SMAP is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 SMAP is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with SMAP.  If not, see <http://www.gnu.org/licenses/>.

 */

/*
 * Quick solution to issue of legacy globals after migrating to AMD / require.js
 */
define('globals',[],function () {
    window.globals = {

        // Security groups
        GROUP_ADMIN: 1,
        GROUP_ANALYST: 2,
        GROUP_ENUM: 3,
        GROUP_ORG_ADMIN : 4,
        GROUP_MANAGE: 5,
        GROUP_SECURITY: 6,
        GROUP_VIEW_DATA: 7,
        GROUP_ENTERPRISE : 8,
        GROUP_OWNER : 9,
		GROUP_VIEW_OWN_DATA : 10,
	    GROUP_MANAGE_TASKS : 11,
	    GROUP_DASHBOARD : 12,
        GROUP_LINKAGES : 13,
        GROUP_CONSOLE_ADMIN : 14,
        GROUP_MCP_ACCESS : 15,

        REC_LIMIT: 200,     // Page size for table views in analysis
	    MAP_REC_LIMIT: 10000,    // Max size for map views in analysis

        gProjectList: undefined,
        gRoleList: undefined,
        gCmSettings: undefined,
        gCurrentProject: 0,
        gCurrentSurvey: 0,
        gCurrentSurveyIdent: undefined,
	    gGroupSurveys: {},
	    gSubForms: {},
        gCurrentForm: 0,
        gCurrentLayer: undefined,
        gLoggedInUser: undefined,
        gEditingReportProject: undefined,   		// Set if fieldAnalysis called to edit a report
        gIsAdministrator: false,
        gIsEnum: false,
        gIsAnalyst: false,
	    gIsDashboard: false,
        gIsManage: false,
        gIsOrgAdministrator: false,
        gIsSecurityAdministrator: false,
        gIsEnterpriseAdministrator: false,
        gIsLinkFollower: false,
        gIsServerOwner: false,
        gIsConsoleAdmin: false,
        gViewData: false,
	    gManageTasks: false,
        gBillingData: false,
        gOrgBillingData: false,
        gSendTrail: 'off',
        gViewIdx: 0,
        gSelector: new Selector(),
        gOrgId: 0,
        gTimezone: undefined,
	    gEnterpriseName: undefined,
	    gSetAsTheme: undefined,
	    gNavbarColor: undefined,

        gRegions: undefined,
        gRegion: {},

        gServerCanSendEmail: false,

        // Reports
        gEmailEnabled: false,
        gFacebookEnabled: false,
        gTwitterEnabled: false,

        // Tasks
        gCurrentUserId: undefined,
        gCurrentUserName: undefined,
        gAssignmentsLayer: undefined,
        gPendingUpdates: [],
        gCurrentTaskGroup: undefined,
	    gCurrentMailout: undefined,
        gTaskList: undefined,
        gCurrentSurveyIndex: 0,
	    gCurrentInstance: undefined,
        gAlertSeen: false,
        gLastAlertTime: undefined,

        // Editor
        gExistingSurvey: false,		// Set true if modifying an existing survey
        gElementIndex: 0,			// Manage creation of unique identifier for each element (question, option) in editor
        gHasItems: false,			// Set true if there are questions or choice lists in the survey
        gNewQuestionButtonIndex: 0,	// Manage creation of unique identifier for buttons that add new questions
        gNewOptionButtonIndex: 0,
        gSId: 0,
        gLanguage: 0,
        gLanguage1: 0,
        gLanguage2: 0,
        errors: [],
        changes: [],
        gErrorPosition: 0,
        gSelProperty: 'label',
        gSelLabel: 'Question Text',
        gSelQuestionProperty: 'label',
        gSelQuestionLabel: 'Question Text',
        gSelChoiceProperty: 'label',
        gSelChoiceLabel: 'Question Text',
        gIsQuestionView: true,
        gShowingChoices: false,
        gMaxOptionList: 0,
        gLatestOptionList: undefined,	// Hack to record the last option list name added
	    gCsvFiles: undefined,

        gListName: undefined,					// Choice Modal parameters, Set if started from choice list view
        gOptionList: undefined,					// The option list name applying to this set of choices
        gSelOptionId: undefined,				// Selected option index
        gFormIndex: undefined,					// Selected form index
        gItemIndex: undefined,					// Selected question index
        gSelectedFilters: undefined,
        gFilterArray: undefined,

        gSaveInProgress: false,

        // Dashboard
        gMainTable: undefined,			// Data tables
        gReports: undefined,			// reports
        gCharts: {},					// charts
	    gRecordMaps: [],                // Maps shown when editing a record
	    gRecordChangeMaps: [],          // Maps shown when viewing change history
        gMapLayersShown: false,
        gViewId: 0,						// Current survey view

	    gTraining: undefined,
	    gRefreshRate: 0,

        gMapboxDefault: undefined,		// Mapbox key
        
        model: new Model()

    }

    return window.globals;

    function Selector() {

        this.dataItems = new Object();
        this.surveys = new Object();
        this.surveysExtended = new Object();
        this.surveyLanguages = new Object();
        this.surveyQuestions = new Object();
        this.surveyMeta = new Object();
        this.surveyAlerts = new Object();
        this.questions = new Object();
        this.allSurveys;				// Simple list of surveys
        this.allRegions;
        this.sharedMaps;
        this.views = [];			// Simple list of views
        this.maps = {};				// map panels indexed by the panel id
        this.changed = false;
        this.SURVEY_KEY_PREFIX = "surveys";
        this.TASK_KEY = "tasks";
        this.TASK_COLOR = "#dd00aa";
        this.SURVEY_COLOR = "#00aa00";
        this.SELECTED_COLOR = "#0000aa";
        this.currentPanel = "map";

        /*
         * Get Functions
         */
        this.getAll = function () {
            return this.dataItems;
        };

        this.getItem = function (key) {
            return this.dataItems[key];
        };

        // Return all the table data available for a survey
        this.getFormItems = function (sId) {
            var tableItems = new Object();
            for (var key in this.dataItems) {
                var item = this.dataItems[key];
                if (item.table == true && item.sId == sId) {
                    tableItems[key] = item;
                }
            }
            return tableItems;
        };

        this.getSurvey = function (key) {
            return this.surveys[key];
        };

        this.getSurveyExtended = function (key) {
            return this.surveysExtended[key];
        };

        this.getSurveyQuestions = function (sId, language) {
            var langQ = this.surveyQuestions[sId];
            if (langQ) {
                return langQ[language];
            } else {
                return null;
            }
        };

        this.getSurveyMeta = function (key) {
            return this.surveyMeta[key];
        };

        this.getSurveyAlerts = function (key) {
            return this.surveyAlerts[key];
        };

        this.getSurveyLanguages = function (key) {
            return this.surveyLanguages[key];
        };

        // Returns the list of surveys on the home server
        this.getSurveyList = function () {
            return this.allSurveys;
        };

        this.getRegionList = function () {
            return this.allRegions;
        };

        this.getSharedMaps = function () {
            return this.sharedMaps;
        };

        // deprecate question meta should be replaced by all question details in the question list
        this.getQuestion = function (qId, language) {
            var langQ = this.questions[qId];
            if (langQ) {
                return langQ[language];
            } else {
                return null;
            }
        };

        /*
         * Get the question details that came with the question list
         * This approach should replace the concept of "question meta"
         */
        this.getQuestionDetails = function (sId, qId, language) {
            var qList = this.getSurveyQuestions(sId, language),
                i;

            if (qList) {
                for (i = 0; i < qList.length; i++) {
                    if (qList[i].id == qId) {
                        return qList[i];
                    }
                }
            }
            return null;
        };

        this.hasQuestion = function (key) {
            if (this.questions[key] != undefined) {
                return true;
            } else {
                return false;
            }
        };

        // Return the list of current views
        this.getViews = function () {
            return this.views;
        };

        // Return a map if it exists
        this.getMap = function (key) {
            return this.maps[key];
        };


        /*
         * Set Functions
         */
        this.addDataItem = function (key, value) {
            this.dataItems[key] = value;
            this.changed = true;
        };

        this.clearDataItems = function () {
            this.dataItems = new Object();
        };

        this.clearSurveys = function () {
            this.surveys = new Object();
            this.surveyLanguages = new Object();
            this.surveyQuestions = new Object();
            this.surveyMeta = new Object();
            this.surveyAlerts = new Object();
            this.questions = new Object();
            this.allSurveys = undefined;
            this.allRegions = undefined;
        };

        this.setSurveyList = function (list) {
            this.allSurveys = list;
            if (typeof list[0] !== "undefined") {
                this.selectedSurvey = list[0].sId;
            }
        };

        this.setSurveyLanguages = function (key, value) {
            this.surveyLanguages[key] = value;
        };

        this.setSurveyQuestions = function (sId, language, value) {
            var langQ = new Object();
            langQ[language] = value;
            this.surveyQuestions[sId] = langQ;
        };

        this.setSurveyMeta = function (key, value) {
            this.surveyMeta[key] = value;
        };

        this.setSurveyAlerts = function (key, value) {
            this.surveyAlerts[key] = value;
        };

        this.setRegionList = function (list) {
            this.allRegions = list;
        };

        this.setSharedMaps = function (list) {
            this.sharedMaps = list;
        };

        this.addSurvey = function (key, value) {
            this.surveys[key] = value;
        };

        this.addSurveyExtended = function (key, value) {
            this.surveysExtended[key] = value;
        };

        this.setSelectedSurvey = function (survey) {
            this.selectedSurvey = survey;
        };

        this.setSelectedQuestion = function (id) {
            this.selectedQuestion = id;
        };

        this.addQuestion = function (qId, language, value) {
            var langQ = this.questions[qId];
            if (!langQ) {
                this.questions[qId] = new Object();
                langQ = this.questions[qId];
            }
            langQ[language] = value;
        };

        // Set the list of views to the passed in array
        this.setViews = function (list) {
            this.views = list;
        };

        // Set the passed in map into the maps object indexed by key
        this.setMap = function (key, value) {
            this.maps[key] = value;
        };

    }

    /*
     * Model for Survey editing
     */
    function Model() {

        this.survey = undefined;
        this.translateChanges = [];
        this.currentTranslateChange = 0;
        this.savedSettings = undefined;
        this.forceSettingsChange = false;

	    // A list of valid appearances for each question type
	    this.qAppearances = {
		    'begin group': ['page', 'w', 'no-collapse'],
            'begin repeat': ['extendable', 'no-collapse'],
		    string: ['numbers', 'thousands-sep', 'w', 'url'],
		    note: ['w'],
            select1: ['select1_type', 'search', 'likert', 'no-buttons', 'w'],
            select: ['select_type', 'search', 'no-buttons', 'w'],
            image: ['image_type', 'selfie', 'new', 'w'],
            int:['thousands-sep', 'w'],
		    geopoint:['placement-map', 'w'],
		    audio:['w', 'new'],
		    video:['selfie', 'w', 'new'],
		    barcode:['read_nfc', 'w'],
		    date:['date_type', 'w'],
		    dateTime:['date_type', 'no-calendar', 'w'],
		    time:['w'],
            decimal:['thousands-sep', 'bearing', 'w'],
		    geotrace:['placement-map', 'w'],
		    geoshape:['placement-map', 'w'],
		    acknowledge:['w'],
		    range:['w', 'rating', 'vertical', 'picker'],
		    file:['w'],
		    rank:['w'],
            geocompound:['w', 'placement-map']
	    };

	    this.appearanceDetails = {
		    'page': {
			    field: 'a_page',
			    type: 'select',
                rex: 'field-list|table-list',
                valIsAppearance: true,
			    value_offset: 0,
                undef_value: ''
		    },
		    'image_type': {
			    field: 'a_image_type',
			    type: 'select',
			    rex: 'annotate|draw|signature',
			    valIsAppearance: true,
			    value_offset: 0,
			    undef_value: ''
		    },
		    'select1_type': {
			    field: 'a_select1_type',
			    type: 'form',
			    rex: 'minimal|quick$|autocomplete|columns|quickcompact|image-map|compact'
		    },
		    'select_type': {
			    field: 'a_select_type',
			    type: 'form',
			    rex: 'minimal|autocomplete|columns|image-map|compact|autocomplete-minimal'
		    },
		    'date_type': {
			    field: 'a_date_type',
			    type: 'select',
			    rex: 'no-calendar|month-year|year|coptic|ethiopian|islamic|myanmar|persian|bikram-sambat',
			    valIsAppearance: true,
			    value_offset: 0,
			    undef_value: ''
		    },
		    'no-calendar': {
			    field: 'a_no_calendar',
			    type: 'boolean',
			    rex: 'no-calendar'
		    },
		    'placement-map': {
			    field: 'a_placement-map',
			    type: 'boolean',
			    rex: 'placement-map'
		    },
		    'search': {
			    field: 'a_search',
			    type: 'form',
			    rex: 'search\\(|lookup_choices\\('
		    },
		    'rating': {
			    field: 'a_rating',
			    type: 'boolean',
			    rex: 'rating'
		    },
		    'likert': {
			    field: 'a_likert',
			    type: 'boolean',
			    rex: 'likert'
		    },
		    'no-buttons': {
			    field: 'a_no_buttons',
			    type: 'boolean',
			    rex: 'no-buttons'
		    },
		    'selfie': {
			    field: 'a_selfie',
			    type: 'boolean',
			    rex: 'selfie'
		    },
		    'new': {
			    field: 'a_new',
			    type: 'boolean',
			    rex: 'new'
		    },
		    'read_nfc': {
			    field: 'a_read_nfc',
			    type: 'boolean',
			    rex: 'read_nfc'
		    },
		    'vertical': {
			    field: 'a_vertical',
			    type: 'boolean',
			    rex: 'vertical'
		    },
		    'picker': {
			    field: 'a_picker',
			    type: 'boolean',
			    rex: 'picker'
		    },
		    'bearing': {
			    field: 'a_bearing',
			    type: 'boolean',
			    rex: 'bearing'
		    },
            'thousands-sep': {
                field: 'a_sep',
                type: 'boolean',
                rex: 'thousands-sep'
            },
		    'no-collapse': {
			    field: 'a_no_collapse',
			    type: 'boolean',
			    rex: 'no-collapse',
                value_offset: 0
		    },
            'extendable': {
                field: 'a_extendable',
                type: 'boolean',
                rex: 'extendable'
            },
		    'numbers': {
			    field: 'a_numbers',
			    type: 'boolean',
			    rex: 'numbers'
		    },
		    'url': {
			    field: 'a_url',
			    type: 'boolean',
			    rex: 'url'
		    },
		    'w': {
			    field: 'a_width',
			    type: 'select',
                rex: 'w10|w[1-9]',
                value_offset: 1,
                undef_value: ''
		    }
	    };

        // A list of valid parameters for each question type
        this.qParams = {
            string: ['rows', 'auto_annotate', 'source', 'from_lang', 'to_lang', 'medical', 'med_type'],
	        calculate: ['auto_annotate', 'source', 'from_lang', 'to_lang', 'medical', 'med_type'],
	        barcode: ['auto'],
            image: ['max-pixels', 'auto'],
	        video: ['auto'],
	        audio: ['auto', 'quality'],
            range: ['start', 'end', 'step'],
            select: ['randomize'],
            select1: ['randomize'],
            rank: ['randomize'],
            parent_form: ['form_identifier', 'key_question', 'auto'],
	        child_form: ['form_identifier', 'key_question', 'auto'],
	        geopoint: ['auto'],
            geotrace: ['geotextlength'],
            geoshape: ['geotextlength'],
            geocompound: ['geotextlength'],
            'begin repeat':['ref', 'instance_order', 'instance_count', 'key_policy'],
	        chart: ['chart_type', 'stacked', 'normalized']
        };

        this.paramDetails = {
	        rows: {
	            field: 'p_rows',
                type: 'integer'
            },
            geotextlength: {
                field: 'p_geotextlength',
                type: 'integer'
            },
            'max-pixels': {
	            field: 'p_max_pixels',
                type: 'integer'
            },
            start: {
	            field: 'p_start',
                type: 'number'
            },
	        end: {
		        field: 'p_end',
		        type: 'number'
	        },
	        step: {
		        field: 'p_step',
		        type: 'number'
	        },
	        randomize: {
		        field: 'p_randomize',
		        type: 'boolean'
	        },
	        auto: {
		        field: 'p_auto',
		        type: 'boolean'
	        },
	        quality: {
		        field: 'p_quality',
		        type: 'select'
	        },
	        auto_annotate: {
		        field: 'p_auto_annotate',
		        type: 'boolean'
	        },
	        medical: {
		        field: 'p_medical',
		        type: 'boolean'
	        },
	        med_type: {
		        field: 'p_med_type',
		        type: 'select'
	        },
	        source: {
		        field: 'p_source',
		        type: 'select'
	        },
	        from_lang: {
		        field: 'from_lang',
		        type: 'select'
	        },
	        to_lang: {
		        field: 'to_lang',
		        type: 'select'
	        },
	        form_identifier: {
		        field: 'p_form_identifier',
		        type: 'select'
	        },
	        key_question: {
		        field: 'p_key_question',
                type: 'select'
            },
	        ref: {
		        field: 'p_ref',
		        type: 'select'
	        },
	        instance_order: {
		        field: 'p_instance_order',
		        type: 'select'
	        },
	        instance_count: {
		        field: 'p_instance_count',
		        type: 'integer'
	        },
	        key_policy: {
		        field: 'p_key_policy',
		        type: 'select'
	        },
	        chart_type: {
		        field: 'p_chart_type',
		        type: 'select'
	        },
	        stacked: {
		        field: 'p_stacked',
		        type: 'boolean'
	        },
	        normalized: {
		        field: 'p_normalized',
		        type: 'boolean'
	        },
	        _other: {
		        field: 'p_other',
		        type: 'text'
	        }
        };

        this.qTypes = [{
	            name: "Text",
	            trans: "rev_text",
	            type: "string",
	            icon: "font",
	            canSelect: true,
	            visible: true,
		        source: "user",
		        compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field", "conversation", "phone"]
            },
            {
                name: "Note",
                type: "note",
                trans: "c_note",
                icon: "pencil-alt",
                canSelect: true,
                visible: true,
                source: "user",
	            compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field", "conversation", "phone"]
            },
            {
                name: "Select One",
                type: "select1",
                trans: "ed_s1",
                image: "/images/select1_64.png",
                canSelect: true,
                visible: true,
				source: "user",
	            compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field", "conversation", "phone"]
            },
            {
                name: "Select Multiple",
                type: "select",
                trans: "ed_s",
                image: "/images/select_64.png",
                canSelect: true,
                visible: true,
                source: "user",
	            compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field", "conversation", "phone"]
            },
            {
                name: "Form",
                type: "begin repeat",
                trans: "c_rep_type",
                icon: "redo-alt",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "Group",
                type: "begin group",
                trans: "sr_g",
                icon: "folder-open",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "Image",
                type: "image",
                trans: "ed_image",
                icon: "camera",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "Integer",
                type: "int",
                trans: "ed_int",
                text: "#",
                canSelect: true,
                visible: true,
                source: "user",
                compatTypes: ["pdf_field"]
            },
            {
                name: "Phone Number",
                type: "phone",
                trans: "c_phone",
                icon: "phone",
                canSelect: true,
                visible: true,
                source: "user",
                compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field", "conversation"]
            },
            {
                name: "Conversation",
                type: "conversation",
                trans: "ed_mat_c",
                icon: "sms",
                canSelect: true,
                visible: true,
                compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field", "phone"]
            },
            {
                name: "GPS Point",
                type: "geopoint",
                trans: "ed_gps",
                icon: "map-marker-alt",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "Calculation",
                type: "calculate",
                trans: "ed_calc",
                calculation: true,
                image: "/images/calc_64.png",
                canSelect: true,
                visible: true,
                source: "user",
	            compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field"]
            },
            {
                name: "Audio",
                type: "audio",
                trans: "ed_audio",
                icon: "volume-up",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "Video",
                type: "video",
                trans: "ed_video",
                icon: "video",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "Barcode",
                type: "barcode",
                trans: "ed_bc",
                icon: "barcode",
                canSelect: true,
                visible: true,
                source: "user",
                compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field"]
            },
            {
                name: "Date",
                type: "date",
                trans: "c_date",
                icon: "calendar-alt",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "Date and Time",
                type: "dateTime",
                trans: "ed_dt",
                icon: "calendar-alt, clock",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "Time",
                type: "time",
                trans: "ed_t",
                icon: "clock",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "Decimal",
                type: "decimal",
                trans: "ed_dec",
                text: "#.#",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "GPS Line",
                type: "geotrace",
                trans: "ed_gps_line",
                image: "/images/linestring_64.png",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "GPS Area",
                type: "geoshape",
                trans: "ed_gps_area",
                image: "/images/polygon_64.png",
                canSelect: true,
                visible: true,
                source: "user"
            },
            {
                name: "Acknowledge",
                type: "acknowledge",
                trans: "ed_ack",
                text: "OK",
                canSelect: true,
                visible: true,
                source: "user",
                compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field"]
            },
            {
                name: "Range",
                type: "range",
                trans: "ed_range",
                icon: "arrows-alt-h",
                text: "Range",
                canSelect: true,
                visible: true,
                source: "user",
	            compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field"]
            },
            {
                name: "Chart",
                type: "chart",
                trans: "c_chart",
                icon: "chart-bar",
                text: "Chart",
                canSelect: true,
                visible: true,
                source: "user",
                compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field"]
            },
	        {
		        name: "Parent Form",
		        type: "parent_form",
		        trans: "c_parent_form",
		        icon: "file-upload",
		        text: "Parent Form",
		        canSelect: true,
		        visible: true,
		        source: "user"
	        },
	        {
		        name: "Child Form",
		        type: "child_form",
		        trans: "c_child_form",
		        icon: "file-download",
		        text: "Child Form",
		        canSelect: true,
		        visible: true,
		        source: "user"
	        },
            {
                name: "File",
                type: "file",
                trans: "c_file",
                icon: "file",
                canSelect: true,
                visible: true,
                source: "user",
                compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field"]
            },
            {
                name: "Rank",
                type: "rank",
                trans: "c_rank",
                icon: "sort-amount-down-alt",
                canSelect: true,
                visible: true,
                source: "user",
                compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field"]
            },
	        {
		        name: "Server Calculation",
		        type: "server_calculate",
		        trans: "ed_s_calc",
		        calculation: true,
		        image: "/images/server_calc_64.png",
		        canSelect: true,
		        visible: true,
		        source: "user",
		        compatTypes: ["string", "select1", "select", "calculate", "rank", "calculate_server", "note", "pdf_field"]
	        },
            {
                name: "Compound Pdf Image",
                type: "pdf_field",
                trans: "ed_ci",
                calculation: false,
                icon: "pallet",
                canSelect: true,
                visible: false,
            },
            {
                name: "Compound map",
                type: "geocompound",
                trans: "ed_cm",
                calculation: false,
                icon: "map-pin",
                canSelect: true,
                visible: false,
                compatTypes: ["geotrace"]
            },
            {
                name: "Unknown Type",
                icon: "ellipses-h",
                canSelect: false
            }
        ];

        // Set the survey model
        this.setSurveyData = function (data) {
            this.survey = data;
            this.survey.forms_orig = $.extend(true, {}, data.forms);
            this.survey.optionLists_orig = $.extend(true, {}, data.optionLists);
        }

        // Save the settings for the survey
        this.save_settings = function () {

            var settings = JSON.stringify(this.getSettings(true));

            addHourglass();
            $.ajax({
                type: "POST",
                contentType: "application/x-www-form-urlencoded",
                cache: false,
                data: { settings: settings },
                url: "/surveyKPI/surveys/save_settings/" + globals.gCurrentSurvey,
                success: function (data, status) {
                    removeHourglass();
                    if(handleLogout(data)) {
                        globals.model.savedSettings = settings;
                        globals.model.survey.pdfTemplateName = data;
                        globals.model.forceSettingsChange = false;
                        $('#save_settings').prop("disabled", true);

                        $('.formName').text(globals.model.survey.displayName);
                        $('#m_media').prop('href', '/app/resources.html?survey=true&survey_name=' + globals.model.survey.displayName);

                        $('#settingsModal').modal("hide");
                    }
                },
                error: function (xhr, textStatus, err) {
                    removeHourglass();
                    if(handleLogout(xhr.responseText)) {
                        if (xhr.readyState == 0 || xhr.status == 0) {
                            return;  // Not an error
                        } else {
                            bootbox.alert("Error saving settings. " + htmlEncode(xhr.responseText));
                        }
                    }
                }
            });

        };

        // Modify a label for a question or an option called from translate where multiple questions can be modified at once if the text is the same
        this.modLabel = function (language, changedQ, newVal, element, prop) {

            var labelMod = {
                changeType: prop,
                action: "update",
                items: []
            }

            var i,
                label = {},
                item,
                item_orig,
                qname,
                translation;


            for (i = 0; i < changedQ.length; i++) {
                translation = {
                    changeType: prop,
                    action: "update",
                    source: "editor"

                };

                // For questions
                if (typeof changedQ[i].form !== "undefined") {

                    label.formIndex = changedQ[i].form;
                    label.itemIndex = changedQ[i].question;
                    item = this.survey.forms[label.formIndex].questions[label.itemIndex];
                    item_orig = this.survey.forms_orig[label.formIndex].questions[label.itemIndex];

                    label.type = "question";
                    label.name = item.name;
                    label.prop = "label";
                    label.qId = item.id;

	                if(changedQ[i].constraint_msg) {
		                label.propType = "constraint_msg";
		                label.oldVal = item_orig.labels[language][label.propType];
	                } else if(changedQ[i].required_msg) {
		                label.propType = "required_msg";
		                label.oldVal = item_orig.labels[language][label.propType];
	                } else if(changedQ[i].guidance_hint) {
		                label.propType = "guidance_hint";
		                label.oldVal = item_orig.labels[language][label.propType];
	                } else if(changedQ[i].hint) {
		                label.propType = "hint";
		                label.oldVal = item_orig.labels[language][label.propType];
	                } else {
		                label.propType = "text";
		                label.oldVal = item_orig.labels[language][element];
	                }

                } else {
	                // For options
                    label.optionList = changedQ[i].optionList;
                    label.optionIdx = changedQ[i].option;

                    item = this.survey.optionLists[label.optionList].options[label.optionIdx];
                    item_orig = this.survey.optionLists_orig[label.optionList].options[label.optionIdx];

                    label.type = "option";
                    label.name = item.value;
	                label.propType = "text";
	                label.oldVal = item_orig.labels[language][element];
                }

                label.newVal = newVal;

                label.element = element;
                label.languageName = language;
                label.allLanguages = false;

                label.languageName = this.survey.languages[language].name;			// For logging the event
                var form = this.survey.forms[label.formIdx];

                if (item.text_id) {
                    label.key = item.text_id;
                } else {
                    // Create reference for this new Label
                    if (typeof changedQ[i].form !== "undefined") {
                        label.key = "/" + form.name + "/" + item.name + ":label";	// TODO hint
                    } else {
                        label.key = "/" + form.name + "/" + qname + "/" + item.name + ":label";
                    }
                }

                translation.property = label;

                labelMod.items.push(translation);
            }

            this.removeDuplicateTranslateChange(this.translateChanges, labelMod);
            if (labelMod.items[0].property.newVal !== labelMod.items[0].property.oldVal) {		// Add if the value has changed
                this.currentTranslateChange = this.translateChanges.push(labelMod) - 1;
                //this.doChange();				// Apply the current change
            }

            $('.m_save_survey').find('.badge').html(this.translateChanges.length);
            if (this.translateChanges.length > 0) {
                $('.m_save_survey').removeClass('disabled').prop('disabled', false);
	            $('#m_auto_translate').closest('li').addClass("disabled").prop("disabled", true);
            } else {
                $('.m_save_survey').addClass('disabled').prop('disabled', true);
	            $('#m_auto_translate').closest('li').removeClass("disabled").prop("disabled", false);
            }
        }

        // Clear the change list
        this.clearChanges = function () {
            this.translateChanges = [];
            $('.m_save_survey').find('.badge').html(this.translateChanges.length);
            if (this.translateChanges.length > 0) {
                $('.m_save_survey').removeClass('disabled').prop('disabled', false);
            } else {
                $('.m_save_survey').addClass('disabled').prop('disabled', true);
	            $('#m_auto_translate').closest('li').removeClass("disabled").prop("disabled", false);
            }
        }

        /*
         * If the label has been modified before then remove it from the change list
         */
        this.removeDuplicateTranslateChange = function () {
            // TODO
        }

        /*
         * Functions for managing settings
         */
        this.getSettings = function (save) {
            var current = this.createSettingsObject(
                $('#set_survey_name').val(),
                $('#set_instance_name').val(),
                $('#set_style').val(),
                $('#set_project_name option:selected').val(),
                $('#set_default_language option:selected').text(),
                $('#task_file').prop('checked'),
                $('#timing_data').prop('checked'),
	            $('#audit_location_data').prop('checked'),
	            $('#track_changes').prop('checked'),
	            $('#hide_on_device').prop('checked'),
	            $('#search_local_data').prop('checked'),
	            $('#data_survey').prop('checked'),
	            $('#oversight_survey').prop('checked'),
                $('#read_only_survey').prop('checked'),
                $('#my_reference_data').prop('checked'),
                $('#exclude_empty').prop('checked'),
                $('#compress_pdf').prop('checked'),
	            $('#default_logo').val()
            );

            // Update the model to reflect the current values
            if (save) {
                this.survey.displayName = current.displayName;
                this.survey.instanceNameDefn = current.instanceNameDefn;
                this.survey.surveyClass = current.surveyClass;
                this.survey.p_id = current.p_id;
                this.survey.def_lang = current.def_lang;
                this.survey.task_file = current.task_file;
                this.survey.timing_data = current.timing_data;
	            this.survey.audit_location_data = current.audit_location_data;
	            this.survey.track_changes = current.track_changes;
	            this.survey.hideOnDevice = current.hideOnDevice;
	            this.survey.searchLocalData = current.searchLocalData;
	            this.survey.dataSurvey = current.dataSurvey;
	            this.survey.oversightSurvey = current.oversightSurvey;
                this.survey.myReferenceData = current.myReferenceData;
                this.survey.readOnlySurvey = current.readOnlySurvey;
                this.survey.exclude_empty = current.exclude_empty;
                this.survey.compress_pdf = current.compress_pdf;
	            this.survey.default_logo = current.default_logo;
            }

            return current;
        }

        this.setSettings = function () {
            this.savedSettings = JSON.stringify(
                this.createSettingsObject(
                    this.survey.displayName,
                    this.survey.instanceNameDefn,
                    this.survey.surveyClass,
                    this.survey.p_id,
                    this.survey.def_lang,
                    this.survey.task_file,
                    this.survey.timing_data,
	                this.survey.audit_location_data,
	                this.survey.track_changes,
                    this.survey.hideOnDevice,
	                this.survey.searchLocalData,
	                this.survey.dataSurvey,
	                this.survey.oversightSurvey,
                    this.survey.myReferenceData,
                    this.survey.readOnlySurvey,
                    this.survey.exclude_empty,
                    this.survey.compress_pdf,
                    this.survey.hrk,
                    this.survey.key_policy,
	                this.survey.default_logo
                ));

            this.forceSettingsChange = false;
        }

        this.createSettingsObject = function (displayName, instanceNameDefn,
                                              surveyClass,
                                              p_id,
                                              def_lang,
                                              task_file,
                                              timing_data,
                                              audit_location_data,
                                              track_changes,
                                              hideOnDevice,
                                              searchLocalData,
                                              dataSurvey,
                                              oversightSurvey,
                                              readOnlySurvey,
                                              myReferenceData,
                                              exclude_empty,
                                              compress_pdf,
                                              default_logo) {

            var projId;
            if (typeof p_id === "string") {
                projId = parseInt(p_id);
            } else {
                projId = p_id;
            }
            return {
                displayName: displayName,
                instanceNameDefn: instanceNameDefn,
                surveyClass: surveyClass,
                p_id: projId,
                def_lang: def_lang,
                task_file: task_file,
                timing_data: timing_data,
	            audit_location_data: audit_location_data,
	            track_changes: track_changes,
                hideOnDevice: hideOnDevice,
                searchLocalData: searchLocalData,
	            dataSurvey: dataSurvey,
	            oversightSurvey: oversightSurvey,
                myReferenceData: myReferenceData,
                readOnlySurvey: readOnlySurvey,
                exclude_empty: exclude_empty,
                compress_pdf: compress_pdf,
	            default_logo: default_logo
            }
        }

        this.settingsChange = function () {
            var current = globals.model.getSettings(false);

            if (JSON.stringify(current) !== globals.model.savedSettings || globals.model.forceSettingsChange) {
                $('#save_settings').prop("disabled", false);
            } else {
                $('#save_settings').prop("disabled", true);
            }
        }

    }
});

;(function (global, factory) {
	typeof exports === 'object' && typeof module !== 'undefined' ? module.exports = factory() :
		typeof define === 'function' && define.amd ? define('moment',factory) :
			global.moment = factory()
}(this, (function () { 'use strict';

	var hookCallback;

	function hooks () {
		return hookCallback.apply(null, arguments);
	}

	// This is done to register the method called with moment()
	// without creating circular dependencies.
	function setHookCallback (callback) {
		hookCallback = callback;
	}

	function isArray(input) {
		return input instanceof Array || Object.prototype.toString.call(input) === '[object Array]';
	}

	function isObject(input) {
		// IE8 will treat undefined and null as object if it wasn't for
		// input != null
		return input != null && Object.prototype.toString.call(input) === '[object Object]';
	}

	function isObjectEmpty(obj) {
		if (Object.getOwnPropertyNames) {
			return (Object.getOwnPropertyNames(obj).length === 0);
		} else {
			var k;
			for (k in obj) {
				if (obj.hasOwnProperty(k)) {
					return false;
				}
			}
			return true;
		}
	}

	function isUndefined(input) {
		return input === void 0;
	}

	function isNumber(input) {
		return typeof input === 'number' || Object.prototype.toString.call(input) === '[object Number]';
	}

	function isDate(input) {
		return input instanceof Date || Object.prototype.toString.call(input) === '[object Date]';
	}

	function map(arr, fn) {
		var res = [], i;
		for (i = 0; i < arr.length; ++i) {
			res.push(fn(arr[i], i));
		}
		return res;
	}

	function hasOwnProp(a, b) {
		return Object.prototype.hasOwnProperty.call(a, b);
	}

	function extend(a, b) {
		for (var i in b) {
			if (hasOwnProp(b, i)) {
				a[i] = b[i];
			}
		}

		if (hasOwnProp(b, 'toString')) {
			a.toString = b.toString;
		}

		if (hasOwnProp(b, 'valueOf')) {
			a.valueOf = b.valueOf;
		}

		return a;
	}

	function createUTC (input, format, locale, strict) {
		return createLocalOrUTC(input, format, locale, strict, true).utc();
	}

	function defaultParsingFlags() {
		// We need to deep clone this object.
		return {
			empty           : false,
			unusedTokens    : [],
			unusedInput     : [],
			overflow        : -2,
			charsLeftOver   : 0,
			nullInput       : false,
			invalidMonth    : null,
			invalidFormat   : false,
			userInvalidated : false,
			iso             : false,
			parsedDateParts : [],
			meridiem        : null,
			rfc2822         : false,
			weekdayMismatch : false
		};
	}

	function getParsingFlags(m) {
		if (m._pf == null) {
			m._pf = defaultParsingFlags();
		}
		return m._pf;
	}

	var some;
	if (Array.prototype.some) {
		some = Array.prototype.some;
	} else {
		some = function (fun) {
			var t = Object(this);
			var len = t.length >>> 0;

			for (var i = 0; i < len; i++) {
				if (i in t && fun.call(this, t[i], i, t)) {
					return true;
				}
			}

			return false;
		};
	}

	function isValid(m) {
		if (m._isValid == null) {
			var flags = getParsingFlags(m);
			var parsedParts = some.call(flags.parsedDateParts, function (i) {
				return i != null;
			});
			var isNowValid = !isNaN(m._d.getTime()) &&
				flags.overflow < 0 &&
				!flags.empty &&
				!flags.invalidMonth &&
				!flags.invalidWeekday &&
				!flags.weekdayMismatch &&
				!flags.nullInput &&
				!flags.invalidFormat &&
				!flags.userInvalidated &&
				(!flags.meridiem || (flags.meridiem && parsedParts));

			if (m._strict) {
				isNowValid = isNowValid &&
					flags.charsLeftOver === 0 &&
					flags.unusedTokens.length === 0 &&
					flags.bigHour === undefined;
			}

			if (Object.isFrozen == null || !Object.isFrozen(m)) {
				m._isValid = isNowValid;
			}
			else {
				return isNowValid;
			}
		}
		return m._isValid;
	}

	function createInvalid (flags) {
		var m = createUTC(NaN);
		if (flags != null) {
			extend(getParsingFlags(m), flags);
		}
		else {
			getParsingFlags(m).userInvalidated = true;
		}

		return m;
	}

	// Plugins that add properties should also add the key here (null value),
	// so we can properly clone ourselves.
	var momentProperties = hooks.momentProperties = [];

	function copyConfig(to, from) {
		var i, prop, val;

		if (!isUndefined(from._isAMomentObject)) {
			to._isAMomentObject = from._isAMomentObject;
		}
		if (!isUndefined(from._i)) {
			to._i = from._i;
		}
		if (!isUndefined(from._f)) {
			to._f = from._f;
		}
		if (!isUndefined(from._l)) {
			to._l = from._l;
		}
		if (!isUndefined(from._strict)) {
			to._strict = from._strict;
		}
		if (!isUndefined(from._tzm)) {
			to._tzm = from._tzm;
		}
		if (!isUndefined(from._isUTC)) {
			to._isUTC = from._isUTC;
		}
		if (!isUndefined(from._offset)) {
			to._offset = from._offset;
		}
		if (!isUndefined(from._pf)) {
			to._pf = getParsingFlags(from);
		}
		if (!isUndefined(from._locale)) {
			to._locale = from._locale;
		}

		if (momentProperties.length > 0) {
			for (i = 0; i < momentProperties.length; i++) {
				prop = momentProperties[i];
				val = from[prop];
				if (!isUndefined(val)) {
					to[prop] = val;
				}
			}
		}

		return to;
	}

	var updateInProgress = false;

	// Moment prototype object
	function Moment(config) {
		copyConfig(this, config);
		this._d = new Date(config._d != null ? config._d.getTime() : NaN);
		if (!this.isValid()) {
			this._d = new Date(NaN);
		}
		// Prevent infinite loop in case updateOffset creates new moment
		// objects.
		if (updateInProgress === false) {
			updateInProgress = true;
			hooks.updateOffset(this);
			updateInProgress = false;
		}
	}

	function isMoment (obj) {
		return obj instanceof Moment || (obj != null && obj._isAMomentObject != null);
	}

	function absFloor (number) {
		if (number < 0) {
			// -0 -> 0
			return Math.ceil(number) || 0;
		} else {
			return Math.floor(number);
		}
	}

	function toInt(argumentForCoercion) {
		var coercedNumber = +argumentForCoercion,
			value = 0;

		if (coercedNumber !== 0 && isFinite(coercedNumber)) {
			value = absFloor(coercedNumber);
		}

		return value;
	}

	// compare two arrays, return the number of differences
	function compareArrays(array1, array2, dontConvert) {
		var len = Math.min(array1.length, array2.length),
			lengthDiff = Math.abs(array1.length - array2.length),
			diffs = 0,
			i;
		for (i = 0; i < len; i++) {
			if ((dontConvert && array1[i] !== array2[i]) ||
				(!dontConvert && toInt(array1[i]) !== toInt(array2[i]))) {
				diffs++;
			}
		}
		return diffs + lengthDiff;
	}

	function warn(msg) {
		if (hooks.suppressDeprecationWarnings === false &&
			(typeof console !==  'undefined') && console.warn) {
			console.warn('Deprecation warning: ' + msg);
		}
	}

	function deprecate(msg, fn) {
		var firstTime = true;

		return extend(function () {
			if (hooks.deprecationHandler != null) {
				hooks.deprecationHandler(null, msg);
			}
			if (firstTime) {
				var args = [];
				var arg;
				for (var i = 0; i < arguments.length; i++) {
					arg = '';
					if (typeof arguments[i] === 'object') {
						arg += '\n[' + i + '] ';
						for (var key in arguments[0]) {
							arg += key + ': ' + arguments[0][key] + ', ';
						}
						arg = arg.slice(0, -2); // Remove trailing comma and space
					} else {
						arg = arguments[i];
					}
					args.push(arg);
				}
				warn(msg + '\nArguments: ' + Array.prototype.slice.call(args).join('') + '\n' + (new Error()).stack);
				firstTime = false;
			}
			return fn.apply(this, arguments);
		}, fn);
	}

	var deprecations = {};

	function deprecateSimple(name, msg) {
		if (hooks.deprecationHandler != null) {
			hooks.deprecationHandler(name, msg);
		}
		if (!deprecations[name]) {
			warn(msg);
			deprecations[name] = true;
		}
	}

	hooks.suppressDeprecationWarnings = false;
	hooks.deprecationHandler = null;

	function isFunction(input) {
		return input instanceof Function || Object.prototype.toString.call(input) === '[object Function]';
	}

	function set (config) {
		var prop, i;
		for (i in config) {
			prop = config[i];
			if (isFunction(prop)) {
				this[i] = prop;
			} else {
				this['_' + i] = prop;
			}
		}
		this._config = config;
		// Lenient ordinal parsing accepts just a number in addition to
		// number + (possibly) stuff coming from _dayOfMonthOrdinalParse.
		// TODO: Remove "ordinalParse" fallback in next major release.
		this._dayOfMonthOrdinalParseLenient = new RegExp(
			(this._dayOfMonthOrdinalParse.source || this._ordinalParse.source) +
			'|' + (/\d{1,2}/).source);
	}

	function mergeConfigs(parentConfig, childConfig) {
		var res = extend({}, parentConfig), prop;
		for (prop in childConfig) {
			if (hasOwnProp(childConfig, prop)) {
				if (isObject(parentConfig[prop]) && isObject(childConfig[prop])) {
					res[prop] = {};
					extend(res[prop], parentConfig[prop]);
					extend(res[prop], childConfig[prop]);
				} else if (childConfig[prop] != null) {
					res[prop] = childConfig[prop];
				} else {
					delete res[prop];
				}
			}
		}
		for (prop in parentConfig) {
			if (hasOwnProp(parentConfig, prop) &&
				!hasOwnProp(childConfig, prop) &&
				isObject(parentConfig[prop])) {
				// make sure changes to properties don't modify parent config
				res[prop] = extend({}, res[prop]);
			}
		}
		return res;
	}

	function Locale(config) {
		if (config != null) {
			this.set(config);
		}
	}

	var keys;

	if (Object.keys) {
		keys = Object.keys;
	} else {
		keys = function (obj) {
			var i, res = [];
			for (i in obj) {
				if (hasOwnProp(obj, i)) {
					res.push(i);
				}
			}
			return res;
		};
	}

	var defaultCalendar = {
		sameDay : '[Today at] LT',
		nextDay : '[Tomorrow at] LT',
		nextWeek : 'dddd [at] LT',
		lastDay : '[Yesterday at] LT',
		lastWeek : '[Last] dddd [at] LT',
		sameElse : 'L'
	};

	function calendar (key, mom, now) {
		var output = this._calendar[key] || this._calendar['sameElse'];
		return isFunction(output) ? output.call(mom, now) : output;
	}

	var defaultLongDateFormat = {
		LTS  : 'h:mm:ss A',
		LT   : 'h:mm A',
		L    : 'MM/DD/YYYY',
		LL   : 'MMMM D, YYYY',
		LLL  : 'MMMM D, YYYY h:mm A',
		LLLL : 'dddd, MMMM D, YYYY h:mm A'
	};

	function longDateFormat (key) {
		var format = this._longDateFormat[key],
			formatUpper = this._longDateFormat[key.toUpperCase()];

		if (format || !formatUpper) {
			return format;
		}

		this._longDateFormat[key] = formatUpper.replace(/MMMM|MM|DD|dddd/g, function (val) {
			return val.slice(1);
		});

		return this._longDateFormat[key];
	}

	var defaultInvalidDate = 'Invalid date';

	function invalidDate () {
		return this._invalidDate;
	}

	var defaultOrdinal = '%d';
	var defaultDayOfMonthOrdinalParse = /\d{1,2}/;

	function ordinal (number) {
		return this._ordinal.replace('%d', number);
	}

	var defaultRelativeTime = {
		future : 'in %s',
		past   : '%s ago',
		s  : 'a few seconds',
		ss : '%d seconds',
		m  : 'a minute',
		mm : '%d minutes',
		h  : 'an hour',
		hh : '%d hours',
		d  : 'a day',
		dd : '%d days',
		M  : 'a month',
		MM : '%d months',
		y  : 'a year',
		yy : '%d years'
	};

	function relativeTime (number, withoutSuffix, string, isFuture) {
		var output = this._relativeTime[string];
		return (isFunction(output)) ?
			output(number, withoutSuffix, string, isFuture) :
			output.replace(/%d/i, number);
	}

	function pastFuture (diff, output) {
		var format = this._relativeTime[diff > 0 ? 'future' : 'past'];
		return isFunction(format) ? format(output) : format.replace(/%s/i, output);
	}

	var aliases = {};

	function addUnitAlias (unit, shorthand) {
		var lowerCase = unit.toLowerCase();
		aliases[lowerCase] = aliases[lowerCase + 's'] = aliases[shorthand] = unit;
	}

	function normalizeUnits(units) {
		return typeof units === 'string' ? aliases[units] || aliases[units.toLowerCase()] : undefined;
	}

	function normalizeObjectUnits(inputObject) {
		var normalizedInput = {},
			normalizedProp,
			prop;

		for (prop in inputObject) {
			if (hasOwnProp(inputObject, prop)) {
				normalizedProp = normalizeUnits(prop);
				if (normalizedProp) {
					normalizedInput[normalizedProp] = inputObject[prop];
				}
			}
		}

		return normalizedInput;
	}

	var priorities = {};

	function addUnitPriority(unit, priority) {
		priorities[unit] = priority;
	}

	function getPrioritizedUnits(unitsObj) {
		var units = [];
		for (var u in unitsObj) {
			units.push({unit: u, priority: priorities[u]});
		}
		units.sort(function (a, b) {
			return a.priority - b.priority;
		});
		return units;
	}

	function zeroFill(number, targetLength, forceSign) {
		var absNumber = '' + Math.abs(number),
			zerosToFill = targetLength - absNumber.length,
			sign = number >= 0;
		return (sign ? (forceSign ? '+' : '') : '-') +
			Math.pow(10, Math.max(0, zerosToFill)).toString().substr(1) + absNumber;
	}

	var formattingTokens = /(\[[^\[]*\])|(\\)?([Hh]mm(ss)?|Mo|MM?M?M?|Do|DDDo|DD?D?D?|ddd?d?|do?|w[o|w]?|W[o|W]?|Qo?|YYYYYY|YYYYY|YYYY|YY|gg(ggg?)?|GG(GGG?)?|e|E|a|A|hh?|HH?|kk?|mm?|ss?|S{1,9}|x|X|zz?|ZZ?|.)/g;

	var localFormattingTokens = /(\[[^\[]*\])|(\\)?(LTS|LT|LL?L?L?|l{1,4})/g;

	var formatFunctions = {};

	var formatTokenFunctions = {};

	// token:    'M'
	// padded:   ['MM', 2]
	// ordinal:  'Mo'
	// callback: function () { this.month() + 1 }
	function addFormatToken (token, padded, ordinal, callback) {
		var func = callback;
		if (typeof callback === 'string') {
			func = function () {
				return this[callback]();
			};
		}
		if (token) {
			formatTokenFunctions[token] = func;
		}
		if (padded) {
			formatTokenFunctions[padded[0]] = function () {
				return zeroFill(func.apply(this, arguments), padded[1], padded[2]);
			};
		}
		if (ordinal) {
			formatTokenFunctions[ordinal] = function () {
				return this.localeData().ordinal(func.apply(this, arguments), token);
			};
		}
	}

	function removeFormattingTokens(input) {
		if (input.match(/\[[\s\S]/)) {
			return input.replace(/^\[|\]$/g, '');
		}
		return input.replace(/\\/g, '');
	}

	function makeFormatFunction(format) {
		var array = format.match(formattingTokens), i, length;

		for (i = 0, length = array.length; i < length; i++) {
			if (formatTokenFunctions[array[i]]) {
				array[i] = formatTokenFunctions[array[i]];
			} else {
				array[i] = removeFormattingTokens(array[i]);
			}
		}

		return function (mom) {
			var output = '', i;
			for (i = 0; i < length; i++) {
				output += isFunction(array[i]) ? array[i].call(mom, format) : array[i];
			}
			return output;
		};
	}

	// format date using native date object
	function formatMoment(m, format) {
		if (!m.isValid()) {
			return m.localeData().invalidDate();
		}

		format = expandFormat(format, m.localeData());
		formatFunctions[format] = formatFunctions[format] || makeFormatFunction(format);

		return formatFunctions[format](m);
	}

	function expandFormat(format, locale) {
		var i = 5;

		function replaceLongDateFormatTokens(input) {
			return locale.longDateFormat(input) || input;
		}

		localFormattingTokens.lastIndex = 0;
		while (i >= 0 && localFormattingTokens.test(format)) {
			format = format.replace(localFormattingTokens, replaceLongDateFormatTokens);
			localFormattingTokens.lastIndex = 0;
			i -= 1;
		}

		return format;
	}

	var match1         = /\d/;            //       0 - 9
	var match2         = /\d\d/;          //      00 - 99
	var match3         = /\d{3}/;         //     000 - 999
	var match4         = /\d{4}/;         //    0000 - 9999
	var match6         = /[+-]?\d{6}/;    // -999999 - 999999
	var match1to2      = /\d\d?/;         //       0 - 99
	var match3to4      = /\d\d\d\d?/;     //     999 - 9999
	var match5to6      = /\d\d\d\d\d\d?/; //   99999 - 999999
	var match1to3      = /\d{1,3}/;       //       0 - 999
	var match1to4      = /\d{1,4}/;       //       0 - 9999
	var match1to6      = /[+-]?\d{1,6}/;  // -999999 - 999999

	var matchUnsigned  = /\d+/;           //       0 - inf
	var matchSigned    = /[+-]?\d+/;      //    -inf - inf

	var matchOffset    = /Z|[+-]\d\d:?\d\d/gi; // +00:00 -00:00 +0000 -0000 or Z
	var matchShortOffset = /Z|[+-]\d\d(?::?\d\d)?/gi; // +00 -00 +00:00 -00:00 +0000 -0000 or Z

	var matchTimestamp = /[+-]?\d+(\.\d{1,3})?/; // 123456789 123456789.123

	// any word (or two) characters or numbers including two/three word month in arabic.
	// includes scottish gaelic two word and hyphenated months
	var matchWord = /[0-9]{0,256}['a-z\u00A0-\u05FF\u0700-\uD7FF\uF900-\uFDCF\uFDF0-\uFF07\uFF10-\uFFEF]{1,256}|[\u0600-\u06FF\/]{1,256}(\s*?[\u0600-\u06FF]{1,256}){1,2}/i;

	var regexes = {};

	function addRegexToken (token, regex, strictRegex) {
		regexes[token] = isFunction(regex) ? regex : function (isStrict, localeData) {
			return (isStrict && strictRegex) ? strictRegex : regex;
		};
	}

	function getParseRegexForToken (token, config) {
		if (!hasOwnProp(regexes, token)) {
			return new RegExp(unescapeFormat(token));
		}

		return regexes[token](config._strict, config._locale);
	}

	// Code from http://stackoverflow.com/questions/3561493/is-there-a-regexp-escape-function-in-javascript
	function unescapeFormat(s) {
		return regexEscape(s.replace('\\', '').replace(/\\(\[)|\\(\])|\[([^\]\[]*)\]|\\(.)/g, function (matched, p1, p2, p3, p4) {
			return p1 || p2 || p3 || p4;
		}));
	}

	function regexEscape(s) {
		return s.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&');
	}

	var tokens = {};

	function addParseToken (token, callback) {
		var i, func = callback;
		if (typeof token === 'string') {
			token = [token];
		}
		if (isNumber(callback)) {
			func = function (input, array) {
				array[callback] = toInt(input);
			};
		}
		for (i = 0; i < token.length; i++) {
			tokens[token[i]] = func;
		}
	}

	function addWeekParseToken (token, callback) {
		addParseToken(token, function (input, array, config, token) {
			config._w = config._w || {};
			callback(input, config._w, config, token);
		});
	}

	function addTimeToArrayFromToken(token, input, config) {
		if (input != null && hasOwnProp(tokens, token)) {
			tokens[token](input, config._a, config, token);
		}
	}

	var YEAR = 0;
	var MONTH = 1;
	var DATE = 2;
	var HOUR = 3;
	var MINUTE = 4;
	var SECOND = 5;
	var MILLISECOND = 6;
	var WEEK = 7;
	var WEEKDAY = 8;

	// FORMATTING

	addFormatToken('Y', 0, 0, function () {
		var y = this.year();
		return y <= 9999 ? '' + y : '+' + y;
	});

	addFormatToken(0, ['YY', 2], 0, function () {
		return this.year() % 100;
	});

	addFormatToken(0, ['YYYY',   4],       0, 'year');
	addFormatToken(0, ['YYYYY',  5],       0, 'year');
	addFormatToken(0, ['YYYYYY', 6, true], 0, 'year');

	// ALIASES

	addUnitAlias('year', 'y');

	// PRIORITIES

	addUnitPriority('year', 1);

	// PARSING

	addRegexToken('Y',      matchSigned);
	addRegexToken('YY',     match1to2, match2);
	addRegexToken('YYYY',   match1to4, match4);
	addRegexToken('YYYYY',  match1to6, match6);
	addRegexToken('YYYYYY', match1to6, match6);

	addParseToken(['YYYYY', 'YYYYYY'], YEAR);
	addParseToken('YYYY', function (input, array) {
		array[YEAR] = input.length === 2 ? hooks.parseTwoDigitYear(input) : toInt(input);
	});
	addParseToken('YY', function (input, array) {
		array[YEAR] = hooks.parseTwoDigitYear(input);
	});
	addParseToken('Y', function (input, array) {
		array[YEAR] = parseInt(input, 10);
	});

	// HELPERS

	function daysInYear(year) {
		return isLeapYear(year) ? 366 : 365;
	}

	function isLeapYear(year) {
		return (year % 4 === 0 && year % 100 !== 0) || year % 400 === 0;
	}

	// HOOKS

	hooks.parseTwoDigitYear = function (input) {
		return toInt(input) + (toInt(input) > 68 ? 1900 : 2000);
	};

	// MOMENTS

	var getSetYear = makeGetSet('FullYear', true);

	function getIsLeapYear () {
		return isLeapYear(this.year());
	}

	function makeGetSet (unit, keepTime) {
		return function (value) {
			if (value != null) {
				set$1(this, unit, value);
				hooks.updateOffset(this, keepTime);
				return this;
			} else {
				return get(this, unit);
			}
		};
	}

	function get (mom, unit) {
		return mom.isValid() ?
			mom._d['get' + (mom._isUTC ? 'UTC' : '') + unit]() : NaN;
	}

	function set$1 (mom, unit, value) {
		if (mom.isValid() && !isNaN(value)) {
			if (unit === 'FullYear' && isLeapYear(mom.year()) && mom.month() === 1 && mom.date() === 29) {
				mom._d['set' + (mom._isUTC ? 'UTC' : '') + unit](value, mom.month(), daysInMonth(value, mom.month()));
			}
			else {
				mom._d['set' + (mom._isUTC ? 'UTC' : '') + unit](value);
			}
		}
	}

	// MOMENTS

	function stringGet (units) {
		units = normalizeUnits(units);
		if (isFunction(this[units])) {
			return this[units]();
		}
		return this;
	}


	function stringSet (units, value) {
		if (typeof units === 'object') {
			units = normalizeObjectUnits(units);
			var prioritized = getPrioritizedUnits(units);
			for (var i = 0; i < prioritized.length; i++) {
				this[prioritized[i].unit](units[prioritized[i].unit]);
			}
		} else {
			units = normalizeUnits(units);
			if (isFunction(this[units])) {
				return this[units](value);
			}
		}
		return this;
	}

	function mod(n, x) {
		return ((n % x) + x) % x;
	}

	var indexOf;

	if (Array.prototype.indexOf) {
		indexOf = Array.prototype.indexOf;
	} else {
		indexOf = function (o) {
			// I know
			var i;
			for (i = 0; i < this.length; ++i) {
				if (this[i] === o) {
					return i;
				}
			}
			return -1;
		};
	}

	function daysInMonth(year, month) {
		if (isNaN(year) || isNaN(month)) {
			return NaN;
		}
		var modMonth = mod(month, 12);
		year += (month - modMonth) / 12;
		return modMonth === 1 ? (isLeapYear(year) ? 29 : 28) : (31 - modMonth % 7 % 2);
	}

	// FORMATTING

	addFormatToken('M', ['MM', 2], 'Mo', function () {
		return this.month() + 1;
	});

	addFormatToken('MMM', 0, 0, function (format) {
		return this.localeData().monthsShort(this, format);
	});

	addFormatToken('MMMM', 0, 0, function (format) {
		return this.localeData().months(this, format);
	});

	// ALIASES

	addUnitAlias('month', 'M');

	// PRIORITY

	addUnitPriority('month', 8);

	// PARSING

	addRegexToken('M',    match1to2);
	addRegexToken('MM',   match1to2, match2);
	addRegexToken('MMM',  function (isStrict, locale) {
		return locale.monthsShortRegex(isStrict);
	});
	addRegexToken('MMMM', function (isStrict, locale) {
		return locale.monthsRegex(isStrict);
	});

	addParseToken(['M', 'MM'], function (input, array) {
		array[MONTH] = toInt(input) - 1;
	});

	addParseToken(['MMM', 'MMMM'], function (input, array, config, token) {
		var month = config._locale.monthsParse(input, token, config._strict);
		// if we didn't find a month name, mark the date as invalid.
		if (month != null) {
			array[MONTH] = month;
		} else {
			getParsingFlags(config).invalidMonth = input;
		}
	});

	// LOCALES

	var MONTHS_IN_FORMAT = /D[oD]?(\[[^\[\]]*\]|\s)+MMMM?/;
	var defaultLocaleMonths = 'January_February_March_April_May_June_July_August_September_October_November_December'.split('_');
	function localeMonths (m, format) {
		if (!m) {
			return isArray(this._months) ? this._months :
				this._months['standalone'];
		}
		return isArray(this._months) ? this._months[m.month()] :
			this._months[(this._months.isFormat || MONTHS_IN_FORMAT).test(format) ? 'format' : 'standalone'][m.month()];
	}

	var defaultLocaleMonthsShort = 'Jan_Feb_Mar_Apr_May_Jun_Jul_Aug_Sep_Oct_Nov_Dec'.split('_');
	function localeMonthsShort (m, format) {
		if (!m) {
			return isArray(this._monthsShort) ? this._monthsShort :
				this._monthsShort['standalone'];
		}
		return isArray(this._monthsShort) ? this._monthsShort[m.month()] :
			this._monthsShort[MONTHS_IN_FORMAT.test(format) ? 'format' : 'standalone'][m.month()];
	}

	function handleStrictParse(monthName, format, strict) {
		var i, ii, mom, llc = monthName.toLocaleLowerCase();
		if (!this._monthsParse) {
			// this is not used
			this._monthsParse = [];
			this._longMonthsParse = [];
			this._shortMonthsParse = [];
			for (i = 0; i < 12; ++i) {
				mom = createUTC([2000, i]);
				this._shortMonthsParse[i] = this.monthsShort(mom, '').toLocaleLowerCase();
				this._longMonthsParse[i] = this.months(mom, '').toLocaleLowerCase();
			}
		}

		if (strict) {
			if (format === 'MMM') {
				ii = indexOf.call(this._shortMonthsParse, llc);
				return ii !== -1 ? ii : null;
			} else {
				ii = indexOf.call(this._longMonthsParse, llc);
				return ii !== -1 ? ii : null;
			}
		} else {
			if (format === 'MMM') {
				ii = indexOf.call(this._shortMonthsParse, llc);
				if (ii !== -1) {
					return ii;
				}
				ii = indexOf.call(this._longMonthsParse, llc);
				return ii !== -1 ? ii : null;
			} else {
				ii = indexOf.call(this._longMonthsParse, llc);
				if (ii !== -1) {
					return ii;
				}
				ii = indexOf.call(this._shortMonthsParse, llc);
				return ii !== -1 ? ii : null;
			}
		}
	}

	function localeMonthsParse (monthName, format, strict) {
		var i, mom, regex;

		if (this._monthsParseExact) {
			return handleStrictParse.call(this, monthName, format, strict);
		}

		if (!this._monthsParse) {
			this._monthsParse = [];
			this._longMonthsParse = [];
			this._shortMonthsParse = [];
		}

		// TODO: add sorting
		// Sorting makes sure if one month (or abbr) is a prefix of another
		// see sorting in computeMonthsParse
		for (i = 0; i < 12; i++) {
			// make the regex if we don't have it already
			mom = createUTC([2000, i]);
			if (strict && !this._longMonthsParse[i]) {
				this._longMonthsParse[i] = new RegExp('^' + this.months(mom, '').replace('.', '') + '$', 'i');
				this._shortMonthsParse[i] = new RegExp('^' + this.monthsShort(mom, '').replace('.', '') + '$', 'i');
			}
			if (!strict && !this._monthsParse[i]) {
				regex = '^' + this.months(mom, '') + '|^' + this.monthsShort(mom, '');
				this._monthsParse[i] = new RegExp(regex.replace('.', ''), 'i');
			}
			// test the regex
			if (strict && format === 'MMMM' && this._longMonthsParse[i].test(monthName)) {
				return i;
			} else if (strict && format === 'MMM' && this._shortMonthsParse[i].test(monthName)) {
				return i;
			} else if (!strict && this._monthsParse[i].test(monthName)) {
				return i;
			}
		}
	}

	// MOMENTS

	function setMonth (mom, value) {
		var dayOfMonth;

		if (!mom.isValid()) {
			// No op
			return mom;
		}

		if (typeof value === 'string') {
			if (/^\d+$/.test(value)) {
				value = toInt(value);
			} else {
				value = mom.localeData().monthsParse(value);
				// TODO: Another silent failure?
				if (!isNumber(value)) {
					return mom;
				}
			}
		}

		dayOfMonth = Math.min(mom.date(), daysInMonth(mom.year(), value));
		mom._d['set' + (mom._isUTC ? 'UTC' : '') + 'Month'](value, dayOfMonth);
		return mom;
	}

	function getSetMonth (value) {
		if (value != null) {
			setMonth(this, value);
			hooks.updateOffset(this, true);
			return this;
		} else {
			return get(this, 'Month');
		}
	}

	function getDaysInMonth () {
		return daysInMonth(this.year(), this.month());
	}

	var defaultMonthsShortRegex = matchWord;
	function monthsShortRegex (isStrict) {
		if (this._monthsParseExact) {
			if (!hasOwnProp(this, '_monthsRegex')) {
				computeMonthsParse.call(this);
			}
			if (isStrict) {
				return this._monthsShortStrictRegex;
			} else {
				return this._monthsShortRegex;
			}
		} else {
			if (!hasOwnProp(this, '_monthsShortRegex')) {
				this._monthsShortRegex = defaultMonthsShortRegex;
			}
			return this._monthsShortStrictRegex && isStrict ?
				this._monthsShortStrictRegex : this._monthsShortRegex;
		}
	}

	var defaultMonthsRegex = matchWord;
	function monthsRegex (isStrict) {
		if (this._monthsParseExact) {
			if (!hasOwnProp(this, '_monthsRegex')) {
				computeMonthsParse.call(this);
			}
			if (isStrict) {
				return this._monthsStrictRegex;
			} else {
				return this._monthsRegex;
			}
		} else {
			if (!hasOwnProp(this, '_monthsRegex')) {
				this._monthsRegex = defaultMonthsRegex;
			}
			return this._monthsStrictRegex && isStrict ?
				this._monthsStrictRegex : this._monthsRegex;
		}
	}

	function computeMonthsParse () {
		function cmpLenRev(a, b) {
			return b.length - a.length;
		}

		var shortPieces = [], longPieces = [], mixedPieces = [],
			i, mom;
		for (i = 0; i < 12; i++) {
			// make the regex if we don't have it already
			mom = createUTC([2000, i]);
			shortPieces.push(this.monthsShort(mom, ''));
			longPieces.push(this.months(mom, ''));
			mixedPieces.push(this.months(mom, ''));
			mixedPieces.push(this.monthsShort(mom, ''));
		}
		// Sorting makes sure if one month (or abbr) is a prefix of another it
		// will match the longer piece.
		shortPieces.sort(cmpLenRev);
		longPieces.sort(cmpLenRev);
		mixedPieces.sort(cmpLenRev);
		for (i = 0; i < 12; i++) {
			shortPieces[i] = regexEscape(shortPieces[i]);
			longPieces[i] = regexEscape(longPieces[i]);
		}
		for (i = 0; i < 24; i++) {
			mixedPieces[i] = regexEscape(mixedPieces[i]);
		}

		this._monthsRegex = new RegExp('^(' + mixedPieces.join('|') + ')', 'i');
		this._monthsShortRegex = this._monthsRegex;
		this._monthsStrictRegex = new RegExp('^(' + longPieces.join('|') + ')', 'i');
		this._monthsShortStrictRegex = new RegExp('^(' + shortPieces.join('|') + ')', 'i');
	}

	function createDate (y, m, d, h, M, s, ms) {
		// can't just apply() to create a date:
		// https://stackoverflow.com/q/181348
		var date;
		// the date constructor remaps years 0-99 to 1900-1999
		if (y < 100 && y >= 0) {
			// preserve leap years using a full 400 year cycle, then reset
			date = new Date(y + 400, m, d, h, M, s, ms);
			if (isFinite(date.getFullYear())) {
				date.setFullYear(y);
			}
		} else {
			date = new Date(y, m, d, h, M, s, ms);
		}

		return date;
	}

	function createUTCDate (y) {
		var date;
		// the Date.UTC function remaps years 0-99 to 1900-1999
		if (y < 100 && y >= 0) {
			var args = Array.prototype.slice.call(arguments);
			// preserve leap years using a full 400 year cycle, then reset
			args[0] = y + 400;
			date = new Date(Date.UTC.apply(null, args));
			if (isFinite(date.getUTCFullYear())) {
				date.setUTCFullYear(y);
			}
		} else {
			date = new Date(Date.UTC.apply(null, arguments));
		}

		return date;
	}

	// start-of-first-week - start-of-year
	function firstWeekOffset(year, dow, doy) {
		var // first-week day -- which january is always in the first week (4 for iso, 1 for other)
			fwd = 7 + dow - doy,
			// first-week day local weekday -- which local weekday is fwd
			fwdlw = (7 + createUTCDate(year, 0, fwd).getUTCDay() - dow) % 7;

		return -fwdlw + fwd - 1;
	}

	// https://en.wikipedia.org/wiki/ISO_week_date#Calculating_a_date_given_the_year.2C_week_number_and_weekday
	function dayOfYearFromWeeks(year, week, weekday, dow, doy) {
		var localWeekday = (7 + weekday - dow) % 7,
			weekOffset = firstWeekOffset(year, dow, doy),
			dayOfYear = 1 + 7 * (week - 1) + localWeekday + weekOffset,
			resYear, resDayOfYear;

		if (dayOfYear <= 0) {
			resYear = year - 1;
			resDayOfYear = daysInYear(resYear) + dayOfYear;
		} else if (dayOfYear > daysInYear(year)) {
			resYear = year + 1;
			resDayOfYear = dayOfYear - daysInYear(year);
		} else {
			resYear = year;
			resDayOfYear = dayOfYear;
		}

		return {
			year: resYear,
			dayOfYear: resDayOfYear
		};
	}

	function weekOfYear(mom, dow, doy) {
		var weekOffset = firstWeekOffset(mom.year(), dow, doy),
			week = Math.floor((mom.dayOfYear() - weekOffset - 1) / 7) + 1,
			resWeek, resYear;

		if (week < 1) {
			resYear = mom.year() - 1;
			resWeek = week + weeksInYear(resYear, dow, doy);
		} else if (week > weeksInYear(mom.year(), dow, doy)) {
			resWeek = week - weeksInYear(mom.year(), dow, doy);
			resYear = mom.year() + 1;
		} else {
			resYear = mom.year();
			resWeek = week;
		}

		return {
			week: resWeek,
			year: resYear
		};
	}

	function weeksInYear(year, dow, doy) {
		var weekOffset = firstWeekOffset(year, dow, doy),
			weekOffsetNext = firstWeekOffset(year + 1, dow, doy);
		return (daysInYear(year) - weekOffset + weekOffsetNext) / 7;
	}

	// FORMATTING

	addFormatToken('w', ['ww', 2], 'wo', 'week');
	addFormatToken('W', ['WW', 2], 'Wo', 'isoWeek');

	// ALIASES

	addUnitAlias('week', 'w');
	addUnitAlias('isoWeek', 'W');

	// PRIORITIES

	addUnitPriority('week', 5);
	addUnitPriority('isoWeek', 5);

	// PARSING

	addRegexToken('w',  match1to2);
	addRegexToken('ww', match1to2, match2);
	addRegexToken('W',  match1to2);
	addRegexToken('WW', match1to2, match2);

	addWeekParseToken(['w', 'ww', 'W', 'WW'], function (input, week, config, token) {
		week[token.substr(0, 1)] = toInt(input);
	});

	// HELPERS

	// LOCALES

	function localeWeek (mom) {
		return weekOfYear(mom, this._week.dow, this._week.doy).week;
	}

	var defaultLocaleWeek = {
		dow : 0, // Sunday is the first day of the week.
		doy : 6  // The week that contains Jan 6th is the first week of the year.
	};

	function localeFirstDayOfWeek () {
		return this._week.dow;
	}

	function localeFirstDayOfYear () {
		return this._week.doy;
	}

	// MOMENTS

	function getSetWeek (input) {
		var week = this.localeData().week(this);
		return input == null ? week : this.add((input - week) * 7, 'd');
	}

	function getSetISOWeek (input) {
		var week = weekOfYear(this, 1, 4).week;
		return input == null ? week : this.add((input - week) * 7, 'd');
	}

	// FORMATTING

	addFormatToken('d', 0, 'do', 'day');

	addFormatToken('dd', 0, 0, function (format) {
		return this.localeData().weekdaysMin(this, format);
	});

	addFormatToken('ddd', 0, 0, function (format) {
		return this.localeData().weekdaysShort(this, format);
	});

	addFormatToken('dddd', 0, 0, function (format) {
		return this.localeData().weekdays(this, format);
	});

	addFormatToken('e', 0, 0, 'weekday');
	addFormatToken('E', 0, 0, 'isoWeekday');

	// ALIASES

	addUnitAlias('day', 'd');
	addUnitAlias('weekday', 'e');
	addUnitAlias('isoWeekday', 'E');

	// PRIORITY
	addUnitPriority('day', 11);
	addUnitPriority('weekday', 11);
	addUnitPriority('isoWeekday', 11);

	// PARSING

	addRegexToken('d',    match1to2);
	addRegexToken('e',    match1to2);
	addRegexToken('E',    match1to2);
	addRegexToken('dd',   function (isStrict, locale) {
		return locale.weekdaysMinRegex(isStrict);
	});
	addRegexToken('ddd',   function (isStrict, locale) {
		return locale.weekdaysShortRegex(isStrict);
	});
	addRegexToken('dddd',   function (isStrict, locale) {
		return locale.weekdaysRegex(isStrict);
	});

	addWeekParseToken(['dd', 'ddd', 'dddd'], function (input, week, config, token) {
		var weekday = config._locale.weekdaysParse(input, token, config._strict);
		// if we didn't get a weekday name, mark the date as invalid
		if (weekday != null) {
			week.d = weekday;
		} else {
			getParsingFlags(config).invalidWeekday = input;
		}
	});

	addWeekParseToken(['d', 'e', 'E'], function (input, week, config, token) {
		week[token] = toInt(input);
	});

	// HELPERS

	function parseWeekday(input, locale) {
		if (typeof input !== 'string') {
			return input;
		}

		if (!isNaN(input)) {
			return parseInt(input, 10);
		}

		input = locale.weekdaysParse(input);
		if (typeof input === 'number') {
			return input;
		}

		return null;
	}

	function parseIsoWeekday(input, locale) {
		if (typeof input === 'string') {
			return locale.weekdaysParse(input) % 7 || 7;
		}
		return isNaN(input) ? null : input;
	}

	// LOCALES
	function shiftWeekdays (ws, n) {
		return ws.slice(n, 7).concat(ws.slice(0, n));
	}

	var defaultLocaleWeekdays = 'Sunday_Monday_Tuesday_Wednesday_Thursday_Friday_Saturday'.split('_');
	function localeWeekdays (m, format) {
		var weekdays = isArray(this._weekdays) ? this._weekdays :
			this._weekdays[(m && m !== true && this._weekdays.isFormat.test(format)) ? 'format' : 'standalone'];
		return (m === true) ? shiftWeekdays(weekdays, this._week.dow)
			: (m) ? weekdays[m.day()] : weekdays;
	}

	var defaultLocaleWeekdaysShort = 'Sun_Mon_Tue_Wed_Thu_Fri_Sat'.split('_');
	function localeWeekdaysShort (m) {
		return (m === true) ? shiftWeekdays(this._weekdaysShort, this._week.dow)
			: (m) ? this._weekdaysShort[m.day()] : this._weekdaysShort;
	}

	var defaultLocaleWeekdaysMin = 'Su_Mo_Tu_We_Th_Fr_Sa'.split('_');
	function localeWeekdaysMin (m) {
		return (m === true) ? shiftWeekdays(this._weekdaysMin, this._week.dow)
			: (m) ? this._weekdaysMin[m.day()] : this._weekdaysMin;
	}

	function handleStrictParse$1(weekdayName, format, strict) {
		var i, ii, mom, llc = weekdayName.toLocaleLowerCase();
		if (!this._weekdaysParse) {
			this._weekdaysParse = [];
			this._shortWeekdaysParse = [];
			this._minWeekdaysParse = [];

			for (i = 0; i < 7; ++i) {
				mom = createUTC([2000, 1]).day(i);
				this._minWeekdaysParse[i] = this.weekdaysMin(mom, '').toLocaleLowerCase();
				this._shortWeekdaysParse[i] = this.weekdaysShort(mom, '').toLocaleLowerCase();
				this._weekdaysParse[i] = this.weekdays(mom, '').toLocaleLowerCase();
			}
		}

		if (strict) {
			if (format === 'dddd') {
				ii = indexOf.call(this._weekdaysParse, llc);
				return ii !== -1 ? ii : null;
			} else if (format === 'ddd') {
				ii = indexOf.call(this._shortWeekdaysParse, llc);
				return ii !== -1 ? ii : null;
			} else {
				ii = indexOf.call(this._minWeekdaysParse, llc);
				return ii !== -1 ? ii : null;
			}
		} else {
			if (format === 'dddd') {
				ii = indexOf.call(this._weekdaysParse, llc);
				if (ii !== -1) {
					return ii;
				}
				ii = indexOf.call(this._shortWeekdaysParse, llc);
				if (ii !== -1) {
					return ii;
				}
				ii = indexOf.call(this._minWeekdaysParse, llc);
				return ii !== -1 ? ii : null;
			} else if (format === 'ddd') {
				ii = indexOf.call(this._shortWeekdaysParse, llc);
				if (ii !== -1) {
					return ii;
				}
				ii = indexOf.call(this._weekdaysParse, llc);
				if (ii !== -1) {
					return ii;
				}
				ii = indexOf.call(this._minWeekdaysParse, llc);
				return ii !== -1 ? ii : null;
			} else {
				ii = indexOf.call(this._minWeekdaysParse, llc);
				if (ii !== -1) {
					return ii;
				}
				ii = indexOf.call(this._weekdaysParse, llc);
				if (ii !== -1) {
					return ii;
				}
				ii = indexOf.call(this._shortWeekdaysParse, llc);
				return ii !== -1 ? ii : null;
			}
		}
	}

	function localeWeekdaysParse (weekdayName, format, strict) {
		var i, mom, regex;

		if (this._weekdaysParseExact) {
			return handleStrictParse$1.call(this, weekdayName, format, strict);
		}

		if (!this._weekdaysParse) {
			this._weekdaysParse = [];
			this._minWeekdaysParse = [];
			this._shortWeekdaysParse = [];
			this._fullWeekdaysParse = [];
		}

		for (i = 0; i < 7; i++) {
			// make the regex if we don't have it already

			mom = createUTC([2000, 1]).day(i);
			if (strict && !this._fullWeekdaysParse[i]) {
				this._fullWeekdaysParse[i] = new RegExp('^' + this.weekdays(mom, '').replace('.', '\\.?') + '$', 'i');
				this._shortWeekdaysParse[i] = new RegExp('^' + this.weekdaysShort(mom, '').replace('.', '\\.?') + '$', 'i');
				this._minWeekdaysParse[i] = new RegExp('^' + this.weekdaysMin(mom, '').replace('.', '\\.?') + '$', 'i');
			}
			if (!this._weekdaysParse[i]) {
				regex = '^' + this.weekdays(mom, '') + '|^' + this.weekdaysShort(mom, '') + '|^' + this.weekdaysMin(mom, '');
				this._weekdaysParse[i] = new RegExp(regex.replace('.', ''), 'i');
			}
			// test the regex
			if (strict && format === 'dddd' && this._fullWeekdaysParse[i].test(weekdayName)) {
				return i;
			} else if (strict && format === 'ddd' && this._shortWeekdaysParse[i].test(weekdayName)) {
				return i;
			} else if (strict && format === 'dd' && this._minWeekdaysParse[i].test(weekdayName)) {
				return i;
			} else if (!strict && this._weekdaysParse[i].test(weekdayName)) {
				return i;
			}
		}
	}

	// MOMENTS

	function getSetDayOfWeek (input) {
		if (!this.isValid()) {
			return input != null ? this : NaN;
		}
		var day = this._isUTC ? this._d.getUTCDay() : this._d.getDay();
		if (input != null) {
			input = parseWeekday(input, this.localeData());
			return this.add(input - day, 'd');
		} else {
			return day;
		}
	}

	function getSetLocaleDayOfWeek (input) {
		if (!this.isValid()) {
			return input != null ? this : NaN;
		}
		var weekday = (this.day() + 7 - this.localeData()._week.dow) % 7;
		return input == null ? weekday : this.add(input - weekday, 'd');
	}

	function getSetISODayOfWeek (input) {
		if (!this.isValid()) {
			return input != null ? this : NaN;
		}

		// behaves the same as moment#day except
		// as a getter, returns 7 instead of 0 (1-7 range instead of 0-6)
		// as a setter, sunday should belong to the previous week.

		if (input != null) {
			var weekday = parseIsoWeekday(input, this.localeData());
			return this.day(this.day() % 7 ? weekday : weekday - 7);
		} else {
			return this.day() || 7;
		}
	}

	var defaultWeekdaysRegex = matchWord;
	function weekdaysRegex (isStrict) {
		if (this._weekdaysParseExact) {
			if (!hasOwnProp(this, '_weekdaysRegex')) {
				computeWeekdaysParse.call(this);
			}
			if (isStrict) {
				return this._weekdaysStrictRegex;
			} else {
				return this._weekdaysRegex;
			}
		} else {
			if (!hasOwnProp(this, '_weekdaysRegex')) {
				this._weekdaysRegex = defaultWeekdaysRegex;
			}
			return this._weekdaysStrictRegex && isStrict ?
				this._weekdaysStrictRegex : this._weekdaysRegex;
		}
	}

	var defaultWeekdaysShortRegex = matchWord;
	function weekdaysShortRegex (isStrict) {
		if (this._weekdaysParseExact) {
			if (!hasOwnProp(this, '_weekdaysRegex')) {
				computeWeekdaysParse.call(this);
			}
			if (isStrict) {
				return this._weekdaysShortStrictRegex;
			} else {
				return this._weekdaysShortRegex;
			}
		} else {
			if (!hasOwnProp(this, '_weekdaysShortRegex')) {
				this._weekdaysShortRegex = defaultWeekdaysShortRegex;
			}
			return this._weekdaysShortStrictRegex && isStrict ?
				this._weekdaysShortStrictRegex : this._weekdaysShortRegex;
		}
	}

	var defaultWeekdaysMinRegex = matchWord;
	function weekdaysMinRegex (isStrict) {
		if (this._weekdaysParseExact) {
			if (!hasOwnProp(this, '_weekdaysRegex')) {
				computeWeekdaysParse.call(this);
			}
			if (isStrict) {
				return this._weekdaysMinStrictRegex;
			} else {
				return this._weekdaysMinRegex;
			}
		} else {
			if (!hasOwnProp(this, '_weekdaysMinRegex')) {
				this._weekdaysMinRegex = defaultWeekdaysMinRegex;
			}
			return this._weekdaysMinStrictRegex && isStrict ?
				this._weekdaysMinStrictRegex : this._weekdaysMinRegex;
		}
	}


	function computeWeekdaysParse () {
		function cmpLenRev(a, b) {
			return b.length - a.length;
		}

		var minPieces = [], shortPieces = [], longPieces = [], mixedPieces = [],
			i, mom, minp, shortp, longp;
		for (i = 0; i < 7; i++) {
			// make the regex if we don't have it already
			mom = createUTC([2000, 1]).day(i);
			minp = this.weekdaysMin(mom, '');
			shortp = this.weekdaysShort(mom, '');
			longp = this.weekdays(mom, '');
			minPieces.push(minp);
			shortPieces.push(shortp);
			longPieces.push(longp);
			mixedPieces.push(minp);
			mixedPieces.push(shortp);
			mixedPieces.push(longp);
		}
		// Sorting makes sure if one weekday (or abbr) is a prefix of another it
		// will match the longer piece.
		minPieces.sort(cmpLenRev);
		shortPieces.sort(cmpLenRev);
		longPieces.sort(cmpLenRev);
		mixedPieces.sort(cmpLenRev);
		for (i = 0; i < 7; i++) {
			shortPieces[i] = regexEscape(shortPieces[i]);
			longPieces[i] = regexEscape(longPieces[i]);
			mixedPieces[i] = regexEscape(mixedPieces[i]);
		}

		this._weekdaysRegex = new RegExp('^(' + mixedPieces.join('|') + ')', 'i');
		this._weekdaysShortRegex = this._weekdaysRegex;
		this._weekdaysMinRegex = this._weekdaysRegex;

		this._weekdaysStrictRegex = new RegExp('^(' + longPieces.join('|') + ')', 'i');
		this._weekdaysShortStrictRegex = new RegExp('^(' + shortPieces.join('|') + ')', 'i');
		this._weekdaysMinStrictRegex = new RegExp('^(' + minPieces.join('|') + ')', 'i');
	}

	// FORMATTING

	function hFormat() {
		return this.hours() % 12 || 12;
	}

	function kFormat() {
		return this.hours() || 24;
	}

	addFormatToken('H', ['HH', 2], 0, 'hour');
	addFormatToken('h', ['hh', 2], 0, hFormat);
	addFormatToken('k', ['kk', 2], 0, kFormat);

	addFormatToken('hmm', 0, 0, function () {
		return '' + hFormat.apply(this) + zeroFill(this.minutes(), 2);
	});

	addFormatToken('hmmss', 0, 0, function () {
		return '' + hFormat.apply(this) + zeroFill(this.minutes(), 2) +
			zeroFill(this.seconds(), 2);
	});

	addFormatToken('Hmm', 0, 0, function () {
		return '' + this.hours() + zeroFill(this.minutes(), 2);
	});

	addFormatToken('Hmmss', 0, 0, function () {
		return '' + this.hours() + zeroFill(this.minutes(), 2) +
			zeroFill(this.seconds(), 2);
	});

	function meridiem (token, lowercase) {
		addFormatToken(token, 0, 0, function () {
			return this.localeData().meridiem(this.hours(), this.minutes(), lowercase);
		});
	}

	meridiem('a', true);
	meridiem('A', false);

	// ALIASES

	addUnitAlias('hour', 'h');

	// PRIORITY
	addUnitPriority('hour', 13);

	// PARSING

	function matchMeridiem (isStrict, locale) {
		return locale._meridiemParse;
	}

	addRegexToken('a',  matchMeridiem);
	addRegexToken('A',  matchMeridiem);
	addRegexToken('H',  match1to2);
	addRegexToken('h',  match1to2);
	addRegexToken('k',  match1to2);
	addRegexToken('HH', match1to2, match2);
	addRegexToken('hh', match1to2, match2);
	addRegexToken('kk', match1to2, match2);

	addRegexToken('hmm', match3to4);
	addRegexToken('hmmss', match5to6);
	addRegexToken('Hmm', match3to4);
	addRegexToken('Hmmss', match5to6);

	addParseToken(['H', 'HH'], HOUR);
	addParseToken(['k', 'kk'], function (input, array, config) {
		var kInput = toInt(input);
		array[HOUR] = kInput === 24 ? 0 : kInput;
	});
	addParseToken(['a', 'A'], function (input, array, config) {
		config._isPm = config._locale.isPM(input);
		config._meridiem = input;
	});
	addParseToken(['h', 'hh'], function (input, array, config) {
		array[HOUR] = toInt(input);
		getParsingFlags(config).bigHour = true;
	});
	addParseToken('hmm', function (input, array, config) {
		var pos = input.length - 2;
		array[HOUR] = toInt(input.substr(0, pos));
		array[MINUTE] = toInt(input.substr(pos));
		getParsingFlags(config).bigHour = true;
	});
	addParseToken('hmmss', function (input, array, config) {
		var pos1 = input.length - 4;
		var pos2 = input.length - 2;
		array[HOUR] = toInt(input.substr(0, pos1));
		array[MINUTE] = toInt(input.substr(pos1, 2));
		array[SECOND] = toInt(input.substr(pos2));
		getParsingFlags(config).bigHour = true;
	});
	addParseToken('Hmm', function (input, array, config) {
		var pos = input.length - 2;
		array[HOUR] = toInt(input.substr(0, pos));
		array[MINUTE] = toInt(input.substr(pos));
	});
	addParseToken('Hmmss', function (input, array, config) {
		var pos1 = input.length - 4;
		var pos2 = input.length - 2;
		array[HOUR] = toInt(input.substr(0, pos1));
		array[MINUTE] = toInt(input.substr(pos1, 2));
		array[SECOND] = toInt(input.substr(pos2));
	});

	// LOCALES

	function localeIsPM (input) {
		// IE8 Quirks Mode & IE7 Standards Mode do not allow accessing strings like arrays
		// Using charAt should be more compatible.
		return ((input + '').toLowerCase().charAt(0) === 'p');
	}

	var defaultLocaleMeridiemParse = /[ap]\.?m?\.?/i;
	function localeMeridiem (hours, minutes, isLower) {
		if (hours > 11) {
			return isLower ? 'pm' : 'PM';
		} else {
			return isLower ? 'am' : 'AM';
		}
	}


	// MOMENTS

	// Setting the hour should keep the time, because the user explicitly
	// specified which hour they want. So trying to maintain the same hour (in
	// a new timezone) makes sense. Adding/subtracting hours does not follow
	// this rule.
	var getSetHour = makeGetSet('Hours', true);

	var baseConfig = {
		calendar: defaultCalendar,
		longDateFormat: defaultLongDateFormat,
		invalidDate: defaultInvalidDate,
		ordinal: defaultOrdinal,
		dayOfMonthOrdinalParse: defaultDayOfMonthOrdinalParse,
		relativeTime: defaultRelativeTime,

		months: defaultLocaleMonths,
		monthsShort: defaultLocaleMonthsShort,

		week: defaultLocaleWeek,

		weekdays: defaultLocaleWeekdays,
		weekdaysMin: defaultLocaleWeekdaysMin,
		weekdaysShort: defaultLocaleWeekdaysShort,

		meridiemParse: defaultLocaleMeridiemParse
	};

	// internal storage for locale config files
	var locales = {};
	var localeFamilies = {};
	var globalLocale;

	function normalizeLocale(key) {
		return key ? key.toLowerCase().replace('_', '-') : key;
	}

	// pick the locale from the array
	// try ['en-au', 'en-gb'] as 'en-au', 'en-gb', 'en', as in move through the list trying each
	// substring from most specific to least, but move to the next array item if it's a more specific variant than the current root
	function chooseLocale(names) {
		var i = 0, j, next, locale, split;

		while (i < names.length) {
			split = normalizeLocale(names[i]).split('-');
			j = split.length;
			next = normalizeLocale(names[i + 1]);
			next = next ? next.split('-') : null;
			while (j > 0) {
				locale = loadLocale(split.slice(0, j).join('-'));
				if (locale) {
					return locale;
				}
				if (next && next.length >= j && compareArrays(split, next, true) >= j - 1) {
					//the next array item is better than a shallower substring of this one
					break;
				}
				j--;
			}
			i++;
		}
		return globalLocale;
	}

	function loadLocale(name) {
		var oldLocale = null;
		// TODO: Find a better way to register and load all the locales in Node
		if (!locales[name] && (typeof module !== 'undefined') &&
			module && module.exports) {
			try {
				oldLocale = globalLocale._abbr;
				var aliasedRequire = require;
				aliasedRequire('./locale/' + name);
				getSetGlobalLocale(oldLocale);
			} catch (e) {}
		}
		return locales[name];
	}

	// This function will load locale and then set the global locale.  If
	// no arguments are passed in, it will simply return the current global
	// locale key.
	function getSetGlobalLocale (key, values) {
		var data;
		if (key) {
			if (isUndefined(values)) {
				data = getLocale(key);
			}
			else {
				data = defineLocale(key, values);
			}

			if (data) {
				// moment.duration._locale = moment._locale = data;
				globalLocale = data;
			}
			else {
				if ((typeof console !==  'undefined') && console.warn) {
					//warn user if arguments are passed but the locale could not be set
					console.warn('Locale ' + key +  ' not found. Did you forget to load it?');
				}
			}
		}

		return globalLocale._abbr;
	}

	function defineLocale (name, config) {
		if (config !== null) {
			var locale, parentConfig = baseConfig;
			config.abbr = name;
			if (locales[name] != null) {
				deprecateSimple('defineLocaleOverride',
					'use moment.updateLocale(localeName, config) to change ' +
					'an existing locale. moment.defineLocale(localeName, ' +
					'config) should only be used for creating a new locale ' +
					'See http://momentjs.com/guides/#/warnings/define-locale/ for more info.');
				parentConfig = locales[name]._config;
			} else if (config.parentLocale != null) {
				if (locales[config.parentLocale] != null) {
					parentConfig = locales[config.parentLocale]._config;
				} else {
					locale = loadLocale(config.parentLocale);
					if (locale != null) {
						parentConfig = locale._config;
					} else {
						if (!localeFamilies[config.parentLocale]) {
							localeFamilies[config.parentLocale] = [];
						}
						localeFamilies[config.parentLocale].push({
							name: name,
							config: config
						});
						return null;
					}
				}
			}
			locales[name] = new Locale(mergeConfigs(parentConfig, config));

			if (localeFamilies[name]) {
				localeFamilies[name].forEach(function (x) {
					defineLocale(x.name, x.config);
				});
			}

			// backwards compat for now: also set the locale
			// make sure we set the locale AFTER all child locales have been
			// created, so we won't end up with the child locale set.
			getSetGlobalLocale(name);


			return locales[name];
		} else {
			// useful for testing
			delete locales[name];
			return null;
		}
	}

	function updateLocale(name, config) {
		if (config != null) {
			var locale, tmpLocale, parentConfig = baseConfig;
			// MERGE
			tmpLocale = loadLocale(name);
			if (tmpLocale != null) {
				parentConfig = tmpLocale._config;
			}
			config = mergeConfigs(parentConfig, config);
			locale = new Locale(config);
			locale.parentLocale = locales[name];
			locales[name] = locale;

			// backwards compat for now: also set the locale
			getSetGlobalLocale(name);
		} else {
			// pass null for config to unupdate, useful for tests
			if (locales[name] != null) {
				if (locales[name].parentLocale != null) {
					locales[name] = locales[name].parentLocale;
				} else if (locales[name] != null) {
					delete locales[name];
				}
			}
		}
		return locales[name];
	}

	// returns locale data
	function getLocale (key) {
		var locale;

		if (key && key._locale && key._locale._abbr) {
			key = key._locale._abbr;
		}

		if (!key) {
			return globalLocale;
		}

		if (!isArray(key)) {
			//short-circuit everything else
			locale = loadLocale(key);
			if (locale) {
				return locale;
			}
			key = [key];
		}

		return chooseLocale(key);
	}

	function listLocales() {
		return keys(locales);
	}

	function checkOverflow (m) {
		var overflow;
		var a = m._a;

		if (a && getParsingFlags(m).overflow === -2) {
			overflow =
				a[MONTH]       < 0 || a[MONTH]       > 11  ? MONTH :
					a[DATE]        < 1 || a[DATE]        > daysInMonth(a[YEAR], a[MONTH]) ? DATE :
						a[HOUR]        < 0 || a[HOUR]        > 24 || (a[HOUR] === 24 && (a[MINUTE] !== 0 || a[SECOND] !== 0 || a[MILLISECOND] !== 0)) ? HOUR :
							a[MINUTE]      < 0 || a[MINUTE]      > 59  ? MINUTE :
								a[SECOND]      < 0 || a[SECOND]      > 59  ? SECOND :
									a[MILLISECOND] < 0 || a[MILLISECOND] > 999 ? MILLISECOND :
										-1;

			if (getParsingFlags(m)._overflowDayOfYear && (overflow < YEAR || overflow > DATE)) {
				overflow = DATE;
			}
			if (getParsingFlags(m)._overflowWeeks && overflow === -1) {
				overflow = WEEK;
			}
			if (getParsingFlags(m)._overflowWeekday && overflow === -1) {
				overflow = WEEKDAY;
			}

			getParsingFlags(m).overflow = overflow;
		}

		return m;
	}

	// Pick the first defined of two or three arguments.
	function defaults(a, b, c) {
		if (a != null) {
			return a;
		}
		if (b != null) {
			return b;
		}
		return c;
	}

	function currentDateArray(config) {
		// hooks is actually the exported moment object
		var nowValue = new Date(hooks.now());
		if (config._useUTC) {
			return [nowValue.getUTCFullYear(), nowValue.getUTCMonth(), nowValue.getUTCDate()];
		}
		return [nowValue.getFullYear(), nowValue.getMonth(), nowValue.getDate()];
	}

	// convert an array to a date.
	// the array should mirror the parameters below
	// note: all values past the year are optional and will default to the lowest possible value.
	// [year, month, day , hour, minute, second, millisecond]
	function configFromArray (config) {
		var i, date, input = [], currentDate, expectedWeekday, yearToUse;

		if (config._d) {
			return;
		}

		currentDate = currentDateArray(config);

		//compute day of the year from weeks and weekdays
		if (config._w && config._a[DATE] == null && config._a[MONTH] == null) {
			dayOfYearFromWeekInfo(config);
		}

		//if the day of the year is set, figure out what it is
		if (config._dayOfYear != null) {
			yearToUse = defaults(config._a[YEAR], currentDate[YEAR]);

			if (config._dayOfYear > daysInYear(yearToUse) || config._dayOfYear === 0) {
				getParsingFlags(config)._overflowDayOfYear = true;
			}

			date = createUTCDate(yearToUse, 0, config._dayOfYear);
			config._a[MONTH] = date.getUTCMonth();
			config._a[DATE] = date.getUTCDate();
		}

		// Default to current date.
		// * if no year, month, day of month are given, default to today
		// * if day of month is given, default month and year
		// * if month is given, default only year
		// * if year is given, don't default anything
		for (i = 0; i < 3 && config._a[i] == null; ++i) {
			config._a[i] = input[i] = currentDate[i];
		}

		// Zero out whatever was not defaulted, including time
		for (; i < 7; i++) {
			config._a[i] = input[i] = (config._a[i] == null) ? (i === 2 ? 1 : 0) : config._a[i];
		}

		// Check for 24:00:00.000
		if (config._a[HOUR] === 24 &&
			config._a[MINUTE] === 0 &&
			config._a[SECOND] === 0 &&
			config._a[MILLISECOND] === 0) {
			config._nextDay = true;
			config._a[HOUR] = 0;
		}

		config._d = (config._useUTC ? createUTCDate : createDate).apply(null, input);
		expectedWeekday = config._useUTC ? config._d.getUTCDay() : config._d.getDay();

		// Apply timezone offset from input. The actual utcOffset can be changed
		// with parseZone.
		if (config._tzm != null) {
			config._d.setUTCMinutes(config._d.getUTCMinutes() - config._tzm);
		}

		if (config._nextDay) {
			config._a[HOUR] = 24;
		}

		// check for mismatching day of week
		if (config._w && typeof config._w.d !== 'undefined' && config._w.d !== expectedWeekday) {
			getParsingFlags(config).weekdayMismatch = true;
		}
	}

	function dayOfYearFromWeekInfo(config) {
		var w, weekYear, week, weekday, dow, doy, temp, weekdayOverflow;

		w = config._w;
		if (w.GG != null || w.W != null || w.E != null) {
			dow = 1;
			doy = 4;

			// TODO: We need to take the current isoWeekYear, but that depends on
			// how we interpret now (local, utc, fixed offset). So create
			// a now version of current config (take local/utc/offset flags, and
			// create now).
			weekYear = defaults(w.GG, config._a[YEAR], weekOfYear(createLocal(), 1, 4).year);
			week = defaults(w.W, 1);
			weekday = defaults(w.E, 1);
			if (weekday < 1 || weekday > 7) {
				weekdayOverflow = true;
			}
		} else {
			dow = config._locale._week.dow;
			doy = config._locale._week.doy;

			var curWeek = weekOfYear(createLocal(), dow, doy);

			weekYear = defaults(w.gg, config._a[YEAR], curWeek.year);

			// Default to current week.
			week = defaults(w.w, curWeek.week);

			if (w.d != null) {
				// weekday -- low day numbers are considered next week
				weekday = w.d;
				if (weekday < 0 || weekday > 6) {
					weekdayOverflow = true;
				}
			} else if (w.e != null) {
				// local weekday -- counting starts from beginning of week
				weekday = w.e + dow;
				if (w.e < 0 || w.e > 6) {
					weekdayOverflow = true;
				}
			} else {
				// default to beginning of week
				weekday = dow;
			}
		}
		if (week < 1 || week > weeksInYear(weekYear, dow, doy)) {
			getParsingFlags(config)._overflowWeeks = true;
		} else if (weekdayOverflow != null) {
			getParsingFlags(config)._overflowWeekday = true;
		} else {
			temp = dayOfYearFromWeeks(weekYear, week, weekday, dow, doy);
			config._a[YEAR] = temp.year;
			config._dayOfYear = temp.dayOfYear;
		}
	}

	// iso 8601 regex
	// 0000-00-00 0000-W00 or 0000-W00-0 + T + 00 or 00:00 or 00:00:00 or 00:00:00.000 + +00:00 or +0000 or +00)
	var extendedIsoRegex = /^\s*((?:[+-]\d{6}|\d{4})-(?:\d\d-\d\d|W\d\d-\d|W\d\d|\d\d\d|\d\d))(?:(T| )(\d\d(?::\d\d(?::\d\d(?:[.,]\d+)?)?)?)([\+\-]\d\d(?::?\d\d)?|\s*Z)?)?$/;
	var basicIsoRegex = /^\s*((?:[+-]\d{6}|\d{4})(?:\d\d\d\d|W\d\d\d|W\d\d|\d\d\d|\d\d))(?:(T| )(\d\d(?:\d\d(?:\d\d(?:[.,]\d+)?)?)?)([\+\-]\d\d(?::?\d\d)?|\s*Z)?)?$/;

	var tzRegex = /Z|[+-]\d\d(?::?\d\d)?/;

	var isoDates = [
		['YYYYYY-MM-DD', /[+-]\d{6}-\d\d-\d\d/],
		['YYYY-MM-DD', /\d{4}-\d\d-\d\d/],
		['GGGG-[W]WW-E', /\d{4}-W\d\d-\d/],
		['GGGG-[W]WW', /\d{4}-W\d\d/, false],
		['YYYY-DDD', /\d{4}-\d{3}/],
		['YYYY-MM', /\d{4}-\d\d/, false],
		['YYYYYYMMDD', /[+-]\d{10}/],
		['YYYYMMDD', /\d{8}/],
		// YYYYMM is NOT allowed by the standard
		['GGGG[W]WWE', /\d{4}W\d{3}/],
		['GGGG[W]WW', /\d{4}W\d{2}/, false],
		['YYYYDDD', /\d{7}/]
	];

	// iso time formats and regexes
	var isoTimes = [
		['HH:mm:ss.SSSS', /\d\d:\d\d:\d\d\.\d+/],
		['HH:mm:ss,SSSS', /\d\d:\d\d:\d\d,\d+/],
		['HH:mm:ss', /\d\d:\d\d:\d\d/],
		['HH:mm', /\d\d:\d\d/],
		['HHmmss.SSSS', /\d\d\d\d\d\d\.\d+/],
		['HHmmss,SSSS', /\d\d\d\d\d\d,\d+/],
		['HHmmss', /\d\d\d\d\d\d/],
		['HHmm', /\d\d\d\d/],
		['HH', /\d\d/]
	];

	var aspNetJsonRegex = /^\/?Date\((\-?\d+)/i;

	// date from iso format
	function configFromISO(config) {
		var i, l,
			string = config._i,
			match = extendedIsoRegex.exec(string) || basicIsoRegex.exec(string),
			allowTime, dateFormat, timeFormat, tzFormat;

		if (match) {
			getParsingFlags(config).iso = true;

			for (i = 0, l = isoDates.length; i < l; i++) {
				if (isoDates[i][1].exec(match[1])) {
					dateFormat = isoDates[i][0];
					allowTime = isoDates[i][2] !== false;
					break;
				}
			}
			if (dateFormat == null) {
				config._isValid = false;
				return;
			}
			if (match[3]) {
				for (i = 0, l = isoTimes.length; i < l; i++) {
					if (isoTimes[i][1].exec(match[3])) {
						// match[2] should be 'T' or space
						timeFormat = (match[2] || ' ') + isoTimes[i][0];
						break;
					}
				}
				if (timeFormat == null) {
					config._isValid = false;
					return;
				}
			}
			if (!allowTime && timeFormat != null) {
				config._isValid = false;
				return;
			}
			if (match[4]) {
				if (tzRegex.exec(match[4])) {
					tzFormat = 'Z';
				} else {
					config._isValid = false;
					return;
				}
			}
			config._f = dateFormat + (timeFormat || '') + (tzFormat || '');
			configFromStringAndFormat(config);
		} else {
			config._isValid = false;
		}
	}

	// RFC 2822 regex: For details see https://tools.ietf.org/html/rfc2822#section-3.3
	var rfc2822 = /^(?:(Mon|Tue|Wed|Thu|Fri|Sat|Sun),?\s)?(\d{1,2})\s(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)\s(\d{2,4})\s(\d\d):(\d\d)(?::(\d\d))?\s(?:(UT|GMT|[ECMP][SD]T)|([Zz])|([+-]\d{4}))$/;

	function extractFromRFC2822Strings(yearStr, monthStr, dayStr, hourStr, minuteStr, secondStr) {
		var result = [
			untruncateYear(yearStr),
			defaultLocaleMonthsShort.indexOf(monthStr),
			parseInt(dayStr, 10),
			parseInt(hourStr, 10),
			parseInt(minuteStr, 10)
		];

		if (secondStr) {
			result.push(parseInt(secondStr, 10));
		}

		return result;
	}

	function untruncateYear(yearStr) {
		var year = parseInt(yearStr, 10);
		if (year <= 49) {
			return 2000 + year;
		} else if (year <= 999) {
			return 1900 + year;
		}
		return year;
	}

	function preprocessRFC2822(s) {
		// Remove comments and folding whitespace and replace multiple-spaces with a single space
		return s.replace(/\([^)]*\)|[\n\t]/g, ' ').replace(/(\s\s+)/g, ' ').replace(/^\s\s*/, '').replace(/\s\s*$/, '');
	}

	function checkWeekday(weekdayStr, parsedInput, config) {
		if (weekdayStr) {
			// TODO: Replace the vanilla JS Date object with an indepentent day-of-week check.
			var weekdayProvided = defaultLocaleWeekdaysShort.indexOf(weekdayStr),
				weekdayActual = new Date(parsedInput[0], parsedInput[1], parsedInput[2]).getDay();
			if (weekdayProvided !== weekdayActual) {
				getParsingFlags(config).weekdayMismatch = true;
				config._isValid = false;
				return false;
			}
		}
		return true;
	}

	var obsOffsets = {
		UT: 0,
		GMT: 0,
		EDT: -4 * 60,
		EST: -5 * 60,
		CDT: -5 * 60,
		CST: -6 * 60,
		MDT: -6 * 60,
		MST: -7 * 60,
		PDT: -7 * 60,
		PST: -8 * 60
	};

	function calculateOffset(obsOffset, militaryOffset, numOffset) {
		if (obsOffset) {
			return obsOffsets[obsOffset];
		} else if (militaryOffset) {
			// the only allowed military tz is Z
			return 0;
		} else {
			var hm = parseInt(numOffset, 10);
			var m = hm % 100, h = (hm - m) / 100;
			return h * 60 + m;
		}
	}

	// date and time from ref 2822 format
	function configFromRFC2822(config) {
		var match = rfc2822.exec(preprocessRFC2822(config._i));
		if (match) {
			var parsedArray = extractFromRFC2822Strings(match[4], match[3], match[2], match[5], match[6], match[7]);
			if (!checkWeekday(match[1], parsedArray, config)) {
				return;
			}

			config._a = parsedArray;
			config._tzm = calculateOffset(match[8], match[9], match[10]);

			config._d = createUTCDate.apply(null, config._a);
			config._d.setUTCMinutes(config._d.getUTCMinutes() - config._tzm);

			getParsingFlags(config).rfc2822 = true;
		} else {
			config._isValid = false;
		}
	}

	// date from iso format or fallback
	function configFromString(config) {
		var matched = aspNetJsonRegex.exec(config._i);

		if (matched !== null) {
			config._d = new Date(+matched[1]);
			return;
		}

		configFromISO(config);
		if (config._isValid === false) {
			delete config._isValid;
		} else {
			return;
		}

		configFromRFC2822(config);
		if (config._isValid === false) {
			delete config._isValid;
		} else {
			return;
		}

		// Final attempt, use Input Fallback
		hooks.createFromInputFallback(config);
	}

	hooks.createFromInputFallback = deprecate(
		'value provided is not in a recognized RFC2822 or ISO format. moment construction falls back to js Date(), ' +
		'which is not reliable across all browsers and versions. Non RFC2822/ISO date formats are ' +
		'discouraged and will be removed in an upcoming major release. Please refer to ' +
		'http://momentjs.com/guides/#/warnings/js-date/ for more info.',
		function (config) {
			config._d = new Date(config._i + (config._useUTC ? ' UTC' : ''));
		}
	);

	// constant that refers to the ISO standard
	hooks.ISO_8601 = function () {};

	// constant that refers to the RFC 2822 form
	hooks.RFC_2822 = function () {};

	// date from string and format string
	function configFromStringAndFormat(config) {
		// TODO: Move this to another part of the creation flow to prevent circular deps
		if (config._f === hooks.ISO_8601) {
			configFromISO(config);
			return;
		}
		if (config._f === hooks.RFC_2822) {
			configFromRFC2822(config);
			return;
		}
		config._a = [];
		getParsingFlags(config).empty = true;

		// This array is used to make a Date, either with `new Date` or `Date.UTC`
		var string = '' + config._i,
			i, parsedInput, tokens, token, skipped,
			stringLength = string.length,
			totalParsedInputLength = 0;

		tokens = expandFormat(config._f, config._locale).match(formattingTokens) || [];

		for (i = 0; i < tokens.length; i++) {
			token = tokens[i];
			parsedInput = (string.match(getParseRegexForToken(token, config)) || [])[0];
			// console.log('token', token, 'parsedInput', parsedInput,
			//         'regex', getParseRegexForToken(token, config));
			if (parsedInput) {
				skipped = string.substr(0, string.indexOf(parsedInput));
				if (skipped.length > 0) {
					getParsingFlags(config).unusedInput.push(skipped);
				}
				string = string.slice(string.indexOf(parsedInput) + parsedInput.length);
				totalParsedInputLength += parsedInput.length;
			}
			// don't parse if it's not a known token
			if (formatTokenFunctions[token]) {
				if (parsedInput) {
					getParsingFlags(config).empty = false;
				}
				else {
					getParsingFlags(config).unusedTokens.push(token);
				}
				addTimeToArrayFromToken(token, parsedInput, config);
			}
			else if (config._strict && !parsedInput) {
				getParsingFlags(config).unusedTokens.push(token);
			}
		}

		// add remaining unparsed input length to the string
		getParsingFlags(config).charsLeftOver = stringLength - totalParsedInputLength;
		if (string.length > 0) {
			getParsingFlags(config).unusedInput.push(string);
		}

		// clear _12h flag if hour is <= 12
		if (config._a[HOUR] <= 12 &&
			getParsingFlags(config).bigHour === true &&
			config._a[HOUR] > 0) {
			getParsingFlags(config).bigHour = undefined;
		}

		getParsingFlags(config).parsedDateParts = config._a.slice(0);
		getParsingFlags(config).meridiem = config._meridiem;
		// handle meridiem
		config._a[HOUR] = meridiemFixWrap(config._locale, config._a[HOUR], config._meridiem);

		configFromArray(config);
		checkOverflow(config);
	}


	function meridiemFixWrap (locale, hour, meridiem) {
		var isPm;

		if (meridiem == null) {
			// nothing to do
			return hour;
		}
		if (locale.meridiemHour != null) {
			return locale.meridiemHour(hour, meridiem);
		} else if (locale.isPM != null) {
			// Fallback
			isPm = locale.isPM(meridiem);
			if (isPm && hour < 12) {
				hour += 12;
			}
			if (!isPm && hour === 12) {
				hour = 0;
			}
			return hour;
		} else {
			// this is not supposed to happen
			return hour;
		}
	}

	// date from string and array of format strings
	function configFromStringAndArray(config) {
		var tempConfig,
			bestMoment,

			scoreToBeat,
			i,
			currentScore;

		if (config._f.length === 0) {
			getParsingFlags(config).invalidFormat = true;
			config._d = new Date(NaN);
			return;
		}

		for (i = 0; i < config._f.length; i++) {
			currentScore = 0;
			tempConfig = copyConfig({}, config);
			if (config._useUTC != null) {
				tempConfig._useUTC = config._useUTC;
			}
			tempConfig._f = config._f[i];
			configFromStringAndFormat(tempConfig);

			if (!isValid(tempConfig)) {
				continue;
			}

			// if there is any input that was not parsed add a penalty for that format
			currentScore += getParsingFlags(tempConfig).charsLeftOver;

			//or tokens
			currentScore += getParsingFlags(tempConfig).unusedTokens.length * 10;

			getParsingFlags(tempConfig).score = currentScore;

			if (scoreToBeat == null || currentScore < scoreToBeat) {
				scoreToBeat = currentScore;
				bestMoment = tempConfig;
			}
		}

		extend(config, bestMoment || tempConfig);
	}

	function configFromObject(config) {
		if (config._d) {
			return;
		}

		var i = normalizeObjectUnits(config._i);
		config._a = map([i.year, i.month, i.day || i.date, i.hour, i.minute, i.second, i.millisecond], function (obj) {
			return obj && parseInt(obj, 10);
		});

		configFromArray(config);
	}

	function createFromConfig (config) {
		var res = new Moment(checkOverflow(prepareConfig(config)));
		if (res._nextDay) {
			// Adding is smart enough around DST
			res.add(1, 'd');
			res._nextDay = undefined;
		}

		return res;
	}

	function prepareConfig (config) {
		var input = config._i,
			format = config._f;

		config._locale = config._locale || getLocale(config._l);

		if (input === null || (format === undefined && input === '')) {
			return createInvalid({nullInput: true});
		}

		if (typeof input === 'string') {
			config._i = input = config._locale.preparse(input);
		}

		if (isMoment(input)) {
			return new Moment(checkOverflow(input));
		} else if (isDate(input)) {
			config._d = input;
		} else if (isArray(format)) {
			configFromStringAndArray(config);
		} else if (format) {
			configFromStringAndFormat(config);
		}  else {
			configFromInput(config);
		}

		if (!isValid(config)) {
			config._d = null;
		}

		return config;
	}

	function configFromInput(config) {
		var input = config._i;
		if (isUndefined(input)) {
			config._d = new Date(hooks.now());
		} else if (isDate(input)) {
			config._d = new Date(input.valueOf());
		} else if (typeof input === 'string') {
			configFromString(config);
		} else if (isArray(input)) {
			config._a = map(input.slice(0), function (obj) {
				return parseInt(obj, 10);
			});
			configFromArray(config);
		} else if (isObject(input)) {
			configFromObject(config);
		} else if (isNumber(input)) {
			// from milliseconds
			config._d = new Date(input);
		} else {
			hooks.createFromInputFallback(config);
		}
	}

	function createLocalOrUTC (input, format, locale, strict, isUTC) {
		var c = {};

		if (locale === true || locale === false) {
			strict = locale;
			locale = undefined;
		}

		if ((isObject(input) && isObjectEmpty(input)) ||
			(isArray(input) && input.length === 0)) {
			input = undefined;
		}
		// object construction must be done this way.
		// https://github.com/moment/moment/issues/1423
		c._isAMomentObject = true;
		c._useUTC = c._isUTC = isUTC;
		c._l = locale;
		c._i = input;
		c._f = format;
		c._strict = strict;

		return createFromConfig(c);
	}

	function createLocal (input, format, locale, strict) {
		return createLocalOrUTC(input, format, locale, strict, false);
	}

	var prototypeMin = deprecate(
		'moment().min is deprecated, use moment.max instead. http://momentjs.com/guides/#/warnings/min-max/',
		function () {
			var other = createLocal.apply(null, arguments);
			if (this.isValid() && other.isValid()) {
				return other < this ? this : other;
			} else {
				return createInvalid();
			}
		}
	);

	var prototypeMax = deprecate(
		'moment().max is deprecated, use moment.min instead. http://momentjs.com/guides/#/warnings/min-max/',
		function () {
			var other = createLocal.apply(null, arguments);
			if (this.isValid() && other.isValid()) {
				return other > this ? this : other;
			} else {
				return createInvalid();
			}
		}
	);

	// Pick a moment m from moments so that m[fn](other) is true for all
	// other. This relies on the function fn to be transitive.
	//
	// moments should either be an array of moment objects or an array, whose
	// first element is an array of moment objects.
	function pickBy(fn, moments) {
		var res, i;
		if (moments.length === 1 && isArray(moments[0])) {
			moments = moments[0];
		}
		if (!moments.length) {
			return createLocal();
		}
		res = moments[0];
		for (i = 1; i < moments.length; ++i) {
			if (!moments[i].isValid() || moments[i][fn](res)) {
				res = moments[i];
			}
		}
		return res;
	}

	// TODO: Use [].sort instead?
	function min () {
		var args = [].slice.call(arguments, 0);

		return pickBy('isBefore', args);
	}

	function max () {
		var args = [].slice.call(arguments, 0);

		return pickBy('isAfter', args);
	}

	var now = function () {
		return Date.now ? Date.now() : +(new Date());
	};

	var ordering = ['year', 'quarter', 'month', 'week', 'day', 'hour', 'minute', 'second', 'millisecond'];

	function isDurationValid(m) {
		for (var key in m) {
			if (!(indexOf.call(ordering, key) !== -1 && (m[key] == null || !isNaN(m[key])))) {
				return false;
			}
		}

		var unitHasDecimal = false;
		for (var i = 0; i < ordering.length; ++i) {
			if (m[ordering[i]]) {
				if (unitHasDecimal) {
					return false; // only allow non-integers for smallest unit
				}
				if (parseFloat(m[ordering[i]]) !== toInt(m[ordering[i]])) {
					unitHasDecimal = true;
				}
			}
		}

		return true;
	}

	function isValid$1() {
		return this._isValid;
	}

	function createInvalid$1() {
		return createDuration(NaN);
	}

	function Duration (duration) {
		var normalizedInput = normalizeObjectUnits(duration),
			years = normalizedInput.year || 0,
			quarters = normalizedInput.quarter || 0,
			months = normalizedInput.month || 0,
			weeks = normalizedInput.week || normalizedInput.isoWeek || 0,
			days = normalizedInput.day || 0,
			hours = normalizedInput.hour || 0,
			minutes = normalizedInput.minute || 0,
			seconds = normalizedInput.second || 0,
			milliseconds = normalizedInput.millisecond || 0;

		this._isValid = isDurationValid(normalizedInput);

		// representation for dateAddRemove
		this._milliseconds = +milliseconds +
			seconds * 1e3 + // 1000
			minutes * 6e4 + // 1000 * 60
			hours * 1000 * 60 * 60; //using 1000 * 60 * 60 instead of 36e5 to avoid floating point rounding errors https://github.com/moment/moment/issues/2978
		// Because of dateAddRemove treats 24 hours as different from a
		// day when working around DST, we need to store them separately
		this._days = +days +
			weeks * 7;
		// It is impossible to translate months into days without knowing
		// which months you are are talking about, so we have to store
		// it separately.
		this._months = +months +
			quarters * 3 +
			years * 12;

		this._data = {};

		this._locale = getLocale();

		this._bubble();
	}

	function isDuration (obj) {
		return obj instanceof Duration;
	}

	function absRound (number) {
		if (number < 0) {
			return Math.round(-1 * number) * -1;
		} else {
			return Math.round(number);
		}
	}

	// FORMATTING

	function offset (token, separator) {
		addFormatToken(token, 0, 0, function () {
			var offset = this.utcOffset();
			var sign = '+';
			if (offset < 0) {
				offset = -offset;
				sign = '-';
			}
			return sign + zeroFill(~~(offset / 60), 2) + separator + zeroFill(~~(offset) % 60, 2);
		});
	}

	offset('Z', ':');
	offset('ZZ', '');

	// PARSING

	addRegexToken('Z',  matchShortOffset);
	addRegexToken('ZZ', matchShortOffset);
	addParseToken(['Z', 'ZZ'], function (input, array, config) {
		config._useUTC = true;
		config._tzm = offsetFromString(matchShortOffset, input);
	});

	// HELPERS

	// timezone chunker
	// '+10:00' > ['10',  '00']
	// '-1530'  > ['-15', '30']
	var chunkOffset = /([\+\-]|\d\d)/gi;

	function offsetFromString(matcher, string) {
		var matches = (string || '').match(matcher);

		if (matches === null) {
			return null;
		}

		var chunk   = matches[matches.length - 1] || [];
		var parts   = (chunk + '').match(chunkOffset) || ['-', 0, 0];
		var minutes = +(parts[1] * 60) + toInt(parts[2]);

		return minutes === 0 ?
			0 :
			parts[0] === '+' ? minutes : -minutes;
	}

	// Return a moment from input, that is local/utc/zone equivalent to model.
	function cloneWithOffset(input, model) {
		var res, diff;
		if (model._isUTC) {
			res = model.clone();
			diff = (isMoment(input) || isDate(input) ? input.valueOf() : createLocal(input).valueOf()) - res.valueOf();
			// Use low-level api, because this fn is low-level api.
			res._d.setTime(res._d.valueOf() + diff);
			hooks.updateOffset(res, false);
			return res;
		} else {
			return createLocal(input).local();
		}
	}

	function getDateOffset (m) {
		// On Firefox.24 Date#getTimezoneOffset returns a floating point.
		// https://github.com/moment/moment/pull/1871
		return -Math.round(m._d.getTimezoneOffset() / 15) * 15;
	}

	// HOOKS

	// This function will be called whenever a moment is mutated.
	// It is intended to keep the offset in sync with the timezone.
	hooks.updateOffset = function () {};

	// MOMENTS

	// keepLocalTime = true means only change the timezone, without
	// affecting the local hour. So 5:31:26 +0300 --[utcOffset(2, true)]-->
	// 5:31:26 +0200 It is possible that 5:31:26 doesn't exist with offset
	// +0200, so we adjust the time as needed, to be valid.
	//
	// Keeping the time actually adds/subtracts (one hour)
	// from the actual represented time. That is why we call updateOffset
	// a second time. In case it wants us to change the offset again
	// _changeInProgress == true case, then we have to adjust, because
	// there is no such time in the given timezone.
	function getSetOffset (input, keepLocalTime, keepMinutes) {
		var offset = this._offset || 0,
			localAdjust;
		if (!this.isValid()) {
			return input != null ? this : NaN;
		}
		if (input != null) {
			if (typeof input === 'string') {
				input = offsetFromString(matchShortOffset, input);
				if (input === null) {
					return this;
				}
			} else if (Math.abs(input) < 16 && !keepMinutes) {
				input = input * 60;
			}
			if (!this._isUTC && keepLocalTime) {
				localAdjust = getDateOffset(this);
			}
			this._offset = input;
			this._isUTC = true;
			if (localAdjust != null) {
				this.add(localAdjust, 'm');
			}
			if (offset !== input) {
				if (!keepLocalTime || this._changeInProgress) {
					addSubtract(this, createDuration(input - offset, 'm'), 1, false);
				} else if (!this._changeInProgress) {
					this._changeInProgress = true;
					hooks.updateOffset(this, true);
					this._changeInProgress = null;
				}
			}
			return this;
		} else {
			return this._isUTC ? offset : getDateOffset(this);
		}
	}

	function getSetZone (input, keepLocalTime) {
		if (input != null) {
			if (typeof input !== 'string') {
				input = -input;
			}

			this.utcOffset(input, keepLocalTime);

			return this;
		} else {
			return -this.utcOffset();
		}
	}

	function setOffsetToUTC (keepLocalTime) {
		return this.utcOffset(0, keepLocalTime);
	}

	function setOffsetToLocal (keepLocalTime) {
		if (this._isUTC) {
			this.utcOffset(0, keepLocalTime);
			this._isUTC = false;

			if (keepLocalTime) {
				this.subtract(getDateOffset(this), 'm');
			}
		}
		return this;
	}

	function setOffsetToParsedOffset () {
		if (this._tzm != null) {
			this.utcOffset(this._tzm, false, true);
		} else if (typeof this._i === 'string') {
			var tZone = offsetFromString(matchOffset, this._i);
			if (tZone != null) {
				this.utcOffset(tZone);
			}
			else {
				this.utcOffset(0, true);
			}
		}
		return this;
	}

	function hasAlignedHourOffset (input) {
		if (!this.isValid()) {
			return false;
		}
		input = input ? createLocal(input).utcOffset() : 0;

		return (this.utcOffset() - input) % 60 === 0;
	}

	function isDaylightSavingTime () {
		return (
			this.utcOffset() > this.clone().month(0).utcOffset() ||
			this.utcOffset() > this.clone().month(5).utcOffset()
		);
	}

	function isDaylightSavingTimeShifted () {
		if (!isUndefined(this._isDSTShifted)) {
			return this._isDSTShifted;
		}

		var c = {};

		copyConfig(c, this);
		c = prepareConfig(c);

		if (c._a) {
			var other = c._isUTC ? createUTC(c._a) : createLocal(c._a);
			this._isDSTShifted = this.isValid() &&
				compareArrays(c._a, other.toArray()) > 0;
		} else {
			this._isDSTShifted = false;
		}

		return this._isDSTShifted;
	}

	function isLocal () {
		return this.isValid() ? !this._isUTC : false;
	}

	function isUtcOffset () {
		return this.isValid() ? this._isUTC : false;
	}

	function isUtc () {
		return this.isValid() ? this._isUTC && this._offset === 0 : false;
	}

	// ASP.NET json date format regex
	var aspNetRegex = /^(\-|\+)?(?:(\d*)[. ])?(\d+)\:(\d+)(?:\:(\d+)(\.\d*)?)?$/;

	// from http://docs.closure-library.googlecode.com/git/closure_goog_date_date.js.source.html
	// somewhat more in line with 4.4.3.2 2004 spec, but allows decimal anywhere
	// and further modified to allow for strings containing both week and day
	var isoRegex = /^(-|\+)?P(?:([-+]?[0-9,.]*)Y)?(?:([-+]?[0-9,.]*)M)?(?:([-+]?[0-9,.]*)W)?(?:([-+]?[0-9,.]*)D)?(?:T(?:([-+]?[0-9,.]*)H)?(?:([-+]?[0-9,.]*)M)?(?:([-+]?[0-9,.]*)S)?)?$/;

	function createDuration (input, key) {
		var duration = input,
			// matching against regexp is expensive, do it on demand
			match = null,
			sign,
			ret,
			diffRes;

		if (isDuration(input)) {
			duration = {
				ms : input._milliseconds,
				d  : input._days,
				M  : input._months
			};
		} else if (isNumber(input)) {
			duration = {};
			if (key) {
				duration[key] = input;
			} else {
				duration.milliseconds = input;
			}
		} else if (!!(match = aspNetRegex.exec(input))) {
			sign = (match[1] === '-') ? -1 : 1;
			duration = {
				y  : 0,
				d  : toInt(match[DATE])                         * sign,
				h  : toInt(match[HOUR])                         * sign,
				m  : toInt(match[MINUTE])                       * sign,
				s  : toInt(match[SECOND])                       * sign,
				ms : toInt(absRound(match[MILLISECOND] * 1000)) * sign // the millisecond decimal point is included in the match
			};
		} else if (!!(match = isoRegex.exec(input))) {
			sign = (match[1] === '-') ? -1 : 1;
			duration = {
				y : parseIso(match[2], sign),
				M : parseIso(match[3], sign),
				w : parseIso(match[4], sign),
				d : parseIso(match[5], sign),
				h : parseIso(match[6], sign),
				m : parseIso(match[7], sign),
				s : parseIso(match[8], sign)
			};
		} else if (duration == null) {// checks for null or undefined
			duration = {};
		} else if (typeof duration === 'object' && ('from' in duration || 'to' in duration)) {
			diffRes = momentsDifference(createLocal(duration.from), createLocal(duration.to));

			duration = {};
			duration.ms = diffRes.milliseconds;
			duration.M = diffRes.months;
		}

		ret = new Duration(duration);

		if (isDuration(input) && hasOwnProp(input, '_locale')) {
			ret._locale = input._locale;
		}

		return ret;
	}

	createDuration.fn = Duration.prototype;
	createDuration.invalid = createInvalid$1;

	function parseIso (inp, sign) {
		// We'd normally use ~~inp for this, but unfortunately it also
		// converts floats to ints.
		// inp may be undefined, so careful calling replace on it.
		var res = inp && parseFloat(inp.replace(',', '.'));
		// apply sign while we're at it
		return (isNaN(res) ? 0 : res) * sign;
	}

	function positiveMomentsDifference(base, other) {
		var res = {};

		res.months = other.month() - base.month() +
			(other.year() - base.year()) * 12;
		if (base.clone().add(res.months, 'M').isAfter(other)) {
			--res.months;
		}

		res.milliseconds = +other - +(base.clone().add(res.months, 'M'));

		return res;
	}

	function momentsDifference(base, other) {
		var res;
		if (!(base.isValid() && other.isValid())) {
			return {milliseconds: 0, months: 0};
		}

		other = cloneWithOffset(other, base);
		if (base.isBefore(other)) {
			res = positiveMomentsDifference(base, other);
		} else {
			res = positiveMomentsDifference(other, base);
			res.milliseconds = -res.milliseconds;
			res.months = -res.months;
		}

		return res;
	}

	// TODO: remove 'name' arg after deprecation is removed
	function createAdder(direction, name) {
		return function (val, period) {
			var dur, tmp;
			//invert the arguments, but complain about it
			if (period !== null && !isNaN(+period)) {
				deprecateSimple(name, 'moment().' + name  + '(period, number) is deprecated. Please use moment().' + name + '(number, period). ' +
					'See http://momentjs.com/guides/#/warnings/add-inverted-param/ for more info.');
				tmp = val; val = period; period = tmp;
			}

			val = typeof val === 'string' ? +val : val;
			dur = createDuration(val, period);
			addSubtract(this, dur, direction);
			return this;
		};
	}

	function addSubtract (mom, duration, isAdding, updateOffset) {
		var milliseconds = duration._milliseconds,
			days = absRound(duration._days),
			months = absRound(duration._months);

		if (!mom.isValid()) {
			// No op
			return;
		}

		updateOffset = updateOffset == null ? true : updateOffset;

		if (months) {
			setMonth(mom, get(mom, 'Month') + months * isAdding);
		}
		if (days) {
			set$1(mom, 'Date', get(mom, 'Date') + days * isAdding);
		}
		if (milliseconds) {
			mom._d.setTime(mom._d.valueOf() + milliseconds * isAdding);
		}
		if (updateOffset) {
			hooks.updateOffset(mom, days || months);
		}
	}

	var add      = createAdder(1, 'add');
	var subtract = createAdder(-1, 'subtract');

	function getCalendarFormat(myMoment, now) {
		var diff = myMoment.diff(now, 'days', true);
		return diff < -6 ? 'sameElse' :
			diff < -1 ? 'lastWeek' :
				diff < 0 ? 'lastDay' :
					diff < 1 ? 'sameDay' :
						diff < 2 ? 'nextDay' :
							diff < 7 ? 'nextWeek' : 'sameElse';
	}

	function calendar$1 (time, formats) {
		// We want to compare the start of today, vs this.
		// Getting start-of-today depends on whether we're local/utc/offset or not.
		var now = time || createLocal(),
			sod = cloneWithOffset(now, this).startOf('day'),
			format = hooks.calendarFormat(this, sod) || 'sameElse';

		var output = formats && (isFunction(formats[format]) ? formats[format].call(this, now) : formats[format]);

		return this.format(output || this.localeData().calendar(format, this, createLocal(now)));
	}

	function clone () {
		return new Moment(this);
	}

	function isAfter (input, units) {
		var localInput = isMoment(input) ? input : createLocal(input);
		if (!(this.isValid() && localInput.isValid())) {
			return false;
		}
		units = normalizeUnits(units) || 'millisecond';
		if (units === 'millisecond') {
			return this.valueOf() > localInput.valueOf();
		} else {
			return localInput.valueOf() < this.clone().startOf(units).valueOf();
		}
	}

	function isBefore (input, units) {
		var localInput = isMoment(input) ? input : createLocal(input);
		if (!(this.isValid() && localInput.isValid())) {
			return false;
		}
		units = normalizeUnits(units) || 'millisecond';
		if (units === 'millisecond') {
			return this.valueOf() < localInput.valueOf();
		} else {
			return this.clone().endOf(units).valueOf() < localInput.valueOf();
		}
	}

	function isBetween (from, to, units, inclusivity) {
		var localFrom = isMoment(from) ? from : createLocal(from),
			localTo = isMoment(to) ? to : createLocal(to);
		if (!(this.isValid() && localFrom.isValid() && localTo.isValid())) {
			return false;
		}
		inclusivity = inclusivity || '()';
		return (inclusivity[0] === '(' ? this.isAfter(localFrom, units) : !this.isBefore(localFrom, units)) &&
			(inclusivity[1] === ')' ? this.isBefore(localTo, units) : !this.isAfter(localTo, units));
	}

	function isSame (input, units) {
		var localInput = isMoment(input) ? input : createLocal(input),
			inputMs;
		if (!(this.isValid() && localInput.isValid())) {
			return false;
		}
		units = normalizeUnits(units) || 'millisecond';
		if (units === 'millisecond') {
			return this.valueOf() === localInput.valueOf();
		} else {
			inputMs = localInput.valueOf();
			return this.clone().startOf(units).valueOf() <= inputMs && inputMs <= this.clone().endOf(units).valueOf();
		}
	}

	function isSameOrAfter (input, units) {
		return this.isSame(input, units) || this.isAfter(input, units);
	}

	function isSameOrBefore (input, units) {
		return this.isSame(input, units) || this.isBefore(input, units);
	}

	function diff (input, units, asFloat) {
		var that,
			zoneDelta,
			output;

		if (!this.isValid()) {
			return NaN;
		}

		that = cloneWithOffset(input, this);

		if (!that.isValid()) {
			return NaN;
		}

		zoneDelta = (that.utcOffset() - this.utcOffset()) * 6e4;

		units = normalizeUnits(units);

		switch (units) {
			case 'year': output = monthDiff(this, that) / 12; break;
			case 'month': output = monthDiff(this, that); break;
			case 'quarter': output = monthDiff(this, that) / 3; break;
			case 'second': output = (this - that) / 1e3; break; // 1000
			case 'minute': output = (this - that) / 6e4; break; // 1000 * 60
			case 'hour': output = (this - that) / 36e5; break; // 1000 * 60 * 60
			case 'day': output = (this - that - zoneDelta) / 864e5; break; // 1000 * 60 * 60 * 24, negate dst
			case 'week': output = (this - that - zoneDelta) / 6048e5; break; // 1000 * 60 * 60 * 24 * 7, negate dst
			default: output = this - that;
		}

		return asFloat ? output : absFloor(output);
	}

	function monthDiff (a, b) {
		// difference in months
		var wholeMonthDiff = ((b.year() - a.year()) * 12) + (b.month() - a.month()),
			// b is in (anchor - 1 month, anchor + 1 month)
			anchor = a.clone().add(wholeMonthDiff, 'months'),
			anchor2, adjust;

		if (b - anchor < 0) {
			anchor2 = a.clone().add(wholeMonthDiff - 1, 'months');
			// linear across the month
			adjust = (b - anchor) / (anchor - anchor2);
		} else {
			anchor2 = a.clone().add(wholeMonthDiff + 1, 'months');
			// linear across the month
			adjust = (b - anchor) / (anchor2 - anchor);
		}

		//check for negative zero, return zero if negative zero
		return -(wholeMonthDiff + adjust) || 0;
	}

	hooks.defaultFormat = 'YYYY-MM-DDTHH:mm:ssZ';
	hooks.defaultFormatUtc = 'YYYY-MM-DDTHH:mm:ss[Z]';

	function toString () {
		return this.clone().locale('en').format('ddd MMM DD YYYY HH:mm:ss [GMT]ZZ');
	}

	function toISOString(keepOffset) {
		if (!this.isValid()) {
			return null;
		}
		var utc = keepOffset !== true;
		var m = utc ? this.clone().utc() : this;
		if (m.year() < 0 || m.year() > 9999) {
			return formatMoment(m, utc ? 'YYYYYY-MM-DD[T]HH:mm:ss.SSS[Z]' : 'YYYYYY-MM-DD[T]HH:mm:ss.SSSZ');
		}
		if (isFunction(Date.prototype.toISOString)) {
			// native implementation is ~50x faster, use it when we can
			if (utc) {
				return this.toDate().toISOString();
			} else {
				return new Date(this.valueOf() + this.utcOffset() * 60 * 1000).toISOString().replace('Z', formatMoment(m, 'Z'));
			}
		}
		return formatMoment(m, utc ? 'YYYY-MM-DD[T]HH:mm:ss.SSS[Z]' : 'YYYY-MM-DD[T]HH:mm:ss.SSSZ');
	}

	/**
	 * Return a human readable representation of a moment that can
	 * also be evaluated to get a new moment which is the same
	 *
	 * @link https://nodejs.org/dist/latest/docs/api/util.html#util_custom_inspect_function_on_objects
	 */
	function inspect () {
		if (!this.isValid()) {
			return 'moment.invalid(/* ' + this._i + ' */)';
		}
		var func = 'moment';
		var zone = '';
		if (!this.isLocal()) {
			func = this.utcOffset() === 0 ? 'moment.utc' : 'moment.parseZone';
			zone = 'Z';
		}
		var prefix = '[' + func + '("]';
		var year = (0 <= this.year() && this.year() <= 9999) ? 'YYYY' : 'YYYYYY';
		var datetime = '-MM-DD[T]HH:mm:ss.SSS';
		var suffix = zone + '[")]';

		return this.format(prefix + year + datetime + suffix);
	}

	function format (inputString) {
		if (!inputString) {
			inputString = this.isUtc() ? hooks.defaultFormatUtc : hooks.defaultFormat;
		}
		var output = formatMoment(this, inputString);
		return this.localeData().postformat(output);
	}

	function from (time, withoutSuffix) {
		if (this.isValid() &&
			((isMoment(time) && time.isValid()) ||
				createLocal(time).isValid())) {
			return createDuration({to: this, from: time}).locale(this.locale()).humanize(!withoutSuffix);
		} else {
			return this.localeData().invalidDate();
		}
	}

	function fromNow (withoutSuffix) {
		return this.from(createLocal(), withoutSuffix);
	}

	function to (time, withoutSuffix) {
		if (this.isValid() &&
			((isMoment(time) && time.isValid()) ||
				createLocal(time).isValid())) {
			return createDuration({from: this, to: time}).locale(this.locale()).humanize(!withoutSuffix);
		} else {
			return this.localeData().invalidDate();
		}
	}

	function toNow (withoutSuffix) {
		return this.to(createLocal(), withoutSuffix);
	}

	// If passed a locale key, it will set the locale for this
	// instance.  Otherwise, it will return the locale configuration
	// variables for this instance.
	function locale (key) {
		var newLocaleData;

		if (key === undefined) {
			return this._locale._abbr;
		} else {
			newLocaleData = getLocale(key);
			if (newLocaleData != null) {
				this._locale = newLocaleData;
			}
			return this;
		}
	}

	var lang = deprecate(
		'moment().lang() is deprecated. Instead, use moment().localeData() to get the language configuration. Use moment().locale() to change languages.',
		function (key) {
			if (key === undefined) {
				return this.localeData();
			} else {
				return this.locale(key);
			}
		}
	);

	function localeData () {
		return this._locale;
	}

	var MS_PER_SECOND = 1000;
	var MS_PER_MINUTE = 60 * MS_PER_SECOND;
	var MS_PER_HOUR = 60 * MS_PER_MINUTE;
	var MS_PER_400_YEARS = (365 * 400 + 97) * 24 * MS_PER_HOUR;

	// actual modulo - handles negative numbers (for dates before 1970):
	function mod$1(dividend, divisor) {
		return (dividend % divisor + divisor) % divisor;
	}

	function localStartOfDate(y, m, d) {
		// the date constructor remaps years 0-99 to 1900-1999
		if (y < 100 && y >= 0) {
			// preserve leap years using a full 400 year cycle, then reset
			return new Date(y + 400, m, d) - MS_PER_400_YEARS;
		} else {
			return new Date(y, m, d).valueOf();
		}
	}

	function utcStartOfDate(y, m, d) {
		// Date.UTC remaps years 0-99 to 1900-1999
		if (y < 100 && y >= 0) {
			// preserve leap years using a full 400 year cycle, then reset
			return Date.UTC(y + 400, m, d) - MS_PER_400_YEARS;
		} else {
			return Date.UTC(y, m, d);
		}
	}

	function startOf (units) {
		var time;
		units = normalizeUnits(units);
		if (units === undefined || units === 'millisecond' || !this.isValid()) {
			return this;
		}

		var startOfDate = this._isUTC ? utcStartOfDate : localStartOfDate;

		switch (units) {
			case 'year':
				time = startOfDate(this.year(), 0, 1);
				break;
			case 'quarter':
				time = startOfDate(this.year(), this.month() - this.month() % 3, 1);
				break;
			case 'month':
				time = startOfDate(this.year(), this.month(), 1);
				break;
			case 'week':
				time = startOfDate(this.year(), this.month(), this.date() - this.weekday());
				break;
			case 'isoWeek':
				time = startOfDate(this.year(), this.month(), this.date() - (this.isoWeekday() - 1));
				break;
			case 'day':
			case 'date':
				time = startOfDate(this.year(), this.month(), this.date());
				break;
			case 'hour':
				time = this._d.valueOf();
				time -= mod$1(time + (this._isUTC ? 0 : this.utcOffset() * MS_PER_MINUTE), MS_PER_HOUR);
				break;
			case 'minute':
				time = this._d.valueOf();
				time -= mod$1(time, MS_PER_MINUTE);
				break;
			case 'second':
				time = this._d.valueOf();
				time -= mod$1(time, MS_PER_SECOND);
				break;
		}

		this._d.setTime(time);
		hooks.updateOffset(this, true);
		return this;
	}

	function endOf (units) {
		var time;
		units = normalizeUnits(units);
		if (units === undefined || units === 'millisecond' || !this.isValid()) {
			return this;
		}

		var startOfDate = this._isUTC ? utcStartOfDate : localStartOfDate;

		switch (units) {
			case 'year':
				time = startOfDate(this.year() + 1, 0, 1) - 1;
				break;
			case 'quarter':
				time = startOfDate(this.year(), this.month() - this.month() % 3 + 3, 1) - 1;
				break;
			case 'month':
				time = startOfDate(this.year(), this.month() + 1, 1) - 1;
				break;
			case 'week':
				time = startOfDate(this.year(), this.month(), this.date() - this.weekday() + 7) - 1;
				break;
			case 'isoWeek':
				time = startOfDate(this.year(), this.month(), this.date() - (this.isoWeekday() - 1) + 7) - 1;
				break;
			case 'day':
			case 'date':
				time = startOfDate(this.year(), this.month(), this.date() + 1) - 1;
				break;
			case 'hour':
				time = this._d.valueOf();
				time += MS_PER_HOUR - mod$1(time + (this._isUTC ? 0 : this.utcOffset() * MS_PER_MINUTE), MS_PER_HOUR) - 1;
				break;
			case 'minute':
				time = this._d.valueOf();
				time += MS_PER_MINUTE - mod$1(time, MS_PER_MINUTE) - 1;
				break;
			case 'second':
				time = this._d.valueOf();
				time += MS_PER_SECOND - mod$1(time, MS_PER_SECOND) - 1;
				break;
		}

		this._d.setTime(time);
		hooks.updateOffset(this, true);
		return this;
	}

	function valueOf () {
		return this._d.valueOf() - ((this._offset || 0) * 60000);
	}

	function unix () {
		return Math.floor(this.valueOf() / 1000);
	}

	function toDate () {
		return new Date(this.valueOf());
	}

	function toArray () {
		var m = this;
		return [m.year(), m.month(), m.date(), m.hour(), m.minute(), m.second(), m.millisecond()];
	}

	function toObject () {
		var m = this;
		return {
			years: m.year(),
			months: m.month(),
			date: m.date(),
			hours: m.hours(),
			minutes: m.minutes(),
			seconds: m.seconds(),
			milliseconds: m.milliseconds()
		};
	}

	function toJSON () {
		// new Date(NaN).toJSON() === null
		return this.isValid() ? this.toISOString() : null;
	}

	function isValid$2 () {
		return isValid(this);
	}

	function parsingFlags () {
		return extend({}, getParsingFlags(this));
	}

	function invalidAt () {
		return getParsingFlags(this).overflow;
	}

	function creationData() {
		return {
			input: this._i,
			format: this._f,
			locale: this._locale,
			isUTC: this._isUTC,
			strict: this._strict
		};
	}

	// FORMATTING

	addFormatToken(0, ['gg', 2], 0, function () {
		return this.weekYear() % 100;
	});

	addFormatToken(0, ['GG', 2], 0, function () {
		return this.isoWeekYear() % 100;
	});

	function addWeekYearFormatToken (token, getter) {
		addFormatToken(0, [token, token.length], 0, getter);
	}

	addWeekYearFormatToken('gggg',     'weekYear');
	addWeekYearFormatToken('ggggg',    'weekYear');
	addWeekYearFormatToken('GGGG',  'isoWeekYear');
	addWeekYearFormatToken('GGGGG', 'isoWeekYear');

	// ALIASES

	addUnitAlias('weekYear', 'gg');
	addUnitAlias('isoWeekYear', 'GG');

	// PRIORITY

	addUnitPriority('weekYear', 1);
	addUnitPriority('isoWeekYear', 1);


	// PARSING

	addRegexToken('G',      matchSigned);
	addRegexToken('g',      matchSigned);
	addRegexToken('GG',     match1to2, match2);
	addRegexToken('gg',     match1to2, match2);
	addRegexToken('GGGG',   match1to4, match4);
	addRegexToken('gggg',   match1to4, match4);
	addRegexToken('GGGGG',  match1to6, match6);
	addRegexToken('ggggg',  match1to6, match6);

	addWeekParseToken(['gggg', 'ggggg', 'GGGG', 'GGGGG'], function (input, week, config, token) {
		week[token.substr(0, 2)] = toInt(input);
	});

	addWeekParseToken(['gg', 'GG'], function (input, week, config, token) {
		week[token] = hooks.parseTwoDigitYear(input);
	});

	// MOMENTS

	function getSetWeekYear (input) {
		return getSetWeekYearHelper.call(this,
			input,
			this.week(),
			this.weekday(),
			this.localeData()._week.dow,
			this.localeData()._week.doy);
	}

	function getSetISOWeekYear (input) {
		return getSetWeekYearHelper.call(this,
			input, this.isoWeek(), this.isoWeekday(), 1, 4);
	}

	function getISOWeeksInYear () {
		return weeksInYear(this.year(), 1, 4);
	}

	function getWeeksInYear () {
		var weekInfo = this.localeData()._week;
		return weeksInYear(this.year(), weekInfo.dow, weekInfo.doy);
	}

	function getSetWeekYearHelper(input, week, weekday, dow, doy) {
		var weeksTarget;
		if (input == null) {
			return weekOfYear(this, dow, doy).year;
		} else {
			weeksTarget = weeksInYear(input, dow, doy);
			if (week > weeksTarget) {
				week = weeksTarget;
			}
			return setWeekAll.call(this, input, week, weekday, dow, doy);
		}
	}

	function setWeekAll(weekYear, week, weekday, dow, doy) {
		var dayOfYearData = dayOfYearFromWeeks(weekYear, week, weekday, dow, doy),
			date = createUTCDate(dayOfYearData.year, 0, dayOfYearData.dayOfYear);

		this.year(date.getUTCFullYear());
		this.month(date.getUTCMonth());
		this.date(date.getUTCDate());
		return this;
	}

	// FORMATTING

	addFormatToken('Q', 0, 'Qo', 'quarter');

	// ALIASES

	addUnitAlias('quarter', 'Q');

	// PRIORITY

	addUnitPriority('quarter', 7);

	// PARSING

	addRegexToken('Q', match1);
	addParseToken('Q', function (input, array) {
		array[MONTH] = (toInt(input) - 1) * 3;
	});

	// MOMENTS

	function getSetQuarter (input) {
		return input == null ? Math.ceil((this.month() + 1) / 3) : this.month((input - 1) * 3 + this.month() % 3);
	}

	// FORMATTING

	addFormatToken('D', ['DD', 2], 'Do', 'date');

	// ALIASES

	addUnitAlias('date', 'D');

	// PRIORITY
	addUnitPriority('date', 9);

	// PARSING

	addRegexToken('D',  match1to2);
	addRegexToken('DD', match1to2, match2);
	addRegexToken('Do', function (isStrict, locale) {
		// TODO: Remove "ordinalParse" fallback in next major release.
		return isStrict ?
			(locale._dayOfMonthOrdinalParse || locale._ordinalParse) :
			locale._dayOfMonthOrdinalParseLenient;
	});

	addParseToken(['D', 'DD'], DATE);
	addParseToken('Do', function (input, array) {
		array[DATE] = toInt(input.match(match1to2)[0]);
	});

	// MOMENTS

	var getSetDayOfMonth = makeGetSet('Date', true);

	// FORMATTING

	addFormatToken('DDD', ['DDDD', 3], 'DDDo', 'dayOfYear');

	// ALIASES

	addUnitAlias('dayOfYear', 'DDD');

	// PRIORITY
	addUnitPriority('dayOfYear', 4);

	// PARSING

	addRegexToken('DDD',  match1to3);
	addRegexToken('DDDD', match3);
	addParseToken(['DDD', 'DDDD'], function (input, array, config) {
		config._dayOfYear = toInt(input);
	});

	// HELPERS

	// MOMENTS

	function getSetDayOfYear (input) {
		var dayOfYear = Math.round((this.clone().startOf('day') - this.clone().startOf('year')) / 864e5) + 1;
		return input == null ? dayOfYear : this.add((input - dayOfYear), 'd');
	}

	// FORMATTING

	addFormatToken('m', ['mm', 2], 0, 'minute');

	// ALIASES

	addUnitAlias('minute', 'm');

	// PRIORITY

	addUnitPriority('minute', 14);

	// PARSING

	addRegexToken('m',  match1to2);
	addRegexToken('mm', match1to2, match2);
	addParseToken(['m', 'mm'], MINUTE);

	// MOMENTS

	var getSetMinute = makeGetSet('Minutes', false);

	// FORMATTING

	addFormatToken('s', ['ss', 2], 0, 'second');

	// ALIASES

	addUnitAlias('second', 's');

	// PRIORITY

	addUnitPriority('second', 15);

	// PARSING

	addRegexToken('s',  match1to2);
	addRegexToken('ss', match1to2, match2);
	addParseToken(['s', 'ss'], SECOND);

	// MOMENTS

	var getSetSecond = makeGetSet('Seconds', false);

	// FORMATTING

	addFormatToken('S', 0, 0, function () {
		return ~~(this.millisecond() / 100);
	});

	addFormatToken(0, ['SS', 2], 0, function () {
		return ~~(this.millisecond() / 10);
	});

	addFormatToken(0, ['SSS', 3], 0, 'millisecond');
	addFormatToken(0, ['SSSS', 4], 0, function () {
		return this.millisecond() * 10;
	});
	addFormatToken(0, ['SSSSS', 5], 0, function () {
		return this.millisecond() * 100;
	});
	addFormatToken(0, ['SSSSSS', 6], 0, function () {
		return this.millisecond() * 1000;
	});
	addFormatToken(0, ['SSSSSSS', 7], 0, function () {
		return this.millisecond() * 10000;
	});
	addFormatToken(0, ['SSSSSSSS', 8], 0, function () {
		return this.millisecond() * 100000;
	});
	addFormatToken(0, ['SSSSSSSSS', 9], 0, function () {
		return this.millisecond() * 1000000;
	});


	// ALIASES

	addUnitAlias('millisecond', 'ms');

	// PRIORITY

	addUnitPriority('millisecond', 16);

	// PARSING

	addRegexToken('S',    match1to3, match1);
	addRegexToken('SS',   match1to3, match2);
	addRegexToken('SSS',  match1to3, match3);

	var token;
	for (token = 'SSSS'; token.length <= 9; token += 'S') {
		addRegexToken(token, matchUnsigned);
	}

	function parseMs(input, array) {
		array[MILLISECOND] = toInt(('0.' + input) * 1000);
	}

	for (token = 'S'; token.length <= 9; token += 'S') {
		addParseToken(token, parseMs);
	}
	// MOMENTS

	var getSetMillisecond = makeGetSet('Milliseconds', false);

	// FORMATTING

	addFormatToken('z',  0, 0, 'zoneAbbr');
	addFormatToken('zz', 0, 0, 'zoneName');

	// MOMENTS

	function getZoneAbbr () {
		return this._isUTC ? 'UTC' : '';
	}

	function getZoneName () {
		return this._isUTC ? 'Coordinated Universal Time' : '';
	}

	var proto = Moment.prototype;

	proto.add               = add;
	proto.calendar          = calendar$1;
	proto.clone             = clone;
	proto.diff              = diff;
	proto.endOf             = endOf;
	proto.format            = format;
	proto.from              = from;
	proto.fromNow           = fromNow;
	proto.to                = to;
	proto.toNow             = toNow;
	proto.get               = stringGet;
	proto.invalidAt         = invalidAt;
	proto.isAfter           = isAfter;
	proto.isBefore          = isBefore;
	proto.isBetween         = isBetween;
	proto.isSame            = isSame;
	proto.isSameOrAfter     = isSameOrAfter;
	proto.isSameOrBefore    = isSameOrBefore;
	proto.isValid           = isValid$2;
	proto.lang              = lang;
	proto.locale            = locale;
	proto.localeData        = localeData;
	proto.max               = prototypeMax;
	proto.min               = prototypeMin;
	proto.parsingFlags      = parsingFlags;
	proto.set               = stringSet;
	proto.startOf           = startOf;
	proto.subtract          = subtract;
	proto.toArray           = toArray;
	proto.toObject          = toObject;
	proto.toDate            = toDate;
	proto.toISOString       = toISOString;
	proto.inspect           = inspect;
	proto.toJSON            = toJSON;
	proto.toString          = toString;
	proto.unix              = unix;
	proto.valueOf           = valueOf;
	proto.creationData      = creationData;
	proto.year       = getSetYear;
	proto.isLeapYear = getIsLeapYear;
	proto.weekYear    = getSetWeekYear;
	proto.isoWeekYear = getSetISOWeekYear;
	proto.quarter = proto.quarters = getSetQuarter;
	proto.month       = getSetMonth;
	proto.daysInMonth = getDaysInMonth;
	proto.week           = proto.weeks        = getSetWeek;
	proto.isoWeek        = proto.isoWeeks     = getSetISOWeek;
	proto.weeksInYear    = getWeeksInYear;
	proto.isoWeeksInYear = getISOWeeksInYear;
	proto.date       = getSetDayOfMonth;
	proto.day        = proto.days             = getSetDayOfWeek;
	proto.weekday    = getSetLocaleDayOfWeek;
	proto.isoWeekday = getSetISODayOfWeek;
	proto.dayOfYear  = getSetDayOfYear;
	proto.hour = proto.hours = getSetHour;
	proto.minute = proto.minutes = getSetMinute;
	proto.second = proto.seconds = getSetSecond;
	proto.millisecond = proto.milliseconds = getSetMillisecond;
	proto.utcOffset            = getSetOffset;
	proto.utc                  = setOffsetToUTC;
	proto.local                = setOffsetToLocal;
	proto.parseZone            = setOffsetToParsedOffset;
	proto.hasAlignedHourOffset = hasAlignedHourOffset;
	proto.isDST                = isDaylightSavingTime;
	proto.isLocal              = isLocal;
	proto.isUtcOffset          = isUtcOffset;
	proto.isUtc                = isUtc;
	proto.isUTC                = isUtc;
	proto.zoneAbbr = getZoneAbbr;
	proto.zoneName = getZoneName;
	proto.dates  = deprecate('dates accessor is deprecated. Use date instead.', getSetDayOfMonth);
	proto.months = deprecate('months accessor is deprecated. Use month instead', getSetMonth);
	proto.years  = deprecate('years accessor is deprecated. Use year instead', getSetYear);
	proto.zone   = deprecate('moment().zone is deprecated, use moment().utcOffset instead. http://momentjs.com/guides/#/warnings/zone/', getSetZone);
	proto.isDSTShifted = deprecate('isDSTShifted is deprecated. See http://momentjs.com/guides/#/warnings/dst-shifted/ for more information', isDaylightSavingTimeShifted);

	function createUnix (input) {
		return createLocal(input * 1000);
	}

	function createInZone () {
		return createLocal.apply(null, arguments).parseZone();
	}

	function preParsePostFormat (string) {
		return string;
	}

	var proto$1 = Locale.prototype;

	proto$1.calendar        = calendar;
	proto$1.longDateFormat  = longDateFormat;
	proto$1.invalidDate     = invalidDate;
	proto$1.ordinal         = ordinal;
	proto$1.preparse        = preParsePostFormat;
	proto$1.postformat      = preParsePostFormat;
	proto$1.relativeTime    = relativeTime;
	proto$1.pastFuture      = pastFuture;
	proto$1.set             = set;

	proto$1.months            =        localeMonths;
	proto$1.monthsShort       =        localeMonthsShort;
	proto$1.monthsParse       =        localeMonthsParse;
	proto$1.monthsRegex       = monthsRegex;
	proto$1.monthsShortRegex  = monthsShortRegex;
	proto$1.week = localeWeek;
	proto$1.firstDayOfYear = localeFirstDayOfYear;
	proto$1.firstDayOfWeek = localeFirstDayOfWeek;

	proto$1.weekdays       =        localeWeekdays;
	proto$1.weekdaysMin    =        localeWeekdaysMin;
	proto$1.weekdaysShort  =        localeWeekdaysShort;
	proto$1.weekdaysParse  =        localeWeekdaysParse;

	proto$1.weekdaysRegex       =        weekdaysRegex;
	proto$1.weekdaysShortRegex  =        weekdaysShortRegex;
	proto$1.weekdaysMinRegex    =        weekdaysMinRegex;

	proto$1.isPM = localeIsPM;
	proto$1.meridiem = localeMeridiem;

	function get$1 (format, index, field, setter) {
		var locale = getLocale();
		var utc = createUTC().set(setter, index);
		return locale[field](utc, format);
	}

	function listMonthsImpl (format, index, field) {
		if (isNumber(format)) {
			index = format;
			format = undefined;
		}

		format = format || '';

		if (index != null) {
			return get$1(format, index, field, 'month');
		}

		var i;
		var out = [];
		for (i = 0; i < 12; i++) {
			out[i] = get$1(format, i, field, 'month');
		}
		return out;
	}

	// ()
	// (5)
	// (fmt, 5)
	// (fmt)
	// (true)
	// (true, 5)
	// (true, fmt, 5)
	// (true, fmt)
	function listWeekdaysImpl (localeSorted, format, index, field) {
		if (typeof localeSorted === 'boolean') {
			if (isNumber(format)) {
				index = format;
				format = undefined;
			}

			format = format || '';
		} else {
			format = localeSorted;
			index = format;
			localeSorted = false;

			if (isNumber(format)) {
				index = format;
				format = undefined;
			}

			format = format || '';
		}

		var locale = getLocale(),
			shift = localeSorted ? locale._week.dow : 0;

		if (index != null) {
			return get$1(format, (index + shift) % 7, field, 'day');
		}

		var i;
		var out = [];
		for (i = 0; i < 7; i++) {
			out[i] = get$1(format, (i + shift) % 7, field, 'day');
		}
		return out;
	}

	function listMonths (format, index) {
		return listMonthsImpl(format, index, 'months');
	}

	function listMonthsShort (format, index) {
		return listMonthsImpl(format, index, 'monthsShort');
	}

	function listWeekdays (localeSorted, format, index) {
		return listWeekdaysImpl(localeSorted, format, index, 'weekdays');
	}

	function listWeekdaysShort (localeSorted, format, index) {
		return listWeekdaysImpl(localeSorted, format, index, 'weekdaysShort');
	}

	function listWeekdaysMin (localeSorted, format, index) {
		return listWeekdaysImpl(localeSorted, format, index, 'weekdaysMin');
	}

	getSetGlobalLocale('en', {
		dayOfMonthOrdinalParse: /\d{1,2}(th|st|nd|rd)/,
		ordinal : function (number) {
			var b = number % 10,
				output = (toInt(number % 100 / 10) === 1) ? 'th' :
					(b === 1) ? 'st' :
						(b === 2) ? 'nd' :
							(b === 3) ? 'rd' : 'th';
			return number + output;
		}
	});

	// Side effect imports

	hooks.lang = deprecate('moment.lang is deprecated. Use moment.locale instead.', getSetGlobalLocale);
	hooks.langData = deprecate('moment.langData is deprecated. Use moment.localeData instead.', getLocale);

	var mathAbs = Math.abs;

	function abs () {
		var data           = this._data;

		this._milliseconds = mathAbs(this._milliseconds);
		this._days         = mathAbs(this._days);
		this._months       = mathAbs(this._months);

		data.milliseconds  = mathAbs(data.milliseconds);
		data.seconds       = mathAbs(data.seconds);
		data.minutes       = mathAbs(data.minutes);
		data.hours         = mathAbs(data.hours);
		data.months        = mathAbs(data.months);
		data.years         = mathAbs(data.years);

		return this;
	}

	function addSubtract$1 (duration, input, value, direction) {
		var other = createDuration(input, value);

		duration._milliseconds += direction * other._milliseconds;
		duration._days         += direction * other._days;
		duration._months       += direction * other._months;

		return duration._bubble();
	}

	// supports only 2.0-style add(1, 's') or add(duration)
	function add$1 (input, value) {
		return addSubtract$1(this, input, value, 1);
	}

	// supports only 2.0-style subtract(1, 's') or subtract(duration)
	function subtract$1 (input, value) {
		return addSubtract$1(this, input, value, -1);
	}

	function absCeil (number) {
		if (number < 0) {
			return Math.floor(number);
		} else {
			return Math.ceil(number);
		}
	}

	function bubble () {
		var milliseconds = this._milliseconds;
		var days         = this._days;
		var months       = this._months;
		var data         = this._data;
		var seconds, minutes, hours, years, monthsFromDays;

		// if we have a mix of positive and negative values, bubble down first
		// check: https://github.com/moment/moment/issues/2166
		if (!((milliseconds >= 0 && days >= 0 && months >= 0) ||
			(milliseconds <= 0 && days <= 0 && months <= 0))) {
			milliseconds += absCeil(monthsToDays(months) + days) * 864e5;
			days = 0;
			months = 0;
		}

		// The following code bubbles up values, see the tests for
		// examples of what that means.
		data.milliseconds = milliseconds % 1000;

		seconds           = absFloor(milliseconds / 1000);
		data.seconds      = seconds % 60;

		minutes           = absFloor(seconds / 60);
		data.minutes      = minutes % 60;

		hours             = absFloor(minutes / 60);
		data.hours        = hours % 24;

		days += absFloor(hours / 24);

		// convert days to months
		monthsFromDays = absFloor(daysToMonths(days));
		months += monthsFromDays;
		days -= absCeil(monthsToDays(monthsFromDays));

		// 12 months -> 1 year
		years = absFloor(months / 12);
		months %= 12;

		data.days   = days;
		data.months = months;
		data.years  = years;

		return this;
	}

	function daysToMonths (days) {
		// 400 years have 146097 days (taking into account leap year rules)
		// 400 years have 12 months === 4800
		return days * 4800 / 146097;
	}

	function monthsToDays (months) {
		// the reverse of daysToMonths
		return months * 146097 / 4800;
	}

	function as (units) {
		if (!this.isValid()) {
			return NaN;
		}
		var days;
		var months;
		var milliseconds = this._milliseconds;

		units = normalizeUnits(units);

		if (units === 'month' || units === 'quarter' || units === 'year') {
			days = this._days + milliseconds / 864e5;
			months = this._months + daysToMonths(days);
			switch (units) {
				case 'month':   return months;
				case 'quarter': return months / 3;
				case 'year':    return months / 12;
			}
		} else {
			// handle milliseconds separately because of floating point math errors (issue #1867)
			days = this._days + Math.round(monthsToDays(this._months));
			switch (units) {
				case 'week'   : return days / 7     + milliseconds / 6048e5;
				case 'day'    : return days         + milliseconds / 864e5;
				case 'hour'   : return days * 24    + milliseconds / 36e5;
				case 'minute' : return days * 1440  + milliseconds / 6e4;
				case 'second' : return days * 86400 + milliseconds / 1000;
				// Math.floor prevents floating point math errors here
				case 'millisecond': return Math.floor(days * 864e5) + milliseconds;
				default: throw new Error('Unknown unit ' + units);
			}
		}
	}

	// TODO: Use this.as('ms')?
	function valueOf$1 () {
		if (!this.isValid()) {
			return NaN;
		}
		return (
			this._milliseconds +
			this._days * 864e5 +
			(this._months % 12) * 2592e6 +
			toInt(this._months / 12) * 31536e6
		);
	}

	function makeAs (alias) {
		return function () {
			return this.as(alias);
		};
	}

	var asMilliseconds = makeAs('ms');
	var asSeconds      = makeAs('s');
	var asMinutes      = makeAs('m');
	var asHours        = makeAs('h');
	var asDays         = makeAs('d');
	var asWeeks        = makeAs('w');
	var asMonths       = makeAs('M');
	var asQuarters     = makeAs('Q');
	var asYears        = makeAs('y');

	function clone$1 () {
		return createDuration(this);
	}

	function get$2 (units) {
		units = normalizeUnits(units);
		return this.isValid() ? this[units + 's']() : NaN;
	}

	function makeGetter(name) {
		return function () {
			return this.isValid() ? this._data[name] : NaN;
		};
	}

	var milliseconds = makeGetter('milliseconds');
	var seconds      = makeGetter('seconds');
	var minutes      = makeGetter('minutes');
	var hours        = makeGetter('hours');
	var days         = makeGetter('days');
	var months       = makeGetter('months');
	var years        = makeGetter('years');

	function weeks () {
		return absFloor(this.days() / 7);
	}

	var round = Math.round;
	var thresholds = {
		ss: 44,         // a few seconds to seconds
		s : 45,         // seconds to minute
		m : 45,         // minutes to hour
		h : 22,         // hours to day
		d : 26,         // days to month
		M : 11          // months to year
	};

	// helper function for moment.fn.from, moment.fn.fromNow, and moment.duration.fn.humanize
	function substituteTimeAgo(string, number, withoutSuffix, isFuture, locale) {
		return locale.relativeTime(number || 1, !!withoutSuffix, string, isFuture);
	}

	function relativeTime$1 (posNegDuration, withoutSuffix, locale) {
		var duration = createDuration(posNegDuration).abs();
		var seconds  = round(duration.as('s'));
		var minutes  = round(duration.as('m'));
		var hours    = round(duration.as('h'));
		var days     = round(duration.as('d'));
		var months   = round(duration.as('M'));
		var years    = round(duration.as('y'));

		var a = seconds <= thresholds.ss && ['s', seconds]  ||
			seconds < thresholds.s   && ['ss', seconds] ||
			minutes <= 1             && ['m']           ||
			minutes < thresholds.m   && ['mm', minutes] ||
			hours   <= 1             && ['h']           ||
			hours   < thresholds.h   && ['hh', hours]   ||
			days    <= 1             && ['d']           ||
			days    < thresholds.d   && ['dd', days]    ||
			months  <= 1             && ['M']           ||
			months  < thresholds.M   && ['MM', months]  ||
			years   <= 1             && ['y']           || ['yy', years];

		a[2] = withoutSuffix;
		a[3] = +posNegDuration > 0;
		a[4] = locale;
		return substituteTimeAgo.apply(null, a);
	}

	// This function allows you to set the rounding function for relative time strings
	function getSetRelativeTimeRounding (roundingFunction) {
		if (roundingFunction === undefined) {
			return round;
		}
		if (typeof(roundingFunction) === 'function') {
			round = roundingFunction;
			return true;
		}
		return false;
	}

	// This function allows you to set a threshold for relative time strings
	function getSetRelativeTimeThreshold (threshold, limit) {
		if (thresholds[threshold] === undefined) {
			return false;
		}
		if (limit === undefined) {
			return thresholds[threshold];
		}
		thresholds[threshold] = limit;
		if (threshold === 's') {
			thresholds.ss = limit - 1;
		}
		return true;
	}

	function humanize (withSuffix) {
		if (!this.isValid()) {
			return this.localeData().invalidDate();
		}

		var locale = this.localeData();
		var output = relativeTime$1(this, !withSuffix, locale);

		if (withSuffix) {
			output = locale.pastFuture(+this, output);
		}

		return locale.postformat(output);
	}

	var abs$1 = Math.abs;

	function sign(x) {
		return ((x > 0) - (x < 0)) || +x;
	}

	function toISOString$1() {
		// for ISO strings we do not use the normal bubbling rules:
		//  * milliseconds bubble up until they become hours
		//  * days do not bubble at all
		//  * months bubble up until they become years
		// This is because there is no context-free conversion between hours and days
		// (think of clock changes)
		// and also not between days and months (28-31 days per month)
		if (!this.isValid()) {
			return this.localeData().invalidDate();
		}

		var seconds = abs$1(this._milliseconds) / 1000;
		var days         = abs$1(this._days);
		var months       = abs$1(this._months);
		var minutes, hours, years;

		// 3600 seconds -> 60 minutes -> 1 hour
		minutes           = absFloor(seconds / 60);
		hours             = absFloor(minutes / 60);
		seconds %= 60;
		minutes %= 60;

		// 12 months -> 1 year
		years  = absFloor(months / 12);
		months %= 12;


		// inspired by https://github.com/dordille/moment-isoduration/blob/master/moment.isoduration.js
		var Y = years;
		var M = months;
		var D = days;
		var h = hours;
		var m = minutes;
		var s = seconds ? seconds.toFixed(3).replace(/\.?0+$/, '') : '';
		var total = this.asSeconds();

		if (!total) {
			// this is the same as C#'s (Noda) and python (isodate)...
			// but not other JS (goog.date)
			return 'P0D';
		}

		var totalSign = total < 0 ? '-' : '';
		var ymSign = sign(this._months) !== sign(total) ? '-' : '';
		var daysSign = sign(this._days) !== sign(total) ? '-' : '';
		var hmsSign = sign(this._milliseconds) !== sign(total) ? '-' : '';

		return totalSign + 'P' +
			(Y ? ymSign + Y + 'Y' : '') +
			(M ? ymSign + M + 'M' : '') +
			(D ? daysSign + D + 'D' : '') +
			((h || m || s) ? 'T' : '') +
			(h ? hmsSign + h + 'H' : '') +
			(m ? hmsSign + m + 'M' : '') +
			(s ? hmsSign + s + 'S' : '');
	}

	var proto$2 = Duration.prototype;

	proto$2.isValid        = isValid$1;
	proto$2.abs            = abs;
	proto$2.add            = add$1;
	proto$2.subtract       = subtract$1;
	proto$2.as             = as;
	proto$2.asMilliseconds = asMilliseconds;
	proto$2.asSeconds      = asSeconds;
	proto$2.asMinutes      = asMinutes;
	proto$2.asHours        = asHours;
	proto$2.asDays         = asDays;
	proto$2.asWeeks        = asWeeks;
	proto$2.asMonths       = asMonths;
	proto$2.asQuarters     = asQuarters;
	proto$2.asYears        = asYears;
	proto$2.valueOf        = valueOf$1;
	proto$2._bubble        = bubble;
	proto$2.clone          = clone$1;
	proto$2.get            = get$2;
	proto$2.milliseconds   = milliseconds;
	proto$2.seconds        = seconds;
	proto$2.minutes        = minutes;
	proto$2.hours          = hours;
	proto$2.days           = days;
	proto$2.weeks          = weeks;
	proto$2.months         = months;
	proto$2.years          = years;
	proto$2.humanize       = humanize;
	proto$2.toISOString    = toISOString$1;
	proto$2.toString       = toISOString$1;
	proto$2.toJSON         = toISOString$1;
	proto$2.locale         = locale;
	proto$2.localeData     = localeData;

	proto$2.toIsoString = deprecate('toIsoString() is deprecated. Please use toISOString() instead (notice the capitals)', toISOString$1);
	proto$2.lang = lang;

	// Side effect imports

	// FORMATTING

	addFormatToken('X', 0, 0, 'unix');
	addFormatToken('x', 0, 0, 'valueOf');

	// PARSING

	addRegexToken('x', matchSigned);
	addRegexToken('X', matchTimestamp);
	addParseToken('X', function (input, array, config) {
		config._d = new Date(parseFloat(input, 10) * 1000);
	});
	addParseToken('x', function (input, array, config) {
		config._d = new Date(toInt(input));
	});

	// Side effect imports

	//! moment.js

	hooks.version = '2.24.0';

	setHookCallback(createLocal);

	hooks.fn                    = proto;
	hooks.min                   = min;
	hooks.max                   = max;
	hooks.now                   = now;
	hooks.utc                   = createUTC;
	hooks.unix                  = createUnix;
	hooks.months                = listMonths;
	hooks.isDate                = isDate;
	hooks.locale                = getSetGlobalLocale;
	hooks.invalid               = createInvalid;
	hooks.duration              = createDuration;
	hooks.isMoment              = isMoment;
	hooks.weekdays              = listWeekdays;
	hooks.parseZone             = createInZone;
	hooks.localeData            = getLocale;
	hooks.isDuration            = isDuration;
	hooks.monthsShort           = listMonthsShort;
	hooks.weekdaysMin           = listWeekdaysMin;
	hooks.defineLocale          = defineLocale;
	hooks.updateLocale          = updateLocale;
	hooks.locales               = listLocales;
	hooks.weekdaysShort         = listWeekdaysShort;
	hooks.normalizeUnits        = normalizeUnits;
	hooks.relativeTimeRounding  = getSetRelativeTimeRounding;
	hooks.relativeTimeThreshold = getSetRelativeTimeThreshold;
	hooks.calendarFormat        = getCalendarFormat;
	hooks.prototype             = proto;

	// currently HTML5 input type only supports 24-hour formats
	hooks.HTML5_FMT = {
		DATETIME_LOCAL: 'YYYY-MM-DDTHH:mm',             // <input type="datetime-local" />
		DATETIME_LOCAL_SECONDS: 'YYYY-MM-DDTHH:mm:ss',  // <input type="datetime-local" step="1" />
		DATETIME_LOCAL_MS: 'YYYY-MM-DDTHH:mm:ss.SSS',   // <input type="datetime-local" step="0.001" />
		DATE: 'YYYY-MM-DD',                             // <input type="date" />
		TIME: 'HH:mm',                                  // <input type="time" />
		TIME_SECONDS: 'HH:mm:ss',                       // <input type="time" step="1" />
		TIME_MS: 'HH:mm:ss.SSS',                        // <input type="time" step="0.001" />
		WEEK: 'GGGG-[W]WW',                             // <input type="week" />
		MONTH: 'YYYY-MM'                                // <input type="month" />
	};

	//! moment.js locale configuration

	hooks.defineLocale('af', {
		months : 'Januarie_Februarie_Maart_April_Mei_Junie_Julie_Augustus_September_Oktober_November_Desember'.split('_'),
		monthsShort : 'Jan_Feb_Mrt_Apr_Mei_Jun_Jul_Aug_Sep_Okt_Nov_Des'.split('_'),
		weekdays : 'Sondag_Maandag_Dinsdag_Woensdag_Donderdag_Vrydag_Saterdag'.split('_'),
		weekdaysShort : 'Son_Maa_Din_Woe_Don_Vry_Sat'.split('_'),
		weekdaysMin : 'So_Ma_Di_Wo_Do_Vr_Sa'.split('_'),
		meridiemParse: /vm|nm/i,
		isPM : function (input) {
			return /^nm$/i.test(input);
		},
		meridiem : function (hours, minutes, isLower) {
			if (hours < 12) {
				return isLower ? 'vm' : 'VM';
			} else {
				return isLower ? 'nm' : 'NM';
			}
		},
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd, D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay : '[Vandag om] LT',
			nextDay : '[Môre om] LT',
			nextWeek : 'dddd [om] LT',
			lastDay : '[Gister om] LT',
			lastWeek : '[Laas] dddd [om] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'oor %s',
			past : '%s gelede',
			s : '\'n paar sekondes',
			ss : '%d sekondes',
			m : '\'n minuut',
			mm : '%d minute',
			h : '\'n uur',
			hh : '%d ure',
			d : '\'n dag',
			dd : '%d dae',
			M : '\'n maand',
			MM : '%d maande',
			y : '\'n jaar',
			yy : '%d jaar'
		},
		dayOfMonthOrdinalParse: /\d{1,2}(ste|de)/,
		ordinal : function (number) {
			return number + ((number === 1 || number === 8 || number >= 20) ? 'ste' : 'de'); // Thanks to Joris Röling : https://github.com/jjupiter
		},
		week : {
			dow : 1, // Maandag is die eerste dag van die week.
			doy : 4  // Die week wat die 4de Januarie bevat is die eerste week van die jaar.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('ar-dz', {
		months : 'جانفي_فيفري_مارس_أفريل_ماي_جوان_جويلية_أوت_سبتمبر_أكتوبر_نوفمبر_ديسمبر'.split('_'),
		monthsShort : 'جانفي_فيفري_مارس_أفريل_ماي_جوان_جويلية_أوت_سبتمبر_أكتوبر_نوفمبر_ديسمبر'.split('_'),
		weekdays : 'الأحد_الإثنين_الثلاثاء_الأربعاء_الخميس_الجمعة_السبت'.split('_'),
		weekdaysShort : 'احد_اثنين_ثلاثاء_اربعاء_خميس_جمعة_سبت'.split('_'),
		weekdaysMin : 'أح_إث_ثلا_أر_خم_جم_سب'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay: '[اليوم على الساعة] LT',
			nextDay: '[غدا على الساعة] LT',
			nextWeek: 'dddd [على الساعة] LT',
			lastDay: '[أمس على الساعة] LT',
			lastWeek: 'dddd [على الساعة] LT',
			sameElse: 'L'
		},
		relativeTime : {
			future : 'في %s',
			past : 'منذ %s',
			s : 'ثوان',
			ss : '%d ثانية',
			m : 'دقيقة',
			mm : '%d دقائق',
			h : 'ساعة',
			hh : '%d ساعات',
			d : 'يوم',
			dd : '%d أيام',
			M : 'شهر',
			MM : '%d أشهر',
			y : 'سنة',
			yy : '%d سنوات'
		},
		week : {
			dow : 0, // Sunday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('ar-kw', {
		months : 'يناير_فبراير_مارس_أبريل_ماي_يونيو_يوليوز_غشت_شتنبر_أكتوبر_نونبر_دجنبر'.split('_'),
		monthsShort : 'يناير_فبراير_مارس_أبريل_ماي_يونيو_يوليوز_غشت_شتنبر_أكتوبر_نونبر_دجنبر'.split('_'),
		weekdays : 'الأحد_الإتنين_الثلاثاء_الأربعاء_الخميس_الجمعة_السبت'.split('_'),
		weekdaysShort : 'احد_اتنين_ثلاثاء_اربعاء_خميس_جمعة_سبت'.split('_'),
		weekdaysMin : 'ح_ن_ث_ر_خ_ج_س'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay: '[اليوم على الساعة] LT',
			nextDay: '[غدا على الساعة] LT',
			nextWeek: 'dddd [على الساعة] LT',
			lastDay: '[أمس على الساعة] LT',
			lastWeek: 'dddd [على الساعة] LT',
			sameElse: 'L'
		},
		relativeTime : {
			future : 'في %s',
			past : 'منذ %s',
			s : 'ثوان',
			ss : '%d ثانية',
			m : 'دقيقة',
			mm : '%d دقائق',
			h : 'ساعة',
			hh : '%d ساعات',
			d : 'يوم',
			dd : '%d أيام',
			M : 'شهر',
			MM : '%d أشهر',
			y : 'سنة',
			yy : '%d سنوات'
		},
		week : {
			dow : 0, // Sunday is the first day of the week.
			doy : 12  // The week that contains Jan 12th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var symbolMap = {
		'1': '1',
		'2': '2',
		'3': '3',
		'4': '4',
		'5': '5',
		'6': '6',
		'7': '7',
		'8': '8',
		'9': '9',
		'0': '0'
	}, pluralForm = function (n) {
		return n === 0 ? 0 : n === 1 ? 1 : n === 2 ? 2 : n % 100 >= 3 && n % 100 <= 10 ? 3 : n % 100 >= 11 ? 4 : 5;
	}, plurals = {
		s : ['أقل من ثانية', 'ثانية واحدة', ['ثانيتان', 'ثانيتين'], '%d ثوان', '%d ثانية', '%d ثانية'],
		m : ['أقل من دقيقة', 'دقيقة واحدة', ['دقيقتان', 'دقيقتين'], '%d دقائق', '%d دقيقة', '%d دقيقة'],
		h : ['أقل من ساعة', 'ساعة واحدة', ['ساعتان', 'ساعتين'], '%d ساعات', '%d ساعة', '%d ساعة'],
		d : ['أقل من يوم', 'يوم واحد', ['يومان', 'يومين'], '%d أيام', '%d يومًا', '%d يوم'],
		M : ['أقل من شهر', 'شهر واحد', ['شهران', 'شهرين'], '%d أشهر', '%d شهرا', '%d شهر'],
		y : ['أقل من عام', 'عام واحد', ['عامان', 'عامين'], '%d أعوام', '%d عامًا', '%d عام']
	}, pluralize = function (u) {
		return function (number, withoutSuffix, string, isFuture) {
			var f = pluralForm(number),
				str = plurals[u][pluralForm(number)];
			if (f === 2) {
				str = str[withoutSuffix ? 0 : 1];
			}
			return str.replace(/%d/i, number);
		};
	}, months$1 = [
		'يناير',
		'فبراير',
		'مارس',
		'أبريل',
		'مايو',
		'يونيو',
		'يوليو',
		'أغسطس',
		'سبتمبر',
		'أكتوبر',
		'نوفمبر',
		'ديسمبر'
	];

	hooks.defineLocale('ar-ly', {
		months : months$1,
		monthsShort : months$1,
		weekdays : 'الأحد_الإثنين_الثلاثاء_الأربعاء_الخميس_الجمعة_السبت'.split('_'),
		weekdaysShort : 'أحد_إثنين_ثلاثاء_أربعاء_خميس_جمعة_سبت'.split('_'),
		weekdaysMin : 'ح_ن_ث_ر_خ_ج_س'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'D/\u200FM/\u200FYYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd D MMMM YYYY HH:mm'
		},
		meridiemParse: /ص|م/,
		isPM : function (input) {
			return 'م' === input;
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 12) {
				return 'ص';
			} else {
				return 'م';
			}
		},
		calendar : {
			sameDay: '[اليوم عند الساعة] LT',
			nextDay: '[غدًا عند الساعة] LT',
			nextWeek: 'dddd [عند الساعة] LT',
			lastDay: '[أمس عند الساعة] LT',
			lastWeek: 'dddd [عند الساعة] LT',
			sameElse: 'L'
		},
		relativeTime : {
			future : 'بعد %s',
			past : 'منذ %s',
			s : pluralize('s'),
			ss : pluralize('s'),
			m : pluralize('m'),
			mm : pluralize('m'),
			h : pluralize('h'),
			hh : pluralize('h'),
			d : pluralize('d'),
			dd : pluralize('d'),
			M : pluralize('M'),
			MM : pluralize('M'),
			y : pluralize('y'),
			yy : pluralize('y')
		},
		preparse: function (string) {
			return string.replace(/،/g, ',');
		},
		postformat: function (string) {
			return string.replace(/\d/g, function (match) {
				return symbolMap[match];
			}).replace(/,/g, '،');
		},
		week : {
			dow : 6, // Saturday is the first day of the week.
			doy : 12  // The week that contains Jan 12th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('ar-ma', {
		months : 'يناير_فبراير_مارس_أبريل_ماي_يونيو_يوليوز_غشت_شتنبر_أكتوبر_نونبر_دجنبر'.split('_'),
		monthsShort : 'يناير_فبراير_مارس_أبريل_ماي_يونيو_يوليوز_غشت_شتنبر_أكتوبر_نونبر_دجنبر'.split('_'),
		weekdays : 'الأحد_الإتنين_الثلاثاء_الأربعاء_الخميس_الجمعة_السبت'.split('_'),
		weekdaysShort : 'احد_اتنين_ثلاثاء_اربعاء_خميس_جمعة_سبت'.split('_'),
		weekdaysMin : 'ح_ن_ث_ر_خ_ج_س'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay: '[اليوم على الساعة] LT',
			nextDay: '[غدا على الساعة] LT',
			nextWeek: 'dddd [على الساعة] LT',
			lastDay: '[أمس على الساعة] LT',
			lastWeek: 'dddd [على الساعة] LT',
			sameElse: 'L'
		},
		relativeTime : {
			future : 'في %s',
			past : 'منذ %s',
			s : 'ثوان',
			ss : '%d ثانية',
			m : 'دقيقة',
			mm : '%d دقائق',
			h : 'ساعة',
			hh : '%d ساعات',
			d : 'يوم',
			dd : '%d أيام',
			M : 'شهر',
			MM : '%d أشهر',
			y : 'سنة',
			yy : '%d سنوات'
		},
		week : {
			dow : 6, // Saturday is the first day of the week.
			doy : 12  // The week that contains Jan 12th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var symbolMap$1 = {
		'1': '١',
		'2': '٢',
		'3': '٣',
		'4': '٤',
		'5': '٥',
		'6': '٦',
		'7': '٧',
		'8': '٨',
		'9': '٩',
		'0': '٠'
	}, numberMap = {
		'١': '1',
		'٢': '2',
		'٣': '3',
		'٤': '4',
		'٥': '5',
		'٦': '6',
		'٧': '7',
		'٨': '8',
		'٩': '9',
		'٠': '0'
	};

	hooks.defineLocale('ar-sa', {
		months : 'يناير_فبراير_مارس_أبريل_مايو_يونيو_يوليو_أغسطس_سبتمبر_أكتوبر_نوفمبر_ديسمبر'.split('_'),
		monthsShort : 'يناير_فبراير_مارس_أبريل_مايو_يونيو_يوليو_أغسطس_سبتمبر_أكتوبر_نوفمبر_ديسمبر'.split('_'),
		weekdays : 'الأحد_الإثنين_الثلاثاء_الأربعاء_الخميس_الجمعة_السبت'.split('_'),
		weekdaysShort : 'أحد_إثنين_ثلاثاء_أربعاء_خميس_جمعة_سبت'.split('_'),
		weekdaysMin : 'ح_ن_ث_ر_خ_ج_س'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd D MMMM YYYY HH:mm'
		},
		meridiemParse: /ص|م/,
		isPM : function (input) {
			return 'م' === input;
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 12) {
				return 'ص';
			} else {
				return 'م';
			}
		},
		calendar : {
			sameDay: '[اليوم على الساعة] LT',
			nextDay: '[غدا على الساعة] LT',
			nextWeek: 'dddd [على الساعة] LT',
			lastDay: '[أمس على الساعة] LT',
			lastWeek: 'dddd [على الساعة] LT',
			sameElse: 'L'
		},
		relativeTime : {
			future : 'في %s',
			past : 'منذ %s',
			s : 'ثوان',
			ss : '%d ثانية',
			m : 'دقيقة',
			mm : '%d دقائق',
			h : 'ساعة',
			hh : '%d ساعات',
			d : 'يوم',
			dd : '%d أيام',
			M : 'شهر',
			MM : '%d أشهر',
			y : 'سنة',
			yy : '%d سنوات'
		},
		preparse: function (string) {
			return string.replace(/[١٢٣٤٥٦٧٨٩٠]/g, function (match) {
				return numberMap[match];
			}).replace(/،/g, ',');
		},
		postformat: function (string) {
			return string.replace(/\d/g, function (match) {
				return symbolMap$1[match];
			}).replace(/,/g, '،');
		},
		week : {
			dow : 0, // Sunday is the first day of the week.
			doy : 6  // The week that contains Jan 6th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('ar-tn', {
		months: 'جانفي_فيفري_مارس_أفريل_ماي_جوان_جويلية_أوت_سبتمبر_أكتوبر_نوفمبر_ديسمبر'.split('_'),
		monthsShort: 'جانفي_فيفري_مارس_أفريل_ماي_جوان_جويلية_أوت_سبتمبر_أكتوبر_نوفمبر_ديسمبر'.split('_'),
		weekdays: 'الأحد_الإثنين_الثلاثاء_الأربعاء_الخميس_الجمعة_السبت'.split('_'),
		weekdaysShort: 'أحد_إثنين_ثلاثاء_أربعاء_خميس_جمعة_سبت'.split('_'),
		weekdaysMin: 'ح_ن_ث_ر_خ_ج_س'.split('_'),
		weekdaysParseExact : true,
		longDateFormat: {
			LT: 'HH:mm',
			LTS: 'HH:mm:ss',
			L: 'DD/MM/YYYY',
			LL: 'D MMMM YYYY',
			LLL: 'D MMMM YYYY HH:mm',
			LLLL: 'dddd D MMMM YYYY HH:mm'
		},
		calendar: {
			sameDay: '[اليوم على الساعة] LT',
			nextDay: '[غدا على الساعة] LT',
			nextWeek: 'dddd [على الساعة] LT',
			lastDay: '[أمس على الساعة] LT',
			lastWeek: 'dddd [على الساعة] LT',
			sameElse: 'L'
		},
		relativeTime: {
			future: 'في %s',
			past: 'منذ %s',
			s: 'ثوان',
			ss : '%d ثانية',
			m: 'دقيقة',
			mm: '%d دقائق',
			h: 'ساعة',
			hh: '%d ساعات',
			d: 'يوم',
			dd: '%d أيام',
			M: 'شهر',
			MM: '%d أشهر',
			y: 'سنة',
			yy: '%d سنوات'
		},
		week: {
			dow: 1, // Monday is the first day of the week.
			doy: 4 // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var symbolMap$2 = {
		'1': '١',
		'2': '٢',
		'3': '٣',
		'4': '٤',
		'5': '٥',
		'6': '٦',
		'7': '٧',
		'8': '٨',
		'9': '٩',
		'0': '٠'
	}, numberMap$1 = {
		'١': '1',
		'٢': '2',
		'٣': '3',
		'٤': '4',
		'٥': '5',
		'٦': '6',
		'٧': '7',
		'٨': '8',
		'٩': '9',
		'٠': '0'
	}, pluralForm$1 = function (n) {
		return n === 0 ? 0 : n === 1 ? 1 : n === 2 ? 2 : n % 100 >= 3 && n % 100 <= 10 ? 3 : n % 100 >= 11 ? 4 : 5;
	}, plurals$1 = {
		s : ['أقل من ثانية', 'ثانية واحدة', ['ثانيتان', 'ثانيتين'], '%d ثوان', '%d ثانية', '%d ثانية'],
		m : ['أقل من دقيقة', 'دقيقة واحدة', ['دقيقتان', 'دقيقتين'], '%d دقائق', '%d دقيقة', '%d دقيقة'],
		h : ['أقل من ساعة', 'ساعة واحدة', ['ساعتان', 'ساعتين'], '%d ساعات', '%d ساعة', '%d ساعة'],
		d : ['أقل من يوم', 'يوم واحد', ['يومان', 'يومين'], '%d أيام', '%d يومًا', '%d يوم'],
		M : ['أقل من شهر', 'شهر واحد', ['شهران', 'شهرين'], '%d أشهر', '%d شهرا', '%d شهر'],
		y : ['أقل من عام', 'عام واحد', ['عامان', 'عامين'], '%d أعوام', '%d عامًا', '%d عام']
	}, pluralize$1 = function (u) {
		return function (number, withoutSuffix, string, isFuture) {
			var f = pluralForm$1(number),
				str = plurals$1[u][pluralForm$1(number)];
			if (f === 2) {
				str = str[withoutSuffix ? 0 : 1];
			}
			return str.replace(/%d/i, number);
		};
	}, months$2 = [
		'يناير',
		'فبراير',
		'مارس',
		'أبريل',
		'مايو',
		'يونيو',
		'يوليو',
		'أغسطس',
		'سبتمبر',
		'أكتوبر',
		'نوفمبر',
		'ديسمبر'
	];

	hooks.defineLocale('ar', {
		months : months$2,
		monthsShort : months$2,
		weekdays : 'الأحد_الإثنين_الثلاثاء_الأربعاء_الخميس_الجمعة_السبت'.split('_'),
		weekdaysShort : 'أحد_إثنين_ثلاثاء_أربعاء_خميس_جمعة_سبت'.split('_'),
		weekdaysMin : 'ح_ن_ث_ر_خ_ج_س'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'D/\u200FM/\u200FYYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd D MMMM YYYY HH:mm'
		},
		meridiemParse: /ص|م/,
		isPM : function (input) {
			return 'م' === input;
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 12) {
				return 'ص';
			} else {
				return 'م';
			}
		},
		calendar : {
			sameDay: '[اليوم عند الساعة] LT',
			nextDay: '[غدًا عند الساعة] LT',
			nextWeek: 'dddd [عند الساعة] LT',
			lastDay: '[أمس عند الساعة] LT',
			lastWeek: 'dddd [عند الساعة] LT',
			sameElse: 'L'
		},
		relativeTime : {
			future : 'بعد %s',
			past : 'منذ %s',
			s : pluralize$1('s'),
			ss : pluralize$1('s'),
			m : pluralize$1('m'),
			mm : pluralize$1('m'),
			h : pluralize$1('h'),
			hh : pluralize$1('h'),
			d : pluralize$1('d'),
			dd : pluralize$1('d'),
			M : pluralize$1('M'),
			MM : pluralize$1('M'),
			y : pluralize$1('y'),
			yy : pluralize$1('y')
		},
		preparse: function (string) {
			return string.replace(/[١٢٣٤٥٦٧٨٩٠]/g, function (match) {
				return numberMap$1[match];
			}).replace(/،/g, ',');
		},
		postformat: function (string) {
			return string.replace(/\d/g, function (match) {
				return symbolMap$2[match];
			}).replace(/,/g, '،');
		},
		week : {
			dow : 6, // Saturday is the first day of the week.
			doy : 12  // The week that contains Jan 12th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var suffixes = {
		1: '-inci',
		5: '-inci',
		8: '-inci',
		70: '-inci',
		80: '-inci',
		2: '-nci',
		7: '-nci',
		20: '-nci',
		50: '-nci',
		3: '-üncü',
		4: '-üncü',
		100: '-üncü',
		6: '-ncı',
		9: '-uncu',
		10: '-uncu',
		30: '-uncu',
		60: '-ıncı',
		90: '-ıncı'
	};

	hooks.defineLocale('az', {
		months : 'yanvar_fevral_mart_aprel_may_iyun_iyul_avqust_sentyabr_oktyabr_noyabr_dekabr'.split('_'),
		monthsShort : 'yan_fev_mar_apr_may_iyn_iyl_avq_sen_okt_noy_dek'.split('_'),
		weekdays : 'Bazar_Bazar ertəsi_Çərşənbə axşamı_Çərşənbə_Cümə axşamı_Cümə_Şənbə'.split('_'),
		weekdaysShort : 'Baz_BzE_ÇAx_Çər_CAx_Cüm_Şən'.split('_'),
		weekdaysMin : 'Bz_BE_ÇA_Çə_CA_Cü_Şə'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd, D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay : '[bugün saat] LT',
			nextDay : '[sabah saat] LT',
			nextWeek : '[gələn həftə] dddd [saat] LT',
			lastDay : '[dünən] LT',
			lastWeek : '[keçən həftə] dddd [saat] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : '%s sonra',
			past : '%s əvvəl',
			s : 'birneçə saniyə',
			ss : '%d saniyə',
			m : 'bir dəqiqə',
			mm : '%d dəqiqə',
			h : 'bir saat',
			hh : '%d saat',
			d : 'bir gün',
			dd : '%d gün',
			M : 'bir ay',
			MM : '%d ay',
			y : 'bir il',
			yy : '%d il'
		},
		meridiemParse: /gecə|səhər|gündüz|axşam/,
		isPM : function (input) {
			return /^(gündüz|axşam)$/.test(input);
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 4) {
				return 'gecə';
			} else if (hour < 12) {
				return 'səhər';
			} else if (hour < 17) {
				return 'gündüz';
			} else {
				return 'axşam';
			}
		},
		dayOfMonthOrdinalParse: /\d{1,2}-(ıncı|inci|nci|üncü|ncı|uncu)/,
		ordinal : function (number) {
			if (number === 0) {  // special case for zero
				return number + '-ıncı';
			}
			var a = number % 10,
				b = number % 100 - a,
				c = number >= 100 ? 100 : null;
			return number + (suffixes[a] || suffixes[b] || suffixes[c]);
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	function plural(word, num) {
		var forms = word.split('_');
		return num % 10 === 1 && num % 100 !== 11 ? forms[0] : (num % 10 >= 2 && num % 10 <= 4 && (num % 100 < 10 || num % 100 >= 20) ? forms[1] : forms[2]);
	}
	function relativeTimeWithPlural(number, withoutSuffix, key) {
		var format = {
			'ss': withoutSuffix ? 'секунда_секунды_секунд' : 'секунду_секунды_секунд',
			'mm': withoutSuffix ? 'хвіліна_хвіліны_хвілін' : 'хвіліну_хвіліны_хвілін',
			'hh': withoutSuffix ? 'гадзіна_гадзіны_гадзін' : 'гадзіну_гадзіны_гадзін',
			'dd': 'дзень_дні_дзён',
			'MM': 'месяц_месяцы_месяцаў',
			'yy': 'год_гады_гадоў'
		};
		if (key === 'm') {
			return withoutSuffix ? 'хвіліна' : 'хвіліну';
		}
		else if (key === 'h') {
			return withoutSuffix ? 'гадзіна' : 'гадзіну';
		}
		else {
			return number + ' ' + plural(format[key], +number);
		}
	}

	hooks.defineLocale('be', {
		months : {
			format: 'студзеня_лютага_сакавіка_красавіка_траўня_чэрвеня_ліпеня_жніўня_верасня_кастрычніка_лістапада_снежня'.split('_'),
			standalone: 'студзень_люты_сакавік_красавік_травень_чэрвень_ліпень_жнівень_верасень_кастрычнік_лістапад_снежань'.split('_')
		},
		monthsShort : 'студ_лют_сак_крас_трав_чэрв_ліп_жнів_вер_каст_ліст_снеж'.split('_'),
		weekdays : {
			format: 'нядзелю_панядзелак_аўторак_сераду_чацвер_пятніцу_суботу'.split('_'),
			standalone: 'нядзеля_панядзелак_аўторак_серада_чацвер_пятніца_субота'.split('_'),
			isFormat: /\[ ?[Ууў] ?(?:мінулую|наступную)? ?\] ?dddd/
		},
		weekdaysShort : 'нд_пн_ат_ср_чц_пт_сб'.split('_'),
		weekdaysMin : 'нд_пн_ат_ср_чц_пт_сб'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D MMMM YYYY г.',
			LLL : 'D MMMM YYYY г., HH:mm',
			LLLL : 'dddd, D MMMM YYYY г., HH:mm'
		},
		calendar : {
			sameDay: '[Сёння ў] LT',
			nextDay: '[Заўтра ў] LT',
			lastDay: '[Учора ў] LT',
			nextWeek: function () {
				return '[У] dddd [ў] LT';
			},
			lastWeek: function () {
				switch (this.day()) {
					case 0:
					case 3:
					case 5:
					case 6:
						return '[У мінулую] dddd [ў] LT';
					case 1:
					case 2:
					case 4:
						return '[У мінулы] dddd [ў] LT';
				}
			},
			sameElse: 'L'
		},
		relativeTime : {
			future : 'праз %s',
			past : '%s таму',
			s : 'некалькі секунд',
			m : relativeTimeWithPlural,
			mm : relativeTimeWithPlural,
			h : relativeTimeWithPlural,
			hh : relativeTimeWithPlural,
			d : 'дзень',
			dd : relativeTimeWithPlural,
			M : 'месяц',
			MM : relativeTimeWithPlural,
			y : 'год',
			yy : relativeTimeWithPlural
		},
		meridiemParse: /ночы|раніцы|дня|вечара/,
		isPM : function (input) {
			return /^(дня|вечара)$/.test(input);
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 4) {
				return 'ночы';
			} else if (hour < 12) {
				return 'раніцы';
			} else if (hour < 17) {
				return 'дня';
			} else {
				return 'вечара';
			}
		},
		dayOfMonthOrdinalParse: /\d{1,2}-(і|ы|га)/,
		ordinal: function (number, period) {
			switch (period) {
				case 'M':
				case 'd':
				case 'DDD':
				case 'w':
				case 'W':
					return (number % 10 === 2 || number % 10 === 3) && (number % 100 !== 12 && number % 100 !== 13) ? number + '-і' : number + '-ы';
				case 'D':
					return number + '-га';
				default:
					return number;
			}
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('bg', {
		months : 'януари_февруари_март_април_май_юни_юли_август_септември_октомври_ноември_декември'.split('_'),
		monthsShort : 'янр_фев_мар_апр_май_юни_юли_авг_сеп_окт_ное_дек'.split('_'),
		weekdays : 'неделя_понеделник_вторник_сряда_четвъртък_петък_събота'.split('_'),
		weekdaysShort : 'нед_пон_вто_сря_чет_пет_съб'.split('_'),
		weekdaysMin : 'нд_пн_вт_ср_чт_пт_сб'.split('_'),
		longDateFormat : {
			LT : 'H:mm',
			LTS : 'H:mm:ss',
			L : 'D.MM.YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY H:mm',
			LLLL : 'dddd, D MMMM YYYY H:mm'
		},
		calendar : {
			sameDay : '[Днес в] LT',
			nextDay : '[Утре в] LT',
			nextWeek : 'dddd [в] LT',
			lastDay : '[Вчера в] LT',
			lastWeek : function () {
				switch (this.day()) {
					case 0:
					case 3:
					case 6:
						return '[В изминалата] dddd [в] LT';
					case 1:
					case 2:
					case 4:
					case 5:
						return '[В изминалия] dddd [в] LT';
				}
			},
			sameElse : 'L'
		},
		relativeTime : {
			future : 'след %s',
			past : 'преди %s',
			s : 'няколко секунди',
			ss : '%d секунди',
			m : 'минута',
			mm : '%d минути',
			h : 'час',
			hh : '%d часа',
			d : 'ден',
			dd : '%d дни',
			M : 'месец',
			MM : '%d месеца',
			y : 'година',
			yy : '%d години'
		},
		dayOfMonthOrdinalParse: /\d{1,2}-(ев|ен|ти|ви|ри|ми)/,
		ordinal : function (number) {
			var lastDigit = number % 10,
				last2Digits = number % 100;
			if (number === 0) {
				return number + '-ев';
			} else if (last2Digits === 0) {
				return number + '-ен';
			} else if (last2Digits > 10 && last2Digits < 20) {
				return number + '-ти';
			} else if (lastDigit === 1) {
				return number + '-ви';
			} else if (lastDigit === 2) {
				return number + '-ри';
			} else if (lastDigit === 7 || lastDigit === 8) {
				return number + '-ми';
			} else {
				return number + '-ти';
			}
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('bm', {
		months : 'Zanwuyekalo_Fewuruyekalo_Marisikalo_Awirilikalo_Mɛkalo_Zuwɛnkalo_Zuluyekalo_Utikalo_Sɛtanburukalo_ɔkutɔburukalo_Nowanburukalo_Desanburukalo'.split('_'),
		monthsShort : 'Zan_Few_Mar_Awi_Mɛ_Zuw_Zul_Uti_Sɛt_ɔku_Now_Des'.split('_'),
		weekdays : 'Kari_Ntɛnɛn_Tarata_Araba_Alamisa_Juma_Sibiri'.split('_'),
		weekdaysShort : 'Kar_Ntɛ_Tar_Ara_Ala_Jum_Sib'.split('_'),
		weekdaysMin : 'Ka_Nt_Ta_Ar_Al_Ju_Si'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'MMMM [tile] D [san] YYYY',
			LLL : 'MMMM [tile] D [san] YYYY [lɛrɛ] HH:mm',
			LLLL : 'dddd MMMM [tile] D [san] YYYY [lɛrɛ] HH:mm'
		},
		calendar : {
			sameDay : '[Bi lɛrɛ] LT',
			nextDay : '[Sini lɛrɛ] LT',
			nextWeek : 'dddd [don lɛrɛ] LT',
			lastDay : '[Kunu lɛrɛ] LT',
			lastWeek : 'dddd [tɛmɛnen lɛrɛ] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : '%s kɔnɔ',
			past : 'a bɛ %s bɔ',
			s : 'sanga dama dama',
			ss : 'sekondi %d',
			m : 'miniti kelen',
			mm : 'miniti %d',
			h : 'lɛrɛ kelen',
			hh : 'lɛrɛ %d',
			d : 'tile kelen',
			dd : 'tile %d',
			M : 'kalo kelen',
			MM : 'kalo %d',
			y : 'san kelen',
			yy : 'san %d'
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var symbolMap$3 = {
			'1': '১',
			'2': '২',
			'3': '৩',
			'4': '৪',
			'5': '৫',
			'6': '৬',
			'7': '৭',
			'8': '৮',
			'9': '৯',
			'0': '০'
		},
		numberMap$2 = {
			'১': '1',
			'২': '2',
			'৩': '3',
			'৪': '4',
			'৫': '5',
			'৬': '6',
			'৭': '7',
			'৮': '8',
			'৯': '9',
			'০': '0'
		};

	hooks.defineLocale('bn', {
		months : 'জানুয়ারী_ফেব্রুয়ারি_মার্চ_এপ্রিল_মে_জুন_জুলাই_আগস্ট_সেপ্টেম্বর_অক্টোবর_নভেম্বর_ডিসেম্বর'.split('_'),
		monthsShort : 'জানু_ফেব_মার্চ_এপ্র_মে_জুন_জুল_আগ_সেপ্ট_অক্টো_নভে_ডিসে'.split('_'),
		weekdays : 'রবিবার_সোমবার_মঙ্গলবার_বুধবার_বৃহস্পতিবার_শুক্রবার_শনিবার'.split('_'),
		weekdaysShort : 'রবি_সোম_মঙ্গল_বুধ_বৃহস্পতি_শুক্র_শনি'.split('_'),
		weekdaysMin : 'রবি_সোম_মঙ্গ_বুধ_বৃহঃ_শুক্র_শনি'.split('_'),
		longDateFormat : {
			LT : 'A h:mm সময়',
			LTS : 'A h:mm:ss সময়',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY, A h:mm সময়',
			LLLL : 'dddd, D MMMM YYYY, A h:mm সময়'
		},
		calendar : {
			sameDay : '[আজ] LT',
			nextDay : '[আগামীকাল] LT',
			nextWeek : 'dddd, LT',
			lastDay : '[গতকাল] LT',
			lastWeek : '[গত] dddd, LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : '%s পরে',
			past : '%s আগে',
			s : 'কয়েক সেকেন্ড',
			ss : '%d সেকেন্ড',
			m : 'এক মিনিট',
			mm : '%d মিনিট',
			h : 'এক ঘন্টা',
			hh : '%d ঘন্টা',
			d : 'এক দিন',
			dd : '%d দিন',
			M : 'এক মাস',
			MM : '%d মাস',
			y : 'এক বছর',
			yy : '%d বছর'
		},
		preparse: function (string) {
			return string.replace(/[১২৩৪৫৬৭৮৯০]/g, function (match) {
				return numberMap$2[match];
			});
		},
		postformat: function (string) {
			return string.replace(/\d/g, function (match) {
				return symbolMap$3[match];
			});
		},
		meridiemParse: /রাত|সকাল|দুপুর|বিকাল|রাত/,
		meridiemHour : function (hour, meridiem) {
			if (hour === 12) {
				hour = 0;
			}
			if ((meridiem === 'রাত' && hour >= 4) ||
				(meridiem === 'দুপুর' && hour < 5) ||
				meridiem === 'বিকাল') {
				return hour + 12;
			} else {
				return hour;
			}
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 4) {
				return 'রাত';
			} else if (hour < 10) {
				return 'সকাল';
			} else if (hour < 17) {
				return 'দুপুর';
			} else if (hour < 20) {
				return 'বিকাল';
			} else {
				return 'রাত';
			}
		},
		week : {
			dow : 0, // Sunday is the first day of the week.
			doy : 6  // The week that contains Jan 6th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var symbolMap$4 = {
			'1': '༡',
			'2': '༢',
			'3': '༣',
			'4': '༤',
			'5': '༥',
			'6': '༦',
			'7': '༧',
			'8': '༨',
			'9': '༩',
			'0': '༠'
		},
		numberMap$3 = {
			'༡': '1',
			'༢': '2',
			'༣': '3',
			'༤': '4',
			'༥': '5',
			'༦': '6',
			'༧': '7',
			'༨': '8',
			'༩': '9',
			'༠': '0'
		};

	hooks.defineLocale('bo', {
		months : 'ཟླ་བ་དང་པོ_ཟླ་བ་གཉིས་པ_ཟླ་བ་གསུམ་པ_ཟླ་བ་བཞི་པ_ཟླ་བ་ལྔ་པ_ཟླ་བ་དྲུག་པ_ཟླ་བ་བདུན་པ_ཟླ་བ་བརྒྱད་པ_ཟླ་བ་དགུ་པ_ཟླ་བ་བཅུ་པ_ཟླ་བ་བཅུ་གཅིག་པ_ཟླ་བ་བཅུ་གཉིས་པ'.split('_'),
		monthsShort : 'ཟླ་བ་དང་པོ_ཟླ་བ་གཉིས་པ_ཟླ་བ་གསུམ་པ_ཟླ་བ་བཞི་པ_ཟླ་བ་ལྔ་པ_ཟླ་བ་དྲུག་པ_ཟླ་བ་བདུན་པ_ཟླ་བ་བརྒྱད་པ_ཟླ་བ་དགུ་པ_ཟླ་བ་བཅུ་པ_ཟླ་བ་བཅུ་གཅིག་པ_ཟླ་བ་བཅུ་གཉིས་པ'.split('_'),
		weekdays : 'གཟའ་ཉི་མ་_གཟའ་ཟླ་བ་_གཟའ་མིག་དམར་_གཟའ་ལྷག་པ་_གཟའ་ཕུར་བུ_གཟའ་པ་སངས་_གཟའ་སྤེན་པ་'.split('_'),
		weekdaysShort : 'ཉི་མ་_ཟླ་བ་_མིག་དམར་_ལྷག་པ་_ཕུར་བུ_པ་སངས་_སྤེན་པ་'.split('_'),
		weekdaysMin : 'ཉི་མ་_ཟླ་བ་_མིག་དམར་_ལྷག་པ་_ཕུར་བུ_པ་སངས་_སྤེན་པ་'.split('_'),
		longDateFormat : {
			LT : 'A h:mm',
			LTS : 'A h:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY, A h:mm',
			LLLL : 'dddd, D MMMM YYYY, A h:mm'
		},
		calendar : {
			sameDay : '[དི་རིང] LT',
			nextDay : '[སང་ཉིན] LT',
			nextWeek : '[བདུན་ཕྲག་རྗེས་མ], LT',
			lastDay : '[ཁ་སང] LT',
			lastWeek : '[བདུན་ཕྲག་མཐའ་མ] dddd, LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : '%s ལ་',
			past : '%s སྔན་ལ',
			s : 'ལམ་སང',
			ss : '%d སྐར་ཆ།',
			m : 'སྐར་མ་གཅིག',
			mm : '%d སྐར་མ',
			h : 'ཆུ་ཚོད་གཅིག',
			hh : '%d ཆུ་ཚོད',
			d : 'ཉིན་གཅིག',
			dd : '%d ཉིན་',
			M : 'ཟླ་བ་གཅིག',
			MM : '%d ཟླ་བ',
			y : 'ལོ་གཅིག',
			yy : '%d ལོ'
		},
		preparse: function (string) {
			return string.replace(/[༡༢༣༤༥༦༧༨༩༠]/g, function (match) {
				return numberMap$3[match];
			});
		},
		postformat: function (string) {
			return string.replace(/\d/g, function (match) {
				return symbolMap$4[match];
			});
		},
		meridiemParse: /མཚན་མོ|ཞོགས་ཀས|ཉིན་གུང|དགོང་དག|མཚན་མོ/,
		meridiemHour : function (hour, meridiem) {
			if (hour === 12) {
				hour = 0;
			}
			if ((meridiem === 'མཚན་མོ' && hour >= 4) ||
				(meridiem === 'ཉིན་གུང' && hour < 5) ||
				meridiem === 'དགོང་དག') {
				return hour + 12;
			} else {
				return hour;
			}
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 4) {
				return 'མཚན་མོ';
			} else if (hour < 10) {
				return 'ཞོགས་ཀས';
			} else if (hour < 17) {
				return 'ཉིན་གུང';
			} else if (hour < 20) {
				return 'དགོང་དག';
			} else {
				return 'མཚན་མོ';
			}
		},
		week : {
			dow : 0, // Sunday is the first day of the week.
			doy : 6  // The week that contains Jan 6th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	function relativeTimeWithMutation(number, withoutSuffix, key) {
		var format = {
			'mm': 'munutenn',
			'MM': 'miz',
			'dd': 'devezh'
		};
		return number + ' ' + mutation(format[key], number);
	}
	function specialMutationForYears(number) {
		switch (lastNumber(number)) {
			case 1:
			case 3:
			case 4:
			case 5:
			case 9:
				return number + ' bloaz';
			default:
				return number + ' vloaz';
		}
	}
	function lastNumber(number) {
		if (number > 9) {
			return lastNumber(number % 10);
		}
		return number;
	}
	function mutation(text, number) {
		if (number === 2) {
			return softMutation(text);
		}
		return text;
	}
	function softMutation(text) {
		var mutationTable = {
			'm': 'v',
			'b': 'v',
			'd': 'z'
		};
		if (mutationTable[text.charAt(0)] === undefined) {
			return text;
		}
		return mutationTable[text.charAt(0)] + text.substring(1);
	}

	hooks.defineLocale('br', {
		months : 'Genver_C\'hwevrer_Meurzh_Ebrel_Mae_Mezheven_Gouere_Eost_Gwengolo_Here_Du_Kerzu'.split('_'),
		monthsShort : 'Gen_C\'hwe_Meu_Ebr_Mae_Eve_Gou_Eos_Gwe_Her_Du_Ker'.split('_'),
		weekdays : 'Sul_Lun_Meurzh_Merc\'her_Yaou_Gwener_Sadorn'.split('_'),
		weekdaysShort : 'Sul_Lun_Meu_Mer_Yao_Gwe_Sad'.split('_'),
		weekdaysMin : 'Su_Lu_Me_Mer_Ya_Gw_Sa'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'h[e]mm A',
			LTS : 'h[e]mm:ss A',
			L : 'DD/MM/YYYY',
			LL : 'D [a viz] MMMM YYYY',
			LLL : 'D [a viz] MMMM YYYY h[e]mm A',
			LLLL : 'dddd, D [a viz] MMMM YYYY h[e]mm A'
		},
		calendar : {
			sameDay : '[Hiziv da] LT',
			nextDay : '[Warc\'hoazh da] LT',
			nextWeek : 'dddd [da] LT',
			lastDay : '[Dec\'h da] LT',
			lastWeek : 'dddd [paset da] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'a-benn %s',
			past : '%s \'zo',
			s : 'un nebeud segondennoù',
			ss : '%d eilenn',
			m : 'ur vunutenn',
			mm : relativeTimeWithMutation,
			h : 'un eur',
			hh : '%d eur',
			d : 'un devezh',
			dd : relativeTimeWithMutation,
			M : 'ur miz',
			MM : relativeTimeWithMutation,
			y : 'ur bloaz',
			yy : specialMutationForYears
		},
		dayOfMonthOrdinalParse: /\d{1,2}(añ|vet)/,
		ordinal : function (number) {
			var output = (number === 1) ? 'añ' : 'vet';
			return number + output;
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	function translate(number, withoutSuffix, key) {
		var result = number + ' ';
		switch (key) {
			case 'ss':
				if (number === 1) {
					result += 'sekunda';
				} else if (number === 2 || number === 3 || number === 4) {
					result += 'sekunde';
				} else {
					result += 'sekundi';
				}
				return result;
			case 'm':
				return withoutSuffix ? 'jedna minuta' : 'jedne minute';
			case 'mm':
				if (number === 1) {
					result += 'minuta';
				} else if (number === 2 || number === 3 || number === 4) {
					result += 'minute';
				} else {
					result += 'minuta';
				}
				return result;
			case 'h':
				return withoutSuffix ? 'jedan sat' : 'jednog sata';
			case 'hh':
				if (number === 1) {
					result += 'sat';
				} else if (number === 2 || number === 3 || number === 4) {
					result += 'sata';
				} else {
					result += 'sati';
				}
				return result;
			case 'dd':
				if (number === 1) {
					result += 'dan';
				} else {
					result += 'dana';
				}
				return result;
			case 'MM':
				if (number === 1) {
					result += 'mjesec';
				} else if (number === 2 || number === 3 || number === 4) {
					result += 'mjeseca';
				} else {
					result += 'mjeseci';
				}
				return result;
			case 'yy':
				if (number === 1) {
					result += 'godina';
				} else if (number === 2 || number === 3 || number === 4) {
					result += 'godine';
				} else {
					result += 'godina';
				}
				return result;
		}
	}

	hooks.defineLocale('bs', {
		months : 'januar_februar_mart_april_maj_juni_juli_august_septembar_oktobar_novembar_decembar'.split('_'),
		monthsShort : 'jan._feb._mar._apr._maj._jun._jul._aug._sep._okt._nov._dec.'.split('_'),
		monthsParseExact: true,
		weekdays : 'nedjelja_ponedjeljak_utorak_srijeda_četvrtak_petak_subota'.split('_'),
		weekdaysShort : 'ned._pon._uto._sri._čet._pet._sub.'.split('_'),
		weekdaysMin : 'ne_po_ut_sr_če_pe_su'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'H:mm',
			LTS : 'H:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D. MMMM YYYY',
			LLL : 'D. MMMM YYYY H:mm',
			LLLL : 'dddd, D. MMMM YYYY H:mm'
		},
		calendar : {
			sameDay  : '[danas u] LT',
			nextDay  : '[sutra u] LT',
			nextWeek : function () {
				switch (this.day()) {
					case 0:
						return '[u] [nedjelju] [u] LT';
					case 3:
						return '[u] [srijedu] [u] LT';
					case 6:
						return '[u] [subotu] [u] LT';
					case 1:
					case 2:
					case 4:
					case 5:
						return '[u] dddd [u] LT';
				}
			},
			lastDay  : '[jučer u] LT',
			lastWeek : function () {
				switch (this.day()) {
					case 0:
					case 3:
						return '[prošlu] dddd [u] LT';
					case 6:
						return '[prošle] [subote] [u] LT';
					case 1:
					case 2:
					case 4:
					case 5:
						return '[prošli] dddd [u] LT';
				}
			},
			sameElse : 'L'
		},
		relativeTime : {
			future : 'za %s',
			past   : 'prije %s',
			s      : 'par sekundi',
			ss     : translate,
			m      : translate,
			mm     : translate,
			h      : translate,
			hh     : translate,
			d      : 'dan',
			dd     : translate,
			M      : 'mjesec',
			MM     : translate,
			y      : 'godinu',
			yy     : translate
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('ca', {
		months : {
			standalone: 'gener_febrer_març_abril_maig_juny_juliol_agost_setembre_octubre_novembre_desembre'.split('_'),
			format: 'de gener_de febrer_de març_d\'abril_de maig_de juny_de juliol_d\'agost_de setembre_d\'octubre_de novembre_de desembre'.split('_'),
			isFormat: /D[oD]?(\s)+MMMM/
		},
		monthsShort : 'gen._febr._març_abr._maig_juny_jul._ag._set._oct._nov._des.'.split('_'),
		monthsParseExact : true,
		weekdays : 'diumenge_dilluns_dimarts_dimecres_dijous_divendres_dissabte'.split('_'),
		weekdaysShort : 'dg._dl._dt._dc._dj._dv._ds.'.split('_'),
		weekdaysMin : 'dg_dl_dt_dc_dj_dv_ds'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'H:mm',
			LTS : 'H:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM [de] YYYY',
			ll : 'D MMM YYYY',
			LLL : 'D MMMM [de] YYYY [a les] H:mm',
			lll : 'D MMM YYYY, H:mm',
			LLLL : 'dddd D MMMM [de] YYYY [a les] H:mm',
			llll : 'ddd D MMM YYYY, H:mm'
		},
		calendar : {
			sameDay : function () {
				return '[avui a ' + ((this.hours() !== 1) ? 'les' : 'la') + '] LT';
			},
			nextDay : function () {
				return '[demà a ' + ((this.hours() !== 1) ? 'les' : 'la') + '] LT';
			},
			nextWeek : function () {
				return 'dddd [a ' + ((this.hours() !== 1) ? 'les' : 'la') + '] LT';
			},
			lastDay : function () {
				return '[ahir a ' + ((this.hours() !== 1) ? 'les' : 'la') + '] LT';
			},
			lastWeek : function () {
				return '[el] dddd [passat a ' + ((this.hours() !== 1) ? 'les' : 'la') + '] LT';
			},
			sameElse : 'L'
		},
		relativeTime : {
			future : 'd\'aquí %s',
			past : 'fa %s',
			s : 'uns segons',
			ss : '%d segons',
			m : 'un minut',
			mm : '%d minuts',
			h : 'una hora',
			hh : '%d hores',
			d : 'un dia',
			dd : '%d dies',
			M : 'un mes',
			MM : '%d mesos',
			y : 'un any',
			yy : '%d anys'
		},
		dayOfMonthOrdinalParse: /\d{1,2}(r|n|t|è|a)/,
		ordinal : function (number, period) {
			var output = (number === 1) ? 'r' :
				(number === 2) ? 'n' :
					(number === 3) ? 'r' :
						(number === 4) ? 't' : 'è';
			if (period === 'w' || period === 'W') {
				output = 'a';
			}
			return number + output;
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var months$3 = 'leden_únor_březen_duben_květen_červen_červenec_srpen_září_říjen_listopad_prosinec'.split('_'),
		monthsShort = 'led_úno_bře_dub_kvě_čvn_čvc_srp_zář_říj_lis_pro'.split('_');

	var monthsParse = [/^led/i, /^úno/i, /^bře/i, /^dub/i, /^kvě/i, /^(čvn|červen$|června)/i, /^(čvc|červenec|července)/i, /^srp/i, /^zář/i, /^říj/i, /^lis/i, /^pro/i];
	// NOTE: 'červen' is substring of 'červenec'; therefore 'červenec' must precede 'červen' in the regex to be fully matched.
	// Otherwise parser matches '1. červenec' as '1. červen' + 'ec'.
	var monthsRegex$1 = /^(leden|únor|březen|duben|květen|červenec|července|červen|června|srpen|září|říjen|listopad|prosinec|led|úno|bře|dub|kvě|čvn|čvc|srp|zář|říj|lis|pro)/i;

	function plural$1(n) {
		return (n > 1) && (n < 5) && (~~(n / 10) !== 1);
	}
	function translate$1(number, withoutSuffix, key, isFuture) {
		var result = number + ' ';
		switch (key) {
			case 's':  // a few seconds / in a few seconds / a few seconds ago
				return (withoutSuffix || isFuture) ? 'pár sekund' : 'pár sekundami';
			case 'ss': // 9 seconds / in 9 seconds / 9 seconds ago
				if (withoutSuffix || isFuture) {
					return result + (plural$1(number) ? 'sekundy' : 'sekund');
				} else {
					return result + 'sekundami';
				}
				break;
			case 'm':  // a minute / in a minute / a minute ago
				return withoutSuffix ? 'minuta' : (isFuture ? 'minutu' : 'minutou');
			case 'mm': // 9 minutes / in 9 minutes / 9 minutes ago
				if (withoutSuffix || isFuture) {
					return result + (plural$1(number) ? 'minuty' : 'minut');
				} else {
					return result + 'minutami';
				}
				break;
			case 'h':  // an hour / in an hour / an hour ago
				return withoutSuffix ? 'hodina' : (isFuture ? 'hodinu' : 'hodinou');
			case 'hh': // 9 hours / in 9 hours / 9 hours ago
				if (withoutSuffix || isFuture) {
					return result + (plural$1(number) ? 'hodiny' : 'hodin');
				} else {
					return result + 'hodinami';
				}
				break;
			case 'd':  // a day / in a day / a day ago
				return (withoutSuffix || isFuture) ? 'den' : 'dnem';
			case 'dd': // 9 days / in 9 days / 9 days ago
				if (withoutSuffix || isFuture) {
					return result + (plural$1(number) ? 'dny' : 'dní');
				} else {
					return result + 'dny';
				}
				break;
			case 'M':  // a month / in a month / a month ago
				return (withoutSuffix || isFuture) ? 'měsíc' : 'měsícem';
			case 'MM': // 9 months / in 9 months / 9 months ago
				if (withoutSuffix || isFuture) {
					return result + (plural$1(number) ? 'měsíce' : 'měsíců');
				} else {
					return result + 'měsíci';
				}
				break;
			case 'y':  // a year / in a year / a year ago
				return (withoutSuffix || isFuture) ? 'rok' : 'rokem';
			case 'yy': // 9 years / in 9 years / 9 years ago
				if (withoutSuffix || isFuture) {
					return result + (plural$1(number) ? 'roky' : 'let');
				} else {
					return result + 'lety';
				}
				break;
		}
	}

	hooks.defineLocale('cs', {
		months : months$3,
		monthsShort : monthsShort,
		monthsRegex : monthsRegex$1,
		monthsShortRegex : monthsRegex$1,
		// NOTE: 'červen' is substring of 'červenec'; therefore 'červenec' must precede 'červen' in the regex to be fully matched.
		// Otherwise parser matches '1. červenec' as '1. červen' + 'ec'.
		monthsStrictRegex : /^(leden|ledna|února|únor|březen|března|duben|dubna|květen|května|červenec|července|červen|června|srpen|srpna|září|říjen|října|listopadu|listopad|prosinec|prosince)/i,
		monthsShortStrictRegex : /^(led|úno|bře|dub|kvě|čvn|čvc|srp|zář|říj|lis|pro)/i,
		monthsParse : monthsParse,
		longMonthsParse : monthsParse,
		shortMonthsParse : monthsParse,
		weekdays : 'neděle_pondělí_úterý_středa_čtvrtek_pátek_sobota'.split('_'),
		weekdaysShort : 'ne_po_út_st_čt_pá_so'.split('_'),
		weekdaysMin : 'ne_po_út_st_čt_pá_so'.split('_'),
		longDateFormat : {
			LT: 'H:mm',
			LTS : 'H:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D. MMMM YYYY',
			LLL : 'D. MMMM YYYY H:mm',
			LLLL : 'dddd D. MMMM YYYY H:mm',
			l : 'D. M. YYYY'
		},
		calendar : {
			sameDay: '[dnes v] LT',
			nextDay: '[zítra v] LT',
			nextWeek: function () {
				switch (this.day()) {
					case 0:
						return '[v neděli v] LT';
					case 1:
					case 2:
						return '[v] dddd [v] LT';
					case 3:
						return '[ve středu v] LT';
					case 4:
						return '[ve čtvrtek v] LT';
					case 5:
						return '[v pátek v] LT';
					case 6:
						return '[v sobotu v] LT';
				}
			},
			lastDay: '[včera v] LT',
			lastWeek: function () {
				switch (this.day()) {
					case 0:
						return '[minulou neděli v] LT';
					case 1:
					case 2:
						return '[minulé] dddd [v] LT';
					case 3:
						return '[minulou středu v] LT';
					case 4:
					case 5:
						return '[minulý] dddd [v] LT';
					case 6:
						return '[minulou sobotu v] LT';
				}
			},
			sameElse: 'L'
		},
		relativeTime : {
			future : 'za %s',
			past : 'před %s',
			s : translate$1,
			ss : translate$1,
			m : translate$1,
			mm : translate$1,
			h : translate$1,
			hh : translate$1,
			d : translate$1,
			dd : translate$1,
			M : translate$1,
			MM : translate$1,
			y : translate$1,
			yy : translate$1
		},
		dayOfMonthOrdinalParse : /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('cv', {
		months : 'кӑрлач_нарӑс_пуш_ака_май_ҫӗртме_утӑ_ҫурла_авӑн_юпа_чӳк_раштав'.split('_'),
		monthsShort : 'кӑр_нар_пуш_ака_май_ҫӗр_утӑ_ҫур_авн_юпа_чӳк_раш'.split('_'),
		weekdays : 'вырсарникун_тунтикун_ытларикун_юнкун_кӗҫнерникун_эрнекун_шӑматкун'.split('_'),
		weekdaysShort : 'выр_тун_ытл_юн_кӗҫ_эрн_шӑм'.split('_'),
		weekdaysMin : 'вр_тн_ыт_юн_кҫ_эр_шм'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD-MM-YYYY',
			LL : 'YYYY [ҫулхи] MMMM [уйӑхӗн] D[-мӗшӗ]',
			LLL : 'YYYY [ҫулхи] MMMM [уйӑхӗн] D[-мӗшӗ], HH:mm',
			LLLL : 'dddd, YYYY [ҫулхи] MMMM [уйӑхӗн] D[-мӗшӗ], HH:mm'
		},
		calendar : {
			sameDay: '[Паян] LT [сехетре]',
			nextDay: '[Ыран] LT [сехетре]',
			lastDay: '[Ӗнер] LT [сехетре]',
			nextWeek: '[Ҫитес] dddd LT [сехетре]',
			lastWeek: '[Иртнӗ] dddd LT [сехетре]',
			sameElse: 'L'
		},
		relativeTime : {
			future : function (output) {
				var affix = /сехет$/i.exec(output) ? 'рен' : /ҫул$/i.exec(output) ? 'тан' : 'ран';
				return output + affix;
			},
			past : '%s каялла',
			s : 'пӗр-ик ҫеккунт',
			ss : '%d ҫеккунт',
			m : 'пӗр минут',
			mm : '%d минут',
			h : 'пӗр сехет',
			hh : '%d сехет',
			d : 'пӗр кун',
			dd : '%d кун',
			M : 'пӗр уйӑх',
			MM : '%d уйӑх',
			y : 'пӗр ҫул',
			yy : '%d ҫул'
		},
		dayOfMonthOrdinalParse: /\d{1,2}-мӗш/,
		ordinal : '%d-мӗш',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('cy', {
		months: 'Ionawr_Chwefror_Mawrth_Ebrill_Mai_Mehefin_Gorffennaf_Awst_Medi_Hydref_Tachwedd_Rhagfyr'.split('_'),
		monthsShort: 'Ion_Chwe_Maw_Ebr_Mai_Meh_Gor_Aws_Med_Hyd_Tach_Rhag'.split('_'),
		weekdays: 'Dydd Sul_Dydd Llun_Dydd Mawrth_Dydd Mercher_Dydd Iau_Dydd Gwener_Dydd Sadwrn'.split('_'),
		weekdaysShort: 'Sul_Llun_Maw_Mer_Iau_Gwe_Sad'.split('_'),
		weekdaysMin: 'Su_Ll_Ma_Me_Ia_Gw_Sa'.split('_'),
		weekdaysParseExact : true,
		// time formats are the same as en-gb
		longDateFormat: {
			LT: 'HH:mm',
			LTS : 'HH:mm:ss',
			L: 'DD/MM/YYYY',
			LL: 'D MMMM YYYY',
			LLL: 'D MMMM YYYY HH:mm',
			LLLL: 'dddd, D MMMM YYYY HH:mm'
		},
		calendar: {
			sameDay: '[Heddiw am] LT',
			nextDay: '[Yfory am] LT',
			nextWeek: 'dddd [am] LT',
			lastDay: '[Ddoe am] LT',
			lastWeek: 'dddd [diwethaf am] LT',
			sameElse: 'L'
		},
		relativeTime: {
			future: 'mewn %s',
			past: '%s yn ôl',
			s: 'ychydig eiliadau',
			ss: '%d eiliad',
			m: 'munud',
			mm: '%d munud',
			h: 'awr',
			hh: '%d awr',
			d: 'diwrnod',
			dd: '%d diwrnod',
			M: 'mis',
			MM: '%d mis',
			y: 'blwyddyn',
			yy: '%d flynedd'
		},
		dayOfMonthOrdinalParse: /\d{1,2}(fed|ain|af|il|ydd|ed|eg)/,
		// traditional ordinal numbers above 31 are not commonly used in colloquial Welsh
		ordinal: function (number) {
			var b = number,
				output = '',
				lookup = [
					'', 'af', 'il', 'ydd', 'ydd', 'ed', 'ed', 'ed', 'fed', 'fed', 'fed', // 1af to 10fed
					'eg', 'fed', 'eg', 'eg', 'fed', 'eg', 'eg', 'fed', 'eg', 'fed' // 11eg to 20fed
				];
			if (b > 20) {
				if (b === 40 || b === 50 || b === 60 || b === 80 || b === 100) {
					output = 'fed'; // not 30ain, 70ain or 90ain
				} else {
					output = 'ain';
				}
			} else if (b > 0) {
				output = lookup[b];
			}
			return number + output;
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('da', {
		months : 'januar_februar_marts_april_maj_juni_juli_august_september_oktober_november_december'.split('_'),
		monthsShort : 'jan_feb_mar_apr_maj_jun_jul_aug_sep_okt_nov_dec'.split('_'),
		weekdays : 'søndag_mandag_tirsdag_onsdag_torsdag_fredag_lørdag'.split('_'),
		weekdaysShort : 'søn_man_tir_ons_tor_fre_lør'.split('_'),
		weekdaysMin : 'sø_ma_ti_on_to_fr_lø'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D. MMMM YYYY',
			LLL : 'D. MMMM YYYY HH:mm',
			LLLL : 'dddd [d.] D. MMMM YYYY [kl.] HH:mm'
		},
		calendar : {
			sameDay : '[i dag kl.] LT',
			nextDay : '[i morgen kl.] LT',
			nextWeek : 'på dddd [kl.] LT',
			lastDay : '[i går kl.] LT',
			lastWeek : '[i] dddd[s kl.] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'om %s',
			past : '%s siden',
			s : 'få sekunder',
			ss : '%d sekunder',
			m : 'et minut',
			mm : '%d minutter',
			h : 'en time',
			hh : '%d timer',
			d : 'en dag',
			dd : '%d dage',
			M : 'en måned',
			MM : '%d måneder',
			y : 'et år',
			yy : '%d år'
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	function processRelativeTime(number, withoutSuffix, key, isFuture) {
		var format = {
			'm': ['eine Minute', 'einer Minute'],
			'h': ['eine Stunde', 'einer Stunde'],
			'd': ['ein Tag', 'einem Tag'],
			'dd': [number + ' Tage', number + ' Tagen'],
			'M': ['ein Monat', 'einem Monat'],
			'MM': [number + ' Monate', number + ' Monaten'],
			'y': ['ein Jahr', 'einem Jahr'],
			'yy': [number + ' Jahre', number + ' Jahren']
		};
		return withoutSuffix ? format[key][0] : format[key][1];
	}

	hooks.defineLocale('de-at', {
		months : 'Jänner_Februar_März_April_Mai_Juni_Juli_August_September_Oktober_November_Dezember'.split('_'),
		monthsShort : 'Jän._Feb._März_Apr._Mai_Juni_Juli_Aug._Sep._Okt._Nov._Dez.'.split('_'),
		monthsParseExact : true,
		weekdays : 'Sonntag_Montag_Dienstag_Mittwoch_Donnerstag_Freitag_Samstag'.split('_'),
		weekdaysShort : 'So._Mo._Di._Mi._Do._Fr._Sa.'.split('_'),
		weekdaysMin : 'So_Mo_Di_Mi_Do_Fr_Sa'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT: 'HH:mm',
			LTS: 'HH:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D. MMMM YYYY',
			LLL : 'D. MMMM YYYY HH:mm',
			LLLL : 'dddd, D. MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay: '[heute um] LT [Uhr]',
			sameElse: 'L',
			nextDay: '[morgen um] LT [Uhr]',
			nextWeek: 'dddd [um] LT [Uhr]',
			lastDay: '[gestern um] LT [Uhr]',
			lastWeek: '[letzten] dddd [um] LT [Uhr]'
		},
		relativeTime : {
			future : 'in %s',
			past : 'vor %s',
			s : 'ein paar Sekunden',
			ss : '%d Sekunden',
			m : processRelativeTime,
			mm : '%d Minuten',
			h : processRelativeTime,
			hh : '%d Stunden',
			d : processRelativeTime,
			dd : processRelativeTime,
			M : processRelativeTime,
			MM : processRelativeTime,
			y : processRelativeTime,
			yy : processRelativeTime
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	function processRelativeTime$1(number, withoutSuffix, key, isFuture) {
		var format = {
			'm': ['eine Minute', 'einer Minute'],
			'h': ['eine Stunde', 'einer Stunde'],
			'd': ['ein Tag', 'einem Tag'],
			'dd': [number + ' Tage', number + ' Tagen'],
			'M': ['ein Monat', 'einem Monat'],
			'MM': [number + ' Monate', number + ' Monaten'],
			'y': ['ein Jahr', 'einem Jahr'],
			'yy': [number + ' Jahre', number + ' Jahren']
		};
		return withoutSuffix ? format[key][0] : format[key][1];
	}

	hooks.defineLocale('de-ch', {
		months : 'Januar_Februar_März_April_Mai_Juni_Juli_August_September_Oktober_November_Dezember'.split('_'),
		monthsShort : 'Jan._Feb._März_Apr._Mai_Juni_Juli_Aug._Sep._Okt._Nov._Dez.'.split('_'),
		monthsParseExact : true,
		weekdays : 'Sonntag_Montag_Dienstag_Mittwoch_Donnerstag_Freitag_Samstag'.split('_'),
		weekdaysShort : 'So_Mo_Di_Mi_Do_Fr_Sa'.split('_'),
		weekdaysMin : 'So_Mo_Di_Mi_Do_Fr_Sa'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT: 'HH:mm',
			LTS: 'HH:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D. MMMM YYYY',
			LLL : 'D. MMMM YYYY HH:mm',
			LLLL : 'dddd, D. MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay: '[heute um] LT [Uhr]',
			sameElse: 'L',
			nextDay: '[morgen um] LT [Uhr]',
			nextWeek: 'dddd [um] LT [Uhr]',
			lastDay: '[gestern um] LT [Uhr]',
			lastWeek: '[letzten] dddd [um] LT [Uhr]'
		},
		relativeTime : {
			future : 'in %s',
			past : 'vor %s',
			s : 'ein paar Sekunden',
			ss : '%d Sekunden',
			m : processRelativeTime$1,
			mm : '%d Minuten',
			h : processRelativeTime$1,
			hh : '%d Stunden',
			d : processRelativeTime$1,
			dd : processRelativeTime$1,
			M : processRelativeTime$1,
			MM : processRelativeTime$1,
			y : processRelativeTime$1,
			yy : processRelativeTime$1
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	function processRelativeTime$2(number, withoutSuffix, key, isFuture) {
		var format = {
			'm': ['eine Minute', 'einer Minute'],
			'h': ['eine Stunde', 'einer Stunde'],
			'd': ['ein Tag', 'einem Tag'],
			'dd': [number + ' Tage', number + ' Tagen'],
			'M': ['ein Monat', 'einem Monat'],
			'MM': [number + ' Monate', number + ' Monaten'],
			'y': ['ein Jahr', 'einem Jahr'],
			'yy': [number + ' Jahre', number + ' Jahren']
		};
		return withoutSuffix ? format[key][0] : format[key][1];
	}

	hooks.defineLocale('de', {
		months : 'Januar_Februar_März_April_Mai_Juni_Juli_August_September_Oktober_November_Dezember'.split('_'),
		monthsShort : 'Jan._Feb._März_Apr._Mai_Juni_Juli_Aug._Sep._Okt._Nov._Dez.'.split('_'),
		monthsParseExact : true,
		weekdays : 'Sonntag_Montag_Dienstag_Mittwoch_Donnerstag_Freitag_Samstag'.split('_'),
		weekdaysShort : 'So._Mo._Di._Mi._Do._Fr._Sa.'.split('_'),
		weekdaysMin : 'So_Mo_Di_Mi_Do_Fr_Sa'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT: 'HH:mm',
			LTS: 'HH:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D. MMMM YYYY',
			LLL : 'D. MMMM YYYY HH:mm',
			LLLL : 'dddd, D. MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay: '[heute um] LT [Uhr]',
			sameElse: 'L',
			nextDay: '[morgen um] LT [Uhr]',
			nextWeek: 'dddd [um] LT [Uhr]',
			lastDay: '[gestern um] LT [Uhr]',
			lastWeek: '[letzten] dddd [um] LT [Uhr]'
		},
		relativeTime : {
			future : 'in %s',
			past : 'vor %s',
			s : 'ein paar Sekunden',
			ss : '%d Sekunden',
			m : processRelativeTime$2,
			mm : '%d Minuten',
			h : processRelativeTime$2,
			hh : '%d Stunden',
			d : processRelativeTime$2,
			dd : processRelativeTime$2,
			M : processRelativeTime$2,
			MM : processRelativeTime$2,
			y : processRelativeTime$2,
			yy : processRelativeTime$2
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var months$4 = [
		'ޖެނުއަރީ',
		'ފެބްރުއަރީ',
		'މާރިޗު',
		'އޭޕްރީލު',
		'މޭ',
		'ޖޫން',
		'ޖުލައި',
		'އޯގަސްޓު',
		'ސެޕްޓެމްބަރު',
		'އޮކްޓޯބަރު',
		'ނޮވެމްބަރު',
		'ޑިސެމްބަރު'
	], weekdays = [
		'އާދިއްތަ',
		'ހޯމަ',
		'އަންގާރަ',
		'ބުދަ',
		'ބުރާސްފަތި',
		'ހުކުރު',
		'ހޮނިހިރު'
	];

	hooks.defineLocale('dv', {
		months : months$4,
		monthsShort : months$4,
		weekdays : weekdays,
		weekdaysShort : weekdays,
		weekdaysMin : 'އާދި_ހޯމަ_އަން_ބުދަ_ބުރާ_ހުކު_ހޮނި'.split('_'),
		longDateFormat : {

			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'D/M/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd D MMMM YYYY HH:mm'
		},
		meridiemParse: /މކ|މފ/,
		isPM : function (input) {
			return 'މފ' === input;
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 12) {
				return 'މކ';
			} else {
				return 'މފ';
			}
		},
		calendar : {
			sameDay : '[މިއަދު] LT',
			nextDay : '[މާދަމާ] LT',
			nextWeek : 'dddd LT',
			lastDay : '[އިއްޔެ] LT',
			lastWeek : '[ފާއިތުވި] dddd LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'ތެރޭގައި %s',
			past : 'ކުރިން %s',
			s : 'ސިކުންތުކޮޅެއް',
			ss : 'd% ސިކުންތު',
			m : 'މިނިޓެއް',
			mm : 'މިނިޓު %d',
			h : 'ގަޑިއިރެއް',
			hh : 'ގަޑިއިރު %d',
			d : 'ދުވަހެއް',
			dd : 'ދުވަސް %d',
			M : 'މަހެއް',
			MM : 'މަސް %d',
			y : 'އަހަރެއް',
			yy : 'އަހަރު %d'
		},
		preparse: function (string) {
			return string.replace(/،/g, ',');
		},
		postformat: function (string) {
			return string.replace(/,/g, '،');
		},
		week : {
			dow : 7,  // Sunday is the first day of the week.
			doy : 12  // The week that contains Jan 12th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('el', {
		monthsNominativeEl : 'Ιανουάριος_Φεβρουάριος_Μάρτιος_Απρίλιος_Μάιος_Ιούνιος_Ιούλιος_Αύγουστος_Σεπτέμβριος_Οκτώβριος_Νοέμβριος_Δεκέμβριος'.split('_'),
		monthsGenitiveEl : 'Ιανουαρίου_Φεβρουαρίου_Μαρτίου_Απριλίου_Μαΐου_Ιουνίου_Ιουλίου_Αυγούστου_Σεπτεμβρίου_Οκτωβρίου_Νοεμβρίου_Δεκεμβρίου'.split('_'),
		months : function (momentToFormat, format) {
			if (!momentToFormat) {
				return this._monthsNominativeEl;
			} else if (typeof format === 'string' && /D/.test(format.substring(0, format.indexOf('MMMM')))) { // if there is a day number before 'MMMM'
				return this._monthsGenitiveEl[momentToFormat.month()];
			} else {
				return this._monthsNominativeEl[momentToFormat.month()];
			}
		},
		monthsShort : 'Ιαν_Φεβ_Μαρ_Απρ_Μαϊ_Ιουν_Ιουλ_Αυγ_Σεπ_Οκτ_Νοε_Δεκ'.split('_'),
		weekdays : 'Κυριακή_Δευτέρα_Τρίτη_Τετάρτη_Πέμπτη_Παρασκευή_Σάββατο'.split('_'),
		weekdaysShort : 'Κυρ_Δευ_Τρι_Τετ_Πεμ_Παρ_Σαβ'.split('_'),
		weekdaysMin : 'Κυ_Δε_Τρ_Τε_Πε_Πα_Σα'.split('_'),
		meridiem : function (hours, minutes, isLower) {
			if (hours > 11) {
				return isLower ? 'μμ' : 'ΜΜ';
			} else {
				return isLower ? 'πμ' : 'ΠΜ';
			}
		},
		isPM : function (input) {
			return ((input + '').toLowerCase()[0] === 'μ');
		},
		meridiemParse : /[ΠΜ]\.?Μ?\.?/i,
		longDateFormat : {
			LT : 'h:mm A',
			LTS : 'h:mm:ss A',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY h:mm A',
			LLLL : 'dddd, D MMMM YYYY h:mm A'
		},
		calendarEl : {
			sameDay : '[Σήμερα {}] LT',
			nextDay : '[Αύριο {}] LT',
			nextWeek : 'dddd [{}] LT',
			lastDay : '[Χθες {}] LT',
			lastWeek : function () {
				switch (this.day()) {
					case 6:
						return '[το προηγούμενο] dddd [{}] LT';
					default:
						return '[την προηγούμενη] dddd [{}] LT';
				}
			},
			sameElse : 'L'
		},
		calendar : function (key, mom) {
			var output = this._calendarEl[key],
				hours = mom && mom.hours();
			if (isFunction(output)) {
				output = output.apply(mom);
			}
			return output.replace('{}', (hours % 12 === 1 ? 'στη' : 'στις'));
		},
		relativeTime : {
			future : 'σε %s',
			past : '%s πριν',
			s : 'λίγα δευτερόλεπτα',
			ss : '%d δευτερόλεπτα',
			m : 'ένα λεπτό',
			mm : '%d λεπτά',
			h : 'μία ώρα',
			hh : '%d ώρες',
			d : 'μία μέρα',
			dd : '%d μέρες',
			M : 'ένας μήνας',
			MM : '%d μήνες',
			y : 'ένας χρόνος',
			yy : '%d χρόνια'
		},
		dayOfMonthOrdinalParse: /\d{1,2}η/,
		ordinal: '%dη',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4st is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('en-SG', {
		months : 'January_February_March_April_May_June_July_August_September_October_November_December'.split('_'),
		monthsShort : 'Jan_Feb_Mar_Apr_May_Jun_Jul_Aug_Sep_Oct_Nov_Dec'.split('_'),
		weekdays : 'Sunday_Monday_Tuesday_Wednesday_Thursday_Friday_Saturday'.split('_'),
		weekdaysShort : 'Sun_Mon_Tue_Wed_Thu_Fri_Sat'.split('_'),
		weekdaysMin : 'Su_Mo_Tu_We_Th_Fr_Sa'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd, D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay : '[Today at] LT',
			nextDay : '[Tomorrow at] LT',
			nextWeek : 'dddd [at] LT',
			lastDay : '[Yesterday at] LT',
			lastWeek : '[Last] dddd [at] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'in %s',
			past : '%s ago',
			s : 'a few seconds',
			ss : '%d seconds',
			m : 'a minute',
			mm : '%d minutes',
			h : 'an hour',
			hh : '%d hours',
			d : 'a day',
			dd : '%d days',
			M : 'a month',
			MM : '%d months',
			y : 'a year',
			yy : '%d years'
		},
		dayOfMonthOrdinalParse: /\d{1,2}(st|nd|rd|th)/,
		ordinal : function (number) {
			var b = number % 10,
				output = (~~(number % 100 / 10) === 1) ? 'th' :
					(b === 1) ? 'st' :
						(b === 2) ? 'nd' :
							(b === 3) ? 'rd' : 'th';
			return number + output;
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('en-au', {
		months : 'January_February_March_April_May_June_July_August_September_October_November_December'.split('_'),
		monthsShort : 'Jan_Feb_Mar_Apr_May_Jun_Jul_Aug_Sep_Oct_Nov_Dec'.split('_'),
		weekdays : 'Sunday_Monday_Tuesday_Wednesday_Thursday_Friday_Saturday'.split('_'),
		weekdaysShort : 'Sun_Mon_Tue_Wed_Thu_Fri_Sat'.split('_'),
		weekdaysMin : 'Su_Mo_Tu_We_Th_Fr_Sa'.split('_'),
		longDateFormat : {
			LT : 'h:mm A',
			LTS : 'h:mm:ss A',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY h:mm A',
			LLLL : 'dddd, D MMMM YYYY h:mm A'
		},
		calendar : {
			sameDay : '[Today at] LT',
			nextDay : '[Tomorrow at] LT',
			nextWeek : 'dddd [at] LT',
			lastDay : '[Yesterday at] LT',
			lastWeek : '[Last] dddd [at] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'in %s',
			past : '%s ago',
			s : 'a few seconds',
			ss : '%d seconds',
			m : 'a minute',
			mm : '%d minutes',
			h : 'an hour',
			hh : '%d hours',
			d : 'a day',
			dd : '%d days',
			M : 'a month',
			MM : '%d months',
			y : 'a year',
			yy : '%d years'
		},
		dayOfMonthOrdinalParse: /\d{1,2}(st|nd|rd|th)/,
		ordinal : function (number) {
			var b = number % 10,
				output = (~~(number % 100 / 10) === 1) ? 'th' :
					(b === 1) ? 'st' :
						(b === 2) ? 'nd' :
							(b === 3) ? 'rd' : 'th';
			return number + output;
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('en-ca', {
		months : 'January_February_March_April_May_June_July_August_September_October_November_December'.split('_'),
		monthsShort : 'Jan_Feb_Mar_Apr_May_Jun_Jul_Aug_Sep_Oct_Nov_Dec'.split('_'),
		weekdays : 'Sunday_Monday_Tuesday_Wednesday_Thursday_Friday_Saturday'.split('_'),
		weekdaysShort : 'Sun_Mon_Tue_Wed_Thu_Fri_Sat'.split('_'),
		weekdaysMin : 'Su_Mo_Tu_We_Th_Fr_Sa'.split('_'),
		longDateFormat : {
			LT : 'h:mm A',
			LTS : 'h:mm:ss A',
			L : 'YYYY-MM-DD',
			LL : 'MMMM D, YYYY',
			LLL : 'MMMM D, YYYY h:mm A',
			LLLL : 'dddd, MMMM D, YYYY h:mm A'
		},
		calendar : {
			sameDay : '[Today at] LT',
			nextDay : '[Tomorrow at] LT',
			nextWeek : 'dddd [at] LT',
			lastDay : '[Yesterday at] LT',
			lastWeek : '[Last] dddd [at] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'in %s',
			past : '%s ago',
			s : 'a few seconds',
			ss : '%d seconds',
			m : 'a minute',
			mm : '%d minutes',
			h : 'an hour',
			hh : '%d hours',
			d : 'a day',
			dd : '%d days',
			M : 'a month',
			MM : '%d months',
			y : 'a year',
			yy : '%d years'
		},
		dayOfMonthOrdinalParse: /\d{1,2}(st|nd|rd|th)/,
		ordinal : function (number) {
			var b = number % 10,
				output = (~~(number % 100 / 10) === 1) ? 'th' :
					(b === 1) ? 'st' :
						(b === 2) ? 'nd' :
							(b === 3) ? 'rd' : 'th';
			return number + output;
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('en-gb', {
		months : 'January_February_March_April_May_June_July_August_September_October_November_December'.split('_'),
		monthsShort : 'Jan_Feb_Mar_Apr_May_Jun_Jul_Aug_Sep_Oct_Nov_Dec'.split('_'),
		weekdays : 'Sunday_Monday_Tuesday_Wednesday_Thursday_Friday_Saturday'.split('_'),
		weekdaysShort : 'Sun_Mon_Tue_Wed_Thu_Fri_Sat'.split('_'),
		weekdaysMin : 'Su_Mo_Tu_We_Th_Fr_Sa'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd, D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay : '[Today at] LT',
			nextDay : '[Tomorrow at] LT',
			nextWeek : 'dddd [at] LT',
			lastDay : '[Yesterday at] LT',
			lastWeek : '[Last] dddd [at] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'in %s',
			past : '%s ago',
			s : 'a few seconds',
			ss : '%d seconds',
			m : 'a minute',
			mm : '%d minutes',
			h : 'an hour',
			hh : '%d hours',
			d : 'a day',
			dd : '%d days',
			M : 'a month',
			MM : '%d months',
			y : 'a year',
			yy : '%d years'
		},
		dayOfMonthOrdinalParse: /\d{1,2}(st|nd|rd|th)/,
		ordinal : function (number) {
			var b = number % 10,
				output = (~~(number % 100 / 10) === 1) ? 'th' :
					(b === 1) ? 'st' :
						(b === 2) ? 'nd' :
							(b === 3) ? 'rd' : 'th';
			return number + output;
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('en-ie', {
		months : 'January_February_March_April_May_June_July_August_September_October_November_December'.split('_'),
		monthsShort : 'Jan_Feb_Mar_Apr_May_Jun_Jul_Aug_Sep_Oct_Nov_Dec'.split('_'),
		weekdays : 'Sunday_Monday_Tuesday_Wednesday_Thursday_Friday_Saturday'.split('_'),
		weekdaysShort : 'Sun_Mon_Tue_Wed_Thu_Fri_Sat'.split('_'),
		weekdaysMin : 'Su_Mo_Tu_We_Th_Fr_Sa'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay : '[Today at] LT',
			nextDay : '[Tomorrow at] LT',
			nextWeek : 'dddd [at] LT',
			lastDay : '[Yesterday at] LT',
			lastWeek : '[Last] dddd [at] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'in %s',
			past : '%s ago',
			s : 'a few seconds',
			ss : '%d seconds',
			m : 'a minute',
			mm : '%d minutes',
			h : 'an hour',
			hh : '%d hours',
			d : 'a day',
			dd : '%d days',
			M : 'a month',
			MM : '%d months',
			y : 'a year',
			yy : '%d years'
		},
		dayOfMonthOrdinalParse: /\d{1,2}(st|nd|rd|th)/,
		ordinal : function (number) {
			var b = number % 10,
				output = (~~(number % 100 / 10) === 1) ? 'th' :
					(b === 1) ? 'st' :
						(b === 2) ? 'nd' :
							(b === 3) ? 'rd' : 'th';
			return number + output;
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('en-il', {
		months : 'January_February_March_April_May_June_July_August_September_October_November_December'.split('_'),
		monthsShort : 'Jan_Feb_Mar_Apr_May_Jun_Jul_Aug_Sep_Oct_Nov_Dec'.split('_'),
		weekdays : 'Sunday_Monday_Tuesday_Wednesday_Thursday_Friday_Saturday'.split('_'),
		weekdaysShort : 'Sun_Mon_Tue_Wed_Thu_Fri_Sat'.split('_'),
		weekdaysMin : 'Su_Mo_Tu_We_Th_Fr_Sa'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd, D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay : '[Today at] LT',
			nextDay : '[Tomorrow at] LT',
			nextWeek : 'dddd [at] LT',
			lastDay : '[Yesterday at] LT',
			lastWeek : '[Last] dddd [at] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'in %s',
			past : '%s ago',
			s : 'a few seconds',
			m : 'a minute',
			mm : '%d minutes',
			h : 'an hour',
			hh : '%d hours',
			d : 'a day',
			dd : '%d days',
			M : 'a month',
			MM : '%d months',
			y : 'a year',
			yy : '%d years'
		},
		dayOfMonthOrdinalParse: /\d{1,2}(st|nd|rd|th)/,
		ordinal : function (number) {
			var b = number % 10,
				output = (~~(number % 100 / 10) === 1) ? 'th' :
					(b === 1) ? 'st' :
						(b === 2) ? 'nd' :
							(b === 3) ? 'rd' : 'th';
			return number + output;
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('en-nz', {
		months : 'January_February_March_April_May_June_July_August_September_October_November_December'.split('_'),
		monthsShort : 'Jan_Feb_Mar_Apr_May_Jun_Jul_Aug_Sep_Oct_Nov_Dec'.split('_'),
		weekdays : 'Sunday_Monday_Tuesday_Wednesday_Thursday_Friday_Saturday'.split('_'),
		weekdaysShort : 'Sun_Mon_Tue_Wed_Thu_Fri_Sat'.split('_'),
		weekdaysMin : 'Su_Mo_Tu_We_Th_Fr_Sa'.split('_'),
		longDateFormat : {
			LT : 'h:mm A',
			LTS : 'h:mm:ss A',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY h:mm A',
			LLLL : 'dddd, D MMMM YYYY h:mm A'
		},
		calendar : {
			sameDay : '[Today at] LT',
			nextDay : '[Tomorrow at] LT',
			nextWeek : 'dddd [at] LT',
			lastDay : '[Yesterday at] LT',
			lastWeek : '[Last] dddd [at] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'in %s',
			past : '%s ago',
			s : 'a few seconds',
			ss : '%d seconds',
			m : 'a minute',
			mm : '%d minutes',
			h : 'an hour',
			hh : '%d hours',
			d : 'a day',
			dd : '%d days',
			M : 'a month',
			MM : '%d months',
			y : 'a year',
			yy : '%d years'
		},
		dayOfMonthOrdinalParse: /\d{1,2}(st|nd|rd|th)/,
		ordinal : function (number) {
			var b = number % 10,
				output = (~~(number % 100 / 10) === 1) ? 'th' :
					(b === 1) ? 'st' :
						(b === 2) ? 'nd' :
							(b === 3) ? 'rd' : 'th';
			return number + output;
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('eo', {
		months : 'januaro_februaro_marto_aprilo_majo_junio_julio_aŭgusto_septembro_oktobro_novembro_decembro'.split('_'),
		monthsShort : 'jan_feb_mar_apr_maj_jun_jul_aŭg_sep_okt_nov_dec'.split('_'),
		weekdays : 'dimanĉo_lundo_mardo_merkredo_ĵaŭdo_vendredo_sabato'.split('_'),
		weekdaysShort : 'dim_lun_mard_merk_ĵaŭ_ven_sab'.split('_'),
		weekdaysMin : 'di_lu_ma_me_ĵa_ve_sa'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'YYYY-MM-DD',
			LL : 'D[-a de] MMMM, YYYY',
			LLL : 'D[-a de] MMMM, YYYY HH:mm',
			LLLL : 'dddd, [la] D[-a de] MMMM, YYYY HH:mm'
		},
		meridiemParse: /[ap]\.t\.m/i,
		isPM: function (input) {
			return input.charAt(0).toLowerCase() === 'p';
		},
		meridiem : function (hours, minutes, isLower) {
			if (hours > 11) {
				return isLower ? 'p.t.m.' : 'P.T.M.';
			} else {
				return isLower ? 'a.t.m.' : 'A.T.M.';
			}
		},
		calendar : {
			sameDay : '[Hodiaŭ je] LT',
			nextDay : '[Morgaŭ je] LT',
			nextWeek : 'dddd [je] LT',
			lastDay : '[Hieraŭ je] LT',
			lastWeek : '[pasinta] dddd [je] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'post %s',
			past : 'antaŭ %s',
			s : 'sekundoj',
			ss : '%d sekundoj',
			m : 'minuto',
			mm : '%d minutoj',
			h : 'horo',
			hh : '%d horoj',
			d : 'tago',//ne 'diurno', ĉar estas uzita por proksimumo
			dd : '%d tagoj',
			M : 'monato',
			MM : '%d monatoj',
			y : 'jaro',
			yy : '%d jaroj'
		},
		dayOfMonthOrdinalParse: /\d{1,2}a/,
		ordinal : '%da',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var monthsShortDot = 'ene._feb._mar._abr._may._jun._jul._ago._sep._oct._nov._dic.'.split('_'),
		monthsShort$1 = 'ene_feb_mar_abr_may_jun_jul_ago_sep_oct_nov_dic'.split('_');

	var monthsParse$1 = [/^ene/i, /^feb/i, /^mar/i, /^abr/i, /^may/i, /^jun/i, /^jul/i, /^ago/i, /^sep/i, /^oct/i, /^nov/i, /^dic/i];
	var monthsRegex$2 = /^(enero|febrero|marzo|abril|mayo|junio|julio|agosto|septiembre|octubre|noviembre|diciembre|ene\.?|feb\.?|mar\.?|abr\.?|may\.?|jun\.?|jul\.?|ago\.?|sep\.?|oct\.?|nov\.?|dic\.?)/i;

	hooks.defineLocale('es-do', {
		months : 'enero_febrero_marzo_abril_mayo_junio_julio_agosto_septiembre_octubre_noviembre_diciembre'.split('_'),
		monthsShort : function (m, format) {
			if (!m) {
				return monthsShortDot;
			} else if (/-MMM-/.test(format)) {
				return monthsShort$1[m.month()];
			} else {
				return monthsShortDot[m.month()];
			}
		},
		monthsRegex: monthsRegex$2,
		monthsShortRegex: monthsRegex$2,
		monthsStrictRegex: /^(enero|febrero|marzo|abril|mayo|junio|julio|agosto|septiembre|octubre|noviembre|diciembre)/i,
		monthsShortStrictRegex: /^(ene\.?|feb\.?|mar\.?|abr\.?|may\.?|jun\.?|jul\.?|ago\.?|sep\.?|oct\.?|nov\.?|dic\.?)/i,
		monthsParse: monthsParse$1,
		longMonthsParse: monthsParse$1,
		shortMonthsParse: monthsParse$1,
		weekdays : 'domingo_lunes_martes_miércoles_jueves_viernes_sábado'.split('_'),
		weekdaysShort : 'dom._lun._mar._mié._jue._vie._sáb.'.split('_'),
		weekdaysMin : 'do_lu_ma_mi_ju_vi_sá'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'h:mm A',
			LTS : 'h:mm:ss A',
			L : 'DD/MM/YYYY',
			LL : 'D [de] MMMM [de] YYYY',
			LLL : 'D [de] MMMM [de] YYYY h:mm A',
			LLLL : 'dddd, D [de] MMMM [de] YYYY h:mm A'
		},
		calendar : {
			sameDay : function () {
				return '[hoy a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
			},
			nextDay : function () {
				return '[mañana a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
			},
			nextWeek : function () {
				return 'dddd [a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
			},
			lastDay : function () {
				return '[ayer a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
			},
			lastWeek : function () {
				return '[el] dddd [pasado a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
			},
			sameElse : 'L'
		},
		relativeTime : {
			future : 'en %s',
			past : 'hace %s',
			s : 'unos segundos',
			ss : '%d segundos',
			m : 'un minuto',
			mm : '%d minutos',
			h : 'una hora',
			hh : '%d horas',
			d : 'un día',
			dd : '%d días',
			M : 'un mes',
			MM : '%d meses',
			y : 'un año',
			yy : '%d años'
		},
		dayOfMonthOrdinalParse : /\d{1,2}º/,
		ordinal : '%dº',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var monthsShortDot$1 = 'ene._feb._mar._abr._may._jun._jul._ago._sep._oct._nov._dic.'.split('_'),
		monthsShort$2 = 'ene_feb_mar_abr_may_jun_jul_ago_sep_oct_nov_dic'.split('_');

	var monthsParse$2 = [/^ene/i, /^feb/i, /^mar/i, /^abr/i, /^may/i, /^jun/i, /^jul/i, /^ago/i, /^sep/i, /^oct/i, /^nov/i, /^dic/i];
	var monthsRegex$3 = /^(enero|febrero|marzo|abril|mayo|junio|julio|agosto|septiembre|octubre|noviembre|diciembre|ene\.?|feb\.?|mar\.?|abr\.?|may\.?|jun\.?|jul\.?|ago\.?|sep\.?|oct\.?|nov\.?|dic\.?)/i;

	hooks.defineLocale('es-us', {
		months : 'enero_febrero_marzo_abril_mayo_junio_julio_agosto_septiembre_octubre_noviembre_diciembre'.split('_'),
		monthsShort : function (m, format) {
			if (!m) {
				return monthsShortDot$1;
			} else if (/-MMM-/.test(format)) {
				return monthsShort$2[m.month()];
			} else {
				return monthsShortDot$1[m.month()];
			}
		},
		monthsRegex: monthsRegex$3,
		monthsShortRegex: monthsRegex$3,
		monthsStrictRegex: /^(enero|febrero|marzo|abril|mayo|junio|julio|agosto|septiembre|octubre|noviembre|diciembre)/i,
		monthsShortStrictRegex: /^(ene\.?|feb\.?|mar\.?|abr\.?|may\.?|jun\.?|jul\.?|ago\.?|sep\.?|oct\.?|nov\.?|dic\.?)/i,
		monthsParse: monthsParse$2,
		longMonthsParse: monthsParse$2,
		shortMonthsParse: monthsParse$2,
		weekdays : 'domingo_lunes_martes_miércoles_jueves_viernes_sábado'.split('_'),
		weekdaysShort : 'dom._lun._mar._mié._jue._vie._sáb.'.split('_'),
		weekdaysMin : 'do_lu_ma_mi_ju_vi_sá'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'h:mm A',
			LTS : 'h:mm:ss A',
			L : 'MM/DD/YYYY',
			LL : 'D [de] MMMM [de] YYYY',
			LLL : 'D [de] MMMM [de] YYYY h:mm A',
			LLLL : 'dddd, D [de] MMMM [de] YYYY h:mm A'
		},
		calendar : {
			sameDay : function () {
				return '[hoy a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
			},
			nextDay : function () {
				return '[mañana a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
			},
			nextWeek : function () {
				return 'dddd [a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
			},
			lastDay : function () {
				return '[ayer a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
			},
			lastWeek : function () {
				return '[el] dddd [pasado a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
			},
			sameElse : 'L'
		},
		relativeTime : {
			future : 'en %s',
			past : 'hace %s',
			s : 'unos segundos',
			ss : '%d segundos',
			m : 'un minuto',
			mm : '%d minutos',
			h : 'una hora',
			hh : '%d horas',
			d : 'un día',
			dd : '%d días',
			M : 'un mes',
			MM : '%d meses',
			y : 'un año',
			yy : '%d años'
		},
		dayOfMonthOrdinalParse : /\d{1,2}º/,
		ordinal : '%dº',
		week : {
			dow : 0, // Sunday is the first day of the week.
			doy : 6  // The week that contains Jan 6th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var monthsShortDot$2 = 'ene._feb._mar._abr._may._jun._jul._ago._sep._oct._nov._dic.'.split('_'),
		monthsShort$3 = 'ene_feb_mar_abr_may_jun_jul_ago_sep_oct_nov_dic'.split('_');

	var monthsParse$3 = [/^ene/i, /^feb/i, /^mar/i, /^abr/i, /^may/i, /^jun/i, /^jul/i, /^ago/i, /^sep/i, /^oct/i, /^nov/i, /^dic/i];
	var monthsRegex$4 = /^(enero|febrero|marzo|abril|mayo|junio|julio|agosto|septiembre|octubre|noviembre|diciembre|ene\.?|feb\.?|mar\.?|abr\.?|may\.?|jun\.?|jul\.?|ago\.?|sep\.?|oct\.?|nov\.?|dic\.?)/i;

	hooks.defineLocale('es', {
		months : 'enero_febrero_marzo_abril_mayo_junio_julio_agosto_septiembre_octubre_noviembre_diciembre'.split('_'),
		monthsShort : function (m, format) {
			if (!m) {
				return monthsShortDot$2;
			} else if (/-MMM-/.test(format)) {
				return monthsShort$3[m.month()];
			} else {
				return monthsShortDot$2[m.month()];
			}
		},
		monthsRegex : monthsRegex$4,
		monthsShortRegex : monthsRegex$4,
		monthsStrictRegex : /^(enero|febrero|marzo|abril|mayo|junio|julio|agosto|septiembre|octubre|noviembre|diciembre)/i,
		monthsShortStrictRegex : /^(ene\.?|feb\.?|mar\.?|abr\.?|may\.?|jun\.?|jul\.?|ago\.?|sep\.?|oct\.?|nov\.?|dic\.?)/i,
		monthsParse : monthsParse$3,
		longMonthsParse : monthsParse$3,
		shortMonthsParse : monthsParse$3,
		weekdays : 'domingo_lunes_martes_miércoles_jueves_viernes_sábado'.split('_'),
		weekdaysShort : 'dom._lun._mar._mié._jue._vie._sáb.'.split('_'),
		weekdaysMin : 'do_lu_ma_mi_ju_vi_sá'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'H:mm',
			LTS : 'H:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D [de] MMMM [de] YYYY',
			LLL : 'D [de] MMMM [de] YYYY H:mm',
			LLLL : 'dddd, D [de] MMMM [de] YYYY H:mm'
		},
		calendar : {
			sameDay : function () {
				return '[hoy a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
			},
			nextDay : function () {
				return '[mañana a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
			},
			nextWeek : function () {
				return 'dddd [a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
			},
			lastDay : function () {
				return '[ayer a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
			},
			lastWeek : function () {
				return '[el] dddd [pasado a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
			},
			sameElse : 'L'
		},
		relativeTime : {
			future : 'en %s',
			past : 'hace %s',
			s : 'unos segundos',
			ss : '%d segundos',
			m : 'un minuto',
			mm : '%d minutos',
			h : 'una hora',
			hh : '%d horas',
			d : 'un día',
			dd : '%d días',
			M : 'un mes',
			MM : '%d meses',
			y : 'un año',
			yy : '%d años'
		},
		dayOfMonthOrdinalParse : /\d{1,2}º/,
		ordinal : '%dº',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	function processRelativeTime$3(number, withoutSuffix, key, isFuture) {
		var format = {
			's' : ['mõne sekundi', 'mõni sekund', 'paar sekundit'],
			'ss': [number + 'sekundi', number + 'sekundit'],
			'm' : ['ühe minuti', 'üks minut'],
			'mm': [number + ' minuti', number + ' minutit'],
			'h' : ['ühe tunni', 'tund aega', 'üks tund'],
			'hh': [number + ' tunni', number + ' tundi'],
			'd' : ['ühe päeva', 'üks päev'],
			'M' : ['kuu aja', 'kuu aega', 'üks kuu'],
			'MM': [number + ' kuu', number + ' kuud'],
			'y' : ['ühe aasta', 'aasta', 'üks aasta'],
			'yy': [number + ' aasta', number + ' aastat']
		};
		if (withoutSuffix) {
			return format[key][2] ? format[key][2] : format[key][1];
		}
		return isFuture ? format[key][0] : format[key][1];
	}

	hooks.defineLocale('et', {
		months        : 'jaanuar_veebruar_märts_aprill_mai_juuni_juuli_august_september_oktoober_november_detsember'.split('_'),
		monthsShort   : 'jaan_veebr_märts_apr_mai_juuni_juuli_aug_sept_okt_nov_dets'.split('_'),
		weekdays      : 'pühapäev_esmaspäev_teisipäev_kolmapäev_neljapäev_reede_laupäev'.split('_'),
		weekdaysShort : 'P_E_T_K_N_R_L'.split('_'),
		weekdaysMin   : 'P_E_T_K_N_R_L'.split('_'),
		longDateFormat : {
			LT   : 'H:mm',
			LTS : 'H:mm:ss',
			L    : 'DD.MM.YYYY',
			LL   : 'D. MMMM YYYY',
			LLL  : 'D. MMMM YYYY H:mm',
			LLLL : 'dddd, D. MMMM YYYY H:mm'
		},
		calendar : {
			sameDay  : '[Täna,] LT',
			nextDay  : '[Homme,] LT',
			nextWeek : '[Järgmine] dddd LT',
			lastDay  : '[Eile,] LT',
			lastWeek : '[Eelmine] dddd LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : '%s pärast',
			past   : '%s tagasi',
			s      : processRelativeTime$3,
			ss     : processRelativeTime$3,
			m      : processRelativeTime$3,
			mm     : processRelativeTime$3,
			h      : processRelativeTime$3,
			hh     : processRelativeTime$3,
			d      : processRelativeTime$3,
			dd     : '%d päeva',
			M      : processRelativeTime$3,
			MM     : processRelativeTime$3,
			y      : processRelativeTime$3,
			yy     : processRelativeTime$3
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('eu', {
		months : 'urtarrila_otsaila_martxoa_apirila_maiatza_ekaina_uztaila_abuztua_iraila_urria_azaroa_abendua'.split('_'),
		monthsShort : 'urt._ots._mar._api._mai._eka._uzt._abu._ira._urr._aza._abe.'.split('_'),
		monthsParseExact : true,
		weekdays : 'igandea_astelehena_asteartea_asteazkena_osteguna_ostirala_larunbata'.split('_'),
		weekdaysShort : 'ig._al._ar._az._og._ol._lr.'.split('_'),
		weekdaysMin : 'ig_al_ar_az_og_ol_lr'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'YYYY-MM-DD',
			LL : 'YYYY[ko] MMMM[ren] D[a]',
			LLL : 'YYYY[ko] MMMM[ren] D[a] HH:mm',
			LLLL : 'dddd, YYYY[ko] MMMM[ren] D[a] HH:mm',
			l : 'YYYY-M-D',
			ll : 'YYYY[ko] MMM D[a]',
			lll : 'YYYY[ko] MMM D[a] HH:mm',
			llll : 'ddd, YYYY[ko] MMM D[a] HH:mm'
		},
		calendar : {
			sameDay : '[gaur] LT[etan]',
			nextDay : '[bihar] LT[etan]',
			nextWeek : 'dddd LT[etan]',
			lastDay : '[atzo] LT[etan]',
			lastWeek : '[aurreko] dddd LT[etan]',
			sameElse : 'L'
		},
		relativeTime : {
			future : '%s barru',
			past : 'duela %s',
			s : 'segundo batzuk',
			ss : '%d segundo',
			m : 'minutu bat',
			mm : '%d minutu',
			h : 'ordu bat',
			hh : '%d ordu',
			d : 'egun bat',
			dd : '%d egun',
			M : 'hilabete bat',
			MM : '%d hilabete',
			y : 'urte bat',
			yy : '%d urte'
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var symbolMap$5 = {
		'1': '۱',
		'2': '۲',
		'3': '۳',
		'4': '۴',
		'5': '۵',
		'6': '۶',
		'7': '۷',
		'8': '۸',
		'9': '۹',
		'0': '۰'
	}, numberMap$4 = {
		'۱': '1',
		'۲': '2',
		'۳': '3',
		'۴': '4',
		'۵': '5',
		'۶': '6',
		'۷': '7',
		'۸': '8',
		'۹': '9',
		'۰': '0'
	};

	hooks.defineLocale('fa', {
		months : 'ژانویه_فوریه_مارس_آوریل_مه_ژوئن_ژوئیه_اوت_سپتامبر_اکتبر_نوامبر_دسامبر'.split('_'),
		monthsShort : 'ژانویه_فوریه_مارس_آوریل_مه_ژوئن_ژوئیه_اوت_سپتامبر_اکتبر_نوامبر_دسامبر'.split('_'),
		weekdays : 'یک\u200cشنبه_دوشنبه_سه\u200cشنبه_چهارشنبه_پنج\u200cشنبه_جمعه_شنبه'.split('_'),
		weekdaysShort : 'یک\u200cشنبه_دوشنبه_سه\u200cشنبه_چهارشنبه_پنج\u200cشنبه_جمعه_شنبه'.split('_'),
		weekdaysMin : 'ی_د_س_چ_پ_ج_ش'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd, D MMMM YYYY HH:mm'
		},
		meridiemParse: /قبل از ظهر|بعد از ظهر/,
		isPM: function (input) {
			return /بعد از ظهر/.test(input);
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 12) {
				return 'قبل از ظهر';
			} else {
				return 'بعد از ظهر';
			}
		},
		calendar : {
			sameDay : '[امروز ساعت] LT',
			nextDay : '[فردا ساعت] LT',
			nextWeek : 'dddd [ساعت] LT',
			lastDay : '[دیروز ساعت] LT',
			lastWeek : 'dddd [پیش] [ساعت] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'در %s',
			past : '%s پیش',
			s : 'چند ثانیه',
			ss : 'ثانیه d%',
			m : 'یک دقیقه',
			mm : '%d دقیقه',
			h : 'یک ساعت',
			hh : '%d ساعت',
			d : 'یک روز',
			dd : '%d روز',
			M : 'یک ماه',
			MM : '%d ماه',
			y : 'یک سال',
			yy : '%d سال'
		},
		preparse: function (string) {
			return string.replace(/[۰-۹]/g, function (match) {
				return numberMap$4[match];
			}).replace(/،/g, ',');
		},
		postformat: function (string) {
			return string.replace(/\d/g, function (match) {
				return symbolMap$5[match];
			}).replace(/,/g, '،');
		},
		dayOfMonthOrdinalParse: /\d{1,2}م/,
		ordinal : '%dم',
		week : {
			dow : 6, // Saturday is the first day of the week.
			doy : 12 // The week that contains Jan 12th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var numbersPast = 'nolla yksi kaksi kolme neljä viisi kuusi seitsemän kahdeksan yhdeksän'.split(' '),
		numbersFuture = [
			'nolla', 'yhden', 'kahden', 'kolmen', 'neljän', 'viiden', 'kuuden',
			numbersPast[7], numbersPast[8], numbersPast[9]
		];
	function translate$2(number, withoutSuffix, key, isFuture) {
		var result = '';
		switch (key) {
			case 's':
				return isFuture ? 'muutaman sekunnin' : 'muutama sekunti';
			case 'ss':
				return isFuture ? 'sekunnin' : 'sekuntia';
			case 'm':
				return isFuture ? 'minuutin' : 'minuutti';
			case 'mm':
				result = isFuture ? 'minuutin' : 'minuuttia';
				break;
			case 'h':
				return isFuture ? 'tunnin' : 'tunti';
			case 'hh':
				result = isFuture ? 'tunnin' : 'tuntia';
				break;
			case 'd':
				return isFuture ? 'päivän' : 'päivä';
			case 'dd':
				result = isFuture ? 'päivän' : 'päivää';
				break;
			case 'M':
				return isFuture ? 'kuukauden' : 'kuukausi';
			case 'MM':
				result = isFuture ? 'kuukauden' : 'kuukautta';
				break;
			case 'y':
				return isFuture ? 'vuoden' : 'vuosi';
			case 'yy':
				result = isFuture ? 'vuoden' : 'vuotta';
				break;
		}
		result = verbalNumber(number, isFuture) + ' ' + result;
		return result;
	}
	function verbalNumber(number, isFuture) {
		return number < 10 ? (isFuture ? numbersFuture[number] : numbersPast[number]) : number;
	}

	hooks.defineLocale('fi', {
		months : 'tammikuu_helmikuu_maaliskuu_huhtikuu_toukokuu_kesäkuu_heinäkuu_elokuu_syyskuu_lokakuu_marraskuu_joulukuu'.split('_'),
		monthsShort : 'tammi_helmi_maalis_huhti_touko_kesä_heinä_elo_syys_loka_marras_joulu'.split('_'),
		weekdays : 'sunnuntai_maanantai_tiistai_keskiviikko_torstai_perjantai_lauantai'.split('_'),
		weekdaysShort : 'su_ma_ti_ke_to_pe_la'.split('_'),
		weekdaysMin : 'su_ma_ti_ke_to_pe_la'.split('_'),
		longDateFormat : {
			LT : 'HH.mm',
			LTS : 'HH.mm.ss',
			L : 'DD.MM.YYYY',
			LL : 'Do MMMM[ta] YYYY',
			LLL : 'Do MMMM[ta] YYYY, [klo] HH.mm',
			LLLL : 'dddd, Do MMMM[ta] YYYY, [klo] HH.mm',
			l : 'D.M.YYYY',
			ll : 'Do MMM YYYY',
			lll : 'Do MMM YYYY, [klo] HH.mm',
			llll : 'ddd, Do MMM YYYY, [klo] HH.mm'
		},
		calendar : {
			sameDay : '[tänään] [klo] LT',
			nextDay : '[huomenna] [klo] LT',
			nextWeek : 'dddd [klo] LT',
			lastDay : '[eilen] [klo] LT',
			lastWeek : '[viime] dddd[na] [klo] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : '%s päästä',
			past : '%s sitten',
			s : translate$2,
			ss : translate$2,
			m : translate$2,
			mm : translate$2,
			h : translate$2,
			hh : translate$2,
			d : translate$2,
			dd : translate$2,
			M : translate$2,
			MM : translate$2,
			y : translate$2,
			yy : translate$2
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('fo', {
		months : 'januar_februar_mars_apríl_mai_juni_juli_august_september_oktober_november_desember'.split('_'),
		monthsShort : 'jan_feb_mar_apr_mai_jun_jul_aug_sep_okt_nov_des'.split('_'),
		weekdays : 'sunnudagur_mánadagur_týsdagur_mikudagur_hósdagur_fríggjadagur_leygardagur'.split('_'),
		weekdaysShort : 'sun_mán_týs_mik_hós_frí_ley'.split('_'),
		weekdaysMin : 'su_má_tý_mi_hó_fr_le'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd D. MMMM, YYYY HH:mm'
		},
		calendar : {
			sameDay : '[Í dag kl.] LT',
			nextDay : '[Í morgin kl.] LT',
			nextWeek : 'dddd [kl.] LT',
			lastDay : '[Í gjár kl.] LT',
			lastWeek : '[síðstu] dddd [kl] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'um %s',
			past : '%s síðani',
			s : 'fá sekund',
			ss : '%d sekundir',
			m : 'ein minuttur',
			mm : '%d minuttir',
			h : 'ein tími',
			hh : '%d tímar',
			d : 'ein dagur',
			dd : '%d dagar',
			M : 'ein mánaður',
			MM : '%d mánaðir',
			y : 'eitt ár',
			yy : '%d ár'
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('fr-ca', {
		months : 'janvier_février_mars_avril_mai_juin_juillet_août_septembre_octobre_novembre_décembre'.split('_'),
		monthsShort : 'janv._févr._mars_avr._mai_juin_juil._août_sept._oct._nov._déc.'.split('_'),
		monthsParseExact : true,
		weekdays : 'dimanche_lundi_mardi_mercredi_jeudi_vendredi_samedi'.split('_'),
		weekdaysShort : 'dim._lun._mar._mer._jeu._ven._sam.'.split('_'),
		weekdaysMin : 'di_lu_ma_me_je_ve_sa'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'YYYY-MM-DD',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay : '[Aujourd’hui à] LT',
			nextDay : '[Demain à] LT',
			nextWeek : 'dddd [à] LT',
			lastDay : '[Hier à] LT',
			lastWeek : 'dddd [dernier à] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'dans %s',
			past : 'il y a %s',
			s : 'quelques secondes',
			ss : '%d secondes',
			m : 'une minute',
			mm : '%d minutes',
			h : 'une heure',
			hh : '%d heures',
			d : 'un jour',
			dd : '%d jours',
			M : 'un mois',
			MM : '%d mois',
			y : 'un an',
			yy : '%d ans'
		},
		dayOfMonthOrdinalParse: /\d{1,2}(er|e)/,
		ordinal : function (number, period) {
			switch (period) {
				// Words with masculine grammatical gender: mois, trimestre, jour
				default:
				case 'M':
				case 'Q':
				case 'D':
				case 'DDD':
				case 'd':
					return number + (number === 1 ? 'er' : 'e');

				// Words with feminine grammatical gender: semaine
				case 'w':
				case 'W':
					return number + (number === 1 ? 're' : 'e');
			}
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('fr-ch', {
		months : 'janvier_février_mars_avril_mai_juin_juillet_août_septembre_octobre_novembre_décembre'.split('_'),
		monthsShort : 'janv._févr._mars_avr._mai_juin_juil._août_sept._oct._nov._déc.'.split('_'),
		monthsParseExact : true,
		weekdays : 'dimanche_lundi_mardi_mercredi_jeudi_vendredi_samedi'.split('_'),
		weekdaysShort : 'dim._lun._mar._mer._jeu._ven._sam.'.split('_'),
		weekdaysMin : 'di_lu_ma_me_je_ve_sa'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay : '[Aujourd’hui à] LT',
			nextDay : '[Demain à] LT',
			nextWeek : 'dddd [à] LT',
			lastDay : '[Hier à] LT',
			lastWeek : 'dddd [dernier à] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'dans %s',
			past : 'il y a %s',
			s : 'quelques secondes',
			ss : '%d secondes',
			m : 'une minute',
			mm : '%d minutes',
			h : 'une heure',
			hh : '%d heures',
			d : 'un jour',
			dd : '%d jours',
			M : 'un mois',
			MM : '%d mois',
			y : 'un an',
			yy : '%d ans'
		},
		dayOfMonthOrdinalParse: /\d{1,2}(er|e)/,
		ordinal : function (number, period) {
			switch (period) {
				// Words with masculine grammatical gender: mois, trimestre, jour
				default:
				case 'M':
				case 'Q':
				case 'D':
				case 'DDD':
				case 'd':
					return number + (number === 1 ? 'er' : 'e');

				// Words with feminine grammatical gender: semaine
				case 'w':
				case 'W':
					return number + (number === 1 ? 're' : 'e');
			}
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('fr', {
		months : 'janvier_février_mars_avril_mai_juin_juillet_août_septembre_octobre_novembre_décembre'.split('_'),
		monthsShort : 'janv._févr._mars_avr._mai_juin_juil._août_sept._oct._nov._déc.'.split('_'),
		monthsParseExact : true,
		weekdays : 'dimanche_lundi_mardi_mercredi_jeudi_vendredi_samedi'.split('_'),
		weekdaysShort : 'dim._lun._mar._mer._jeu._ven._sam.'.split('_'),
		weekdaysMin : 'di_lu_ma_me_je_ve_sa'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay : '[Aujourd’hui à] LT',
			nextDay : '[Demain à] LT',
			nextWeek : 'dddd [à] LT',
			lastDay : '[Hier à] LT',
			lastWeek : 'dddd [dernier à] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'dans %s',
			past : 'il y a %s',
			s : 'quelques secondes',
			ss : '%d secondes',
			m : 'une minute',
			mm : '%d minutes',
			h : 'une heure',
			hh : '%d heures',
			d : 'un jour',
			dd : '%d jours',
			M : 'un mois',
			MM : '%d mois',
			y : 'un an',
			yy : '%d ans'
		},
		dayOfMonthOrdinalParse: /\d{1,2}(er|)/,
		ordinal : function (number, period) {
			switch (period) {
				// TODO: Return 'e' when day of month > 1. Move this case inside
				// block for masculine words below.
				// See https://github.com/moment/moment/issues/3375
				case 'D':
					return number + (number === 1 ? 'er' : '');

				// Words with masculine grammatical gender: mois, trimestre, jour
				default:
				case 'M':
				case 'Q':
				case 'DDD':
				case 'd':
					return number + (number === 1 ? 'er' : 'e');

				// Words with feminine grammatical gender: semaine
				case 'w':
				case 'W':
					return number + (number === 1 ? 're' : 'e');
			}
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var monthsShortWithDots = 'jan._feb._mrt._apr._mai_jun._jul._aug._sep._okt._nov._des.'.split('_'),
		monthsShortWithoutDots = 'jan_feb_mrt_apr_mai_jun_jul_aug_sep_okt_nov_des'.split('_');

	hooks.defineLocale('fy', {
		months : 'jannewaris_febrewaris_maart_april_maaie_juny_july_augustus_septimber_oktober_novimber_desimber'.split('_'),
		monthsShort : function (m, format) {
			if (!m) {
				return monthsShortWithDots;
			} else if (/-MMM-/.test(format)) {
				return monthsShortWithoutDots[m.month()];
			} else {
				return monthsShortWithDots[m.month()];
			}
		},
		monthsParseExact : true,
		weekdays : 'snein_moandei_tiisdei_woansdei_tongersdei_freed_sneon'.split('_'),
		weekdaysShort : 'si._mo._ti._wo._to._fr._so.'.split('_'),
		weekdaysMin : 'Si_Mo_Ti_Wo_To_Fr_So'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD-MM-YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay: '[hjoed om] LT',
			nextDay: '[moarn om] LT',
			nextWeek: 'dddd [om] LT',
			lastDay: '[juster om] LT',
			lastWeek: '[ôfrûne] dddd [om] LT',
			sameElse: 'L'
		},
		relativeTime : {
			future : 'oer %s',
			past : '%s lyn',
			s : 'in pear sekonden',
			ss : '%d sekonden',
			m : 'ien minút',
			mm : '%d minuten',
			h : 'ien oere',
			hh : '%d oeren',
			d : 'ien dei',
			dd : '%d dagen',
			M : 'ien moanne',
			MM : '%d moannen',
			y : 'ien jier',
			yy : '%d jierren'
		},
		dayOfMonthOrdinalParse: /\d{1,2}(ste|de)/,
		ordinal : function (number) {
			return number + ((number === 1 || number === 8 || number >= 20) ? 'ste' : 'de');
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration


	var months$5 = [
		'Eanáir', 'Feabhra', 'Márta', 'Aibreán', 'Bealtaine', 'Méitheamh', 'Iúil', 'Lúnasa', 'Meán Fómhair', 'Deaireadh Fómhair', 'Samhain', 'Nollaig'
	];

	var monthsShort$4 = ['Eaná', 'Feab', 'Márt', 'Aibr', 'Beal', 'Méit', 'Iúil', 'Lúna', 'Meán', 'Deai', 'Samh', 'Noll'];

	var weekdays$1 = ['Dé Domhnaigh', 'Dé Luain', 'Dé Máirt', 'Dé Céadaoin', 'Déardaoin', 'Dé hAoine', 'Dé Satharn'];

	var weekdaysShort = ['Dom', 'Lua', 'Mái', 'Céa', 'Déa', 'hAo', 'Sat'];

	var weekdaysMin = ['Do', 'Lu', 'Má', 'Ce', 'Dé', 'hA', 'Sa'];

	hooks.defineLocale('ga', {
		months: months$5,
		monthsShort: monthsShort$4,
		monthsParseExact: true,
		weekdays: weekdays$1,
		weekdaysShort: weekdaysShort,
		weekdaysMin: weekdaysMin,
		longDateFormat: {
			LT: 'HH:mm',
			LTS: 'HH:mm:ss',
			L: 'DD/MM/YYYY',
			LL: 'D MMMM YYYY',
			LLL: 'D MMMM YYYY HH:mm',
			LLLL: 'dddd, D MMMM YYYY HH:mm'
		},
		calendar: {
			sameDay: '[Inniu ag] LT',
			nextDay: '[Amárach ag] LT',
			nextWeek: 'dddd [ag] LT',
			lastDay: '[Inné aig] LT',
			lastWeek: 'dddd [seo caite] [ag] LT',
			sameElse: 'L'
		},
		relativeTime: {
			future: 'i %s',
			past: '%s ó shin',
			s: 'cúpla soicind',
			ss: '%d soicind',
			m: 'nóiméad',
			mm: '%d nóiméad',
			h: 'uair an chloig',
			hh: '%d uair an chloig',
			d: 'lá',
			dd: '%d lá',
			M: 'mí',
			MM: '%d mí',
			y: 'bliain',
			yy: '%d bliain'
		},
		dayOfMonthOrdinalParse: /\d{1,2}(d|na|mh)/,
		ordinal: function (number) {
			var output = number === 1 ? 'd' : number % 10 === 2 ? 'na' : 'mh';
			return number + output;
		},
		week: {
			dow: 1, // Monday is the first day of the week.
			doy: 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var months$6 = [
		'Am Faoilleach', 'An Gearran', 'Am Màrt', 'An Giblean', 'An Cèitean', 'An t-Ògmhios', 'An t-Iuchar', 'An Lùnastal', 'An t-Sultain', 'An Dàmhair', 'An t-Samhain', 'An Dùbhlachd'
	];

	var monthsShort$5 = ['Faoi', 'Gear', 'Màrt', 'Gibl', 'Cèit', 'Ògmh', 'Iuch', 'Lùn', 'Sult', 'Dàmh', 'Samh', 'Dùbh'];

	var weekdays$2 = ['Didòmhnaich', 'Diluain', 'Dimàirt', 'Diciadain', 'Diardaoin', 'Dihaoine', 'Disathairne'];

	var weekdaysShort$1 = ['Did', 'Dil', 'Dim', 'Dic', 'Dia', 'Dih', 'Dis'];

	var weekdaysMin$1 = ['Dò', 'Lu', 'Mà', 'Ci', 'Ar', 'Ha', 'Sa'];

	hooks.defineLocale('gd', {
		months : months$6,
		monthsShort : monthsShort$5,
		monthsParseExact : true,
		weekdays : weekdays$2,
		weekdaysShort : weekdaysShort$1,
		weekdaysMin : weekdaysMin$1,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd, D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay : '[An-diugh aig] LT',
			nextDay : '[A-màireach aig] LT',
			nextWeek : 'dddd [aig] LT',
			lastDay : '[An-dè aig] LT',
			lastWeek : 'dddd [seo chaidh] [aig] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'ann an %s',
			past : 'bho chionn %s',
			s : 'beagan diogan',
			ss : '%d diogan',
			m : 'mionaid',
			mm : '%d mionaidean',
			h : 'uair',
			hh : '%d uairean',
			d : 'latha',
			dd : '%d latha',
			M : 'mìos',
			MM : '%d mìosan',
			y : 'bliadhna',
			yy : '%d bliadhna'
		},
		dayOfMonthOrdinalParse : /\d{1,2}(d|na|mh)/,
		ordinal : function (number) {
			var output = number === 1 ? 'd' : number % 10 === 2 ? 'na' : 'mh';
			return number + output;
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('gl', {
		months : 'xaneiro_febreiro_marzo_abril_maio_xuño_xullo_agosto_setembro_outubro_novembro_decembro'.split('_'),
		monthsShort : 'xan._feb._mar._abr._mai._xuñ._xul._ago._set._out._nov._dec.'.split('_'),
		monthsParseExact: true,
		weekdays : 'domingo_luns_martes_mércores_xoves_venres_sábado'.split('_'),
		weekdaysShort : 'dom._lun._mar._mér._xov._ven._sáb.'.split('_'),
		weekdaysMin : 'do_lu_ma_mé_xo_ve_sá'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'H:mm',
			LTS : 'H:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D [de] MMMM [de] YYYY',
			LLL : 'D [de] MMMM [de] YYYY H:mm',
			LLLL : 'dddd, D [de] MMMM [de] YYYY H:mm'
		},
		calendar : {
			sameDay : function () {
				return '[hoxe ' + ((this.hours() !== 1) ? 'ás' : 'á') + '] LT';
			},
			nextDay : function () {
				return '[mañá ' + ((this.hours() !== 1) ? 'ás' : 'á') + '] LT';
			},
			nextWeek : function () {
				return 'dddd [' + ((this.hours() !== 1) ? 'ás' : 'a') + '] LT';
			},
			lastDay : function () {
				return '[onte ' + ((this.hours() !== 1) ? 'á' : 'a') + '] LT';
			},
			lastWeek : function () {
				return '[o] dddd [pasado ' + ((this.hours() !== 1) ? 'ás' : 'a') + '] LT';
			},
			sameElse : 'L'
		},
		relativeTime : {
			future : function (str) {
				if (str.indexOf('un') === 0) {
					return 'n' + str;
				}
				return 'en ' + str;
			},
			past : 'hai %s',
			s : 'uns segundos',
			ss : '%d segundos',
			m : 'un minuto',
			mm : '%d minutos',
			h : 'unha hora',
			hh : '%d horas',
			d : 'un día',
			dd : '%d días',
			M : 'un mes',
			MM : '%d meses',
			y : 'un ano',
			yy : '%d anos'
		},
		dayOfMonthOrdinalParse : /\d{1,2}º/,
		ordinal : '%dº',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	function processRelativeTime$4(number, withoutSuffix, key, isFuture) {
		var format = {
			's': ['thodde secondanim', 'thodde second'],
			'ss': [number + ' secondanim', number + ' second'],
			'm': ['eka mintan', 'ek minute'],
			'mm': [number + ' mintanim', number + ' mintam'],
			'h': ['eka voran', 'ek vor'],
			'hh': [number + ' voranim', number + ' voram'],
			'd': ['eka disan', 'ek dis'],
			'dd': [number + ' disanim', number + ' dis'],
			'M': ['eka mhoinean', 'ek mhoino'],
			'MM': [number + ' mhoineanim', number + ' mhoine'],
			'y': ['eka vorsan', 'ek voros'],
			'yy': [number + ' vorsanim', number + ' vorsam']
		};
		return withoutSuffix ? format[key][0] : format[key][1];
	}

	hooks.defineLocale('gom-latn', {
		months : 'Janer_Febrer_Mars_Abril_Mai_Jun_Julai_Agost_Setembr_Otubr_Novembr_Dezembr'.split('_'),
		monthsShort : 'Jan._Feb._Mars_Abr._Mai_Jun_Jul._Ago._Set._Otu._Nov._Dez.'.split('_'),
		monthsParseExact : true,
		weekdays : 'Aitar_Somar_Mongllar_Budvar_Brestar_Sukrar_Son\'var'.split('_'),
		weekdaysShort : 'Ait._Som._Mon._Bud._Bre._Suk._Son.'.split('_'),
		weekdaysMin : 'Ai_Sm_Mo_Bu_Br_Su_Sn'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'A h:mm [vazta]',
			LTS : 'A h:mm:ss [vazta]',
			L : 'DD-MM-YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY A h:mm [vazta]',
			LLLL : 'dddd, MMMM[achea] Do, YYYY, A h:mm [vazta]',
			llll: 'ddd, D MMM YYYY, A h:mm [vazta]'
		},
		calendar : {
			sameDay: '[Aiz] LT',
			nextDay: '[Faleam] LT',
			nextWeek: '[Ieta to] dddd[,] LT',
			lastDay: '[Kal] LT',
			lastWeek: '[Fatlo] dddd[,] LT',
			sameElse: 'L'
		},
		relativeTime : {
			future : '%s',
			past : '%s adim',
			s : processRelativeTime$4,
			ss : processRelativeTime$4,
			m : processRelativeTime$4,
			mm : processRelativeTime$4,
			h : processRelativeTime$4,
			hh : processRelativeTime$4,
			d : processRelativeTime$4,
			dd : processRelativeTime$4,
			M : processRelativeTime$4,
			MM : processRelativeTime$4,
			y : processRelativeTime$4,
			yy : processRelativeTime$4
		},
		dayOfMonthOrdinalParse : /\d{1,2}(er)/,
		ordinal : function (number, period) {
			switch (period) {
				// the ordinal 'er' only applies to day of the month
				case 'D':
					return number + 'er';
				default:
				case 'M':
				case 'Q':
				case 'DDD':
				case 'd':
				case 'w':
				case 'W':
					return number;
			}
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		},
		meridiemParse: /rati|sokalli|donparam|sanje/,
		meridiemHour : function (hour, meridiem) {
			if (hour === 12) {
				hour = 0;
			}
			if (meridiem === 'rati') {
				return hour < 4 ? hour : hour + 12;
			} else if (meridiem === 'sokalli') {
				return hour;
			} else if (meridiem === 'donparam') {
				return hour > 12 ? hour : hour + 12;
			} else if (meridiem === 'sanje') {
				return hour + 12;
			}
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 4) {
				return 'rati';
			} else if (hour < 12) {
				return 'sokalli';
			} else if (hour < 16) {
				return 'donparam';
			} else if (hour < 20) {
				return 'sanje';
			} else {
				return 'rati';
			}
		}
	});

	//! moment.js locale configuration

	var symbolMap$6 = {
			'1': '૧',
			'2': '૨',
			'3': '૩',
			'4': '૪',
			'5': '૫',
			'6': '૬',
			'7': '૭',
			'8': '૮',
			'9': '૯',
			'0': '૦'
		},
		numberMap$5 = {
			'૧': '1',
			'૨': '2',
			'૩': '3',
			'૪': '4',
			'૫': '5',
			'૬': '6',
			'૭': '7',
			'૮': '8',
			'૯': '9',
			'૦': '0'
		};

	hooks.defineLocale('gu', {
		months: 'જાન્યુઆરી_ફેબ્રુઆરી_માર્ચ_એપ્રિલ_મે_જૂન_જુલાઈ_ઑગસ્ટ_સપ્ટેમ્બર_ઑક્ટ્બર_નવેમ્બર_ડિસેમ્બર'.split('_'),
		monthsShort: 'જાન્યુ._ફેબ્રુ._માર્ચ_એપ્રિ._મે_જૂન_જુલા._ઑગ._સપ્ટે._ઑક્ટ્._નવે._ડિસે.'.split('_'),
		monthsParseExact: true,
		weekdays: 'રવિવાર_સોમવાર_મંગળવાર_બુધ્વાર_ગુરુવાર_શુક્રવાર_શનિવાર'.split('_'),
		weekdaysShort: 'રવિ_સોમ_મંગળ_બુધ્_ગુરુ_શુક્ર_શનિ'.split('_'),
		weekdaysMin: 'ર_સો_મં_બુ_ગુ_શુ_શ'.split('_'),
		longDateFormat: {
			LT: 'A h:mm વાગ્યે',
			LTS: 'A h:mm:ss વાગ્યે',
			L: 'DD/MM/YYYY',
			LL: 'D MMMM YYYY',
			LLL: 'D MMMM YYYY, A h:mm વાગ્યે',
			LLLL: 'dddd, D MMMM YYYY, A h:mm વાગ્યે'
		},
		calendar: {
			sameDay: '[આજ] LT',
			nextDay: '[કાલે] LT',
			nextWeek: 'dddd, LT',
			lastDay: '[ગઇકાલે] LT',
			lastWeek: '[પાછલા] dddd, LT',
			sameElse: 'L'
		},
		relativeTime: {
			future: '%s મા',
			past: '%s પેહલા',
			s: 'અમુક પળો',
			ss: '%d સેકંડ',
			m: 'એક મિનિટ',
			mm: '%d મિનિટ',
			h: 'એક કલાક',
			hh: '%d કલાક',
			d: 'એક દિવસ',
			dd: '%d દિવસ',
			M: 'એક મહિનો',
			MM: '%d મહિનો',
			y: 'એક વર્ષ',
			yy: '%d વર્ષ'
		},
		preparse: function (string) {
			return string.replace(/[૧૨૩૪૫૬૭૮૯૦]/g, function (match) {
				return numberMap$5[match];
			});
		},
		postformat: function (string) {
			return string.replace(/\d/g, function (match) {
				return symbolMap$6[match];
			});
		},
		// Gujarati notation for meridiems are quite fuzzy in practice. While there exists
		// a rigid notion of a 'Pahar' it is not used as rigidly in modern Gujarati.
		meridiemParse: /રાત|બપોર|સવાર|સાંજ/,
		meridiemHour: function (hour, meridiem) {
			if (hour === 12) {
				hour = 0;
			}
			if (meridiem === 'રાત') {
				return hour < 4 ? hour : hour + 12;
			} else if (meridiem === 'સવાર') {
				return hour;
			} else if (meridiem === 'બપોર') {
				return hour >= 10 ? hour : hour + 12;
			} else if (meridiem === 'સાંજ') {
				return hour + 12;
			}
		},
		meridiem: function (hour, minute, isLower) {
			if (hour < 4) {
				return 'રાત';
			} else if (hour < 10) {
				return 'સવાર';
			} else if (hour < 17) {
				return 'બપોર';
			} else if (hour < 20) {
				return 'સાંજ';
			} else {
				return 'રાત';
			}
		},
		week: {
			dow: 0, // Sunday is the first day of the week.
			doy: 6 // The week that contains Jan 6th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('he', {
		months : 'ינואר_פברואר_מרץ_אפריל_מאי_יוני_יולי_אוגוסט_ספטמבר_אוקטובר_נובמבר_דצמבר'.split('_'),
		monthsShort : 'ינו׳_פבר׳_מרץ_אפר׳_מאי_יוני_יולי_אוג׳_ספט׳_אוק׳_נוב׳_דצמ׳'.split('_'),
		weekdays : 'ראשון_שני_שלישי_רביעי_חמישי_שישי_שבת'.split('_'),
		weekdaysShort : 'א׳_ב׳_ג׳_ד׳_ה׳_ו׳_ש׳'.split('_'),
		weekdaysMin : 'א_ב_ג_ד_ה_ו_ש'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D [ב]MMMM YYYY',
			LLL : 'D [ב]MMMM YYYY HH:mm',
			LLLL : 'dddd, D [ב]MMMM YYYY HH:mm',
			l : 'D/M/YYYY',
			ll : 'D MMM YYYY',
			lll : 'D MMM YYYY HH:mm',
			llll : 'ddd, D MMM YYYY HH:mm'
		},
		calendar : {
			sameDay : '[היום ב־]LT',
			nextDay : '[מחר ב־]LT',
			nextWeek : 'dddd [בשעה] LT',
			lastDay : '[אתמול ב־]LT',
			lastWeek : '[ביום] dddd [האחרון בשעה] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'בעוד %s',
			past : 'לפני %s',
			s : 'מספר שניות',
			ss : '%d שניות',
			m : 'דקה',
			mm : '%d דקות',
			h : 'שעה',
			hh : function (number) {
				if (number === 2) {
					return 'שעתיים';
				}
				return number + ' שעות';
			},
			d : 'יום',
			dd : function (number) {
				if (number === 2) {
					return 'יומיים';
				}
				return number + ' ימים';
			},
			M : 'חודש',
			MM : function (number) {
				if (number === 2) {
					return 'חודשיים';
				}
				return number + ' חודשים';
			},
			y : 'שנה',
			yy : function (number) {
				if (number === 2) {
					return 'שנתיים';
				} else if (number % 10 === 0 && number !== 10) {
					return number + ' שנה';
				}
				return number + ' שנים';
			}
		},
		meridiemParse: /אחה"צ|לפנה"צ|אחרי הצהריים|לפני הצהריים|לפנות בוקר|בבוקר|בערב/i,
		isPM : function (input) {
			return /^(אחה"צ|אחרי הצהריים|בערב)$/.test(input);
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 5) {
				return 'לפנות בוקר';
			} else if (hour < 10) {
				return 'בבוקר';
			} else if (hour < 12) {
				return isLower ? 'לפנה"צ' : 'לפני הצהריים';
			} else if (hour < 18) {
				return isLower ? 'אחה"צ' : 'אחרי הצהריים';
			} else {
				return 'בערב';
			}
		}
	});

	//! moment.js locale configuration

	var symbolMap$7 = {
			'1': '१',
			'2': '२',
			'3': '३',
			'4': '४',
			'5': '५',
			'6': '६',
			'7': '७',
			'8': '८',
			'9': '९',
			'0': '०'
		},
		numberMap$6 = {
			'१': '1',
			'२': '2',
			'३': '3',
			'४': '4',
			'५': '5',
			'६': '6',
			'७': '7',
			'८': '8',
			'९': '9',
			'०': '0'
		};

	hooks.defineLocale('hi', {
		months : 'जनवरी_फ़रवरी_मार्च_अप्रैल_मई_जून_जुलाई_अगस्त_सितम्बर_अक्टूबर_नवम्बर_दिसम्बर'.split('_'),
		monthsShort : 'जन._फ़र._मार्च_अप्रै._मई_जून_जुल._अग._सित._अक्टू._नव._दिस.'.split('_'),
		monthsParseExact: true,
		weekdays : 'रविवार_सोमवार_मंगलवार_बुधवार_गुरूवार_शुक्रवार_शनिवार'.split('_'),
		weekdaysShort : 'रवि_सोम_मंगल_बुध_गुरू_शुक्र_शनि'.split('_'),
		weekdaysMin : 'र_सो_मं_बु_गु_शु_श'.split('_'),
		longDateFormat : {
			LT : 'A h:mm बजे',
			LTS : 'A h:mm:ss बजे',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY, A h:mm बजे',
			LLLL : 'dddd, D MMMM YYYY, A h:mm बजे'
		},
		calendar : {
			sameDay : '[आज] LT',
			nextDay : '[कल] LT',
			nextWeek : 'dddd, LT',
			lastDay : '[कल] LT',
			lastWeek : '[पिछले] dddd, LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : '%s में',
			past : '%s पहले',
			s : 'कुछ ही क्षण',
			ss : '%d सेकंड',
			m : 'एक मिनट',
			mm : '%d मिनट',
			h : 'एक घंटा',
			hh : '%d घंटे',
			d : 'एक दिन',
			dd : '%d दिन',
			M : 'एक महीने',
			MM : '%d महीने',
			y : 'एक वर्ष',
			yy : '%d वर्ष'
		},
		preparse: function (string) {
			return string.replace(/[१२३४५६७८९०]/g, function (match) {
				return numberMap$6[match];
			});
		},
		postformat: function (string) {
			return string.replace(/\d/g, function (match) {
				return symbolMap$7[match];
			});
		},
		// Hindi notation for meridiems are quite fuzzy in practice. While there exists
		// a rigid notion of a 'Pahar' it is not used as rigidly in modern Hindi.
		meridiemParse: /रात|सुबह|दोपहर|शाम/,
		meridiemHour : function (hour, meridiem) {
			if (hour === 12) {
				hour = 0;
			}
			if (meridiem === 'रात') {
				return hour < 4 ? hour : hour + 12;
			} else if (meridiem === 'सुबह') {
				return hour;
			} else if (meridiem === 'दोपहर') {
				return hour >= 10 ? hour : hour + 12;
			} else if (meridiem === 'शाम') {
				return hour + 12;
			}
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 4) {
				return 'रात';
			} else if (hour < 10) {
				return 'सुबह';
			} else if (hour < 17) {
				return 'दोपहर';
			} else if (hour < 20) {
				return 'शाम';
			} else {
				return 'रात';
			}
		},
		week : {
			dow : 0, // Sunday is the first day of the week.
			doy : 6  // The week that contains Jan 6th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	function translate$3(number, withoutSuffix, key) {
		var result = number + ' ';
		switch (key) {
			case 'ss':
				if (number === 1) {
					result += 'sekunda';
				} else if (number === 2 || number === 3 || number === 4) {
					result += 'sekunde';
				} else {
					result += 'sekundi';
				}
				return result;
			case 'm':
				return withoutSuffix ? 'jedna minuta' : 'jedne minute';
			case 'mm':
				if (number === 1) {
					result += 'minuta';
				} else if (number === 2 || number === 3 || number === 4) {
					result += 'minute';
				} else {
					result += 'minuta';
				}
				return result;
			case 'h':
				return withoutSuffix ? 'jedan sat' : 'jednog sata';
			case 'hh':
				if (number === 1) {
					result += 'sat';
				} else if (number === 2 || number === 3 || number === 4) {
					result += 'sata';
				} else {
					result += 'sati';
				}
				return result;
			case 'dd':
				if (number === 1) {
					result += 'dan';
				} else {
					result += 'dana';
				}
				return result;
			case 'MM':
				if (number === 1) {
					result += 'mjesec';
				} else if (number === 2 || number === 3 || number === 4) {
					result += 'mjeseca';
				} else {
					result += 'mjeseci';
				}
				return result;
			case 'yy':
				if (number === 1) {
					result += 'godina';
				} else if (number === 2 || number === 3 || number === 4) {
					result += 'godine';
				} else {
					result += 'godina';
				}
				return result;
		}
	}

	hooks.defineLocale('hr', {
		months : {
			format: 'siječnja_veljače_ožujka_travnja_svibnja_lipnja_srpnja_kolovoza_rujna_listopada_studenoga_prosinca'.split('_'),
			standalone: 'siječanj_veljača_ožujak_travanj_svibanj_lipanj_srpanj_kolovoz_rujan_listopad_studeni_prosinac'.split('_')
		},
		monthsShort : 'sij._velj._ožu._tra._svi._lip._srp._kol._ruj._lis._stu._pro.'.split('_'),
		monthsParseExact: true,
		weekdays : 'nedjelja_ponedjeljak_utorak_srijeda_četvrtak_petak_subota'.split('_'),
		weekdaysShort : 'ned._pon._uto._sri._čet._pet._sub.'.split('_'),
		weekdaysMin : 'ne_po_ut_sr_če_pe_su'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'H:mm',
			LTS : 'H:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D. MMMM YYYY',
			LLL : 'D. MMMM YYYY H:mm',
			LLLL : 'dddd, D. MMMM YYYY H:mm'
		},
		calendar : {
			sameDay  : '[danas u] LT',
			nextDay  : '[sutra u] LT',
			nextWeek : function () {
				switch (this.day()) {
					case 0:
						return '[u] [nedjelju] [u] LT';
					case 3:
						return '[u] [srijedu] [u] LT';
					case 6:
						return '[u] [subotu] [u] LT';
					case 1:
					case 2:
					case 4:
					case 5:
						return '[u] dddd [u] LT';
				}
			},
			lastDay  : '[jučer u] LT',
			lastWeek : function () {
				switch (this.day()) {
					case 0:
					case 3:
						return '[prošlu] dddd [u] LT';
					case 6:
						return '[prošle] [subote] [u] LT';
					case 1:
					case 2:
					case 4:
					case 5:
						return '[prošli] dddd [u] LT';
				}
			},
			sameElse : 'L'
		},
		relativeTime : {
			future : 'za %s',
			past   : 'prije %s',
			s      : 'par sekundi',
			ss     : translate$3,
			m      : translate$3,
			mm     : translate$3,
			h      : translate$3,
			hh     : translate$3,
			d      : 'dan',
			dd     : translate$3,
			M      : 'mjesec',
			MM     : translate$3,
			y      : 'godinu',
			yy     : translate$3
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var weekEndings = 'vasárnap hétfőn kedden szerdán csütörtökön pénteken szombaton'.split(' ');
	function translate$4(number, withoutSuffix, key, isFuture) {
		var num = number;
		switch (key) {
			case 's':
				return (isFuture || withoutSuffix) ? 'néhány másodperc' : 'néhány másodperce';
			case 'ss':
				return num + (isFuture || withoutSuffix) ? ' másodperc' : ' másodperce';
			case 'm':
				return 'egy' + (isFuture || withoutSuffix ? ' perc' : ' perce');
			case 'mm':
				return num + (isFuture || withoutSuffix ? ' perc' : ' perce');
			case 'h':
				return 'egy' + (isFuture || withoutSuffix ? ' óra' : ' órája');
			case 'hh':
				return num + (isFuture || withoutSuffix ? ' óra' : ' órája');
			case 'd':
				return 'egy' + (isFuture || withoutSuffix ? ' nap' : ' napja');
			case 'dd':
				return num + (isFuture || withoutSuffix ? ' nap' : ' napja');
			case 'M':
				return 'egy' + (isFuture || withoutSuffix ? ' hónap' : ' hónapja');
			case 'MM':
				return num + (isFuture || withoutSuffix ? ' hónap' : ' hónapja');
			case 'y':
				return 'egy' + (isFuture || withoutSuffix ? ' év' : ' éve');
			case 'yy':
				return num + (isFuture || withoutSuffix ? ' év' : ' éve');
		}
		return '';
	}
	function week(isFuture) {
		return (isFuture ? '' : '[múlt] ') + '[' + weekEndings[this.day()] + '] LT[-kor]';
	}

	hooks.defineLocale('hu', {
		months : 'január_február_március_április_május_június_július_augusztus_szeptember_október_november_december'.split('_'),
		monthsShort : 'jan_feb_márc_ápr_máj_jún_júl_aug_szept_okt_nov_dec'.split('_'),
		weekdays : 'vasárnap_hétfő_kedd_szerda_csütörtök_péntek_szombat'.split('_'),
		weekdaysShort : 'vas_hét_kedd_sze_csüt_pén_szo'.split('_'),
		weekdaysMin : 'v_h_k_sze_cs_p_szo'.split('_'),
		longDateFormat : {
			LT : 'H:mm',
			LTS : 'H:mm:ss',
			L : 'YYYY.MM.DD.',
			LL : 'YYYY. MMMM D.',
			LLL : 'YYYY. MMMM D. H:mm',
			LLLL : 'YYYY. MMMM D., dddd H:mm'
		},
		meridiemParse: /de|du/i,
		isPM: function (input) {
			return input.charAt(1).toLowerCase() === 'u';
		},
		meridiem : function (hours, minutes, isLower) {
			if (hours < 12) {
				return isLower === true ? 'de' : 'DE';
			} else {
				return isLower === true ? 'du' : 'DU';
			}
		},
		calendar : {
			sameDay : '[ma] LT[-kor]',
			nextDay : '[holnap] LT[-kor]',
			nextWeek : function () {
				return week.call(this, true);
			},
			lastDay : '[tegnap] LT[-kor]',
			lastWeek : function () {
				return week.call(this, false);
			},
			sameElse : 'L'
		},
		relativeTime : {
			future : '%s múlva',
			past : '%s',
			s : translate$4,
			ss : translate$4,
			m : translate$4,
			mm : translate$4,
			h : translate$4,
			hh : translate$4,
			d : translate$4,
			dd : translate$4,
			M : translate$4,
			MM : translate$4,
			y : translate$4,
			yy : translate$4
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('hy-am', {
		months : {
			format: 'հունվարի_փետրվարի_մարտի_ապրիլի_մայիսի_հունիսի_հուլիսի_օգոստոսի_սեպտեմբերի_հոկտեմբերի_նոյեմբերի_դեկտեմբերի'.split('_'),
			standalone: 'հունվար_փետրվար_մարտ_ապրիլ_մայիս_հունիս_հուլիս_օգոստոս_սեպտեմբեր_հոկտեմբեր_նոյեմբեր_դեկտեմբեր'.split('_')
		},
		monthsShort : 'հնվ_փտր_մրտ_ապր_մյս_հնս_հլս_օգս_սպտ_հկտ_նմբ_դկտ'.split('_'),
		weekdays : 'կիրակի_երկուշաբթի_երեքշաբթի_չորեքշաբթի_հինգշաբթի_ուրբաթ_շաբաթ'.split('_'),
		weekdaysShort : 'կրկ_երկ_երք_չրք_հնգ_ուրբ_շբթ'.split('_'),
		weekdaysMin : 'կրկ_երկ_երք_չրք_հնգ_ուրբ_շբթ'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D MMMM YYYY թ.',
			LLL : 'D MMMM YYYY թ., HH:mm',
			LLLL : 'dddd, D MMMM YYYY թ., HH:mm'
		},
		calendar : {
			sameDay: '[այսօր] LT',
			nextDay: '[վաղը] LT',
			lastDay: '[երեկ] LT',
			nextWeek: function () {
				return 'dddd [օրը ժամը] LT';
			},
			lastWeek: function () {
				return '[անցած] dddd [օրը ժամը] LT';
			},
			sameElse: 'L'
		},
		relativeTime : {
			future : '%s հետո',
			past : '%s առաջ',
			s : 'մի քանի վայրկյան',
			ss : '%d վայրկյան',
			m : 'րոպե',
			mm : '%d րոպե',
			h : 'ժամ',
			hh : '%d ժամ',
			d : 'օր',
			dd : '%d օր',
			M : 'ամիս',
			MM : '%d ամիս',
			y : 'տարի',
			yy : '%d տարի'
		},
		meridiemParse: /գիշերվա|առավոտվա|ցերեկվա|երեկոյան/,
		isPM: function (input) {
			return /^(ցերեկվա|երեկոյան)$/.test(input);
		},
		meridiem : function (hour) {
			if (hour < 4) {
				return 'գիշերվա';
			} else if (hour < 12) {
				return 'առավոտվա';
			} else if (hour < 17) {
				return 'ցերեկվա';
			} else {
				return 'երեկոյան';
			}
		},
		dayOfMonthOrdinalParse: /\d{1,2}|\d{1,2}-(ին|րդ)/,
		ordinal: function (number, period) {
			switch (period) {
				case 'DDD':
				case 'w':
				case 'W':
				case 'DDDo':
					if (number === 1) {
						return number + '-ին';
					}
					return number + '-րդ';
				default:
					return number;
			}
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('id', {
		months : 'Januari_Februari_Maret_April_Mei_Juni_Juli_Agustus_September_Oktober_November_Desember'.split('_'),
		monthsShort : 'Jan_Feb_Mar_Apr_Mei_Jun_Jul_Agt_Sep_Okt_Nov_Des'.split('_'),
		weekdays : 'Minggu_Senin_Selasa_Rabu_Kamis_Jumat_Sabtu'.split('_'),
		weekdaysShort : 'Min_Sen_Sel_Rab_Kam_Jum_Sab'.split('_'),
		weekdaysMin : 'Mg_Sn_Sl_Rb_Km_Jm_Sb'.split('_'),
		longDateFormat : {
			LT : 'HH.mm',
			LTS : 'HH.mm.ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY [pukul] HH.mm',
			LLLL : 'dddd, D MMMM YYYY [pukul] HH.mm'
		},
		meridiemParse: /pagi|siang|sore|malam/,
		meridiemHour : function (hour, meridiem) {
			if (hour === 12) {
				hour = 0;
			}
			if (meridiem === 'pagi') {
				return hour;
			} else if (meridiem === 'siang') {
				return hour >= 11 ? hour : hour + 12;
			} else if (meridiem === 'sore' || meridiem === 'malam') {
				return hour + 12;
			}
		},
		meridiem : function (hours, minutes, isLower) {
			if (hours < 11) {
				return 'pagi';
			} else if (hours < 15) {
				return 'siang';
			} else if (hours < 19) {
				return 'sore';
			} else {
				return 'malam';
			}
		},
		calendar : {
			sameDay : '[Hari ini pukul] LT',
			nextDay : '[Besok pukul] LT',
			nextWeek : 'dddd [pukul] LT',
			lastDay : '[Kemarin pukul] LT',
			lastWeek : 'dddd [lalu pukul] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'dalam %s',
			past : '%s yang lalu',
			s : 'beberapa detik',
			ss : '%d detik',
			m : 'semenit',
			mm : '%d menit',
			h : 'sejam',
			hh : '%d jam',
			d : 'sehari',
			dd : '%d hari',
			M : 'sebulan',
			MM : '%d bulan',
			y : 'setahun',
			yy : '%d tahun'
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	function plural$2(n) {
		if (n % 100 === 11) {
			return true;
		} else if (n % 10 === 1) {
			return false;
		}
		return true;
	}
	function translate$5(number, withoutSuffix, key, isFuture) {
		var result = number + ' ';
		switch (key) {
			case 's':
				return withoutSuffix || isFuture ? 'nokkrar sekúndur' : 'nokkrum sekúndum';
			case 'ss':
				if (plural$2(number)) {
					return result + (withoutSuffix || isFuture ? 'sekúndur' : 'sekúndum');
				}
				return result + 'sekúnda';
			case 'm':
				return withoutSuffix ? 'mínúta' : 'mínútu';
			case 'mm':
				if (plural$2(number)) {
					return result + (withoutSuffix || isFuture ? 'mínútur' : 'mínútum');
				} else if (withoutSuffix) {
					return result + 'mínúta';
				}
				return result + 'mínútu';
			case 'hh':
				if (plural$2(number)) {
					return result + (withoutSuffix || isFuture ? 'klukkustundir' : 'klukkustundum');
				}
				return result + 'klukkustund';
			case 'd':
				if (withoutSuffix) {
					return 'dagur';
				}
				return isFuture ? 'dag' : 'degi';
			case 'dd':
				if (plural$2(number)) {
					if (withoutSuffix) {
						return result + 'dagar';
					}
					return result + (isFuture ? 'daga' : 'dögum');
				} else if (withoutSuffix) {
					return result + 'dagur';
				}
				return result + (isFuture ? 'dag' : 'degi');
			case 'M':
				if (withoutSuffix) {
					return 'mánuður';
				}
				return isFuture ? 'mánuð' : 'mánuði';
			case 'MM':
				if (plural$2(number)) {
					if (withoutSuffix) {
						return result + 'mánuðir';
					}
					return result + (isFuture ? 'mánuði' : 'mánuðum');
				} else if (withoutSuffix) {
					return result + 'mánuður';
				}
				return result + (isFuture ? 'mánuð' : 'mánuði');
			case 'y':
				return withoutSuffix || isFuture ? 'ár' : 'ári';
			case 'yy':
				if (plural$2(number)) {
					return result + (withoutSuffix || isFuture ? 'ár' : 'árum');
				}
				return result + (withoutSuffix || isFuture ? 'ár' : 'ári');
		}
	}

	hooks.defineLocale('is', {
		months : 'janúar_febrúar_mars_apríl_maí_júní_júlí_ágúst_september_október_nóvember_desember'.split('_'),
		monthsShort : 'jan_feb_mar_apr_maí_jún_júl_ágú_sep_okt_nóv_des'.split('_'),
		weekdays : 'sunnudagur_mánudagur_þriðjudagur_miðvikudagur_fimmtudagur_föstudagur_laugardagur'.split('_'),
		weekdaysShort : 'sun_mán_þri_mið_fim_fös_lau'.split('_'),
		weekdaysMin : 'Su_Má_Þr_Mi_Fi_Fö_La'.split('_'),
		longDateFormat : {
			LT : 'H:mm',
			LTS : 'H:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D. MMMM YYYY',
			LLL : 'D. MMMM YYYY [kl.] H:mm',
			LLLL : 'dddd, D. MMMM YYYY [kl.] H:mm'
		},
		calendar : {
			sameDay : '[í dag kl.] LT',
			nextDay : '[á morgun kl.] LT',
			nextWeek : 'dddd [kl.] LT',
			lastDay : '[í gær kl.] LT',
			lastWeek : '[síðasta] dddd [kl.] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'eftir %s',
			past : 'fyrir %s síðan',
			s : translate$5,
			ss : translate$5,
			m : translate$5,
			mm : translate$5,
			h : 'klukkustund',
			hh : translate$5,
			d : translate$5,
			dd : translate$5,
			M : translate$5,
			MM : translate$5,
			y : translate$5,
			yy : translate$5
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('it-ch', {
		months : 'gennaio_febbraio_marzo_aprile_maggio_giugno_luglio_agosto_settembre_ottobre_novembre_dicembre'.split('_'),
		monthsShort : 'gen_feb_mar_apr_mag_giu_lug_ago_set_ott_nov_dic'.split('_'),
		weekdays : 'domenica_lunedì_martedì_mercoledì_giovedì_venerdì_sabato'.split('_'),
		weekdaysShort : 'dom_lun_mar_mer_gio_ven_sab'.split('_'),
		weekdaysMin : 'do_lu_ma_me_gi_ve_sa'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay: '[Oggi alle] LT',
			nextDay: '[Domani alle] LT',
			nextWeek: 'dddd [alle] LT',
			lastDay: '[Ieri alle] LT',
			lastWeek: function () {
				switch (this.day()) {
					case 0:
						return '[la scorsa] dddd [alle] LT';
					default:
						return '[lo scorso] dddd [alle] LT';
				}
			},
			sameElse: 'L'
		},
		relativeTime : {
			future : function (s) {
				return ((/^[0-9].+$/).test(s) ? 'tra' : 'in') + ' ' + s;
			},
			past : '%s fa',
			s : 'alcuni secondi',
			ss : '%d secondi',
			m : 'un minuto',
			mm : '%d minuti',
			h : 'un\'ora',
			hh : '%d ore',
			d : 'un giorno',
			dd : '%d giorni',
			M : 'un mese',
			MM : '%d mesi',
			y : 'un anno',
			yy : '%d anni'
		},
		dayOfMonthOrdinalParse : /\d{1,2}º/,
		ordinal: '%dº',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('it', {
		months : 'gennaio_febbraio_marzo_aprile_maggio_giugno_luglio_agosto_settembre_ottobre_novembre_dicembre'.split('_'),
		monthsShort : 'gen_feb_mar_apr_mag_giu_lug_ago_set_ott_nov_dic'.split('_'),
		weekdays : 'domenica_lunedì_martedì_mercoledì_giovedì_venerdì_sabato'.split('_'),
		weekdaysShort : 'dom_lun_mar_mer_gio_ven_sab'.split('_'),
		weekdaysMin : 'do_lu_ma_me_gi_ve_sa'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay: '[Oggi alle] LT',
			nextDay: '[Domani alle] LT',
			nextWeek: 'dddd [alle] LT',
			lastDay: '[Ieri alle] LT',
			lastWeek: function () {
				switch (this.day()) {
					case 0:
						return '[la scorsa] dddd [alle] LT';
					default:
						return '[lo scorso] dddd [alle] LT';
				}
			},
			sameElse: 'L'
		},
		relativeTime : {
			future : function (s) {
				return ((/^[0-9].+$/).test(s) ? 'tra' : 'in') + ' ' + s;
			},
			past : '%s fa',
			s : 'alcuni secondi',
			ss : '%d secondi',
			m : 'un minuto',
			mm : '%d minuti',
			h : 'un\'ora',
			hh : '%d ore',
			d : 'un giorno',
			dd : '%d giorni',
			M : 'un mese',
			MM : '%d mesi',
			y : 'un anno',
			yy : '%d anni'
		},
		dayOfMonthOrdinalParse : /\d{1,2}º/,
		ordinal: '%dº',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('ja', {
		months : '一月_二月_三月_四月_五月_六月_七月_八月_九月_十月_十一月_十二月'.split('_'),
		monthsShort : '1月_2月_3月_4月_5月_6月_7月_8月_9月_10月_11月_12月'.split('_'),
		weekdays : '日曜日_月曜日_火曜日_水曜日_木曜日_金曜日_土曜日'.split('_'),
		weekdaysShort : '日_月_火_水_木_金_土'.split('_'),
		weekdaysMin : '日_月_火_水_木_金_土'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'YYYY/MM/DD',
			LL : 'YYYY年M月D日',
			LLL : 'YYYY年M月D日 HH:mm',
			LLLL : 'YYYY年M月D日 dddd HH:mm',
			l : 'YYYY/MM/DD',
			ll : 'YYYY年M月D日',
			lll : 'YYYY年M月D日 HH:mm',
			llll : 'YYYY年M月D日(ddd) HH:mm'
		},
		meridiemParse: /午前|午後/i,
		isPM : function (input) {
			return input === '午後';
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 12) {
				return '午前';
			} else {
				return '午後';
			}
		},
		calendar : {
			sameDay : '[今日] LT',
			nextDay : '[明日] LT',
			nextWeek : function (now) {
				if (now.week() < this.week()) {
					return '[来週]dddd LT';
				} else {
					return 'dddd LT';
				}
			},
			lastDay : '[昨日] LT',
			lastWeek : function (now) {
				if (this.week() < now.week()) {
					return '[先週]dddd LT';
				} else {
					return 'dddd LT';
				}
			},
			sameElse : 'L'
		},
		dayOfMonthOrdinalParse : /\d{1,2}日/,
		ordinal : function (number, period) {
			switch (period) {
				case 'd':
				case 'D':
				case 'DDD':
					return number + '日';
				default:
					return number;
			}
		},
		relativeTime : {
			future : '%s後',
			past : '%s前',
			s : '数秒',
			ss : '%d秒',
			m : '1分',
			mm : '%d分',
			h : '1時間',
			hh : '%d時間',
			d : '1日',
			dd : '%d日',
			M : '1ヶ月',
			MM : '%dヶ月',
			y : '1年',
			yy : '%d年'
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('jv', {
		months : 'Januari_Februari_Maret_April_Mei_Juni_Juli_Agustus_September_Oktober_Nopember_Desember'.split('_'),
		monthsShort : 'Jan_Feb_Mar_Apr_Mei_Jun_Jul_Ags_Sep_Okt_Nop_Des'.split('_'),
		weekdays : 'Minggu_Senen_Seloso_Rebu_Kemis_Jemuwah_Septu'.split('_'),
		weekdaysShort : 'Min_Sen_Sel_Reb_Kem_Jem_Sep'.split('_'),
		weekdaysMin : 'Mg_Sn_Sl_Rb_Km_Jm_Sp'.split('_'),
		longDateFormat : {
			LT : 'HH.mm',
			LTS : 'HH.mm.ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY [pukul] HH.mm',
			LLLL : 'dddd, D MMMM YYYY [pukul] HH.mm'
		},
		meridiemParse: /enjing|siyang|sonten|ndalu/,
		meridiemHour : function (hour, meridiem) {
			if (hour === 12) {
				hour = 0;
			}
			if (meridiem === 'enjing') {
				return hour;
			} else if (meridiem === 'siyang') {
				return hour >= 11 ? hour : hour + 12;
			} else if (meridiem === 'sonten' || meridiem === 'ndalu') {
				return hour + 12;
			}
		},
		meridiem : function (hours, minutes, isLower) {
			if (hours < 11) {
				return 'enjing';
			} else if (hours < 15) {
				return 'siyang';
			} else if (hours < 19) {
				return 'sonten';
			} else {
				return 'ndalu';
			}
		},
		calendar : {
			sameDay : '[Dinten puniko pukul] LT',
			nextDay : '[Mbenjang pukul] LT',
			nextWeek : 'dddd [pukul] LT',
			lastDay : '[Kala wingi pukul] LT',
			lastWeek : 'dddd [kepengker pukul] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'wonten ing %s',
			past : '%s ingkang kepengker',
			s : 'sawetawis detik',
			ss : '%d detik',
			m : 'setunggal menit',
			mm : '%d menit',
			h : 'setunggal jam',
			hh : '%d jam',
			d : 'sedinten',
			dd : '%d dinten',
			M : 'sewulan',
			MM : '%d wulan',
			y : 'setaun',
			yy : '%d taun'
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('ka', {
		months : {
			standalone: 'იანვარი_თებერვალი_მარტი_აპრილი_მაისი_ივნისი_ივლისი_აგვისტო_სექტემბერი_ოქტომბერი_ნოემბერი_დეკემბერი'.split('_'),
			format: 'იანვარს_თებერვალს_მარტს_აპრილის_მაისს_ივნისს_ივლისს_აგვისტს_სექტემბერს_ოქტომბერს_ნოემბერს_დეკემბერს'.split('_')
		},
		monthsShort : 'იან_თებ_მარ_აპრ_მაი_ივნ_ივლ_აგვ_სექ_ოქტ_ნოე_დეკ'.split('_'),
		weekdays : {
			standalone: 'კვირა_ორშაბათი_სამშაბათი_ოთხშაბათი_ხუთშაბათი_პარასკევი_შაბათი'.split('_'),
			format: 'კვირას_ორშაბათს_სამშაბათს_ოთხშაბათს_ხუთშაბათს_პარასკევს_შაბათს'.split('_'),
			isFormat: /(წინა|შემდეგ)/
		},
		weekdaysShort : 'კვი_ორშ_სამ_ოთხ_ხუთ_პარ_შაბ'.split('_'),
		weekdaysMin : 'კვ_ორ_სა_ოთ_ხუ_პა_შა'.split('_'),
		longDateFormat : {
			LT : 'h:mm A',
			LTS : 'h:mm:ss A',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY h:mm A',
			LLLL : 'dddd, D MMMM YYYY h:mm A'
		},
		calendar : {
			sameDay : '[დღეს] LT[-ზე]',
			nextDay : '[ხვალ] LT[-ზე]',
			lastDay : '[გუშინ] LT[-ზე]',
			nextWeek : '[შემდეგ] dddd LT[-ზე]',
			lastWeek : '[წინა] dddd LT-ზე',
			sameElse : 'L'
		},
		relativeTime : {
			future : function (s) {
				return (/(წამი|წუთი|საათი|წელი)/).test(s) ?
					s.replace(/ი$/, 'ში') :
					s + 'ში';
			},
			past : function (s) {
				if ((/(წამი|წუთი|საათი|დღე|თვე)/).test(s)) {
					return s.replace(/(ი|ე)$/, 'ის წინ');
				}
				if ((/წელი/).test(s)) {
					return s.replace(/წელი$/, 'წლის წინ');
				}
			},
			s : 'რამდენიმე წამი',
			ss : '%d წამი',
			m : 'წუთი',
			mm : '%d წუთი',
			h : 'საათი',
			hh : '%d საათი',
			d : 'დღე',
			dd : '%d დღე',
			M : 'თვე',
			MM : '%d თვე',
			y : 'წელი',
			yy : '%d წელი'
		},
		dayOfMonthOrdinalParse: /0|1-ლი|მე-\d{1,2}|\d{1,2}-ე/,
		ordinal : function (number) {
			if (number === 0) {
				return number;
			}
			if (number === 1) {
				return number + '-ლი';
			}
			if ((number < 20) || (number <= 100 && (number % 20 === 0)) || (number % 100 === 0)) {
				return 'მე-' + number;
			}
			return number + '-ე';
		},
		week : {
			dow : 1,
			doy : 7
		}
	});

	//! moment.js locale configuration

	var suffixes$1 = {
		0: '-ші',
		1: '-ші',
		2: '-ші',
		3: '-ші',
		4: '-ші',
		5: '-ші',
		6: '-шы',
		7: '-ші',
		8: '-ші',
		9: '-шы',
		10: '-шы',
		20: '-шы',
		30: '-шы',
		40: '-шы',
		50: '-ші',
		60: '-шы',
		70: '-ші',
		80: '-ші',
		90: '-шы',
		100: '-ші'
	};

	hooks.defineLocale('kk', {
		months : 'қаңтар_ақпан_наурыз_сәуір_мамыр_маусым_шілде_тамыз_қыркүйек_қазан_қараша_желтоқсан'.split('_'),
		monthsShort : 'қаң_ақп_нау_сәу_мам_мау_шіл_там_қыр_қаз_қар_жел'.split('_'),
		weekdays : 'жексенбі_дүйсенбі_сейсенбі_сәрсенбі_бейсенбі_жұма_сенбі'.split('_'),
		weekdaysShort : 'жек_дүй_сей_сәр_бей_жұм_сен'.split('_'),
		weekdaysMin : 'жк_дй_сй_ср_бй_жм_сн'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd, D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay : '[Бүгін сағат] LT',
			nextDay : '[Ертең сағат] LT',
			nextWeek : 'dddd [сағат] LT',
			lastDay : '[Кеше сағат] LT',
			lastWeek : '[Өткен аптаның] dddd [сағат] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : '%s ішінде',
			past : '%s бұрын',
			s : 'бірнеше секунд',
			ss : '%d секунд',
			m : 'бір минут',
			mm : '%d минут',
			h : 'бір сағат',
			hh : '%d сағат',
			d : 'бір күн',
			dd : '%d күн',
			M : 'бір ай',
			MM : '%d ай',
			y : 'бір жыл',
			yy : '%d жыл'
		},
		dayOfMonthOrdinalParse: /\d{1,2}-(ші|шы)/,
		ordinal : function (number) {
			var a = number % 10,
				b = number >= 100 ? 100 : null;
			return number + (suffixes$1[number] || suffixes$1[a] || suffixes$1[b]);
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var symbolMap$8 = {
		'1': '១',
		'2': '២',
		'3': '៣',
		'4': '៤',
		'5': '៥',
		'6': '៦',
		'7': '៧',
		'8': '៨',
		'9': '៩',
		'0': '០'
	}, numberMap$7 = {
		'១': '1',
		'២': '2',
		'៣': '3',
		'៤': '4',
		'៥': '5',
		'៦': '6',
		'៧': '7',
		'៨': '8',
		'៩': '9',
		'០': '0'
	};

	hooks.defineLocale('km', {
		months: 'មករា_កុម្ភៈ_មីនា_មេសា_ឧសភា_មិថុនា_កក្កដា_សីហា_កញ្ញា_តុលា_វិច្ឆិកា_ធ្នូ'.split(
			'_'
		),
		monthsShort: 'មករា_កុម្ភៈ_មីនា_មេសា_ឧសភា_មិថុនា_កក្កដា_សីហា_កញ្ញា_តុលា_វិច្ឆិកា_ធ្នូ'.split(
			'_'
		),
		weekdays: 'អាទិត្យ_ច័ន្ទ_អង្គារ_ពុធ_ព្រហស្បតិ៍_សុក្រ_សៅរ៍'.split('_'),
		weekdaysShort: 'អា_ច_អ_ព_ព្រ_សុ_ស'.split('_'),
		weekdaysMin: 'អា_ច_អ_ព_ព្រ_សុ_ស'.split('_'),
		weekdaysParseExact: true,
		longDateFormat: {
			LT: 'HH:mm',
			LTS: 'HH:mm:ss',
			L: 'DD/MM/YYYY',
			LL: 'D MMMM YYYY',
			LLL: 'D MMMM YYYY HH:mm',
			LLLL: 'dddd, D MMMM YYYY HH:mm'
		},
		meridiemParse: /ព្រឹក|ល្ងាច/,
		isPM: function (input) {
			return input === 'ល្ងាច';
		},
		meridiem: function (hour, minute, isLower) {
			if (hour < 12) {
				return 'ព្រឹក';
			} else {
				return 'ល្ងាច';
			}
		},
		calendar: {
			sameDay: '[ថ្ងៃនេះ ម៉ោង] LT',
			nextDay: '[ស្អែក ម៉ោង] LT',
			nextWeek: 'dddd [ម៉ោង] LT',
			lastDay: '[ម្សិលមិញ ម៉ោង] LT',
			lastWeek: 'dddd [សប្តាហ៍មុន] [ម៉ោង] LT',
			sameElse: 'L'
		},
		relativeTime: {
			future: '%sទៀត',
			past: '%sមុន',
			s: 'ប៉ុន្មានវិនាទី',
			ss: '%d វិនាទី',
			m: 'មួយនាទី',
			mm: '%d នាទី',
			h: 'មួយម៉ោង',
			hh: '%d ម៉ោង',
			d: 'មួយថ្ងៃ',
			dd: '%d ថ្ងៃ',
			M: 'មួយខែ',
			MM: '%d ខែ',
			y: 'មួយឆ្នាំ',
			yy: '%d ឆ្នាំ'
		},
		dayOfMonthOrdinalParse : /ទី\d{1,2}/,
		ordinal : 'ទី%d',
		preparse: function (string) {
			return string.replace(/[១២៣៤៥៦៧៨៩០]/g, function (match) {
				return numberMap$7[match];
			});
		},
		postformat: function (string) {
			return string.replace(/\d/g, function (match) {
				return symbolMap$8[match];
			});
		},
		week: {
			dow: 1, // Monday is the first day of the week.
			doy: 4 // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var symbolMap$9 = {
			'1': '೧',
			'2': '೨',
			'3': '೩',
			'4': '೪',
			'5': '೫',
			'6': '೬',
			'7': '೭',
			'8': '೮',
			'9': '೯',
			'0': '೦'
		},
		numberMap$8 = {
			'೧': '1',
			'೨': '2',
			'೩': '3',
			'೪': '4',
			'೫': '5',
			'೬': '6',
			'೭': '7',
			'೮': '8',
			'೯': '9',
			'೦': '0'
		};

	hooks.defineLocale('kn', {
		months : 'ಜನವರಿ_ಫೆಬ್ರವರಿ_ಮಾರ್ಚ್_ಏಪ್ರಿಲ್_ಮೇ_ಜೂನ್_ಜುಲೈ_ಆಗಸ್ಟ್_ಸೆಪ್ಟೆಂಬರ್_ಅಕ್ಟೋಬರ್_ನವೆಂಬರ್_ಡಿಸೆಂಬರ್'.split('_'),
		monthsShort : 'ಜನ_ಫೆಬ್ರ_ಮಾರ್ಚ್_ಏಪ್ರಿಲ್_ಮೇ_ಜೂನ್_ಜುಲೈ_ಆಗಸ್ಟ್_ಸೆಪ್ಟೆಂ_ಅಕ್ಟೋ_ನವೆಂ_ಡಿಸೆಂ'.split('_'),
		monthsParseExact: true,
		weekdays : 'ಭಾನುವಾರ_ಸೋಮವಾರ_ಮಂಗಳವಾರ_ಬುಧವಾರ_ಗುರುವಾರ_ಶುಕ್ರವಾರ_ಶನಿವಾರ'.split('_'),
		weekdaysShort : 'ಭಾನು_ಸೋಮ_ಮಂಗಳ_ಬುಧ_ಗುರು_ಶುಕ್ರ_ಶನಿ'.split('_'),
		weekdaysMin : 'ಭಾ_ಸೋ_ಮಂ_ಬು_ಗು_ಶು_ಶ'.split('_'),
		longDateFormat : {
			LT : 'A h:mm',
			LTS : 'A h:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY, A h:mm',
			LLLL : 'dddd, D MMMM YYYY, A h:mm'
		},
		calendar : {
			sameDay : '[ಇಂದು] LT',
			nextDay : '[ನಾಳೆ] LT',
			nextWeek : 'dddd, LT',
			lastDay : '[ನಿನ್ನೆ] LT',
			lastWeek : '[ಕೊನೆಯ] dddd, LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : '%s ನಂತರ',
			past : '%s ಹಿಂದೆ',
			s : 'ಕೆಲವು ಕ್ಷಣಗಳು',
			ss : '%d ಸೆಕೆಂಡುಗಳು',
			m : 'ಒಂದು ನಿಮಿಷ',
			mm : '%d ನಿಮಿಷ',
			h : 'ಒಂದು ಗಂಟೆ',
			hh : '%d ಗಂಟೆ',
			d : 'ಒಂದು ದಿನ',
			dd : '%d ದಿನ',
			M : 'ಒಂದು ತಿಂಗಳು',
			MM : '%d ತಿಂಗಳು',
			y : 'ಒಂದು ವರ್ಷ',
			yy : '%d ವರ್ಷ'
		},
		preparse: function (string) {
			return string.replace(/[೧೨೩೪೫೬೭೮೯೦]/g, function (match) {
				return numberMap$8[match];
			});
		},
		postformat: function (string) {
			return string.replace(/\d/g, function (match) {
				return symbolMap$9[match];
			});
		},
		meridiemParse: /ರಾತ್ರಿ|ಬೆಳಿಗ್ಗೆ|ಮಧ್ಯಾಹ್ನ|ಸಂಜೆ/,
		meridiemHour : function (hour, meridiem) {
			if (hour === 12) {
				hour = 0;
			}
			if (meridiem === 'ರಾತ್ರಿ') {
				return hour < 4 ? hour : hour + 12;
			} else if (meridiem === 'ಬೆಳಿಗ್ಗೆ') {
				return hour;
			} else if (meridiem === 'ಮಧ್ಯಾಹ್ನ') {
				return hour >= 10 ? hour : hour + 12;
			} else if (meridiem === 'ಸಂಜೆ') {
				return hour + 12;
			}
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 4) {
				return 'ರಾತ್ರಿ';
			} else if (hour < 10) {
				return 'ಬೆಳಿಗ್ಗೆ';
			} else if (hour < 17) {
				return 'ಮಧ್ಯಾಹ್ನ';
			} else if (hour < 20) {
				return 'ಸಂಜೆ';
			} else {
				return 'ರಾತ್ರಿ';
			}
		},
		dayOfMonthOrdinalParse: /\d{1,2}(ನೇ)/,
		ordinal : function (number) {
			return number + 'ನೇ';
		},
		week : {
			dow : 0, // Sunday is the first day of the week.
			doy : 6  // The week that contains Jan 6th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('ko', {
		months : '1월_2월_3월_4월_5월_6월_7월_8월_9월_10월_11월_12월'.split('_'),
		monthsShort : '1월_2월_3월_4월_5월_6월_7월_8월_9월_10월_11월_12월'.split('_'),
		weekdays : '일요일_월요일_화요일_수요일_목요일_금요일_토요일'.split('_'),
		weekdaysShort : '일_월_화_수_목_금_토'.split('_'),
		weekdaysMin : '일_월_화_수_목_금_토'.split('_'),
		longDateFormat : {
			LT : 'A h:mm',
			LTS : 'A h:mm:ss',
			L : 'YYYY.MM.DD.',
			LL : 'YYYY년 MMMM D일',
			LLL : 'YYYY년 MMMM D일 A h:mm',
			LLLL : 'YYYY년 MMMM D일 dddd A h:mm',
			l : 'YYYY.MM.DD.',
			ll : 'YYYY년 MMMM D일',
			lll : 'YYYY년 MMMM D일 A h:mm',
			llll : 'YYYY년 MMMM D일 dddd A h:mm'
		},
		calendar : {
			sameDay : '오늘 LT',
			nextDay : '내일 LT',
			nextWeek : 'dddd LT',
			lastDay : '어제 LT',
			lastWeek : '지난주 dddd LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : '%s 후',
			past : '%s 전',
			s : '몇 초',
			ss : '%d초',
			m : '1분',
			mm : '%d분',
			h : '한 시간',
			hh : '%d시간',
			d : '하루',
			dd : '%d일',
			M : '한 달',
			MM : '%d달',
			y : '일 년',
			yy : '%d년'
		},
		dayOfMonthOrdinalParse : /\d{1,2}(일|월|주)/,
		ordinal : function (number, period) {
			switch (period) {
				case 'd':
				case 'D':
				case 'DDD':
					return number + '일';
				case 'M':
					return number + '월';
				case 'w':
				case 'W':
					return number + '주';
				default:
					return number;
			}
		},
		meridiemParse : /오전|오후/,
		isPM : function (token) {
			return token === '오후';
		},
		meridiem : function (hour, minute, isUpper) {
			return hour < 12 ? '오전' : '오후';
		}
	});

	//! moment.js locale configuration

	var symbolMap$a = {
			'1': '١',
			'2': '٢',
			'3': '٣',
			'4': '٤',
			'5': '٥',
			'6': '٦',
			'7': '٧',
			'8': '٨',
			'9': '٩',
			'0': '٠'
		}, numberMap$9 = {
			'١': '1',
			'٢': '2',
			'٣': '3',
			'٤': '4',
			'٥': '5',
			'٦': '6',
			'٧': '7',
			'٨': '8',
			'٩': '9',
			'٠': '0'
		},
		months$7 = [
			'کانونی دووەم',
			'شوبات',
			'ئازار',
			'نیسان',
			'ئایار',
			'حوزەیران',
			'تەمموز',
			'ئاب',
			'ئەیلوول',
			'تشرینی یەكەم',
			'تشرینی دووەم',
			'كانونی یەکەم'
		];


	hooks.defineLocale('ku', {
		months : months$7,
		monthsShort : months$7,
		weekdays : 'یه‌كشه‌ممه‌_دووشه‌ممه‌_سێشه‌ممه‌_چوارشه‌ممه‌_پێنجشه‌ممه‌_هه‌ینی_شه‌ممه‌'.split('_'),
		weekdaysShort : 'یه‌كشه‌م_دووشه‌م_سێشه‌م_چوارشه‌م_پێنجشه‌م_هه‌ینی_شه‌ممه‌'.split('_'),
		weekdaysMin : 'ی_د_س_چ_پ_ه_ش'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd, D MMMM YYYY HH:mm'
		},
		meridiemParse: /ئێواره‌|به‌یانی/,
		isPM: function (input) {
			return /ئێواره‌/.test(input);
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 12) {
				return 'به‌یانی';
			} else {
				return 'ئێواره‌';
			}
		},
		calendar : {
			sameDay : '[ئه‌مرۆ كاتژمێر] LT',
			nextDay : '[به‌یانی كاتژمێر] LT',
			nextWeek : 'dddd [كاتژمێر] LT',
			lastDay : '[دوێنێ كاتژمێر] LT',
			lastWeek : 'dddd [كاتژمێر] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'له‌ %s',
			past : '%s',
			s : 'چه‌ند چركه‌یه‌ك',
			ss : 'چركه‌ %d',
			m : 'یه‌ك خوله‌ك',
			mm : '%d خوله‌ك',
			h : 'یه‌ك كاتژمێر',
			hh : '%d كاتژمێر',
			d : 'یه‌ك ڕۆژ',
			dd : '%d ڕۆژ',
			M : 'یه‌ك مانگ',
			MM : '%d مانگ',
			y : 'یه‌ك ساڵ',
			yy : '%d ساڵ'
		},
		preparse: function (string) {
			return string.replace(/[١٢٣٤٥٦٧٨٩٠]/g, function (match) {
				return numberMap$9[match];
			}).replace(/،/g, ',');
		},
		postformat: function (string) {
			return string.replace(/\d/g, function (match) {
				return symbolMap$a[match];
			}).replace(/,/g, '،');
		},
		week : {
			dow : 6, // Saturday is the first day of the week.
			doy : 12 // The week that contains Jan 12th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var suffixes$2 = {
		0: '-чү',
		1: '-чи',
		2: '-чи',
		3: '-чү',
		4: '-чү',
		5: '-чи',
		6: '-чы',
		7: '-чи',
		8: '-чи',
		9: '-чу',
		10: '-чу',
		20: '-чы',
		30: '-чу',
		40: '-чы',
		50: '-чү',
		60: '-чы',
		70: '-чи',
		80: '-чи',
		90: '-чу',
		100: '-чү'
	};

	hooks.defineLocale('ky', {
		months : 'январь_февраль_март_апрель_май_июнь_июль_август_сентябрь_октябрь_ноябрь_декабрь'.split('_'),
		monthsShort : 'янв_фев_март_апр_май_июнь_июль_авг_сен_окт_ноя_дек'.split('_'),
		weekdays : 'Жекшемби_Дүйшөмбү_Шейшемби_Шаршемби_Бейшемби_Жума_Ишемби'.split('_'),
		weekdaysShort : 'Жек_Дүй_Шей_Шар_Бей_Жум_Ише'.split('_'),
		weekdaysMin : 'Жк_Дй_Шй_Шр_Бй_Жм_Иш'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd, D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay : '[Бүгүн саат] LT',
			nextDay : '[Эртең саат] LT',
			nextWeek : 'dddd [саат] LT',
			lastDay : '[Кечээ саат] LT',
			lastWeek : '[Өткөн аптанын] dddd [күнү] [саат] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : '%s ичинде',
			past : '%s мурун',
			s : 'бирнече секунд',
			ss : '%d секунд',
			m : 'бир мүнөт',
			mm : '%d мүнөт',
			h : 'бир саат',
			hh : '%d саат',
			d : 'бир күн',
			dd : '%d күн',
			M : 'бир ай',
			MM : '%d ай',
			y : 'бир жыл',
			yy : '%d жыл'
		},
		dayOfMonthOrdinalParse: /\d{1,2}-(чи|чы|чү|чу)/,
		ordinal : function (number) {
			var a = number % 10,
				b = number >= 100 ? 100 : null;
			return number + (suffixes$2[number] || suffixes$2[a] || suffixes$2[b]);
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	function processRelativeTime$5(number, withoutSuffix, key, isFuture) {
		var format = {
			'm': ['eng Minutt', 'enger Minutt'],
			'h': ['eng Stonn', 'enger Stonn'],
			'd': ['een Dag', 'engem Dag'],
			'M': ['ee Mount', 'engem Mount'],
			'y': ['ee Joer', 'engem Joer']
		};
		return withoutSuffix ? format[key][0] : format[key][1];
	}
	function processFutureTime(string) {
		var number = string.substr(0, string.indexOf(' '));
		if (eifelerRegelAppliesToNumber(number)) {
			return 'a ' + string;
		}
		return 'an ' + string;
	}
	function processPastTime(string) {
		var number = string.substr(0, string.indexOf(' '));
		if (eifelerRegelAppliesToNumber(number)) {
			return 'viru ' + string;
		}
		return 'virun ' + string;
	}
	/**
	 * Returns true if the word before the given number loses the '-n' ending.
	 * e.g. 'an 10 Deeg' but 'a 5 Deeg'
	 *
	 * @param number {integer}
	 * @returns {boolean}
	 */
	function eifelerRegelAppliesToNumber(number) {
		number = parseInt(number, 10);
		if (isNaN(number)) {
			return false;
		}
		if (number < 0) {
			// Negative Number --> always true
			return true;
		} else if (number < 10) {
			// Only 1 digit
			if (4 <= number && number <= 7) {
				return true;
			}
			return false;
		} else if (number < 100) {
			// 2 digits
			var lastDigit = number % 10, firstDigit = number / 10;
			if (lastDigit === 0) {
				return eifelerRegelAppliesToNumber(firstDigit);
			}
			return eifelerRegelAppliesToNumber(lastDigit);
		} else if (number < 10000) {
			// 3 or 4 digits --> recursively check first digit
			while (number >= 10) {
				number = number / 10;
			}
			return eifelerRegelAppliesToNumber(number);
		} else {
			// Anything larger than 4 digits: recursively check first n-3 digits
			number = number / 1000;
			return eifelerRegelAppliesToNumber(number);
		}
	}

	hooks.defineLocale('lb', {
		months: 'Januar_Februar_Mäerz_Abrëll_Mee_Juni_Juli_August_September_Oktober_November_Dezember'.split('_'),
		monthsShort: 'Jan._Febr._Mrz._Abr._Mee_Jun._Jul._Aug._Sept._Okt._Nov._Dez.'.split('_'),
		monthsParseExact : true,
		weekdays: 'Sonndeg_Méindeg_Dënschdeg_Mëttwoch_Donneschdeg_Freideg_Samschdeg'.split('_'),
		weekdaysShort: 'So._Mé._Dë._Më._Do._Fr._Sa.'.split('_'),
		weekdaysMin: 'So_Mé_Dë_Më_Do_Fr_Sa'.split('_'),
		weekdaysParseExact : true,
		longDateFormat: {
			LT: 'H:mm [Auer]',
			LTS: 'H:mm:ss [Auer]',
			L: 'DD.MM.YYYY',
			LL: 'D. MMMM YYYY',
			LLL: 'D. MMMM YYYY H:mm [Auer]',
			LLLL: 'dddd, D. MMMM YYYY H:mm [Auer]'
		},
		calendar: {
			sameDay: '[Haut um] LT',
			sameElse: 'L',
			nextDay: '[Muer um] LT',
			nextWeek: 'dddd [um] LT',
			lastDay: '[Gëschter um] LT',
			lastWeek: function () {
				// Different date string for 'Dënschdeg' (Tuesday) and 'Donneschdeg' (Thursday) due to phonological rule
				switch (this.day()) {
					case 2:
					case 4:
						return '[Leschten] dddd [um] LT';
					default:
						return '[Leschte] dddd [um] LT';
				}
			}
		},
		relativeTime : {
			future : processFutureTime,
			past : processPastTime,
			s : 'e puer Sekonnen',
			ss : '%d Sekonnen',
			m : processRelativeTime$5,
			mm : '%d Minutten',
			h : processRelativeTime$5,
			hh : '%d Stonnen',
			d : processRelativeTime$5,
			dd : '%d Deeg',
			M : processRelativeTime$5,
			MM : '%d Méint',
			y : processRelativeTime$5,
			yy : '%d Joer'
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal: '%d.',
		week: {
			dow: 1, // Monday is the first day of the week.
			doy: 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('lo', {
		months : 'ມັງກອນ_ກຸມພາ_ມີນາ_ເມສາ_ພຶດສະພາ_ມິຖຸນາ_ກໍລະກົດ_ສິງຫາ_ກັນຍາ_ຕຸລາ_ພະຈິກ_ທັນວາ'.split('_'),
		monthsShort : 'ມັງກອນ_ກຸມພາ_ມີນາ_ເມສາ_ພຶດສະພາ_ມິຖຸນາ_ກໍລະກົດ_ສິງຫາ_ກັນຍາ_ຕຸລາ_ພະຈິກ_ທັນວາ'.split('_'),
		weekdays : 'ອາທິດ_ຈັນ_ອັງຄານ_ພຸດ_ພະຫັດ_ສຸກ_ເສົາ'.split('_'),
		weekdaysShort : 'ທິດ_ຈັນ_ອັງຄານ_ພຸດ_ພະຫັດ_ສຸກ_ເສົາ'.split('_'),
		weekdaysMin : 'ທ_ຈ_ອຄ_ພ_ພຫ_ສກ_ສ'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'ວັນdddd D MMMM YYYY HH:mm'
		},
		meridiemParse: /ຕອນເຊົ້າ|ຕອນແລງ/,
		isPM: function (input) {
			return input === 'ຕອນແລງ';
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 12) {
				return 'ຕອນເຊົ້າ';
			} else {
				return 'ຕອນແລງ';
			}
		},
		calendar : {
			sameDay : '[ມື້ນີ້ເວລາ] LT',
			nextDay : '[ມື້ອື່ນເວລາ] LT',
			nextWeek : '[ວັນ]dddd[ໜ້າເວລາ] LT',
			lastDay : '[ມື້ວານນີ້ເວລາ] LT',
			lastWeek : '[ວັນ]dddd[ແລ້ວນີ້ເວລາ] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'ອີກ %s',
			past : '%sຜ່ານມາ',
			s : 'ບໍ່ເທົ່າໃດວິນາທີ',
			ss : '%d ວິນາທີ' ,
			m : '1 ນາທີ',
			mm : '%d ນາທີ',
			h : '1 ຊົ່ວໂມງ',
			hh : '%d ຊົ່ວໂມງ',
			d : '1 ມື້',
			dd : '%d ມື້',
			M : '1 ເດືອນ',
			MM : '%d ເດືອນ',
			y : '1 ປີ',
			yy : '%d ປີ'
		},
		dayOfMonthOrdinalParse: /(ທີ່)\d{1,2}/,
		ordinal : function (number) {
			return 'ທີ່' + number;
		}
	});

	//! moment.js locale configuration

	var units = {
		'ss' : 'sekundė_sekundžių_sekundes',
		'm' : 'minutė_minutės_minutę',
		'mm': 'minutės_minučių_minutes',
		'h' : 'valanda_valandos_valandą',
		'hh': 'valandos_valandų_valandas',
		'd' : 'diena_dienos_dieną',
		'dd': 'dienos_dienų_dienas',
		'M' : 'mėnuo_mėnesio_mėnesį',
		'MM': 'mėnesiai_mėnesių_mėnesius',
		'y' : 'metai_metų_metus',
		'yy': 'metai_metų_metus'
	};
	function translateSeconds(number, withoutSuffix, key, isFuture) {
		if (withoutSuffix) {
			return 'kelios sekundės';
		} else {
			return isFuture ? 'kelių sekundžių' : 'kelias sekundes';
		}
	}
	function translateSingular(number, withoutSuffix, key, isFuture) {
		return withoutSuffix ? forms(key)[0] : (isFuture ? forms(key)[1] : forms(key)[2]);
	}
	function special(number) {
		return number % 10 === 0 || (number > 10 && number < 20);
	}
	function forms(key) {
		return units[key].split('_');
	}
	function translate$6(number, withoutSuffix, key, isFuture) {
		var result = number + ' ';
		if (number === 1) {
			return result + translateSingular(number, withoutSuffix, key[0], isFuture);
		} else if (withoutSuffix) {
			return result + (special(number) ? forms(key)[1] : forms(key)[0]);
		} else {
			if (isFuture) {
				return result + forms(key)[1];
			} else {
				return result + (special(number) ? forms(key)[1] : forms(key)[2]);
			}
		}
	}
	hooks.defineLocale('lt', {
		months : {
			format: 'sausio_vasario_kovo_balandžio_gegužės_birželio_liepos_rugpjūčio_rugsėjo_spalio_lapkričio_gruodžio'.split('_'),
			standalone: 'sausis_vasaris_kovas_balandis_gegužė_birželis_liepa_rugpjūtis_rugsėjis_spalis_lapkritis_gruodis'.split('_'),
			isFormat: /D[oD]?(\[[^\[\]]*\]|\s)+MMMM?|MMMM?(\[[^\[\]]*\]|\s)+D[oD]?/
		},
		monthsShort : 'sau_vas_kov_bal_geg_bir_lie_rgp_rgs_spa_lap_grd'.split('_'),
		weekdays : {
			format: 'sekmadienį_pirmadienį_antradienį_trečiadienį_ketvirtadienį_penktadienį_šeštadienį'.split('_'),
			standalone: 'sekmadienis_pirmadienis_antradienis_trečiadienis_ketvirtadienis_penktadienis_šeštadienis'.split('_'),
			isFormat: /dddd HH:mm/
		},
		weekdaysShort : 'Sek_Pir_Ant_Tre_Ket_Pen_Šeš'.split('_'),
		weekdaysMin : 'S_P_A_T_K_Pn_Š'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'YYYY-MM-DD',
			LL : 'YYYY [m.] MMMM D [d.]',
			LLL : 'YYYY [m.] MMMM D [d.], HH:mm [val.]',
			LLLL : 'YYYY [m.] MMMM D [d.], dddd, HH:mm [val.]',
			l : 'YYYY-MM-DD',
			ll : 'YYYY [m.] MMMM D [d.]',
			lll : 'YYYY [m.] MMMM D [d.], HH:mm [val.]',
			llll : 'YYYY [m.] MMMM D [d.], ddd, HH:mm [val.]'
		},
		calendar : {
			sameDay : '[Šiandien] LT',
			nextDay : '[Rytoj] LT',
			nextWeek : 'dddd LT',
			lastDay : '[Vakar] LT',
			lastWeek : '[Praėjusį] dddd LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'po %s',
			past : 'prieš %s',
			s : translateSeconds,
			ss : translate$6,
			m : translateSingular,
			mm : translate$6,
			h : translateSingular,
			hh : translate$6,
			d : translateSingular,
			dd : translate$6,
			M : translateSingular,
			MM : translate$6,
			y : translateSingular,
			yy : translate$6
		},
		dayOfMonthOrdinalParse: /\d{1,2}-oji/,
		ordinal : function (number) {
			return number + '-oji';
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var units$1 = {
		'ss': 'sekundes_sekundēm_sekunde_sekundes'.split('_'),
		'm': 'minūtes_minūtēm_minūte_minūtes'.split('_'),
		'mm': 'minūtes_minūtēm_minūte_minūtes'.split('_'),
		'h': 'stundas_stundām_stunda_stundas'.split('_'),
		'hh': 'stundas_stundām_stunda_stundas'.split('_'),
		'd': 'dienas_dienām_diena_dienas'.split('_'),
		'dd': 'dienas_dienām_diena_dienas'.split('_'),
		'M': 'mēneša_mēnešiem_mēnesis_mēneši'.split('_'),
		'MM': 'mēneša_mēnešiem_mēnesis_mēneši'.split('_'),
		'y': 'gada_gadiem_gads_gadi'.split('_'),
		'yy': 'gada_gadiem_gads_gadi'.split('_')
	};
	/**
	 * @param withoutSuffix boolean true = a length of time; false = before/after a period of time.
	 */
	function format$1(forms, number, withoutSuffix) {
		if (withoutSuffix) {
			// E.g. "21 minūte", "3 minūtes".
			return number % 10 === 1 && number % 100 !== 11 ? forms[2] : forms[3];
		} else {
			// E.g. "21 minūtes" as in "pēc 21 minūtes".
			// E.g. "3 minūtēm" as in "pēc 3 minūtēm".
			return number % 10 === 1 && number % 100 !== 11 ? forms[0] : forms[1];
		}
	}
	function relativeTimeWithPlural$1(number, withoutSuffix, key) {
		return number + ' ' + format$1(units$1[key], number, withoutSuffix);
	}
	function relativeTimeWithSingular(number, withoutSuffix, key) {
		return format$1(units$1[key], number, withoutSuffix);
	}
	function relativeSeconds(number, withoutSuffix) {
		return withoutSuffix ? 'dažas sekundes' : 'dažām sekundēm';
	}

	hooks.defineLocale('lv', {
		months : 'janvāris_februāris_marts_aprīlis_maijs_jūnijs_jūlijs_augusts_septembris_oktobris_novembris_decembris'.split('_'),
		monthsShort : 'jan_feb_mar_apr_mai_jūn_jūl_aug_sep_okt_nov_dec'.split('_'),
		weekdays : 'svētdiena_pirmdiena_otrdiena_trešdiena_ceturtdiena_piektdiena_sestdiena'.split('_'),
		weekdaysShort : 'Sv_P_O_T_C_Pk_S'.split('_'),
		weekdaysMin : 'Sv_P_O_T_C_Pk_S'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD.MM.YYYY.',
			LL : 'YYYY. [gada] D. MMMM',
			LLL : 'YYYY. [gada] D. MMMM, HH:mm',
			LLLL : 'YYYY. [gada] D. MMMM, dddd, HH:mm'
		},
		calendar : {
			sameDay : '[Šodien pulksten] LT',
			nextDay : '[Rīt pulksten] LT',
			nextWeek : 'dddd [pulksten] LT',
			lastDay : '[Vakar pulksten] LT',
			lastWeek : '[Pagājušā] dddd [pulksten] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'pēc %s',
			past : 'pirms %s',
			s : relativeSeconds,
			ss : relativeTimeWithPlural$1,
			m : relativeTimeWithSingular,
			mm : relativeTimeWithPlural$1,
			h : relativeTimeWithSingular,
			hh : relativeTimeWithPlural$1,
			d : relativeTimeWithSingular,
			dd : relativeTimeWithPlural$1,
			M : relativeTimeWithSingular,
			MM : relativeTimeWithPlural$1,
			y : relativeTimeWithSingular,
			yy : relativeTimeWithPlural$1
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var translator = {
		words: { //Different grammatical cases
			ss: ['sekund', 'sekunda', 'sekundi'],
			m: ['jedan minut', 'jednog minuta'],
			mm: ['minut', 'minuta', 'minuta'],
			h: ['jedan sat', 'jednog sata'],
			hh: ['sat', 'sata', 'sati'],
			dd: ['dan', 'dana', 'dana'],
			MM: ['mjesec', 'mjeseca', 'mjeseci'],
			yy: ['godina', 'godine', 'godina']
		},
		correctGrammaticalCase: function (number, wordKey) {
			return number === 1 ? wordKey[0] : (number >= 2 && number <= 4 ? wordKey[1] : wordKey[2]);
		},
		translate: function (number, withoutSuffix, key) {
			var wordKey = translator.words[key];
			if (key.length === 1) {
				return withoutSuffix ? wordKey[0] : wordKey[1];
			} else {
				return number + ' ' + translator.correctGrammaticalCase(number, wordKey);
			}
		}
	};

	hooks.defineLocale('me', {
		months: 'januar_februar_mart_april_maj_jun_jul_avgust_septembar_oktobar_novembar_decembar'.split('_'),
		monthsShort: 'jan._feb._mar._apr._maj_jun_jul_avg._sep._okt._nov._dec.'.split('_'),
		monthsParseExact : true,
		weekdays: 'nedjelja_ponedjeljak_utorak_srijeda_četvrtak_petak_subota'.split('_'),
		weekdaysShort: 'ned._pon._uto._sri._čet._pet._sub.'.split('_'),
		weekdaysMin: 'ne_po_ut_sr_če_pe_su'.split('_'),
		weekdaysParseExact : true,
		longDateFormat: {
			LT: 'H:mm',
			LTS : 'H:mm:ss',
			L: 'DD.MM.YYYY',
			LL: 'D. MMMM YYYY',
			LLL: 'D. MMMM YYYY H:mm',
			LLLL: 'dddd, D. MMMM YYYY H:mm'
		},
		calendar: {
			sameDay: '[danas u] LT',
			nextDay: '[sjutra u] LT',

			nextWeek: function () {
				switch (this.day()) {
					case 0:
						return '[u] [nedjelju] [u] LT';
					case 3:
						return '[u] [srijedu] [u] LT';
					case 6:
						return '[u] [subotu] [u] LT';
					case 1:
					case 2:
					case 4:
					case 5:
						return '[u] dddd [u] LT';
				}
			},
			lastDay  : '[juče u] LT',
			lastWeek : function () {
				var lastWeekDays = [
					'[prošle] [nedjelje] [u] LT',
					'[prošlog] [ponedjeljka] [u] LT',
					'[prošlog] [utorka] [u] LT',
					'[prošle] [srijede] [u] LT',
					'[prošlog] [četvrtka] [u] LT',
					'[prošlog] [petka] [u] LT',
					'[prošle] [subote] [u] LT'
				];
				return lastWeekDays[this.day()];
			},
			sameElse : 'L'
		},
		relativeTime : {
			future : 'za %s',
			past   : 'prije %s',
			s      : 'nekoliko sekundi',
			ss     : translator.translate,
			m      : translator.translate,
			mm     : translator.translate,
			h      : translator.translate,
			hh     : translator.translate,
			d      : 'dan',
			dd     : translator.translate,
			M      : 'mjesec',
			MM     : translator.translate,
			y      : 'godinu',
			yy     : translator.translate
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('mi', {
		months: 'Kohi-tāte_Hui-tanguru_Poutū-te-rangi_Paenga-whāwhā_Haratua_Pipiri_Hōngoingoi_Here-turi-kōkā_Mahuru_Whiringa-ā-nuku_Whiringa-ā-rangi_Hakihea'.split('_'),
		monthsShort: 'Kohi_Hui_Pou_Pae_Hara_Pipi_Hōngoi_Here_Mahu_Whi-nu_Whi-ra_Haki'.split('_'),
		monthsRegex: /(?:['a-z\u0101\u014D\u016B]+\-?){1,3}/i,
		monthsStrictRegex: /(?:['a-z\u0101\u014D\u016B]+\-?){1,3}/i,
		monthsShortRegex: /(?:['a-z\u0101\u014D\u016B]+\-?){1,3}/i,
		monthsShortStrictRegex: /(?:['a-z\u0101\u014D\u016B]+\-?){1,2}/i,
		weekdays: 'Rātapu_Mane_Tūrei_Wenerei_Tāite_Paraire_Hātarei'.split('_'),
		weekdaysShort: 'Ta_Ma_Tū_We_Tāi_Pa_Hā'.split('_'),
		weekdaysMin: 'Ta_Ma_Tū_We_Tāi_Pa_Hā'.split('_'),
		longDateFormat: {
			LT: 'HH:mm',
			LTS: 'HH:mm:ss',
			L: 'DD/MM/YYYY',
			LL: 'D MMMM YYYY',
			LLL: 'D MMMM YYYY [i] HH:mm',
			LLLL: 'dddd, D MMMM YYYY [i] HH:mm'
		},
		calendar: {
			sameDay: '[i teie mahana, i] LT',
			nextDay: '[apopo i] LT',
			nextWeek: 'dddd [i] LT',
			lastDay: '[inanahi i] LT',
			lastWeek: 'dddd [whakamutunga i] LT',
			sameElse: 'L'
		},
		relativeTime: {
			future: 'i roto i %s',
			past: '%s i mua',
			s: 'te hēkona ruarua',
			ss: '%d hēkona',
			m: 'he meneti',
			mm: '%d meneti',
			h: 'te haora',
			hh: '%d haora',
			d: 'he ra',
			dd: '%d ra',
			M: 'he marama',
			MM: '%d marama',
			y: 'he tau',
			yy: '%d tau'
		},
		dayOfMonthOrdinalParse: /\d{1,2}º/,
		ordinal: '%dº',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('mk', {
		months : 'јануари_февруари_март_април_мај_јуни_јули_август_септември_октомври_ноември_декември'.split('_'),
		monthsShort : 'јан_фев_мар_апр_мај_јун_јул_авг_сеп_окт_ное_дек'.split('_'),
		weekdays : 'недела_понеделник_вторник_среда_четврток_петок_сабота'.split('_'),
		weekdaysShort : 'нед_пон_вто_сре_чет_пет_саб'.split('_'),
		weekdaysMin : 'нe_пo_вт_ср_че_пе_сa'.split('_'),
		longDateFormat : {
			LT : 'H:mm',
			LTS : 'H:mm:ss',
			L : 'D.MM.YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY H:mm',
			LLLL : 'dddd, D MMMM YYYY H:mm'
		},
		calendar : {
			sameDay : '[Денес во] LT',
			nextDay : '[Утре во] LT',
			nextWeek : '[Во] dddd [во] LT',
			lastDay : '[Вчера во] LT',
			lastWeek : function () {
				switch (this.day()) {
					case 0:
					case 3:
					case 6:
						return '[Изминатата] dddd [во] LT';
					case 1:
					case 2:
					case 4:
					case 5:
						return '[Изминатиот] dddd [во] LT';
				}
			},
			sameElse : 'L'
		},
		relativeTime : {
			future : 'после %s',
			past : 'пред %s',
			s : 'неколку секунди',
			ss : '%d секунди',
			m : 'минута',
			mm : '%d минути',
			h : 'час',
			hh : '%d часа',
			d : 'ден',
			dd : '%d дена',
			M : 'месец',
			MM : '%d месеци',
			y : 'година',
			yy : '%d години'
		},
		dayOfMonthOrdinalParse: /\d{1,2}-(ев|ен|ти|ви|ри|ми)/,
		ordinal : function (number) {
			var lastDigit = number % 10,
				last2Digits = number % 100;
			if (number === 0) {
				return number + '-ев';
			} else if (last2Digits === 0) {
				return number + '-ен';
			} else if (last2Digits > 10 && last2Digits < 20) {
				return number + '-ти';
			} else if (lastDigit === 1) {
				return number + '-ви';
			} else if (lastDigit === 2) {
				return number + '-ри';
			} else if (lastDigit === 7 || lastDigit === 8) {
				return number + '-ми';
			} else {
				return number + '-ти';
			}
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('ml', {
		months : 'ജനുവരി_ഫെബ്രുവരി_മാർച്ച്_ഏപ്രിൽ_മേയ്_ജൂൺ_ജൂലൈ_ഓഗസ്റ്റ്_സെപ്റ്റംബർ_ഒക്ടോബർ_നവംബർ_ഡിസംബർ'.split('_'),
		monthsShort : 'ജനു._ഫെബ്രു._മാർ._ഏപ്രി._മേയ്_ജൂൺ_ജൂലൈ._ഓഗ._സെപ്റ്റ._ഒക്ടോ._നവം._ഡിസം.'.split('_'),
		monthsParseExact : true,
		weekdays : 'ഞായറാഴ്ച_തിങ്കളാഴ്ച_ചൊവ്വാഴ്ച_ബുധനാഴ്ച_വ്യാഴാഴ്ച_വെള്ളിയാഴ്ച_ശനിയാഴ്ച'.split('_'),
		weekdaysShort : 'ഞായർ_തിങ്കൾ_ചൊവ്വ_ബുധൻ_വ്യാഴം_വെള്ളി_ശനി'.split('_'),
		weekdaysMin : 'ഞാ_തി_ചൊ_ബു_വ്യാ_വെ_ശ'.split('_'),
		longDateFormat : {
			LT : 'A h:mm -നു',
			LTS : 'A h:mm:ss -നു',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY, A h:mm -നു',
			LLLL : 'dddd, D MMMM YYYY, A h:mm -നു'
		},
		calendar : {
			sameDay : '[ഇന്ന്] LT',
			nextDay : '[നാളെ] LT',
			nextWeek : 'dddd, LT',
			lastDay : '[ഇന്നലെ] LT',
			lastWeek : '[കഴിഞ്ഞ] dddd, LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : '%s കഴിഞ്ഞ്',
			past : '%s മുൻപ്',
			s : 'അൽപ നിമിഷങ്ങൾ',
			ss : '%d സെക്കൻഡ്',
			m : 'ഒരു മിനിറ്റ്',
			mm : '%d മിനിറ്റ്',
			h : 'ഒരു മണിക്കൂർ',
			hh : '%d മണിക്കൂർ',
			d : 'ഒരു ദിവസം',
			dd : '%d ദിവസം',
			M : 'ഒരു മാസം',
			MM : '%d മാസം',
			y : 'ഒരു വർഷം',
			yy : '%d വർഷം'
		},
		meridiemParse: /രാത്രി|രാവിലെ|ഉച്ച കഴിഞ്ഞ്|വൈകുന്നേരം|രാത്രി/i,
		meridiemHour : function (hour, meridiem) {
			if (hour === 12) {
				hour = 0;
			}
			if ((meridiem === 'രാത്രി' && hour >= 4) ||
				meridiem === 'ഉച്ച കഴിഞ്ഞ്' ||
				meridiem === 'വൈകുന്നേരം') {
				return hour + 12;
			} else {
				return hour;
			}
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 4) {
				return 'രാത്രി';
			} else if (hour < 12) {
				return 'രാവിലെ';
			} else if (hour < 17) {
				return 'ഉച്ച കഴിഞ്ഞ്';
			} else if (hour < 20) {
				return 'വൈകുന്നേരം';
			} else {
				return 'രാത്രി';
			}
		}
	});

	//! moment.js locale configuration

	function translate$7(number, withoutSuffix, key, isFuture) {
		switch (key) {
			case 's':
				return withoutSuffix ? 'хэдхэн секунд' : 'хэдхэн секундын';
			case 'ss':
				return number + (withoutSuffix ? ' секунд' : ' секундын');
			case 'm':
			case 'mm':
				return number + (withoutSuffix ? ' минут' : ' минутын');
			case 'h':
			case 'hh':
				return number + (withoutSuffix ? ' цаг' : ' цагийн');
			case 'd':
			case 'dd':
				return number + (withoutSuffix ? ' өдөр' : ' өдрийн');
			case 'M':
			case 'MM':
				return number + (withoutSuffix ? ' сар' : ' сарын');
			case 'y':
			case 'yy':
				return number + (withoutSuffix ? ' жил' : ' жилийн');
			default:
				return number;
		}
	}

	hooks.defineLocale('mn', {
		months : 'Нэгдүгээр сар_Хоёрдугаар сар_Гуравдугаар сар_Дөрөвдүгээр сар_Тавдугаар сар_Зургадугаар сар_Долдугаар сар_Наймдугаар сар_Есдүгээр сар_Аравдугаар сар_Арван нэгдүгээр сар_Арван хоёрдугаар сар'.split('_'),
		monthsShort : '1 сар_2 сар_3 сар_4 сар_5 сар_6 сар_7 сар_8 сар_9 сар_10 сар_11 сар_12 сар'.split('_'),
		monthsParseExact : true,
		weekdays : 'Ням_Даваа_Мягмар_Лхагва_Пүрэв_Баасан_Бямба'.split('_'),
		weekdaysShort : 'Ням_Дав_Мяг_Лха_Пүр_Баа_Бям'.split('_'),
		weekdaysMin : 'Ня_Да_Мя_Лх_Пү_Ба_Бя'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'YYYY-MM-DD',
			LL : 'YYYY оны MMMMын D',
			LLL : 'YYYY оны MMMMын D HH:mm',
			LLLL : 'dddd, YYYY оны MMMMын D HH:mm'
		},
		meridiemParse: /ҮӨ|ҮХ/i,
		isPM : function (input) {
			return input === 'ҮХ';
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 12) {
				return 'ҮӨ';
			} else {
				return 'ҮХ';
			}
		},
		calendar : {
			sameDay : '[Өнөөдөр] LT',
			nextDay : '[Маргааш] LT',
			nextWeek : '[Ирэх] dddd LT',
			lastDay : '[Өчигдөр] LT',
			lastWeek : '[Өнгөрсөн] dddd LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : '%s дараа',
			past : '%s өмнө',
			s : translate$7,
			ss : translate$7,
			m : translate$7,
			mm : translate$7,
			h : translate$7,
			hh : translate$7,
			d : translate$7,
			dd : translate$7,
			M : translate$7,
			MM : translate$7,
			y : translate$7,
			yy : translate$7
		},
		dayOfMonthOrdinalParse: /\d{1,2} өдөр/,
		ordinal : function (number, period) {
			switch (period) {
				case 'd':
				case 'D':
				case 'DDD':
					return number + ' өдөр';
				default:
					return number;
			}
		}
	});

	//! moment.js locale configuration

	var symbolMap$b = {
			'1': '१',
			'2': '२',
			'3': '३',
			'4': '४',
			'5': '५',
			'6': '६',
			'7': '७',
			'8': '८',
			'9': '९',
			'0': '०'
		},
		numberMap$a = {
			'१': '1',
			'२': '2',
			'३': '3',
			'४': '4',
			'५': '5',
			'६': '6',
			'७': '7',
			'८': '8',
			'९': '9',
			'०': '0'
		};

	function relativeTimeMr(number, withoutSuffix, string, isFuture)
	{
		var output = '';
		if (withoutSuffix) {
			switch (string) {
				case 's': output = 'काही सेकंद'; break;
				case 'ss': output = '%d सेकंद'; break;
				case 'm': output = 'एक मिनिट'; break;
				case 'mm': output = '%d मिनिटे'; break;
				case 'h': output = 'एक तास'; break;
				case 'hh': output = '%d तास'; break;
				case 'd': output = 'एक दिवस'; break;
				case 'dd': output = '%d दिवस'; break;
				case 'M': output = 'एक महिना'; break;
				case 'MM': output = '%d महिने'; break;
				case 'y': output = 'एक वर्ष'; break;
				case 'yy': output = '%d वर्षे'; break;
			}
		}
		else {
			switch (string) {
				case 's': output = 'काही सेकंदां'; break;
				case 'ss': output = '%d सेकंदां'; break;
				case 'm': output = 'एका मिनिटा'; break;
				case 'mm': output = '%d मिनिटां'; break;
				case 'h': output = 'एका तासा'; break;
				case 'hh': output = '%d तासां'; break;
				case 'd': output = 'एका दिवसा'; break;
				case 'dd': output = '%d दिवसां'; break;
				case 'M': output = 'एका महिन्या'; break;
				case 'MM': output = '%d महिन्यां'; break;
				case 'y': output = 'एका वर्षा'; break;
				case 'yy': output = '%d वर्षां'; break;
			}
		}
		return output.replace(/%d/i, number);
	}

	hooks.defineLocale('mr', {
		months : 'जानेवारी_फेब्रुवारी_मार्च_एप्रिल_मे_जून_जुलै_ऑगस्ट_सप्टेंबर_ऑक्टोबर_नोव्हेंबर_डिसेंबर'.split('_'),
		monthsShort: 'जाने._फेब्रु._मार्च._एप्रि._मे._जून._जुलै._ऑग._सप्टें._ऑक्टो._नोव्हें._डिसें.'.split('_'),
		monthsParseExact : true,
		weekdays : 'रविवार_सोमवार_मंगळवार_बुधवार_गुरूवार_शुक्रवार_शनिवार'.split('_'),
		weekdaysShort : 'रवि_सोम_मंगळ_बुध_गुरू_शुक्र_शनि'.split('_'),
		weekdaysMin : 'र_सो_मं_बु_गु_शु_श'.split('_'),
		longDateFormat : {
			LT : 'A h:mm वाजता',
			LTS : 'A h:mm:ss वाजता',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY, A h:mm वाजता',
			LLLL : 'dddd, D MMMM YYYY, A h:mm वाजता'
		},
		calendar : {
			sameDay : '[आज] LT',
			nextDay : '[उद्या] LT',
			nextWeek : 'dddd, LT',
			lastDay : '[काल] LT',
			lastWeek: '[मागील] dddd, LT',
			sameElse : 'L'
		},
		relativeTime : {
			future: '%sमध्ये',
			past: '%sपूर्वी',
			s: relativeTimeMr,
			ss: relativeTimeMr,
			m: relativeTimeMr,
			mm: relativeTimeMr,
			h: relativeTimeMr,
			hh: relativeTimeMr,
			d: relativeTimeMr,
			dd: relativeTimeMr,
			M: relativeTimeMr,
			MM: relativeTimeMr,
			y: relativeTimeMr,
			yy: relativeTimeMr
		},
		preparse: function (string) {
			return string.replace(/[१२३४५६७८९०]/g, function (match) {
				return numberMap$a[match];
			});
		},
		postformat: function (string) {
			return string.replace(/\d/g, function (match) {
				return symbolMap$b[match];
			});
		},
		meridiemParse: /रात्री|सकाळी|दुपारी|सायंकाळी/,
		meridiemHour : function (hour, meridiem) {
			if (hour === 12) {
				hour = 0;
			}
			if (meridiem === 'रात्री') {
				return hour < 4 ? hour : hour + 12;
			} else if (meridiem === 'सकाळी') {
				return hour;
			} else if (meridiem === 'दुपारी') {
				return hour >= 10 ? hour : hour + 12;
			} else if (meridiem === 'सायंकाळी') {
				return hour + 12;
			}
		},
		meridiem: function (hour, minute, isLower) {
			if (hour < 4) {
				return 'रात्री';
			} else if (hour < 10) {
				return 'सकाळी';
			} else if (hour < 17) {
				return 'दुपारी';
			} else if (hour < 20) {
				return 'सायंकाळी';
			} else {
				return 'रात्री';
			}
		},
		week : {
			dow : 0, // Sunday is the first day of the week.
			doy : 6  // The week that contains Jan 6th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('ms-my', {
		months : 'Januari_Februari_Mac_April_Mei_Jun_Julai_Ogos_September_Oktober_November_Disember'.split('_'),
		monthsShort : 'Jan_Feb_Mac_Apr_Mei_Jun_Jul_Ogs_Sep_Okt_Nov_Dis'.split('_'),
		weekdays : 'Ahad_Isnin_Selasa_Rabu_Khamis_Jumaat_Sabtu'.split('_'),
		weekdaysShort : 'Ahd_Isn_Sel_Rab_Kha_Jum_Sab'.split('_'),
		weekdaysMin : 'Ah_Is_Sl_Rb_Km_Jm_Sb'.split('_'),
		longDateFormat : {
			LT : 'HH.mm',
			LTS : 'HH.mm.ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY [pukul] HH.mm',
			LLLL : 'dddd, D MMMM YYYY [pukul] HH.mm'
		},
		meridiemParse: /pagi|tengahari|petang|malam/,
		meridiemHour: function (hour, meridiem) {
			if (hour === 12) {
				hour = 0;
			}
			if (meridiem === 'pagi') {
				return hour;
			} else if (meridiem === 'tengahari') {
				return hour >= 11 ? hour : hour + 12;
			} else if (meridiem === 'petang' || meridiem === 'malam') {
				return hour + 12;
			}
		},
		meridiem : function (hours, minutes, isLower) {
			if (hours < 11) {
				return 'pagi';
			} else if (hours < 15) {
				return 'tengahari';
			} else if (hours < 19) {
				return 'petang';
			} else {
				return 'malam';
			}
		},
		calendar : {
			sameDay : '[Hari ini pukul] LT',
			nextDay : '[Esok pukul] LT',
			nextWeek : 'dddd [pukul] LT',
			lastDay : '[Kelmarin pukul] LT',
			lastWeek : 'dddd [lepas pukul] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'dalam %s',
			past : '%s yang lepas',
			s : 'beberapa saat',
			ss : '%d saat',
			m : 'seminit',
			mm : '%d minit',
			h : 'sejam',
			hh : '%d jam',
			d : 'sehari',
			dd : '%d hari',
			M : 'sebulan',
			MM : '%d bulan',
			y : 'setahun',
			yy : '%d tahun'
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('ms', {
		months : 'Januari_Februari_Mac_April_Mei_Jun_Julai_Ogos_September_Oktober_November_Disember'.split('_'),
		monthsShort : 'Jan_Feb_Mac_Apr_Mei_Jun_Jul_Ogs_Sep_Okt_Nov_Dis'.split('_'),
		weekdays : 'Ahad_Isnin_Selasa_Rabu_Khamis_Jumaat_Sabtu'.split('_'),
		weekdaysShort : 'Ahd_Isn_Sel_Rab_Kha_Jum_Sab'.split('_'),
		weekdaysMin : 'Ah_Is_Sl_Rb_Km_Jm_Sb'.split('_'),
		longDateFormat : {
			LT : 'HH.mm',
			LTS : 'HH.mm.ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY [pukul] HH.mm',
			LLLL : 'dddd, D MMMM YYYY [pukul] HH.mm'
		},
		meridiemParse: /pagi|tengahari|petang|malam/,
		meridiemHour: function (hour, meridiem) {
			if (hour === 12) {
				hour = 0;
			}
			if (meridiem === 'pagi') {
				return hour;
			} else if (meridiem === 'tengahari') {
				return hour >= 11 ? hour : hour + 12;
			} else if (meridiem === 'petang' || meridiem === 'malam') {
				return hour + 12;
			}
		},
		meridiem : function (hours, minutes, isLower) {
			if (hours < 11) {
				return 'pagi';
			} else if (hours < 15) {
				return 'tengahari';
			} else if (hours < 19) {
				return 'petang';
			} else {
				return 'malam';
			}
		},
		calendar : {
			sameDay : '[Hari ini pukul] LT',
			nextDay : '[Esok pukul] LT',
			nextWeek : 'dddd [pukul] LT',
			lastDay : '[Kelmarin pukul] LT',
			lastWeek : 'dddd [lepas pukul] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'dalam %s',
			past : '%s yang lepas',
			s : 'beberapa saat',
			ss : '%d saat',
			m : 'seminit',
			mm : '%d minit',
			h : 'sejam',
			hh : '%d jam',
			d : 'sehari',
			dd : '%d hari',
			M : 'sebulan',
			MM : '%d bulan',
			y : 'setahun',
			yy : '%d tahun'
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('mt', {
		months : 'Jannar_Frar_Marzu_April_Mejju_Ġunju_Lulju_Awwissu_Settembru_Ottubru_Novembru_Diċembru'.split('_'),
		monthsShort : 'Jan_Fra_Mar_Apr_Mej_Ġun_Lul_Aww_Set_Ott_Nov_Diċ'.split('_'),
		weekdays : 'Il-Ħadd_It-Tnejn_It-Tlieta_L-Erbgħa_Il-Ħamis_Il-Ġimgħa_Is-Sibt'.split('_'),
		weekdaysShort : 'Ħad_Tne_Tli_Erb_Ħam_Ġim_Sib'.split('_'),
		weekdaysMin : 'Ħa_Tn_Tl_Er_Ħa_Ġi_Si'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd, D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay : '[Illum fil-]LT',
			nextDay : '[Għada fil-]LT',
			nextWeek : 'dddd [fil-]LT',
			lastDay : '[Il-bieraħ fil-]LT',
			lastWeek : 'dddd [li għadda] [fil-]LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'f’ %s',
			past : '%s ilu',
			s : 'ftit sekondi',
			ss : '%d sekondi',
			m : 'minuta',
			mm : '%d minuti',
			h : 'siegħa',
			hh : '%d siegħat',
			d : 'ġurnata',
			dd : '%d ġranet',
			M : 'xahar',
			MM : '%d xhur',
			y : 'sena',
			yy : '%d sni'
		},
		dayOfMonthOrdinalParse : /\d{1,2}º/,
		ordinal: '%dº',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var symbolMap$c = {
		'1': '၁',
		'2': '၂',
		'3': '၃',
		'4': '၄',
		'5': '၅',
		'6': '၆',
		'7': '၇',
		'8': '၈',
		'9': '၉',
		'0': '၀'
	}, numberMap$b = {
		'၁': '1',
		'၂': '2',
		'၃': '3',
		'၄': '4',
		'၅': '5',
		'၆': '6',
		'၇': '7',
		'၈': '8',
		'၉': '9',
		'၀': '0'
	};

	hooks.defineLocale('my', {
		months: 'ဇန်နဝါရီ_ဖေဖော်ဝါရီ_မတ်_ဧပြီ_မေ_ဇွန်_ဇူလိုင်_သြဂုတ်_စက်တင်ဘာ_အောက်တိုဘာ_နိုဝင်ဘာ_ဒီဇင်ဘာ'.split('_'),
		monthsShort: 'ဇန်_ဖေ_မတ်_ပြီ_မေ_ဇွန်_လိုင်_သြ_စက်_အောက်_နို_ဒီ'.split('_'),
		weekdays: 'တနင်္ဂနွေ_တနင်္လာ_အင်္ဂါ_ဗုဒ္ဓဟူး_ကြာသပတေး_သောကြာ_စနေ'.split('_'),
		weekdaysShort: 'နွေ_လာ_ဂါ_ဟူး_ကြာ_သော_နေ'.split('_'),
		weekdaysMin: 'နွေ_လာ_ဂါ_ဟူး_ကြာ_သော_နေ'.split('_'),

		longDateFormat: {
			LT: 'HH:mm',
			LTS: 'HH:mm:ss',
			L: 'DD/MM/YYYY',
			LL: 'D MMMM YYYY',
			LLL: 'D MMMM YYYY HH:mm',
			LLLL: 'dddd D MMMM YYYY HH:mm'
		},
		calendar: {
			sameDay: '[ယနေ.] LT [မှာ]',
			nextDay: '[မနက်ဖြန်] LT [မှာ]',
			nextWeek: 'dddd LT [မှာ]',
			lastDay: '[မနေ.က] LT [မှာ]',
			lastWeek: '[ပြီးခဲ့သော] dddd LT [မှာ]',
			sameElse: 'L'
		},
		relativeTime: {
			future: 'လာမည့် %s မှာ',
			past: 'လွန်ခဲ့သော %s က',
			s: 'စက္ကန်.အနည်းငယ်',
			ss : '%d စက္ကန့်',
			m: 'တစ်မိနစ်',
			mm: '%d မိနစ်',
			h: 'တစ်နာရီ',
			hh: '%d နာရီ',
			d: 'တစ်ရက်',
			dd: '%d ရက်',
			M: 'တစ်လ',
			MM: '%d လ',
			y: 'တစ်နှစ်',
			yy: '%d နှစ်'
		},
		preparse: function (string) {
			return string.replace(/[၁၂၃၄၅၆၇၈၉၀]/g, function (match) {
				return numberMap$b[match];
			});
		},
		postformat: function (string) {
			return string.replace(/\d/g, function (match) {
				return symbolMap$c[match];
			});
		},
		week: {
			dow: 1, // Monday is the first day of the week.
			doy: 4 // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('nb', {
		months : 'januar_februar_mars_april_mai_juni_juli_august_september_oktober_november_desember'.split('_'),
		monthsShort : 'jan._feb._mars_april_mai_juni_juli_aug._sep._okt._nov._des.'.split('_'),
		monthsParseExact : true,
		weekdays : 'søndag_mandag_tirsdag_onsdag_torsdag_fredag_lørdag'.split('_'),
		weekdaysShort : 'sø._ma._ti._on._to._fr._lø.'.split('_'),
		weekdaysMin : 'sø_ma_ti_on_to_fr_lø'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D. MMMM YYYY',
			LLL : 'D. MMMM YYYY [kl.] HH:mm',
			LLLL : 'dddd D. MMMM YYYY [kl.] HH:mm'
		},
		calendar : {
			sameDay: '[i dag kl.] LT',
			nextDay: '[i morgen kl.] LT',
			nextWeek: 'dddd [kl.] LT',
			lastDay: '[i går kl.] LT',
			lastWeek: '[forrige] dddd [kl.] LT',
			sameElse: 'L'
		},
		relativeTime : {
			future : 'om %s',
			past : '%s siden',
			s : 'noen sekunder',
			ss : '%d sekunder',
			m : 'ett minutt',
			mm : '%d minutter',
			h : 'en time',
			hh : '%d timer',
			d : 'en dag',
			dd : '%d dager',
			M : 'en måned',
			MM : '%d måneder',
			y : 'ett år',
			yy : '%d år'
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var symbolMap$d = {
			'1': '१',
			'2': '२',
			'3': '३',
			'4': '४',
			'5': '५',
			'6': '६',
			'7': '७',
			'8': '८',
			'9': '९',
			'0': '०'
		},
		numberMap$c = {
			'१': '1',
			'२': '2',
			'३': '3',
			'४': '4',
			'५': '5',
			'६': '6',
			'७': '7',
			'८': '8',
			'९': '9',
			'०': '0'
		};

	hooks.defineLocale('ne', {
		months : 'जनवरी_फेब्रुवरी_मार्च_अप्रिल_मई_जुन_जुलाई_अगष्ट_सेप्टेम्बर_अक्टोबर_नोभेम्बर_डिसेम्बर'.split('_'),
		monthsShort : 'जन._फेब्रु._मार्च_अप्रि._मई_जुन_जुलाई._अग._सेप्ट._अक्टो._नोभे._डिसे.'.split('_'),
		monthsParseExact : true,
		weekdays : 'आइतबार_सोमबार_मङ्गलबार_बुधबार_बिहिबार_शुक्रबार_शनिबार'.split('_'),
		weekdaysShort : 'आइत._सोम._मङ्गल._बुध._बिहि._शुक्र._शनि.'.split('_'),
		weekdaysMin : 'आ._सो._मं._बु._बि._शु._श.'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'Aको h:mm बजे',
			LTS : 'Aको h:mm:ss बजे',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY, Aको h:mm बजे',
			LLLL : 'dddd, D MMMM YYYY, Aको h:mm बजे'
		},
		preparse: function (string) {
			return string.replace(/[१२३४५६७८९०]/g, function (match) {
				return numberMap$c[match];
			});
		},
		postformat: function (string) {
			return string.replace(/\d/g, function (match) {
				return symbolMap$d[match];
			});
		},
		meridiemParse: /राति|बिहान|दिउँसो|साँझ/,
		meridiemHour : function (hour, meridiem) {
			if (hour === 12) {
				hour = 0;
			}
			if (meridiem === 'राति') {
				return hour < 4 ? hour : hour + 12;
			} else if (meridiem === 'बिहान') {
				return hour;
			} else if (meridiem === 'दिउँसो') {
				return hour >= 10 ? hour : hour + 12;
			} else if (meridiem === 'साँझ') {
				return hour + 12;
			}
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 3) {
				return 'राति';
			} else if (hour < 12) {
				return 'बिहान';
			} else if (hour < 16) {
				return 'दिउँसो';
			} else if (hour < 20) {
				return 'साँझ';
			} else {
				return 'राति';
			}
		},
		calendar : {
			sameDay : '[आज] LT',
			nextDay : '[भोलि] LT',
			nextWeek : '[आउँदो] dddd[,] LT',
			lastDay : '[हिजो] LT',
			lastWeek : '[गएको] dddd[,] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : '%sमा',
			past : '%s अगाडि',
			s : 'केही क्षण',
			ss : '%d सेकेण्ड',
			m : 'एक मिनेट',
			mm : '%d मिनेट',
			h : 'एक घण्टा',
			hh : '%d घण्टा',
			d : 'एक दिन',
			dd : '%d दिन',
			M : 'एक महिना',
			MM : '%d महिना',
			y : 'एक बर्ष',
			yy : '%d बर्ष'
		},
		week : {
			dow : 0, // Sunday is the first day of the week.
			doy : 6  // The week that contains Jan 6th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var monthsShortWithDots$1 = 'jan._feb._mrt._apr._mei_jun._jul._aug._sep._okt._nov._dec.'.split('_'),
		monthsShortWithoutDots$1 = 'jan_feb_mrt_apr_mei_jun_jul_aug_sep_okt_nov_dec'.split('_');

	var monthsParse$4 = [/^jan/i, /^feb/i, /^maart|mrt.?$/i, /^apr/i, /^mei$/i, /^jun[i.]?$/i, /^jul[i.]?$/i, /^aug/i, /^sep/i, /^okt/i, /^nov/i, /^dec/i];
	var monthsRegex$5 = /^(januari|februari|maart|april|mei|ju[nl]i|augustus|september|oktober|november|december|jan\.?|feb\.?|mrt\.?|apr\.?|ju[nl]\.?|aug\.?|sep\.?|okt\.?|nov\.?|dec\.?)/i;

	hooks.defineLocale('nl-be', {
		months : 'januari_februari_maart_april_mei_juni_juli_augustus_september_oktober_november_december'.split('_'),
		monthsShort : function (m, format) {
			if (!m) {
				return monthsShortWithDots$1;
			} else if (/-MMM-/.test(format)) {
				return monthsShortWithoutDots$1[m.month()];
			} else {
				return monthsShortWithDots$1[m.month()];
			}
		},

		monthsRegex: monthsRegex$5,
		monthsShortRegex: monthsRegex$5,
		monthsStrictRegex: /^(januari|februari|maart|april|mei|ju[nl]i|augustus|september|oktober|november|december)/i,
		monthsShortStrictRegex: /^(jan\.?|feb\.?|mrt\.?|apr\.?|mei|ju[nl]\.?|aug\.?|sep\.?|okt\.?|nov\.?|dec\.?)/i,

		monthsParse : monthsParse$4,
		longMonthsParse : monthsParse$4,
		shortMonthsParse : monthsParse$4,

		weekdays : 'zondag_maandag_dinsdag_woensdag_donderdag_vrijdag_zaterdag'.split('_'),
		weekdaysShort : 'zo._ma._di._wo._do._vr._za.'.split('_'),
		weekdaysMin : 'zo_ma_di_wo_do_vr_za'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay: '[vandaag om] LT',
			nextDay: '[morgen om] LT',
			nextWeek: 'dddd [om] LT',
			lastDay: '[gisteren om] LT',
			lastWeek: '[afgelopen] dddd [om] LT',
			sameElse: 'L'
		},
		relativeTime : {
			future : 'over %s',
			past : '%s geleden',
			s : 'een paar seconden',
			ss : '%d seconden',
			m : 'één minuut',
			mm : '%d minuten',
			h : 'één uur',
			hh : '%d uur',
			d : 'één dag',
			dd : '%d dagen',
			M : 'één maand',
			MM : '%d maanden',
			y : 'één jaar',
			yy : '%d jaar'
		},
		dayOfMonthOrdinalParse: /\d{1,2}(ste|de)/,
		ordinal : function (number) {
			return number + ((number === 1 || number === 8 || number >= 20) ? 'ste' : 'de');
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var monthsShortWithDots$2 = 'jan._feb._mrt._apr._mei_jun._jul._aug._sep._okt._nov._dec.'.split('_'),
		monthsShortWithoutDots$2 = 'jan_feb_mrt_apr_mei_jun_jul_aug_sep_okt_nov_dec'.split('_');

	var monthsParse$5 = [/^jan/i, /^feb/i, /^maart|mrt.?$/i, /^apr/i, /^mei$/i, /^jun[i.]?$/i, /^jul[i.]?$/i, /^aug/i, /^sep/i, /^okt/i, /^nov/i, /^dec/i];
	var monthsRegex$6 = /^(januari|februari|maart|april|mei|ju[nl]i|augustus|september|oktober|november|december|jan\.?|feb\.?|mrt\.?|apr\.?|ju[nl]\.?|aug\.?|sep\.?|okt\.?|nov\.?|dec\.?)/i;

	hooks.defineLocale('nl', {
		months : 'januari_februari_maart_april_mei_juni_juli_augustus_september_oktober_november_december'.split('_'),
		monthsShort : function (m, format) {
			if (!m) {
				return monthsShortWithDots$2;
			} else if (/-MMM-/.test(format)) {
				return monthsShortWithoutDots$2[m.month()];
			} else {
				return monthsShortWithDots$2[m.month()];
			}
		},

		monthsRegex: monthsRegex$6,
		monthsShortRegex: monthsRegex$6,
		monthsStrictRegex: /^(januari|februari|maart|april|mei|ju[nl]i|augustus|september|oktober|november|december)/i,
		monthsShortStrictRegex: /^(jan\.?|feb\.?|mrt\.?|apr\.?|mei|ju[nl]\.?|aug\.?|sep\.?|okt\.?|nov\.?|dec\.?)/i,

		monthsParse : monthsParse$5,
		longMonthsParse : monthsParse$5,
		shortMonthsParse : monthsParse$5,

		weekdays : 'zondag_maandag_dinsdag_woensdag_donderdag_vrijdag_zaterdag'.split('_'),
		weekdaysShort : 'zo._ma._di._wo._do._vr._za.'.split('_'),
		weekdaysMin : 'zo_ma_di_wo_do_vr_za'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD-MM-YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay: '[vandaag om] LT',
			nextDay: '[morgen om] LT',
			nextWeek: 'dddd [om] LT',
			lastDay: '[gisteren om] LT',
			lastWeek: '[afgelopen] dddd [om] LT',
			sameElse: 'L'
		},
		relativeTime : {
			future : 'over %s',
			past : '%s geleden',
			s : 'een paar seconden',
			ss : '%d seconden',
			m : 'één minuut',
			mm : '%d minuten',
			h : 'één uur',
			hh : '%d uur',
			d : 'één dag',
			dd : '%d dagen',
			M : 'één maand',
			MM : '%d maanden',
			y : 'één jaar',
			yy : '%d jaar'
		},
		dayOfMonthOrdinalParse: /\d{1,2}(ste|de)/,
		ordinal : function (number) {
			return number + ((number === 1 || number === 8 || number >= 20) ? 'ste' : 'de');
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('nn', {
		months : 'januar_februar_mars_april_mai_juni_juli_august_september_oktober_november_desember'.split('_'),
		monthsShort : 'jan_feb_mar_apr_mai_jun_jul_aug_sep_okt_nov_des'.split('_'),
		weekdays : 'sundag_måndag_tysdag_onsdag_torsdag_fredag_laurdag'.split('_'),
		weekdaysShort : 'sun_mån_tys_ons_tor_fre_lau'.split('_'),
		weekdaysMin : 'su_må_ty_on_to_fr_lø'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D. MMMM YYYY',
			LLL : 'D. MMMM YYYY [kl.] H:mm',
			LLLL : 'dddd D. MMMM YYYY [kl.] HH:mm'
		},
		calendar : {
			sameDay: '[I dag klokka] LT',
			nextDay: '[I morgon klokka] LT',
			nextWeek: 'dddd [klokka] LT',
			lastDay: '[I går klokka] LT',
			lastWeek: '[Føregåande] dddd [klokka] LT',
			sameElse: 'L'
		},
		relativeTime : {
			future : 'om %s',
			past : '%s sidan',
			s : 'nokre sekund',
			ss : '%d sekund',
			m : 'eit minutt',
			mm : '%d minutt',
			h : 'ein time',
			hh : '%d timar',
			d : 'ein dag',
			dd : '%d dagar',
			M : 'ein månad',
			MM : '%d månader',
			y : 'eit år',
			yy : '%d år'
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var symbolMap$e = {
			'1': '੧',
			'2': '੨',
			'3': '੩',
			'4': '੪',
			'5': '੫',
			'6': '੬',
			'7': '੭',
			'8': '੮',
			'9': '੯',
			'0': '੦'
		},
		numberMap$d = {
			'੧': '1',
			'੨': '2',
			'੩': '3',
			'੪': '4',
			'੫': '5',
			'੬': '6',
			'੭': '7',
			'੮': '8',
			'੯': '9',
			'੦': '0'
		};

	hooks.defineLocale('pa-in', {
		// There are months name as per Nanakshahi Calendar but they are not used as rigidly in modern Punjabi.
		months : 'ਜਨਵਰੀ_ਫ਼ਰਵਰੀ_ਮਾਰਚ_ਅਪ੍ਰੈਲ_ਮਈ_ਜੂਨ_ਜੁਲਾਈ_ਅਗਸਤ_ਸਤੰਬਰ_ਅਕਤੂਬਰ_ਨਵੰਬਰ_ਦਸੰਬਰ'.split('_'),
		monthsShort : 'ਜਨਵਰੀ_ਫ਼ਰਵਰੀ_ਮਾਰਚ_ਅਪ੍ਰੈਲ_ਮਈ_ਜੂਨ_ਜੁਲਾਈ_ਅਗਸਤ_ਸਤੰਬਰ_ਅਕਤੂਬਰ_ਨਵੰਬਰ_ਦਸੰਬਰ'.split('_'),
		weekdays : 'ਐਤਵਾਰ_ਸੋਮਵਾਰ_ਮੰਗਲਵਾਰ_ਬੁਧਵਾਰ_ਵੀਰਵਾਰ_ਸ਼ੁੱਕਰਵਾਰ_ਸ਼ਨੀਚਰਵਾਰ'.split('_'),
		weekdaysShort : 'ਐਤ_ਸੋਮ_ਮੰਗਲ_ਬੁਧ_ਵੀਰ_ਸ਼ੁਕਰ_ਸ਼ਨੀ'.split('_'),
		weekdaysMin : 'ਐਤ_ਸੋਮ_ਮੰਗਲ_ਬੁਧ_ਵੀਰ_ਸ਼ੁਕਰ_ਸ਼ਨੀ'.split('_'),
		longDateFormat : {
			LT : 'A h:mm ਵਜੇ',
			LTS : 'A h:mm:ss ਵਜੇ',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY, A h:mm ਵਜੇ',
			LLLL : 'dddd, D MMMM YYYY, A h:mm ਵਜੇ'
		},
		calendar : {
			sameDay : '[ਅਜ] LT',
			nextDay : '[ਕਲ] LT',
			nextWeek : '[ਅਗਲਾ] dddd, LT',
			lastDay : '[ਕਲ] LT',
			lastWeek : '[ਪਿਛਲੇ] dddd, LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : '%s ਵਿੱਚ',
			past : '%s ਪਿਛਲੇ',
			s : 'ਕੁਝ ਸਕਿੰਟ',
			ss : '%d ਸਕਿੰਟ',
			m : 'ਇਕ ਮਿੰਟ',
			mm : '%d ਮਿੰਟ',
			h : 'ਇੱਕ ਘੰਟਾ',
			hh : '%d ਘੰਟੇ',
			d : 'ਇੱਕ ਦਿਨ',
			dd : '%d ਦਿਨ',
			M : 'ਇੱਕ ਮਹੀਨਾ',
			MM : '%d ਮਹੀਨੇ',
			y : 'ਇੱਕ ਸਾਲ',
			yy : '%d ਸਾਲ'
		},
		preparse: function (string) {
			return string.replace(/[੧੨੩੪੫੬੭੮੯੦]/g, function (match) {
				return numberMap$d[match];
			});
		},
		postformat: function (string) {
			return string.replace(/\d/g, function (match) {
				return symbolMap$e[match];
			});
		},
		// Punjabi notation for meridiems are quite fuzzy in practice. While there exists
		// a rigid notion of a 'Pahar' it is not used as rigidly in modern Punjabi.
		meridiemParse: /ਰਾਤ|ਸਵੇਰ|ਦੁਪਹਿਰ|ਸ਼ਾਮ/,
		meridiemHour : function (hour, meridiem) {
			if (hour === 12) {
				hour = 0;
			}
			if (meridiem === 'ਰਾਤ') {
				return hour < 4 ? hour : hour + 12;
			} else if (meridiem === 'ਸਵੇਰ') {
				return hour;
			} else if (meridiem === 'ਦੁਪਹਿਰ') {
				return hour >= 10 ? hour : hour + 12;
			} else if (meridiem === 'ਸ਼ਾਮ') {
				return hour + 12;
			}
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 4) {
				return 'ਰਾਤ';
			} else if (hour < 10) {
				return 'ਸਵੇਰ';
			} else if (hour < 17) {
				return 'ਦੁਪਹਿਰ';
			} else if (hour < 20) {
				return 'ਸ਼ਾਮ';
			} else {
				return 'ਰਾਤ';
			}
		},
		week : {
			dow : 0, // Sunday is the first day of the week.
			doy : 6  // The week that contains Jan 6th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var monthsNominative = 'styczeń_luty_marzec_kwiecień_maj_czerwiec_lipiec_sierpień_wrzesień_październik_listopad_grudzień'.split('_'),
		monthsSubjective = 'stycznia_lutego_marca_kwietnia_maja_czerwca_lipca_sierpnia_września_października_listopada_grudnia'.split('_');
	function plural$3(n) {
		return (n % 10 < 5) && (n % 10 > 1) && ((~~(n / 10) % 10) !== 1);
	}
	function translate$8(number, withoutSuffix, key) {
		var result = number + ' ';
		switch (key) {
			case 'ss':
				return result + (plural$3(number) ? 'sekundy' : 'sekund');
			case 'm':
				return withoutSuffix ? 'minuta' : 'minutę';
			case 'mm':
				return result + (plural$3(number) ? 'minuty' : 'minut');
			case 'h':
				return withoutSuffix  ? 'godzina'  : 'godzinę';
			case 'hh':
				return result + (plural$3(number) ? 'godziny' : 'godzin');
			case 'MM':
				return result + (plural$3(number) ? 'miesiące' : 'miesięcy');
			case 'yy':
				return result + (plural$3(number) ? 'lata' : 'lat');
		}
	}

	hooks.defineLocale('pl', {
		months : function (momentToFormat, format) {
			if (!momentToFormat) {
				return monthsNominative;
			} else if (format === '') {
				// Hack: if format empty we know this is used to generate
				// RegExp by moment. Give then back both valid forms of months
				// in RegExp ready format.
				return '(' + monthsSubjective[momentToFormat.month()] + '|' + monthsNominative[momentToFormat.month()] + ')';
			} else if (/D MMMM/.test(format)) {
				return monthsSubjective[momentToFormat.month()];
			} else {
				return monthsNominative[momentToFormat.month()];
			}
		},
		monthsShort : 'sty_lut_mar_kwi_maj_cze_lip_sie_wrz_paź_lis_gru'.split('_'),
		weekdays : 'niedziela_poniedziałek_wtorek_środa_czwartek_piątek_sobota'.split('_'),
		weekdaysShort : 'ndz_pon_wt_śr_czw_pt_sob'.split('_'),
		weekdaysMin : 'Nd_Pn_Wt_Śr_Cz_Pt_So'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd, D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay: '[Dziś o] LT',
			nextDay: '[Jutro o] LT',
			nextWeek: function () {
				switch (this.day()) {
					case 0:
						return '[W niedzielę o] LT';

					case 2:
						return '[We wtorek o] LT';

					case 3:
						return '[W środę o] LT';

					case 6:
						return '[W sobotę o] LT';

					default:
						return '[W] dddd [o] LT';
				}
			},
			lastDay: '[Wczoraj o] LT',
			lastWeek: function () {
				switch (this.day()) {
					case 0:
						return '[W zeszłą niedzielę o] LT';
					case 3:
						return '[W zeszłą środę o] LT';
					case 6:
						return '[W zeszłą sobotę o] LT';
					default:
						return '[W zeszły] dddd [o] LT';
				}
			},
			sameElse: 'L'
		},
		relativeTime : {
			future : 'za %s',
			past : '%s temu',
			s : 'kilka sekund',
			ss : translate$8,
			m : translate$8,
			mm : translate$8,
			h : translate$8,
			hh : translate$8,
			d : '1 dzień',
			dd : '%d dni',
			M : 'miesiąc',
			MM : translate$8,
			y : 'rok',
			yy : translate$8
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('pt-br', {
		months : 'Janeiro_Fevereiro_Março_Abril_Maio_Junho_Julho_Agosto_Setembro_Outubro_Novembro_Dezembro'.split('_'),
		monthsShort : 'Jan_Fev_Mar_Abr_Mai_Jun_Jul_Ago_Set_Out_Nov_Dez'.split('_'),
		weekdays : 'Domingo_Segunda-feira_Terça-feira_Quarta-feira_Quinta-feira_Sexta-feira_Sábado'.split('_'),
		weekdaysShort : 'Dom_Seg_Ter_Qua_Qui_Sex_Sáb'.split('_'),
		weekdaysMin : 'Do_2ª_3ª_4ª_5ª_6ª_Sá'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D [de] MMMM [de] YYYY',
			LLL : 'D [de] MMMM [de] YYYY [às] HH:mm',
			LLLL : 'dddd, D [de] MMMM [de] YYYY [às] HH:mm'
		},
		calendar : {
			sameDay: '[Hoje às] LT',
			nextDay: '[Amanhã às] LT',
			nextWeek: 'dddd [às] LT',
			lastDay: '[Ontem às] LT',
			lastWeek: function () {
				return (this.day() === 0 || this.day() === 6) ?
					'[Último] dddd [às] LT' : // Saturday + Sunday
					'[Última] dddd [às] LT'; // Monday - Friday
			},
			sameElse: 'L'
		},
		relativeTime : {
			future : 'em %s',
			past : 'há %s',
			s : 'poucos segundos',
			ss : '%d segundos',
			m : 'um minuto',
			mm : '%d minutos',
			h : 'uma hora',
			hh : '%d horas',
			d : 'um dia',
			dd : '%d dias',
			M : 'um mês',
			MM : '%d meses',
			y : 'um ano',
			yy : '%d anos'
		},
		dayOfMonthOrdinalParse: /\d{1,2}º/,
		ordinal : '%dº'
	});

	//! moment.js locale configuration

	hooks.defineLocale('pt', {
		months : 'Janeiro_Fevereiro_Março_Abril_Maio_Junho_Julho_Agosto_Setembro_Outubro_Novembro_Dezembro'.split('_'),
		monthsShort : 'Jan_Fev_Mar_Abr_Mai_Jun_Jul_Ago_Set_Out_Nov_Dez'.split('_'),
		weekdays : 'Domingo_Segunda-feira_Terça-feira_Quarta-feira_Quinta-feira_Sexta-feira_Sábado'.split('_'),
		weekdaysShort : 'Dom_Seg_Ter_Qua_Qui_Sex_Sáb'.split('_'),
		weekdaysMin : 'Do_2ª_3ª_4ª_5ª_6ª_Sá'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D [de] MMMM [de] YYYY',
			LLL : 'D [de] MMMM [de] YYYY HH:mm',
			LLLL : 'dddd, D [de] MMMM [de] YYYY HH:mm'
		},
		calendar : {
			sameDay: '[Hoje às] LT',
			nextDay: '[Amanhã às] LT',
			nextWeek: 'dddd [às] LT',
			lastDay: '[Ontem às] LT',
			lastWeek: function () {
				return (this.day() === 0 || this.day() === 6) ?
					'[Último] dddd [às] LT' : // Saturday + Sunday
					'[Última] dddd [às] LT'; // Monday - Friday
			},
			sameElse: 'L'
		},
		relativeTime : {
			future : 'em %s',
			past : 'há %s',
			s : 'segundos',
			ss : '%d segundos',
			m : 'um minuto',
			mm : '%d minutos',
			h : 'uma hora',
			hh : '%d horas',
			d : 'um dia',
			dd : '%d dias',
			M : 'um mês',
			MM : '%d meses',
			y : 'um ano',
			yy : '%d anos'
		},
		dayOfMonthOrdinalParse: /\d{1,2}º/,
		ordinal : '%dº',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	function relativeTimeWithPlural$2(number, withoutSuffix, key) {
		var format = {
				'ss': 'secunde',
				'mm': 'minute',
				'hh': 'ore',
				'dd': 'zile',
				'MM': 'luni',
				'yy': 'ani'
			},
			separator = ' ';
		if (number % 100 >= 20 || (number >= 100 && number % 100 === 0)) {
			separator = ' de ';
		}
		return number + separator + format[key];
	}

	hooks.defineLocale('ro', {
		months : 'ianuarie_februarie_martie_aprilie_mai_iunie_iulie_august_septembrie_octombrie_noiembrie_decembrie'.split('_'),
		monthsShort : 'ian._febr._mart._apr._mai_iun._iul._aug._sept._oct._nov._dec.'.split('_'),
		monthsParseExact: true,
		weekdays : 'duminică_luni_marți_miercuri_joi_vineri_sâmbătă'.split('_'),
		weekdaysShort : 'Dum_Lun_Mar_Mie_Joi_Vin_Sâm'.split('_'),
		weekdaysMin : 'Du_Lu_Ma_Mi_Jo_Vi_Sâ'.split('_'),
		longDateFormat : {
			LT : 'H:mm',
			LTS : 'H:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY H:mm',
			LLLL : 'dddd, D MMMM YYYY H:mm'
		},
		calendar : {
			sameDay: '[azi la] LT',
			nextDay: '[mâine la] LT',
			nextWeek: 'dddd [la] LT',
			lastDay: '[ieri la] LT',
			lastWeek: '[fosta] dddd [la] LT',
			sameElse: 'L'
		},
		relativeTime : {
			future : 'peste %s',
			past : '%s în urmă',
			s : 'câteva secunde',
			ss : relativeTimeWithPlural$2,
			m : 'un minut',
			mm : relativeTimeWithPlural$2,
			h : 'o oră',
			hh : relativeTimeWithPlural$2,
			d : 'o zi',
			dd : relativeTimeWithPlural$2,
			M : 'o lună',
			MM : relativeTimeWithPlural$2,
			y : 'un an',
			yy : relativeTimeWithPlural$2
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	function plural$4(word, num) {
		var forms = word.split('_');
		return num % 10 === 1 && num % 100 !== 11 ? forms[0] : (num % 10 >= 2 && num % 10 <= 4 && (num % 100 < 10 || num % 100 >= 20) ? forms[1] : forms[2]);
	}
	function relativeTimeWithPlural$3(number, withoutSuffix, key) {
		var format = {
			'ss': withoutSuffix ? 'секунда_секунды_секунд' : 'секунду_секунды_секунд',
			'mm': withoutSuffix ? 'минута_минуты_минут' : 'минуту_минуты_минут',
			'hh': 'час_часа_часов',
			'dd': 'день_дня_дней',
			'MM': 'месяц_месяца_месяцев',
			'yy': 'год_года_лет'
		};
		if (key === 'm') {
			return withoutSuffix ? 'минута' : 'минуту';
		}
		else {
			return number + ' ' + plural$4(format[key], +number);
		}
	}
	var monthsParse$6 = [/^янв/i, /^фев/i, /^мар/i, /^апр/i, /^ма[йя]/i, /^июн/i, /^июл/i, /^авг/i, /^сен/i, /^окт/i, /^ноя/i, /^дек/i];

	// http://new.gramota.ru/spravka/rules/139-prop : § 103
	// Сокращения месяцев: http://new.gramota.ru/spravka/buro/search-answer?s=242637
	// CLDR data:          http://www.unicode.org/cldr/charts/28/summary/ru.html#1753
	hooks.defineLocale('ru', {
		months : {
			format: 'января_февраля_марта_апреля_мая_июня_июля_августа_сентября_октября_ноября_декабря'.split('_'),
			standalone: 'январь_февраль_март_апрель_май_июнь_июль_август_сентябрь_октябрь_ноябрь_декабрь'.split('_')
		},
		monthsShort : {
			// по CLDR именно "июл." и "июн.", но какой смысл менять букву на точку ?
			format: 'янв._февр._мар._апр._мая_июня_июля_авг._сент._окт._нояб._дек.'.split('_'),
			standalone: 'янв._февр._март_апр._май_июнь_июль_авг._сент._окт._нояб._дек.'.split('_')
		},
		weekdays : {
			standalone: 'воскресенье_понедельник_вторник_среда_четверг_пятница_суббота'.split('_'),
			format: 'воскресенье_понедельник_вторник_среду_четверг_пятницу_субботу'.split('_'),
			isFormat: /\[ ?[Вв] ?(?:прошлую|следующую|эту)? ?\] ?dddd/
		},
		weekdaysShort : 'вс_пн_вт_ср_чт_пт_сб'.split('_'),
		weekdaysMin : 'вс_пн_вт_ср_чт_пт_сб'.split('_'),
		monthsParse : monthsParse$6,
		longMonthsParse : monthsParse$6,
		shortMonthsParse : monthsParse$6,

		// полные названия с падежами, по три буквы, для некоторых, по 4 буквы, сокращения с точкой и без точки
		monthsRegex: /^(январ[ья]|янв\.?|феврал[ья]|февр?\.?|марта?|мар\.?|апрел[ья]|апр\.?|ма[йя]|июн[ья]|июн\.?|июл[ья]|июл\.?|августа?|авг\.?|сентябр[ья]|сент?\.?|октябр[ья]|окт\.?|ноябр[ья]|нояб?\.?|декабр[ья]|дек\.?)/i,

		// копия предыдущего
		monthsShortRegex: /^(январ[ья]|янв\.?|феврал[ья]|февр?\.?|марта?|мар\.?|апрел[ья]|апр\.?|ма[йя]|июн[ья]|июн\.?|июл[ья]|июл\.?|августа?|авг\.?|сентябр[ья]|сент?\.?|октябр[ья]|окт\.?|ноябр[ья]|нояб?\.?|декабр[ья]|дек\.?)/i,

		// полные названия с падежами
		monthsStrictRegex: /^(январ[яь]|феврал[яь]|марта?|апрел[яь]|ма[яй]|июн[яь]|июл[яь]|августа?|сентябр[яь]|октябр[яь]|ноябр[яь]|декабр[яь])/i,

		// Выражение, которое соотвествует только сокращённым формам
		monthsShortStrictRegex: /^(янв\.|февр?\.|мар[т.]|апр\.|ма[яй]|июн[ья.]|июл[ья.]|авг\.|сент?\.|окт\.|нояб?\.|дек\.)/i,
		longDateFormat : {
			LT : 'H:mm',
			LTS : 'H:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D MMMM YYYY г.',
			LLL : 'D MMMM YYYY г., H:mm',
			LLLL : 'dddd, D MMMM YYYY г., H:mm'
		},
		calendar : {
			sameDay: '[Сегодня, в] LT',
			nextDay: '[Завтра, в] LT',
			lastDay: '[Вчера, в] LT',
			nextWeek: function (now) {
				if (now.week() !== this.week()) {
					switch (this.day()) {
						case 0:
							return '[В следующее] dddd, [в] LT';
						case 1:
						case 2:
						case 4:
							return '[В следующий] dddd, [в] LT';
						case 3:
						case 5:
						case 6:
							return '[В следующую] dddd, [в] LT';
					}
				} else {
					if (this.day() === 2) {
						return '[Во] dddd, [в] LT';
					} else {
						return '[В] dddd, [в] LT';
					}
				}
			},
			lastWeek: function (now) {
				if (now.week() !== this.week()) {
					switch (this.day()) {
						case 0:
							return '[В прошлое] dddd, [в] LT';
						case 1:
						case 2:
						case 4:
							return '[В прошлый] dddd, [в] LT';
						case 3:
						case 5:
						case 6:
							return '[В прошлую] dddd, [в] LT';
					}
				} else {
					if (this.day() === 2) {
						return '[Во] dddd, [в] LT';
					} else {
						return '[В] dddd, [в] LT';
					}
				}
			},
			sameElse: 'L'
		},
		relativeTime : {
			future : 'через %s',
			past : '%s назад',
			s : 'несколько секунд',
			ss : relativeTimeWithPlural$3,
			m : relativeTimeWithPlural$3,
			mm : relativeTimeWithPlural$3,
			h : 'час',
			hh : relativeTimeWithPlural$3,
			d : 'день',
			dd : relativeTimeWithPlural$3,
			M : 'месяц',
			MM : relativeTimeWithPlural$3,
			y : 'год',
			yy : relativeTimeWithPlural$3
		},
		meridiemParse: /ночи|утра|дня|вечера/i,
		isPM : function (input) {
			return /^(дня|вечера)$/.test(input);
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 4) {
				return 'ночи';
			} else if (hour < 12) {
				return 'утра';
			} else if (hour < 17) {
				return 'дня';
			} else {
				return 'вечера';
			}
		},
		dayOfMonthOrdinalParse: /\d{1,2}-(й|го|я)/,
		ordinal: function (number, period) {
			switch (period) {
				case 'M':
				case 'd':
				case 'DDD':
					return number + '-й';
				case 'D':
					return number + '-го';
				case 'w':
				case 'W':
					return number + '-я';
				default:
					return number;
			}
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var months$8 = [
		'جنوري',
		'فيبروري',
		'مارچ',
		'اپريل',
		'مئي',
		'جون',
		'جولاءِ',
		'آگسٽ',
		'سيپٽمبر',
		'آڪٽوبر',
		'نومبر',
		'ڊسمبر'
	];
	var days$1 = [
		'آچر',
		'سومر',
		'اڱارو',
		'اربع',
		'خميس',
		'جمع',
		'ڇنڇر'
	];

	hooks.defineLocale('sd', {
		months : months$8,
		monthsShort : months$8,
		weekdays : days$1,
		weekdaysShort : days$1,
		weekdaysMin : days$1,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd، D MMMM YYYY HH:mm'
		},
		meridiemParse: /صبح|شام/,
		isPM : function (input) {
			return 'شام' === input;
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 12) {
				return 'صبح';
			}
			return 'شام';
		},
		calendar : {
			sameDay : '[اڄ] LT',
			nextDay : '[سڀاڻي] LT',
			nextWeek : 'dddd [اڳين هفتي تي] LT',
			lastDay : '[ڪالهه] LT',
			lastWeek : '[گزريل هفتي] dddd [تي] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : '%s پوء',
			past : '%s اڳ',
			s : 'چند سيڪنڊ',
			ss : '%d سيڪنڊ',
			m : 'هڪ منٽ',
			mm : '%d منٽ',
			h : 'هڪ ڪلاڪ',
			hh : '%d ڪلاڪ',
			d : 'هڪ ڏينهن',
			dd : '%d ڏينهن',
			M : 'هڪ مهينو',
			MM : '%d مهينا',
			y : 'هڪ سال',
			yy : '%d سال'
		},
		preparse: function (string) {
			return string.replace(/،/g, ',');
		},
		postformat: function (string) {
			return string.replace(/,/g, '،');
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('se', {
		months : 'ođđajagemánnu_guovvamánnu_njukčamánnu_cuoŋománnu_miessemánnu_geassemánnu_suoidnemánnu_borgemánnu_čakčamánnu_golggotmánnu_skábmamánnu_juovlamánnu'.split('_'),
		monthsShort : 'ođđj_guov_njuk_cuo_mies_geas_suoi_borg_čakč_golg_skáb_juov'.split('_'),
		weekdays : 'sotnabeaivi_vuossárga_maŋŋebárga_gaskavahkku_duorastat_bearjadat_lávvardat'.split('_'),
		weekdaysShort : 'sotn_vuos_maŋ_gask_duor_bear_láv'.split('_'),
		weekdaysMin : 's_v_m_g_d_b_L'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'MMMM D. [b.] YYYY',
			LLL : 'MMMM D. [b.] YYYY [ti.] HH:mm',
			LLLL : 'dddd, MMMM D. [b.] YYYY [ti.] HH:mm'
		},
		calendar : {
			sameDay: '[otne ti] LT',
			nextDay: '[ihttin ti] LT',
			nextWeek: 'dddd [ti] LT',
			lastDay: '[ikte ti] LT',
			lastWeek: '[ovddit] dddd [ti] LT',
			sameElse: 'L'
		},
		relativeTime : {
			future : '%s geažes',
			past : 'maŋit %s',
			s : 'moadde sekunddat',
			ss: '%d sekunddat',
			m : 'okta minuhta',
			mm : '%d minuhtat',
			h : 'okta diimmu',
			hh : '%d diimmut',
			d : 'okta beaivi',
			dd : '%d beaivvit',
			M : 'okta mánnu',
			MM : '%d mánut',
			y : 'okta jahki',
			yy : '%d jagit'
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	/*jshint -W100*/
	hooks.defineLocale('si', {
		months : 'ජනවාරි_පෙබරවාරි_මාර්තු_අප්‍රේල්_මැයි_ජූනි_ජූලි_අගෝස්තු_සැප්තැම්බර්_ඔක්තෝබර්_නොවැම්බර්_දෙසැම්බර්'.split('_'),
		monthsShort : 'ජන_පෙබ_මාර්_අප්_මැයි_ජූනි_ජූලි_අගෝ_සැප්_ඔක්_නොවැ_දෙසැ'.split('_'),
		weekdays : 'ඉරිදා_සඳුදා_අඟහරුවාදා_බදාදා_බ්‍රහස්පතින්දා_සිකුරාදා_සෙනසුරාදා'.split('_'),
		weekdaysShort : 'ඉරි_සඳු_අඟ_බදා_බ්‍රහ_සිකු_සෙන'.split('_'),
		weekdaysMin : 'ඉ_ස_අ_බ_බ්‍ර_සි_සෙ'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'a h:mm',
			LTS : 'a h:mm:ss',
			L : 'YYYY/MM/DD',
			LL : 'YYYY MMMM D',
			LLL : 'YYYY MMMM D, a h:mm',
			LLLL : 'YYYY MMMM D [වැනි] dddd, a h:mm:ss'
		},
		calendar : {
			sameDay : '[අද] LT[ට]',
			nextDay : '[හෙට] LT[ට]',
			nextWeek : 'dddd LT[ට]',
			lastDay : '[ඊයේ] LT[ට]',
			lastWeek : '[පසුගිය] dddd LT[ට]',
			sameElse : 'L'
		},
		relativeTime : {
			future : '%sකින්',
			past : '%sකට පෙර',
			s : 'තත්පර කිහිපය',
			ss : 'තත්පර %d',
			m : 'මිනිත්තුව',
			mm : 'මිනිත්තු %d',
			h : 'පැය',
			hh : 'පැය %d',
			d : 'දිනය',
			dd : 'දින %d',
			M : 'මාසය',
			MM : 'මාස %d',
			y : 'වසර',
			yy : 'වසර %d'
		},
		dayOfMonthOrdinalParse: /\d{1,2} වැනි/,
		ordinal : function (number) {
			return number + ' වැනි';
		},
		meridiemParse : /පෙර වරු|පස් වරු|පෙ.ව|ප.ව./,
		isPM : function (input) {
			return input === 'ප.ව.' || input === 'පස් වරු';
		},
		meridiem : function (hours, minutes, isLower) {
			if (hours > 11) {
				return isLower ? 'ප.ව.' : 'පස් වරු';
			} else {
				return isLower ? 'පෙ.ව.' : 'පෙර වරු';
			}
		}
	});

	//! moment.js locale configuration

	var months$9 = 'január_február_marec_apríl_máj_jún_júl_august_september_október_november_december'.split('_'),
		monthsShort$6 = 'jan_feb_mar_apr_máj_jún_júl_aug_sep_okt_nov_dec'.split('_');
	function plural$5(n) {
		return (n > 1) && (n < 5);
	}
	function translate$9(number, withoutSuffix, key, isFuture) {
		var result = number + ' ';
		switch (key) {
			case 's':  // a few seconds / in a few seconds / a few seconds ago
				return (withoutSuffix || isFuture) ? 'pár sekúnd' : 'pár sekundami';
			case 'ss': // 9 seconds / in 9 seconds / 9 seconds ago
				if (withoutSuffix || isFuture) {
					return result + (plural$5(number) ? 'sekundy' : 'sekúnd');
				} else {
					return result + 'sekundami';
				}
				break;
			case 'm':  // a minute / in a minute / a minute ago
				return withoutSuffix ? 'minúta' : (isFuture ? 'minútu' : 'minútou');
			case 'mm': // 9 minutes / in 9 minutes / 9 minutes ago
				if (withoutSuffix || isFuture) {
					return result + (plural$5(number) ? 'minúty' : 'minút');
				} else {
					return result + 'minútami';
				}
				break;
			case 'h':  // an hour / in an hour / an hour ago
				return withoutSuffix ? 'hodina' : (isFuture ? 'hodinu' : 'hodinou');
			case 'hh': // 9 hours / in 9 hours / 9 hours ago
				if (withoutSuffix || isFuture) {
					return result + (plural$5(number) ? 'hodiny' : 'hodín');
				} else {
					return result + 'hodinami';
				}
				break;
			case 'd':  // a day / in a day / a day ago
				return (withoutSuffix || isFuture) ? 'deň' : 'dňom';
			case 'dd': // 9 days / in 9 days / 9 days ago
				if (withoutSuffix || isFuture) {
					return result + (plural$5(number) ? 'dni' : 'dní');
				} else {
					return result + 'dňami';
				}
				break;
			case 'M':  // a month / in a month / a month ago
				return (withoutSuffix || isFuture) ? 'mesiac' : 'mesiacom';
			case 'MM': // 9 months / in 9 months / 9 months ago
				if (withoutSuffix || isFuture) {
					return result + (plural$5(number) ? 'mesiace' : 'mesiacov');
				} else {
					return result + 'mesiacmi';
				}
				break;
			case 'y':  // a year / in a year / a year ago
				return (withoutSuffix || isFuture) ? 'rok' : 'rokom';
			case 'yy': // 9 years / in 9 years / 9 years ago
				if (withoutSuffix || isFuture) {
					return result + (plural$5(number) ? 'roky' : 'rokov');
				} else {
					return result + 'rokmi';
				}
				break;
		}
	}

	hooks.defineLocale('sk', {
		months : months$9,
		monthsShort : monthsShort$6,
		weekdays : 'nedeľa_pondelok_utorok_streda_štvrtok_piatok_sobota'.split('_'),
		weekdaysShort : 'ne_po_ut_st_št_pi_so'.split('_'),
		weekdaysMin : 'ne_po_ut_st_št_pi_so'.split('_'),
		longDateFormat : {
			LT: 'H:mm',
			LTS : 'H:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D. MMMM YYYY',
			LLL : 'D. MMMM YYYY H:mm',
			LLLL : 'dddd D. MMMM YYYY H:mm'
		},
		calendar : {
			sameDay: '[dnes o] LT',
			nextDay: '[zajtra o] LT',
			nextWeek: function () {
				switch (this.day()) {
					case 0:
						return '[v nedeľu o] LT';
					case 1:
					case 2:
						return '[v] dddd [o] LT';
					case 3:
						return '[v stredu o] LT';
					case 4:
						return '[vo štvrtok o] LT';
					case 5:
						return '[v piatok o] LT';
					case 6:
						return '[v sobotu o] LT';
				}
			},
			lastDay: '[včera o] LT',
			lastWeek: function () {
				switch (this.day()) {
					case 0:
						return '[minulú nedeľu o] LT';
					case 1:
					case 2:
						return '[minulý] dddd [o] LT';
					case 3:
						return '[minulú stredu o] LT';
					case 4:
					case 5:
						return '[minulý] dddd [o] LT';
					case 6:
						return '[minulú sobotu o] LT';
				}
			},
			sameElse: 'L'
		},
		relativeTime : {
			future : 'za %s',
			past : 'pred %s',
			s : translate$9,
			ss : translate$9,
			m : translate$9,
			mm : translate$9,
			h : translate$9,
			hh : translate$9,
			d : translate$9,
			dd : translate$9,
			M : translate$9,
			MM : translate$9,
			y : translate$9,
			yy : translate$9
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	function processRelativeTime$6(number, withoutSuffix, key, isFuture) {
		var result = number + ' ';
		switch (key) {
			case 's':
				return withoutSuffix || isFuture ? 'nekaj sekund' : 'nekaj sekundami';
			case 'ss':
				if (number === 1) {
					result += withoutSuffix ? 'sekundo' : 'sekundi';
				} else if (number === 2) {
					result += withoutSuffix || isFuture ? 'sekundi' : 'sekundah';
				} else if (number < 5) {
					result += withoutSuffix || isFuture ? 'sekunde' : 'sekundah';
				} else {
					result += 'sekund';
				}
				return result;
			case 'm':
				return withoutSuffix ? 'ena minuta' : 'eno minuto';
			case 'mm':
				if (number === 1) {
					result += withoutSuffix ? 'minuta' : 'minuto';
				} else if (number === 2) {
					result += withoutSuffix || isFuture ? 'minuti' : 'minutama';
				} else if (number < 5) {
					result += withoutSuffix || isFuture ? 'minute' : 'minutami';
				} else {
					result += withoutSuffix || isFuture ? 'minut' : 'minutami';
				}
				return result;
			case 'h':
				return withoutSuffix ? 'ena ura' : 'eno uro';
			case 'hh':
				if (number === 1) {
					result += withoutSuffix ? 'ura' : 'uro';
				} else if (number === 2) {
					result += withoutSuffix || isFuture ? 'uri' : 'urama';
				} else if (number < 5) {
					result += withoutSuffix || isFuture ? 'ure' : 'urami';
				} else {
					result += withoutSuffix || isFuture ? 'ur' : 'urami';
				}
				return result;
			case 'd':
				return withoutSuffix || isFuture ? 'en dan' : 'enim dnem';
			case 'dd':
				if (number === 1) {
					result += withoutSuffix || isFuture ? 'dan' : 'dnem';
				} else if (number === 2) {
					result += withoutSuffix || isFuture ? 'dni' : 'dnevoma';
				} else {
					result += withoutSuffix || isFuture ? 'dni' : 'dnevi';
				}
				return result;
			case 'M':
				return withoutSuffix || isFuture ? 'en mesec' : 'enim mesecem';
			case 'MM':
				if (number === 1) {
					result += withoutSuffix || isFuture ? 'mesec' : 'mesecem';
				} else if (number === 2) {
					result += withoutSuffix || isFuture ? 'meseca' : 'mesecema';
				} else if (number < 5) {
					result += withoutSuffix || isFuture ? 'mesece' : 'meseci';
				} else {
					result += withoutSuffix || isFuture ? 'mesecev' : 'meseci';
				}
				return result;
			case 'y':
				return withoutSuffix || isFuture ? 'eno leto' : 'enim letom';
			case 'yy':
				if (number === 1) {
					result += withoutSuffix || isFuture ? 'leto' : 'letom';
				} else if (number === 2) {
					result += withoutSuffix || isFuture ? 'leti' : 'letoma';
				} else if (number < 5) {
					result += withoutSuffix || isFuture ? 'leta' : 'leti';
				} else {
					result += withoutSuffix || isFuture ? 'let' : 'leti';
				}
				return result;
		}
	}

	hooks.defineLocale('sl', {
		months : 'januar_februar_marec_april_maj_junij_julij_avgust_september_oktober_november_december'.split('_'),
		monthsShort : 'jan._feb._mar._apr._maj._jun._jul._avg._sep._okt._nov._dec.'.split('_'),
		monthsParseExact: true,
		weekdays : 'nedelja_ponedeljek_torek_sreda_četrtek_petek_sobota'.split('_'),
		weekdaysShort : 'ned._pon._tor._sre._čet._pet._sob.'.split('_'),
		weekdaysMin : 'ne_po_to_sr_če_pe_so'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'H:mm',
			LTS : 'H:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D. MMMM YYYY',
			LLL : 'D. MMMM YYYY H:mm',
			LLLL : 'dddd, D. MMMM YYYY H:mm'
		},
		calendar : {
			sameDay  : '[danes ob] LT',
			nextDay  : '[jutri ob] LT',

			nextWeek : function () {
				switch (this.day()) {
					case 0:
						return '[v] [nedeljo] [ob] LT';
					case 3:
						return '[v] [sredo] [ob] LT';
					case 6:
						return '[v] [soboto] [ob] LT';
					case 1:
					case 2:
					case 4:
					case 5:
						return '[v] dddd [ob] LT';
				}
			},
			lastDay  : '[včeraj ob] LT',
			lastWeek : function () {
				switch (this.day()) {
					case 0:
						return '[prejšnjo] [nedeljo] [ob] LT';
					case 3:
						return '[prejšnjo] [sredo] [ob] LT';
					case 6:
						return '[prejšnjo] [soboto] [ob] LT';
					case 1:
					case 2:
					case 4:
					case 5:
						return '[prejšnji] dddd [ob] LT';
				}
			},
			sameElse : 'L'
		},
		relativeTime : {
			future : 'čez %s',
			past   : 'pred %s',
			s      : processRelativeTime$6,
			ss     : processRelativeTime$6,
			m      : processRelativeTime$6,
			mm     : processRelativeTime$6,
			h      : processRelativeTime$6,
			hh     : processRelativeTime$6,
			d      : processRelativeTime$6,
			dd     : processRelativeTime$6,
			M      : processRelativeTime$6,
			MM     : processRelativeTime$6,
			y      : processRelativeTime$6,
			yy     : processRelativeTime$6
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('sq', {
		months : 'Janar_Shkurt_Mars_Prill_Maj_Qershor_Korrik_Gusht_Shtator_Tetor_Nëntor_Dhjetor'.split('_'),
		monthsShort : 'Jan_Shk_Mar_Pri_Maj_Qer_Kor_Gus_Sht_Tet_Nën_Dhj'.split('_'),
		weekdays : 'E Diel_E Hënë_E Martë_E Mërkurë_E Enjte_E Premte_E Shtunë'.split('_'),
		weekdaysShort : 'Die_Hën_Mar_Mër_Enj_Pre_Sht'.split('_'),
		weekdaysMin : 'D_H_Ma_Më_E_P_Sh'.split('_'),
		weekdaysParseExact : true,
		meridiemParse: /PD|MD/,
		isPM: function (input) {
			return input.charAt(0) === 'M';
		},
		meridiem : function (hours, minutes, isLower) {
			return hours < 12 ? 'PD' : 'MD';
		},
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd, D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay : '[Sot në] LT',
			nextDay : '[Nesër në] LT',
			nextWeek : 'dddd [në] LT',
			lastDay : '[Dje në] LT',
			lastWeek : 'dddd [e kaluar në] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'në %s',
			past : '%s më parë',
			s : 'disa sekonda',
			ss : '%d sekonda',
			m : 'një minutë',
			mm : '%d minuta',
			h : 'një orë',
			hh : '%d orë',
			d : 'një ditë',
			dd : '%d ditë',
			M : 'një muaj',
			MM : '%d muaj',
			y : 'një vit',
			yy : '%d vite'
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var translator$1 = {
		words: { //Different grammatical cases
			ss: ['секунда', 'секунде', 'секунди'],
			m: ['један минут', 'једне минуте'],
			mm: ['минут', 'минуте', 'минута'],
			h: ['један сат', 'једног сата'],
			hh: ['сат', 'сата', 'сати'],
			dd: ['дан', 'дана', 'дана'],
			MM: ['месец', 'месеца', 'месеци'],
			yy: ['година', 'године', 'година']
		},
		correctGrammaticalCase: function (number, wordKey) {
			return number === 1 ? wordKey[0] : (number >= 2 && number <= 4 ? wordKey[1] : wordKey[2]);
		},
		translate: function (number, withoutSuffix, key) {
			var wordKey = translator$1.words[key];
			if (key.length === 1) {
				return withoutSuffix ? wordKey[0] : wordKey[1];
			} else {
				return number + ' ' + translator$1.correctGrammaticalCase(number, wordKey);
			}
		}
	};

	hooks.defineLocale('sr-cyrl', {
		months: 'јануар_фебруар_март_април_мај_јун_јул_август_септембар_октобар_новембар_децембар'.split('_'),
		monthsShort: 'јан._феб._мар._апр._мај_јун_јул_авг._сеп._окт._нов._дец.'.split('_'),
		monthsParseExact: true,
		weekdays: 'недеља_понедељак_уторак_среда_четвртак_петак_субота'.split('_'),
		weekdaysShort: 'нед._пон._уто._сре._чет._пет._суб.'.split('_'),
		weekdaysMin: 'не_по_ут_ср_че_пе_су'.split('_'),
		weekdaysParseExact : true,
		longDateFormat: {
			LT: 'H:mm',
			LTS : 'H:mm:ss',
			L: 'DD.MM.YYYY',
			LL: 'D. MMMM YYYY',
			LLL: 'D. MMMM YYYY H:mm',
			LLLL: 'dddd, D. MMMM YYYY H:mm'
		},
		calendar: {
			sameDay: '[данас у] LT',
			nextDay: '[сутра у] LT',
			nextWeek: function () {
				switch (this.day()) {
					case 0:
						return '[у] [недељу] [у] LT';
					case 3:
						return '[у] [среду] [у] LT';
					case 6:
						return '[у] [суботу] [у] LT';
					case 1:
					case 2:
					case 4:
					case 5:
						return '[у] dddd [у] LT';
				}
			},
			lastDay  : '[јуче у] LT',
			lastWeek : function () {
				var lastWeekDays = [
					'[прошле] [недеље] [у] LT',
					'[прошлог] [понедељка] [у] LT',
					'[прошлог] [уторка] [у] LT',
					'[прошле] [среде] [у] LT',
					'[прошлог] [четвртка] [у] LT',
					'[прошлог] [петка] [у] LT',
					'[прошле] [суботе] [у] LT'
				];
				return lastWeekDays[this.day()];
			},
			sameElse : 'L'
		},
		relativeTime : {
			future : 'за %s',
			past   : 'пре %s',
			s      : 'неколико секунди',
			ss     : translator$1.translate,
			m      : translator$1.translate,
			mm     : translator$1.translate,
			h      : translator$1.translate,
			hh     : translator$1.translate,
			d      : 'дан',
			dd     : translator$1.translate,
			M      : 'месец',
			MM     : translator$1.translate,
			y      : 'годину',
			yy     : translator$1.translate
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var translator$2 = {
		words: { //Different grammatical cases
			ss: ['sekunda', 'sekunde', 'sekundi'],
			m: ['jedan minut', 'jedne minute'],
			mm: ['minut', 'minute', 'minuta'],
			h: ['jedan sat', 'jednog sata'],
			hh: ['sat', 'sata', 'sati'],
			dd: ['dan', 'dana', 'dana'],
			MM: ['mesec', 'meseca', 'meseci'],
			yy: ['godina', 'godine', 'godina']
		},
		correctGrammaticalCase: function (number, wordKey) {
			return number === 1 ? wordKey[0] : (number >= 2 && number <= 4 ? wordKey[1] : wordKey[2]);
		},
		translate: function (number, withoutSuffix, key) {
			var wordKey = translator$2.words[key];
			if (key.length === 1) {
				return withoutSuffix ? wordKey[0] : wordKey[1];
			} else {
				return number + ' ' + translator$2.correctGrammaticalCase(number, wordKey);
			}
		}
	};

	hooks.defineLocale('sr', {
		months: 'januar_februar_mart_april_maj_jun_jul_avgust_septembar_oktobar_novembar_decembar'.split('_'),
		monthsShort: 'jan._feb._mar._apr._maj_jun_jul_avg._sep._okt._nov._dec.'.split('_'),
		monthsParseExact: true,
		weekdays: 'nedelja_ponedeljak_utorak_sreda_četvrtak_petak_subota'.split('_'),
		weekdaysShort: 'ned._pon._uto._sre._čet._pet._sub.'.split('_'),
		weekdaysMin: 'ne_po_ut_sr_če_pe_su'.split('_'),
		weekdaysParseExact : true,
		longDateFormat: {
			LT: 'H:mm',
			LTS : 'H:mm:ss',
			L: 'DD.MM.YYYY',
			LL: 'D. MMMM YYYY',
			LLL: 'D. MMMM YYYY H:mm',
			LLLL: 'dddd, D. MMMM YYYY H:mm'
		},
		calendar: {
			sameDay: '[danas u] LT',
			nextDay: '[sutra u] LT',
			nextWeek: function () {
				switch (this.day()) {
					case 0:
						return '[u] [nedelju] [u] LT';
					case 3:
						return '[u] [sredu] [u] LT';
					case 6:
						return '[u] [subotu] [u] LT';
					case 1:
					case 2:
					case 4:
					case 5:
						return '[u] dddd [u] LT';
				}
			},
			lastDay  : '[juče u] LT',
			lastWeek : function () {
				var lastWeekDays = [
					'[prošle] [nedelje] [u] LT',
					'[prošlog] [ponedeljka] [u] LT',
					'[prošlog] [utorka] [u] LT',
					'[prošle] [srede] [u] LT',
					'[prošlog] [četvrtka] [u] LT',
					'[prošlog] [petka] [u] LT',
					'[prošle] [subote] [u] LT'
				];
				return lastWeekDays[this.day()];
			},
			sameElse : 'L'
		},
		relativeTime : {
			future : 'za %s',
			past   : 'pre %s',
			s      : 'nekoliko sekundi',
			ss     : translator$2.translate,
			m      : translator$2.translate,
			mm     : translator$2.translate,
			h      : translator$2.translate,
			hh     : translator$2.translate,
			d      : 'dan',
			dd     : translator$2.translate,
			M      : 'mesec',
			MM     : translator$2.translate,
			y      : 'godinu',
			yy     : translator$2.translate
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('ss', {
		months : "Bhimbidvwane_Indlovana_Indlov'lenkhulu_Mabasa_Inkhwekhweti_Inhlaba_Kholwane_Ingci_Inyoni_Imphala_Lweti_Ingongoni".split('_'),
		monthsShort : 'Bhi_Ina_Inu_Mab_Ink_Inh_Kho_Igc_Iny_Imp_Lwe_Igo'.split('_'),
		weekdays : 'Lisontfo_Umsombuluko_Lesibili_Lesitsatfu_Lesine_Lesihlanu_Umgcibelo'.split('_'),
		weekdaysShort : 'Lis_Umb_Lsb_Les_Lsi_Lsh_Umg'.split('_'),
		weekdaysMin : 'Li_Us_Lb_Lt_Ls_Lh_Ug'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'h:mm A',
			LTS : 'h:mm:ss A',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY h:mm A',
			LLLL : 'dddd, D MMMM YYYY h:mm A'
		},
		calendar : {
			sameDay : '[Namuhla nga] LT',
			nextDay : '[Kusasa nga] LT',
			nextWeek : 'dddd [nga] LT',
			lastDay : '[Itolo nga] LT',
			lastWeek : 'dddd [leliphelile] [nga] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'nga %s',
			past : 'wenteka nga %s',
			s : 'emizuzwana lomcane',
			ss : '%d mzuzwana',
			m : 'umzuzu',
			mm : '%d emizuzu',
			h : 'lihora',
			hh : '%d emahora',
			d : 'lilanga',
			dd : '%d emalanga',
			M : 'inyanga',
			MM : '%d tinyanga',
			y : 'umnyaka',
			yy : '%d iminyaka'
		},
		meridiemParse: /ekuseni|emini|entsambama|ebusuku/,
		meridiem : function (hours, minutes, isLower) {
			if (hours < 11) {
				return 'ekuseni';
			} else if (hours < 15) {
				return 'emini';
			} else if (hours < 19) {
				return 'entsambama';
			} else {
				return 'ebusuku';
			}
		},
		meridiemHour : function (hour, meridiem) {
			if (hour === 12) {
				hour = 0;
			}
			if (meridiem === 'ekuseni') {
				return hour;
			} else if (meridiem === 'emini') {
				return hour >= 11 ? hour : hour + 12;
			} else if (meridiem === 'entsambama' || meridiem === 'ebusuku') {
				if (hour === 0) {
					return 0;
				}
				return hour + 12;
			}
		},
		dayOfMonthOrdinalParse: /\d{1,2}/,
		ordinal : '%d',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('sv', {
		months : 'januari_februari_mars_april_maj_juni_juli_augusti_september_oktober_november_december'.split('_'),
		monthsShort : 'jan_feb_mar_apr_maj_jun_jul_aug_sep_okt_nov_dec'.split('_'),
		weekdays : 'söndag_måndag_tisdag_onsdag_torsdag_fredag_lördag'.split('_'),
		weekdaysShort : 'sön_mån_tis_ons_tor_fre_lör'.split('_'),
		weekdaysMin : 'sö_må_ti_on_to_fr_lö'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'YYYY-MM-DD',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY [kl.] HH:mm',
			LLLL : 'dddd D MMMM YYYY [kl.] HH:mm',
			lll : 'D MMM YYYY HH:mm',
			llll : 'ddd D MMM YYYY HH:mm'
		},
		calendar : {
			sameDay: '[Idag] LT',
			nextDay: '[Imorgon] LT',
			lastDay: '[Igår] LT',
			nextWeek: '[På] dddd LT',
			lastWeek: '[I] dddd[s] LT',
			sameElse: 'L'
		},
		relativeTime : {
			future : 'om %s',
			past : 'för %s sedan',
			s : 'några sekunder',
			ss : '%d sekunder',
			m : 'en minut',
			mm : '%d minuter',
			h : 'en timme',
			hh : '%d timmar',
			d : 'en dag',
			dd : '%d dagar',
			M : 'en månad',
			MM : '%d månader',
			y : 'ett år',
			yy : '%d år'
		},
		dayOfMonthOrdinalParse: /\d{1,2}(e|a)/,
		ordinal : function (number) {
			var b = number % 10,
				output = (~~(number % 100 / 10) === 1) ? 'e' :
					(b === 1) ? 'a' :
						(b === 2) ? 'a' :
							(b === 3) ? 'e' : 'e';
			return number + output;
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('sw', {
		months : 'Januari_Februari_Machi_Aprili_Mei_Juni_Julai_Agosti_Septemba_Oktoba_Novemba_Desemba'.split('_'),
		monthsShort : 'Jan_Feb_Mac_Apr_Mei_Jun_Jul_Ago_Sep_Okt_Nov_Des'.split('_'),
		weekdays : 'Jumapili_Jumatatu_Jumanne_Jumatano_Alhamisi_Ijumaa_Jumamosi'.split('_'),
		weekdaysShort : 'Jpl_Jtat_Jnne_Jtan_Alh_Ijm_Jmos'.split('_'),
		weekdaysMin : 'J2_J3_J4_J5_Al_Ij_J1'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd, D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay : '[leo saa] LT',
			nextDay : '[kesho saa] LT',
			nextWeek : '[wiki ijayo] dddd [saat] LT',
			lastDay : '[jana] LT',
			lastWeek : '[wiki iliyopita] dddd [saat] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : '%s baadaye',
			past : 'tokea %s',
			s : 'hivi punde',
			ss : 'sekunde %d',
			m : 'dakika moja',
			mm : 'dakika %d',
			h : 'saa limoja',
			hh : 'masaa %d',
			d : 'siku moja',
			dd : 'masiku %d',
			M : 'mwezi mmoja',
			MM : 'miezi %d',
			y : 'mwaka mmoja',
			yy : 'miaka %d'
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var symbolMap$f = {
		'1': '௧',
		'2': '௨',
		'3': '௩',
		'4': '௪',
		'5': '௫',
		'6': '௬',
		'7': '௭',
		'8': '௮',
		'9': '௯',
		'0': '௦'
	}, numberMap$e = {
		'௧': '1',
		'௨': '2',
		'௩': '3',
		'௪': '4',
		'௫': '5',
		'௬': '6',
		'௭': '7',
		'௮': '8',
		'௯': '9',
		'௦': '0'
	};

	hooks.defineLocale('ta', {
		months : 'ஜனவரி_பிப்ரவரி_மார்ச்_ஏப்ரல்_மே_ஜூன்_ஜூலை_ஆகஸ்ட்_செப்டெம்பர்_அக்டோபர்_நவம்பர்_டிசம்பர்'.split('_'),
		monthsShort : 'ஜனவரி_பிப்ரவரி_மார்ச்_ஏப்ரல்_மே_ஜூன்_ஜூலை_ஆகஸ்ட்_செப்டெம்பர்_அக்டோபர்_நவம்பர்_டிசம்பர்'.split('_'),
		weekdays : 'ஞாயிற்றுக்கிழமை_திங்கட்கிழமை_செவ்வாய்கிழமை_புதன்கிழமை_வியாழக்கிழமை_வெள்ளிக்கிழமை_சனிக்கிழமை'.split('_'),
		weekdaysShort : 'ஞாயிறு_திங்கள்_செவ்வாய்_புதன்_வியாழன்_வெள்ளி_சனி'.split('_'),
		weekdaysMin : 'ஞா_தி_செ_பு_வி_வெ_ச'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY, HH:mm',
			LLLL : 'dddd, D MMMM YYYY, HH:mm'
		},
		calendar : {
			sameDay : '[இன்று] LT',
			nextDay : '[நாளை] LT',
			nextWeek : 'dddd, LT',
			lastDay : '[நேற்று] LT',
			lastWeek : '[கடந்த வாரம்] dddd, LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : '%s இல்',
			past : '%s முன்',
			s : 'ஒரு சில விநாடிகள்',
			ss : '%d விநாடிகள்',
			m : 'ஒரு நிமிடம்',
			mm : '%d நிமிடங்கள்',
			h : 'ஒரு மணி நேரம்',
			hh : '%d மணி நேரம்',
			d : 'ஒரு நாள்',
			dd : '%d நாட்கள்',
			M : 'ஒரு மாதம்',
			MM : '%d மாதங்கள்',
			y : 'ஒரு வருடம்',
			yy : '%d ஆண்டுகள்'
		},
		dayOfMonthOrdinalParse: /\d{1,2}வது/,
		ordinal : function (number) {
			return number + 'வது';
		},
		preparse: function (string) {
			return string.replace(/[௧௨௩௪௫௬௭௮௯௦]/g, function (match) {
				return numberMap$e[match];
			});
		},
		postformat: function (string) {
			return string.replace(/\d/g, function (match) {
				return symbolMap$f[match];
			});
		},
		// refer http://ta.wikipedia.org/s/1er1
		meridiemParse: /யாமம்|வைகறை|காலை|நண்பகல்|எற்பாடு|மாலை/,
		meridiem : function (hour, minute, isLower) {
			if (hour < 2) {
				return ' யாமம்';
			} else if (hour < 6) {
				return ' வைகறை';  // வைகறை
			} else if (hour < 10) {
				return ' காலை'; // காலை
			} else if (hour < 14) {
				return ' நண்பகல்'; // நண்பகல்
			} else if (hour < 18) {
				return ' எற்பாடு'; // எற்பாடு
			} else if (hour < 22) {
				return ' மாலை'; // மாலை
			} else {
				return ' யாமம்';
			}
		},
		meridiemHour : function (hour, meridiem) {
			if (hour === 12) {
				hour = 0;
			}
			if (meridiem === 'யாமம்') {
				return hour < 2 ? hour : hour + 12;
			} else if (meridiem === 'வைகறை' || meridiem === 'காலை') {
				return hour;
			} else if (meridiem === 'நண்பகல்') {
				return hour >= 10 ? hour : hour + 12;
			} else {
				return hour + 12;
			}
		},
		week : {
			dow : 0, // Sunday is the first day of the week.
			doy : 6  // The week that contains Jan 6th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('te', {
		months : 'జనవరి_ఫిబ్రవరి_మార్చి_ఏప్రిల్_మే_జూన్_జులై_ఆగస్టు_సెప్టెంబర్_అక్టోబర్_నవంబర్_డిసెంబర్'.split('_'),
		monthsShort : 'జన._ఫిబ్ర._మార్చి_ఏప్రి._మే_జూన్_జులై_ఆగ._సెప్._అక్టో._నవ._డిసె.'.split('_'),
		monthsParseExact : true,
		weekdays : 'ఆదివారం_సోమవారం_మంగళవారం_బుధవారం_గురువారం_శుక్రవారం_శనివారం'.split('_'),
		weekdaysShort : 'ఆది_సోమ_మంగళ_బుధ_గురు_శుక్ర_శని'.split('_'),
		weekdaysMin : 'ఆ_సో_మం_బు_గు_శు_శ'.split('_'),
		longDateFormat : {
			LT : 'A h:mm',
			LTS : 'A h:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY, A h:mm',
			LLLL : 'dddd, D MMMM YYYY, A h:mm'
		},
		calendar : {
			sameDay : '[నేడు] LT',
			nextDay : '[రేపు] LT',
			nextWeek : 'dddd, LT',
			lastDay : '[నిన్న] LT',
			lastWeek : '[గత] dddd, LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : '%s లో',
			past : '%s క్రితం',
			s : 'కొన్ని క్షణాలు',
			ss : '%d సెకన్లు',
			m : 'ఒక నిమిషం',
			mm : '%d నిమిషాలు',
			h : 'ఒక గంట',
			hh : '%d గంటలు',
			d : 'ఒక రోజు',
			dd : '%d రోజులు',
			M : 'ఒక నెల',
			MM : '%d నెలలు',
			y : 'ఒక సంవత్సరం',
			yy : '%d సంవత్సరాలు'
		},
		dayOfMonthOrdinalParse : /\d{1,2}వ/,
		ordinal : '%dవ',
		meridiemParse: /రాత్రి|ఉదయం|మధ్యాహ్నం|సాయంత్రం/,
		meridiemHour : function (hour, meridiem) {
			if (hour === 12) {
				hour = 0;
			}
			if (meridiem === 'రాత్రి') {
				return hour < 4 ? hour : hour + 12;
			} else if (meridiem === 'ఉదయం') {
				return hour;
			} else if (meridiem === 'మధ్యాహ్నం') {
				return hour >= 10 ? hour : hour + 12;
			} else if (meridiem === 'సాయంత్రం') {
				return hour + 12;
			}
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 4) {
				return 'రాత్రి';
			} else if (hour < 10) {
				return 'ఉదయం';
			} else if (hour < 17) {
				return 'మధ్యాహ్నం';
			} else if (hour < 20) {
				return 'సాయంత్రం';
			} else {
				return 'రాత్రి';
			}
		},
		week : {
			dow : 0, // Sunday is the first day of the week.
			doy : 6  // The week that contains Jan 6th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('tet', {
		months : 'Janeiru_Fevereiru_Marsu_Abril_Maiu_Juñu_Jullu_Agustu_Setembru_Outubru_Novembru_Dezembru'.split('_'),
		monthsShort : 'Jan_Fev_Mar_Abr_Mai_Jun_Jul_Ago_Set_Out_Nov_Dez'.split('_'),
		weekdays : 'Domingu_Segunda_Tersa_Kuarta_Kinta_Sesta_Sabadu'.split('_'),
		weekdaysShort : 'Dom_Seg_Ters_Kua_Kint_Sest_Sab'.split('_'),
		weekdaysMin : 'Do_Seg_Te_Ku_Ki_Ses_Sa'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd, D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay: '[Ohin iha] LT',
			nextDay: '[Aban iha] LT',
			nextWeek: 'dddd [iha] LT',
			lastDay: '[Horiseik iha] LT',
			lastWeek: 'dddd [semana kotuk] [iha] LT',
			sameElse: 'L'
		},
		relativeTime : {
			future : 'iha %s',
			past : '%s liuba',
			s : 'minutu balun',
			ss : 'minutu %d',
			m : 'minutu ida',
			mm : 'minutu %d',
			h : 'oras ida',
			hh : 'oras %d',
			d : 'loron ida',
			dd : 'loron %d',
			M : 'fulan ida',
			MM : 'fulan %d',
			y : 'tinan ida',
			yy : 'tinan %d'
		},
		dayOfMonthOrdinalParse: /\d{1,2}(st|nd|rd|th)/,
		ordinal : function (number) {
			var b = number % 10,
				output = (~~(number % 100 / 10) === 1) ? 'th' :
					(b === 1) ? 'st' :
						(b === 2) ? 'nd' :
							(b === 3) ? 'rd' : 'th';
			return number + output;
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var suffixes$3 = {
		0: '-ум',
		1: '-ум',
		2: '-юм',
		3: '-юм',
		4: '-ум',
		5: '-ум',
		6: '-ум',
		7: '-ум',
		8: '-ум',
		9: '-ум',
		10: '-ум',
		12: '-ум',
		13: '-ум',
		20: '-ум',
		30: '-юм',
		40: '-ум',
		50: '-ум',
		60: '-ум',
		70: '-ум',
		80: '-ум',
		90: '-ум',
		100: '-ум'
	};

	hooks.defineLocale('tg', {
		months : 'январ_феврал_март_апрел_май_июн_июл_август_сентябр_октябр_ноябр_декабр'.split('_'),
		monthsShort : 'янв_фев_мар_апр_май_июн_июл_авг_сен_окт_ноя_дек'.split('_'),
		weekdays : 'якшанбе_душанбе_сешанбе_чоршанбе_панҷшанбе_ҷумъа_шанбе'.split('_'),
		weekdaysShort : 'яшб_дшб_сшб_чшб_пшб_ҷум_шнб'.split('_'),
		weekdaysMin : 'яш_дш_сш_чш_пш_ҷм_шб'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd, D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay : '[Имрӯз соати] LT',
			nextDay : '[Пагоҳ соати] LT',
			lastDay : '[Дирӯз соати] LT',
			nextWeek : 'dddd[и] [ҳафтаи оянда соати] LT',
			lastWeek : 'dddd[и] [ҳафтаи гузашта соати] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'баъди %s',
			past : '%s пеш',
			s : 'якчанд сония',
			m : 'як дақиқа',
			mm : '%d дақиқа',
			h : 'як соат',
			hh : '%d соат',
			d : 'як рӯз',
			dd : '%d рӯз',
			M : 'як моҳ',
			MM : '%d моҳ',
			y : 'як сол',
			yy : '%d сол'
		},
		meridiemParse: /шаб|субҳ|рӯз|бегоҳ/,
		meridiemHour: function (hour, meridiem) {
			if (hour === 12) {
				hour = 0;
			}
			if (meridiem === 'шаб') {
				return hour < 4 ? hour : hour + 12;
			} else if (meridiem === 'субҳ') {
				return hour;
			} else if (meridiem === 'рӯз') {
				return hour >= 11 ? hour : hour + 12;
			} else if (meridiem === 'бегоҳ') {
				return hour + 12;
			}
		},
		meridiem: function (hour, minute, isLower) {
			if (hour < 4) {
				return 'шаб';
			} else if (hour < 11) {
				return 'субҳ';
			} else if (hour < 16) {
				return 'рӯз';
			} else if (hour < 19) {
				return 'бегоҳ';
			} else {
				return 'шаб';
			}
		},
		dayOfMonthOrdinalParse: /\d{1,2}-(ум|юм)/,
		ordinal: function (number) {
			var a = number % 10,
				b = number >= 100 ? 100 : null;
			return number + (suffixes$3[number] || suffixes$3[a] || suffixes$3[b]);
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 1th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('th', {
		months : 'มกราคม_กุมภาพันธ์_มีนาคม_เมษายน_พฤษภาคม_มิถุนายน_กรกฎาคม_สิงหาคม_กันยายน_ตุลาคม_พฤศจิกายน_ธันวาคม'.split('_'),
		monthsShort : 'ม.ค._ก.พ._มี.ค._เม.ย._พ.ค._มิ.ย._ก.ค._ส.ค._ก.ย._ต.ค._พ.ย._ธ.ค.'.split('_'),
		monthsParseExact: true,
		weekdays : 'อาทิตย์_จันทร์_อังคาร_พุธ_พฤหัสบดี_ศุกร์_เสาร์'.split('_'),
		weekdaysShort : 'อาทิตย์_จันทร์_อังคาร_พุธ_พฤหัส_ศุกร์_เสาร์'.split('_'), // yes, three characters difference
		weekdaysMin : 'อา._จ._อ._พ._พฤ._ศ._ส.'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'H:mm',
			LTS : 'H:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY เวลา H:mm',
			LLLL : 'วันddddที่ D MMMM YYYY เวลา H:mm'
		},
		meridiemParse: /ก่อนเที่ยง|หลังเที่ยง/,
		isPM: function (input) {
			return input === 'หลังเที่ยง';
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 12) {
				return 'ก่อนเที่ยง';
			} else {
				return 'หลังเที่ยง';
			}
		},
		calendar : {
			sameDay : '[วันนี้ เวลา] LT',
			nextDay : '[พรุ่งนี้ เวลา] LT',
			nextWeek : 'dddd[หน้า เวลา] LT',
			lastDay : '[เมื่อวานนี้ เวลา] LT',
			lastWeek : '[วัน]dddd[ที่แล้ว เวลา] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'อีก %s',
			past : '%sที่แล้ว',
			s : 'ไม่กี่วินาที',
			ss : '%d วินาที',
			m : '1 นาที',
			mm : '%d นาที',
			h : '1 ชั่วโมง',
			hh : '%d ชั่วโมง',
			d : '1 วัน',
			dd : '%d วัน',
			M : '1 เดือน',
			MM : '%d เดือน',
			y : '1 ปี',
			yy : '%d ปี'
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('tl-ph', {
		months : 'Enero_Pebrero_Marso_Abril_Mayo_Hunyo_Hulyo_Agosto_Setyembre_Oktubre_Nobyembre_Disyembre'.split('_'),
		monthsShort : 'Ene_Peb_Mar_Abr_May_Hun_Hul_Ago_Set_Okt_Nob_Dis'.split('_'),
		weekdays : 'Linggo_Lunes_Martes_Miyerkules_Huwebes_Biyernes_Sabado'.split('_'),
		weekdaysShort : 'Lin_Lun_Mar_Miy_Huw_Biy_Sab'.split('_'),
		weekdaysMin : 'Li_Lu_Ma_Mi_Hu_Bi_Sab'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'MM/D/YYYY',
			LL : 'MMMM D, YYYY',
			LLL : 'MMMM D, YYYY HH:mm',
			LLLL : 'dddd, MMMM DD, YYYY HH:mm'
		},
		calendar : {
			sameDay: 'LT [ngayong araw]',
			nextDay: '[Bukas ng] LT',
			nextWeek: 'LT [sa susunod na] dddd',
			lastDay: 'LT [kahapon]',
			lastWeek: 'LT [noong nakaraang] dddd',
			sameElse: 'L'
		},
		relativeTime : {
			future : 'sa loob ng %s',
			past : '%s ang nakalipas',
			s : 'ilang segundo',
			ss : '%d segundo',
			m : 'isang minuto',
			mm : '%d minuto',
			h : 'isang oras',
			hh : '%d oras',
			d : 'isang araw',
			dd : '%d araw',
			M : 'isang buwan',
			MM : '%d buwan',
			y : 'isang taon',
			yy : '%d taon'
		},
		dayOfMonthOrdinalParse: /\d{1,2}/,
		ordinal : function (number) {
			return number;
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var numbersNouns = 'pagh_wa’_cha’_wej_loS_vagh_jav_Soch_chorgh_Hut'.split('_');

	function translateFuture(output) {
		var time = output;
		time = (output.indexOf('jaj') !== -1) ?
			time.slice(0, -3) + 'leS' :
			(output.indexOf('jar') !== -1) ?
				time.slice(0, -3) + 'waQ' :
				(output.indexOf('DIS') !== -1) ?
					time.slice(0, -3) + 'nem' :
					time + ' pIq';
		return time;
	}

	function translatePast(output) {
		var time = output;
		time = (output.indexOf('jaj') !== -1) ?
			time.slice(0, -3) + 'Hu’' :
			(output.indexOf('jar') !== -1) ?
				time.slice(0, -3) + 'wen' :
				(output.indexOf('DIS') !== -1) ?
					time.slice(0, -3) + 'ben' :
					time + ' ret';
		return time;
	}

	function translate$a(number, withoutSuffix, string, isFuture) {
		var numberNoun = numberAsNoun(number);
		switch (string) {
			case 'ss':
				return numberNoun + ' lup';
			case 'mm':
				return numberNoun + ' tup';
			case 'hh':
				return numberNoun + ' rep';
			case 'dd':
				return numberNoun + ' jaj';
			case 'MM':
				return numberNoun + ' jar';
			case 'yy':
				return numberNoun + ' DIS';
		}
	}

	function numberAsNoun(number) {
		var hundred = Math.floor((number % 1000) / 100),
			ten = Math.floor((number % 100) / 10),
			one = number % 10,
			word = '';
		if (hundred > 0) {
			word += numbersNouns[hundred] + 'vatlh';
		}
		if (ten > 0) {
			word += ((word !== '') ? ' ' : '') + numbersNouns[ten] + 'maH';
		}
		if (one > 0) {
			word += ((word !== '') ? ' ' : '') + numbersNouns[one];
		}
		return (word === '') ? 'pagh' : word;
	}

	hooks.defineLocale('tlh', {
		months : 'tera’ jar wa’_tera’ jar cha’_tera’ jar wej_tera’ jar loS_tera’ jar vagh_tera’ jar jav_tera’ jar Soch_tera’ jar chorgh_tera’ jar Hut_tera’ jar wa’maH_tera’ jar wa’maH wa’_tera’ jar wa’maH cha’'.split('_'),
		monthsShort : 'jar wa’_jar cha’_jar wej_jar loS_jar vagh_jar jav_jar Soch_jar chorgh_jar Hut_jar wa’maH_jar wa’maH wa’_jar wa’maH cha’'.split('_'),
		monthsParseExact : true,
		weekdays : 'lojmItjaj_DaSjaj_povjaj_ghItlhjaj_loghjaj_buqjaj_ghInjaj'.split('_'),
		weekdaysShort : 'lojmItjaj_DaSjaj_povjaj_ghItlhjaj_loghjaj_buqjaj_ghInjaj'.split('_'),
		weekdaysMin : 'lojmItjaj_DaSjaj_povjaj_ghItlhjaj_loghjaj_buqjaj_ghInjaj'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd, D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay: '[DaHjaj] LT',
			nextDay: '[wa’leS] LT',
			nextWeek: 'LLL',
			lastDay: '[wa’Hu’] LT',
			lastWeek: 'LLL',
			sameElse: 'L'
		},
		relativeTime : {
			future : translateFuture,
			past : translatePast,
			s : 'puS lup',
			ss : translate$a,
			m : 'wa’ tup',
			mm : translate$a,
			h : 'wa’ rep',
			hh : translate$a,
			d : 'wa’ jaj',
			dd : translate$a,
			M : 'wa’ jar',
			MM : translate$a,
			y : 'wa’ DIS',
			yy : translate$a
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	var suffixes$4 = {
		1: '\'inci',
		5: '\'inci',
		8: '\'inci',
		70: '\'inci',
		80: '\'inci',
		2: '\'nci',
		7: '\'nci',
		20: '\'nci',
		50: '\'nci',
		3: '\'üncü',
		4: '\'üncü',
		100: '\'üncü',
		6: '\'ncı',
		9: '\'uncu',
		10: '\'uncu',
		30: '\'uncu',
		60: '\'ıncı',
		90: '\'ıncı'
	};

	hooks.defineLocale('tr', {
		months : 'Ocak_Şubat_Mart_Nisan_Mayıs_Haziran_Temmuz_Ağustos_Eylül_Ekim_Kasım_Aralık'.split('_'),
		monthsShort : 'Oca_Şub_Mar_Nis_May_Haz_Tem_Ağu_Eyl_Eki_Kas_Ara'.split('_'),
		weekdays : 'Pazar_Pazartesi_Salı_Çarşamba_Perşembe_Cuma_Cumartesi'.split('_'),
		weekdaysShort : 'Paz_Pts_Sal_Çar_Per_Cum_Cts'.split('_'),
		weekdaysMin : 'Pz_Pt_Sa_Ça_Pe_Cu_Ct'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd, D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay : '[bugün saat] LT',
			nextDay : '[yarın saat] LT',
			nextWeek : '[gelecek] dddd [saat] LT',
			lastDay : '[dün] LT',
			lastWeek : '[geçen] dddd [saat] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : '%s sonra',
			past : '%s önce',
			s : 'birkaç saniye',
			ss : '%d saniye',
			m : 'bir dakika',
			mm : '%d dakika',
			h : 'bir saat',
			hh : '%d saat',
			d : 'bir gün',
			dd : '%d gün',
			M : 'bir ay',
			MM : '%d ay',
			y : 'bir yıl',
			yy : '%d yıl'
		},
		ordinal: function (number, period) {
			switch (period) {
				case 'd':
				case 'D':
				case 'Do':
				case 'DD':
					return number;
				default:
					if (number === 0) {  // special case for zero
						return number + '\'ıncı';
					}
					var a = number % 10,
						b = number % 100 - a,
						c = number >= 100 ? 100 : null;
					return number + (suffixes$4[a] || suffixes$4[b] || suffixes$4[c]);
			}
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	// After the year there should be a slash and the amount of years since December 26, 1979 in Roman numerals.
	// This is currently too difficult (maybe even impossible) to add.
	hooks.defineLocale('tzl', {
		months : 'Januar_Fevraglh_Març_Avrïu_Mai_Gün_Julia_Guscht_Setemvar_Listopäts_Noemvar_Zecemvar'.split('_'),
		monthsShort : 'Jan_Fev_Mar_Avr_Mai_Gün_Jul_Gus_Set_Lis_Noe_Zec'.split('_'),
		weekdays : 'Súladi_Lúneçi_Maitzi_Márcuri_Xhúadi_Viénerçi_Sáturi'.split('_'),
		weekdaysShort : 'Súl_Lún_Mai_Már_Xhú_Vié_Sát'.split('_'),
		weekdaysMin : 'Sú_Lú_Ma_Má_Xh_Vi_Sá'.split('_'),
		longDateFormat : {
			LT : 'HH.mm',
			LTS : 'HH.mm.ss',
			L : 'DD.MM.YYYY',
			LL : 'D. MMMM [dallas] YYYY',
			LLL : 'D. MMMM [dallas] YYYY HH.mm',
			LLLL : 'dddd, [li] D. MMMM [dallas] YYYY HH.mm'
		},
		meridiemParse: /d\'o|d\'a/i,
		isPM : function (input) {
			return 'd\'o' === input.toLowerCase();
		},
		meridiem : function (hours, minutes, isLower) {
			if (hours > 11) {
				return isLower ? 'd\'o' : 'D\'O';
			} else {
				return isLower ? 'd\'a' : 'D\'A';
			}
		},
		calendar : {
			sameDay : '[oxhi à] LT',
			nextDay : '[demà à] LT',
			nextWeek : 'dddd [à] LT',
			lastDay : '[ieiri à] LT',
			lastWeek : '[sür el] dddd [lasteu à] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'osprei %s',
			past : 'ja%s',
			s : processRelativeTime$7,
			ss : processRelativeTime$7,
			m : processRelativeTime$7,
			mm : processRelativeTime$7,
			h : processRelativeTime$7,
			hh : processRelativeTime$7,
			d : processRelativeTime$7,
			dd : processRelativeTime$7,
			M : processRelativeTime$7,
			MM : processRelativeTime$7,
			y : processRelativeTime$7,
			yy : processRelativeTime$7
		},
		dayOfMonthOrdinalParse: /\d{1,2}\./,
		ordinal : '%d.',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	function processRelativeTime$7(number, withoutSuffix, key, isFuture) {
		var format = {
			's': ['viensas secunds', '\'iensas secunds'],
			'ss': [number + ' secunds', '' + number + ' secunds'],
			'm': ['\'n míut', '\'iens míut'],
			'mm': [number + ' míuts', '' + number + ' míuts'],
			'h': ['\'n þora', '\'iensa þora'],
			'hh': [number + ' þoras', '' + number + ' þoras'],
			'd': ['\'n ziua', '\'iensa ziua'],
			'dd': [number + ' ziuas', '' + number + ' ziuas'],
			'M': ['\'n mes', '\'iens mes'],
			'MM': [number + ' mesen', '' + number + ' mesen'],
			'y': ['\'n ar', '\'iens ar'],
			'yy': [number + ' ars', '' + number + ' ars']
		};
		return isFuture ? format[key][0] : (withoutSuffix ? format[key][0] : format[key][1]);
	}

	//! moment.js locale configuration

	hooks.defineLocale('tzm-latn', {
		months : 'innayr_brˤayrˤ_marˤsˤ_ibrir_mayyw_ywnyw_ywlywz_ɣwšt_šwtanbir_ktˤwbrˤ_nwwanbir_dwjnbir'.split('_'),
		monthsShort : 'innayr_brˤayrˤ_marˤsˤ_ibrir_mayyw_ywnyw_ywlywz_ɣwšt_šwtanbir_ktˤwbrˤ_nwwanbir_dwjnbir'.split('_'),
		weekdays : 'asamas_aynas_asinas_akras_akwas_asimwas_asiḍyas'.split('_'),
		weekdaysShort : 'asamas_aynas_asinas_akras_akwas_asimwas_asiḍyas'.split('_'),
		weekdaysMin : 'asamas_aynas_asinas_akras_akwas_asimwas_asiḍyas'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay: '[asdkh g] LT',
			nextDay: '[aska g] LT',
			nextWeek: 'dddd [g] LT',
			lastDay: '[assant g] LT',
			lastWeek: 'dddd [g] LT',
			sameElse: 'L'
		},
		relativeTime : {
			future : 'dadkh s yan %s',
			past : 'yan %s',
			s : 'imik',
			ss : '%d imik',
			m : 'minuḍ',
			mm : '%d minuḍ',
			h : 'saɛa',
			hh : '%d tassaɛin',
			d : 'ass',
			dd : '%d ossan',
			M : 'ayowr',
			MM : '%d iyyirn',
			y : 'asgas',
			yy : '%d isgasn'
		},
		week : {
			dow : 6, // Saturday is the first day of the week.
			doy : 12  // The week that contains Jan 12th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('tzm', {
		months : 'ⵉⵏⵏⴰⵢⵔ_ⴱⵕⴰⵢⵕ_ⵎⴰⵕⵚ_ⵉⴱⵔⵉⵔ_ⵎⴰⵢⵢⵓ_ⵢⵓⵏⵢⵓ_ⵢⵓⵍⵢⵓⵣ_ⵖⵓⵛⵜ_ⵛⵓⵜⴰⵏⴱⵉⵔ_ⴽⵟⵓⴱⵕ_ⵏⵓⵡⴰⵏⴱⵉⵔ_ⴷⵓⵊⵏⴱⵉⵔ'.split('_'),
		monthsShort : 'ⵉⵏⵏⴰⵢⵔ_ⴱⵕⴰⵢⵕ_ⵎⴰⵕⵚ_ⵉⴱⵔⵉⵔ_ⵎⴰⵢⵢⵓ_ⵢⵓⵏⵢⵓ_ⵢⵓⵍⵢⵓⵣ_ⵖⵓⵛⵜ_ⵛⵓⵜⴰⵏⴱⵉⵔ_ⴽⵟⵓⴱⵕ_ⵏⵓⵡⴰⵏⴱⵉⵔ_ⴷⵓⵊⵏⴱⵉⵔ'.split('_'),
		weekdays : 'ⴰⵙⴰⵎⴰⵙ_ⴰⵢⵏⴰⵙ_ⴰⵙⵉⵏⴰⵙ_ⴰⴽⵔⴰⵙ_ⴰⴽⵡⴰⵙ_ⴰⵙⵉⵎⵡⴰⵙ_ⴰⵙⵉⴹⵢⴰⵙ'.split('_'),
		weekdaysShort : 'ⴰⵙⴰⵎⴰⵙ_ⴰⵢⵏⴰⵙ_ⴰⵙⵉⵏⴰⵙ_ⴰⴽⵔⴰⵙ_ⴰⴽⵡⴰⵙ_ⴰⵙⵉⵎⵡⴰⵙ_ⴰⵙⵉⴹⵢⴰⵙ'.split('_'),
		weekdaysMin : 'ⴰⵙⴰⵎⴰⵙ_ⴰⵢⵏⴰⵙ_ⴰⵙⵉⵏⴰⵙ_ⴰⴽⵔⴰⵙ_ⴰⴽⵡⴰⵙ_ⴰⵙⵉⵎⵡⴰⵙ_ⴰⵙⵉⴹⵢⴰⵙ'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS: 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay: '[ⴰⵙⴷⵅ ⴴ] LT',
			nextDay: '[ⴰⵙⴽⴰ ⴴ] LT',
			nextWeek: 'dddd [ⴴ] LT',
			lastDay: '[ⴰⵚⴰⵏⵜ ⴴ] LT',
			lastWeek: 'dddd [ⴴ] LT',
			sameElse: 'L'
		},
		relativeTime : {
			future : 'ⴷⴰⴷⵅ ⵙ ⵢⴰⵏ %s',
			past : 'ⵢⴰⵏ %s',
			s : 'ⵉⵎⵉⴽ',
			ss : '%d ⵉⵎⵉⴽ',
			m : 'ⵎⵉⵏⵓⴺ',
			mm : '%d ⵎⵉⵏⵓⴺ',
			h : 'ⵙⴰⵄⴰ',
			hh : '%d ⵜⴰⵙⵙⴰⵄⵉⵏ',
			d : 'ⴰⵙⵙ',
			dd : '%d oⵙⵙⴰⵏ',
			M : 'ⴰⵢoⵓⵔ',
			MM : '%d ⵉⵢⵢⵉⵔⵏ',
			y : 'ⴰⵙⴳⴰⵙ',
			yy : '%d ⵉⵙⴳⴰⵙⵏ'
		},
		week : {
			dow : 6, // Saturday is the first day of the week.
			doy : 12  // The week that contains Jan 12th is the first week of the year.
		}
	});

	//! moment.js language configuration

	hooks.defineLocale('ug-cn', {
		months: 'يانۋار_فېۋرال_مارت_ئاپرېل_ماي_ئىيۇن_ئىيۇل_ئاۋغۇست_سېنتەبىر_ئۆكتەبىر_نويابىر_دېكابىر'.split(
			'_'
		),
		monthsShort: 'يانۋار_فېۋرال_مارت_ئاپرېل_ماي_ئىيۇن_ئىيۇل_ئاۋغۇست_سېنتەبىر_ئۆكتەبىر_نويابىر_دېكابىر'.split(
			'_'
		),
		weekdays: 'يەكشەنبە_دۈشەنبە_سەيشەنبە_چارشەنبە_پەيشەنبە_جۈمە_شەنبە'.split(
			'_'
		),
		weekdaysShort: 'يە_دۈ_سە_چا_پە_جۈ_شە'.split('_'),
		weekdaysMin: 'يە_دۈ_سە_چا_پە_جۈ_شە'.split('_'),
		longDateFormat: {
			LT: 'HH:mm',
			LTS: 'HH:mm:ss',
			L: 'YYYY-MM-DD',
			LL: 'YYYY-يىلىM-ئاينىڭD-كۈنى',
			LLL: 'YYYY-يىلىM-ئاينىڭD-كۈنى، HH:mm',
			LLLL: 'dddd، YYYY-يىلىM-ئاينىڭD-كۈنى، HH:mm'
		},
		meridiemParse: /يېرىم كېچە|سەھەر|چۈشتىن بۇرۇن|چۈش|چۈشتىن كېيىن|كەچ/,
		meridiemHour: function (hour, meridiem) {
			if (hour === 12) {
				hour = 0;
			}
			if (
				meridiem === 'يېرىم كېچە' ||
				meridiem === 'سەھەر' ||
				meridiem === 'چۈشتىن بۇرۇن'
			) {
				return hour;
			} else if (meridiem === 'چۈشتىن كېيىن' || meridiem === 'كەچ') {
				return hour + 12;
			} else {
				return hour >= 11 ? hour : hour + 12;
			}
		},
		meridiem: function (hour, minute, isLower) {
			var hm = hour * 100 + minute;
			if (hm < 600) {
				return 'يېرىم كېچە';
			} else if (hm < 900) {
				return 'سەھەر';
			} else if (hm < 1130) {
				return 'چۈشتىن بۇرۇن';
			} else if (hm < 1230) {
				return 'چۈش';
			} else if (hm < 1800) {
				return 'چۈشتىن كېيىن';
			} else {
				return 'كەچ';
			}
		},
		calendar: {
			sameDay: '[بۈگۈن سائەت] LT',
			nextDay: '[ئەتە سائەت] LT',
			nextWeek: '[كېلەركى] dddd [سائەت] LT',
			lastDay: '[تۆنۈگۈن] LT',
			lastWeek: '[ئالدىنقى] dddd [سائەت] LT',
			sameElse: 'L'
		},
		relativeTime: {
			future: '%s كېيىن',
			past: '%s بۇرۇن',
			s: 'نەچچە سېكونت',
			ss: '%d سېكونت',
			m: 'بىر مىنۇت',
			mm: '%d مىنۇت',
			h: 'بىر سائەت',
			hh: '%d سائەت',
			d: 'بىر كۈن',
			dd: '%d كۈن',
			M: 'بىر ئاي',
			MM: '%d ئاي',
			y: 'بىر يىل',
			yy: '%d يىل'
		},

		dayOfMonthOrdinalParse: /\d{1,2}(-كۈنى|-ئاي|-ھەپتە)/,
		ordinal: function (number, period) {
			switch (period) {
				case 'd':
				case 'D':
				case 'DDD':
					return number + '-كۈنى';
				case 'w':
				case 'W':
					return number + '-ھەپتە';
				default:
					return number;
			}
		},
		preparse: function (string) {
			return string.replace(/،/g, ',');
		},
		postformat: function (string) {
			return string.replace(/,/g, '،');
		},
		week: {
			// GB/T 7408-1994《数据元和交换格式·信息交换·日期和时间表示法》与ISO 8601:1988等效
			dow: 1, // Monday is the first day of the week.
			doy: 7 // The week that contains Jan 1st is the first week of the year.
		}
	});

	//! moment.js locale configuration

	function plural$6(word, num) {
		var forms = word.split('_');
		return num % 10 === 1 && num % 100 !== 11 ? forms[0] : (num % 10 >= 2 && num % 10 <= 4 && (num % 100 < 10 || num % 100 >= 20) ? forms[1] : forms[2]);
	}
	function relativeTimeWithPlural$4(number, withoutSuffix, key) {
		var format = {
			'ss': withoutSuffix ? 'секунда_секунди_секунд' : 'секунду_секунди_секунд',
			'mm': withoutSuffix ? 'хвилина_хвилини_хвилин' : 'хвилину_хвилини_хвилин',
			'hh': withoutSuffix ? 'година_години_годин' : 'годину_години_годин',
			'dd': 'день_дні_днів',
			'MM': 'місяць_місяці_місяців',
			'yy': 'рік_роки_років'
		};
		if (key === 'm') {
			return withoutSuffix ? 'хвилина' : 'хвилину';
		}
		else if (key === 'h') {
			return withoutSuffix ? 'година' : 'годину';
		}
		else {
			return number + ' ' + plural$6(format[key], +number);
		}
	}
	function weekdaysCaseReplace(m, format) {
		var weekdays = {
			'nominative': 'неділя_понеділок_вівторок_середа_четвер_п’ятниця_субота'.split('_'),
			'accusative': 'неділю_понеділок_вівторок_середу_четвер_п’ятницю_суботу'.split('_'),
			'genitive': 'неділі_понеділка_вівторка_середи_четверга_п’ятниці_суботи'.split('_')
		};

		if (m === true) {
			return weekdays['nominative'].slice(1, 7).concat(weekdays['nominative'].slice(0, 1));
		}
		if (!m) {
			return weekdays['nominative'];
		}

		var nounCase = (/(\[[ВвУу]\]) ?dddd/).test(format) ?
			'accusative' :
			((/\[?(?:минулої|наступної)? ?\] ?dddd/).test(format) ?
				'genitive' :
				'nominative');
		return weekdays[nounCase][m.day()];
	}
	function processHoursFunction(str) {
		return function () {
			return str + 'о' + (this.hours() === 11 ? 'б' : '') + '] LT';
		};
	}

	hooks.defineLocale('uk', {
		months : {
			'format': 'січня_лютого_березня_квітня_травня_червня_липня_серпня_вересня_жовтня_листопада_грудня'.split('_'),
			'standalone': 'січень_лютий_березень_квітень_травень_червень_липень_серпень_вересень_жовтень_листопад_грудень'.split('_')
		},
		monthsShort : 'січ_лют_бер_квіт_трав_черв_лип_серп_вер_жовт_лист_груд'.split('_'),
		weekdays : weekdaysCaseReplace,
		weekdaysShort : 'нд_пн_вт_ср_чт_пт_сб'.split('_'),
		weekdaysMin : 'нд_пн_вт_ср_чт_пт_сб'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD.MM.YYYY',
			LL : 'D MMMM YYYY р.',
			LLL : 'D MMMM YYYY р., HH:mm',
			LLLL : 'dddd, D MMMM YYYY р., HH:mm'
		},
		calendar : {
			sameDay: processHoursFunction('[Сьогодні '),
			nextDay: processHoursFunction('[Завтра '),
			lastDay: processHoursFunction('[Вчора '),
			nextWeek: processHoursFunction('[У] dddd ['),
			lastWeek: function () {
				switch (this.day()) {
					case 0:
					case 3:
					case 5:
					case 6:
						return processHoursFunction('[Минулої] dddd [').call(this);
					case 1:
					case 2:
					case 4:
						return processHoursFunction('[Минулого] dddd [').call(this);
				}
			},
			sameElse: 'L'
		},
		relativeTime : {
			future : 'за %s',
			past : '%s тому',
			s : 'декілька секунд',
			ss : relativeTimeWithPlural$4,
			m : relativeTimeWithPlural$4,
			mm : relativeTimeWithPlural$4,
			h : 'годину',
			hh : relativeTimeWithPlural$4,
			d : 'день',
			dd : relativeTimeWithPlural$4,
			M : 'місяць',
			MM : relativeTimeWithPlural$4,
			y : 'рік',
			yy : relativeTimeWithPlural$4
		},
		// M. E.: those two are virtually unused but a user might want to implement them for his/her website for some reason
		meridiemParse: /ночі|ранку|дня|вечора/,
		isPM: function (input) {
			return /^(дня|вечора)$/.test(input);
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 4) {
				return 'ночі';
			} else if (hour < 12) {
				return 'ранку';
			} else if (hour < 17) {
				return 'дня';
			} else {
				return 'вечора';
			}
		},
		dayOfMonthOrdinalParse: /\d{1,2}-(й|го)/,
		ordinal: function (number, period) {
			switch (period) {
				case 'M':
				case 'd':
				case 'DDD':
				case 'w':
				case 'W':
					return number + '-й';
				case 'D':
					return number + '-го';
				default:
					return number;
			}
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	var months$a = [
		'جنوری',
		'فروری',
		'مارچ',
		'اپریل',
		'مئی',
		'جون',
		'جولائی',
		'اگست',
		'ستمبر',
		'اکتوبر',
		'نومبر',
		'دسمبر'
	];
	var days$2 = [
		'اتوار',
		'پیر',
		'منگل',
		'بدھ',
		'جمعرات',
		'جمعہ',
		'ہفتہ'
	];

	hooks.defineLocale('ur', {
		months : months$a,
		monthsShort : months$a,
		weekdays : days$2,
		weekdaysShort : days$2,
		weekdaysMin : days$2,
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd، D MMMM YYYY HH:mm'
		},
		meridiemParse: /صبح|شام/,
		isPM : function (input) {
			return 'شام' === input;
		},
		meridiem : function (hour, minute, isLower) {
			if (hour < 12) {
				return 'صبح';
			}
			return 'شام';
		},
		calendar : {
			sameDay : '[آج بوقت] LT',
			nextDay : '[کل بوقت] LT',
			nextWeek : 'dddd [بوقت] LT',
			lastDay : '[گذشتہ روز بوقت] LT',
			lastWeek : '[گذشتہ] dddd [بوقت] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : '%s بعد',
			past : '%s قبل',
			s : 'چند سیکنڈ',
			ss : '%d سیکنڈ',
			m : 'ایک منٹ',
			mm : '%d منٹ',
			h : 'ایک گھنٹہ',
			hh : '%d گھنٹے',
			d : 'ایک دن',
			dd : '%d دن',
			M : 'ایک ماہ',
			MM : '%d ماہ',
			y : 'ایک سال',
			yy : '%d سال'
		},
		preparse: function (string) {
			return string.replace(/،/g, ',');
		},
		postformat: function (string) {
			return string.replace(/,/g, '،');
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('uz-latn', {
		months : 'Yanvar_Fevral_Mart_Aprel_May_Iyun_Iyul_Avgust_Sentabr_Oktabr_Noyabr_Dekabr'.split('_'),
		monthsShort : 'Yan_Fev_Mar_Apr_May_Iyun_Iyul_Avg_Sen_Okt_Noy_Dek'.split('_'),
		weekdays : 'Yakshanba_Dushanba_Seshanba_Chorshanba_Payshanba_Juma_Shanba'.split('_'),
		weekdaysShort : 'Yak_Dush_Sesh_Chor_Pay_Jum_Shan'.split('_'),
		weekdaysMin : 'Ya_Du_Se_Cho_Pa_Ju_Sha'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'D MMMM YYYY, dddd HH:mm'
		},
		calendar : {
			sameDay : '[Bugun soat] LT [da]',
			nextDay : '[Ertaga] LT [da]',
			nextWeek : 'dddd [kuni soat] LT [da]',
			lastDay : '[Kecha soat] LT [da]',
			lastWeek : '[O\'tgan] dddd [kuni soat] LT [da]',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'Yaqin %s ichida',
			past : 'Bir necha %s oldin',
			s : 'soniya',
			ss : '%d soniya',
			m : 'bir daqiqa',
			mm : '%d daqiqa',
			h : 'bir soat',
			hh : '%d soat',
			d : 'bir kun',
			dd : '%d kun',
			M : 'bir oy',
			MM : '%d oy',
			y : 'bir yil',
			yy : '%d yil'
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 7th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('uz', {
		months : 'январ_феврал_март_апрел_май_июн_июл_август_сентябр_октябр_ноябр_декабр'.split('_'),
		monthsShort : 'янв_фев_мар_апр_май_июн_июл_авг_сен_окт_ноя_дек'.split('_'),
		weekdays : 'Якшанба_Душанба_Сешанба_Чоршанба_Пайшанба_Жума_Шанба'.split('_'),
		weekdaysShort : 'Якш_Душ_Сеш_Чор_Пай_Жум_Шан'.split('_'),
		weekdaysMin : 'Як_Ду_Се_Чо_Па_Жу_Ша'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'D MMMM YYYY, dddd HH:mm'
		},
		calendar : {
			sameDay : '[Бугун соат] LT [да]',
			nextDay : '[Эртага] LT [да]',
			nextWeek : 'dddd [куни соат] LT [да]',
			lastDay : '[Кеча соат] LT [да]',
			lastWeek : '[Утган] dddd [куни соат] LT [да]',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'Якин %s ичида',
			past : 'Бир неча %s олдин',
			s : 'фурсат',
			ss : '%d фурсат',
			m : 'бир дакика',
			mm : '%d дакика',
			h : 'бир соат',
			hh : '%d соат',
			d : 'бир кун',
			dd : '%d кун',
			M : 'бир ой',
			MM : '%d ой',
			y : 'бир йил',
			yy : '%d йил'
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 7  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('vi', {
		months : 'tháng 1_tháng 2_tháng 3_tháng 4_tháng 5_tháng 6_tháng 7_tháng 8_tháng 9_tháng 10_tháng 11_tháng 12'.split('_'),
		monthsShort : 'Th01_Th02_Th03_Th04_Th05_Th06_Th07_Th08_Th09_Th10_Th11_Th12'.split('_'),
		monthsParseExact : true,
		weekdays : 'chủ nhật_thứ hai_thứ ba_thứ tư_thứ năm_thứ sáu_thứ bảy'.split('_'),
		weekdaysShort : 'CN_T2_T3_T4_T5_T6_T7'.split('_'),
		weekdaysMin : 'CN_T2_T3_T4_T5_T6_T7'.split('_'),
		weekdaysParseExact : true,
		meridiemParse: /sa|ch/i,
		isPM : function (input) {
			return /^ch$/i.test(input);
		},
		meridiem : function (hours, minutes, isLower) {
			if (hours < 12) {
				return isLower ? 'sa' : 'SA';
			} else {
				return isLower ? 'ch' : 'CH';
			}
		},
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM [năm] YYYY',
			LLL : 'D MMMM [năm] YYYY HH:mm',
			LLLL : 'dddd, D MMMM [năm] YYYY HH:mm',
			l : 'DD/M/YYYY',
			ll : 'D MMM YYYY',
			lll : 'D MMM YYYY HH:mm',
			llll : 'ddd, D MMM YYYY HH:mm'
		},
		calendar : {
			sameDay: '[Hôm nay lúc] LT',
			nextDay: '[Ngày mai lúc] LT',
			nextWeek: 'dddd [tuần tới lúc] LT',
			lastDay: '[Hôm qua lúc] LT',
			lastWeek: 'dddd [tuần rồi lúc] LT',
			sameElse: 'L'
		},
		relativeTime : {
			future : '%s tới',
			past : '%s trước',
			s : 'vài giây',
			ss : '%d giây' ,
			m : 'một phút',
			mm : '%d phút',
			h : 'một giờ',
			hh : '%d giờ',
			d : 'một ngày',
			dd : '%d ngày',
			M : 'một tháng',
			MM : '%d tháng',
			y : 'một năm',
			yy : '%d năm'
		},
		dayOfMonthOrdinalParse: /\d{1,2}/,
		ordinal : function (number) {
			return number;
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('x-pseudo', {
		months : 'J~áñúá~rý_F~ébrú~árý_~Márc~h_Áp~ríl_~Máý_~Júñé~_Júl~ý_Áú~gúst~_Sép~témb~ér_Ó~ctób~ér_Ñ~óvém~bér_~Décé~mbér'.split('_'),
		monthsShort : 'J~áñ_~Féb_~Már_~Ápr_~Máý_~Júñ_~Júl_~Áúg_~Sép_~Óct_~Ñóv_~Déc'.split('_'),
		monthsParseExact : true,
		weekdays : 'S~úñdá~ý_Mó~ñdáý~_Túé~sdáý~_Wéd~ñésd~áý_T~húrs~dáý_~Fríd~áý_S~átúr~dáý'.split('_'),
		weekdaysShort : 'S~úñ_~Móñ_~Túé_~Wéd_~Thú_~Frí_~Sát'.split('_'),
		weekdaysMin : 'S~ú_Mó~_Tú_~Wé_T~h_Fr~_Sá'.split('_'),
		weekdaysParseExact : true,
		longDateFormat : {
			LT : 'HH:mm',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY HH:mm',
			LLLL : 'dddd, D MMMM YYYY HH:mm'
		},
		calendar : {
			sameDay : '[T~ódá~ý át] LT',
			nextDay : '[T~ómó~rró~w át] LT',
			nextWeek : 'dddd [át] LT',
			lastDay : '[Ý~ést~érdá~ý át] LT',
			lastWeek : '[L~ást] dddd [át] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'í~ñ %s',
			past : '%s á~gó',
			s : 'á ~féw ~sécó~ñds',
			ss : '%d s~écóñ~ds',
			m : 'á ~míñ~úté',
			mm : '%d m~íñú~tés',
			h : 'á~ñ hó~úr',
			hh : '%d h~óúrs',
			d : 'á ~dáý',
			dd : '%d d~áýs',
			M : 'á ~móñ~th',
			MM : '%d m~óñt~hs',
			y : 'á ~ýéár',
			yy : '%d ý~éárs'
		},
		dayOfMonthOrdinalParse: /\d{1,2}(th|st|nd|rd)/,
		ordinal : function (number) {
			var b = number % 10,
				output = (~~(number % 100 / 10) === 1) ? 'th' :
					(b === 1) ? 'st' :
						(b === 2) ? 'nd' :
							(b === 3) ? 'rd' : 'th';
			return number + output;
		},
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('yo', {
		months : 'Sẹ́rẹ́_Èrèlè_Ẹrẹ̀nà_Ìgbé_Èbibi_Òkùdu_Agẹmo_Ògún_Owewe_Ọ̀wàrà_Bélú_Ọ̀pẹ̀̀'.split('_'),
		monthsShort : 'Sẹ́r_Èrl_Ẹrn_Ìgb_Èbi_Òkù_Agẹ_Ògú_Owe_Ọ̀wà_Bél_Ọ̀pẹ̀̀'.split('_'),
		weekdays : 'Àìkú_Ajé_Ìsẹ́gun_Ọjọ́rú_Ọjọ́bọ_Ẹtì_Àbámẹ́ta'.split('_'),
		weekdaysShort : 'Àìk_Ajé_Ìsẹ́_Ọjr_Ọjb_Ẹtì_Àbá'.split('_'),
		weekdaysMin : 'Àì_Aj_Ìs_Ọr_Ọb_Ẹt_Àb'.split('_'),
		longDateFormat : {
			LT : 'h:mm A',
			LTS : 'h:mm:ss A',
			L : 'DD/MM/YYYY',
			LL : 'D MMMM YYYY',
			LLL : 'D MMMM YYYY h:mm A',
			LLLL : 'dddd, D MMMM YYYY h:mm A'
		},
		calendar : {
			sameDay : '[Ònì ni] LT',
			nextDay : '[Ọ̀la ni] LT',
			nextWeek : 'dddd [Ọsẹ̀ tón\'bọ] [ni] LT',
			lastDay : '[Àna ni] LT',
			lastWeek : 'dddd [Ọsẹ̀ tólọ́] [ni] LT',
			sameElse : 'L'
		},
		relativeTime : {
			future : 'ní %s',
			past : '%s kọjá',
			s : 'ìsẹjú aayá die',
			ss :'aayá %d',
			m : 'ìsẹjú kan',
			mm : 'ìsẹjú %d',
			h : 'wákati kan',
			hh : 'wákati %d',
			d : 'ọjọ́ kan',
			dd : 'ọjọ́ %d',
			M : 'osù kan',
			MM : 'osù %d',
			y : 'ọdún kan',
			yy : 'ọdún %d'
		},
		dayOfMonthOrdinalParse : /ọjọ́\s\d{1,2}/,
		ordinal : 'ọjọ́ %d',
		week : {
			dow : 1, // Monday is the first day of the week.
			doy : 4 // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('zh-cn', {
		months : '一月_二月_三月_四月_五月_六月_七月_八月_九月_十月_十一月_十二月'.split('_'),
		monthsShort : '1月_2月_3月_4月_5月_6月_7月_8月_9月_10月_11月_12月'.split('_'),
		weekdays : '星期日_星期一_星期二_星期三_星期四_星期五_星期六'.split('_'),
		weekdaysShort : '周日_周一_周二_周三_周四_周五_周六'.split('_'),
		weekdaysMin : '日_一_二_三_四_五_六'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'YYYY/MM/DD',
			LL : 'YYYY年M月D日',
			LLL : 'YYYY年M月D日Ah点mm分',
			LLLL : 'YYYY年M月D日ddddAh点mm分',
			l : 'YYYY/M/D',
			ll : 'YYYY年M月D日',
			lll : 'YYYY年M月D日 HH:mm',
			llll : 'YYYY年M月D日dddd HH:mm'
		},
		meridiemParse: /凌晨|早上|上午|中午|下午|晚上/,
		meridiemHour: function (hour, meridiem) {
			if (hour === 12) {
				hour = 0;
			}
			if (meridiem === '凌晨' || meridiem === '早上' ||
				meridiem === '上午') {
				return hour;
			} else if (meridiem === '下午' || meridiem === '晚上') {
				return hour + 12;
			} else {
				// '中午'
				return hour >= 11 ? hour : hour + 12;
			}
		},
		meridiem : function (hour, minute, isLower) {
			var hm = hour * 100 + minute;
			if (hm < 600) {
				return '凌晨';
			} else if (hm < 900) {
				return '早上';
			} else if (hm < 1130) {
				return '上午';
			} else if (hm < 1230) {
				return '中午';
			} else if (hm < 1800) {
				return '下午';
			} else {
				return '晚上';
			}
		},
		calendar : {
			sameDay : '[今天]LT',
			nextDay : '[明天]LT',
			nextWeek : '[下]ddddLT',
			lastDay : '[昨天]LT',
			lastWeek : '[上]ddddLT',
			sameElse : 'L'
		},
		dayOfMonthOrdinalParse: /\d{1,2}(日|月|周)/,
		ordinal : function (number, period) {
			switch (period) {
				case 'd':
				case 'D':
				case 'DDD':
					return number + '日';
				case 'M':
					return number + '月';
				case 'w':
				case 'W':
					return number + '周';
				default:
					return number;
			}
		},
		relativeTime : {
			future : '%s内',
			past : '%s前',
			s : '几秒',
			ss : '%d 秒',
			m : '1 分钟',
			mm : '%d 分钟',
			h : '1 小时',
			hh : '%d 小时',
			d : '1 天',
			dd : '%d 天',
			M : '1 个月',
			MM : '%d 个月',
			y : '1 年',
			yy : '%d 年'
		},
		week : {
			// GB/T 7408-1994《数据元和交换格式·信息交换·日期和时间表示法》与ISO 8601:1988等效
			dow : 1, // Monday is the first day of the week.
			doy : 4  // The week that contains Jan 4th is the first week of the year.
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('zh-hk', {
		months : '一月_二月_三月_四月_五月_六月_七月_八月_九月_十月_十一月_十二月'.split('_'),
		monthsShort : '1月_2月_3月_4月_5月_6月_7月_8月_9月_10月_11月_12月'.split('_'),
		weekdays : '星期日_星期一_星期二_星期三_星期四_星期五_星期六'.split('_'),
		weekdaysShort : '週日_週一_週二_週三_週四_週五_週六'.split('_'),
		weekdaysMin : '日_一_二_三_四_五_六'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'YYYY/MM/DD',
			LL : 'YYYY年M月D日',
			LLL : 'YYYY年M月D日 HH:mm',
			LLLL : 'YYYY年M月D日dddd HH:mm',
			l : 'YYYY/M/D',
			ll : 'YYYY年M月D日',
			lll : 'YYYY年M月D日 HH:mm',
			llll : 'YYYY年M月D日dddd HH:mm'
		},
		meridiemParse: /凌晨|早上|上午|中午|下午|晚上/,
		meridiemHour : function (hour, meridiem) {
			if (hour === 12) {
				hour = 0;
			}
			if (meridiem === '凌晨' || meridiem === '早上' || meridiem === '上午') {
				return hour;
			} else if (meridiem === '中午') {
				return hour >= 11 ? hour : hour + 12;
			} else if (meridiem === '下午' || meridiem === '晚上') {
				return hour + 12;
			}
		},
		meridiem : function (hour, minute, isLower) {
			var hm = hour * 100 + minute;
			if (hm < 600) {
				return '凌晨';
			} else if (hm < 900) {
				return '早上';
			} else if (hm < 1130) {
				return '上午';
			} else if (hm < 1230) {
				return '中午';
			} else if (hm < 1800) {
				return '下午';
			} else {
				return '晚上';
			}
		},
		calendar : {
			sameDay : '[今天]LT',
			nextDay : '[明天]LT',
			nextWeek : '[下]ddddLT',
			lastDay : '[昨天]LT',
			lastWeek : '[上]ddddLT',
			sameElse : 'L'
		},
		dayOfMonthOrdinalParse: /\d{1,2}(日|月|週)/,
		ordinal : function (number, period) {
			switch (period) {
				case 'd' :
				case 'D' :
				case 'DDD' :
					return number + '日';
				case 'M' :
					return number + '月';
				case 'w' :
				case 'W' :
					return number + '週';
				default :
					return number;
			}
		},
		relativeTime : {
			future : '%s內',
			past : '%s前',
			s : '幾秒',
			ss : '%d 秒',
			m : '1 分鐘',
			mm : '%d 分鐘',
			h : '1 小時',
			hh : '%d 小時',
			d : '1 天',
			dd : '%d 天',
			M : '1 個月',
			MM : '%d 個月',
			y : '1 年',
			yy : '%d 年'
		}
	});

	//! moment.js locale configuration

	hooks.defineLocale('zh-tw', {
		months : '一月_二月_三月_四月_五月_六月_七月_八月_九月_十月_十一月_十二月'.split('_'),
		monthsShort : '1月_2月_3月_4月_5月_6月_7月_8月_9月_10月_11月_12月'.split('_'),
		weekdays : '星期日_星期一_星期二_星期三_星期四_星期五_星期六'.split('_'),
		weekdaysShort : '週日_週一_週二_週三_週四_週五_週六'.split('_'),
		weekdaysMin : '日_一_二_三_四_五_六'.split('_'),
		longDateFormat : {
			LT : 'HH:mm',
			LTS : 'HH:mm:ss',
			L : 'YYYY/MM/DD',
			LL : 'YYYY年M月D日',
			LLL : 'YYYY年M月D日 HH:mm',
			LLLL : 'YYYY年M月D日dddd HH:mm',
			l : 'YYYY/M/D',
			ll : 'YYYY年M月D日',
			lll : 'YYYY年M月D日 HH:mm',
			llll : 'YYYY年M月D日dddd HH:mm'
		},
		meridiemParse: /凌晨|早上|上午|中午|下午|晚上/,
		meridiemHour : function (hour, meridiem) {
			if (hour === 12) {
				hour = 0;
			}
			if (meridiem === '凌晨' || meridiem === '早上' || meridiem === '上午') {
				return hour;
			} else if (meridiem === '中午') {
				return hour >= 11 ? hour : hour + 12;
			} else if (meridiem === '下午' || meridiem === '晚上') {
				return hour + 12;
			}
		},
		meridiem : function (hour, minute, isLower) {
			var hm = hour * 100 + minute;
			if (hm < 600) {
				return '凌晨';
			} else if (hm < 900) {
				return '早上';
			} else if (hm < 1130) {
				return '上午';
			} else if (hm < 1230) {
				return '中午';
			} else if (hm < 1800) {
				return '下午';
			} else {
				return '晚上';
			}
		},
		calendar : {
			sameDay : '[今天] LT',
			nextDay : '[明天] LT',
			nextWeek : '[下]dddd LT',
			lastDay : '[昨天] LT',
			lastWeek : '[上]dddd LT',
			sameElse : 'L'
		},
		dayOfMonthOrdinalParse: /\d{1,2}(日|月|週)/,
		ordinal : function (number, period) {
			switch (period) {
				case 'd' :
				case 'D' :
				case 'DDD' :
					return number + '日';
				case 'M' :
					return number + '月';
				case 'w' :
				case 'W' :
					return number + '週';
				default :
					return number;
			}
		},
		relativeTime : {
			future : '%s內',
			past : '%s前',
			s : '幾秒',
			ss : '%d 秒',
			m : '1 分鐘',
			mm : '%d 分鐘',
			h : '1 小時',
			hh : '%d 小時',
			d : '1 天',
			dd : '%d 天',
			M : '1 個月',
			MM : '%d 個月',
			y : '1 年',
			yy : '%d 年'
		}
	});

	hooks.locale('en');

	return hooks;

})));
/*! pace 1.0.2 */
(function(){var a,b,c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,x,y,z,A,B,C,D,E,F,G,H,I,J,K,L,M,N,O,P,Q,R,S,T,U,V,W,X=[].slice,Y={}.hasOwnProperty,Z=function(a,b){function c(){this.constructor=a}for(var d in b)Y.call(b,d)&&(a[d]=b[d]);return c.prototype=b.prototype,a.prototype=new c,a.__super__=b.prototype,a},$=[].indexOf||function(a){for(var b=0,c=this.length;c>b;b++)if(b in this&&this[b]===a)return b;return-1};for(u={catchupTime:100,initialRate:.03,minTime:250,ghostTime:100,maxProgressPerFrame:20,easeFactor:1.25,startOnPageLoad:!0,restartOnPushState:!0,restartOnRequestAfter:500,target:"body",elements:{checkInterval:100,selectors:["body"]},eventLag:{minSamples:10,sampleCount:3,lagThreshold:3},ajax:{trackMethods:["GET"],trackWebSockets:!0,ignoreURLs:[]}},C=function(){var a;return null!=(a="undefined"!=typeof performance&&null!==performance&&"function"==typeof performance.now?performance.now():void 0)?a:+new Date},E=window.requestAnimationFrame||window.mozRequestAnimationFrame||window.webkitRequestAnimationFrame||window.msRequestAnimationFrame,t=window.cancelAnimationFrame||window.mozCancelAnimationFrame,null==E&&(E=function(a){return setTimeout(a,50)},t=function(a){return clearTimeout(a)}),G=function(a){var b,c;return b=C(),(c=function(){var d;return d=C()-b,d>=33?(b=C(),a(d,function(){return E(c)})):setTimeout(c,33-d)})()},F=function(){var a,b,c;return c=arguments[0],b=arguments[1],a=3<=arguments.length?X.call(arguments,2):[],"function"==typeof c[b]?c[b].apply(c,a):c[b]},v=function(){var a,b,c,d,e,f,g;for(b=arguments[0],d=2<=arguments.length?X.call(arguments,1):[],f=0,g=d.length;g>f;f++)if(c=d[f])for(a in c)Y.call(c,a)&&(e=c[a],null!=b[a]&&"object"==typeof b[a]&&null!=e&&"object"==typeof e?v(b[a],e):b[a]=e);return b},q=function(a){var b,c,d,e,f;for(c=b=0,e=0,f=a.length;f>e;e++)d=a[e],c+=Math.abs(d),b++;return c/b},x=function(a,b){var c,d,e;if(null==a&&(a="options"),null==b&&(b=!0),e=document.querySelector("[data-pace-"+a+"]")){if(c=e.getAttribute("data-pace-"+a),!b)return c;try{return JSON.parse(c)}catch(f){return d=f,"undefined"!=typeof console&&null!==console?console.error("Error parsing inline pace options",d):void 0}}},g=function(){function a(){}return a.prototype.on=function(a,b,c,d){var e;return null==d&&(d=!1),null==this.bindings&&(this.bindings={}),null==(e=this.bindings)[a]&&(e[a]=[]),this.bindings[a].push({handler:b,ctx:c,once:d})},a.prototype.once=function(a,b,c){return this.on(a,b,c,!0)},a.prototype.off=function(a,b){var c,d,e;if(null!=(null!=(d=this.bindings)?d[a]:void 0)){if(null==b)return delete this.bindings[a];for(c=0,e=[];c<this.bindings[a].length;)e.push(this.bindings[a][c].handler===b?this.bindings[a].splice(c,1):c++);return e}},a.prototype.trigger=function(){var a,b,c,d,e,f,g,h,i;if(c=arguments[0],a=2<=arguments.length?X.call(arguments,1):[],null!=(g=this.bindings)?g[c]:void 0){for(e=0,i=[];e<this.bindings[c].length;)h=this.bindings[c][e],d=h.handler,b=h.ctx,f=h.once,d.apply(null!=b?b:this,a),i.push(f?this.bindings[c].splice(e,1):e++);return i}},a}(),j=window.Pace||{},window.Pace=j,v(j,g.prototype),D=j.options=v({},u,window.paceOptions,x()),U=["ajax","document","eventLag","elements"],Q=0,S=U.length;S>Q;Q++)K=U[Q],D[K]===!0&&(D[K]=u[K]);i=function(a){function b(){return V=b.__super__.constructor.apply(this,arguments)}return Z(b,a),b}(Error),b=function(){function a(){this.progress=0}return a.prototype.getElement=function(){var a;if(null==this.el){if(a=document.querySelector(D.target),!a)throw new i;this.el=document.createElement("div"),this.el.className="pace pace-active",document.body.className=document.body.className.replace(/pace-done/g,""),document.body.className+=" pace-running",this.el.innerHTML='<div class="pace-progress">\n  <div class="pace-progress-inner"></div>\n</div>\n<div class="pace-activity"></div>',null!=a.firstChild?a.insertBefore(this.el,a.firstChild):a.appendChild(this.el)}return this.el},a.prototype.finish=function(){var a;return a=this.getElement(),a.className=a.className.replace("pace-active",""),a.className+=" pace-inactive",document.body.className=document.body.className.replace("pace-running",""),document.body.className+=" pace-done"},a.prototype.update=function(a){return this.progress=a,this.render()},a.prototype.destroy=function(){try{this.getElement().parentNode.removeChild(this.getElement())}catch(a){i=a}return this.el=void 0},a.prototype.render=function(){var a,b,c,d,e,f,g;if(null==document.querySelector(D.target))return!1;for(a=this.getElement(),d="translate3d("+this.progress+"%, 0, 0)",g=["webkitTransform","msTransform","transform"],e=0,f=g.length;f>e;e++)b=g[e],a.children[0].style[b]=d;return(!this.lastRenderedProgress||this.lastRenderedProgress|0!==this.progress|0)&&(a.children[0].setAttribute("data-progress-text",""+(0|this.progress)+"%"),this.progress>=100?c="99":(c=this.progress<10?"0":"",c+=0|this.progress),a.children[0].setAttribute("data-progress",""+c)),this.lastRenderedProgress=this.progress},a.prototype.done=function(){return this.progress>=100},a}(),h=function(){function a(){this.bindings={}}return a.prototype.trigger=function(a,b){var c,d,e,f,g;if(null!=this.bindings[a]){for(f=this.bindings[a],g=[],d=0,e=f.length;e>d;d++)c=f[d],g.push(c.call(this,b));return g}},a.prototype.on=function(a,b){var c;return null==(c=this.bindings)[a]&&(c[a]=[]),this.bindings[a].push(b)},a}(),P=window.XMLHttpRequest,O=window.XDomainRequest,N=window.WebSocket,w=function(a,b){var c,d,e;e=[];for(d in b.prototype)try{e.push(null==a[d]&&"function"!=typeof b[d]?"function"==typeof Object.defineProperty?Object.defineProperty(a,d,{get:function(){return b.prototype[d]},configurable:!0,enumerable:!0}):a[d]=b.prototype[d]:void 0)}catch(f){c=f}return e},A=[],j.ignore=function(){var a,b,c;return b=arguments[0],a=2<=arguments.length?X.call(arguments,1):[],A.unshift("ignore"),c=b.apply(null,a),A.shift(),c},j.track=function(){var a,b,c;return b=arguments[0],a=2<=arguments.length?X.call(arguments,1):[],A.unshift("track"),c=b.apply(null,a),A.shift(),c},J=function(a){var b;if(null==a&&(a="GET"),"track"===A[0])return"force";if(!A.length&&D.ajax){if("socket"===a&&D.ajax.trackWebSockets)return!0;if(b=a.toUpperCase(),$.call(D.ajax.trackMethods,b)>=0)return!0}return!1},k=function(a){function b(){var a,c=this;b.__super__.constructor.apply(this,arguments),a=function(a){var b;return b=a.open,a.open=function(d,e){return J(d)&&c.trigger("request",{type:d,url:e,request:a}),b.apply(a,arguments)}},window.XMLHttpRequest=function(b){var c;return c=new P(b),a(c),c};try{w(window.XMLHttpRequest,P)}catch(d){}if(null!=O){window.XDomainRequest=function(){var b;return b=new O,a(b),b};try{w(window.XDomainRequest,O)}catch(d){}}if(null!=N&&D.ajax.trackWebSockets){window.WebSocket=function(a,b){var d;return d=null!=b?new N(a,b):new N(a),J("socket")&&c.trigger("request",{type:"socket",url:a,protocols:b,request:d}),d};try{w(window.WebSocket,N)}catch(d){}}}return Z(b,a),b}(h),R=null,y=function(){return null==R&&(R=new k),R},I=function(a){var b,c,d,e;for(e=D.ajax.ignoreURLs,c=0,d=e.length;d>c;c++)if(b=e[c],"string"==typeof b){if(-1!==a.indexOf(b))return!0}else if(b.test(a))return!0;return!1},y().on("request",function(b){var c,d,e,f,g;return f=b.type,e=b.request,g=b.url,I(g)?void 0:j.running||D.restartOnRequestAfter===!1&&"force"!==J(f)?void 0:(d=arguments,c=D.restartOnRequestAfter||0,"boolean"==typeof c&&(c=0),setTimeout(function(){var b,c,g,h,i,k;if(b="socket"===f?e.readyState<2:0<(h=e.readyState)&&4>h){for(j.restart(),i=j.sources,k=[],c=0,g=i.length;g>c;c++){if(K=i[c],K instanceof a){K.watch.apply(K,d);break}k.push(void 0)}return k}},c))}),a=function(){function a(){var a=this;this.elements=[],y().on("request",function(){return a.watch.apply(a,arguments)})}return a.prototype.watch=function(a){var b,c,d,e;return d=a.type,b=a.request,e=a.url,I(e)?void 0:(c="socket"===d?new n(b):new o(b),this.elements.push(c))},a}(),o=function(){function a(a){var b,c,d,e,f,g,h=this;if(this.progress=0,null!=window.ProgressEvent)for(c=null,a.addEventListener("progress",function(a){return h.progress=a.lengthComputable?100*a.loaded/a.total:h.progress+(100-h.progress)/2},!1),g=["load","abort","timeout","error"],d=0,e=g.length;e>d;d++)b=g[d],a.addEventListener(b,function(){return h.progress=100},!1);else f=a.onreadystatechange,a.onreadystatechange=function(){var b;return 0===(b=a.readyState)||4===b?h.progress=100:3===a.readyState&&(h.progress=50),"function"==typeof f?f.apply(null,arguments):void 0}}return a}(),n=function(){function a(a){var b,c,d,e,f=this;for(this.progress=0,e=["error","open"],c=0,d=e.length;d>c;c++)b=e[c],a.addEventListener(b,function(){return f.progress=100},!1)}return a}(),d=function(){function a(a){var b,c,d,f;for(null==a&&(a={}),this.elements=[],null==a.selectors&&(a.selectors=[]),f=a.selectors,c=0,d=f.length;d>c;c++)b=f[c],this.elements.push(new e(b))}return a}(),e=function(){function a(a){this.selector=a,this.progress=0,this.check()}return a.prototype.check=function(){var a=this;return document.querySelector(this.selector)?this.done():setTimeout(function(){return a.check()},D.elements.checkInterval)},a.prototype.done=function(){return this.progress=100},a}(),c=function(){function a(){var a,b,c=this;this.progress=null!=(b=this.states[document.readyState])?b:100,a=document.onreadystatechange,document.onreadystatechange=function(){return null!=c.states[document.readyState]&&(c.progress=c.states[document.readyState]),"function"==typeof a?a.apply(null,arguments):void 0}}return a.prototype.states={loading:0,interactive:50,complete:100},a}(),f=function(){function a(){var a,b,c,d,e,f=this;this.progress=0,a=0,e=[],d=0,c=C(),b=setInterval(function(){var g;return g=C()-c-50,c=C(),e.push(g),e.length>D.eventLag.sampleCount&&e.shift(),a=q(e),++d>=D.eventLag.minSamples&&a<D.eventLag.lagThreshold?(f.progress=100,clearInterval(b)):f.progress=100*(3/(a+3))},50)}return a}(),m=function(){function a(a){this.source=a,this.last=this.sinceLastUpdate=0,this.rate=D.initialRate,this.catchup=0,this.progress=this.lastProgress=0,null!=this.source&&(this.progress=F(this.source,"progress"))}return a.prototype.tick=function(a,b){var c;return null==b&&(b=F(this.source,"progress")),b>=100&&(this.done=!0),b===this.last?this.sinceLastUpdate+=a:(this.sinceLastUpdate&&(this.rate=(b-this.last)/this.sinceLastUpdate),this.catchup=(b-this.progress)/D.catchupTime,this.sinceLastUpdate=0,this.last=b),b>this.progress&&(this.progress+=this.catchup*a),c=1-Math.pow(this.progress/100,D.easeFactor),this.progress+=c*this.rate*a,this.progress=Math.min(this.lastProgress+D.maxProgressPerFrame,this.progress),this.progress=Math.max(0,this.progress),this.progress=Math.min(100,this.progress),this.lastProgress=this.progress,this.progress},a}(),L=null,H=null,r=null,M=null,p=null,s=null,j.running=!1,z=function(){return D.restartOnPushState?j.restart():void 0},null!=window.history.pushState&&(T=window.history.pushState,window.history.pushState=function(){return z(),T.apply(window.history,arguments)}),null!=window.history.replaceState&&(W=window.history.replaceState,window.history.replaceState=function(){return z(),W.apply(window.history,arguments)}),l={ajax:a,elements:d,document:c,eventLag:f},(B=function(){var a,c,d,e,f,g,h,i;for(j.sources=L=[],g=["ajax","elements","document","eventLag"],c=0,e=g.length;e>c;c++)a=g[c],D[a]!==!1&&L.push(new l[a](D[a]));for(i=null!=(h=D.extraSources)?h:[],d=0,f=i.length;f>d;d++)K=i[d],L.push(new K(D));return j.bar=r=new b,H=[],M=new m})(),j.stop=function(){return j.trigger("stop"),j.running=!1,r.destroy(),s=!0,null!=p&&("function"==typeof t&&t(p),p=null),B()},j.restart=function(){return j.trigger("restart"),j.stop(),j.start()},j.go=function(){var a;return j.running=!0,r.render(),a=C(),s=!1,p=G(function(b,c){var d,e,f,g,h,i,k,l,n,o,p,q,t,u,v,w;for(l=100-r.progress,e=p=0,f=!0,i=q=0,u=L.length;u>q;i=++q)for(K=L[i],o=null!=H[i]?H[i]:H[i]=[],h=null!=(w=K.elements)?w:[K],k=t=0,v=h.length;v>t;k=++t)g=h[k],n=null!=o[k]?o[k]:o[k]=new m(g),f&=n.done,n.done||(e++,p+=n.tick(b));return d=p/e,r.update(M.tick(b,d)),r.done()||f||s?(r.update(100),j.trigger("done"),setTimeout(function(){return r.finish(),j.running=!1,j.trigger("hide")},Math.max(D.ghostTime,Math.max(D.minTime-(C()-a),0)))):c()})},j.start=function(a){v(D,a),j.running=!0;try{r.render()}catch(b){i=b}return document.querySelector(".pace")?(j.trigger("start"),j.go()):setTimeout(j.start,50)},"function"==typeof define&&define.amd?define('pace',["pace"],function(){return j}):"object"==typeof exports?module.exports=j:D.startOnPageLoad&&j.start()}).call(this);
/*
This file is part of SMAP.

SMAP is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

SMAP is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with SMAP.  If not, see <http://www.gnu.org/licenses/>.

*/

/*
 * Purpose: Manage the panels that display graphs, maps etc of results data
 */

var gUserLocale = navigator.language;
if (Modernizr.localstorage) {
	gUserLocale = localStorage.getItem('user_locale') || navigator.language;
} 

requirejs.config({
    baseUrl: 'js/libs',
    locale: gUserLocale,
    waitSeconds: 0,
    paths: {
     	app: '../app',
     	i18n: '../../../../js/libs/i18n',
     	async: '../../../../js/libs/async',
     	localise: '../../../../js/app/localise',
	    globals: '../../../../js/app/globals',
    	modernizr: '../../../../js/libs/modernizr',
	    moment: '../../../../js/libs/moment-with-locales.2.24.0',
    	common: '../../../../js/app/common',
    	lang_location: '../../../../js',
    	pace: '../../../../js/libs/wb/pace/pace.min'
    },
    shim: {

    	'common': ['jquery'],
    	'bootstrap': ['jquery']
    	}
    });

require([
         'jquery',
         'common',
         'localise',
		 'globals',
		 'moment',
         'pace'

         ], function($,
        		 common,
        		 localise,
        		 globals,
		         moment) {

	var table;
	var gSelectedIndexes;
	var gSelectedRecord;
	var gSelectedId;
	
	$(document).ready(function() {

        setCustomSubs();
		setTheme();
		setupUserProfile(true);
		localise.setlang();		// Localise HTML


		getLoggedInUser(gotUser, false, false, undefined);

		
	});

	function gotUser() {
		var tzString = globals.gTimezone ? "&tz=" + encodeURIComponent(globals.gTimezone) : "";
		table = $('#sub_table').DataTable({
			processing: true,
			scrollY: '70vh',
			scrollX: true,
			scrollCollapse: true,
			ajax: "/surveyKPI/api/subscriptions?dt=true" + tzString,
			select: true,
			rowId: 'id',
			columns: [
				{ "data": "email" },
				{ "data": "name" },
				{ "data": "status_loc"  },
				{ "data": "time_changed"  }
			],
			order: [[ 0, "asc" ]],
			initComplete: function () {
				this.api().columns().every( function () {
					var column = this;
					if (column.index() === 1 || column.index() === 2) {
						var select = $('<select><option value=""></option></select>')
							.appendTo($(column.footer()).empty())
							.on('change', function () {
								var val = $.fn.dataTable.util.escapeRegex(
									$(this).val()
								);

								column
									.search(val ? '^' + val + '$' : '', true, false)
									.draw();
							});

						column.data().unique().sort().each(function (d, j) {
							select.append('<option value="' + d + '">' + d + '</option>')
						});
					}
				} );

			}

		});

		$('#sub_table').find('td').css('white-space','initial').css('word-wrap', 'break-word');

		// Respond to selection of a row
		table.off('select').on('select', function (e, dt, type, indexes) {
			recordSelected(indexes);
		});
		table.off('deselect').on('deselect', function (e, dt, type, indexes) {
			$('.selectedOnly').hide();
		});

		$('#m_refresh').click(function(e) {	// Add refresh action
			table.ajax.reload();
		});

		$('#m_add').click(function(e){
			e.preventDefault();
			gSelectedId = -1;
			initPersonDialog(-1);
			$('#addPersonPopup').modal("show");
		});

		$('#m_edit').click(function(e){
			e.preventDefault();
			gSelectedId = gSelectedRecord.id;
			initPersonDialog(gSelectedRecord.id);
			$('#addPersonPopup').modal("show");
		});

		$('#m_del').click(function(e){
			e.preventDefault();
			gSelectedId = gSelectedRecord.id;

			msg = localise.set["msg_confirm_del"];
			msg += ": ";
			msg += gSelectedRecord.name;

			bootbox.confirm(htmlEncode(msg), function(result){
				if(result) {
					deletePerson(gSelectedId);
				}
			});
		});


		$('#savePerson').click(function(){savePerson();});
	}
	/*
	 * Initialise the dialog
	 */
	function initPersonDialog(idx) {

		if(idx > 0) {
			// existing
			$('#p_email').val(gSelectedRecord.email);
			$('#p_name').val(gSelectedRecord.name);
		} else {
			// new
			$('#p_email').val("");
			$('#p_name').val("");
		}

	}

	/*
	 * Delete the entry for a person
	 */
	function deletePerson(id) {

		var url = '/surveyKPI/people/' + id;
		addHourglass();
		$.ajax({
			type: "DELETE",
			cache: false,
			async: true,
			url: url,
			success: function(data, status) {
				removeHourglass();
				table.rows( { selected: true } ).deselect();
				table.ajax.reload();
			},
			error: function(xhr, textStatus, err) {
				removeHourglass();
				if(xhr.readyState == 0 || xhr.status == 0) {
					return;  // Not an error
				} else {
					alert(localise.set["msg_err_del"] + " " + xhr.responseText);
				}
			}
		});
	}

	/*
	 * Save a person
	 */
	function savePerson() {

		var url,
			person = {},
			personString,
			errorMsg;

		person.email = $('#p_email').val();
		person.name = $('#p_name').val();

		/*
		 * Validation
		 *
		 * email
		 */

		if(person.email && person.email.trim().length > 0) {
			var emailArray = person.email.split(",");
			if(emailArray.length > 1) {
				errorMsg = localise.set["msg_inv_email"];
			}
			if (!validateEmails(person.email)) {
				errorMsg = localise.set["msg_inv_email"];
			}
		} else {
			errorMsg = localise.set["msg_inv_email"];
		}

		// name
		if(!errorMsg) {
			if(!person.name || person.name.trim().length == 0) {
				errorMsg = localise.set["msg_val_name"];
			}
		}

		if(!errorMsg) {

			person.id = gSelectedId;
			personString = JSON.stringify(person);
			url = '/surveyKPI/people'

			addHourglass();
			$.ajax({
				type: "POST",
				dataType: 'text',
				cache: false,
				async: true,
				url: url,
				data: { person: personString },
				success: function(data, status) {
					removeHourglass();
					table.ajax.reload();
					$('#addPersonPopup').modal("hide");
				},
				error: function(xhr, textStatus, err) {
					removeHourglass();
					if(xhr.readyState == 0 || xhr.status == 0) {
						return;  // Not an error
					} else {
						alert(localise.set["msg_err_save"] + " " + xhr.responseText);
					}
				}
			});

		} else {
			alert(errorMsg);
		}
	}

	function recordSelected(indexes) {
		$('.selectedOnly').show();
		gSelectedIndexes = indexes;
		gSelectedRecord = table.rows(gSelectedIndexes).data().toArray()[0];
	}
});


define("../contacts", function(){});

